/*
 * Copyright (c) 2001 Christoph Hellwig.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *	$Id: sysent.c,v 1.9 2001/07/13 13:46:56 hch Exp $
 */

/*
 * Wyse/386 personality switch.
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/personality.h>
#include <linux/sched.h>
#include <linux/syscall.h>

#include <abi/svr4/sysent.h>
#include <abi/wyse/sysent.h>

#include <abi/abi.h>
#include <abi/socket.h>
#include <abi/sysent.h>
#include <abi/svr4.h>


extern void	svr4_class_cxenix(struct pt_regs *);
static void	wyse_class_nfs(struct pt_regs *);
static void	wyse_class_tcp(struct pt_regs *);


static u_char wyse_err_table[] = {
	/*   0 -   9 */   0,   1,   2,    3,   4,   5,   6,   7,   8,   9,
	/*  10 -  19 */   10,  11,  12,  13,  14,  15,  16,  17,  18,  19,
	/*  20 -  29 */   20,  21,  22,  23,  24,  25,  26,  27,  28,  29,
	/*  30 -  39 */   30,  31,  32,  33,  34,  45,  78,  46,  228, 46,
	/*  40 -  49 */   22, 231, 227, 200,  37,  38,  39,  40,  41,  42,
	/*  50 -  59 */   43,  44,  50,  51,  52,  53,  54,  55,  56,  57,
	/*  60 -  69 */   60,  61,  62,  63,  64,  65,  66,  67,  68,  69,
	/*  70 -  79 */   70,  71,  74,  76,  77,  22,  80,  81,  82,  83,
	/*  80 -  89 */   84,  85,  86,  87,  22,   4,  22, 233, 203, 204,
	/*  90 -  99 */  205, 206, 207, 208, 209, 210, 211, 212, 213, 214,
	/* 100 - 109 */  215, 216, 217, 218, 219, 220, 221, 222, 223, 224,
	/* 110 - 119 */  225, 226, 229, 230, 202, 201, 237, 135, 137, 138,
	/* 120 - 122 */  139, 140, 234
};

/*
 * Map Linux RESTART* values (512,513,514) to EINTR
 */
static u_char lnx_err_table[] = {
	/* 512 - 514 */ EINTR, EINTR, EINTR
};

static struct map_segment wyse_err_map[] = {
	{ 0,	0 + sizeof(wyse_err_table) - 1,		wyse_err_table },
	{ 512,	512 + sizeof(lnx_err_table) - 1,	lnx_err_table },
	{ -1 }
};

static abi_func_t wyse_nfscall_table[] = {
   { 0,			Ukn	ITR(1, "nfs_svc",	"")	}, /*  0 */
   { 0,			Ukn	ITR(1, "async_daemon",	"")	}, /*  1 */
   { 0,			Ukn	ITR(1, "nfs_getfh",	"")	}, /*  2 */
   { 0,			Ukn	ITR(1, "nfsmount",	"")	}  /*  3 */
};

static abi_func_t wyse_tcpcall_table[] = {
   { SC(_newselect),	-5      ITR(0, "select",        "dxxxx")},/*  0 */
   { abi_socket,	Spl     ITR(0, "socket",        "ddd")  }, /*  1 */
   { abi_connect,	Spl     ITR(0, "connect",       "dxd")  }, /*  2 */
   { abi_accept,	Spl     ITR(0, "accept",        "dxx")  }, /*  3 */
   { abi_send,		Spl     ITR(0, "send",          "dxdd")}, /*  4 */
   { abi_recv,		Spl     ITR(0, "recv",          "dxdd")}, /*  5 */
   { abi_bind,		Spl     ITR(0, "bind",          "dxd")  }, /*  6 */
   { abi_setsockopt,	Spl     ITR(0, "setsockopt",    "")     },  /*  7 */
   { abi_listen,	Spl     ITR(0, "listen",        "dd")   }, /*  8 */
   { 0,			3       ITR(1, "recvmsg",       "dxd")  }, /*  9 */
   { 0,			3       ITR(1, "sendmsg",       "dxd")  }, /* 10 */
   { abi_getsockopt,	Spl     ITR(0, "getsockopt",    "dddxx")}, /* 11 */
   { abi_recvfrom,	Spl     ITR(0, "recvfrom",      "dxddxd")},/* 12 */
   { abi_sendto,	Spl     ITR(0, "sendto",        "dxddxd")},/* 13 */
   { abi_shutdown,	Spl     ITR(0, "shutdown",      "dd")   }, /* 14 */
   { abi_socketpair,	Spl     ITR(0, "socketpair",    "dddx")},  /* 15 */
   { 0,			Ukn     ITR(1, "trace",         "")     }, /* 16 */
   { abi_getpeername,	Spl     ITR(0, "getpeername",   "dxx")  }, /* 17 */
   { abi_getsockname,	Spl     ITR(0, "getsockname",   "")     }, /* 18 */
   { abi_wait3,		1       ITR(0, "wait3",         "x")    }, /* 19 */
};


static abi_func_t wyse_syscall_table[] = {
   { abi_syscall,	Fast	ITR(0, "syscall",	"")	}, /*    0 */
   { SC(exit),		-1	ITR(0, "exit",		"d")	}, /*    1 */
   { abi_fork,		Spl	ITR(0, "fork",		"")	}, /*    2 */
   { abi_read,		3	ITR(0, "read",		"dpd")	}, /*    3 */
   { SC(write),		-3	ITR(0, "write",		"dpd")	}, /*    4 */
   { svr4_open,		3	ITR(0, "open",		"soo")	}, /*    5 */
   { SC(close),		-1	ITR(0, "close",		"d")	}, /*    6 */
   { abi_wait,		Spl	ITR(0, "wait",		"xxx")	}, /*    7 */
   { SC(creat),		-2	ITR(0, "creat",		"so")	}, /*    8 */
   { SC(link),		-2	ITR(0, "link",		"ss")	}, /*    9 */
   { SC(unlink),	-1	ITR(0, "unlink",	"s")	}, /*   10 */
   { abi_exec,		Spl	ITR(0, "exec",		"sxx")	}, /*   11 */
   { SC(chdir),		-1	ITR(0, "chdir",		"s")	}, /*   12 */
   { abi_time,		0	ITR(0, "time",		"")	}, /*   13 */
   { abi_mknod,		3	ITR(0, "mknod",		"soo")	}, /*   14 */
   { SC(chmod),		-2	ITR(0, "chmod",		"so")	}, /*   15 */
   { SC(chown),		-3	ITR(0, "chown",		"sdd")	}, /*   16 */
   { abi_brk,		1	ITR(0, "brk/break",	"x")	}, /*   17 */
   { abi_stat,		2	ITR(0, "stat",		"sp")	}, /*   18 */
   { sys_lseek,		3	ITR(0, "seek/lseek",	"ddd")	}, /*   19 */
   { abi_getpid,	Spl	ITR(0, "getpid",	"")	}, /*   20 */
   { 0,			Ukn	ITR(1, "mount",		"")	}, /*   21 */
   { SC(umount),	-1	ITR(0, "umount",	"s")	}, /*   22 */
   { SC(setuid),	-1	ITR(0, "setuid",	"d")	}, /*   23 */
   { abi_getuid,	Spl	ITR(0, "getuid",	"")	}, /*   24 */
   { SC(stime),		-1	ITR(0, "stime",		"d")	}, /*   25 */
   { wyse_ptrace,	4	ITR(0, "ptrace",	"xdxx") }, /*   26 */
   { SC(alarm),		-1	ITR(0, "alarm",		"d")	}, /*   27 */
   { abi_fstat,		2	ITR(0, "fstat",		"dp")	}, /*   28 */
   { SC(pause),		-ZERO	ITR(0, "pause",		"")	}, /*   29 */
   { SC(utime),		-2	ITR(0, "utime",		"xx")	}, /*   30 */
   { 0,			Ukn	ITR(0, "stty",		"")	}, /*   31 */
   { 0,			Ukn	ITR(1, "gtty",		"")	}, /*   32 */
   { SC(access),	-2	ITR(0, "access",	"so")	}, /*   33 */
   { SC(nice),		-1	ITR(0, "nice",		"d")	}, /*   34 */
   { svr4_statfs,	4	ITR(0, "statfs",	"spdd")	}, /*   35 */
   { SC(sync),		-ZERO	ITR(0, "sync",		"")	}, /*   36 */
   { abi_kill,		2	ITR(0, "kill",		"dd")	}, /*   37 */
   { svr4_fstatfs,	4	ITR(0, "fstatfs",	"dpdd")	}, /*   38 */
   { abi_procids,	Spl	ITR(0, "ibcs_procids",	"d")	}, /*   39 */
   { svr4_class_cxenix,	Fast	ITR(0, "cxenix",	"")	}, /*   40 */
   { SC(dup),		-1	ITR(0, "dup",		"d")	}, /*   41 */
   { abi_pipe,		Spl	ITR(0, "pipe",		"")	}, /*   42 */
   { SC(times),		-1	ITR(0, "times",		"p")	}, /*   43 */
   { SC(profil),	-4	ITR(0, "prof",		"xxxx")	}, /*   44 */
   { 0,			Ukn	ITR(1, "lock/plock",	"")	}, /*   45 */
   { SC(setgid),	-1	ITR(0, "setgid",	"d")	}, /*   46 */
   { abi_getgid,	Spl	ITR(0, "getgid",	"")	}, /*   47 */
   { abi_sigfunc,	Fast	ITR(0, "sigfunc",	"xxx")	}, /*   48 */
   { svr4_msgsys,	Spl	ITR(0, "msgsys",	"dxddd")}, /*   49 */
   { svr4_sysi86,	Spl	ITR(0, "sysi86/sys3b",	"d")	}, /*   50 */
   { SC(acct),		-1	ITR(0, "acct/sysacct",	"x")	}, /*   51 */
   { svr4_shmsys,	Fast	ITR(0, "shmsys",	"ddxo")	}, /*   52 */
   { svr4_semsys,	Spl	ITR(0, "semsys",	"dddx")	}, /*   53 */
   { svr4_ioctl,	Spl	ITR(0, "ioctl",		"dxx")	}, /*   54 */
   { 0,			3	ITR(0, "uadmin",	"xxx")	}, /*   55 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   56 */
   { v7_utsname,	1	ITR(0, "utsys",		"x")	}, /*   57 */
   { SC(fsync),		-1	ITR(0, "fsync",		"d")	}, /*   58 */
   { abi_exec,		Spl	ITR(0, "execv",		"spp")	}, /*   59 */
   { SC(umask),		-1	ITR(0, "umask",		"o")	}, /*   60 */
   { SC(chroot),	-1	ITR(0, "chroot",	"s")	}, /*   61 */
   { svr4_fcntl,	Spl	ITR(0, "fcntl",		"dxx")	}, /*   62 */
   { svr4_ulimit,	2	ITR(0, "ulimit",	"xx")	}, /*   63 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   64 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   65 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   66 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   67 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   68 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   69 */
   { 0,			Ukn	ITR(1, "advfs",		"")	}, /*   70 */
   { 0,			Ukn	ITR(1, "unadvfs",	"")	}, /*   71 */
   { 0,			Ukn	ITR(1, "rmount",	"")	}, /*   72 */
   { 0,			Ukn	ITR(1, "rumount",	"")	}, /*   73 */
   { 0,			Ukn	ITR(1, "rfstart",	"")	}, /*   74 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   75 */
   { 0,			Ukn	ITR(1, "rdebug",	"")	}, /*   76 */
   { 0,			Ukn	ITR(1, "rfstop",	"")	}, /*   77 */
   { 0,			Ukn	ITR(1, "rfsys",		"")	}, /*   78 */
   { SC(rmdir),		-1	ITR(0, "rmdir",		"s")	}, /*   79 */
   { abi_mkdir,		2	ITR(0, "mkdir",		"so")	}, /*   80 */
   { svr4_getdents,	3	ITR(0, "getdents",	"dxd")	}, /*   81 */
   { 0,			Ukn	ITR(1, "libattach",	"")	}, /*   82 */
   { 0,			Ukn	ITR(1, "libdetach",	"")	}, /*   83 */
   { svr4_sysfs,	3	ITR(0, "sysfs",		"dxx")	}, /*   84 */
   { svr4_getmsg,	Spl	ITR(0, "getmsg",	"dxxx")	}, /*   85 */
   { svr4_putmsg,	Spl	ITR(0, "putmsg",	"dxxd")	}, /*   86 */
   { SC(poll),		-3	ITR(0, "poll",		"xdd")	}, /*   87 */
   { 0,			Ukn	ITR(0, "nosys88",	"")	}, /*   88 */
   { 0,			Ukn	ITR(0, "nosys89",	"")	}, /*   89 */
   { 0,			Ukn	ITR(0, "nosys90",	"")	}, /*   90 */
   { 0,			Ukn	ITR(0, "nosys91",	"")	}, /*   91 */
   { 0,			Ukn	ITR(0, "nosys92",	"")	}, /*   92 */
   { 0,			Ukn	ITR(0, "nosys93",	"")	}, /*   93 */
   { 0,			Ukn	ITR(0, "nosys94",	"")	}, /*   94 */
   { 0,			Ukn	ITR(0, "nosys95",	"")	}, /*   95 */
   { 0,			Ukn	ITR(0, "nosys96",	"")	}, /*   96 */
   { 0,			Ukn	ITR(0, "nosys97",	"")	}, /*   97 */
   { 0,			Ukn	ITR(0, "nosys98",	"")	}, /*   98 */
   { 0,			Ukn	ITR(0, "nosys99",	"")	}, /*   99 */
   { 0,			Ukn	ITR(0, "nosys100",	"")	}, /*  100 */
   { 0,			Ukn	ITR(0, "nosys101",	"")	}, /*  101 */
   { 0,			Ukn	ITR(0, "nosys102",	"")	}, /*  102 */
   { 0,			Ukn	ITR(0, "nosys103",	"")	}, /*  103 */
   { 0,			Ukn	ITR(0, "nosys104",	"")	}, /*  104 */
   { 0,			Ukn	ITR(0, "nosys105",	"")	}, /*  105 */
   { 0,			Ukn	ITR(0, "nosys106",	"")	}, /*  106 */
   { 0,			Ukn	ITR(0, "nosys107",	"")	}, /*  107 */
   { 0,			Ukn	ITR(0, "nosys108",	"")	}, /*  108 */
   { 0,			Ukn	ITR(0, "nosys109",	"")	}, /*  109 */
   { 0,			Ukn	ITR(0, "nosys110",	"")	}, /*  110 */
   { 0,			Ukn	ITR(0, "nosys111",	"")	}, /*  111 */
   { 0,			Ukn	ITR(0, "nosys112",	"")	}, /*  112 */
   { 0,			Ukn	ITR(0, "nosys113",	"")	}, /*  113 */
   { 0,			Ukn	ITR(0, "nosys114",	"")	}, /*  114 */
   { 0,			Ukn	ITR(0, "nosys115",	"")	}, /*  115 */
   { 0,			Ukn	ITR(0, "nosys116",	"")	}, /*  116 */
   { 0,			Ukn	ITR(0, "nosys117",	"")	}, /*  117 */
   { 0,			Ukn	ITR(0, "nosys118",	"")	}, /*  118 */
   { 0,			Ukn	ITR(0, "nosys119",	"")	}, /*  119 */
   { 0,			Ukn	ITR(0, "nosys120",	"")	}, /*  120 */
   { 0,			Ukn	ITR(0, "nosys121",	"")	}, /*  121 */
   { 0,			Ukn	ITR(0, "nosys122",	"")	}, /*  122 */
   { 0,			Ukn	ITR(0, "nosys123",	"")	}, /*  123 */
   { 0,			Ukn	ITR(0, "nosys124",	"")	}, /*  124 */
   { 0,			Ukn	ITR(0, "nosys125",	"")	}, /*  125 */
   { 0,			Ukn	ITR(0, "nosys126",	"")	}, /*  126 */
   { 0,			Ukn	ITR(0, "nosys127",	"")	}, /*  127 */
   { abi_lstat,		2	ITR(0, "lstat",		"sp")	}, /*  128 */
   { SC(readlink),	-3	ITR(0, "readlink",	"spd")	}, /*  129 */
   { SC(symlink),	-2	ITR(0, "symlink",	"ss")	}, /*  130 */
   { wyse_class_tcp,	Fast	ITR(0, "?",		"")	}, /*  131 */
   { wyse_class_nfs,	Fast	ITR(0, "?",		"")	}, /*  132 */
   { abi_gethostname,	2	ITR(0, "gethostname",	"xd")	}, /*  133 */
   { SC(sethostname),	-2	ITR(0, "sethostname",	"sd")	}, /*  134 */
   { abi_getdomainname,	2	ITR(0, "getdomainname","xd")	}, /*  135 */
   { SC(setdomainname),	-2	ITR(0, "setdomainname","sd")	}, /*  136 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*  137 */
   { SC(setreuid),	-2	ITR(0, "setreuid",	"dd")	}, /*  138 */
   { SC(setregid),	-2	ITR(0, "setregid",	"dd")	}, /*  139 */
};

static void
wyse_class_nfs(struct pt_regs *regp)
{
	int			i = regp->eax >> 8;

	if (i <= 3) {
		abi_dispatch(regp, &wyse_nfscall_table[i], 1);
		return;
	}
	
	regp->eax = iABI_errors(-EINVAL);
	regp->eflags |= 1;
}

static void
wyse_class_tcp(struct pt_regs *regp)
{
	int			i = regp->eax >> 8;
	
	if (i <= 19) {
		abi_dispatch(regp, &wyse_tcpcall_table[i], 1);
		return;
	}
		
	regp->eax = iABI_errors(-EINVAL);
	regp->eflags |= 1;
}

static void
wyse_lcall7(int segment, struct pt_regs * regp)
{
	abi_dispatch(regp, &wyse_syscall_table[regp->eax & 0xff], 1);
}

struct exec_domain wyse_exec_domain = {
	name:		"Wyse/386",
	handler:	wyse_lcall7,
	pers_low:	4 /* PER_WYSEV386 */,
	pers_high:	4 /* PER_WYSEV386 */,
	signal_map:	ibcs_to_linux_signals,
	signal_invmap:	linux_to_ibcs_signals,
	err_map:	wyse_err_map,
	socktype_map:	svr4_socktype_map,
	sockopt_map:	abi_sockopt_map,
	af_map:		abi_af_map,
	module:		THIS_MODULE
};


static int __init
wyse_module_init(void)
{
	return (register_exec_domain(&wyse_exec_domain));
}

static void __exit
wyse_module_exit(void)
{
	unregister_exec_domain(&wyse_exec_domain);
}

module_init(wyse_module_init);
module_exit(wyse_module_exit);
