/*
 * Copyright (c) 2001 Christoph Hellwig.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ident "$Id: sysent.c,v 1.4 2001/07/13 13:41:19 hch Exp $"

/*
 * SVR4 personality switch.
 * 
 * This file is somewhat icky as we put things together that don't
 * belong together.  At least Xenix should get it's own syscall table.
 * Maybe ISC should also be sorted out.
 */
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/personality.h>
#include <linux/syscall.h>

#include <abi/svr4/sysent.h>

#include <abi/abi.h>
#include <abi/signal.h>
#include <abi/sysent.h>


extern void svr4_class_cxenix(struct pt_regs *regs);
extern void iBCS_class_ISC(struct pt_regs *regs);

extern struct map_segment svr4_err_map[];
extern struct map_segment svr4_socktype_map[];
extern struct map_segment abi_sockopt_map[];
extern struct map_segment abi_af_map[];



static abi_func_t ibcs_syscall_table[] = {
   { abi_syscall,	Fast	ITR(0, "syscall",	"")	}, /*    0 */
   { SC(exit),		-1	ITR(0, "exit",		"d")	}, /*    1 */
   { abi_fork,		Spl	ITR(0, "fork",		"")	}, /*    2 */
   { abi_read,		3	ITR(0, "read",		"dpd")	}, /*    3 */
   { SC(write),		-3	ITR(0, "write",		"dpd")	}, /*    4 */
   { svr4_open,		3	ITR(0, "open",		"soo")	}, /*    5 */
   { SC(close),		-1	ITR(0, "close",		"d")	}, /*    6 */
   { abi_wait,		Spl	ITR(0, "wait",		"xxx")	}, /*    7 */
   { SC(creat),		-2	ITR(0, "creat",		"so")	}, /*    8 */
   { SC(link),		-2	ITR(0, "link",		"ss")	}, /*    9 */
   { SC(unlink),	-1	ITR(0, "unlink",	"s")	}, /*   10 */
   { abi_exec,		Spl	ITR(0, "exec",		"sxx")	}, /*   11 */
   { SC(chdir),		-1	ITR(0, "chdir",		"s")	}, /*   12 */
   { abi_time,		0	ITR(0, "time",		"")	}, /*   13 */
   { abi_mknod,		3	ITR(0, "mknod",		"soo")	}, /*   14 */
   { SC(chmod),		-2	ITR(0, "chmod",		"so")	}, /*   15 */
   { SC(chown),		-3	ITR(0, "chown",		"sdd")	}, /*   16 */
   { abi_brk,		1	ITR(0, "brk/break",	"x")	}, /*   17 */
   { abi_stat,		2	ITR(0, "stat",		"sp")	}, /*   18 */
   { sys_lseek,		3	ITR(0, "seek/lseek",	"ddd")	}, /*   19 */
   { abi_getpid,	Spl	ITR(0, "getpid",	"")	}, /*   20 */
   { 0,			Ukn	ITR(1, "mount",		"")	}, /*   21 */
   { SC(umount),	-1	ITR(0, "umount",	"s")	}, /*   22 */
   { SC(setuid),	-1	ITR(0, "setuid",	"d")	}, /*   23 */
   { abi_getuid,	Spl	ITR(0, "getuid",	"")	}, /*   24 */
   { SC(stime),		-1	ITR(0, "stime",		"d")	}, /*   25 */
   { 0,			Ukn	ITR(0, "ptrace",	"")	}, /*   26 */
   { SC(alarm),		-1	ITR(0, "alarm",		"d")	}, /*   27 */
   { abi_fstat,		2	ITR(0, "fstat",		"dp")	}, /*   28 */
   { SC(pause),		-ZERO	ITR(0, "pause",		"")	}, /*   29 */
   { SC(utime),		-2	ITR(0, "utime",		"xx")	}, /*   30 */
   { 0,			Ukn	ITR(0, "stty",		"")	}, /*   31 */
   { 0,			Ukn	ITR(1, "gtty",		"")	}, /*   32 */
   { SC(access),	-2	ITR(0, "access",	"so")	}, /*   33 */
   { SC(nice),		-1	ITR(0, "nice",		"d")	}, /*   34 */
   { svr4_statfs,	4	ITR(0, "statfs",	"spdd")	}, /*   35 */
   { SC(sync),		-ZERO	ITR(0, "sync",		"")	}, /*   36 */
   { abi_kill,		2	ITR(0, "kill",		"dd")	}, /*   37 */
   { svr4_fstatfs,	4	ITR(0, "fstatfs",	"dpdd")	}, /*   38 */
   { abi_procids,	Spl	ITR(0, "ibcs_procids",	"d")	}, /*   39 */
   { svr4_class_cxenix,	Fast	ITR(0, "cxenix",	"")	}, /*   40 */
   { SC(dup),		-1	ITR(0, "dup",		"d")	}, /*   41 */
   { abi_pipe,		Spl	ITR(0, "pipe",		"")	}, /*   42 */
   { SC(times),		-1	ITR(0, "times",		"p")	}, /*   43 */
   { SC(profil),	-4	ITR(0, "prof",		"xxxx")}, /*   44 */
   { 0,			Ukn	ITR(1, "lock/plock",	"")	}, /*   45 */
   { SC(setgid),	-1	ITR(0, "setgid",	"d")	}, /*   46 */
   { abi_getgid,	Spl	ITR(0, "getgid",	"")	}, /*   47 */
   { abi_sigfunc,	Fast	ITR(0, "sigfunc",	"xxx")	}, /*   48 */
   { svr4_msgsys,	Spl	ITR(0, "msgsys",	"dxddd")}, /*   49 */
   { svr4_sysi86,	Spl	ITR(0, "sysi86/sys3b",	"d")	}, /*   50 */
   { SC(acct),		-1	ITR(0, "acct/sysacct",	"x")	}, /*   51 */
   { svr4_shmsys,	Fast	ITR(0, "shmsys",	"ddxo")}, /*   52 */
   { svr4_semsys,	Spl	ITR(0, "semsys",	"dddx")}, /*   53 */
   { svr4_ioctl,	Spl	ITR(0, "ioctl",		"dxx")	}, /*   54 */
   { 0,			3	ITR(0, "uadmin",	"xxx")	}, /*   55 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   56 */
   { v7_utsname,	1	ITR(0, "utsys",		"x")	}, /*   57 */
   { SC(fsync),		-1	ITR(0, "fsync",		"d")	}, /*   58 */
   { abi_exec,		Spl	ITR(0, "execv",		"spp")	}, /*   59 */
   { SC(umask),		-1	ITR(0, "umask",		"o")	}, /*   60 */
   { SC(chroot),	-1	ITR(0, "chroot",	"s")	}, /*   61 */
   { svr4_fcntl,	Spl	ITR(0, "fcntl",		"dxx")	}, /*   62 */
   { svr4_ulimit,	2	ITR(0, "ulimit",	"xx")	}, /*   63 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   64 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   65 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   66 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   67 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   68 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   69 */
   { 0,			Ukn	ITR(1, "advfs",		"")	}, /*   70 */
   { 0,			Ukn	ITR(1, "unadvfs",	"")	}, /*   71 */
   { 0,			Ukn	ITR(1, "rmount",	"")	}, /*   72 */
   { 0,			Ukn	ITR(1, "rumount",	"")	}, /*   73 */
   { 0,			Ukn	ITR(1, "rfstart",	"")	}, /*   74 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   75 */
   { 0,			Ukn	ITR(1, "rdebug",	"")	}, /*   76 */
   { 0,			Ukn	ITR(1, "rfstop",	"")	}, /*   77 */
   { 0,			Ukn	ITR(1, "rfsys",		"")	}, /*   78 */
   { SC(rmdir),		-1	ITR(0, "rmdir",		"s")	}, /*   79 */
   { abi_mkdir,		2	ITR(0, "mkdir",		"so")	}, /*   80 */
   { svr4_getdents,	3	ITR(0, "getdents",	"dxd")	}, /*   81 */
   { 0,			Ukn	ITR(1, "libattach",	"")	}, /*   82 */
   { 0,			Ukn	ITR(1, "libdetach",	"")	}, /*   83 */
   { svr4_sysfs,	3	ITR(0, "sysfs",		"dxx")	}, /*   84 */
   { svr4_getmsg,	Spl	ITR(0, "getmsg",	"dxxx")	}, /*   85 */
   { svr4_putmsg,	Spl	ITR(0, "putmsg",	"dxxd")	}, /*   86 */
   { SC(poll),		-3	ITR(0, "poll",		"xdd")	}, /*   87 */
   { abi_lstat,		2	ITR(0, "lstat",		"sp")	}, /*   88 */
   { SC(symlink),	-2	ITR(0, "symlink",	"ss")	}, /*   89 */
   { SC(readlink),	-3	ITR(0, "readlink",	"spd")	}, /*   90 */
   { sys_setgroups,	2	ITR(0, "setgroups",	"dp")	}, /*   91 */
   { sys_getgroups,	2	ITR(0, "getgroups",	"dp")	}, /*   92 */
   { SC(fchmod),	-2	ITR(0, "fchmod",	"do")	}, /*   93 */
   { SC(fchown),	-3	ITR(0, "fchown",	"ddd")	}, /*   94 */
   { abi_sigprocmask,	3	ITR(0, "sigprocmask",	"dxx")	}, /*   95 */
   { abi_sigsuspend,	Spl	ITR(0, "sigsuspend",	"x")	}, /*   96 */
   { 0,			2	ITR(1, "sigaltstack",	"xx")	}, /*   97 */
   { abi_sigaction,	3	ITR(0, "sigaction",	"dxx")	}, /*   98 */
   { svr4_sigpending,	2	ITR(1, "sigpending",	"dp")	}, /*   99 */
   { svr4_context,	Spl	ITR(0, "context",	"")	}, /*   100 */
   { 0,			Ukn	ITR(1, "evsys",		"")	}, /*   101 */
   { 0,			Ukn	ITR(1, "evtrapret",	"")	}, /*   102 */
   { abi_statvfs,	2	ITR(0, "statvfs",	"sp")	}, /*   103 */
   { abi_fstatvfs,	2	ITR(0, "fstatvfs",	"dp")	}, /*   104 */
   { iBCS_class_ISC,	Fast	ITR(0, "sysisc",	"")	}, /*   105 */
   { 0,			Ukn	ITR(1, "nfssys",	"")	}, /*   106 */
   { 0,			4	ITR(0, "waitid",	"ddxd")	}, /*   107 */
   { 0,			3	ITR(1, "sigsendsys",	"ddd")	}, /*   108 */
   { svr4_hrtsys,	Spl	ITR(0, "hrtsys",	"xxx")	}, /*   109 */
   { 0,			3	ITR(1, "acancel",	"dxd")	}, /*   110 */
   { 0,			Ukn	ITR(1, "async",		"")	}, /*   111 */
   { 0,			Ukn	ITR(1, "priocntlsys",	"")	}, /*   112 */
   { svr4_pathconf,	2	ITR(1, "pathconf",	"sd")	}, /*   113 */
   { 0,			3	ITR(1, "mincore",	"xdx")	}, /*   114 */
   { svr4_mmap,		6	ITR(0, "mmap",		"xxxxdx") },/*   115 */
   { SC(mprotect),	-3	ITR(0, "mprotect",	"xdx")  },/*   116 */
   { SC(munmap),	-2	ITR(0, "munmap",	"xd")   },/*   117 */
   { svr4_fpathconf,	2	ITR(1, "fpathconf",	"dd")	}, /*   118 */
   { abi_fork,		Spl	ITR(0, "vfork",		"")	}, /*   119 */
   { SC(fchdir),	-1	ITR(0, "fchdir",	"d")	}, /*   120 */
   { SC(readv),		-3	ITR(0, "readv",		"dxd")	}, /*   121 */
   { SC(writev),	-3	ITR(0, "writev",	"dxd")	}, /*   122 */
   { svr4_xstat,	3	ITR(0, "xstat",		"dsx")	}, /*   123 */
   { svr4_lxstat,     	3	ITR(0, "lxstat",	"dsx")	}, /*   124 */
   { svr4_fxstat,	3	ITR(0, "fxstat",	"ddx")	}, /*   125 */
   { svr4_xmknod,	4	ITR(0, "xmknod",	"dsox")}, /*   126 */
   { 0,			Ukn	ITR(0, "syslocal",	"d")	}, /*   127 */
   { svr4_getrlimit,	2	ITR(0, "setrlimit",	"dx")	}, /*   128 */
   { svr4_setrlimit,	2	ITR(0, "getrlimit",	"dx")	}, /*   129 */
   { 0,			3	ITR(1, "lchown",	"sdd")	}, /*   130 */
   { 0,			Ukn	ITR(1, "memcntl",	"")	}, /*   131 */
#ifdef CONFIG_ABI_XTI
   { svr4_getpmsg,	5	ITR(0, "getpmsg",	"dxxxx")}, /*   132 */
   { svr4_putpmsg,	5	ITR(0, "putpmsg",	"dxxdd")}, /*   133 */
#else
   { 0,			5	ITR(0, "getpmsg",	"dxxxx")}, /*   132 */
   { 0,			5	ITR(0, "putpmsg",	"dxxdd")}, /*   133 */
#endif
   { SC(rename),	-2	ITR(0, "rename",	"ss")	}, /*   134 */
   { abi_utsname,	1	ITR(0, "uname",		"x")	}, /*   135 */
   { svr4_setegid,	1	ITR(1, "setegid",	"d")	}, /*   136 */
   { svr4_sysconfig,	1	ITR(0, "sysconfig",	"d")	}, /*   137 */
   { 0,			Ukn	ITR(1, "adjtime",	"")	}, /*   138 */
   { svr4_sysinfo,	3	ITR(0, "systeminfo",	"dsd")	}, /*   139 */
   { socksys_syscall,	1	ITR(0, "socksys_syscall","x")	}, /*   140 */
   { svr4_seteuid,	1	ITR(1, "seteuid",	"d")	}, /*   141 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   142 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*   143 */
   { 0,			2	ITR(1, "secsys",	"dx")	}, /*	144 */
   { 0,			4	ITR(1, "filepriv",	"sdxd")	}, /*	145 */
   { 0,			3	ITR(1, "procpriv",	"dxd")	}, /*	146 */
   { 0,			3	ITR(1, "devstat",	"sdx")	}, /*	147 */
   { 0,			5	ITR(1, "aclipc",	"ddddx")}, /*	148 */
   { 0,			3	ITR(1, "fdevstat",	"ddx")	}, /*	149 */
   { 0,			3	ITR(1, "flvlfile",	"ddx")	}, /*	150 */
   { 0,			3	ITR(1, "lvlfile",	"sdx")	}, /*	151 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*	152 */
   { 0,			2	ITR(1, "lvlequal",	"xx")	}, /*	153 */
   { 0,			2	ITR(1, "lvlproc",	"dx")	}, /*	154 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*	155 */
   { 0,			4	ITR(1, "lvlipc",	"dddx")	}, /*	156 */
   { 0,			4	ITR(1, "acl",		"sddx")	}, /*	157 */
   { 0,			Ukn	ITR(1, "auditevt",	"")	}, /*	158 */
   { 0,			Ukn	ITR(1, "auditctl",	"")	}, /*	159 */
   { 0,			Ukn	ITR(1, "auditdmp",	"")	}, /*	160 */
   { 0,			Ukn	ITR(1, "auditlog",	"")	}, /*	161 */
   { 0,			Ukn	ITR(1, "auditbuf",	"")	}, /*	162 */
   { 0,			2	ITR(1, "lvldom",	"xx")	}, /*	163 */
   { 0,			Ukn	ITR(1, "lvlvfs",	"")	}, /*	164 */
   { 0,			2	ITR(1, "mkmld",		"so")	}, /*	165 */
   { 0,			Ukn	ITR(1, "mlddone",	"")	}, /*	166 */
   { 0,			2	ITR(0, "secadvise",	"xx")	}, /*	167 */
   { 0,			Ukn	ITR(1, "online",	"")	}, /*	168 */
   { SC(setitimer),	-3	ITR(0, "setitimer",	"dxx")	}, /*	169 */
   { SC(getitimer),	-2	ITR(0, "getitimer",	"dx")	}, /*	170 */
   { SC(gettimeofday),	-2	ITR(0, "gettimeofday",	"xx")	}, /*	171 */
   { SC(settimeofday),	-2	ITR(0, "settimeofday",	"xx")	}, /*	172 */
   { 0,			Ukn	ITR(1, "lwpcreate",	"")	}, /*	173 */
   { 0,			Ukn	ITR(1, "lwpexit",	"")	}, /*	174 */
   { 0,			Ukn	ITR(1, "lwpwait",	"")	}, /*	175 */
   { 0,			Ukn	ITR(1, "lwpself",	"")	}, /*	176 */
   { 0,			Ukn	ITR(1, "lwpinfo",	"")	}, /*	177 */
   { 0,			Ukn	ITR(1, "lwpprivate",	"")	}, /*	178 */
   { 0,			Ukn	ITR(1, "processorbind",	"")	}, /*	179 */
   { 0,			Ukn	ITR(1, "processorexbind","")	}, /*	180 */
   { 0,			Ukn	ITR(1, "",		"")	}, /*	181 */
   { 0,			Ukn	ITR(1, "sync_mailbox",	"")	}, /*	182 */
   { 0,			Ukn	ITR(1, "prepblock",	"")	}, /*	183 */
   { 0,			Ukn	ITR(1, "block",		"")	}, /*	184 */
   { 0,			Ukn	ITR(1, "rdblock",	"")	}, /*	185 */
   { 0,			Ukn	ITR(1, "unblock",	"")	}, /*	186 */
   { 0,			Ukn	ITR(1, "cancelblock",	"")	}, /*	187 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*	188 */
   { 0,			Ukn	ITR(1, "pread",		"")	}, /*	189 */
   { 0,			Ukn	ITR(1, "pwrite",	"")	}, /*	190 */
   { SC(truncate),	-2	ITR(0, "truncate",	"sd")	}, /*	191 */
   { SC(ftruncate),	-2	ITR(0, "ftruncate",	"dd")	}, /*	192 */
   { 0,			Ukn	ITR(1, "lwpkill",	"")	}, /*	193 */
   { 0,			Ukn	ITR(1, "sigwait",	"")	}, /*	194 */
   { 0,			Ukn	ITR(1, "fork1",		"")	}, /*	195 */
   { 0,			Ukn	ITR(1, "forkall",	"")	}, /*	196 */
   { 0,			Ukn	ITR(1, "modload",	"")	}, /*	197 */
   { 0,			Ukn	ITR(1, "moduload",	"")	}, /*	198 */
   { 0,			Ukn	ITR(1, "modpath",	"")	}, /*	199 */
   { 0,			Ukn	ITR(1, "modstat",	"")	}, /*	200 */
   { 0,			Ukn	ITR(1, "modadm",	"")	}, /*	201 */
   { 0,			Ukn	ITR(1, "getksym",	"")	}, /*	202 */
   { 0,			Ukn	ITR(1, "lwpsuspend",	"")	}, /*	203 */
   { 0,			Ukn	ITR(1, "lwpcontinue",	"")	}, /*	204 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*	205 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*	206 */
   { 0,			Ukn	ITR(1, "?",		"")	}, /*	207 */
   { 0,			Ukn	ITR(1, "?",		"")	},
   { 0,			Ukn	ITR(1, "?",		"")	},
   { 0,			Ukn	ITR(1, "?",		"")	},
   { 0,			Ukn	ITR(1, "?",		"")	},
   { 0,			Ukn	ITR(1, "?",		"")	},
   { 0,			Ukn	ITR(1, "?",		"")	},
   { 0,			Fast	ITR(1, "?",		"")	},
   { 0,			Ukn	ITR(1, "?",		"")	}
};

static void
ibcs_lcall7(int segment, struct pt_regs *regp)
{
	abi_dispatch(regp, &ibcs_syscall_table[regp->eax & 0xff], 1);
}


long linux_to_ibcs_signals[NSIGNALS+1] = {
	0,
	IBCS_SIGHUP,	IBCS_SIGINT,	IBCS_SIGQUIT,	IBCS_SIGILL,
	IBCS_SIGTRAP,	IBCS_SIGABRT,	-1,		IBCS_SIGFPE,
	IBCS_SIGKILL,	IBCS_SIGUSR1,	IBCS_SIGSEGV,	IBCS_SIGUSR2,
	IBCS_SIGPIPE,	IBCS_SIGALRM,	IBCS_SIGTERM,	IBCS_SIGSEGV,
	IBCS_SIGCHLD,	IBCS_SIGCONT,	IBCS_SIGSTOP,	IBCS_SIGTSTP,
	IBCS_SIGTTIN,	IBCS_SIGTTOU,	IBCS_SIGURG,	IBCS_SIGGXCPU,
	IBCS_SIGGXFSZ,	IBCS_SIGVTALRM,	IBCS_SIGPROF,	IBCS_SIGWINCH,
	IBCS_SIGIO,	IBCS_SIGPWR,	-1,		-1
};

long ibcs_to_linux_signals[NSIGNALS+1] = {
	0,
	SIGHUP,		SIGINT,		SIGQUIT,	SIGILL,
	SIGTRAP,	SIGIOT,		SIGUNUSED,	SIGFPE,
	SIGKILL,	SIGUNUSED,	SIGSEGV,	SIGUNUSED,
	SIGPIPE,	SIGALRM,	SIGTERM,	SIGUSR1,
	SIGUSR2,	SIGCHLD,	SIGPWR,		SIGWINCH,
	SIGURG,		SIGPOLL,	SIGSTOP,	SIGTSTP,
	SIGCONT,	SIGTTIN,	SIGTTOU,	SIGVTALRM,
	SIGPROF,	SIGXCPU,	SIGXFSZ,	-1
};

#ifdef CONFIG_ABI_ISC
static long linux_to_isc_signals[NSIGNALS+1] = {
	0,
	IBCS_SIGHUP,	IBCS_SIGINT,	IBCS_SIGQUIT,	IBCS_SIGILL,
	IBCS_SIGTRAP,	IBCS_SIGABRT,	-1,		IBCS_SIGFPE,
	IBCS_SIGKILL,	IBCS_SIGUSR1,	IBCS_SIGSEGV,	IBCS_SIGUSR2,
	IBCS_SIGPIPE,	IBCS_SIGALRM,	IBCS_SIGTERM,	IBCS_SIGSEGV,
	IBCS_SIGCHLD,	ISC_SIGCONT,	ISC_SIGSTOP,	ISC_SIGTSTP,
	IBCS_SIGTTIN,	IBCS_SIGTTOU,	IBCS_SIGUSR1,	IBCS_SIGGXCPU,
	IBCS_SIGGXFSZ,	IBCS_SIGVTALRM,	IBCS_SIGPROF,	IBCS_SIGWINCH,
	IBCS_SIGIO,	IBCS_SIGPWR,	-1,		-1
};

static long isc_to_linux_signals[NSIGNALS+1] = {
	0,
	SIGHUP,		SIGINT,		SIGQUIT,	SIGILL,
	SIGTRAP,	SIGIOT,		SIGUNUSED,	SIGFPE,
	SIGKILL,	SIGUNUSED,	SIGSEGV,	SIGUNUSED,
	SIGPIPE,	SIGALRM,	SIGTERM,	SIGUSR1,
	SIGUSR2,	SIGCHLD,	SIGPWR,		SIGWINCH,
	-1,		SIGPOLL,	SIGCONT,	SIGSTOP,
	SIGTSTP,	SIGTTIN,	SIGTTOU,	SIGVTALRM,
	SIGPROF,	SIGXCPU,	SIGXFSZ,	-1
};

static struct exec_domain isc_exec_domain = {
	name:		"ISC",
	handler:	ibcs_lcall7,
	pers_low:	5 /* PER_ISCR4 */,
	pers_high:	5 /* PER_ISCR4 */,
	signal_map:	isc_to_linux_signals,
	signal_invmap:	linux_to_isc_signals,
	err_map:	svr4_err_map,
	socktype_map:	svr4_socktype_map,
	sockopt_map:	abi_sockopt_map,
	af_map:		abi_af_map,
	module:		THIS_MODULE
};
#endif /* CONFIG_ABI_ISC */

#ifdef CONFIG_ABI_XENIX
static long linux_to_xnx_signals[NSIGNALS+1] = {
	0,
	IBCS_SIGHUP,	IBCS_SIGINT,	IBCS_SIGQUIT,	IBCS_SIGILL,
	IBCS_SIGTRAP,	IBCS_SIGABRT,	-1,		IBCS_SIGFPE,
	IBCS_SIGKILL,	IBCS_SIGUSR1,	IBCS_SIGSEGV,	IBCS_SIGUSR2,
	IBCS_SIGPIPE,	IBCS_SIGALRM,	IBCS_SIGTERM,	IBCS_SIGSEGV,
	IBCS_SIGCHLD,	IBCS_SIGCONT,	IBCS_SIGSTOP,	IBCS_SIGTSTP,
	IBCS_SIGTTIN,	IBCS_SIGTTOU,	IBCS_SIGUSR1,	IBCS_SIGGXCPU,
	IBCS_SIGGXFSZ,	IBCS_SIGVTALRM,	IBCS_SIGPROF,	IBCS_SIGWINCH,
	20 /*XNX_SIGIO*/, IBCS_SIGPWR,	-1,		-1
};

static long xnx_to_linux_signals[NSIGNALS+1] = {
	0,
	SIGHUP,		SIGINT,		SIGQUIT,	SIGILL,
	SIGTRAP,	SIGIOT,		SIGUNUSED,	SIGFPE,
	SIGKILL,	SIGUNUSED,	SIGSEGV,	SIGUNUSED,
	SIGPIPE,	SIGALRM,	SIGTERM,	SIGUSR1,
	SIGUSR2,	SIGCHLD,	SIGPWR,		SIGPOLL,
	-1,		-1,		-1,		-1,
	-1,		-1,		-1,		-1,
	-1,		-1,		-1,		-1
};

static struct exec_domain xnx_exec_domain = {
	name:		"Xenix",
	handler:	ibcs_lcall7,
	pers_low:	7 /* PER_XENIX */,
	pers_high:	7 /* PER_XENIX */,
	signal_map:	xnx_to_linux_signals,
	signal_invmap:	linux_to_xnx_signals,
	err_map:	svr4_err_map,
	socktype_map:	svr4_socktype_map,
	sockopt_map:	abi_sockopt_map,
	af_map:		abi_af_map,
	module:		THIS_MODULE
};
#endif /* CONFIG_ABI_XENIX */

static struct exec_domain ibcs_exec_domain = {
	name:		"iBCS2/iABI",
	handler:	ibcs_lcall7,
	pers_low:	1 /* PER_SVR4 */,
	pers_high:	2 /* PER_SVR3 */,
	signal_map:	ibcs_to_linux_signals,
	signal_invmap:	linux_to_ibcs_signals,
	err_map:	svr4_err_map,
	socktype_map:	svr4_socktype_map,
	sockopt_map:	abi_sockopt_map,
	af_map:		abi_af_map,
	module:		THIS_MODULE
};


/*
 * XXX
 * XXX	A proper error handling is needed here, but with the current
 * XXX	structure it would be ifdef-mania...
 * XXX
 */
static int __init
ibcs_init(void)
{
	register_exec_domain(&ibcs_exec_domain);
#ifdef CONFIG_ABI_XENIX
	register_exec_domain(&xnx_exec_domain);
#endif
#ifdef CONFIG_ABI_ISC
	register_exec_domain(&isc_exec_domain);
#endif

	return 0;
}


static void __exit
ibcs_exit(void)
{
#ifdef CONFIG_ABI_ISC
	unregister_exec_domain(&isc_exec_domain);
#endif
#ifdef CONFIG_ABI_XENIX
	unregister_exec_domain(&xnx_exec_domain);
#endif
	unregister_exec_domain(&ibcs_exec_domain);
}

module_init(ibcs_init);
module_exit(ibcs_exit);
