/* 
 * Real-Time and Multimedia Systems Laboratory
 * Copyright (c) 2000 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Real-Time and Multimedia Systems Laboratory
 *  Attn: Prof. Raj Rajkumar
 *  Electrical and Computer Engineering, and Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 *  or via email to raj@ece.cmu.edu
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* Simple schuduling test for RKs....					*/
/*			By Luca Abeni					*/

/*
 * This program creates two resource sets RS1 and RS2, and attaches a cpu
 * reserve to each RS. The two cpu resereves RSV1 and RSV2 are created
 * using the parameters C1, T1, C2, and T2 defined below.
 * Two time consuming tasks are attached to the two resource sets, and
 * the tasks execution are tracked and registered in two files
 * res<pid1> and res<pid2>. The two files can be visulalized (to show the
 * tasks' schedule) using gnuplot:
 * gnuplot> plot "res<pid1>" w i
 * gnuplot> replot "res<pid2>" w i
 *
 * In order to synchronize tasks and reserves, the execution is assumed to
 * start at time 3000
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <wait.h>
#include <string.h>

#include <rk/rk.h>
#include <rk/rk_error.h>
#include <rk/posix_timers.h>
#include <rk/timespec.h>

/* Reserves Parameters:
 * edit them to change the CPU reserve set.
 */

/* Warning!!! This set of reserves can be guaranteed only using EDF!!!
 * If you run this program under the RM or DM scheduling policy, you will
 * obatin a ``Cannot create CPU reserve # 2'' error.
 * Use ``rksetschedulingpolicy 2''
 */
#define C1 25 
#define T1 100
#define C2 55
#define T2 125
#define C3 15
#define T3 250

/* Number of iteration per task */
#define ITER1 500
#define ITER2 500
#define ITER3 500

unsigned long int timevect[10000];

/* Program start time */
unsigned long int firstime;
struct timespec fnow;


/* Destroy the resource sets... end exit!!! */
void shutdown(rk_resource_set_t rs1, rk_resource_set_t rs2, rk_resource_set_t rs3)
{
    int res;

    res = rk_resource_set_destroy(rs1);
    if (res != 0) {
	printf("rk_resource_set_destroy failed destroying RS %p", rs1);
    }

    res = rk_resource_set_destroy(rs2);
    if (res != 0) {
	printf("rk_resource_set_destroy failed destroying RS %p", rs2);
    }

    res = rk_resource_set_destroy(rs3);
    if (res != 0) {
	printf("rk_resource_set_destroy failed destroying RS %p", rs2);
    }
    printf("Everithing is finished... I am going to exit!\n");
    exit(0);
}

/* This is the body of the time consuming tasks */
void childbody(int childnum)
{
    pid_t whoami;
    int i, j, count;
    struct timespec now;
    struct timespec per;
    char filename[256];
    FILE *f;
    int iter;

    iter = 500; // by default
    switch (childnum) {
    case 1: iter = ITER1; break;
    case 2: iter = ITER2; break;
    case 3: iter = ITER3; break;
    } 
    whoami = getpid();
    printf("Child %d starting\n", whoami);
    sprintf(filename, "res%d", whoami);
    f = fopen(filename, "w");

    /* Synchronize with the other child */
    //    fnow.tv_sec += 2;

    /*
     * The use of periodic tasks is a little tricky here...
     * The task becomes periodic only to wait for the start time.
     */
    rt_make_periodic(&per, &fnow);
    rt_wait_for_start_time();

    /* And this is the time-consuming loop... */
    count = 7000;
    for (i = 0; i < iter; i++) {
	for (j = 0; j < count; j++) {
	    clock_gettime(CLOCK_REALTIME, &now);
	    timevect[i] = now.tv_nsec / 1000000 + now.tv_sec * 1000;
	}
    }

    printf("Child %d saving data\n", whoami);
    for (i = 0; i < iter; i++) {
	fprintf(f, "%lu\t%d\n", timevect[i] - firstime, childnum);
    }

    fclose(f);

    printf("Child %d diing\n", whoami);
    exit(0);
}

void print_usage(char *cmdname)
{
    printf("usage: %s n (ms/sec) \n", cmdname);
}

int main(int argc, char *argv[])
{
    pid_t child1, child2, child3;
    rk_resource_set_t rs1, rs2, rs3;
    rk_reserve_t cpu_rsv1, cpu_rsv2, cpu_rsv3;
    rk_reserve_param_data_t p;
    struct timespec c, t;
    cpu_reserve_attr_data_t attr;
    char name[256];
    int res;

    /* create resource sets */
    sprintf(name, "RS 1");
    rs1 = rk_resource_set_create(name);
    if (((int) rs1 == -1) || (rs1 == NULL)) {
	printf("Cannot create resource set 1!!!\n");
	exit(-1);
    }
    sprintf(name, "RS 2");
    rs2 = rk_resource_set_create(name);
    if (((int) rs2 == -1) || (rs2 == NULL)) {
	printf("Cannot create resource set 2!!!\n");
	shutdown(rs1, 0, 0);
    }
    sprintf(name, "RS 3");
    rs3 = rk_resource_set_create(name);
    if (((int) rs3 == -1) || (rs2 == NULL)) {
	printf("Cannot create resource set 3!!!\n");
	shutdown(rs1, rs2, 0);
    }

    /* And now, the reserves... */
    clock_gettime(CLOCK_REALTIME, &fnow);
    firstime = fnow.tv_nsec / 1000000 + fnow.tv_sec * 1000;

    c.tv_sec = C1 / 1000, c.tv_nsec = (C1 % 1000) * 1000 * 1000;
    t.tv_sec = T1 / 1000, t.tv_nsec = (T1 % 1000) * 1000 * 1000;

    p.enf_mode = RSV_HARD;
//  p.enf_mode = RSV_SOFT;
    p.sch_mode = p.rep_mode = RSV_HARD;
    attr.compute_time = c;
    attr.period = t;
    attr.deadline = t;
    attr.blocking_time.tv_sec = 0;
    attr.blocking_time.tv_nsec = 0;
    memcpy(&attr.start_time, &fnow, sizeof(struct timespec));
    attr.start_time.tv_sec += 1;
    attr.reserve_type = p;

    cpu_rsv1 = rk_cpu_reserve_create(rs1, &attr);
    if (cpu_rsv1 == 0) {
	printf("Cannot create CPU reserve # 1\n");
	shutdown(rs1, rs2, rs3);
    }

    c.tv_sec = C2 / 1000, c.tv_nsec = (C2 % 1000) * 1000 * 1000;
    t.tv_sec = T2 / 1000, t.tv_nsec = (T2 % 1000) * 1000 * 1000;


    p.enf_mode = RSV_HARD;
//  p.enf_mode = RSV_SOFT;
    p.sch_mode = p.rep_mode = RSV_HARD;
    attr.compute_time = c;
    attr.period = t;
    attr.deadline = t;
    attr.blocking_time.tv_sec = 0;
    attr.blocking_time.tv_nsec = 0;
    memcpy(&attr.start_time, &fnow, sizeof(struct timespec));
    attr.start_time.tv_sec += 1;
    attr.reserve_type = p;

    cpu_rsv2 = rk_cpu_reserve_create(rs2, &attr);
    if (cpu_rsv2 == 0) {
	printf("Cannot create CPU reserve # 2\n");
	/* Well, we have created cpu resv #1, but it isn't attached to a
	   resource set yet... In order to destroy it, we need to wait that
	   it is attached to its RS!!!
	 */
	while (rk_resource_set_get_cpu_rsv(rs1) == 0);
	shutdown(rs1, rs2, rs3);
    }

    c.tv_sec = C3 / 1000, c.tv_nsec = (C3 % 1000) * 1000 * 1000;
    t.tv_sec = T3 / 1000, t.tv_nsec = (T3 % 1000) * 1000 * 1000;


    p.enf_mode = RSV_HARD;
//  p.enf_mode = RSV_SOFT;
    p.sch_mode = p.rep_mode = RSV_HARD;
    attr.compute_time = c;
    attr.period = t;
    attr.deadline = t;
    attr.blocking_time.tv_sec = 0;
    attr.blocking_time.tv_nsec = 0;
    memcpy(&attr.start_time, &fnow, sizeof(struct timespec));
    attr.start_time.tv_sec += 1;
    attr.reserve_type = p;

    cpu_rsv3 = rk_cpu_reserve_create(rs3, &attr);
    if (cpu_rsv3 == 0) {
	printf("Cannot create CPU reserve # 3\n");
	/* Well, we have created cpu resv #1 and #2 but they aren't attached 
	   to aresource set yet... In order to destroy it, we need to wait 
	   until they are attached to its RS!!!
	 */
	while (rk_resource_set_get_cpu_rsv(rs1) == 0);
	while (rk_resource_set_get_cpu_rsv(rs2) == 0);
	shutdown(rs1, rs2, rs3);
    }


    /* Let's wait that the reserves are attached to their resource sets */
    while (rk_resource_set_get_cpu_rsv(rs1) == 0);
    while (rk_resource_set_get_cpu_rsv(rs2) == 0);
    while (rk_resource_set_get_cpu_rsv(rs3) == 0);

    /* Create child 1 & attach it to RS 1 */
    child1 = fork();
    if (child1 < 0) {
	perror("Error in fork()");
	shutdown(rs1, rs2, rs3);
    }
    if (child1 == 0) {
	childbody(1);

	/* The child exits, so execution cannot arrive here */
	printf("ARRRG!!!! Why am I here?\n");
	exit(-1);
    }


    /* Create child 2 & attach it to RS 2 */
    child2 = fork();
    if (child2 < 0) {
	perror("Error in fork()");
	shutdown(rs1, rs2, rs3);
    }
    if (child2 == 0) {
	childbody(2);

	/* The child exits, so execution cannot arrive here */
	printf("ARRRG!!!! Why am I here?\n");
	exit(-1);
    }

    /* Create child 3 & attach it to RS 3 */
    child3 = fork();
    if (child3 < 0) {
	perror("Error in fork()");
	shutdown(rs1, rs2, rs3);
    }
    if (child3 == 0) {
	childbody(3);

	/* The child exits, so execution cannot arrive here */
	printf("ARRRG!!!! Why am I here?\n");
	exit(-1);
    }


    printf("Now, attaching the resource sets to children \n");


    res = rk_resource_set_attach_process(rs1, child1);
    if (res != 0) {
	printf
	    ("rk_resource_set_attach_proces failed for Resource Set 1...\n");
	shutdown(rs1, rs2, rs3);
    }

    res = rk_resource_set_attach_process(rs2, child2);
    if (res != 0) {
	printf
	    ("rk_resource_set_attach_proces failed for Resource Set 2...\n");
	shutdown(rs1, rs2, rs3);
    }

    res = rk_resource_set_attach_process(rs3, child3);
    if (res != 0) {
	printf
	    ("rk_resource_set_attach_proces failed for Resource Set 3...\n");
	shutdown(rs1, rs2, rs3);
    }

    printf("Done...\n");

    /* And now, as a good father, wait for the children to die... */
    wait(&res);
    wait(&res);
    wait(&res);

    /* ...and destroy the resource sets!!! */
    shutdown(rs1, rs2, rs3);
    return 0;
}
