/*
 * OHCI HCD (Host Controller Driver) for USB.
 *
 * (C) Copyright 1999 Roman Weissgaerber <weissg@vienna.at>
 * (C) Copyright 2000-2002 David Brownell <dbrownell@users.sourceforge.net>
 * (C) Copyright 2002 Hewlett-Packard Company
 * (C) Copyright 2004 Intel Company
 *
 * PXA Bus Glue
 *
 * Written by Christopher Hoover <ch@hpl.hp.com>
 * Based on fragments of previous driver by Rusell King et al.
 *
 * Modified for PXA27x from ohci-sa1111.c by Intel.
 *
 * This file is licenced under the GPL.
 */

#include <asm/hardware.h>
#include <asm/mach-types.h>
#include <asm/arch/pxa-regs.h>

extern int usb_disabled(void);

#define PMM_NPS_MODE	       1
#define PMM_GLOBAL_MODE        2
#define PMM_PERPORT_MODE       3

#define PXA_UHC_MAX_PORTNUM    3

static int     pxa_ohci_pmm_state;

/*-------------------------------------------------------------------------*/
static void pxa_hcd_set_gpio(void)
{
	pxa_gpio_mode(GPIO88_USBH1_PWR_MD);
	pxa_gpio_mode(GPIO89_USBH1_PEN_MD);
}

static int pxa_hcd_select_pmm( int mode )
{
	pxa_ohci_pmm_state = mode;
	switch ( mode ) {
	case PMM_NPS_MODE:
		UHCRHDA |= RH_A_NPS;
		break;
	case PMM_GLOBAL_MODE:
		UHCRHDA &= ~(RH_A_NPS & RH_A_PSM);
		break;
	case PMM_PERPORT_MODE:
		UHCRHDA &= ~(RH_A_NPS);
		UHCRHDA |= RH_A_PSM;

		/* Set port power control mask bits, only 3 ports. */
		UHCRHDB |= (0x7<<17);
		break;
	default:
		printk( KERN_ERR
			"Invalid mode %d, set to non-power switch mode.\n",
			mode );

		UHCRHDA |= RH_A_NPS;
	}

	return 0;
}

static int pxa_hcd_set_port_power( int port )
{
	if ((pxa_ohci_pmm_state == PMM_PERPORT_MODE)
	     && (port > 0) && (port < PXA_UHC_MAX_PORTNUM) ) {
		UHCRHPS(port) |= 0x100;
		return 0;
	}
	return -1;
}

/*
    If you select PMM_PERPORT_MODE, you should set the port power
*/
static int pxa_hcd_clear_port_power( int port )
{
	if ((pxa_ohci_pmm_state == PMM_PERPORT_MODE)
	     && (port > 0) && (port < PXA_UHC_MAX_PORTNUM) ) {
		UHCRHPS(port) |= 0x200;
		return 0;
	}

	return -1;
}

static void pxa_start_hc(struct platform_device *dev)
{
	printk(KERN_DEBUG"OHCI_PXA: starting PXA OHCI USB Controller\n");

	pxa_hcd_set_gpio();

	CKEN |= CKEN10_USBHOST;

	UHCHR |= UHCHR_FHR;
	udelay(11);
	UHCHR &= ~UHCHR_FHR;

	UHCHR |= UHCHR_FSBIR;
	while ( UHCHR & UHCHR_FSBIR )
		;

	if (machine_is_mainstone() || machine_is_glencoe())
		UHCHR |= (UHCHR_PCPL | UHCHR_PSPL);	// Set Signal Polarity to Active Low

	UHCHR &= ~UHCHR_SSEP3;
	UHCHR &= ~UHCHR_SSEP2;
	UHCHR &= ~UHCHR_SSEP1;
	UHCHR &= ~UHCHR_SSE;
}

static void pxa_stop_hc(struct platform_device *dev)
{
	printk(KERN_DEBUG"OHCI_PXA: stopping PXA OHCI USB Controller\n");

	/*
	 * Put the USB host controller into reset.
	 */
	UHCHR |= (UHCHR_FHR|UHCHR_FSBIR);

	/*
	 * Stop the USB clock.
	 */
	CKEN &= ~CKEN10_USBHOST;
}

/*-------------------------------------------------------------------------*/
static irqreturn_t usb_hcd_pxa_hcim_irq (int irq, void *__hcd, struct pt_regs * r)
{
	struct usb_hcd *hcd = __hcd;

	return usb_hcd_irq(irq, hcd, r);
}

/*-------------------------------------------------------------------------*/

void usb_hcd_pxa_remove (struct usb_hcd *, struct platform_device *);

/* configure so an HC device and id are always provided */
/* always called with process context; sleeping is OK */


/**
 * usb_hcd_pxa_probe - initialize PXA-based HCDs
 * Context: !in_interrupt()
 *
 * Allocates basic resources for this USB host controller, and
 * then invokes the start() method for the HCD associated with it
 * through the hotplug entry's driver_data.
 *
 * Store this function in the HCD's struct pci_driver as probe().
 */
int usb_hcd_pxa_probe (const struct hc_driver *driver,
			  struct usb_hcd **hcd_out,
			  struct platform_device *dev)
{
	int retval;
	struct usb_hcd *hcd = 0;

	if (!request_mem_region(dev->resource[0].start,
				dev->resource[0].end -
				dev->resource[0].start + 1, hcd_name)) {
		dbg("request_mem_region failed");
		return -EBUSY;
	}

	pxa_start_hc(dev);

	pxa_hcd_select_pmm( PMM_PERPORT_MODE );
	if ( pxa_hcd_set_port_power( 1 )<0 ) {
		printk(KERN_ERR "Setting port 1 power failed.\n");
	}

	if ( pxa_hcd_clear_port_power( 2 )<0 ) {
		printk(KERN_ERR "Setting port 2 power failed.\n");
	}
	
	if ( pxa_hcd_clear_port_power( 3 )<0 ) {
		printk(KERN_ERR "Setting port 3 power failed.\n");
	}

	hcd = driver->hcd_alloc ();
	if (hcd == NULL){
		dbg ("hcd_alloc failed");
		retval = -ENOMEM;
		goto err1;
	}

	hcd->driver = (struct hc_driver *) driver;
	hcd->description = driver->description;
	hcd->irq = dev->resource[1].start;
	hcd->regs = (void*) io_p2v(dev->resource[0].start);
	hcd->self.controller = &dev->dev;

	retval = hcd_buffer_create (hcd);
	if (retval != 0) {
		dbg ("pool alloc fail");
		goto err1;
	}
	
	retval = request_irq (hcd->irq, usb_hcd_pxa_hcim_irq, SA_INTERRUPT,
			      hcd->description, hcd);
	if (retval != 0) {
		dbg("request_irq failed");
		retval = -EBUSY;
		goto err2;
	}

	info ("%s (PXA) at 0x%p, irq %d\n",
	      hcd->description, hcd->regs, hcd->irq);

	usb_bus_init (&hcd->self);
	hcd->self.op = &usb_hcd_operations;
	hcd->self.hcpriv = (void *) hcd;
	hcd->self.bus_name = "pxa27x";
	hcd->product_desc = "PXA27x OHCI";

	INIT_LIST_HEAD (&hcd->dev_list);

	usb_register_bus (&hcd->self);

	if ((retval = driver->start (hcd)) < 0)
	{
		usb_hcd_pxa_remove(hcd, dev);
		return retval;
	}

	*hcd_out = hcd;
	return 0;

 err2:
	hcd_buffer_destroy (hcd);
	if (hcd)
		driver->hcd_free(hcd);
 err1:
	pxa_stop_hc(dev);
	release_mem_region(dev->resource[0].start,
			   dev->resource[0].end - dev->resource[0].start + 1);
	return retval;
}


/* may be called without controller electrically present */
/* may be called with controller, bus, and devices active */

/**
 * usb_hcd_pxa_remove - shutdown processing for PXA-based HCDs
 * @dev: USB Host Controller being removed
 * Context: !in_interrupt()
 *
 * Reverses the effect of usb_hcd_pxa_probe(), first invoking
 * the HCD's stop() method.  It is always called from a thread
 * context, normally "rmmod", "apmd", or something similar.
 *
 */
void usb_hcd_pxa_remove (struct usb_hcd *hcd, struct platform_device *dev)
{
	void *base;
	info ("remove: %s, state %x", hcd->self.bus_name, hcd->state);

	if (in_interrupt ())
		BUG ();

	hcd->state = USB_STATE_QUIESCING;

	dbg ("%s: roothub graceful disconnect", hcd->self.bus_name);
	usb_disconnect (&hcd->self.root_hub);

	hcd->driver->stop (hcd);
	hcd->state = USB_STATE_HALT;

	free_irq (hcd->irq, hcd);
	hcd_buffer_destroy (hcd);

	usb_deregister_bus (&hcd->self);

	base = hcd->regs;
	hcd->driver->hcd_free (hcd);

	pxa_stop_hc(dev);
	release_mem_region(dev->resource[0].start,
			   dev->resource[0].end - dev->resource[0].start + 1);
}

/*-------------------------------------------------------------------------*/

static int __devinit
ohci_pxa_start (struct usb_hcd *hcd)
{
	struct ohci_hcd *ohci = hcd_to_ohci (hcd);
	int		ret;

	ohci->hcca = dma_alloc_coherent (hcd->self.controller,
			sizeof *ohci->hcca, &ohci->hcca_dma, 0);
	if (!ohci->hcca)
		return -ENOMEM;

	memset (ohci->hcca, 0, sizeof (struct ohci_hcca));
	if ((ret = ohci_mem_init (ohci)) < 0) {
		ohci_stop (hcd);
		return ret;
	}
	ohci->regs = hcd->regs;

	if (hc_reset (ohci) < 0) {
		ohci_stop (hcd);
		return -ENODEV;
	}

	if (hc_start (ohci) < 0) {
		err ("can't start %s", ohci->hcd.self.bus_name);
		ohci_stop (hcd);
		return -EBUSY;
	}
	create_debug_files (ohci);

#ifdef	DEBUG
	ohci_dump (ohci, 1);
#endif
	return 0;
}

/*-------------------------------------------------------------------------*/

static const struct hc_driver ohci_pxa_hc_driver = {
	.description =		hcd_name,

	/*
	 * generic hardware linkage
	 */
	.irq =			ohci_irq,
	.flags =		HCD_USB11,

	/*
	 * basic lifecycle operations
	 */
	.start =		ohci_pxa_start,
#ifdef	CONFIG_PM
	/* suspend:		ohci_pxa_suspend,  -- tbd */
	/* resume:		ohci_pxa_resume,   -- tbd */
#endif
	.stop =			ohci_stop,

	/*
	 * memory lifecycle (except per-request)
	 */
	.hcd_alloc =		ohci_hcd_alloc,
	.hcd_free =		ohci_hcd_free,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue =		ohci_urb_enqueue,
	.urb_dequeue =		ohci_urb_dequeue,
	.endpoint_disable =	ohci_endpoint_disable,

	/*
	 * scheduling support
	 */
	.get_frame_number =	ohci_get_frame,

	/*
	 * root hub support
	 */
	.hub_status_data =	ohci_hub_status_data,
	.hub_control =		ohci_hub_control,
#ifdef	CONFIG_USB_SUSPEND
	.hub_suspend =		ohci_hub_suspend,
	.hub_resume =		ohci_hub_resume,
#endif
};

/*-------------------------------------------------------------------------*/

static int ohci_hcd_pxa_drv_probe(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct usb_hcd *hcd = NULL;
	int ret;

	if (usb_disabled())
		return -ENODEV;

	ret = usb_hcd_pxa_probe(&ohci_pxa_hc_driver, &hcd, pdev);

	if (ret == 0)
		dev_set_drvdata(dev, hcd);

	return ret;
}

static int ohci_hcd_pxa_drv_remove(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct usb_hcd *hcd = dev_get_drvdata(dev);

	usb_hcd_pxa_remove(hcd, pdev);

	dev_set_drvdata(dev, NULL);

	return 0;
}

static int ohci_hcd_pxa_drv_suspend(struct device * dev, u32 state, u32 level)
{
//	printk(KERN_ERR"state:%d, level:%d\n", state, level);
	if ( level == SUSPEND_POWER_DOWN)
		pxa_stop_hc(NULL);
	return 0;
}

static int ohci_hcd_pxa_drv_resume(struct device * dev, u32 level)
{
//	printk(KERN_ERR"level:%d\n", level);
	if (level == RESUME_POWER_ON) {
		struct usb_hcd *hcd = dev_get_drvdata(dev);
		struct ohci_hcd * ohci = hcd_to_ohci(hcd);

		pxa_start_hc(NULL);
		if (hc_reset (ohci) < 0) {
			ohci_stop (hcd);
			return -ENODEV;
		}

		if (hc_start (ohci) < 0) {
			err ("can't start %s", ohci->hcd.self.bus_name);
			ohci_stop (hcd);
			return -EBUSY;
		}
	}
	return 0;
}

static u64 ohci_hcd_pxa_dmamask = 0xffffffffUL;

static struct resource pxa_ohci_resources[] = {
	[0] = {
		.start= 0x4C000000,
		.end= 0x4C000fff,
		.flags= IORESOURCE_MEM,
	},
	[1] = {
		.start= IRQ_USBH1,
		.end= IRQ_USBH1,
		.flags= IORESOURCE_IRQ,
	},
};

static void ohci_hcd_pxa_device_release(struct device *dev)
{
	/* Keep this function empty. */
}

static struct platform_device ohci_hcd_pxa_device = {
	.name = "pxa-ohci",
	.id = -1,
	.dev		= {
		.dma_mask = &ohci_hcd_pxa_dmamask,
		.coherent_dma_mask = 0xffffffff,
		.release = ohci_hcd_pxa_device_release,
	},

	.num_resources = ARRAY_SIZE(pxa_ohci_resources),
	.resource      = pxa_ohci_resources,
};

static struct device_driver ohci_hcd_pxa_driver = {
	.name		= "pxa-ohci",
	.bus		= &platform_bus_type,
	.probe		= ohci_hcd_pxa_drv_probe,
	.remove		= ohci_hcd_pxa_drv_remove,
	.suspend	= ohci_hcd_pxa_drv_suspend,
	.resume		= ohci_hcd_pxa_drv_resume,
};

static int __init ohci_hcd_pxa_init (void)
{
	int ret;

	dbg (DRIVER_INFO " (PXA-27x)");
	dbg ("block sizes: ed %d td %d",
		sizeof (struct ed), sizeof (struct td));

	ret = platform_device_register(&ohci_hcd_pxa_device);
	if (!ret) {
		ret = driver_register(&ohci_hcd_pxa_driver);
		if (ret)
			platform_device_unregister(&ohci_hcd_pxa_device);
	}

	return ret;
}
static void __exit ohci_hcd_pxa_cleanup (void)
{
	driver_unregister(&ohci_hcd_pxa_driver);
	platform_device_unregister(&ohci_hcd_pxa_device);
}

module_init (ohci_hcd_pxa_init);
module_exit (ohci_hcd_pxa_cleanup);
