/*
 * xircom_gem.c: adds entries to /proc fs to control Xircom/Intel modem
 * - write 1 to assert, delay 250ms, and deassert pin
 * - write 2 to assert pin, write 0 to deassert pin
 *
  *  Copyright (c) 2005 Matthias Ihmig <m.ihmig@gmx.net>
 *
 * This program is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more
 * details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/delay.h>
#include <asm/uaccess.h>

#include <asm/hardware.h>
#include <asm/arch/pxa-regs.h>
#include <asm/arch/glencoe.h>

#define MODULE_VERS "1.0"
#define MODULE_NAME "xircom_gem"

static struct proc_dir_entry *gem_dir, *asleep, *power_down, *wake_up, *reset;

static int proc_read_gem(char *page, char **start, off_t off, int count,
			     int *eof, void *data)
{
	return sprintf(page, "%s\n", (GPIO_lev( (int)data) ? "0" : "1")); // active low signal -> invert
}

static int proc_write_gem(struct file *file, const char *buffer,
			     unsigned long count, void *data)
{
	int len=0;
	char value='0';

	if(count >= 1)
		len = 1;
	else
		return -EFAULT;

	if(copy_from_user(&value, buffer, len))
		return -EFAULT;

	if (value == '1' || value == '2')
		GPIO_clr( (int)data );		// assert pin (active low)
	if (value == '1')
		mdelay(250);			// wait 250ms when value=1
	if (value == '1' || value == '0')
		GPIO_set( (int)data );		// deassert pin
	return len;
}

static int __init init_xircom_gem(void)
{
	int rv = 0;
	
	// Initialize GPIO registers
	pxa_gpio_mode(GLN_GPIO_nGEM_WAKE  | GPIO_OUT);
	pxa_gpio_mode(GLN_GPIO_nGEM_SLEEP | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_nGEM_RST   | GPIO_OUT);
	pxa_gpio_mode(GLN_GPIO_nGEM_PWR_DWN|GPIO_OUT);
	
	/* create directory */
	gem_dir = proc_mkdir("driver/" MODULE_NAME, NULL);
	if(gem_dir == NULL) {
		rv = -ENOMEM; goto out;
	}
	gem_dir->owner = THIS_MODULE;
	
	asleep = create_proc_read_entry("asleep", 0444, gem_dir, proc_read_gem, (void*)GLN_GPIO_nGEM_SLEEP);
	if(asleep == NULL) {
		rv  = -ENOMEM; goto no_asleep;
	}
	asleep->owner = THIS_MODULE;

	
	power_down = create_proc_entry("power_down", 0644, gem_dir);
	if(power_down == NULL) {
		rv = -ENOMEM;
		goto no_power_down;
	}
	power_down->data = (void*)GLN_GPIO_nGEM_PWR_DWN;
	power_down->read_proc = proc_read_gem;
	power_down->write_proc = proc_write_gem;
	power_down->owner = THIS_MODULE;

	
	wake_up = create_proc_entry("wake_up", 0644, gem_dir);
	if(wake_up == NULL) {
		rv = -ENOMEM;
		goto no_wake_up;
	}
	wake_up->data = (void*)GLN_GPIO_nGEM_WAKE;
	wake_up->read_proc = proc_read_gem;
	wake_up->write_proc = proc_write_gem;
	wake_up->owner = THIS_MODULE;
		
	
	reset = create_proc_entry("reset", 0644, gem_dir);
	if(reset == NULL) {
		rv = -ENOMEM;
		goto no_reset;
	}
	reset->data = (void*)GLN_GPIO_nGEM_RST;
	reset->read_proc = proc_read_gem;
	reset->write_proc = proc_write_gem;
	reset->owner = THIS_MODULE;
	
	/* everything OK */
	printk(KERN_INFO "%s %s proc interface initialised\n",
	       MODULE_NAME, MODULE_VERS);
	return 0;

no_reset:
	remove_proc_entry("wake_up", gem_dir);
no_wake_up:
	remove_proc_entry("power_down", gem_dir);
no_power_down:
	remove_proc_entry("asleep", gem_dir);
no_asleep:			      
	remove_proc_entry(MODULE_NAME, NULL);
out:
	return rv;
}

static void __exit cleanup_xircom_gem(void)
{
	remove_proc_entry("power_down", gem_dir);
	remove_proc_entry("wake_up", gem_dir);
	remove_proc_entry("reset", gem_dir);
	remove_proc_entry("asleep", gem_dir);
	remove_proc_entry(MODULE_NAME, NULL);

	printk(KERN_INFO "%s %s interface removed\n",
	       MODULE_NAME, MODULE_VERS);
}

module_init(init_xircom_gem);
module_exit(cleanup_xircom_gem);

MODULE_AUTHOR("Matthias Ihmig <m.ihmig@mytum.de>");
MODULE_DESCRIPTION("Xircom/Intel GEM module /proc interface");
MODULE_LICENSE("GPL");

