/*
 * wm9713-touch.c  --  Touch screen driver for Wolfson WM9713 AC97 Codecs.
 *
 * Copyright 2003, 2004, 2005 Wolfson Microelectronics PLC.
 * Author: Liam Girdwood
 *         liam.girdwood@wolfsonmicro.com or linux@wolfsonmicro.com
 * Parts Copyright : Ian Molton <spyro@f2s.com>
 *                   Andrew Zabolotny <zap@homelink.ru>
 *                   Russell King <rmk@arm.linux.org.uk>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Notes:
 *
 *  Features:
 *       - polling mode
 *       - coordinate polling
 *       - continuous mode (arch-dependent)
 *       - adjustable rpu/dpp settings
 *       - adjustable pressure current
 *       - adjustable sample settle delay
 *       - pen down detection
 *       - battery monitor
 *       - sample AUX adc's
 *       - power management
 *       - codec GPIO
 *       - codec event notification
 *
 *  Revision history
 *    7th May 2003   Initial version.
 *    6th June 2003  Added non module support and AC97 registration.
 *   18th June 2003  Added AUX adc sampling.
 *   23rd June 2003  Did some minimal reformatting, fixed a couple of
 *                   locking bugs and noted a race to fix.
 *   24th June 2003  Added power management and fixed race condition.
 *   10th July 2003  Changed to a misc device.
 *   31st July 2003  Moved TS_EVENT and TS_CAL to wm97xx.h
 *    8th Aug  2003  Added option for read() calling wm97xx_sample_touch()
 *                   because some ac97_read/ac_97_write call schedule()
 *    7th Nov  2003  Added Input touch event interface, stanley.cai@intel.com
 *   13th Nov  2003  Removed h3600 touch interface, added interrupt based
 *                   pen down notification and implemented continous mode
 *                   on XScale arch.
 *   16th Nov  2003  Ian Molton <spyro@f2s.com>
 *                   Modified so that it suits the new 2.6 driver model.
 *   25th Jan  2004  Andrew Zabolotny <zap@homelink.ru>
 *                   Implemented IRQ-driven pen down detection, implemented
 *                   the private API meant to be exposed to platform-specific
 *                   drivers, reorganized the driver so that it supports
 *                   an arbitrary number of devices.
 *    1st Feb  2004  Moved continuous mode handling to a separate
 *                   architecture-dependent file. For now only PXA
 *                   built-in AC97 controller is supported (pxa-ac97-wm97xx.c).
 *    11th Feb 2004  Reduced CPU usage by keeping a cached copy of both
 *                   digitizer registers instead of reading them every time.
 *                   A reorganization of the whole code for better
 *                   error handling.
 *    17th Apr 2004  Added BMON support.
 *    17th Nov 2004  Added codec GPIO, codec event handling (real and virtual
 *                   GPIOs) and 2.6 power management. 
 *    29th Nov 2004  Added WM9713 support.
 *     1st Mar 2005  Changed from OSS plugin to ALSA audio bus device. 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/input.h>
#include <linux/delay.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/pm.h>
#include <linux/interrupt.h>
#include <linux/bitops.h>
#include <linux/workqueue.h>
#include <linux/device.h>
#include <linux/list.h>
#include <asm/uaccess.h>
#include <asm/io.h>

#include "wm97xx.h"
#include "mcp.h"

#define TS_NAME			"wm9713"
#define WM_TS_VERSION	"0.30"
#define DEFAULT_PRESSURE	0xb0c0

/*
 * Debug
 */
#if 0
#define dbg(format, arg...) printk(KERN_DEBUG TS_NAME ": " format "\n" , ## arg)
#else
#define dbg(format, arg...)
#endif
#define err(format, arg...) printk(KERN_ERR TS_NAME ": " format "\n" , ## arg)
#define info(format, arg...) printk(KERN_INFO TS_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) printk(KERN_WARNING TS_NAME ": " format "\n" , ## arg)

/*
 * Module parameters
 */

/*
 * Set the codec sample mode.
 *
 * The WM9713 can sample touchscreen data in 3 different operating
 * modes. i.e. polling, coordinate and continous.
 *
 * Polling:      The driver polls the codec and issues 3 seperate commands
 *               over the AC97 link to read X,Y and pressure.
 *
 * Coordinate:   The driver polls the codec and only issues 1 command over
 *               the AC97 link to read X,Y and pressure. This mode has
 *               strict timing requirements and may drop samples if
 *               interrupted. However, it is less demanding on the AC97
 *               link. Note: this mode requires a larger delay than polling
 *               mode.
 *
 * Continuous:   The codec automatically samples X,Y and pressure and then
 *               sends the data over the AC97 link in slowm-> This is the
 *               same method used by the codec when recording audio.
 *               This mode is not selectable by this parameter because
 *               it needs separate architecture-dependent support. The
 *               following drivers will add continuouse mode support:
 *
 *               pxa-wm97xx-touch.c - for Intel PXA built-in AC97 controller.
 */
static int mode;
module_param (mode, int, 0);
MODULE_PARM_DESC(mode, "WM97XX operation mode (0:polling 1:coordinate)");

/*
 * WM9713 - Set internal pull up for pen detect.
 *
 * Pull up is in the range 1.02k (least sensitive) to 64k (most sensitive)
 * i.e. pull up resistance = 64k Ohms / rpu.
 *
 * Adjust this value if you are having problems with pen detect not
 * detecting any down evenwm->
 */
static int rpu;
module_param(rpu, int, 0);
MODULE_PARM_DESC(rpu, "Set internal pull up resitor for pen detect.");


/*
 * Set current used for pressure measurement.
 *
 * Set pil = 2 to use 400uA
 *     pil = 1 to use 200uA and
 *     pil = 0 to disable pressure measurement.
 *
 * This is used to increase the range of values returned by the adc
 * when measureing touchpanel pressure.
 */
static int pil = 0;
module_param(pil, int, 0);
MODULE_PARM_DESC(pil, "Set current used for pressure measurement.");

/*
 * Set threshold for pressure measurement.
 *
 * Pen down pressure below threshold is ignored.
 */
static int pressure = DEFAULT_PRESSURE & 0xfff;
module_param(pressure, int, 0);
MODULE_PARM_DESC(pressure, "Set threshold for pressure measurement.");


/*
 * Set adc sample delay.
 *
 * For accurate touchpanel measurements, some settling time may be
 * required between the switch matrix applying a voltage across the
 * touchpanel plate and the ADC sampling the signal.
 *
 * This delay can be set by setting delay = n, where n is the array
 * position of the delay in the array delay_table below.
 * Long delays > 1ms are supported for completeness, but are not
 * recommended.
 */
static int delay = 4;
module_param(delay, int, 0);
MODULE_PARM_DESC(delay, "Set adc sample delay.");


/*
 * Touchscreen absolute values
 *
 * These parameters are used to help the input layer discard out of
 * range readings and reduce jitter etc. 
 * 
 *   o min, max:- indicate the min and max values your touch screen returns
 *   o fuzz:- use a higher number to reduce jitter
 *  	
 * The default values correspond to Mainstone II in QVGA mode
 *	
 * Please read 
 * Documentation/input/input-programming.txt for more details.	
 */

static int num_params;

static int abs_x[3] = {350,3900,5};
module_param_array(abs_x, int, num_params, 0);
MODULE_PARM_DESC(abs_x, "Touchscreen absolute X min, max, fuzz");

static int abs_y[3] = {320,3750,40};
module_param_array(abs_y, int, num_params, 0);
MODULE_PARM_DESC(abs_y, "Touchscreen absolute Y min, max, fuzz");

static int abs_p[3] = {0,150,4};
module_param_array(abs_p, int, num_params, 0);
MODULE_PARM_DESC(abs_p, "Touchscreen absolute Pressure min, max, fuzz");


/*
 * ADC sample delay times in uS
 */
static const int delay_table[] = {
	21,    // 1 AC97 Link frames
	42,    // 2
	84,    // 4
	167,   // 8
	333,   // 16
	667,   // 32
	1000,  // 48
	1333,  // 64
	2000,  // 96
	2667,  // 128
	3333,  // 160
	4000,  // 192
	4667,  // 224
	5333,  // 256
	6000,  // 288
	0      // No delay, switch matrix always on
};

/*
 * Delay after issuing a POLL command.
 *
 * The delay is 3 AC97 link frames + the touchpanel settling delay
 */
static inline void poll_delay(int d)
{
	udelay (3 * AC97_LINK_FRAME + delay_table [d]);
}



static void wm9713_digitiser_save(struct wm97xx* wm)
{
	wm->dig1 = mcp_reg_read(wm->mcp, AC97_WM9713_DIG1);
	wm->dig2 = mcp_reg_read(wm->mcp, AC97_WM9713_DIG2);
	wm->dig3 = mcp_reg_read(wm->mcp, AC97_WM9713_DIG3);
}


static void wm9713_digitiser_restore(struct wm97xx* wm)
{
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG1, wm->dig1);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG2, wm->dig2);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG3, wm->dig3);
}


static void wm9713_aux_prepare(struct wm97xx* wm)
{
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG1, 0);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG2, 0);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG3, WM97XX_PRP_DET_DIG);
}


static void wm9713_digitiser_start(struct wm97xx* wm)
{
	u16 val = mcp_reg_read(wm->mcp, AC97_EXTENDED_MID);
	mcp_reg_write(wm->mcp, AC97_EXTENDED_MID, val & 0x7fff);
	val = mcp_reg_read(wm->mcp, AC97_WM9713_DIG3);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG3, val | WM97XX_PRP_DET_DIG);
	mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD); /* dummy read */
}

static void wm9713_digitiser_stop(struct wm97xx* wm)
{
	u16 val = mcp_reg_read(wm->mcp, AC97_WM9713_DIG3);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG3, val & ~WM97XX_PRP_DET_DIG);
	val = mcp_reg_read(wm->mcp, AC97_EXTENDED_MID);
	mcp_reg_write(wm->mcp, AC97_EXTENDED_MID, val | 0x8000);
}

static inline int is_pden (struct wm97xx* wm)
{
	return wm->dig3 & WM9713_PDEN;
}

/*
 * Read a sample from the WM9713 adc in polling mode.
 */
static int wm9713_poll_sample (struct wm97xx* wm, int adcsel, int *sample)
{
	u16 dig1;

	int timeout = 5 * delay;

	if (!wm->pen_probably_down) {
		u16 data = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);
		if (!(data & WM97XX_PEN_DOWN))
			return RC_PENUP;
		wm->pen_probably_down = 1;
	}

	/* set up digitiser */
    if (adcsel & 0x8000)
        adcsel = 1 << ((adcsel & 0x7fff) + 4);
	dig1 = mcp_reg_read(wm->mcp, AC97_WM9713_DIG1);
	dig1 &= ~WM9713_ADCSEL_MASK; 
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG1, dig1 | adcsel |WM9713_POLL); 

	/* wait 3 AC97 time slots + delay for conversion */
	poll_delay(delay);

	/* wait for POLL to go low */
	while ((mcp_reg_read(wm->mcp, AC97_WM9713_DIG1) & WM9713_POLL) && timeout) {
		udelay(AC97_LINK_FRAME);
		timeout--;
	}
	
	if (timeout <= 0) {
		/* If PDEN is set, we can get a timeout when pen goes up */
		if (is_pden (wm))
			wm->pen_probably_down = 0;
		else {
			dbg ("adc sample timeout");
		}
		return RC_PENUP;
	}
	*sample = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);

	/* check we have correct sample */
	if ((*sample & WM97XX_ADCSRC_MASK) != generic_ffs(adcsel >> 1) << 12) {
		dbg ("adc wrong sample, read %x got %x", adcsel,
		     *sample & WM97XX_ADCSRC_MASK);
		return RC_PENUP;
	}

	if (!(*sample & WM97XX_PEN_DOWN)) {
		wm->pen_probably_down = 0;
		return RC_PENUP;
	}

	return RC_VALID;
}


/*
 * Read WM9713 sample data (x,y,[p]) from the adc in coordinate mode.
 */
static int wm9713_poll_coord_touch(struct wm97xx* wm, struct wm97xx_data *data)
{
	u16 dig1;
	int timeout = 5 * delay;

	if (!wm->pen_probably_down) {
		u16 data = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);
		if (!(data & WM97XX_PEN_DOWN))
			return RC_PENUP;
		wm->pen_probably_down = 1;
	}

	/* set up digitiser */
	dig1 = mcp_reg_read(wm->mcp, AC97_WM9713_DIG1);
	dig1 &= ~WM9713_ADCSEL_MASK; 
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG1, dig1 | WM9713_ADCSEL_PRES |
		WM9713_POLL | WM9713_COO); 

	/* wait 3 AC97 time slots + delay for conversion */
	poll_delay(delay);

	/* read X then wait for 1 AC97 link frame + settling delay */
	data->x = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);
	udelay (AC97_LINK_FRAME + delay_table[delay]);

	/* read Y */
	data->y = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);

	/* wait for POLL to go low and then read pressure */
	while ((mcp_reg_read(wm->mcp, AC97_WM9713_DIG1) & WM9713_POLL)&& timeout) {
			udelay(AC97_LINK_FRAME);
			timeout--;
	}

	if (timeout <= 0) {
		if (is_pden (wm))
			wm->pen_probably_down = 0;
		else {
			dbg ("adc sample timeout");
		}
		return RC_PENUP;
	}

	/* read pressure */
	data->p = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);

	/* check we have correct samples */
	if (((data->x & WM97XX_ADCSRC_MASK) != 0x1000) ||
	    ((data->y & WM97XX_ADCSRC_MASK) != 0x2000) ||
	    ((data->p & WM97XX_ADCSRC_MASK) != 0x3000)) {
		dbg ("wrong sample order");
		return RC_PENUP;
	}

	if (!(data->x & WM97XX_PEN_DOWN)) {
		wm->pen_probably_down = 0;
		return RC_PENUP;
	}

	return RC_VALID;
}


/*
 * Sample the WM9713 touchscreen in polling mode
 */
static int wm9713_poll_touch(struct wm97xx* wm, struct wm97xx_data *data)
{
	int rc;

	if ((rc = wm9713_poll_sample(wm, WM9713_ADCSEL_X, &data->x)) != RC_VALID)
		return rc;
	if ((rc = wm9713_poll_sample(wm, WM9713_ADCSEL_Y, &data->y)) != RC_VALID)
		return rc;
	if (pil) {
		if ((rc = wm9713_poll_sample(wm, WM9713_ADCSEL_PRES, &data->p)) != RC_VALID)
			return rc;
	} else
		data->p = DEFAULT_PRESSURE;

	return RC_VALID;
}

/*
 * Enable WM9713 continuous mode, i.e. touch data is streamed across an AC97 slot
 */
static void wm9713_init_continuous_mode (struct wm97xx* wm, int enable, int slot, int rate)
{
	u16 dig1, dig2, dig3;

	//spin_lock(&wm->lock);

	dig1 = wm->dig1;
	dig2 = wm->dig2;
	dig3 = wm->dig3;

	if (enable) {
		/* continous mode */
		dbg("Enabling continous mode");
		dig1 &= ~WM9713_ADCSEL_MASK;
        dig1 |= WM9713_CTC | WM9713_COO | WM9713_ADCSEL_X | WM9713_ADCSEL_Y; 
        if (pil)
				dig1 |= WM9713_ADCSEL_PRES;
        dig2 &= ~(WM97XX_DELAY_MASK | WM97XX_SLT_MASK  | WM97XX_CM_RATE_MASK);
        dig2 |= WM97XX_SLEN | WM97XX_DELAY (delay) | WM97XX_SLT (slot) | WM97XX_RATE (rate);
		dig3 |= WM9713_PDEN;
	} else {
		dig1 &= ~(WM97XX_CTC | WM97XX_COO);
		dig2 &= ~WM97XX_SLEN;
		if (mode == 1)
			/* coordinate mode */
			dig1 |= WM97XX_COO;
		dig3 &= ~WM9713_PDEN;
	}

	mcp_reg_write(wm->mcp, AC97_WM9713_DIG1, wm->dig1 = dig1);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG2, wm->dig2 = dig2);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG3, wm->dig3 = dig3);
	//spin_unlock (&wm->lock);
}

/*
 * set up the physical settings of the WM9713 
 */
static void init_wm9713_phy(struct wm97xx* wm)
{
	u16 dig1 = 0, dig2, dig3;
	
	/* default values */
	dig2 = WM97XX_DELAY(4) | WM97XX_SLT(5);
	dig3= WM9712_RPU(1);
      
	/* rpu */
	if (rpu) {
		dig3 &= 0xffc0;
		dig3 |= WM9712_RPU(rpu);
		info("setting pen detect pull-up to %d Ohms",64000 / rpu);
	}
      
	/* touchpanel pressure */
	if  (pil == 2) {
		dig3 |= WM9712_PIL;
		info("setting pressure measurement current to 400uA.");
	} else if (pil) 
		info ("setting pressure measurement current to 200uA.");
	if(!pil)
		pressure = 0;
      
	/* sample settling delay */
	if (delay!=4) {
		if (delay < 0 || delay > 15) {
			info ("supplied delay out of range.");
			delay = 4;
			info("setting adc sample delay to %d u Secs.", delay_table[delay]);
		}
	}
	dig2 &= 0xff0f;
	dig2 |= WM97XX_DELAY(delay);
      
	/* coordinate mode */
	if (mode == 1) {
		dig1 |= WM9713_COO;
		info("using coordinate mode");
	}

	mcp_reg_write(wm->mcp, AC97_WM9713_DIG1, wm->dig1 = dig1);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG2, wm->dig2 = dig2);
	mcp_reg_write(wm->mcp, AC97_WM9713_DIG3, wm->dig3 = dig3);
}

void wm97xx_set_continuous_mode (struct wm97xx *wm, codec_read_t read, int enable, int slot, int rate)
{
   wm->cont_read_sample = read;
   wm->wm97xx_codec->set_continuous_mode(wm, enable, slot, rate);
}
EXPORT_SYMBOL_GPL(wm97xx_set_continuous_mode);

static struct wm97xx_codec wm9713_codec = {
	.poll_sample = wm9713_poll_sample,
	.poll_coord =  wm9713_poll_coord_touch,
	.poll_touch = wm9713_poll_touch,
	.init_phy = init_wm9713_phy,
//	.codec_suspend = wm9713_suspend,
//	.codec_resume = wm9713_resume,
	.digitiser_save = wm9713_digitiser_save,
	.digitiser_restore = wm9713_digitiser_restore,
	.aux_prepare = wm9713_aux_prepare,
	.set_continuous_mode = wm9713_init_continuous_mode,
	.digitiser_start = wm9713_digitiser_start,
	.digitiser_stop = wm9713_digitiser_stop,
};


static int wm9713_ts_add(struct class_device *dev)
{
	u16 id1, id2;
	struct wm97xx *wm = classdev_to_wm97xx(dev);

	dbg("adding..\n");

	info("Wolfson WM9713 Touchscreen Controller driver");
	info("Version %s  liam.girdwood@wolfsonmicro.com", WM_TS_VERSION);

	/* test for a WM9713 */
	id1 = mcp_reg_read(wm->mcp, AC97_VENDOR_ID1);
	id2 = mcp_reg_read(wm->mcp, AC97_VENDOR_ID2);
	if (id1 != WM97XX_ID1 || id2 != WM9713_ID2) {
		err("could not find a WM9713 codec. Found a 0x%4x:0x%4x instead", id1, id2);
		return -ENODEV;
	}

	wm->wm97xx_codec = &wm9713_codec;
	memset (&wm->wm97xx_input, 0, sizeof (struct input_dev));
	wm->wm97xx_input.name = "wm9713 touchscreen";
	wm->wm97xx_input.open = wm97xx_ts_input_open;
	wm->wm97xx_input.close = wm97xx_ts_input_close;
	set_bit(EV_ABS, wm->wm97xx_input.evbit);
	set_bit(ABS_X, wm->wm97xx_input.absbit);
	set_bit(ABS_Y, wm->wm97xx_input.absbit);
	set_bit(ABS_PRESSURE, wm->wm97xx_input.absbit);
	wm->wm97xx_input.absmax[ABS_X] = abs_x[1];
	wm->wm97xx_input.absmax[ABS_Y] = abs_y[1];
	wm->wm97xx_input.absmax[ABS_PRESSURE] = abs_p[1];
	wm->wm97xx_input.absmin[ABS_X] = abs_x[0];
	wm->wm97xx_input.absmin[ABS_Y] = abs_y[0];
	wm->wm97xx_input.absmin[ABS_PRESSURE] = abs_p[0];
	wm->wm97xx_input.absfuzz[ABS_X] = abs_x[2];
	wm->wm97xx_input.absfuzz[ABS_Y] = abs_y[2];
	wm->wm97xx_input.absfuzz[ABS_PRESSURE] = abs_p[2];
	wm->wm97xx_input.private = wm;
	input_register_device(&wm->wm97xx_input);

	/* set up physical characteristics */
	init_wm9713_phy(wm);
	return 0;
}

static void wm9713_ts_remove(struct class_device *dev)
{
	struct wm97xx *wm = classdev_to_wm97xx(dev);

	input_unregister_device(&wm->wm97xx_input);
}

static struct class_interface wm9713_ts_interface = {
	.add		= wm9713_ts_add,
	.remove		= wm9713_ts_remove,
};

static int __init wm9713_ts_init(void)
{
	dbg("ts_init: register_interface..\n");
	return wm97xx_register_interface(&wm9713_ts_interface);
}

static void __exit wm9713_ts_exit(void)
{
	wm97xx_unregister_interface(&wm9713_ts_interface);
}

/* Module information */
MODULE_AUTHOR("Liam Girdwood, liam.girdwood@wolfsonmicro.com, www.wolfsonmicro.com");
MODULE_DESCRIPTION("WM9713 Touch Screen Driver");
MODULE_LICENSE("GPL");

module_init(wm9713_ts_init);
module_exit(wm9713_ts_exit);
