/*
 * wm9705-touch.c  --  Touch screen driver for Wolfson WM9713 AC97 Codecs.
 *
 * Copyright 2003, 2004, 2005 Wolfson Microelectronics PLC.
 * Author: Liam Girdwood
 *         liam.girdwood@wolfsonmicro.com or linux@wolfsonmicro.com
 * Parts Copyright : Ian Molton <spyro@f2s.com>
 *                   Andrew Zabolotny <zap@homelink.ru>
 *                   Russell King <rmk@arm.linux.org.uk>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Notes:
 *
 *  Features:
 *       - polling mode
 *       - coordinate polling
 *       - continuous mode (arch-dependent)
 *       - adjustable rpu/dpp settings
 *       - adjustable pressure current
 *       - adjustable sample settle delay
 *       - pen down detection
 *       - battery monitor
 *       - sample AUX adc's
 *       - power management
 *       - codec GPIO
 *       - codec event notification
 *
 *  Revision history
 *    7th May 2003   Initial version.
 *    6th June 2003  Added non module support and AC97 registration.
 *   18th June 2003  Added AUX adc sampling.
 *   23rd June 2003  Did some minimal reformatting, fixed a couple of
 *                   locking bugs and noted a race to fix.
 *   24th June 2003  Added power management and fixed race condition.
 *   10th July 2003  Changed to a misc device.
 *   31st July 2003  Moved TS_EVENT and TS_CAL to wm97xx.h
 *    8th Aug  2003  Added option for read() calling wm97xx_sample_touch()
 *                   because some ac97_read/ac_97_write call schedule()
 *    7th Nov  2003  Added Input touch event interface, stanley.cai@intel.com
 *   13th Nov  2003  Removed h3600 touch interface, added interrupt based
 *                   pen down notification and implemented continous mode
 *                   on XScale arch.
 *   16th Nov  2003  Ian Molton <spyro@f2s.com>
 *                   Modified so that it suits the new 2.6 driver model.
 *   25th Jan  2004  Andrew Zabolotny <zap@homelink.ru>
 *                   Implemented IRQ-driven pen down detection, implemented
 *                   the private API meant to be exposed to platform-specific
 *                   drivers, reorganized the driver so that it supports
 *                   an arbitrary number of devices.
 *    1st Feb  2004  Moved continuous mode handling to a separate
 *                   architecture-dependent file. For now only PXA
 *                   built-in AC97 controller is supported (pxa-ac97-wm97xx.c).
 *    11th Feb 2004  Reduced CPU usage by keeping a cached copy of both
 *                   digitizer registers instead of reading them every time.
 *                   A reorganization of the whole code for better
 *                   error handling.
 *    17th Apr 2004  Added BMON support.
 *    17th Nov 2004  Added codec GPIO, codec event handling (real and virtual
 *                   GPIOs) and 2.6 power management. 
 *    29th Nov 2004  Added WM9713 support.
 *     1st Mar 2005  Changed from OSS plugin to ALSA audio bus device. 
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/input.h>
#include <linux/delay.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/pm.h>
#include <linux/interrupt.h>
#include <linux/bitops.h>
#include <linux/workqueue.h>
#include <linux/device.h>
#include <linux/list.h>
#include <asm/uaccess.h>
#include <asm/io.h>

#include "wm97xx.h"
#include "mcp.h"

#define TS_NAME			"wm9705"
#define WM_TS_VERSION	"0.30"
#define DEFAULT_PRESSURE	0xb0c0

/*
 * Debug
 */
#if 1
#define dbg(format, arg...) printk(KERN_DEBUG TS_NAME ": " format "\n" , ## arg)
#else
#define dbg(format, arg...)
#endif
#define err(format, arg...) printk(KERN_ERR TS_NAME ": " format "\n" , ## arg)
#define info(format, arg...) printk(KERN_INFO TS_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) printk(KERN_WARNING TS_NAME ": " format "\n" , ## arg)

/*
 * Module parameters
 */

/*
 * Set the codec sample mode.
 *
 * The WM9713 can sample touchscreen data in 3 different operating
 * modes. i.e. polling, coordinate and continous.
 *
 * Polling:      The driver polls the codec and issues 3 seperate commands
 *               over the AC97 link to read X,Y and pressure.
 *
 * Coordinate:   The driver polls the codec and only issues 1 command over
 *               the AC97 link to read X,Y and pressure. This mode has
 *               strict timing requirements and may drop samples if
 *               interrupted. However, it is less demanding on the AC97
 *               link. Note: this mode requires a larger delay than polling
 *               mode.
 *
 * Continuous:   The codec automatically samples X,Y and pressure and then
 *               sends the data over the AC97 link in slowm-> This is the
 *               same method used by the codec when recording audio.
 *               This mode is not selectable by this parameter because
 *               it needs separate architecture-dependent support. The
 *               following drivers will add continuouse mode support:
 *
 *               pxa-wm97xx-touch.c - for Intel PXA built-in AC97 controller.
 */
static int mode;
module_param (mode, int, 0);
MODULE_PARM_DESC(mode, "WM97XX operation mode (0:polling 1:coordinate)");

/*
 * WM9713 - Set internal pull up for pen detect.
 *
 * Pull up is in the range 1.02k (least sensitive) to 64k (most sensitive)
 * i.e. pull up resistance = 64k Ohms / rpu.
 *
 * Adjust this value if you are having problems with pen detect not
 * detecting any down evenwm->
 */
static int rpu;
module_param(rpu, int, 0);
MODULE_PARM_DESC(rpu, "Set internal pull up resitor for pen detect.");


/*
 * Set current used for pressure measurement.
 *
 * Set pil = 2 to use 400uA
 *     pil = 1 to use 200uA and
 *     pil = 0 to disable pressure measurement.
 *
 * This is used to increase the range of values returned by the adc
 * when measureing touchpanel pressure.
 */
static int pil = 0;
module_param(pil, int, 0);
MODULE_PARM_DESC(pil, "Set current used for pressure measurement.");

/*
 * Set threshold for pressure measurement.
 *
 * Pen down pressure below threshold is ignored.
 */
static int pressure = DEFAULT_PRESSURE & 0xfff;
module_param(pressure, int, 0);
MODULE_PARM_DESC(pressure, "Set threshold for pressure measurement.");


/*
 * Set adc sample delay.
 *
 * For accurate touchpanel measurements, some settling time may be
 * required between the switch matrix applying a voltage across the
 * touchpanel plate and the ADC sampling the signal.
 *
 * This delay can be set by setting delay = n, where n is the array
 * position of the delay in the array delay_table below.
 * Long delays > 1ms are supported for completeness, but are not
 * recommended.
 */
static int delay = 4;
module_param(delay, int, 0);
MODULE_PARM_DESC(delay, "Set adc sample delay.");

/*
 * WM9705 - Pen detect comparator threshold.
 *
 * 0 to Vmid in 15 steps, 0 = use zero power comparator with Vmid threshold
 * i.e. 1 =  Vmid/15 threshold
 *      15 =  Vmid/1 threshold
 *
 * Adjust this value if you are having problems with pen detect not
 * detecting any down events.
 */
static int pdd = 8;
module_param(pdd, int, 0);
MODULE_PARM_DESC(pdd, "Set pen detect comparator threshold");

/*
 * WM9712 - Set five_wire = 1 to use a 5 wire touchscreen.
 *
 * NOTE: Five wire mode does not allow for readback of pressure.
 */
static int five_wire;
module_param(five_wire, int, 0);
MODULE_PARM_DESC(five_wire, "Set to '1' to use 5-wire touchscreen.");

/*
 * Touchscreen absolute values
 *
 * These parameters are used to help the input layer discard out of
 * range readings and reduce jitter etc. 
 * 
 *   o min, max:- indicate the min and max values your touch screen returns
 *   o fuzz:- use a higher number to reduce jitter
 *  	
 * The default values correspond to Mainstone II in QVGA mode
 *	
 * Please read 
 * Documentation/input/input-programming.txt for more details.	
 */
static int num_params;

static int abs_x[3] = {350,3900,5};
module_param_array(abs_x, int, num_params, 0);
MODULE_PARM_DESC(abs_x, "Touchscreen absolute X min, max, fuzz");

static int abs_y[3] = {320,3750,40};
module_param_array(abs_y, int, num_params, 0);
MODULE_PARM_DESC(abs_y, "Touchscreen absolute Y min, max, fuzz");

static int abs_p[3] = {0,150,4};
module_param_array(abs_p, int, num_params, 0);
MODULE_PARM_DESC(abs_p, "Touchscreen absolute Pressure min, max, fuzz");


/*
 * ADC sample delay times in uS
 */
static const int delay_table[] = {
	21,    // 1 AC97 Link frames
	42,    // 2
	84,    // 4
	167,   // 8
	333,   // 16
	667,   // 32
	1000,  // 48
	1333,  // 64
	2000,  // 96
	2667,  // 128
	3333,  // 160
	4000,  // 192
	4667,  // 224
	5333,  // 256
	6000,  // 288
	0      // No delay, switch matrix always on
};

/*
 * Delay after issuing a POLL command.
 *
 * The delay is 3 AC97 link frames + the touchpanel settling delay
 */
static inline void poll_delay(int d)
{
	udelay (3 * AC97_LINK_FRAME + delay_table [d]);
}



static void wm9705_digitiser_save(struct wm97xx* wm)
{
    wm->dig1 = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER1);
    wm->dig2 = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER2);
}


static void wm9705_digitiser_restore(struct wm97xx* wm)
{
    mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER1, wm->dig1);
   mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER2, wm->dig2);
}


static void wm9705_aux_prepare(struct wm97xx* wm)
{
mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER1, 0);
 mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER2, WM97XX_PRP_DET_DIG);
}


static void wm9705_digitiser_start(struct wm97xx* wm)
{
    u16 dig2 = wm->dig2 | WM97XX_PRP_DET_DIG;
   mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER2, wm->dig2 = dig2);
 mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD); /* dummy read */
}

static void wm9705_digitiser_stop(struct wm97xx* wm)
{
	u16 val = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER2);
 mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER2, val & ~WM97XX_PRP_DET_DIG);
}

static inline int is_pden (struct wm97xx* wm)
{
    int pden = 0;
	switch (wm->id) {
     case WM9705_ID2:
           pden = wm->dig2 & WM9705_PDEN;
          break;
     case WM9712_ID2:
           pden = wm->dig2 & WM9712_PDEN;
          break;
    }
    return pden;
}

/*
 * Read a sample from the WM9713 adc in polling mode.
 */
static int wm9705_poll_sample (struct wm97xx* wm, int adcsel, int *sample)
{
    int timeout = 5 * delay;

  if (!wm->pen_probably_down) {
       u16 data = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);
     if (!(data & WM97XX_PEN_DOWN))
         return RC_PENUP;
       wm->pen_probably_down = 1;
  }

 /* set up digitiser */
 mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER1, adcsel | WM97XX_POLL | WM97XX_DELAY(delay));

    /* wait 3 AC97 time slots + delay for conversion */
    poll_delay (delay);

   /* wait for POLL to go low */
  while ((mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER1) & WM97XX_POLL) && timeout) {
      udelay(AC97_LINK_FRAME);
       timeout--;
 }

 if (timeout <= 0) {
        /* If PDEN is set, we can get a timeout when pen goes up */
        if (is_pden(wm))
         wm->pen_probably_down = 0;
      else {
         dbg ("adc sample timeout");
        }
      return 0;
  }

 *sample = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);

 /* check we have correct sample */
 if ((*sample & WM97XX_ADCSEL_MASK) != adcsel) {
     dbg ("adc wrong sample, read %x got %x", adcsel,
            *sample & WM97XX_ADCSEL_MASK);
        return 0;
  }

 if (!(*sample & WM97XX_PEN_DOWN)) {
        wm->pen_probably_down = 0;
      return RC_PENUP;
   }

 return RC_VALID;
}


/*
 * Read WM9713 sample data (x,y,[p]) from the adc in coordinate mode.
 */
static int wm9705_poll_coord_touch(struct wm97xx* wm, struct wm97xx_data *data)
{
    u16 dig1;
  int timeout = 5 * delay;

  if (!wm->pen_probably_down) {
       u16 data = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);
     if (!(data & WM97XX_PEN_DOWN))
         return RC_PENUP;
       wm->pen_probably_down = 1;
  }

 /* set up digitiser */
 dig1 = (wm->dig1 & ~WM97XX_ADCSEL_MASK) | WM97XX_ADCSEL_PRES;
   mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER1,
             (wm->dig1 = dig1) | WM97XX_POLL);

   /* wait 3 AC97 time slots + delay for conversion */
    poll_delay(delay);

    /* read X then wait for 1 AC97 link frame + settling delay */
  data->x = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);
  udelay (AC97_LINK_FRAME + delay_table[delay]);

    /* read Y */
   data->y = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);

 /* wait for POLL to go low and then read pressure */
   while ((mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER1) & WM97XX_POLL)&& timeout) {
           udelay(AC97_LINK_FRAME);
           timeout--;
 }

 if (timeout <= 0) {
        if (is_pden(wm))
         wm->pen_probably_down = 0;
      else {

         dbg ("adc sample timeout");
        }
      return 0;
  }

 /* read pressure */
    data->p = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);

 /* check we have correct samples */
    if (((data->x & WM97XX_ADCSRC_MASK) != 0x1000) ||
      ((data->y & WM97XX_ADCSRC_MASK) != 0x2000) ||
      ((data->p & WM97XX_ADCSRC_MASK) != 0x3000)) {
     dbg ("wrong sample order");
        return 0;
  }

 if (!(data->x & WM97XX_PEN_DOWN)) {
        wm->pen_probably_down = 0;
      return RC_PENUP;
   }

 return RC_VALID;
}


/*
 * Sample the WM9713 touchscreen in polling mode
 */
static int wm9705_poll_touch(struct wm97xx* wm, struct wm97xx_data *data)
{
    int rc;

   if ((rc = wm9705_poll_sample(wm, WM97XX_ADCSEL_X, &data->x)) != RC_VALID)
      return rc;
 if ((rc = wm9705_poll_sample(wm, WM97XX_ADCSEL_Y, &data->y)) != RC_VALID)
      return rc;
 if (pil && !five_wire) {
       if ((rc = wm9705_poll_sample(wm, WM97XX_ADCSEL_PRES, &data->p)) != RC_VALID)
           return rc;
 } else
     data->p = DEFAULT_PRESSURE;

   return RC_VALID;
}

/*
 * Enable WM9713 continuous mode, i.e. touch data is streamed across an AC97 slot
 */
static void wm9705_init_continuous_mode (struct wm97xx* wm, int enable, int slot, int rate)
{
    u16 dig1, dig2;

  // spin_lock(&wm->lock);

  dig1 = wm->dig1;
    dig2 = wm->dig2;

   if (enable) {
      /* continous mode */
       dbg("Enabling continous mode");
        //wm->cont_read_sample = dev->cont_read;
      dig1 &= ~(WM97XX_CM_RATE_MASK | WM97XX_ADCSEL_MASK |
             WM97XX_DELAY_MASK | WM97XX_SLT_MASK);
                dig1 |= WM97XX_CTC | WM97XX_COO | WM97XX_SLEN |
            WM97XX_DELAY (delay) |
         WM97XX_SLT (slot) |
            WM97XX_RATE (rate);
            if (pil)
               dig1 |= WM97XX_ADCSEL_PRES;
            if (wm->id == WM9712_ID2)
                dig2 |= WM9712_PDEN;
           else
               dig2 |= WM9705_PDEN;
   } else {
       wm->cont_read_sample = NULL;
        dig1 &= ~(WM97XX_CTC | WM97XX_COO | WM97XX_SLEN);
      if (mode == 1)
         /* coordinate mode */
          dig1 |= WM97XX_COO;
        
       if (wm->id == WM9712_ID2)
                dig2 &= ~WM9712_PDEN;
          else
               dig2 &= ~WM9705_PDEN;
  }

 mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER1, wm->dig1 = dig1);
 mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER2, wm->dig2 = dig2);

   // spin_unlock(&wm->lock);
}

/*
 * set up the physical settings of the WM9705
 */
static void init_wm9705_phy(struct wm97xx* wm)
{
   wm->dig1 = 0;

  wm->dig2 = WM97XX_RPR;

 /*
  * mute VIDEO and AUX as they share X and Y touchscreen
     * inputs on the WM9705 
    */
    mcp_reg_write(wm->mcp, AC97_AUX, 0x8000);
   mcp_reg_write(wm->mcp, AC97_VIDEO, 0x8000);
      
  /* touchpanel pressure current*/
   if  (pil == 2) {
       wm->dig2 |= WM9705_PIL;
     dbg("setting pressure measurement current to 400uA.");
 } else if (pil) 
       dbg("setting pressure measurement current to 200uA.");
 if(!pil)
       pressure = 0;
      
   /* polling mode sample settling delay */
   if (delay!=4) {
        if (delay < 0 || delay > 15) {
         dbg("supplied delay out of range.");
           delay = 4;
     }
  }
  wm->dig1 &= 0xff0f;
 wm->dig1 |= WM97XX_DELAY(delay);
    dbg("setting adc sample delay to %d u Secs.", delay_table[delay]);
      
  /* coordinate mode */
  if (mode == 1) {
       wm->dig1 |= WM97XX_COO;
     dbg("using coordinate mode");
  }
      
   /* WM9705 pdd */
   wm->dig2 |= (pdd & 0x000f);
 dbg("setting pdd to Vmid/%d", 1 - (pdd & 0x000f));
      
  mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER1, wm->dig1);
    mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER2, wm->dig2);     
}

/*
 * set up the physical settings of the WM9712
 */
static void init_wm9712_phy(struct wm97xx* wm)
{
   wm->dig1 = 0;
   wm->dig2 = WM97XX_RPR | WM9712_RPU(1);
      
   /* WM9712 rpu */
   if (rpu) {
     wm->dig2 &= 0xffc0;
     wm->dig2 |= WM9712_RPU(rpu);
        dbg("setting pen detect pull-up to %d Ohms",64000 / rpu);
  }
      
   /* touchpanel pressure current*/
   if  (pil == 2) {
       wm->dig2 |= WM9712_PIL;
     dbg("setting pressure measurement current to 400uA.");
 } else if (pil) 
       dbg("setting pressure measurement current to 200uA.");
 if(!pil)
       pressure = 0;
      
   /* WM9712 five wire */
 if (five_wire) {
       wm->dig2 |= WM9712_45W;
     dbg("setting 5-wire touchscreen mode.");
   }
      
   /* polling mode sample settling delay */
   if (delay!=4) {
        if (delay < 0 || delay > 15) {
         dbg("supplied delay out of range.");
           delay = 4;
     }
  }
  wm->dig1 &= 0xff0f;
 wm->dig1 |= WM97XX_DELAY(delay);
    dbg("setting adc sample delay to %d u Secs.", delay_table[delay]);
      
  /* coordinate mode */
  if (mode == 1) {
       wm->dig1 |= WM97XX_COO;
     dbg("using coordinate mode");
  }
      
   mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER1, wm->dig1);
    mcp_reg_write(wm->mcp, AC97_WM97XX_DIGITISER2, wm->dig2);     
}

void wm97xx_set_continuous_mode (struct wm97xx *wm, codec_read_t read, int enable, int slot, int rate)
{
   wm->cont_read_sample = read;
   wm->wm97xx_codec->set_continuous_mode(wm, enable, slot, rate);
}
EXPORT_SYMBOL_GPL(wm97xx_set_continuous_mode);

static struct wm97xx_codec wm9705_codec = {
	.poll_sample = wm9705_poll_sample,
	.poll_coord =  wm9705_poll_coord_touch,
	.poll_touch = wm9705_poll_touch,
	.init_phy = init_wm9705_phy,
	.digitiser_save = wm9705_digitiser_save,
	.digitiser_restore = wm9705_digitiser_restore,
	.aux_prepare = wm9705_aux_prepare,
	.set_continuous_mode = wm9705_init_continuous_mode,
	.digitiser_start = wm9705_digitiser_start,
	.digitiser_stop = wm9705_digitiser_stop,
};


static int wm9705_ts_add(struct class_device *dev)
{
	u16 id1, id2;
	struct wm97xx *wm = classdev_to_wm97xx(dev);

	info("Wolfson WM9705/12 Touchscreen Controller driver");
	info("Version %s  liam.girdwood@wolfsonmicro.com", WM_TS_VERSION);

	/* test for a WM9705/12 */
	id1 = mcp_reg_read(wm->mcp, AC97_VENDOR_ID1);
	id2 = mcp_reg_read(wm->mcp, AC97_VENDOR_ID2);
	if (id1 != WM97XX_ID1 || (id2 != WM9705_ID2 && id2 != WM9712_ID2)) {
		err("could not find a WM9705/12 codec. Found a 0x%4x:0x%4x instead", id1, id2);
		return -ENODEV;
	}

	wm->wm97xx_codec = &wm9705_codec;
    if(id2 == WM9712_ID2)
        wm->wm97xx_codec->init_phy= init_wm9712_phy;
	memset (&wm->wm97xx_input, 0, sizeof (struct input_dev));
	wm->wm97xx_input.name = "wm9705 touchscreen";
	wm->wm97xx_input.open = wm97xx_ts_input_open;
	wm->wm97xx_input.close = wm97xx_ts_input_close;
	set_bit(EV_ABS, wm->wm97xx_input.evbit);
	set_bit(ABS_X, wm->wm97xx_input.absbit);
	set_bit(ABS_Y, wm->wm97xx_input.absbit);
	set_bit(ABS_PRESSURE, wm->wm97xx_input.absbit);
	wm->wm97xx_input.absmax[ABS_X] = abs_x[1];
	wm->wm97xx_input.absmax[ABS_Y] = abs_y[1];
	wm->wm97xx_input.absmax[ABS_PRESSURE] = abs_p[1];
	wm->wm97xx_input.absmin[ABS_X] = abs_x[0];
	wm->wm97xx_input.absmin[ABS_Y] = abs_y[0];
	wm->wm97xx_input.absmin[ABS_PRESSURE] = abs_p[0];
	wm->wm97xx_input.absfuzz[ABS_X] = abs_x[2];
	wm->wm97xx_input.absfuzz[ABS_Y] = abs_y[2];
	wm->wm97xx_input.absfuzz[ABS_PRESSURE] = abs_p[2];
	wm->wm97xx_input.private = wm;
	input_register_device(&wm->wm97xx_input);

	/* set up physical characteristics */
	init_wm9705_phy(wm);
	return 0;
}

static void wm9705_ts_remove(struct class_device *dev)
{
	struct wm97xx *wm = classdev_to_wm97xx(dev);

	input_unregister_device(&wm->wm97xx_input);
}

static struct class_interface wm9705_ts_interface = {
	.add		= wm9705_ts_add,
	.remove		= wm9705_ts_remove,
};

static int __init wm9705_ts_init(void)
{
	return wm97xx_register_interface(&wm9705_ts_interface);
}

static void __exit wm9705_ts_exit(void)
{
	wm97xx_unregister_interface(&wm9705_ts_interface);
}

/* Module information */
MODULE_AUTHOR("Liam Girdwood, liam.girdwood@wolfsonmicro.com, www.wolfsonmicro.com");
MODULE_DESCRIPTION("WM9705/12 Touch Screen Driver");
MODULE_LICENSE("GPL");

module_init(wm9705_ts_init);
module_exit(wm9705_ts_exit);
