/* 
    camera - header file for pxa camera driver

    Copyright (C) 2003, Intel Corporation

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/ 

#ifndef _CAMERA_H_
#define _CAMERA_H_
#include <linux/pxa_camera.h>

#define VID_HARDWARE_PXA_CAMERA 	50 /* subject to change */

#define STATUS_FAILURE	(0)
#define STATUS_SUCCESS	(1)
#define STATUS_WRONG_PARAMETER  -1



/*
Macros
*/
/*
Sensor type
*/
#define CAMERA_TYPE_ADCM_2650               01
#define CAMERA_TYPE_ADCM_2670               02
#define CAMERA_TYPE_ADCM_2700               03
#define CAMERA_TYPE_OMNIVISION_9640         04
#define CAMERA_TYPE_MAX                     CAMERA_TYPE_OMNIVISION_9640

/*
Image format definition
*/
#define CAMERA_IMAGE_FORMAT_MAX                CAMERA_IMAGE_FORMAT_YCBCR444_PLANAR

// Interrupt mask
#define CAMERA_INTMASK_FIFO_OVERRUN            0x0001
#define CAMERA_INTMASK_END_OF_FRAME            0x0002  
#define CAMERA_INTMASK_START_OF_FRAME          0x0004
#define CAMERA_INTMASK_CI_DISABLE_DONE         0x0008
#define CAMERA_INTMASK_CI_QUICK_DISABLE        0x0010
#define CAMERA_INTMASK_PARITY_ERROR            0x0020
#define CAMERA_INTMASK_END_OF_LINE             0x0040
#define CAMERA_INTMASK_FIFO_EMPTY              0x0080
#define CAMERA_INTMASK_RCV_DATA_AVALIBLE       0x0100
#define CAMERA_INTMASK_TIME_OUT                0x0200
#define CAMERA_INTMASK_END_OF_DMA              0x0400

// Interrupt status
#define CAMERA_INTSTATUS_FIFO_OVERRUN_0        0x00000001
#define CAMERA_INTSTATUS_FIFO_OVERRUN_1        0x00000002
#define CAMERA_INTSTATUS_FIFO_OVERRUN_2        0x00000004
#define CAMERA_INTSTATUS_END_OF_FRAME          0x00000008  
#define CAMERA_INTSTATUS_START_OF_FRAME        0x00000010
#define CAMERA_INTSTATUS_CI_DISABLE_DONE       0x00000020
#define CAMERA_INTSTATUS_CI_QUICK_DISABLE      0x00000040
#define CAMERA_INTSTATUS_PARITY_ERROR          0x00000080
#define CAMERA_INTSTATUS_END_OF_LINE           0x00000100
#define CAMERA_INTSTATUS_FIFO_EMPTY_0          0x00000200
#define CAMERA_INTSTATUS_FIFO_EMPTY_1          0x00000400
#define CAMERA_INTSTATUS_FIFO_EMPTY_2          0x00000800
#define CAMERA_INTSTATUS_RCV_DATA_AVALIBLE_0   0x00001000
#define CAMERA_INTSTATUS_RCV_DATA_AVALIBLE_1   0x00002000
#define CAMERA_INTSTATUS_RCV_DATA_AVALIBLE_2   0x00004000
#define CAMERA_INTSTATUS_TIME_OUT              0x00008000
#define CAMERA_INTSTATUS_END_OF_DMA            0x00010000

// Capture status
#define CAMERA_STATUS_VIDEO_CAPTURE_IN_PROCESS 0x0001
#define CAMERA_STATUS_RING_BUFFER_FULL         0x0002

/*
Structures
*/
typedef struct camera_context_s camera_context_t, *p_camera_context_t;

typedef struct {
    int (*init)(p_camera_context_t context);
    int (*deinit)(p_camera_context_t);   
    int (*set_capture_format)(p_camera_context_t);
    int (*start_capture)(p_camera_context_t, unsigned int frames);
    int (*stop_capture)(p_camera_context_t);
} camera_function_t, *p_camera_function_t;

// context 
struct camera_context_s {
    // syncronization stuff
    atomic_t refcount;

    /*
     DRIVER FILLED PARAMTER
    */
    // sensor info  
    unsigned int sensor_type;
    
    // capture image info
    unsigned int capture_width; 
    unsigned int capture_height;
    unsigned int capture_input_format;
    unsigned int capture_output_format;
    
    // frame rate control
    unsigned int frame_rate;

    // ring buffers
    // note: must pass in 8 bytes aligned address
    void *buffer_virtual;
    void *buffer_physical;
    unsigned int buf_size;

    // memory for dma descriptors, layout:
    //      dma descriptor chain 0,
    //      dma descriptor chain 1,
    //      ...  
    void *dma_descriptors_virtual;
    void *dma_descriptors_physical;
    unsigned int dma_descriptors_size;
    
    // os mapped register address       
    unsigned int clk_reg_base;
    unsigned int ost_reg_base;
    unsigned int gpio_reg_base;
    unsigned int ci_reg_base;
    unsigned int board_reg_base;

	// function dispatch table
    p_camera_function_t camera_functions;

    /*
     FILLED PARAMTER
    */
    int dma_channels[3];
    unsigned int capture_status;
    
    /*
     INTERNALLY USED: DON'T TOUCH!
    */
    unsigned int block_number, block_size;
    unsigned int block_header, block_tail;
    unsigned int fifo0_descriptors_virtual, fifo0_descriptors_physical;
    unsigned int fifo1_descriptors_virtual, fifo1_descriptors_physical;
    unsigned int fifo2_descriptors_virtual, fifo2_descriptors_physical;
    unsigned int fifo0_num_descriptors;
    unsigned int fifo1_num_descriptors;
    unsigned int fifo2_num_descriptors;
    unsigned int fifo0_transfer_size;
    unsigned int fifo1_transfer_size;
    unsigned int fifo2_transfer_size;
};





/*
Prototypes
*/
/***********************************************************************
 *
 * Init/Deinit APIs
 *
 ***********************************************************************/
// Setup the sensor type, configure image capture format (RGB, yuv 444, yuv 422, yuv 420, packed | planar, MJPEG) regardless
// of current operating mode (i.e. sets mode for both still capture and video capture)
int camera_init( p_camera_context_t camera_context );

// Power off sensor
int camera_deinit( p_camera_context_t camera_context );


/***********************************************************************
 *
 * Capture APIs
 *
 ***********************************************************************/
// Set the image format
int camera_set_capture_format( p_camera_context_t camera_context );

// take a picture and copy it into the ring buffer
int camera_capture_still_image( p_camera_context_t camera_context, unsigned int block_id );

// capture motion video and copy it the ring buffer
int camera_start_video_capture( p_camera_context_t camera_context, unsigned int block_id );

// disable motion video image capture
void camera_stop_video_capture( p_camera_context_t camera_context );


/***********************************************************************
 *
 * Flow Control APIs
 *
 ***********************************************************************/
// continue capture image to next available buffer
// Returns the continued buffer id, -1 means buffer full and no transfer started
void camera_continue_transfer( p_camera_context_t camera_context ); 

// Return 1: there is available buffer, 0: buffer is full
int camera_next_buffer_available( p_camera_context_t camera_context );

// Application supplies the FrameBufferID to the driver to tell it that the application has completed processing of the 
// given frame buffer, and that buffer is now available for re-use.
void camera_release_frame_buffer( p_camera_context_t camera_context, unsigned int frame_buffer_id ); 

// Returns the FrameBufferID for the first filled frame
// Note: -1 represents buffer empty
int camera_get_first_frame_buffer_id( p_camera_context_t camera_context ); 

/*
Returns the FrameBufferID for the last filled frame, this would be used if we were polling for image completion data, 
or we wanted to make sure there were no frames waiting for us to process.
Note: -1 represents buffer empty
*/
int camera_get_last_frame_buffer_id( p_camera_context_t camera_context ); 


/***********************************************************************
 *
 * Buffer Info APIs
 *
 ***********************************************************************/
// Return: the number of frame buffers allocated for use.
unsigned int camera_get_num_frame_buffers( p_camera_context_t camera_context );

/* 
FrameBufferID is a number between 0 and N-1, where N is the total number of frame buffers in use.  
Returns the address of the given frame buffer.  
The application will call this once for each frame buffer at application initialization only.
*/
void* camera_get_frame_buffer_addr( p_camera_context_t camera_context, unsigned int frame_buffer_id );

// Return the block id
int camera_get_frame_buffer_id( p_camera_context_t camera_context, void* address );

/***********************************************************************
 *
 * Frame rate APIs
 *
 ***********************************************************************/
// Set desired frame rate
void camera_set_capture_frame_rate( p_camera_context_t camera_context ); 

// return current setting
void camera_get_capture_frame_rate( p_camera_context_t camera_context ); 


/***********************************************************************
 *
 * Interrupt APIs
 *
 ***********************************************************************/
// set interrupt mask 
void camera_set_int_mask( p_camera_context_t camera_context, unsigned int mask ); 

// get interrupt mask 
unsigned int camera_get_int_mask( p_camera_context_t camera_context ); 

// clear interrupt status
void camera_clear_int_status( p_camera_context_t camera_context, unsigned int status );

// gpio init
void camera_gpio_init(void);

#endif
