/*
 *  Keyboard input driver for the Fastap keypad, interfaced through serial port (B9600, 8n1)
 *  and 8 direct keys, interfaced through PXA27x's DKPIN input (on PXA27x machine only)
 *
 *  Fastap sample key: key down: 0x80 0x1b or 0x82 0x1b if previous key not yet released     key up (key independent): 0x81 0x0 *
 * Special mappings:
 *  Shift + No     -> Fn key
 *  Alt   + Cursor -> Home/End/PgUp/PgDown
 *
 * Usage:
 *  To connect keypad with serial port, use inputattach tool, 
 *  available from http://linuxconsole.sourceforge.net/quick.html 
 *  For compiling hints: http://agendawiki.com/cgi-bin/aw.pl?DatacompSerialKeyboard
 * 
 *  $ inputattach -ftap /dev/ttyS1 &
 *
 * Changelog:
 *  2/13/2005 : initial version
 *
 *  Copyright (c) 2005 Matthias Ihmig <m.ihmig@mytum.de>
 *  
 */

/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/input.h>
#include <linux/init.h>
#include <linux/serio.h>

#ifdef CONFIG_PXA27x
#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/arch/pxa-regs.h>
#endif

#define DRIVER_DESC	"Fastap serial & DKIN keyboard driver v1.0"

// defined in linux/serio.h: #define SERIO_FASTAP 0x29

// Key array sizes
#define KEYCODE_SIZE    96
#define DIRECTKEY_SIZE  12
#define REMAP_SIZE      20

#define DIRECTKEY_REMAP  4
#define DIRECTKEY_OFFSET 4

// Remapped keys use a special lookup table
#define KMAPBASE 230
#define KMAP(x) (KMAPBASE+x)

// states
// KPRESS: next byte is scancode
// KRELEASE: next byte is 0x0
#define KEMPTY   0
#define KPRESS   1
#define KRELEASE 2

// QT specific key events, see http://doc.trolltech.com/qtopia2.1/html/devices-keys-buttons.html for details
#define QT_MENU  65	/* Application menu key */
#define QT_BACK  67	/* Accept/Close dialog key */
#define QT_CALL  87	/* Start call */
#define QT_HANG  88	/* End call */
#define QT_SEL   116	/* Select menu option, checkbox, pressbutton */
#define QT_CON1  124	/* First Context button */
#define QT_CON2  125	/* Second Context button */

MODULE_AUTHOR("Matthias Ihmig <m.ihmig@mytum.de>");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");

// Mappings for keys without modifier, <scancode/index>-<key>
static unsigned char fastap_keycode[KEYCODE_SIZE] = {
// 0x00-a   01-1   02-b   03-~!    04-c   05-2   06-d   07-|?    08-e   09-3   0A-f   B  C  D  E  F
     KEY_A, KEY_1, KEY_B, KMAP(0), KEY_C, KEY_2, KEY_D, KMAP(1), KEY_E, KEY_3, KEY_F, 0, 0, 0, 0, 0,
// 0x10-g   11-4   12-h   13-^$    14-i   15-5   16-j   17-_&   18-k   19-6   1A-l   B  C  D  E  F
     KEY_G, KEY_4, KEY_H, KMAP(2), KEY_I, KEY_5, KEY_J, KMAP(3), KEY_K, KEY_6, KEY_L, 0, 0, 0, 0, 0,
// 0x20-m   21-7   22-n   23-+:    24-o   25-8   26-p   27--;    28-q   29-9   2A-r   B  C  D  E  F
     KEY_M, KEY_7, KEY_N, KMAP(4), KEY_O, KEY_8, KEY_P, KMAP(5), KEY_Q, KEY_9, KEY_R, 0, 0, 0, 0, 0,
// 0x30-s   31-*            32-t   33-"'    34-u    35-0   36-v  37-%,    38-w   39-#    3A-x   B  C  D  E  F
     KEY_S, KEY_KPASTERISK, KEY_T, KMAP(6), KEY_U, KEY_0, KEY_V, KMAP(7), KEY_W, KMAP(8), KEY_X, 0, 0, 0, 0, 0,
// 0x40-<.    41-=tab   42->@     43-[(     44-y   45-del\bs    46-z   47-])     48-{/     49-enter   4A-)\     B  C  D  E  F
     KMAP(9), KMAP(10), KMAP(11), KMAP(12), KEY_Y, KMAP(13),    KEY_Z, KMAP(14), KMAP(15), KEY_ENTER, KMAP(16), 0, 0, 0, 0, 0,
// 0x50-home  51 52-arrow up    53 54-lspace  55 56-rspace   57 58-blue     59 5A-esc   B  C  D  E  F
     QT_MENU, 0, QT_SEL,        0, KEY_SPACE, 0, KEY_SPACE,  0, KEY_INSERT, 0, QT_BACK, 0, 0, 0, 0, 0,
};

// Mappings from Direct Key Input to scancodes
static unsigned char fastap_directkey[DIRECTKEY_SIZE] = {
// 0x00-top left,  01-top right, 02-green,     03-red,     04-left,  05-right,  06-up,  07-down
     KEY_LEFTSHIFT, KEY_LEFTALT,  KEY_LEFTCTRL, KEY_ESC,   KEY_LEFT, KEY_RIGHT, KEY_UP, KEY_DOWN,
// keys with Alt pressed, use scancode offset by DIRECTKEY_OFFSET
                                                          KEY_HOME, KEY_END, KEY_PAGEUP, KEY_PAGEDOWN
};

// key mappings when shift on keyboard IS NOT pressed: index: KMAP(#),
//  { AT scancode, AT scancode needs shift (insert)}, always: need_shift_cancel=0
static unsigned char fastap_remap_noshift[REMAP_SIZE][2] = {
/*    0-!           1-?               2-$          3-&          */
    { KEY_1, 1 }, { KEY_SLASH, 1 }, { KEY_4, 1 }, { KEY_7, 1 },
/*    4-:                   5-;                   6-'                    7-,  */
    { KEY_SEMICOLON, 1 }, { KEY_SEMICOLON, 0 }, { KEY_APOSTROPHE, 0 }, { KEY_COMMA, 0 }, 
/*    8-#          9-.             10-tab        11-@           */
    { KEY_3, 1}, { KEY_DOT, 0 }, { KEY_TAB, 0 }, { KEY_2, 1 },
/*    12-(         13-bs                 14-)          15-/              16-\    */
    { KEY_9, 1}, { KEY_BACKSPACE, 0 }, { KEY_0, 1 }, { KEY_SLASH, 0 }, { KEY_BACKSLASH, 0 }
};

// key mappings when shift on keyboard IS pressed: index=KMAP(#)
//  { AT scancode, AT scancode needs shift= do not cancel shift}, always: need_shift_insert=0
static unsigned char fastap_remap_withshift[REMAP_SIZE][2] = {
//     0-~              1-|                   2-^           3-_ 
    { KEY_GRAVE, 1 }, { KEY_BACKSLASH, 1 }, { KEY_6, 1 }, { KEY_MINUS, 1 },
//     4-+              5--               6-"                    7-% 
    { KEY_EQUAL, 1 }, { KEY_MINUS, 0 }, { KEY_APOSTROPHE, 1 }, { KEY_5, 1 },
//    8-`               9-<               10-=              11->
    { KEY_GRAVE, 0 }, { KEY_COMMA, 1 }, { KEY_EQUAL, 0 }, { KEY_DOT, 1 },
//    12-[                  13-del             14-]                   15-{    16-}
    { KEY_LEFTBRACE, 0 }, { KEY_DELETE, 0 }, { KEY_RIGHTBRACE, 0 }, { KEY_LEFTBRACE, 1 }, { KEY_RIGHTBRACE, 1 }  
};

static char *fastap_name = "Fastap Keyboard";

struct fkbd {
	unsigned char keycode[KEYCODE_SIZE];
	unsigned char directkey[DIRECTKEY_SIZE];
	unsigned char remap_noshift[REMAP_SIZE][2];
	unsigned char remap_withshift[REMAP_SIZE][2];
	unsigned char lastkey;			// last pressed and still down fastap-key, 0xFF if released
	unsigned char status;			// what to do next..
	unsigned char isKeyboardShift;		// true if user holds shift on keyboard, controlled by DirectKey Shift
	unsigned char isKeyboardCtrl;		// true if user holds ctrl  on keyboard, controlled by DirectKey Ctrl
	unsigned char isKeyboardAlt;		// true if user holds alt   on keyboard, controlled by DirectKey Alt
	unsigned char directkeystate;		// last state of direct keys (DKIN mapping)
	struct input_dev dev;
	struct serio *serio;
	char phys[32];
};

struct fscancode {
	unsigned char scancode;			// AT scancode
	char need_shift_insert;			// key needs new shift inserted
	char need_shift_cancel;			// key needs previous shift cancelled
};

// Takes Fastap keycode and returns AT key with shift insertion hints
void fkbd_getcode(struct fkbd *fkbd, unsigned char keycode, struct fscancode *sc)
{
	unsigned char code;
	 
	if (keycode >= 128)
		printk (KERN_WARNING "fastap_serial: invalid key received\n");
	
	code = fkbd->keycode[keycode];
	
	if (code >= KMAP(0)) {			// Fastap Remapping required
		code -= KMAP(0);		// "strip" KMAP
		if (!fkbd->isKeyboardShift) {	// just the key, not shift key is held
			sc->scancode          = fkbd->remap_noshift[code][0];
			sc->need_shift_insert = fkbd->remap_noshift[code][1];
			sc->need_shift_cancel = 0;
		} else {
			sc->scancode          = fkbd->remap_withshift[code][0];
			sc->need_shift_insert = 0;
			sc->need_shift_cancel = 1 - fkbd->remap_withshift[code][1];
		}
	} else {						// no fastap remapping
		sc->need_shift_insert = 0;
		sc->need_shift_cancel = 0;
		if (fkbd->isKeyboardShift && (code >= KEY_1 && code <= KEY_0) ) {	// shift+No -> Fn. Key
			code += (KEY_F1 - KEY_1);		// Convert to Fn. Key
			sc->need_shift_cancel = 1;
		}
		sc->scancode = code;
	}
}

irqreturn_t fkbd_interrupt(struct serio *serio,
		unsigned char data, unsigned int flags, struct pt_regs *regs)
{
	struct fkbd *fkbd = serio->private;
	struct fscancode scancode;

	if ((data & 0xFC) == 0x80) {			// new key press or new key release following
		if (data & 0x01)
			fkbd->status = KRELEASE;
		  else
		 	fkbd->status = KPRESS;
	} else {							// received scancode 
		if (fkbd->lastkey != 0xFF) {				// old key still down -> release it
			fkbd_getcode(fkbd, fkbd->lastkey, &scancode);
			input_regs(&fkbd->dev, regs);
			input_report_key(&fkbd->dev, scancode.scancode, 0);	// release key
			
			if ( scancode.need_shift_insert )
				input_report_key(&fkbd->dev, KEY_LEFTSHIFT, 0);	// cancel manually inserted shift
			if ( scancode.need_shift_cancel )
				input_report_key(&fkbd->dev, KEY_LEFTSHIFT, 1);	// restore shift 
				
			input_sync(&fkbd->dev);
			fkbd->lastkey = 0xFF;
		}
		fkbd_getcode(fkbd, data, &scancode);
		if ((fkbd->status == KPRESS) && scancode.scancode) {		// new and valid key arrived -> set it
			input_regs(&fkbd->dev, regs);
			if ( scancode.need_shift_insert )		// cancel manually inserted shift
				input_report_key(&fkbd->dev, KEY_LEFTSHIFT, 1);	// insert shift 
			if ( scancode.need_shift_cancel )
				input_report_key(&fkbd->dev, KEY_LEFTSHIFT, 0);	// cancel shift

			input_report_key(&fkbd->dev, scancode.scancode, 1);	// press new key
			input_sync(&fkbd->dev);
			fkbd->lastkey = data;
		}
		fkbd->status = KEMPTY;
	}
	return IRQ_HANDLED;
}

#ifdef CONFIG_PXA27x
static irqreturn_t fkbd_directkey_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned char dkxor;
	unsigned char code, val;
	char doremap, i, p;
	unsigned char newstate, kpc;
	struct fkbd *fkbd = dev_id;
	
	newstate = (unsigned char) KPDK & 0xFF;
	kpc = (unsigned char) KPC;

	dkxor = (newstate) ^ fkbd->directkeystate;
	fkbd->directkeystate = newstate;

	input_regs(&fkbd->dev, regs);
	for (i=0; i<8; i++) {
		p = 1 << i;
		if (dkxor & p) {
			// if Alt is pressed for remapping and not a modifier key -> add remap offset
			doremap = (fkbd->isKeyboardAlt) && (i >= DIRECTKEY_REMAP );
			code = fkbd->directkey[i + (doremap ? DIRECTKEY_OFFSET : 0)];
			val  = ((fkbd->directkeystate & p) != 0 ? 1 : 0);

			if (doremap && val)	// key pressed and remapping -> cancel Alt to make Home/End/PgUp/PgDown work
				input_report_key(&fkbd->dev, KEY_LEFTALT, 0);
			if (doremap && !val)	// key released and remapping -> restore Alt
				input_report_key(&fkbd->dev, KEY_LEFTALT, 1);
			
			input_report_key(&fkbd->dev, code, val);
	
			if (code == KEY_LEFTSHIFT)
				fkbd->isKeyboardShift = val;
			if (code == KEY_LEFTCTRL)
				fkbd->isKeyboardCtrl  = val;
			if (code == KEY_LEFTALT)
				fkbd->isKeyboardAlt   = val;
		}
	}
	input_sync(&fkbd->dev);

	return IRQ_HANDLED;
}
#endif

void fkbd_connect(struct serio *serio, struct serio_driver *drv)
{
	struct fkbd *fkbd;
	int i;
	unsigned char c;

	if (serio->type != (SERIO_RS232 | SERIO_FASTAP))
		return;

	if (!(fkbd = kmalloc(sizeof(struct fkbd), GFP_KERNEL)))
		return;

	memset(fkbd, 0, sizeof(struct fkbd));

	fkbd->dev.evbit[0] = BIT(EV_KEY) | BIT(EV_REP);
	fkbd->serio = serio;

	init_input_dev(&fkbd->dev);
	
	fkbd->dev.keycode = fkbd->keycode;
	fkbd->dev.keycodesize = sizeof(unsigned char);
	fkbd->dev.keycodemax = ARRAY_SIZE(fastap_keycode);
	fkbd->dev.private = fkbd;
	serio->private = fkbd;

	if (serio_open(serio, drv)) {
		kfree(fkbd);
		return;
	}

	memcpy(fkbd->keycode, fastap_keycode, sizeof(fkbd->keycode));
	memcpy(fkbd->directkey, fastap_directkey, sizeof(fkbd->directkey));
	memcpy(fkbd->remap_noshift, fastap_remap_noshift, sizeof(fkbd->remap_noshift));
	memcpy(fkbd->remap_withshift, fastap_remap_withshift, sizeof(fkbd->remap_withshift));

	for (i = 0; i < 128; i++) {
		c = fkbd->keycode[i];
		if (c < KMAP(0))
			set_bit(c, fkbd->dev.keybit);
	}
	for (i = 0; i < ARRAY_SIZE(fkbd->directkey); i++)
		set_bit(fkbd->directkey[i], fkbd->dev.keybit);
	for (i = 0; i < ARRAY_SIZE(fkbd->remap_noshift); i++)
		set_bit(fkbd->remap_noshift[i][0], fkbd->dev.keybit);
	for (i = 0; i < ARRAY_SIZE(fkbd->remap_withshift); i++)
		set_bit(fkbd->remap_withshift[i][0], fkbd->dev.keybit);
	for (i = KEY_F1; i <= KEY_F10; i++) 
		set_bit(i, fkbd->dev.keybit);
		
	clear_bit(0, fkbd->dev.keybit);
	
	sprintf(fkbd->phys, "%s/input0", serio->phys);

	fkbd->dev.name = fastap_name;
	fkbd->dev.phys = fkbd->phys;
	fkbd->dev.id.bustype = BUS_RS232;
	fkbd->dev.id.vendor = SERIO_FASTAP;
	fkbd->dev.id.product = 0x0001;
	fkbd->dev.id.version = 0x0001;

	input_register_device(&fkbd->dev);

	printk(KERN_INFO "fastap_serial: %s connected to %s\n", fastap_name, serio->phys);
	
#ifdef CONFIG_PXA27x
	// Configure DKIN Input pins, enable DKIN and request IRQ for Direct Keys
	
	pxa_set_cken(CKEN19_KEYPAD, 1);
	KPKDI = 60;	// debounce in ms
	
	// Configure Pins as DKIN
	pxa_gpio_mode( 93 | GPIO_ALT_FN_1_IN );
	pxa_gpio_mode( 94 | GPIO_ALT_FN_1_IN );
	pxa_gpio_mode( 95 | GPIO_ALT_FN_1_IN );
	pxa_gpio_mode( 96 | GPIO_ALT_FN_1_IN );
	pxa_gpio_mode( 98 | GPIO_ALT_FN_1_IN );
	pxa_gpio_mode( 99 | GPIO_ALT_FN_1_IN );
	pxa_gpio_mode( 13 | GPIO_ALT_FN_2_IN );

	set_irq_type(IRQ_KEYPAD, IRQT_BOTHEDGE);

	if (request_irq(IRQ_KEYPAD, fkbd_directkey_interrupt, SA_INTERRUPT, "Fastap Buttons", fkbd)) {
                printk(KERN_ERR "fastap_serial: Can't allocate irq %d for direct key input.\n", IRQ_KEYPAD);
                return;
        }
   	KPC = (7 << 6) | KPC_DIE | KPC_DE ;	// 7+1=8 input pins used
	printk(KERN_INFO "fastap_serial: Direct Key input pins enabled\n");
#endif
}

void fkbd_disconnect(struct serio *serio)
{
	struct fkbd *fkbd = serio->private;
#ifdef CONFIG_PXA27x
	KPC = 0 ; 			// Disable DKIN
	pxa_set_cken(CKEN19_KEYPAD, 0);
	free_irq(IRQ_KEYPAD, fkbd); 
#endif	
	input_unregister_device(&fkbd->dev);
	serio_close(serio);
	kfree(fkbd);
}

struct serio_driver fkbd_drv = {
	.driver		= {
		.name	= "fastapkbd",
	},
	.description	= DRIVER_DESC,
	.interrupt	= fkbd_interrupt,
	.connect	= fkbd_connect,
	.disconnect	= fkbd_disconnect,
};

int __init fkbd_init(void)
{
	serio_register_driver(&fkbd_drv);
	return 0;
}

void __exit fkbd_exit(void)
{
	serio_unregister_driver(&fkbd_drv);
}

module_init(fkbd_init);
module_exit(fkbd_exit);
