/*
 *  linux/arch/arm/mach-pxa/glencoe.c
 *
 *  Support for the Intel B3G Glencoe Platform.
 *	based on mainstone.c
 *  Changed by Matthias Ihmig <m.ihmig@mytum.de>
 *
 *  Original author:	Nicolas Pitre
 *  Created:		Nov 05, 2002
 *  Copyright:		MontaVista Software Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

#include <linux/init.h>
#include <linux/device.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/bitops.h>
#include <linux/fb.h>
#include <linux/mmc/host.h>

#include <asm/types.h>
#include <asm/setup.h>
#include <asm/memory.h>
#include <asm/mach-types.h>
#include <asm/hardware.h>
#include <asm/irq.h>

#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/mach/irq.h>

#include <asm/arch/pxa-regs.h>
#include <asm/arch/glencoe.h>
#include <asm/arch/audio.h>
#include <asm/arch/pxafb.h>
#include <asm/arch/mmc.h>

#include "generic.h"


static void __init glencoe_init_irq(void)
{
	pxa_init_irq();
}
/* Framework future power code.
static int gln_audio_startup(snd_pcm_substream_t *substream, void *priv)
{
	// audio routing/power up/GPIOs/...
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) ;
	return 0;
}
static void gln_audio_shutdown(snd_pcm_substream_t *substream, void *priv)
{
	if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) ;
}
static void gln_audio_suspend(void *priv)
{ }
static void gln_audio_resume(void *priv)
{ }
*/

static pxa2xx_audio_ops_t gln_audio_ops = {
/*	.startup	= gln_audio_startup,
	.shutdown	= gln_audio_shutdown,
	.suspend	= gln_audio_suspend,
	.resume		= gln_audio_resume,*/
};

static struct platform_device gln_audio_device = {
	.name		= "pxa2xx-ac97",
	.id		= -1,
	.dev		= { .platform_data = &gln_audio_ops },
};

static void glencoe_backlight_power(int on)
{
	if (on) {
		pxa_gpio_mode(GLN_GPIO_BL_PWR_ON | GPIO_OUT);
		pxa_gpio_mode(GLN_BACKLIGHT_PWM_MD);
		pxa_set_cken(GLN_BACKLIGHT_PWM_CKEN, 1);
		GLN_BACKLIGHT_PWM_CTRL = 0;
		GLN_BACKLIGHT_PWM_PWDUTY = 0x3ff;
		GLN_BACKLIGHT_PWM_PERVAL = 0x3ff;
		GPIO_set(GLN_GPIO_BL_PWR_ON);
	} else {
		PWM_CTRL0 = 0;
		GLN_BACKLIGHT_PWM_PWDUTY = 0x0;
		GLN_BACKLIGHT_PWM_PERVAL = 0x3ff;
		pxa_set_cken(GLN_BACKLIGHT_PWM_CKEN, 0);
		GPIO_clr(GLN_GPIO_BL_PWR_ON);
	}
}

static void glencoe_lcd_power(int on)
{
	if (on) {
		GPIO_set(GLN_GPIO_LCD_PWR_ON);
	} else {
		GPIO_clr(GLN_GPIO_LCD_PWR_ON);
	}
}

static struct pxafb_mach_info toshiba_ltm04c380k __initdata = {
	.pixclock		= 50000,
	.xres			= 640,
	.yres			= 480,
	.bpp			= 16,
	.hsync_len		= 1,
	.left_margin		= 0x9f,
	.right_margin		= 1,
	.vsync_len		= 44,
	.upper_margin		= 0,
	.lower_margin		= 0,
	.sync			= FB_SYNC_HOR_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT,
	.lccr0			= LCCR0_Act,
	.lccr3			= LCCR3_PCP,
	.pxafb_backlight_power	= glencoe_backlight_power,
	.pxafb_lcd_power	= glencoe_lcd_power,
};

static struct pxafb_mach_info toshiba_ltm035a776c __initdata = {
	.pixclock		= 110000,
	.xres			= 240,
	.yres			= 320,
	.bpp			= 16,
	.hsync_len		= 4,
	.left_margin		= 8+4,	/* 8 */
	.right_margin		= 20,	/* 14 */
	.vsync_len		= 3,
	.upper_margin		= 1+4,	/* 1 */
	.lower_margin		= 10,	/* 6 */
	.sync			= FB_SYNC_HOR_HIGH_ACT|FB_SYNC_VERT_HIGH_ACT,
	.lccr0			= LCCR0_Act,
	.lccr3			= LCCR3_PCP,
	.pxafb_backlight_power	= glencoe_backlight_power,
	.pxafb_lcd_power	= glencoe_lcd_power,
};

/*
 * MMC/SD Device
 *
 * The card detect interrupt isn't debounced and even triggers
 * too early on Glencoe, so delay it by HZ/4 to give the card
 * a chance to fully insert/eject.
 */
static struct mmc_detect {
	struct timer_list detect_timer;
	void *devid;
} mmc_detect;

static void mmc_detect_callback(unsigned long data)
{
#ifdef CONFIG_MMC
	mmc_detect_change(mmc_detect.devid);
#endif
}

static irqreturn_t glencoe_mmc_detect_int(int irq, void *devid, struct pt_regs *regs)
{
	mmc_detect.devid=devid;
	mod_timer(&mmc_detect.detect_timer, jiffies + HZ/4);
	return IRQ_HANDLED;
}

static int glencoe_mci_init(struct device *dev, irqreturn_t (*unused_detect_int)(int, void *, struct pt_regs *), void *data)
{
	struct mmc_host *mmc = (struct mmc_host*)data;
	int err;

	pxa_set_cken(CKEN12_MMC, 0);
	pxa_gpio_mode(GPIO32_MMCCLK_MD);
	pxa_gpio_mode(GPIO112_MMCCMD_MD);
	pxa_gpio_mode(GPIO92_MMCDAT0_MD);
	pxa_gpio_mode(GPIO109_MMCDAT1_MD);
	pxa_gpio_mode(GPIO110_MMCDAT2_MD);
	pxa_gpio_mode(GPIO111_MMCDAT3_MD);

	pxa_gpio_mode(GLN_GPIO_nMMC_CARD_DET | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_MMC_PWR_ON | GPIO_OUT);
	GPIO_clr(GLN_GPIO_MMC_PWR_ON);
	

	init_timer(&mmc_detect.detect_timer);
	mmc_detect.detect_timer.function = mmc_detect_callback;
	mmc_detect.detect_timer.data = (unsigned long) &mmc_detect;
	
	err = request_irq(GLENCOE_MMC_IRQ, glencoe_mmc_detect_int, SA_INTERRUPT,
			     "MMC card detect", (void *)mmc);
	if (err) {
		printk(KERN_ERR "glencoe_mci_init: MMC/SD: can't request MMC card detect IRQ\n");
		return -1;
	}
	set_irq_type(GLENCOE_MMC_IRQ, IRQT_BOTHEDGE);
	return 0;
}

static void glencoe_mci_setpower(struct device *dev, unsigned int vdd)
{
	struct pxamci_platform_data* p_d = dev->platform_data; 

	if ( vdd == 0 ) {
		printk(KERN_DEBUG "glencoe_mci_setpower: off\n");
		pxa_gpio_mode(GLN_GPIO_MMC_PWR_ON | GPIO_OUT);
		GPIO_clr(GLN_GPIO_MMC_PWR_ON);

	} else if ( ( 1 << vdd ) & p_d->ocr_mask ) {
		printk(KERN_DEBUG "glencoe_mci_setpower: on\n");
		pxa_gpio_mode(GLN_GPIO_MMC_PWR_ON | GPIO_OUT);
		GPIO_set(GLN_GPIO_MMC_PWR_ON);
	
	} else 
		printk(KERN_ERR "glencoe_mci_setpower: unable to set power\n");
	
}

static void glencoe_mci_exit(struct device *dev, void *data)
{ 
	struct mmc_host *mmc = (struct mmc_host*)data;
        unsigned long flags;

        local_irq_save(flags); 
	free_irq(GLENCOE_MMC_IRQ, (void *)mmc);
	del_timer(&mmc_detect.detect_timer);
        local_irq_restore(flags);
}

static struct pxamci_platform_data glencoe_mci_platform_data = {
	.ocr_mask	= MMC_VDD_32_33|MMC_VDD_33_34,
	.init 		= glencoe_mci_init,
	.setpower 	= glencoe_mci_setpower,
	.exit		= glencoe_mci_exit,
};

static void __init glencoe_init(void)
{
 	/* system bus arbiter setting
	 * - Core_Park
	 * - LCD_wt:DMA_wt:CORE_Wt = 2:3:4
	 */
	ARB_CNTRL = ARB_CORE_PARK | 0x234;

	GLN_SW_GREEN_LED_ON;

#ifdef CONFIG_FB_PXA_LCD_VGA
		set_pxa_fb_info(&toshiba_ltm04c380k);
#elif	CONFIG_FB_PXA_LCD_QVGA
		set_pxa_fb_info(&toshiba_ltm035a776c);
#endif

	pxa_set_mci_info(&glencoe_mci_platform_data);

	// audio init
	pxa_gpio_mode(GLN_GPIO_nAUDIO_PWR_DWN | GPIO_OUT);
	GPIO_set(GLN_GPIO_nAUDIO_PWR_DWN);	// power up audio/ts codec
	platform_device_register(&gln_audio_device);
}

static void __init glencoe_map_io(void)
{
	pxa_map_io();
}

MACHINE_START(GLENCOE, "Intel Beyond 3G Glencoe Platform")
	MAINTAINER("Intel")
	BOOT_MEM(0xa0000000, 0x40000000, io_p2v(0x40000000))
	BOOT_PARAMS(0xa0000100)
	MAPIO(glencoe_map_io)
	INITIRQ(glencoe_init_irq)
	INITTIME(pxa_init_time)
	INIT_MACHINE(glencoe_init)
MACHINE_END
