/*
 *
 * Bulverde voltage change driver. 
 *
 * Author:  Cain Yuan <cain.yuan@intel.com>
 * 
 * Upgrade for 2.6 kernel by Chao Xie <chao.xie@intel.com>
 *
 * Copyright (C) 2003-2004 Intel Corporation.
 *
 * This software program is licensed subject to the GNU General Public License
 * (GPL).Version 2,June 1991, available at http://www.fsf.org/copyleft/gpl.html
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <asm/hardware.h>
#include <asm/arch/pxa-regs.h>
#include <linux/cpufreq.h>

#include "cpu-freq-voltage-pxa27x.h"

/*
 *  Transfer desired mv to  required  DAC value.
 *  Vcore = 1.5v - ( 587uv * DACIn )
 */
#define MV2DAC(mv) ((1500-mv)*1000/587)
struct fv_table ipm_fv_table[] = {
	{26000,   900},
	{104000, 1100},
	{208000, 1200},
	{312000, 1300},
	{416000, 1400},
	{520000, 1500},
	{624000, 1500}, 
	{0,         0}
};

static unsigned int cpuvoltage_cur = PXA27X_DEFAULT_VOL;
static DECLARE_MUTEX    (pxa27x_voltage_sem);

extern void vm_setvoltage(unsigned int);
extern void vm_pre_setvoltage(unsigned int);

static unsigned int mv2DAC(unsigned int mv)
{
	if (mv > PXA27X_MAX_VOL)
		return MV2DAC(mv);
	if (mv < PXA27X_MIN_VOL)
		return MV2DAC(mv);
	return MV2DAC(mv);
}

static int check_voltage(unsigned int freq, unsigned int vol)
{
	int i = 0;

	printk("check:freq %u, vol %u",freq, vol);
	while (ipm_fv_table[i].freq != 0 ) {
		if (freq <= ipm_fv_table[i].freq) { 
			if(vol < ipm_fv_table[i].voltage) {
				printk(KERN_WARNING "Warnning: CPU voltage is lower than standard, may cause error.\n");
				return 1;
			}
			else
				return 0;
		}
		i++;
	}
	printk(KERN_ERR "Error: Can not find the voltage for frequency %u.\n", freq);
	return -EINVAL;
}

unsigned int pxa27x_find_voltage(unsigned int freq)
{
	int i = 0;

	while (ipm_fv_table[i].freq != 0 ) {
                if (freq <= ipm_fv_table[i].freq) {
                        return ipm_fv_table[i].voltage;
                }
                i++;
        }
	return 0;
}
EXPORT_SYMBOL_GPL(pxa27x_find_voltage);

/*
 * According to bulverde's manual, set the core's voltage.
 * mv == 0 indicates using default voltage arrcording frequency.
 * If mv == cpuvoltage_cur, no voltage change happens. Because there
 * is no convenient way of reading back voltage, it is important to mainstain
 * cpuvoltage_cur correctly.
 * combo == 0 means this is seperated voltage change process, otherwise it is 
 * combo change.
 */
int pxa27x_set_voltage(unsigned int freq, unsigned int mv, unsigned int combo)
{
	int ret;

	/*
	 * Currently voltage change is seperated with frequency change. 
	 */
	if ((mv < PXA27X_MIN_VOL) && (mv > PXA27X_MAX_VOL) && mv != 0)
		return -EINVAL;
	if (mv == 0) {
		if ((mv = pxa27x_find_voltage(freq)) == 0)
			return -EINVAL;
	}
	else {
		ret = check_voltage(freq, mv);
		if (ret < 0)
			return ret;
	}
	down(&pxa27x_voltage_sem);
	if (mv == cpuvoltage_cur) {
		up(&pxa27x_voltage_sem);
		return 1;
	}
	if (!combo)
		vm_setvoltage(mv2DAC(mv));
	else
		vm_pre_setvoltage(mv2DAC(mv));
	cpuvoltage_cur = mv;
	up(&pxa27x_voltage_sem);
	return 0;
}
EXPORT_SYMBOL_GPL(pxa27x_set_voltage);

unsigned int pxa27x_get_voltage(void)
{
	int vol;

	down(&pxa27x_voltage_sem);
	vol = cpuvoltage_cur;
	up(&pxa27x_voltage_sem);

	return vol;
}
EXPORT_SYMBOL_GPL(pxa27x_get_voltage);
#if 0
int pxa27x_combo_set_voltage(unsigned int freq)
{
	unsigned int mv;
	
	if ((mv = pxa27x_find_voltage(freq)) == 0)
		return -EINVAL;
	vm_pre_setvoltage(mv2DAC(mv));
	cpuvoltage_cur = mv;
	return mv;
}
/* This is the notifier of seperated voltage change for frequency change */
static int voltage_cpufreq_notifier(struct notifier_block *nb, 
					unsigned long val, void *data)
{
	struct cpufreq_freqs *freq = data;

	if ((val == CPUFREQ_PRECHANGE  && freq->old < freq->new) ||
            (val == CPUFREQ_POSTCHANGE && freq->old > freq->new)) {
		/* using default voltage according to the frequency */
		return pxa27x_set_voltage(freq->new, 0); 
	}
	return 0;
}

static struct notifier_block voltage_cpufreq_notifier_block = {
        .notifier_call  = voltage_cpufreq_notifier
};

/* This is the notifier of combo voltage change for frequency change */
static int combo_voltage_cpufreq_notifier(struct notifier_block *nb,
                                        unsigned long val, void *data)
{
        struct cpufreq_freqs *freq = data;

        if (val == CPUFREQ_PRECHANGE)  {
                return pxa27x_combo_set_voltage(freq->new);
        }
	return 0;
}

static struct notifier_block combo_voltage_cpufreq_notifier_block = {
        .notifier_call  = combo_voltage_cpufreq_notifier
};


static int __init pxa27x_voltage_init(void)
{
	int ret;

	ret = cpufreq_register_notifier(&voltage_cpufreq_notifier_block,
                                        CPUFREQ_TRANSITION_NOTIFIER);	
	return ret;
}
late_initcall(pxa27x_voltage_init);
#endif
