/* 
 * CPUfreq driver for PXA27x
 * 
 * cpufreq driver for PXA27x by Chao Xie <chao.xie@intel.com>
 * Copyright (C) 2004, Intel Corporation
 *
 * This software program is licensed subject to the GNU General Public License
 * (GPL).Version 2,June 1991, available at http://www.fsf.org/copyleft/gpl.html
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/cpufreq.h>
#include <linux/config.h>
#include <linux/ioctl.h>
#include <linux/delay.h>
#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/delay.h>
#include <asm/mach-types.h>
#include <asm/arch/pxa-regs.h>

#include "cpu-freq-voltage-pxa27x.h"

#define N_NUM 8
#define L_NUM 31

#define CPUFREQ_SET_BASE	(0x1 << 8)
#define CPUFREQ_SET_FREQ	CPUFREQ_SET_BASE << 0x1
#define CPUFREQ_SET_B		CPUFREQ_SET_BASE << 0x2
#define CPUFREQ_SET_T_HT	CPUFREQ_SET_BASE << 0x3
#define CPUFREQ_SET_A		CPUFREQ_SET_BASE << 0x4

#define MSC0_MASK	0x7ff07ff0
#define DRI_MASK	0xfff

struct pxa27x_mem_set {
	unsigned int memc_clock;
	unsigned int msc0;
	unsigned int dri;
};

static struct pxa27x_mem_set mem_op_val[] = {
	{13000, 0x11101110, 0x002},
	{91000, 0x12701270, 0x015},
	{104000, 0x12801280, 0x018},
	{208000, 0x15d015d0, 0x031},
	{0, 0, 0},
};

static unsigned int cpufreq_matrix[N_NUM-1][L_NUM-1];
static unsigned int cpufreq_cur;
static DECLARE_MUTEX    (pxa27x_freq_sem);

struct cpufreq_info {
	unsigned int l;
	unsigned int n2;
	unsigned int b;
	unsigned int t;
	unsigned int a;
};

static struct cpufreq_info cpufreq_mode;	

extern unsigned int get_clk_frequency_khz(int);
extern int pxa27x_set_voltage(unsigned int, unsigned int, unsigned int);
extern unsigned int pxa27x_get_voltage(void);
extern struct cpufreq_governor cpufreq_gov_performance;

static int pxa27x_cpufreq_governor(struct cpufreq_policy *policy, 
					unsigned int event)
{
        return 0;
}

static struct cpufreq_governor pxa27x_cpufreq_gov = {
        .name           = "pxa27x_governor",
        .governor       = pxa27x_cpufreq_governor,
        .owner          = THIS_MODULE,
};

static inline int set_mode(unsigned int l, unsigned int n2, unsigned int b, 
				unsigned int t, unsigned int a)
{
	unsigned int ret = 0;

	if (n2 == cpufreq_mode.n2 && l == cpufreq_mode.l && b == cpufreq_mode.b
		&& t == cpufreq_mode.t && a == cpufreq_mode.a)
		return 0;
	if (l == cpufreq_mode.l && n2 == cpufreq_mode.n2) {
		if (b != cpufreq_mode.b)
			ret |= CPUFREQ_SET_B;
		if (t != cpufreq_mode.t)
			ret |= CPUFREQ_SET_T_HT;
		if (a != cpufreq_mode.a)
			ret |= CPUFREQ_SET_A;
	}
	else {
		ret |= CPUFREQ_SET_FREQ;
		cpufreq_mode.l = l;
		cpufreq_mode.n2 = n2;
	}
	cpufreq_mode.b = b;
	cpufreq_mode.t = t;
	cpufreq_mode.a = a;
	return ret;
}

#ifdef CONFIG_FAST_DVFM
void pxa27x_setspeed(unsigned int CLKCFGValue)
#else
static void pxa27x_setspeed(unsigned int CLKCFGValue)
#endif
{
        unsigned long flags;
        unsigned int unused;

        /* NOTE: IRQs are already turned off in dpm idle, so it is not
           necessary to do it here. */
        local_irq_save(flags);

	__asm__ __volatile__(" \n\
		ldr     r4, [%1]                        @load MDREFR \n\
		mcr     p14, 0, %2, c6, c0, 0           @ set CCLKCFG[FCS] \n\
		ldr     r5, =0xe3dfefff \n\
		and     r4, r4, r5      \n\
		str     r4,  [%1]                       @restore \n\
		"
		: "=&r" (unused)
		: "r" (&MDREFR), "r" (CLKCFGValue)
		: "r4", "r5");

        /* NOTE: if we don't turn off IRQs up top, there is no point
           to restoring them here. */
        local_irq_restore(flags);
}
#ifdef CONFIG_FAST_DVFM
EXPORT_SYMBOL(pxa27x_setspeed);
#endif

#ifdef CONFIG_FAST_DVFM
extern void pxafb_wait_for_eof(unsigned int);
#else
extern void pxafb_wait_for_eof(void);
#endif

static void pxa27x_change_freq(unsigned int CLKCFGvalue)
{
#ifdef CONFIG_FAST_DVFM
	pxafb_wait_for_eof(CLKCFGvalue);
#else
	pxafb_wait_for_eof();
	pxa27x_setspeed(CLKCFGvalue);
#endif
}

static void pxa27x_set_mem_before_freq(unsigned int memc_clock, 
		unsigned int *pmdrefr, unsigned int *pmsc0, unsigned int *pdri)
{
	int i = 0; 

	while(mem_op_val[i].memc_clock < memc_clock && mem_op_val[i].memc_clock != 0)
		i++;
	/* if current msc0 > next msc0, change after */
	if ( (MSC0 & MSC0_MASK) > mem_op_val[i].msc0) {
		*pmsc0 = mem_op_val[i].msc0;	
	}
	else {
		*pmsc0 = 0;
		MSC0 = (MSC0 & ~MSC0_MASK) | mem_op_val[i].msc0;
	}
	/* if current dri > next dri, change after */
	if ( (MDREFR & DRI_MASK) > mem_op_val[i].dri) {
		*pdri = mem_op_val[i].dri;	
	}
	else {
		*pdri = 0;
		MDREFR = (MDREFR & ~DRI_MASK) | mem_op_val[i].dri;
	}

	if (memc_clock > 104000) { 
		/* K1DB2, K2DB2 set; memc/2 for sdram */
		/* K0DB4 set; memc/4 for flash */
		MDREFR |= ((1 << 14) | (1 << 17) | (1 << 19) | (1 << 29));
		*pmdrefr = 0;
	}
	else if (memc_clock > 52000) {
		MDREFR |= (1 << 14);
		/* K1DB2, K2DB2 will be clear; memc for sdram */
		/* K0DB4 will be clear; memc/2 for flash */
		*pmdrefr = (1 << 17) | (1 << 19) | (1 << 29);
	} 
	else {
		/* memc for flash and sdram*/
		*pmdrefr = (1 << 14) | (1 << 17) | (1 << 19) | (1 << 29); 
	}
	
}

void pxa27x_set_mem_after_freq(unsigned int mdrefr, unsigned int msc0, 
			unsigned int dri)
{
	if (mdrefr || dri)
		MDREFR = (MDREFR & (~DRI_MASK | mdrefr)) | dri;
	if (msc0)
		MSC0 = (MSC0 & ~MSC0_MASK) | msc0;
}

/* turbo_mode_flag = 2 : indicates Turbo 
 * turbo_mode_flag = 1 : indicates HalfTurbo
 */
static void pxa27x_set_core_freq(unsigned int l, unsigned int n2, 
				unsigned int fast_bus_mode_flag,
				unsigned int turbo_mode_flag,
				unsigned int memc_clock_flag,
				unsigned int relation)
{
	int CLKCFGValue = 0;
	int turbo_mode = 0;
	int memc_clock, m, l_pll;
	unsigned int mdrefr, msc0, dri;

	if (turbo_mode_flag == 2)
		turbo_mode = 1;
	else if (turbo_mode_flag == 1)
		turbo_mode = 1 << 2;
		
	m = (l <= 10) ? 1 : (l <= 20) ? 2 : 4;
	l_pll = l*13000;
	memc_clock = (!memc_clock_flag) ? (l_pll/m) : ((fast_bus_mode_flag) ? l_pll : (l_pll/2));
	pxa27x_set_mem_before_freq(memc_clock, &mdrefr, &msc0, &dri); 	
	/* SDRAM can not be greater than 104Mhz, and flash can not be greater 
	   than 52Mhz*/ 	
	if (relation & (CPUFREQ_SET_FREQ | CPUFREQ_SET_B | CPUFREQ_SET_A)) {
		CCCR = (n2 << 7) | l | (memc_clock_flag << 25); 

		/*  set up the new CLKCFG value, for an new frequency       */
		/*  0x2 is used to set FC bit in CLKCFG. */
		CLKCFGValue = ( 0x2 | (fast_bus_mode_flag << 3) | turbo_mode);
	}
	else { 
		CLKCFGValue = (fast_bus_mode_flag << 3) | turbo_mode;
	}
	
	/*
	 * Program the CLKCFG (CP14, reigster 6) for frequency change
	 */
	pxa27x_change_freq(CLKCFGValue);
	pxa27x_set_mem_after_freq(mdrefr, msc0, dri);
}

unsigned int pxa27x_read_clkcfg(void)
{
	unsigned int value=0;
	unsigned int un_used;

	__asm__ __volatile__("mrc       p14, 0, %1, c6, c0, 0" :"=&r" (un_used)
			:"r" (value));

	return value;
}
EXPORT_SYMBOL_GPL(pxa27x_read_clkcfg);

#define CCCR_CPDIS_BIT_ON               (1 << 31)
#define CCCR_PPDIS_BIT_ON               (1 << 30)
#define CCCR_PLL_EARLY_EN_BIT_ON        (1 << 26)
#define CCCR_LCD_26_BIT_ON              (1 << 27)

static int old_clkcfg;
static int old_mdrefr;
static int old_msc0;
static int enter_13M(void)
{
	int cccr;

	if (CCCR & (1 << 31))
		return 1;
	old_clkcfg = pxa27x_read_clkcfg();
	old_mdrefr = MDREFR;
	old_msc0 = MSC0;
#ifdef CONFIG_13M_TWOSTEP
	if (old_clkcfg & 0x5) {
		pxa27x_setspeed(old_clkcfg & 0x8);
		cccr = CCCR;
		while (pxa27x_read_clkcfg() != (old_clkcfg & 0x8))
			;

	}
#endif
	cccr = CCCR;
	cccr |= CCCR_CPDIS_BIT_ON;
	cccr &= ~CCCR_PPDIS_BIT_ON;
	cccr |= CCCR_LCD_26_BIT_ON;
	CCCR = cccr;
	LCCR4 |= (1 << 25);
	pxa27x_setspeed(0x2);
	MDREFR = (MDREFR & ~((1 << 14) | (1 << 17) | (1 << 19) | (1 << 29) | DRI_MASK)) | 0x002; 
	/* maybe this can optmize 13Mhz */
	MSC0 = (MSC0 & ~MSC0_MASK) | 0x11101110;
	return 0;
}

/* using new L, N2 to enbale CPDIS first, then change frequency */
static void exit_13M_change_freq(unsigned int l, unsigned int n2,
                                unsigned int fast_bus_mode_flag,
				unsigned int turbo_mode_flag,
				unsigned int memc_clock_flag)
{
	int cccr, clkcfg;
	unsigned int time_delay;

	if (!(CCCR & (1 << 31)))
		return;

	clkcfg = 0x2;
	if (turbo_mode_flag)
		clkcfg |= 0x1;
	if (fast_bus_mode_flag)
		clkcfg |= (0x1 << 3);

	cccr = CCCR;
	cccr |= (CCCR_CPDIS_BIT_ON | CCCR_PLL_EARLY_EN_BIT_ON);
	/* clear L and 2N, A bit */
	cccr &= ~(0x3ffffff);
	cccr |= l | n2 << 7 | (memc_clock_flag << 25);
	CCCR = cccr;

	/* Step 2 */
	cccr = CCCR;
	if ((cccr & 0x84000000) != 0x84000000) { /* CPDIS, pll_early_en */
		printk("DPM: Warning: CPDIS or PLL_EARLY_EN not on\n");
	}
	time_delay = OSCR;
	while ((OSCR - time_delay) < 390)
		;
	/* Step 3 */
	while ((CCSR & 0x30000000) == 0x30000000)
		break;
	/* Step 4: NOP */

	/* Step 5 */
	/* Now clear the PLL disable bits */
	/* But leave EARLY_EN on; it will be cleared by the frequency change */
	cccr &= ~(CCCR_CPDIS_BIT_ON | CCCR_PPDIS_BIT_ON); /* Not ON */
	cccr |= CCCR_PLL_EARLY_EN_BIT_ON;
	CCCR = cccr;
	MDREFR = old_mdrefr;
	MSC0 = old_msc0;
#ifdef CONFIG_13M_TWOSTEP
	pxa27x_setspeed(0x2);
#endif
	LCCR4 &= ~((1 << 25));
#ifdef CONFIG_13M_TWOSTEP
	cccr = CCCR;
	while(pxa27x_read_clkcfg() != 0x2)
		;
	if (turbo_mode_flag || fast_bus_mode_flag) {
		CKEN &= ~CKEN16_LCD;
		pxa27x_setspeed(clkcfg);
		CKEN |= CKEN16_LCD;
	}
#else
	pxa27x_setspeed(clkcfg);
#endif
}

void pxa27x_cpufreq_restore(void)
{
	int CLKCFGValue = 0;
	int turbo_mode = 0;
	int m, l_pll, memc_clock;
	unsigned int mdrefr, msc0, dri;

	if (cpufreq_cur == 13000) {
		enter_13M();
		return;
	}
	m = (cpufreq_mode.l <= 10) ? 1 : (cpufreq_mode.l <= 20) ? 2 : 4;
	l_pll = cpufreq_mode.l*13000;
	memc_clock = (!cpufreq_mode.a) ? (l_pll/m) : ((cpufreq_mode.b) ? l_pll : (l_pll/2));
	pxa27x_set_mem_before_freq(memc_clock, &mdrefr, &msc0, &dri); 	
	if (cpufreq_mode.t == 2)
		turbo_mode = 1;
	else if (cpufreq_mode.t == 1)
		turbo_mode = 1 << 2;

	CCCR = (cpufreq_mode.n2 << 7) | cpufreq_mode.l | (cpufreq_mode.a << 25); 
	CLKCFGValue = ( 0x2 | (cpufreq_mode.b << 3) | turbo_mode);
	pxa27x_setspeed(CLKCFGValue);
	pxa27x_set_mem_after_freq(mdrefr, msc0, dri);
}
EXPORT_SYMBOL_GPL(pxa27x_cpufreq_restore);

static int pxa27x_cpufreq_init(struct cpufreq_policy *policy)
{
	unsigned int n2, l;

	/* we only have one CPU */
	if (policy->cpu != 0)
		return -EINVAL;
	/* init policy */
	policy->cur = cpufreq_cur;
	/* we use CPU limits as default */
	policy->min = PXA27X_MIN_FREQ;
	policy->max = PXA27X_MAX_FREQ;
	policy->cpuinfo.min_freq = PXA27X_MIN_FREQ;
	policy->cpuinfo.max_freq = PXA27X_MAX_FREQ;
	policy->cpuinfo.transition_latency = CPUFREQ_ETERNAL;
	/* Only simple governer is needed. Directly using pxa27x 
	 * governor. In fact it is a NULL governor.
	 */
	policy->governor = &pxa27x_cpufreq_gov;
	/* based on manual to set freq table */
	memset(&cpufreq_matrix, 0 ,sizeof(cpufreq_matrix));
	for (n2 = 2; n2 <= N_NUM; n2++) {
		for (l = 2; l <= L_NUM; l++) {
			cpufreq_matrix[n2-2][l-2] = n2*l*6500;
			if( cpufreq_matrix[n2-2][l-2] > PXA27X_MAX_FREQ )
				cpufreq_matrix[n2-2][l-2] = 0;
		}
	}
	return 0;	
}

static int pxa27x_cpufreq_exit(struct cpufreq_policy *policy)
{
	return 0;
}

/* the frequency is gotten form L, N base on cpufreq_matrix */
static int pxa27x_cpufreq_verify(struct cpufreq_policy *policy)
{
	if (policy->min < PXA27X_MIN_FREQ || policy->max > PXA27X_MAX_FREQ)
		return -1;
	return 0;
}

/* every time invoke the function, frequency is set even nothing has changed */
static int pxa27x_cpufreq_target (struct cpufreq_policy *policy,
				unsigned int target_freq,
				unsigned int relation)
{
	unsigned int n2, l, b, t = 0, a;
	struct cpufreq_freqs freqs;

	freqs.cpu = 0;
	freqs.old = cpufreq_cur;
	freqs.new = target_freq;

	if (target_freq == 13000)  {
		cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);
	 	enter_13M();
		cpufreq_cur = target_freq;
		cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);
		calibrate_delay();
		return 0;
	}
	l = cpufreq_mode.l;
	n2 = cpufreq_mode.n2;
	b = cpufreq_mode.b;
	t = cpufreq_mode.t;
	a = cpufreq_mode.a;

/* System hang when enabling RUN/TURBO switching at 520MHZ */
#ifdef CONFIG_PXA27x_E37
	if (target_freq == 520000 && (relation & CPUFREQ_SET_T_HT)) {
		printk("Workround for E37 about enabling RUN/TURBO switching at 520MHZ.\n If you want to disable workround, DO NOT set \"CONFIG_PXA27x_E37=y\"\n");
		cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);
		enter_13M();
		exit_13M_change_freq(l, n2, b, t, a);
		cpufreq_cur = target_freq;
		cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);
		calibrate_delay();
		return 0;
	}
#endif
	cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);
	/* Using CCCR to make sure it is 13M */
	if ((cpufreq_cur == 13000) && (CCSR & (1 << 31))) 
		exit_13M_change_freq(l, n2, b, t, a);
	else
		pxa27x_set_core_freq(l, n2, b, t, a, relation);
	cpufreq_cur = target_freq;
	cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);
	calibrate_delay();
	return 0;

}
static unsigned int pxa27x_get_cur_freq(unsigned int cpu)
{
	if (cpu)
		return 0;
	return get_clk_frequency_khz(0);
}

static ssize_t show_available_freqs (struct cpufreq_policy *policy, char *buf)
{
	unsigned int n2, l, count = 0;

	count += sprintf(&buf[count], "%s\t%s\t%s\n", "L", "2N", "Freq"); 
	for (n2 = 2; n2 <= N_NUM ; n2++) {
		for (l = 2; l <= L_NUM; l++) {
			if (cpufreq_matrix[n2-2][l-2] >= policy->min 
				&& cpufreq_matrix[n2-2][l-2] <= policy->max)
				count += sprintf(&buf[count], "%d\t%d\t%d\n", l,
						 n2, cpufreq_matrix[n2-2][l-2]);
		}
	}
	return count;
}

static struct freq_attr pxa27x_available_freqs = {
        .attr = { .name = "scaling_available_frequencies", .mode = 0444 },
        .show = show_available_freqs,
};

static ssize_t show_freq_voltage (struct cpufreq_policy *policy, char *buf)
{
	int i = 0, count = 0;

	count += sprintf(&buf[count], "%s\t%s\n", "Freq", "Vol");
	while (ipm_fv_table[i].freq != 0 ) {
                count += sprintf(&buf[count], "%u\t%u\n", ipm_fv_table[i].freq, ipm_fv_table[i].voltage);
		i++;
        }
	return count;
}

static struct freq_attr pxa27x_freq_voltage = {
        .attr = { .name = "show_frequency_and_voltage", .mode = 0444 },
        .show = show_freq_voltage,
};

static ssize_t show_cpu_voltage (struct cpufreq_policy *policy, char *buf)
{
	unsigned int vol, count;

	vol = pxa27x_get_voltage();
	count = sprintf(&buf[0], "%dmv\n", vol);
	return count;
}

static ssize_t store_cpu_voltage (struct cpufreq_policy *policy, 
					const char *buf, size_t count)
{
	unsigned int vol, ret, n;
	
	n = sscanf(buf, "%u", &vol);
	ret = pxa27x_set_voltage(cpufreq_cur, vol, 0);
	if (ret < 0) {
		printk(KERN_ERR "Error:Wrong voltage");
	}
	return count;
}

static struct freq_attr pxa27x_cpu_voltage = {
	.attr = { .name = "cpu-voltage", .mode = 0644 },
	.show = show_cpu_voltage,
	.store = store_cpu_voltage,
};

static struct freq_attr* pxa27x_attr[] = {
        &pxa27x_available_freqs,
	&pxa27x_cpu_voltage,
	&pxa27x_freq_voltage,
        NULL,
};

static struct cpufreq_driver pxa27x_cpufreq_driver = {
	.name           = "pxa27x", /* there's a 16 char limit */
	.flags		= CPUFREQ_CONST_LOOPS,
	.init           = pxa27x_cpufreq_init,
	.exit           = pxa27x_cpufreq_exit,
	.verify         = pxa27x_cpufreq_verify,
	.target         = pxa27x_cpufreq_target,
	.get            = pxa27x_get_cur_freq,
	.attr           = pxa27x_attr,
	.owner          = THIS_MODULE,
};

/* 
 * validate l, n2 , b, t 
*/
int pxa27x_validate(unsigned int l, unsigned int n2, unsigned int b, 
			unsigned int t, unsigned int a)
{
	printk("L:%u, 2N:%u, b:%u, t:%u\n", l, n2 , b, t);
#ifdef CONFIG_PXA27x_E38
	if (t == 1) {
		printk("HaltTurbo mode is disabled because it may hang the system.\n If you want to enbale it, DO NOT set \"CONFIG_PXA27x_E38=y\"\n");
		return -EINVAL;
	}
#endif
	/* This is 13M mode */
	if (b > 1 || t > 2 || a > 1)
                return -EINVAL;
	if (l == 1 && n2 == 2 && b == 0 && t == 0)
		return 0;
	if ((n2 > N_NUM) || (n2 < 2))
                return -EINVAL;
	if (l < 2 || l > L_NUM)
		return -EINVAL;
       	/* when B = 1, Max L = 16 */
	if (b && l > 16)
		return -EINVAL;
	if (t == 1 && (n2 != 6 && n2 != 8))
		return -EINVAL;
	if (cpufreq_matrix[n2-2][l-2] == 0)
		return -EINVAL;
	return 0;
}

EXPORT_SYMBOL(pxa27x_validate);
/* 
 * The interface for other module to change the frequency. 
 * If freq, n2, turbo, b, a are same as before, no frequency change occurs.
 */
int pxa27x_set_cpufreq(unsigned int l, unsigned int n2, unsigned int b, 
			unsigned int t, unsigned int a)
{
	struct cpufreq_policy policy;
	int ret = 0;
	unsigned int freq;

	down(&pxa27x_freq_sem);
	ret = set_mode(l, n2, b, t, a);
	if (!ret) {
		up(&pxa27x_freq_sem);
		return ret;
	}
	policy.cpu = 0;
	freq = l*((t == 0)? 13000 : ((t == 2)? 6500*n2 : 3250*n2)); 	
	printk("Set frequency to %u with L:%u, 2N:%u, B:%u, %s, A:%u\n", freq, 
		l, n2, b, (t==0)? "Run mode":(t==2)? "Turbo Mode":"Half Turbo mode", a);
	ret = cpufreq_driver_target(&policy, freq, ret);
	up(&pxa27x_freq_sem);
	return ret;
}

EXPORT_SYMBOL(pxa27x_set_cpufreq);

static int __init pxa27x_init_cpufreq(void)
{
	unsigned int clkcfg, ccsr, cccr, l, n2, b, t, a;
	int ret;
	
	ret = cpufreq_register_governor(&pxa27x_cpufreq_gov);
	if (ret)
		return ret;
	clkcfg = pxa27x_read_clkcfg(); 
	ccsr = CCSR;
	cccr = CCCR;
	cpufreq_cur = get_clk_frequency_khz(0);
	l = ccsr & 0x1F;
	n2 = (ccsr >> 7) & 0xF;
	b = clkcfg & (0x1 << 3);
	t = (clkcfg & 0x1)? 2: (clkcfg & (0x1 << 2))? 1:0;
	a = (cccr >> 25) & 0x1;
	set_mode(l, n2, b, t, a);  	 
        return cpufreq_register_driver(&pxa27x_cpufreq_driver);
}

static void __exit pxa27x_exit_cpufreq(void)
{
        cpufreq_unregister_driver(&pxa27x_cpufreq_driver);
	cpufreq_unregister_governor(&pxa27x_cpufreq_gov);
}

MODULE_AUTHOR ("Chao Xie <chao.xie@intel.com>");
MODULE_DESCRIPTION ("cpufreq driver for Intel pxa27x.");
MODULE_LICENSE ("GPL");

late_initcall(pxa27x_init_cpufreq);
module_exit(pxa27x_exit_cpufreq);

