/*
 *      Access
 *
 *      Authors:
 *      Juha Mynttinen            <jmynttin@cc.hut.fi>
 *
 *      $Id: access.c,v 1.12 2002/01/13 23:27:22 antti Exp $
 *
 *      This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */

#include <linux/autoconf.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/in6.h>
#include <linux/spinlock.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#endif

#include <net/mipv6.h>
#include "access.h"
#include "mempool.h"
#include "debug.h"
#include "util.h"

#define ACL_SIZE 30 /* Fixed maximum size now. Adding an entry fails if full */

struct mipv6_access_entry {
	struct mipv6_access_entry *next; /* a linked list */
  	struct in6_addr home_addr;
	__u8 prefix;
	__u8 action;                     /* ALLOW or DENY */
};

struct mipv6_access_list {
	struct mipv6_allocation_pool *entries;
	struct mipv6_access_entry *first;
	__u16 size;
	rwlock_t lock;
};

static struct mipv6_access_list acl_head;

/* deletes all access list entries */
static void empty_access(void)
{
	unsigned long flags;
	struct mipv6_access_entry *entry, *next;

	write_lock_irqsave(&acl_head.lock, flags);

	if (acl_head.entries == NULL || acl_head.first == NULL) {
		DEBUG((DBG_INFO, "empty_access: acl with NULL values"));
		write_unlock_irqrestore(&acl_head.lock, flags);
		return;
	}
	entry = acl_head.first;
	while (entry != NULL) {
		next = entry->next;
		mipv6_free_element(acl_head.entries, entry);
		entry = next;
	}  
	acl_head.first = NULL;

	write_unlock_irqrestore(&acl_head.lock, flags);

	return;
}

int mipv6_is_allowed_home_addr(struct in6_addr *home_addr)
{
	unsigned long flags;
	struct mipv6_access_entry *entry;
	struct in6_addr *addr;
	__u8 prefix;
	
	if (home_addr == NULL) {
		DEBUG((DBG_WARNING, "mipv6_is_allowed_home_addr called with NULL value"));
		return -1;
	}

	read_lock_irqsave(&acl_head.lock, flags);

	entry = acl_head.first;
	if (entry == NULL) {
		read_unlock_irqrestore(&acl_head.lock, flags);
		return 1;
	}

	while (entry != NULL) {
		addr = &entry->home_addr;
		prefix = entry->prefix;
		if(mipv6_prefix_compare(home_addr, addr, prefix)) {
			read_unlock_irqrestore(&acl_head.lock, flags);
			return (entry->action == ALLOW);
		}
		entry = entry->next;
	}

	read_unlock_irqrestore(&acl_head.lock, flags);
	
	/* acl is interpreted so that the last line is DENY all */
	return 0;  
}

/* adds an entry to the acl */
static int add_access_list_entry(struct in6_addr *home_addr,
				 __u8 prefix, __u8 action)
{
	unsigned long flags;
	struct mipv6_access_entry *entry, *temp;

	if (home_addr == NULL || prefix > 128 ||
	    (action != ALLOW && action != DENY)) {
		DEBUG((DBG_ERROR, "Error in add_access_entry parameters"));
		return -1;
	}

	write_lock_irqsave(&acl_head.lock, flags);

	entry = mipv6_allocate_element(acl_head.entries);
	if (entry == NULL) {
		DEBUG((DBG_WARNING, "add_access_list_entry: entry == NULL"));
		write_unlock_irqrestore(&acl_head.lock, flags);
		return -1;
	}

	entry->next = NULL;
	memcpy(&entry->home_addr, home_addr, sizeof(struct in6_addr));
	entry->prefix = prefix;
	entry->action = action;

	if ((temp = acl_head.first) == NULL) {
		acl_head.first = entry;
	} else {
		while (temp->next != NULL) temp = temp->next;
		temp->next = entry;
	}

	write_unlock_irqrestore(&acl_head.lock, flags);

	return 0;
}

int mipv6_acl_add(struct mipv6_acl_record *rule)
{
	if (rule->action == FLUSH) {
		empty_access();
		return 0;
	}

	return add_access_list_entry(
		&rule->mask, rule->prefix_len, rule->action);
}

#ifdef CONFIG_PROC_FS

static char *proc_buf;

static int access_proc_info(char *buffer, char **start, off_t offset,
			    int length)
{
	unsigned long flags;
	__u8 prefix, action;
	int len = 0;
	struct mipv6_access_entry *entry;
	struct in6_addr *address;

	proc_buf = buffer + len;
	read_lock_irqsave(&acl_head.lock, flags); 
	entry = acl_head.first;

	while (entry != NULL) {
		action = entry->action ? '+' : '-';
		prefix = entry->prefix;
		address = &entry->home_addr;
		proc_buf += sprintf(proc_buf, 
				    "%c %08x%08x%08x%08x %02x\n",
				    action,
				    (int)ntohl(address->s6_addr32[0]), 
				    (int)ntohl(address->s6_addr32[1]),
				    (int)ntohl(address->s6_addr32[2]), 
				    (int)ntohl(address->s6_addr32[3]),
				    prefix);
		entry = entry->next;
	}

	read_unlock_irqrestore(&acl_head.lock, flags);

	len = proc_buf - (buffer + len);
	*start = buffer + offset;
	len -= offset;
	if (len > length) len = length;
	
	return len;
}
#endif /* CONFIG_PROC_FS */

int mipv6_initialize_access()
{
	acl_head.entries = mipv6_create_allocation_pool(
		ACL_SIZE,
		sizeof(struct mipv6_access_entry), 
		GFP_KERNEL);
	if (acl_head.entries == NULL) {
		return -1;
	}
  
	acl_head.size = ACL_SIZE;
	acl_head.first = NULL;
	acl_head.lock = RW_LOCK_UNLOCKED;

#ifdef CONFIG_PROC_FS
	proc_net_create("mip6_access", 0, access_proc_info); 
#endif

	return 0;
}

int mipv6_destroy_access(void)
{
	unsigned long flags;
	struct mipv6_access_entry *entry, *next;

	write_lock_irqsave(&acl_head.lock, flags);

	entry = acl_head.first;
	while (entry != NULL) {
		next = entry->next;
		mipv6_free_element(acl_head.entries,
				   entry);
		entry = next;
	}

	mipv6_free_allocation_pool(acl_head.entries);

	write_unlock_irqrestore(&acl_head.lock, flags);

#ifdef CONFIG_PROC_FS
	proc_net_remove("mip6_access");
#endif

	return 0;
}
