/*
 * Host AP (software wireless LAN access point) driver for Intersil Prism2
 * Copyright (c) 2001-2002, SSH Communications Security Corp
 * Jouni Malinen <jkm@ssh.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 *
 * FIX:
 * - shutting down pcmcia-cs while wlan# interface is up does not work
 *   (it does not remove the driver and eventually the kernel will panic when
 *   card is removed); when interface is first configured down, pcmcia-cs can
 *   be shut down; update: this seems to work at least with kernel-based
 *   pcmcia and Linux 2.4.13..
 * - SIOCSIWRATE can set TX rate for adhoc and managed modes, but this is not
 *   currently used in host-based TX rate control; it could be used also for
 *   this (i.e., allow AP to be configured to limit/force TX rate)
 * - there is currently no way of associating TX packets to correct wds device
 *   when TX Exc/OK event occurs, so all tx_packets and some
 *   tx_errors/tx_dropped are added to the main netdevice; using sw_support
 *   field in txdesc might be used to fix this (using Alloc event to increment
 *   tx_packets would need some further info in txfid table)
 *
 * Buffer Access Path (BAP) usage:
 *   BAP0 is used for sending data from the host computer to the card and for
 *   RID read/writes; it is protected with local->baplock (spin_lock_bh) and it
 *   must not be used from hardware interrupt context.
 *   BAP1 is used for receiving data from the card to the host computer; it is
 *   used only in hardware interrupt handler and kernel is assumed to not call
 *   the same handler simultaneously for the same interrupt even on SMP
 *   systems (this removes need for spin_lock protecting BAP1 access.
 */


#include <linux/config.h>
#include <linux/version.h>

/* define PRISM2_MONITOR to add support for raw WLAN frame sniffing */
#define PRISM2_MONITOR

/* define PRISM2_EXTRA_FROM_BAP_TESTS to perform extra tests when copying data
 * from BAP; this is mainly for debugging problems with packet flooding */
/* #define PRISM2_EXTRA_FROM_BAP_TESTS */

/* define PRISM2_USE_TX_INTERRUPT to instruct the driver to handle TX
 * interrupts and increments tx_packets counter only for successfully sent
 * frames. This will produce more exact statistics by removing, e.g., TX errors
 * from the counter, but also more load due to extra interrupts for all TX
 * frames. If this is commented, the extra TX interrupts are masked out and
 * tx_packets counter is incremented from Alloc events. */
/* #define PRISM2_USE_TX_INTERRUPT */

/* define PRISM2_USE_CMD_COMPL_INTERRUPT to instruct the driver to use command
 * completion interrupt for transmit commands instead of using busy wait
 * (of about 150 usec with e.g., Compaq WL100) with each TX frame */
/* #define PRISM2_USE_CMD_COMPL_INTERRUPT */

/* define PRISM2_HOSTAPD to use user space daemon to handle management frames;
 */
/* #define PRISM2_HOSTAPD */


#ifndef __IN_PCMCIA_PACKAGE__

/* Kernel tree PCMCIA compilation */
#include <asm/uaccess.h>

#define add_rx_bytes(stats, n) do { (stats)->rx_bytes += n; } while (0)
#define add_tx_bytes(stats, n) do { (stats)->tx_bytes += n; } while (0)
#define init_dev_name(dev, node) do { } while (0)
#define copy_dev_name(node, dev) strcpy((node).dev_name, (dev)->name)

#else /* __IN_PCMCIA_PACKAGE__ */

/* External pcmcia-cs compilation */
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>

#endif /* __IN_PCMCIA_PACKAGE__ */


#include <asm/io.h>
#include <asm/delay.h>

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/cisreg.h>
#include <pcmcia/ds.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/proc_fs.h>
#include <linux/if_arp.h>

#ifdef PRISM2_MONITOR
#ifdef __IN_PCMCIA_PACKAGE__
/* net/sock.h (at least in 2.2.17) does not like min()/max() macros from
 * pcmcia-cs's kernel compat header */
#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif
#endif /* __IN_PCMCIA_PACKAGE__ */
#include <net/sock.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#endif /* PRISM2_MONITOR */
#include <linux/delay.h>

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
/* 2.2 compatibility */
#ifndef spin_lock_bh
#define spin_lock_bh(lock) spin_lock_irq(lock)
#define spin_unlock_bh(lock) spin_unlock_irq(lock)
#endif
#ifndef __constant_cpu_to_le16
#define __constant_cpu_to_le16 __cpu_to_le16
#endif

#ifndef rtnl_shlock_nowait 
static struct semaphore rtnl_sem = MUTEX;
#define rtnl_shlock_nowait()    down_trylock(&rtnl_sem)
#endif
/* end 2.2 compatibility */
#endif /* kernel < 2.4.0 */


#include "prism2_wlan.h"
#include "prism2_ap.h"


/* #define final_version */


static char *version =
"prism2.c 0.0.0 2002-04-13 (SSH Communications Security Corp, Jouni Malinen)";
static dev_info_t dev_info = "prism2";
static dev_link_t *dev_list = NULL;
static struct proc_dir_entry *prism2_proc = NULL;

MODULE_AUTHOR("SSH Communications Security Corp, Jouni Malinen");
MODULE_DESCRIPTION("Support for Intersil Prism2-based 802.11 wireless LAN "
		   "cards.");
MODULE_SUPPORTED_DEVICE("Intersil Prism2-based WLAN cards");
#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif


static int mtu = 1500;
MODULE_PARM(mtu, "i");
MODULE_PARM_DESC(mtu, "Maximum transfer unit");


static unsigned int irq_mask = 0xdeb8;
MODULE_PARM(irq_mask, "i");

static int irq_list[4] = { -1 };
MODULE_PARM(irq_list, "1-4i");



static int channel = 3;
MODULE_PARM(channel, "i");
MODULE_PARM_DESC(channel, "Initial channel");

static char essid[MAX_SSID_LEN + 1] = "test";
MODULE_PARM(essid, "c" __MODULE_STRING(MAX_SSID_LEN));
MODULE_PARM_DESC(essid, "Host AP's ESSID");

static int iw_mode = IW_MODE_MASTER;
MODULE_PARM(iw_mode, "i");
MODULE_PARM_DESC(iw_mode, "Initial operation mode");

static int beacon_int = 100;
MODULE_PARM(beacon_int, "i");
MODULE_PARM_DESC(beacon_int, "Beacon interval (1 = 1024 usec)");

static int dtim_period = 1;
MODULE_PARM(dtim_period, "i");
MODULE_PARM_DESC(dtim_period, "DTIM period");

static int ignore_cis_vcc = 0;
MODULE_PARM(ignore_cis_vcc, "i");
MODULE_PARM_DESC(ignore_cis_vcc, "Ignore broken CIS VCC entry");

/* Ethernet-II snap header */
static unsigned char snap_header[] =
{ 0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00 };

static const long freq_list[] = { 2412, 2417, 2422, 2427, 2432, 2437, 2442,
				  2447, 2452, 2457, 2462, 2467, 2472, 2484 };
#define FREQ_COUNT (sizeof(freq_list) / sizeof(freq_list[0]))

static void prism2_hw_reset(struct net_device *dev);
static int prism2_wds_add(local_info_t *local, u8 *remote_addr,
			  int rtnl_locked);
static int prism2_wds_del(local_info_t *local, u8 *remote_addr,
			  int rtnl_locked, int do_not_remove);


#ifdef final_version
#define EXTRA_EVENTS_WTERR 0
#else
/* check WTERR events (Wait Time-out) in development versions */
#define EXTRA_EVENTS_WTERR HFA384X_EV_WTERR
#endif

#ifdef PRISM2_USE_TX_INTERRUPT
#define EXTRA_EVENTS_TX HFA384X_EV_TX
#define EXTRA_TX_CTRL HFA384X_TX_CTRL_TX_OK
#else
#define EXTRA_EVENTS_TX 0
#define EXTRA_TX_CTRL 0
#endif

#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
#define EXTRA_EVENTS_CMD HFA384X_EV_CMD
#else
#define EXTRA_EVENTS_CMD 0
#endif

/* event mask, i.e., events that will result in an interrupt */
#define HFA384X_EVENT_MASK \
	(HFA384X_EV_TXEXC | HFA384X_EV_RX | HFA384X_EV_ALLOC | \
	HFA384X_EV_INFO | EXTRA_EVENTS_WTERR | EXTRA_EVENTS_TX | \
	EXTRA_EVENTS_CMD)

#define HFA384X_TX_CTRL_FLAGS \
	(HFA384X_TX_CTRL_802_11 | HFA384X_TX_CTRL_TX_EX | EXTRA_TX_CTRL)


#include "prism2_wep.c"
#include "prism2_ap.c"


/* ca. 1 usec */
#define HFA384X_CMD_BUSY_TIMEOUT 1000
#define HFA384X_BAP_BUSY_TIMEOUT 5000

/* ca. 10 usec */
#define HFA384X_INIT_TIMEOUT 50000
#define HFA384X_CMD_COMPL_TIMEOUT 20000
#define HFA384X_ALLOC_COMPL_TIMEOUT 1000


#define TX_TIMEOUT (2 * HZ)
#define PRISM2_MAX_FRAME_SIZE 2304
#define PRISM2_MIN_MTU 256
/* FIX: */
#define PRISM2_MAX_MTU (PRISM2_MAX_FRAME_SIZE - sizeof(snap_header))


static void prism2_detach(dev_link_t *link);
static void prism2_release(u_long arg);
static int prism2_event(event_t event, int priority,
			event_callback_args_t *args);

#ifndef final_version
/* magic value written to SWSUPPORT0 reg. for detecting whether card is still
 * present */
#define HFA384X_MAGIC 0x8A32
#endif


#define HFA384X_OUTB(v,a) outb((v), dev->base_addr + (a))
#define HFA384X_INB(a) inb(dev->base_addr + (a))
#define HFA384X_OUTW(v,a) outw((v), dev->base_addr + (a))
#define HFA384X_INW(a) inw(dev->base_addr + (a))
#define HFA384X_INSW(a, buf, wc) insw(dev->base_addr + (a), buf, wc)
#define HFA384X_OUTSW(a, buf, wc) outsw(dev->base_addr + (a), buf, wc)



static u16 hfa384x_read_reg(struct net_device *dev, u16 reg)
{
	return HFA384X_INW(reg);
}


static int hfa384x_cmd(struct net_device *dev, u16 cmd, u16 param0,
		       u16 *param1, u16 *resp0)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int tries, res;
	unsigned long flags;
	u16 reg;

	spin_lock_irqsave(&local->cmdlock, flags);

	/* wait until busy bit is clear */
	tries = HFA384X_CMD_BUSY_TIMEOUT;
	while (HFA384X_INW(HFA384X_CMD_OFF) & HFA384X_CMD_BUSY && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_CMD_OFF);
		spin_unlock_irqrestore(&local->cmdlock, flags);
		printk("%s: hfa384x_cmd - timeout - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	/* write command */
	HFA384X_OUTW(param0, HFA384X_PARAM0_OFF);
	if (param1 != NULL)
		HFA384X_OUTW(*param1, HFA384X_PARAM1_OFF);
	HFA384X_OUTW(cmd, HFA384X_CMD_OFF);

	/* wait for command completion */
	tries = HFA384X_CMD_COMPL_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_CMD) &&
	       tries > 0) {
		tries--;
		udelay(10);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_EVSTAT_OFF);
		spin_unlock_irqrestore(&local->cmdlock, flags);
		printk("%s: hfa384x_cmd - timeout2 - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	if (resp0 != NULL)
		*resp0 = HFA384X_INW(HFA384X_RESP0_OFF);
	res = (HFA384X_INW(HFA384X_STATUS_OFF) &
	       (BIT(14) | BIT(13) | BIT(12) | BIT(11) | BIT(10) | BIT(9) |
		BIT(8))) >> 8;
#ifndef final_version
	if (res) {
		u16 resp0 = HFA384X_INW(HFA384X_RESP0_OFF);
		printk("%s: CMD=0x%04x => res=0x%02x, resp0=0x%04x\n",
		       dev->name, cmd, res, resp0);
	}
#endif

	HFA384X_OUTW(HFA384X_EV_CMD, HFA384X_EVACK_OFF);

	spin_unlock_irqrestore(&local->cmdlock, flags);
	return res;
}


static int hfa384x_cmd_no_wait(struct net_device *dev, u16 cmd, u16 param0)
{
	int tries;
	u16 reg;

	/* wait until busy bit is clear */
	tries = HFA384X_CMD_BUSY_TIMEOUT;
	while (HFA384X_INW(HFA384X_CMD_OFF) & HFA384X_CMD_BUSY && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		reg = HFA384X_INW(HFA384X_CMD_OFF);
		printk("%s: hfa384x_cmd - timeout - reg=0x%04x\n", dev->name,
		       reg);
		return -ETIMEDOUT;
	}

	/* write command */
	HFA384X_OUTW(param0, HFA384X_PARAM0_OFF);
	HFA384X_OUTW(cmd, HFA384X_CMD_OFF);

	return 0;
}


/* Offset must be even */
int hfa384x_setup_bap(struct net_device *dev, u16 bap, u16 id, int offset)
{
	u16 o_off, tmp;
	int tries, ret = 0;

	if (offset % 2 || bap > 1)
		return -EINVAL;

	o_off = (bap == 1) ? HFA384X_OFFSET1_OFF : HFA384X_OFFSET0_OFF;

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk("%s: hfa384x_setup_bap - timeout1\n", dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}

	HFA384X_OUTW(id,
		     (bap == 1) ? HFA384X_SELECT1_OFF : HFA384X_SELECT0_OFF);

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk("%s: hfa384x_setup_bap - timeout2\n", dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}

	HFA384X_OUTW(offset, o_off);

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while (((tmp = HFA384X_INW(o_off)) & HFA384X_OFFSET_BUSY) &&
	       tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0) {
		printk("%s: hfa384x_setup_bap - timeout3\n", dev->name);
		ret = -ETIMEDOUT;
		goto out;
	}
#ifndef final_version
	else if (tmp & HFA384X_OFFSET_ERR) {
		printk("%s: hfa384x_setup_bap - offset error "
		       "(%d,%d,%d,0x%04x)\n",
		       dev->name, bap, id, offset, tmp);
		ret = tmp;
	}
#endif

 out:
	return ret;
}


int hfa384x_from_bap(struct net_device *dev, u16 bap, void *buf, int len)
{
	u16 d_off;
	u16 *pos;

#ifdef PRISM2_EXTRA_FROM_BAP_TESTS

#ifndef final_version
	u16 o_off;
	int tries;

	o_off = (bap == 1) ? HFA384X_OFFSET1_OFF : HFA384X_OFFSET0_OFF;

	tries = HFA384X_BAP_BUSY_TIMEOUT;
	while ((HFA384X_INW(o_off) & HFA384X_OFFSET_BUSY) && tries > 0) {
		tries--;
		udelay(1);
	}
	if (tries == 0)
		printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - timeout\n",
		       dev->name, bap);
	else if (tries < HFA384X_BAP_BUSY_TIMEOUT)
		printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - offset was "
		       "busy\n", dev->name, bap);
#endif

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;
	for ( ; len > 1; len -= 2) {
#ifndef final_version
		if (HFA384X_INW(o_off) & HFA384X_OFFSET_BUSY)
			printk(KERN_DEBUG "%s: hfa384x_from_bap(%d) - BAP "
			       "busy during read\n", dev->name, bap);
#endif
		*pos++ = __le16_to_cpu(HFA384X_INW(d_off));
	}
	if (len > 0)
		*((u8 *) pos) = HFA384X_INB(d_off);

#else /* PRISM2_EXTRA_FROM_BAP_TESTS */

	int word_count;

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;

	word_count = len / 2;
	if (word_count)
		HFA384X_INSW(d_off, buf, word_count);

	if (len & 1)
		*((u8 *)(pos + word_count)) = HFA384X_INB(d_off);

#endif /* PRISM2_EXTRA_FROM_BAP_TESTS */

	return 0;
}


int hfa384x_to_bap(struct net_device *dev, u16 bap, void *buf, int len)
{
	u16 d_off;
	u16 *pos;
	int word_count;

	d_off = (bap == 1) ? HFA384X_DATA1_OFF : HFA384X_DATA0_OFF;

	pos = (u16 *) buf;
	word_count = len / 2;
	if (word_count)
		HFA384X_OUTSW(d_off, buf, word_count);
	if (len & 1)
		HFA384X_OUTB(*((char *) (pos + word_count)), d_off);

	return 0;
}


static int hfa384x_get_rid(struct net_device *dev, u16 rid, void *buf, int len,
			   int exact_len)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res, rlen = 0;
	struct hfa384x_rid_hdr rec;

	res = hfa384x_cmd(dev, HFA384X_CMDCODE_ACCESS, rid, NULL, NULL);
	if (res) {
		printk("%s: hfa384x_get_rid: CMDCODE_ACCESS failed (res=%d)\n",
		       dev->name, res);
		return res;
	}

	spin_lock_bh(&local->baplock);

	res = hfa384x_setup_bap(dev, BAP0, rid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP0, &rec, sizeof(rec));
	if (res) {
		printk("%s: hfa384x_get_rid - from BAP0 failed\n",
		       dev->name);
		goto fail;
	}

	rlen = (__le16_to_cpu(rec.len) - 1) * 2;
	if (exact_len && rlen != len) {
		printk("RID len mismatch: rid=0x%04x, len=%d (expected %d)\n",
		       rid, rlen, len);
		res = -ENODATA;
		goto fail;
	}

	res = hfa384x_from_bap(dev, BAP0, buf, len);
	if (res) {
		printk("%s: hfa384x_get_rid - from BAP0(2) failed\n",
		       dev->name);
		goto fail;
	}

 fail:
	spin_unlock_bh(&local->baplock);

	if (res) {
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return res;
	}
	return rlen;
}


static int hfa384x_set_rid(struct net_device *dev, u16 rid, void *buf, int len)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res;
	struct hfa384x_rid_hdr rec;

	rec.rid = __cpu_to_le16(rid);
	/* RID len in words and +1 for rec.rid */
	rec.len = __cpu_to_le16(len / 2 + len % 2 + 1);

	spin_lock_bh(&local->baplock);

	res = hfa384x_setup_bap(dev, BAP0, rid, 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &rec, sizeof(rec));
	if (res) {
		printk("%s: hfa384x_set_rid - to BAP0 failed\n",
		       dev->name);
		goto fail;
	}

	res = hfa384x_to_bap(dev, BAP0, buf, len);
	if (res) {
		printk("%s: hfa384x_set_rid - to BAP0(2) failed\n", dev->name);
		goto fail;
	}

	res = hfa384x_cmd(dev, HFA384X_CMDCODE_ACCESS_WRITE, rid, NULL, NULL);
	if (res) {
		printk("%s: hfa384x_set_rid: CMDCODE_ACCESS_WRITE failed "
		       "(res=%d)\n", dev->name, res);
		goto fail;
	}

 fail:
	spin_unlock_bh(&local->baplock);

	if (res == -ETIMEDOUT)
		prism2_hw_reset(dev);

	return res;
}


static void hfa384x_disable_interrupts(struct net_device *dev)
{
	/* disable interrupts and clear event status */
	HFA384X_OUTW(0, HFA384X_INTEN_OFF);
	HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
}


static void hfa384x_enable_interrupts(struct net_device *dev)
{
	/* ack pending events and enable interrupts from selected events */
	HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
	HFA384X_OUTW(HFA384X_EVENT_MASK, HFA384X_INTEN_OFF);
}


static u16 hfa384x_allocate_fid(struct net_device *dev, int len)
{
	int tries;
	u16 fid;

	if (hfa384x_cmd(dev, HFA384X_CMDCODE_ALLOC, len, NULL, NULL)) {
		printk("%s: cannot allocate fid, len=%d\n", dev->name, len);
		return 0xffff;
	}

	tries = HFA384X_ALLOC_COMPL_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_ALLOC) &&
	       tries > 0) {
		tries--;
		udelay(10);
	}
	if (tries == 0) {
		printk("%s: fid allocate, len=%d - timeout\n", dev->name, len);
		return 0xffff;
	}

	fid = HFA384X_INW(HFA384X_ALLOCFID_OFF);
	HFA384X_OUTW(HFA384X_EV_ALLOC, HFA384X_EVACK_OFF);

	return fid;
}


static int prism2_reset_port(struct net_device *dev)
{
	int res;
	res = hfa384x_cmd(dev, HFA384X_CMDCODE_DISABLE, 0,
			  NULL, NULL);
	if (!res)
		res = hfa384x_cmd(dev, HFA384X_CMDCODE_ENABLE, 0,
				  NULL, NULL);

	return res;
}


int prism2_set_tim(struct net_device *dev, u16 aid, int set)
{
	u16 tmp;

	if (set) {
		PDEBUG(DEBUG_PS2, "Setting TIM bit for AID %i\n", aid);
		tmp = __cpu_to_le16(0x8000 | aid);
	} else {
		PDEBUG(DEBUG_PS2, "Clearing TIM bit for AID %i\n", aid);
		tmp = __cpu_to_le16(aid);
	}
	if (hfa384x_set_rid(dev, HFA384X_RID_CNFTIMCTRL, &tmp, 2)) {
		PDEBUG(DEBUG_PS, "TIM AID %d setting (set=%d) failed\n",
		       aid, set);
		return -1;
	}
	return 0;
}


static void prism2_get_version_info(struct net_device *dev, u16 rid,
				    const char *txt)
{
	struct hfa384x_comp_ident comp;

	if (hfa384x_get_rid(dev, rid, &comp, sizeof(comp), 1) < 0)
		printk("Could not get RID for component %s\n", txt);
	else
		printk("%s: %s: id=0x%02x v%d.%d.%d\n", dev->name, txt,
		       __le16_to_cpu(comp.id), __le16_to_cpu(comp.major),
		       __le16_to_cpu(comp.minor), __le16_to_cpu(comp.variant));
}


static int prism2_set_string(struct net_device *dev, int rid, const char *val)
{
	char buf[MAX_SSID_LEN + 2];
	int len;

	len = strlen(val);
	if (len > MAX_SSID_LEN)
		return -1;
	memset(buf, 0, sizeof(buf));
	buf[0] = len; /* little endian 16 bit word */
	memcpy(buf + 2, val, len);

	return hfa384x_set_rid(dev, rid, &buf, MAX_SSID_LEN + 2);
}


static int prism2_setup_rids(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 tmp;
	int ret = 0;

	if (local->iw_mode == IW_MODE_ADHOC && local->pseudo_adhoc)
		tmp = HFA384X_PORTTYPE_PSEUDO_IBSS;
	else if (local->iw_mode == IW_MODE_ADHOC)
		tmp = HFA384X_PORTTYPE_IBSS;
	else if (local->iw_mode == IW_MODE_INFRA)
		tmp = HFA384X_PORTTYPE_BSS;
	else
		tmp = HFA384X_PORTTYPE_HOSTAP;
	cpu_to_le16s(&tmp);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &tmp, 2);
	if (ret) {
		printk("%s: Port type setting to %d failed\n", dev->name,
		       __le16_to_cpu(tmp));
		goto fail;
	}

	ret = prism2_set_string(dev, HFA384X_RID_CNFOWNSSID, local->essid);
	if (ret) {
		printk("%s: AP own SSID setting failed\n", dev->name);
		goto fail;
	}

	tmp = __constant_cpu_to_le16(PRISM2_DATA_MAXLEN);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFMAXDATALEN, &tmp, 2);
	if (ret) {
		printk("%s: MAC data length setting to %d failed\n",
		       dev->name, PRISM2_DATA_MAXLEN);
		goto fail;
	}

	if (hfa384x_get_rid(dev, HFA384X_RID_CHANNELLIST, &tmp, 2, 1) < 0) {
		printk("%s: Channel list read failed\n", dev->name);
		ret = -EINVAL;
		goto fail;
	}
	local->channel_mask = __le16_to_cpu(tmp);

	if (local->channel < 1 || local->channel > FREQ_COUNT ||
	    !(local->channel_mask & (1 << (local->channel - 1)))) {
		printk(KERN_WARNING "%s: Channel setting out of range "
		       "(%d)!\n", dev->name, local->channel);
		ret = -EBUSY;
		goto fail;
	}

	tmp = __cpu_to_le16(local->channel);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFOWNCHANNEL, &tmp, 2);
	if (ret) {
		printk("%s: Channel setting to %d failed\n",
		       dev->name, local->channel);
		goto fail;
	}

	tmp = __cpu_to_le16(local->beacon_int);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFBEACONINT, &tmp, 2);
	if (ret) {
		printk("%s: Beacon interval setting to %d failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	tmp = __constant_cpu_to_le16(HFA384X_RATES_1MBPS |
				     HFA384X_RATES_2MBPS);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFBASICRATES, &tmp, 2);
	if (ret) {
		printk("%s: Basic rates setting to 0x%04x failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	tmp = __constant_cpu_to_le16(HFA384X_RATES_1MBPS |
				     HFA384X_RATES_2MBPS |
				     HFA384X_RATES_5MBPS |
				     HFA384X_RATES_11MBPS);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFSUPPORTEDRATES, &tmp, 2);
	if (ret) {
		printk("%s: Supported rates setting to 0x%04x failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	tmp = __cpu_to_le16(local->dtim_period);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CNFOWNDTIMPERIOD, &tmp, 2);
	if (ret) {
		printk("%s: DTIM period setting to %d failed\n",
		       dev->name, __le16_to_cpu(tmp));
		if (!local->is_symbol && !local->is_lucent)
			goto fail;
	}

	ret = prism2_set_string(dev, HFA384X_RID_CNFDESIREDSSID, local->essid);
	if (ret) {
		printk("%s: Desired SSID setting failed\n", dev->name);
		goto fail;
	}

	/* Setup TXRateControl, defaults to allow use of 1, 2, 5.5, and
	 * 11 Mbps in automatic TX rate fallback */
	if (local->tx_rate_control == 0) {
		local->tx_rate_control =
			HFA384X_RATES_1MBPS |
			HFA384X_RATES_2MBPS |
			HFA384X_RATES_5MBPS |
			HFA384X_RATES_11MBPS;
	}
	tmp = __cpu_to_le16(local->tx_rate_control);
	ret = hfa384x_set_rid(dev, HFA384X_RID_TXRATECONTROL, &tmp, 2);
	if (ret) {
		printk("%s: TXRateControl setting to %d failed\n",
		       dev->name, __le16_to_cpu(tmp));
		goto fail;
	}

	tmp = __constant_cpu_to_le16(1);
	ret = hfa384x_set_rid(dev, HFA384X_RID_CREATEIBSS, &tmp, 2);
	if (ret) {
		printk("%s: Create IBSS setting to 1 failed\n", dev->name);
	}

	if (local->name_set)
		(void) prism2_set_string(dev, HFA384X_RID_CNFOWNNAME,
					 local->name);


 fail:
	return ret;
}


static int prism2_hw_config(struct net_device *dev, int initial)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int ret, i, len;

	PDEBUG(DEBUG_FLOW, "prism2_hw_config()\n");

	/* initialize HFA 384x */
	ret = hfa384x_cmd_no_wait(dev, HFA384X_CMDCODE_INIT, 0);
	if (ret) {
		printk("%s: first command failed - is the card compatible?\n",
		       dev_info);
		goto failed;
	}
	i = HFA384X_INIT_TIMEOUT;
	while (!(HFA384X_INW(HFA384X_EVSTAT_OFF) & HFA384X_EV_CMD) && i > 0) {
		i--;
		udelay(10);
	}
	if (i == 0) {
		printk("%s: card initialization timed out\n", dev_info);
		goto failed;
	}
	printk(KERN_DEBUG "prism2_hw_config: initialized in %d iterations\n",
	       HFA384X_INIT_TIMEOUT - i);
	HFA384X_OUTW(HFA384X_EV_CMD, HFA384X_EVACK_OFF);

	hfa384x_disable_interrupts(dev);

#ifndef final_version
	HFA384X_OUTW(HFA384X_MAGIC, HFA384X_SWSUPPORT0_OFF);
	if (HFA384X_INW(HFA384X_SWSUPPORT0_OFF) != HFA384X_MAGIC) {
		printk("SWSUPPORT0 write/read failed: %04X != %04X\n",
		       HFA384X_INW(HFA384X_SWSUPPORT0_OFF), HFA384X_MAGIC);
		goto failed;
	}
#endif

	/* allocate TX FIDs */
	len = local->is_symbol ? 1600 : PRISM2_TXFID_LEN;
	for (i = 0; i < PRISM2_TXFID_COUNT; i++) {
		local->txfid[i] = hfa384x_allocate_fid(dev, len);
		if (local->txfid[i] == 0xffff)
			goto failed;
		local->intransmitfid[i] = PRISM2_TXFID_EMPTY;
	}
#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
	local->last_txfid_idx = PRISM2_DUMMY_FID;
#endif

	if (initial) {
		/* get card version information */
		prism2_get_version_info(dev, HFA384X_RID_NICID, "NIC");
		prism2_get_version_info(dev, HFA384X_RID_PRIID, "PRI");
		prism2_get_version_info(dev, HFA384X_RID_STAID, "STA");

		if (hfa384x_get_rid(dev, HFA384X_RID_CNFOWNMACADDR,
				    &dev->dev_addr, 6, 1) < 0) {
			printk("%s: could not get own MAC address\n",
			       dev->name);
		}
#ifdef PRISM2_HOSTAPD
		memcpy(local->apdev->dev_addr, dev->dev_addr, ETH_ALEN);
#endif /* PRISM2_HOSTAPD */
	}

	prism2_setup_rids(dev);

	ret = hfa384x_cmd(dev, HFA384X_CMDCODE_ENABLE, 0, NULL, NULL);
	if (ret) {
		printk("%s: MAC port 0 enabling failed\n", dev->name);
		goto failed;
	}
	/* at least D-Link DWL-650 seems to require additional port reset
	 * before it starts acting as an AP, so reset port automatically
	 * here just in case */
	if (prism2_reset_port(dev)) {
		printk("%s: MAC port 0 reseting failed\n", dev->name);
		goto failed;
	}

	local->hw_ready = 1;
	local->hw_reset_tries = 0;
	hfa384x_enable_interrupts(dev);
	return 0;

 failed:
	printk(KERN_WARNING "%s: Initialization failed\n", dev_info);
	return 1;
}


static void prism2_hw_shutdown(struct net_device *dev, int no_disable)
{
	local_info_t *local = (local_info_t *) dev->priv;

	local->hw_ready = 0;
	if (local->link != NULL &&
	    (local->link->state & DEV_PRESENT) != DEV_PRESENT) {
		printk("%s: card already removed\n", dev_info);
		return;
	}
	if (local->link != NULL && !(local->link->state & DEV_CONFIG)) {
		printk("%s: card not configured\n", dev_info);
		return;
	}

	hfa384x_disable_interrupts(dev);

	if (!no_disable &&
	    hfa384x_cmd(dev, HFA384X_CMDCODE_DISABLE, 0, NULL, NULL))
		printk(KERN_WARNING "%s: Shutdown failed\n", dev_info);
}

static void prism2_cor_sreset(local_info_t *local)
{
	int res;
	conf_reg_t reg;

	reg.Function = 0;
	reg.Action = CS_READ;
	reg.Offset = CISREG_COR;
	reg.Value = 0;
	res = CardServices(AccessConfigurationRegister, local->link->handle,
			   &reg);
	if (res != CS_SUCCESS) {
		printk(KERN_DEBUG "prism2_cor_sreset failed 1 (%d)\n", res);
		return;
	}
	printk(KERN_DEBUG "prism2_cor_sreset: original COR %02x\n", reg.Value);

	reg.Action = CS_WRITE;
	reg.Value |= COR_SOFT_RESET;
	res = CardServices(AccessConfigurationRegister, local->link->handle,
			   &reg);
	if (res != CS_SUCCESS) {
		printk(KERN_DEBUG "prism2_cor_sreset failed 1 (%d)\n", res);
		return;
	}

	mdelay(1);

	reg.Value &= ~COR_SOFT_RESET;
	res = CardServices(AccessConfigurationRegister, local->link->handle,
			   &reg);
	if (res != CS_SUCCESS) {
		printk(KERN_DEBUG "prism2_cor_sreset failed 1 (%d)\n", res);
		return;
	}

	mdelay(1);
}

static void prism2_hw_reset(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

#if 0
	static long last_reset = 0;

	/* do not reset card more than once per second to avoid ending up in a
	 * busy loop reseting the card */
	if (last_reset + HZ >= jiffies)
		return;
	last_reset = jiffies;
#endif

	if (local->hw_resetting) {
		printk(KERN_WARNING "%s: %s: already resetting card - "
		       "ignoring reset request\n", dev_info, dev->name);
		return;
	}

	local->hw_reset_tries++;
	if (local->hw_reset_tries > 10) {
		printk(KERN_WARNING "%s: too many reset tries, skipping\n",
		       dev->name);
		return;
	}

	printk(KERN_WARNING "%s: %s: resetting card\n", dev_info, dev->name);
	local->hw_resetting = 1;
	hfa384x_disable_interrupts(dev);
	prism2_cor_sreset(local);
	prism2_hw_shutdown(dev, 1);
	prism2_hw_config(dev, 0);
	local->hw_resetting = 0;
}


static void cs_error(client_handle_t handle, int func, int ret)
{
	error_info_t err = { func, ret };
	CardServices(ReportError, handle, &err);
}


static struct net_device_stats *prism2_get_stats(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	if (local->dev != dev) {
		prism2_wds_info_t *wds = (prism2_wds_info_t *) dev;
		return &wds->stats;
	}
	return &local->stats;
}


#ifdef WIRELESS_EXT
static struct iw_statistics *prism2_get_wireless_stats(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	local->wstats.status = 0;
	local->wstats.discard.code =
		local->comm_tallies.rx_discards_wep_undecryptable;
	local->wstats.discard.misc =
		local->comm_tallies.rx_fcs_errors +
		local->comm_tallies.rx_discards_no_buffer +
		local->comm_tallies.tx_discards_wrong_sa;

	if (local->iw_mode != IW_MODE_MASTER) {
		struct hfa384x_comms_quality sq;
		if (hfa384x_get_rid(local->dev,
					   HFA384X_RID_COMMSQUALITY,
					   &sq, sizeof(sq), 1)) {
			local->wstats.qual.qual = le16_to_cpu(sq.comm_qual);
			local->wstats.qual.level = HFA384X_LEVEL_TO_dBm(
				le16_to_cpu(sq.signal_level));
			local->wstats.qual.noise = HFA384X_LEVEL_TO_dBm(
				le16_to_cpu(sq.noise_level));
			local->wstats.qual.updated = 7;
		}
	}

	return &local->wstats;
}
#endif /* WIRELESS_EXT */


/* wake all netif queues in use */
static void prism2_netif_wake_queues(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	prism2_wds_info_t *wds;

	if (local->dev)
		netif_wake_queue(local->dev);

#ifdef PRISM2_HOSTAPD
	if (local->apdev)
		netif_wake_queue(local->apdev);
#endif /* PRISM2_HOSTAPD */

	wds = local->wds;
	while (wds != NULL) {
		netif_wake_queue(&wds->dev);
		wds = wds->next;
	}
}


/* stop all netif queues in use */
static void prism2_netif_stop_queues(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	prism2_wds_info_t *wds;

	if (local->dev)
		netif_stop_queue(local->dev);

#ifdef PRISM2_HOSTAPD
	if (local->apdev)
		netif_stop_queue(local->apdev);
#endif /* PRISM2_HOSTAPD */

	spin_lock(&local->wdslock);
	wds = local->wds;
	while (wds != NULL) {
		netif_stop_queue(&wds->dev);
		wds = wds->next;
	}
	spin_unlock(&local->wdslock);
}


static int prism2_open(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	PDEBUG(DEBUG_FLOW, "%s: prism2_open\n", dev->name);
	netif_device_attach(dev);
	netif_start_queue(dev);
	local->link->open++;

	MOD_INC_USE_COUNT;

	return 0;
}


static int prism2_close(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;

	PDEBUG(DEBUG_FLOW, "%s: prism2_close\n", dev->name);
	if (!local->link->open) {
		printk(KERN_WARNING "%s: prism2_close(): link not open?!\n",
		       dev->name);
		return 0;
	}

	local->link->open--;
	MOD_DEC_USE_COUNT;

	if (netif_running(dev)) {
		netif_stop_queue(dev);
		netif_device_detach(dev);
	} else
		if (local->link->state & DEV_STALE_CONFIG)
			mod_timer(&local->link->release, jiffies + HZ / 20);

	return 0;
}


static int prism2_change_mtu(struct net_device *dev, int new_mtu)
{
	if (new_mtu < PRISM2_MIN_MTU || new_mtu > PRISM2_MAX_MTU)
		return -EINVAL;

	dev->mtu = new_mtu;
	return 0;
}


#ifdef HAVE_TX_TIMEOUT
static void prism2_tx_timeout(struct net_device *dev)
{
	printk(KERN_WARNING "%s: %s Tx timed out! Resetting card\n",
	       dev_info, dev->name);

	prism2_hw_reset(dev);

	if (netif_queue_stopped(dev)) {
		int i;
		local_info_t *local = (local_info_t *) dev->priv;

		for (i = 0; i < PRISM2_TXFID_COUNT; i++)
			if (local->intransmitfid[i] == PRISM2_TXFID_EMPTY) {
				PDEBUG(DEBUG_EXTRA, "prism2_tx_timeout: "
				       "wake up queue\n");
				netif_wake_queue(dev);
				break;
			}
	}
}
#endif /* HAVE_TX_TIMEOUT */


int prism2_get_txfid_idx(local_info_t *local)
{
	int idx;
	unsigned long flags;

	spin_lock_irqsave(&local->txfidlock, flags);
	idx = local->next_txfid;
	do {
		if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY) {
			local->intransmitfid[idx] = PRISM2_TXFID_RESERVED;
			spin_unlock_irqrestore(&local->txfidlock, flags);
			return idx;
		}
		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_txfid);
	spin_unlock_irqrestore(&local->txfidlock, flags);

	PDEBUG(DEBUG_EXTRA2, "prism2_get_txfid_idx: no room in txfid buf: "
	       "packet dropped\n");
	local->stats.tx_dropped++;

	return -1;
}


#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT

/* Called only from software IRQ */
int prism2_transmit(struct net_device *dev, int idx)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res;

	/* The driver tries to stop netif queue so that there would not be
	 * more than one attempt to transmit frames going on; check that this
	 * is really the case */
	if (local->last_txfid_idx != PRISM2_DUMMY_FID) {
		int tries = 1000;
		while (tries > 0 && local->last_txfid_idx != PRISM2_DUMMY_FID)
		{
			udelay(10);
			tries--;
		}

		if (local->last_txfid_idx != PRISM2_DUMMY_FID) {
			printk(KERN_DEBUG "%s: timeout on waiting pending "
			       "transmit command..\n", dev->name);
			return 1;
		}
		printk(KERN_DEBUG "%s: %d x 10 usec cleared pending TX\n",
		       dev->name, 1000 - tries);
	}

	/* stop the queue for the time that last_txfid_idx is in use */
	prism2_netif_stop_queues(dev);

	local->last_txfid_idx = idx;

	/* transmit packet */
	res = hfa384x_cmd_no_wait(
		dev,
		HFA384X_CMDCODE_TRANSMIT | HFA384X_CMD_TX_RECLAIM,
		local->txfid[idx]);

	if (res) {
		struct net_device_stats *stats;
		printk(KERN_DEBUG "%s: prism2_transmit: CMDCODE_TRANSMIT "
		       "failed (res=%d)\n", dev->name, res);
		stats = prism2_get_stats(dev);
		stats->tx_dropped++;
		return 1;
	}
	dev->trans_start = jiffies;

	/* Since we did not wait for command completion, the card continues
	 * to process on the background and we will finish handling when
	 * command completion event is handled (prism2_cmd_ev() function) */

	return 0;
}


/* Called only from hardware IRQ */
static void prism2_cmd_ev(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int idx = local->last_txfid_idx;
	u16 resp0;

	if (local->last_txfid_idx == PRISM2_DUMMY_FID) {
		/* no pending transmit command; should not happen since we
		 * wait for command completion (and ACK event) with every other
		 * command */
		printk(KERN_DEBUG "%s: CMD EV - no pending transmit\n",
		       dev->name);
		return;
	}

	local->last_txfid_idx = PRISM2_DUMMY_FID;

	resp0 = HFA384X_INW(HFA384X_RESP0_OFF);

	if (netif_queue_stopped(dev)) {
		/* last_txfid_idx is free for next TX, so wake up queue that
		 * was stopped in prism2_transmit() */
		prism2_netif_wake_queues(dev);
	}

	/* FIX: is some locking needed for txfid data? */

	/* With reclaim, Resp0 contains new txfid for transmit; the old txfid
	 * will be automatically allocated for the next TX frame */
	local->intransmitfid[idx] = resp0;

	PDEBUG(DEBUG_FID, "%s: prism2_cmd_ev: txfid[%d]=0x%04x, resp0=0x%04x, "
	       "transmit_txfid=0x%04x\n", dev->name, idx, local->txfid[idx],
	       resp0, local->intransmitfid[local->next_txfid]);

	idx++;
	if (idx >= PRISM2_TXFID_COUNT)
		idx = 0;
	local->next_txfid = idx;

	/* check if all TX buffers are occupied */
	do {
		if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY) {
			return;
		}
		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_txfid);

	/* no empty TX buffers, stop queue */
	prism2_netif_stop_queues(dev);
}

#else /* PRISM2_USE_CMD_COMPL_INTERRUPT */

/* Called only from software IRQ */
int prism2_transmit(struct net_device *dev, int idx)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res;
	unsigned long flags;
	u16 resp0;

	spin_lock_irqsave(&local->txfidlock, flags);

	/* transmit packet */
	res = hfa384x_cmd(dev,
			  HFA384X_CMDCODE_TRANSMIT | HFA384X_CMD_TX_RECLAIM,
			  local->txfid[idx], NULL, &resp0);
	if (res) {
		struct net_device_stats *stats;
		spin_unlock_irqrestore(&local->txfidlock, flags);
		printk("%s: prism2_transmit: CMDCODE_TRANSMIT failed "
		       "(res=%d, resp0=0x%04x)\n", dev->name, res, resp0);
		stats = prism2_get_stats(dev);
		stats->tx_dropped++;
		return 1;
	}
	dev->trans_start = jiffies;

	/* With reclaim, Resp0 contains new txfid for transmit; the old txfid
	 * will be automatically allocated for the next TX frame */
	local->intransmitfid[idx] = resp0;

	PDEBUG(DEBUG_FID, "prism2_transmit: [%d] txfid=0x%04x, "
	       "transmit_txfid=0x%04x\n", idx, local->txfid[idx],
	       local->intransmitfid[local->next_txfid]);

	idx++;
	if (idx >= PRISM2_TXFID_COUNT)
		idx = 0;
	local->next_txfid = idx;

	/* check if all TX buffers are occupied */
	do {
		if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY) {
			spin_unlock_irqrestore(&local->txfidlock, flags);
			return 0;
		}
		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_txfid);
	spin_unlock_irqrestore(&local->txfidlock, flags);

	/* no empty TX buffers, stop queue */
	prism2_netif_stop_queues(dev);

	return 0;
}

#endif /* PRISM2_USE_CMD_COMPL_INTERRUPT */


void prism2_dump_tx_header(const char *name, const struct hfa384x_tx_frame *tx)
{
	u16 fc;

	printk(KERN_DEBUG "%s: TX status=0x%04x retry_count=%d tx_rate=%d "
	       "tx_control=0x%04x; jiffies=%ld\n",
	       name, __le16_to_cpu(tx->status), tx->retry_count, tx->tx_rate,
	       __le16_to_cpu(tx->tx_control), jiffies);

	fc = __le16_to_cpu(tx->frame_control);
	printk(KERN_DEBUG "   FC=0x%04x (type=%d:%d) dur=0x%04x seq=0x%04x "
	       "data_len=%d%s%s\n",
	       fc, WLAN_FC_GET_TYPE(fc), WLAN_FC_GET_STYPE(fc),
	       __le16_to_cpu(tx->duration_id), __le16_to_cpu(tx->seq_ctrl),
	       __le16_to_cpu(tx->data_len),
	       fc & WLAN_FC_TODS ? " [ToDS]" : "",
	       fc & WLAN_FC_FROMDS ? " [FromDS]" : "");

	printk(KERN_DEBUG "   A1=" MACSTR " A2=" MACSTR " A3=" MACSTR " A4="
	       MACSTR "\n",
	       MAC2STR(tx->addr1), MAC2STR(tx->addr2), MAC2STR(tx->addr3),
	       MAC2STR(tx->addr4));

	printk(KERN_DEBUG "   dst=" MACSTR " src=" MACSTR " len=%d\n",
	       MAC2STR(tx->dst_addr), MAC2STR(tx->src_addr),
	       __be16_to_cpu(tx->len));
}

/* Called only from software IRQ */
static int prism2_tx(struct sk_buff *skb, struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int res, idx = -1, ret = 1, data_len;
	struct hfa384x_tx_frame txdesc;
	struct sta_info *sta = NULL;
	unsigned long flags;
	u16 fc;
	prism2_wds_info_t *wds = NULL;
	struct net_device_stats *stats;
	u8 *wepbuf = NULL;
	int wepbuf_len = 0;

	stats = prism2_get_stats(dev);

	if ((local->link->state & (DEV_PRESENT | DEV_CONFIG)) !=
	    (DEV_PRESENT | DEV_CONFIG) ||
	    !local->hw_ready) {
		printk(KERN_DEBUG "%s: prism2_tx: hw not ready - skipping\n",
		       dev->name);
		ret = 0;
		goto tx_exit;
	}

	if (skb->len < ETH_HLEN) {
		printk(KERN_DEBUG "%s: prism2_tx: short skb (len=%d)\n",
		       dev->name, skb->len);
		ret = 0;
		goto tx_exit;
	}

	if (local->dev != dev)
		wds = (prism2_wds_info_t *) dev;

	/* Incoming skb->data: dst_addr[6], src_addr[6], proto[2], payload
	 * ==>
	 * Prism2 TX frame with 802.11 header:
	 * txdesc (address order depending on used mode; includes dst_addr and
	 * src_addr), snap_header[6], proto[2], payload {, possible addr4[6]}
	 */

	memset(&txdesc, 0, sizeof(txdesc));

	txdesc.tx_control = __cpu_to_le16(local->tx_control);

	/* Length of data after txdesc */
	data_len = skb->len - 2 * ETH_ALEN + sizeof(snap_header);

	fc = (WLAN_FC_TYPE_DATA << 2) | (WLAN_FC_STYPE_DATA << 4);
	if (wds) {
		/* Note! Prism2 station firmware has problems with sending real
		 * 802.11 frames with four addresses; until these problems can
		 * be fixed or worked around, 4-addr frames needed for WDS are
		 * using incompatible format: FromDS flag is not set and the
		 * fourth address is added after the frame payload; it is
		 * assumed, that the receiving station knows how to handle this
		 * frame format */

#ifdef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
		fc |= WLAN_FC_FROMDS | WLAN_FC_TODS;
		/* From&To DS: Addr1 = RA, Addr2 = TA, Addr3 = DA, Addr4 = SA
		 */
		memcpy(&txdesc.addr1, wds->remote_addr, ETH_ALEN);
		memcpy(&txdesc.addr2, dev->dev_addr, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data, ETH_ALEN);
		memcpy(&txdesc.addr4, skb->data + ETH_ALEN, ETH_ALEN);
#else /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */
		/* bogus 4-addr format to workaround Prism2 station f/w bug */
		fc |= WLAN_FC_TODS;
		/* From DS: Addr1 = DA (used as RA),
		 * Addr2 = BSSID (used as TA), Addr3 = SA (used as DA) */
		memcpy(&txdesc.addr1, wds->remote_addr, ETH_ALEN);
		memcpy(&txdesc.addr2, dev->dev_addr, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data, ETH_ALEN);
		/* SA from skb->data + ETH_ALEN will be added after frame
		 * payload */
		data_len += ETH_ALEN;
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */
		memcpy(&txdesc.dst_addr, &txdesc.addr3, ETH_ALEN);
		memcpy(&txdesc.src_addr, &txdesc.addr2, ETH_ALEN);
	} else if (local->iw_mode == IW_MODE_MASTER) {
		fc |= WLAN_FC_FROMDS;
		/* From DS: Addr1 = DA, Addr2 = BSSID, Addr3 = SA */
		memcpy(&txdesc.addr1, skb->data, ETH_ALEN);
		/* FIX - addr2 replaced by f/w, so no need to fill it now(?) */
		memcpy(&txdesc.addr2, dev->dev_addr, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data + ETH_ALEN, ETH_ALEN);
	} else if (local->iw_mode == IW_MODE_INFRA) {
		fc |= WLAN_FC_TODS;
		/* To DS: Addr1 = BSSID, Addr2 = SA, Addr3 = DA;
		 * firmware sets BSSID */
		/* memcpy(&txdesc.addr1, local->bssid, ETH_ALEN); */
		memcpy(&txdesc.addr2, skb->data + ETH_ALEN, ETH_ALEN);
		memcpy(&txdesc.addr3, skb->data, ETH_ALEN);
	} else if (local->iw_mode == IW_MODE_ADHOC) {
		/* not From/To DS: Addr1 = DA, Addr2 = SA, Addr3 = BSSID */
		memcpy(&txdesc.addr1, skb->data, ETH_ALEN);
		memcpy(&txdesc.addr2, skb->data + ETH_ALEN, ETH_ALEN);
		memcpy(&txdesc.addr3, local->bssid, ETH_ALEN);
	}

	if (local->use_wep) {
		fc |= WLAN_FC_ISWEP;
		if (local->iw_mode == IW_MODE_MASTER || wds)
			data_len += 8; /* IV + ICV */
	}

	txdesc.frame_control = __cpu_to_le16(fc);
	txdesc.data_len = __cpu_to_le16(data_len);
	txdesc.len = __cpu_to_be16(data_len);
	if (!wds)
		memcpy(&txdesc.dst_addr, skb->data, 2 * ETH_ALEN);

	if (local->iw_mode == IW_MODE_MASTER &&
	    (txdesc.addr1[0] & 0x01) == 0x00) {
		/* unicast packet - check whether destination STA is
		 * associated */
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		sta = ap_get_sta(local->ap, txdesc.addr1);
		if (sta)
			atomic_inc(&sta->users);
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		if (!sta && !wds) {
			/* remove FromDS flag from (pseudo) ad-hoc style
			 * communication between APs */
			txdesc.frame_control &=
				~(__constant_cpu_to_le16(WLAN_FC_FROMDS));

			printk(KERN_DEBUG "AP: packet to non-associated STA "
			       MACSTR "\n", MAC2STR(txdesc.addr1));
		}
	}

	/* Set tx_rate if using host-based TX rate control */
	if (!local->fw_tx_rate_control && sta != NULL) {
		txdesc.tx_rate = sta->tx_rate;
		sta->tx_count[sta->tx_rate_idx]++;
		sta->tx_since_last_failure++;
		if (sta->tx_since_last_failure > WLAN_RATE_UPDATE_COUNT &&
		    sta->tx_rate_idx < sta->tx_max_rate) {
			/* use next higher rate */
			while (sta->tx_rate_idx < sta->tx_max_rate) {
				sta->tx_rate_idx++;
				if (sta->tx_supp_rates &
				    (1 << sta->tx_rate_idx))
					break;
			}
			switch (sta->tx_rate_idx) {
			case 0: sta->tx_rate = 10; break;
			case 1: sta->tx_rate = 20; break;
			case 2: sta->tx_rate = 55; break;
			case 3: sta->tx_rate = 110; break;
			default: sta->tx_rate = 0; break;
			}
			PDEBUG(DEBUG_AP, "%s: STA " MACSTR " TX rate raised to"
			       " %d\n", dev->name, MAC2STR(sta->addr),
			       sta->tx_rate);
			sta->tx_since_last_failure = 0;
		}
	}

	if (sta && sta->flags & WLAN_STA_PS) {
		struct sta_buffer_frame *fbuf;
		int set_tim = 0;

		/* STA in PS mode, buffer frame for later delivery */
		if (sta->buffer_count >= STA_MAX_TX_BUFFER) {
			PDEBUG(DEBUG_PS, "No more space in this STA's PS mode "
			       "buffer\n");
			ret = 0; /* drop packet */
			stats->tx_dropped++;
			goto fail;
		}

		fbuf = (struct sta_buffer_frame *)
			kmalloc(sizeof(struct sta_buffer_frame), GFP_ATOMIC);
		if (fbuf == NULL) {
			printk("Malloc failed for STA's PS mode buffer\n");
			goto fail;
		}
		fbuf->next = NULL;
		fbuf->rx_time = jiffies;
		memcpy(&fbuf->txdesc, &txdesc, sizeof(txdesc));
		fbuf->skb = skb;
		spin_lock_irqsave(&local->ap->sta_table_lock, flags);
		sta->buffer_count++;
		if (sta->tx_buf_head == NULL) {
			sta->tx_buf_head = sta->tx_buf_tail = fbuf;
			set_tim = 1;
		} else {
			sta->tx_buf_tail->next = fbuf;
			sta->tx_buf_tail = fbuf;
		}
		spin_unlock_irqrestore(&local->ap->sta_table_lock, flags);

		if (set_tim)
			prism2_set_tim(dev, sta->aid, 1);

		atomic_dec(&sta->users);

		/* do not free skb here, it will be freed when the buffered
		 * frame is sent/timed out */
		ret = 0;
		goto tx_exit;
	}

	if (local->use_wep && (local->iw_mode == IW_MODE_MASTER || wds)) {
		u8 *pos;
		wepbuf_len = 4 + sizeof(snap_header) +
			skb->len - 2 * ETH_ALEN + 4;
#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
		if (wds)
			wepbuf_len += ETH_ALEN; /* Addr4 */
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */
		wepbuf = (u8 *) kmalloc(wepbuf_len, GFP_ATOMIC);
		if (wepbuf == NULL) {
			printk(KERN_DEBUG "%s: could not allocate TX wepbuf\n",
			       dev->name);
			goto fail;
		}

		pos = wepbuf + 4;
		memcpy(pos, snap_header, sizeof(snap_header));
		pos += sizeof(snap_header);
		memcpy(pos, skb->data + 2 * ETH_ALEN, skb->len - 2 * ETH_ALEN);
#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
		if (wds) {
			pos += skb->len - 2 * ETH_ALEN;
			memcpy(pos, skb->data + ETH_ALEN, ETH_ALEN);
		}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

		prism2_wep_encrypt(local, wepbuf, wepbuf_len - 8, NULL);
	}

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		goto fail;

	if (local->frame_dump & PRISM2_DUMP_TX_HDR)
		prism2_dump_tx_header(dev->name, &txdesc);

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &txdesc, sizeof(txdesc));
	if (!res && !wepbuf)
		res = hfa384x_to_bap(dev, BAP0, &snap_header,
				     sizeof(snap_header));
#ifdef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
	if (!res && !wepbuf)
		res = hfa384x_to_bap(dev, BAP0, skb->data + 2 * ETH_ALEN,
				     skb->len - 2 * ETH_ALEN);
#else /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */
	if (!res && !wepbuf) {
		int wlen = skb->len - 2 * ETH_ALEN;
		int is_odd = wlen & 1;
		if (wds && is_odd)
			wlen--; /* need to avoid using odd offset */
		res = hfa384x_to_bap(dev, BAP0, skb->data + 2 * ETH_ALEN,
				     wlen);

		/* add addr4 (SA) to bogus frame format if WDS is used */
		if (!res && wds && is_odd) {
			char tmpbuf[ETH_ALEN + 1];
			tmpbuf[0] = *(skb->data + skb->len - 1);
			memcpy(tmpbuf + 1, skb->data + ETH_ALEN, ETH_ALEN);
			res = hfa384x_to_bap(dev, BAP0, tmpbuf, ETH_ALEN + 1);
		} else if (!res && wds) {
			res = hfa384x_to_bap(dev, BAP0, skb->data + ETH_ALEN,
					     ETH_ALEN);
		}
	}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

	if (!res && wepbuf) {
		res = hfa384x_to_bap(dev, BAP0, wepbuf, wepbuf_len);
		kfree(wepbuf);
	}
	spin_unlock_bh(&local->baplock);

	if (!res)
		res = prism2_transmit(dev, idx);
	if (res) {
		printk("%s: prism2_tx - to BAP0 failed\n", dev->name);
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		prism2_hw_reset(dev);
		goto fail;
	}

	add_tx_bytes(stats, data_len + 36);

	if (sta) {
		sta->tx_packets++;
		sta->tx_bytes += data_len + 36;
		sta->last_tx = jiffies;
	}

	ret = 0;

 fail:
	if (sta)
		atomic_dec(&sta->users);

	if (!ret)
		dev_kfree_skb(skb);

 tx_exit:
	return ret;
}


#ifdef PRISM2_HOSTAPD

/* Called only from software IRQ */
static int prism2_tx_80211(struct sk_buff *skb, struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct hfa384x_tx_frame txdesc;
	int hdr_len, data_len, ret = 1, idx, res;
	u16 fc;

	if ((local->link->state & (DEV_PRESENT | DEV_CONFIG)) !=
	    (DEV_PRESENT | DEV_CONFIG) ||
	    !local->hw_ready) {
		printk(KERN_DEBUG "%s: prism2_tx_80211: hw not ready - "
		       "skipping\n", dev->name);
		ret = 0;
		goto tx_exit;
	}

	if (skb->len < 24) {
		printk(KERN_DEBUG "%s: prism2_tx_80211: short skb (len=%d)\n",
		       dev->name, skb->len);
		ret = 0;
		goto tx_exit;
	}

	memset(&txdesc, 0, sizeof(txdesc));
	txdesc.tx_control = __cpu_to_le16(local->tx_control);
	/* txdesc.tx_rate might need to be set if f/w does not select suitable
	 * TX rate */

	/* skb->data starts with txdesc->frame_control */
	hdr_len = 24;
	memcpy(&txdesc.frame_control, skb->data, hdr_len);
 	fc = __le16_to_cpu(txdesc.frame_control);
	if (WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_DATA &&
	    (fc & WLAN_FC_FROMDS) && (fc & WLAN_FC_TODS) && skb->len >= 30) {
		/* Addr4 */
		memcpy(txdesc.addr4, skb->data + hdr_len, ETH_ALEN);
		hdr_len += ETH_ALEN;
	}
	
	data_len = skb->len - hdr_len;
	txdesc.data_len = __cpu_to_le16(data_len);
	txdesc.len = __cpu_to_be16(data_len);

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		goto fail;

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, &txdesc, sizeof(txdesc));
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, skb->data + hdr_len,
				     skb->len - hdr_len);
	spin_unlock_bh(&local->baplock);

	if (!res)
		res = prism2_transmit(dev, idx);
	if (res) {
		printk("%s: prism2_tx_80211 - to BAP0 failed\n", dev->name);
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		prism2_hw_reset(dev);
		goto fail;
	}

	ret = 0;

 fail:
	if (!ret)
		dev_kfree_skb(skb);

 tx_exit:
	return ret;
}

#endif /* PRISM2_HOSTAPD */


#if defined(PRISM2_MONITOR) || defined(PRISM2_HOSTAPD)

enum {
	PRISM2_RX_MONITOR, PRISM2_RX_MGMT, PRISM2_RX_NON_ASSOC,
	PRISM2_RX_NULLFUNC_ACK
};

/* Called only from hardware IRQ */
static int prism2_rx_80211(struct net_device *dev,
			   struct hfa384x_rx_frame *rxdesc,
			   int type, char *extra, int extra_len)
{
	struct sk_buff *skb;
	int res, hdrlen;
	u16 len, fc;
	unsigned char *pos;

	hdrlen = 24;
	fc = __le16_to_cpu(rxdesc->frame_control);
	if (WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_DATA &&
	    (fc & WLAN_FC_FROMDS) && (fc & WLAN_FC_TODS))
		hdrlen += 6; /* Addr4 */

	if (extra) {
		/* set 802.11 protocol version to 3 to indicate extra data
		 * after the payload */
		fc |= WLAN_FC_PVER;
		rxdesc->frame_control = __cpu_to_le16(fc);
	}

	len = __le16_to_cpu(rxdesc->data_len);
	if (len > PRISM2_DATA_MAXLEN || extra_len > 65535) {
		printk(KERN_WARNING "%s: prism2_rx_80211: len(%d) > "
		       "MAX(%d)\n", dev->name, len, PRISM2_DATA_MAXLEN);
		return 0;
	}

	skb = dev_alloc_skb(hdrlen + len + 2 + extra_len + 2);
	if (!skb) {
		printk(KERN_WARNING "%s: prism2_rx_80211 cannot allocate "
		       "buffer\n", dev->name);
		return 0;
	}

	/* align IP on 16b boundary */
	skb_reserve(skb, 2);
	dev->last_rx = jiffies;

	skb_put(skb, hdrlen + len);
	pos = skb->data;
	memcpy(pos, &rxdesc->frame_control, hdrlen);
	pos += hdrlen;
	res = hfa384x_from_bap(dev, BAP1, pos, len);
	pos += len;

	/* TODO: could add 'type' information into the end of the data if it
	 * is needed in the user space daemon */
	if (extra != NULL) {
		u16 *elen;

		skb_put(skb, extra_len + 2);
		memcpy(pos, extra, extra_len);
		pos += extra_len;
		elen = (u16 *) pos;
		*elen = __cpu_to_le16(extra_len);
	}

	if (res) {
		printk(KERN_WARNING "%s: prism2_rx_80211 from_bap failed\n",
		       dev->name);
		dev_kfree_skb(skb);
		return 0;
	}

	skb->dev = dev;
	skb->mac.raw = skb->data;
	skb_pull(skb, hdrlen);
	skb->pkt_type = PACKET_OTHERHOST;
	skb->protocol = htons(ETH_P_802_2);
	netif_rx(skb);

	return (hdrlen + len);
}


static int prism2_80211_header_parse(struct sk_buff *skb, unsigned char *haddr)
{
	memcpy(haddr, skb->mac.raw + 10, ETH_ALEN); /* addr2 */
	return ETH_ALEN;
}
#endif /* defined(PRISM2_MONITOR) || defined(PRISM2_HOSTAPD) */


#ifdef PRISM2_MONITOR

/* Called only from hardware IRQ */
static void monitor_rx_nl(struct net_device *dev,
			  struct hfa384x_rx_frame *rxdesc)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sk_buff *skb;
	int res;
	u16 len;

	if (local->nl_monitor == NULL) {
		printk(KERN_DEBUG "%s: monitor_rx_nl - netlink not open\n",
		       dev->name);
		return;
	}

	len = __le16_to_cpu(rxdesc->data_len);
	if (len > PRISM2_DATA_MAXLEN) {
		printk(KERN_WARNING "%s: monitor_rx_nl: len(%d) > MAX(%d)\n",
		       dev->name, len, PRISM2_DATA_MAXLEN);
		return;
	}

	skb = dev_alloc_skb(sizeof(struct hfa384x_rx_frame) + len);
	if (!skb) {
		printk(KERN_WARNING "%s: monitor_rx_nl cannot allocate "
		       "buffer\n", dev->name);
		netlink_set_err(local->nl_monitor, 0, PRISM2_MONITOR_GROUP,
				ENOBUFS);
		return;
	}

	memcpy(skb->data, rxdesc, sizeof(struct hfa384x_rx_frame));

	res = hfa384x_from_bap(dev, BAP1, skb->data +
			       sizeof(struct hfa384x_rx_frame), len);

	skb_put(skb, sizeof(struct hfa384x_rx_frame) + len);

	if (res)
		printk(KERN_WARNING "%s: monitor_rx_nl from_bap failed\n",
		       dev->name);

	skb->dev = dev;

	NETLINK_CB(skb).dst_groups = PRISM2_MONITOR_GROUP;
	netlink_broadcast(local->nl_monitor, skb, 0, PRISM2_MONITOR_GROUP,
			  GFP_ATOMIC);
}

/* Called only from hardware IRQ */
static void monitor_rx_dev(struct net_device *dev,
			   struct hfa384x_rx_frame *rxdesc)
{
	struct net_device_stats *stats;
	int len;

	stats = prism2_get_stats(dev);
	len = prism2_rx_80211(dev, rxdesc, PRISM2_RX_MONITOR, NULL, 0);
	stats->rx_packets++;
	add_rx_bytes(stats, len);
}

/* Called only from hardware IRQ */
static void monitor_rx(struct net_device *dev, struct hfa384x_rx_frame *rxdesc)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (le16_to_cpu(rxdesc->status) & HFA384X_RX_STATUS_FCSERR &&
		!local->monitor_allow_fcserr)
		return;

	if (local->monitor_type == PRISM2_MONITOR_NL)
		monitor_rx_nl(dev, rxdesc);
	else
		monitor_rx_dev(dev, rxdesc);
}

static void monitor_tx_skb(struct net_device *dev, struct sk_buff *skb)
{
	local_info_t *local;
	int idx, res;

	local = (local_info_t *) dev->priv;

	idx = prism2_get_txfid_idx(local);
	if (idx < 0)
		return;

	if (local->frame_dump & PRISM2_DUMP_TX_HDR &&
	    skb->len >= sizeof(struct hfa384x_tx_frame))
		prism2_dump_tx_header(dev->name,
				      (struct hfa384x_tx_frame *) skb->data);

	spin_lock_bh(&local->baplock);
	res = hfa384x_setup_bap(dev, BAP0, local->txfid[idx], 0);
	if (!res)
		res = hfa384x_to_bap(dev, BAP0, skb->data, skb->len);
	spin_unlock_bh(&local->baplock);

	if (!res)
		res = prism2_transmit(dev, idx);

	if (res) {
		printk("%s: monitor_tx_skb - to BAP0 failed\n", dev->name);
		local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}
}


static void monitor_tx(struct sock *sk, int len)
{
	struct net_device *dev;
	local_info_t *local;
	struct sk_buff *skb;

	printk("monitor_tx(%p, %d)\n", sk, len);

	if (dev_list == NULL)
		return;

	dev = dev_list->priv;
	local = (local_info_t *) dev->priv;

	if (local == NULL || local->nl_monitor == NULL)
		return;

	do {
		if (rtnl_shlock_nowait())
			return;
		while ((skb = skb_dequeue(&sk->receive_queue)) != NULL) {
			printk("monitor_tx: got skb, len=%d\n", skb->len);
			monitor_tx_skb(dev, skb);
			kfree_skb(skb);
		}
		up(&rtnl_sem);
	} while (local->nl_monitor && local->nl_monitor->receive_queue.qlen);
}
#endif /* PRISM2_MONITOR */

static void prism2_dump_rx_header(const char *name,
				  const struct hfa384x_rx_frame *rx)
{
	u16 status, fc;


	status = __le16_to_cpu(rx->status);

	printk(KERN_DEBUG "%s: RX status=0x%04x (port=%d, type=%d, "
	       "fcserr=%d) silence=%d signal=%d rate=%d rxflow=%d; "
	       "jiffies=%ld\n",
	       name, status, (status >> 8) & 0x07, status >> 13, status & 1,
	       rx->silence, rx->signal, rx->rate, rx->rxflow, jiffies);

	fc = __le16_to_cpu(rx->frame_control);
	printk(KERN_DEBUG "   FC=0x%04x (type=%d:%d) dur=0x%04x seq=0x%04x "
	       "data_len=%d%s%s\n",
	       fc, WLAN_FC_GET_TYPE(fc), WLAN_FC_GET_STYPE(fc),
	       __le16_to_cpu(rx->duration_id), __le16_to_cpu(rx->seq_ctrl),
	       __le16_to_cpu(rx->data_len),
	       fc & WLAN_FC_TODS ? " [ToDS]" : "",
	       fc & WLAN_FC_FROMDS ? " [FromDS]" : "");

	printk(KERN_DEBUG "   A1=" MACSTR " A2=" MACSTR " A3=" MACSTR " A4="
	       MACSTR "\n",
	       MAC2STR(rx->addr1), MAC2STR(rx->addr2), MAC2STR(rx->addr3),
	       MAC2STR(rx->addr4));

	printk(KERN_DEBUG "   dst=" MACSTR " src=" MACSTR " len=%d\n",
	       MAC2STR(rx->dst_addr), MAC2STR(rx->src_addr),
	       __be16_to_cpu(rx->len));
}

/* Called only from software IRQ */
static void handle_bridged_queue(void *data)
{
	local_info_t *local = (local_info_t *) data;
	struct sk_buff *skb;

	while ((skb = skb_dequeue(&local->bridge_list)) != NULL)
		dev_queue_xmit(skb);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0))
	MOD_DEC_USE_COUNT;
#endif
}

/* Called only from hardware IRQ */
static void prism2_rx(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sk_buff *skb, *skb2 = NULL;
	int res;
	u16 len = 0;
	u16 rxfid, status, macport, msg_type, fc, type, stype;
	struct hfa384x_rx_frame rxdesc;
	struct sta_info *sta = NULL;
	prism2_wds_info_t *wds = NULL;
	struct net_device_stats *stats;

	stats = prism2_get_stats(dev);

	rxfid = HFA384X_INW(HFA384X_RXFID_OFF);
#ifndef final_version
	if (rxfid == 0) {
		rxfid = HFA384X_INW(HFA384X_RXFID_OFF);
		printk(KERN_DEBUG "prism2_rx: rxfid=0 (next 0x%04x)\n",
		       rxfid);
		prism2_hw_reset(dev);
		goto rx_dropped;
	}
#endif

	res = hfa384x_setup_bap(dev, BAP1, rxfid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &rxdesc, sizeof(rxdesc));
	if (res) {
		printk("copy from BAP1 failed %d\n", res);
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		goto rx_dropped;
	}

	if (__le16_to_cpu(rxdesc.data_len) & 0x8000) {
		/* data register seems to give 0x8000 in some error cases even
		 * though busy bit is not set in offset register; re-reading
		 * the data seems to fix at least some of the cases */
		printk(KERN_DEBUG "%s: prism2_rx: re-reading rxdesc to fix "
		       "possibly corrupted BAP read\n", dev->name);
		res = hfa384x_setup_bap(dev, BAP1, rxfid, 0);
		if (!res)
			res = hfa384x_from_bap(dev, BAP1, &rxdesc,
					       sizeof(rxdesc));
		if (res)
			printk(KERN_DEBUG "prism2_rx: could not re-read "
			       "rxdesc\n");
		if (__le16_to_cpu(rxdesc.data_len) & 0x8000) {
			printk(KERN_DEBUG "prism2_rx: re-read did not fix "
			       "rxdesc\n");
			goto rx_dropped;
		}
	}


	status = __le16_to_cpu(rxdesc.status);
	macport = (status >> 8) & 0x07;

	if (local->frame_dump & PRISM2_DUMP_RX_HDR)
		prism2_dump_rx_header(dev->name, &rxdesc);

	if (macport != 0) {
#ifdef PRISM2_MONITOR
		if (macport == 7) {
			monitor_rx(dev, &rxdesc);
		} else {
#else
		{
#endif
			printk(KERN_DEBUG "RX: Unknown MACPort %d\n", macport);
		}
		goto rx_dropped;
	}

	/* FCS errors should not come this far, but let's make sure that frames
	 * with errors will be dropped even in Host AP mode */
	if (status & HFA384X_RX_STATUS_FCSERR) {
		printk(KERN_DEBUG "%s: prism2_rx: dropped FCSErr frame "
		       "(status=%02X)\n", dev->name, status);
		goto rx_dropped;
	}

	msg_type = status >> 13;

	fc = __le16_to_cpu(rxdesc.frame_control);
	type = WLAN_FC_GET_TYPE(fc);
	stype = WLAN_FC_GET_STYPE(fc);

	if (msg_type == HFA384X_RX_MSGTYPE_MGMT) {
#ifdef PRISM2_HOSTAPD
		/* send management frames to the user space daemon for
		 * processing */
		if (type == WLAN_FC_TYPE_MGMT) {
			prism2_rx_80211(local->apdev, &rxdesc, PRISM2_RX_MGMT,
					NULL, 0);
			goto rx_exit;
		}
#endif /* PRISM2_HOSTAPD */

		if (local->iw_mode == IW_MODE_MASTER) {
			/* FIX: should STA's PS flag be also checked here? */
			hostap_rx(dev, &rxdesc, 0);
		} else {
			printk(KERN_DEBUG "%s: prism2_rx: management frame "
			       "received in non-Host AP mode\n", dev->name);
			prism2_dump_rx_header(dev->name, &rxdesc);
			goto rx_dropped;
		}
		goto rx_exit;
	}

	if (msg_type != HFA384X_RX_MSGTYPE_NORMAL &&
	    msg_type != HFA384X_RX_MSGTYPE_RFC1042 &&
	    msg_type != HFA384X_RX_MSGTYPE_BRIDGETUNNEL) {
		printk(KERN_DEBUG "%s: prism2_rx: dropped frame "
		       "(msg_type=%d)\n", dev->name, msg_type);
		goto rx_dropped;
	}

	if (type != WLAN_FC_TYPE_DATA) {
		printk(KERN_DEBUG "%s: prism2_rx: dropped non-data frame "
		       "(type=0x%02x, subtype=0x%02x)\n",
		       dev->name, type, stype);
		goto rx_dropped;
	}

	if (local->iw_mode == IW_MODE_MASTER) {
		spin_lock(&local->ap->sta_table_lock);
		sta = ap_get_sta(local->ap, rxdesc.addr2);
		if (sta)
			atomic_inc(&sta->users);
		spin_unlock(&local->ap->sta_table_lock);

		if (fc & WLAN_FC_TODS) {
			if (memcmp(rxdesc.addr1, dev->dev_addr, 6) != 0) {
				/* BSSID is not ours; drop */
				goto rx_dropped;
			}

			/* check if the frame came from a registered WDS
			 * connection */
			spin_lock(&local->wdslock);
			wds = local->wds;
			while (wds != NULL &&
			       memcmp(wds->remote_addr, rxdesc.addr2,
				      ETH_ALEN) != 0)
				wds = wds->next;
			spin_unlock(&local->wdslock);
			if (wds) {
				dev = &wds->dev;
				stats = prism2_get_stats(dev);
			}

			if (wds == NULL &&
			    (sta == NULL || !(sta->flags & WLAN_STA_ASSOC))) {
#ifdef PRISM2_HOSTAPD
				prism2_rx_80211(local->apdev, &rxdesc,
						PRISM2_RX_NON_ASSOC, NULL, 0);
#else /* PRISM2_HOSTAPD */
				printk(KERN_DEBUG "%s: dropped received packet"
				       " from non-associated STA " MACSTR
				       " (type=0x%02x, subtype=0x%02x)\n",
				       dev->name,
				       MAC2STR(rxdesc.addr2), type, stype);
				hostap_rx(dev, &rxdesc, 1);
#endif /* PRISM2_HOSTAPD */
				goto rx_dropped;
			}
		} else if (fc & WLAN_FC_FROMDS) {
			/* FromDS frame - not for us; probably
			 * broadcast/multicast in another BSS - drop */
			if (memcmp(rxdesc.addr1, dev->dev_addr, 6) == 0) {
				printk(KERN_DEBUG "Odd.. FromDS packet "
				       "received with own BSSID\n");
				prism2_dump_rx_header(dev->name, &rxdesc);
			}
			goto rx_dropped;
		} else if (stype == WLAN_FC_STYPE_NULLFUNC && sta == NULL &&
			   memcmp(rxdesc.addr1, dev->dev_addr, 6) == 0) {
#ifdef PRISM2_HOSTAPD
			prism2_rx_80211(local->apdev, &rxdesc,
					PRISM2_RX_NON_ASSOC, NULL, 0);
#else /* PRISM2_HOSTAPD */
			/* At least Lucent f/w seems to send data::nullfunc
			 * frames with no ToDS flag when the current AP returns
			 * after being unavailable for some time. Speed up
			 * re-association by informing the station about it not
			 * being associated. */
			printk(KERN_DEBUG "%s: rejected received nullfunc "
			       "frame without ToDS from not associated STA "
			       MACSTR "\n",
			       dev->name, MAC2STR(rxdesc.addr2));
			hostap_rx(dev, &rxdesc, 1);
#endif /* PRISM2_HOSTAPD */
			goto rx_dropped;
		} else if (stype == WLAN_FC_STYPE_NULLFUNC) {
			/* At least Lucent cards seem to send periodic nullfunc
			 * frames with ToDS. Let these through to update SQ
			 * stats and PS state. Nullfunc frames do not contain
			 * any data and they will be dropped below. */
		} else {
			printk(KERN_DEBUG "%s: dropped received packet from "
			       MACSTR " with no ToDS flag (type=0x%02x, "
			       "subtype=0x%02x)\n", dev->name,
			       MAC2STR(rxdesc.addr2), type, stype);
			prism2_dump_rx_header(dev->name, &rxdesc);
			goto rx_dropped;
		}
	}

	if (sta) {
		prism2_ap_update_sq(sta, &rxdesc);

		if ((fc & WLAN_FC_PWRMGT) && !(sta->flags & WLAN_STA_PS)) {
			sta->flags |= WLAN_STA_PS;
			PDEBUG(DEBUG_PS2, "STA " MACSTR " changed to use PS "
			       "mode (type=0x%02X, stype=0x%02X)\n",
			       MAC2STR(rxdesc.addr2), type, stype);
		} else if (!(fc & WLAN_FC_PWRMGT) &&
			   (sta->flags & WLAN_STA_PS)) {
			sta->flags &= ~WLAN_STA_PS;
			PDEBUG(DEBUG_PS2, "STA " MACSTR " changed to not use "
			       "PS mode (type=0x%02X, stype=0x%02X)\n",
			       MAC2STR(rxdesc.addr2), type, stype);
			schedule_packet_send(local, sta);
		}
	}

	if (stype != WLAN_FC_STYPE_DATA &&
	    stype != WLAN_FC_STYPE_DATA_CFACK &&
	    stype != WLAN_FC_STYPE_DATA_CFPOLL &&
	    stype != WLAN_FC_STYPE_DATA_CFACKPOLL) {
		if (local->iw_mode == IW_MODE_MASTER &&
		    local->ap && local->ap->nullfunc_ack &&
		    stype == WLAN_FC_STYPE_NULLFUNC &&
		    fc & WLAN_FC_TODS) {
#ifdef PRISM2_HOSTAPD
			prism2_rx_80211(local->apdev, &rxdesc,
					PRISM2_RX_NULLFUNC_ACK, NULL, 0);
#else /* PRISM2_HOSTAPD */
			/* some STA f/w's seem to require control::ACK frame
			 * for data::nullfunc, but Prism2 f/w 0.8.0 (at least
			 * from Compaq) does not send this.. Try to generate
			 * ACK for these frames from the host driver to make
			 * power saving work with, e.g., Lucent WaveLAN f/w */
			hostap_rx(dev, &rxdesc, 1);
#endif /* PRISM2_HOSTAPD */
			goto rx_exit;
		}
		if (stype != WLAN_FC_STYPE_NULLFUNC)
			printk(KERN_DEBUG "%s: prism2_rx: dropped data frame "
			       "with no data (type=0x%02x, subtype=0x%02x)\n",
			       dev->name, type, stype);
		goto rx_dropped;
	}

	len = __le16_to_cpu(rxdesc.data_len);
	if (len > PRISM2_DATA_MAXLEN) {
		printk(KERN_WARNING "%s: Rx: len(%d) > MAX(%d)\n",
		       dev->name, len, PRISM2_DATA_MAXLEN);
		goto rx_dropped;
	}

#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
	if (wds) {
		if (len < ETH_ALEN) {
			printk(KERN_DEBUG "%s: Rx (WDS): len(%d) < %d\n",
			       dev->name, len, ETH_ALEN);
			goto rx_dropped;
		}
		len -= ETH_ALEN; /* addr4 */
	}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

	skb = dev_alloc_skb(len + ETH_HLEN + 2);
	if (!skb) {
		printk(KERN_WARNING "%s Rx cannot allocate buffer\n",
		       dev->name);
		goto rx_dropped;
	}

	/* align IP on 16b boundary */
	skb_reserve(skb, 2);
	dev->last_rx = jiffies;

	/* receive packet (decapsulated Ethernet frame) */
	memcpy(skb->data, &rxdesc.dst_addr, ETH_HLEN);

	if (sta) {
		sta->rx_packets++;
		sta->rx_bytes += len;
		sta->last_rx = jiffies;
	}

	if (len >= 8) {
		/* try to remove Ethernet-II snap header */
		res = hfa384x_from_bap(dev, BAP1, skb->data + ETH_HLEN, 6);
		if (res)
			goto copy_failed;

		if (memcmp(skb->data + ETH_HLEN, snap_header, 3) == 0) {
			res = hfa384x_from_bap(dev, BAP1, skb->data + 12,
					       len - 6);
			skb_put(skb, 6 + len);
		} else {
			PDEBUG(DEBUG_EXTRA,  "no SNAP? (" MACSTR ")\n",
			       MAC2STR(skb->data + ETH_HLEN));
			res = hfa384x_from_bap(dev, BAP1,
					       skb->data + ETH_HLEN + 6,
					       len - 6);
			skb_put(skb, ETH_HLEN + len);
		}
	} else {
		res = hfa384x_from_bap(dev, BAP1, skb->data + ETH_HLEN, len);
		skb_put(skb, ETH_HLEN + len);
	}

#ifndef PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG
	if (!res && wds) {
		/* get addr4 from its bogus location */
		if (len & 1) {
			/* BAP offset is odd; must arrange it to be even */
			u8 tmpbuf[ETH_ALEN + 1];

			res = hfa384x_setup_bap(dev, BAP1, rxfid,
						sizeof(rxdesc) + len - 1);
			if (!res) {
				res = hfa384x_from_bap(dev, BAP1, tmpbuf,
						       ETH_ALEN + 1);
				memcpy(skb->data + ETH_ALEN, tmpbuf + 1,
				       ETH_ALEN);
			}
		} else {
			res = hfa384x_from_bap(dev, BAP1, skb->data + ETH_ALEN,
					       ETH_ALEN);
		}
	}
#endif /* PRISM2_STA_HAS_NO_HOSTAP_WDS_BUG */

 copy_failed:
	if (res) {
		printk("%s: data copy from BAP1 failed %d\n",
		       dev->name, res);
		dev_kfree_skb_irq(skb);
		goto rx_dropped;
	}

	skb->dev = dev;

	if (!wds && local->ap->bridge_packets) {
		if (rxdesc.addr3[0] & 0x01) {
			/* copy multicast frame both to the higher layers and
			 * to the wireless media */
			skb2 = skb_clone(skb, GFP_ATOMIC);
			if (skb2 == NULL)
				printk("%s: skb_clone failed for multicast "
				       "frame\n", dev->name);
		} else {
			struct sta_info *dest_sta;
			spin_lock(&local->ap->sta_table_lock);
			dest_sta = ap_get_sta(local->ap, rxdesc.addr3);
			if (dest_sta != NULL &&
			    (dest_sta->flags & WLAN_STA_ASSOC)) {
				/* send frame directly to the associated STA
				 * using wireless media and not passing to
				 * higher layers */
				skb2 = skb;
				skb = NULL;
			}
			spin_unlock(&local->ap->sta_table_lock);
		}
	}

	if (skb2 != NULL) {
		/* send to wireless media (put in a queue and send later) */
		skb2->mac.raw = skb2->data;
		skb2->nh.raw = skb2->data + ETH_HLEN;
		skb_queue_tail(&local->bridge_list, skb2);
/* tq_scheduler was removed in 2.4.0-test12 */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
		queue_task(&local->bridge_queue, &tq_scheduler);
#else
		MOD_INC_USE_COUNT;
		if (schedule_task(&local->bridge_queue) == 0)
			MOD_DEC_USE_COUNT;
#endif

		if ((rxdesc.addr3[0] & 0x01) == 0x01)
			local->ap->bridged_multicast++;
		else
			local->ap->bridged_unicast++;
	}

	if (skb) {
		skb->protocol = eth_type_trans(skb, dev);
		netif_rx(skb);
	}

	stats->rx_packets++;
	add_rx_bytes(stats, len);

	/* FIX: add WIRELESS_EXT & SPY handling */

 rx_exit:
	if (sta)
		atomic_dec(&sta->users);
	return;

 rx_dropped:
	stats->rx_dropped++;
	goto rx_exit;
}


#ifdef WIRELESS_EXT

#if WIRELESS_EXT > 8
static int prism2_get_key_idx(struct net_device *dev, int idx)
{
	u16 val;

	if (idx < 1 || idx > 4) {
		if (hfa384x_get_rid(dev,
				    HFA384X_RID_CNFWEPDEFAULTKEYID,
				    &val, 2, 1) < 0)
			return -1;
		le16_to_cpus(&val);
		if (val > 3) {
			printk("Invalid CNFWEPDEFAULT_KEYID %d\n", val);
			return -1;
		}

		return val;
	}

	return idx - 1;
}


static int prism2_ioctl_siwencode(struct net_device *dev, struct iw_point *erq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int i;
	u16 val;

	i = prism2_get_key_idx(dev, erq->flags & IW_ENCODE_INDEX);
	if (i < 0 || i >= WEP_KEYS)
		return -EINVAL;

	if (erq->pointer != (caddr_t) 0) {
		char keybuf[WEP_KEY_LEN + 1];
		int keylen;

		if (erq->length > WEP_KEY_LEN)
			return -EINVAL;

		memset(keybuf, 0, sizeof(keybuf));
		if (copy_from_user(keybuf, erq->pointer, erq->length))
			return -EFAULT;
		memcpy(local->wep_keys[i], keybuf, WEP_KEY_LEN + 1);
		local->wep_key_len[i] = erq->length <= 5 ? 5 : 13;

		keylen = erq->length;
		if (keylen > 0 && keylen < 5)
			keylen = 6; /* first 5 octets */
		else  if (keylen > 5)
			keylen = WEP_KEY_LEN + 1; /* first 13 octets */
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFDEFAULTKEY0 + i,
				    keybuf, keylen))
			return -EINVAL;
	} else {
		local->wep_tx_key = i;
		val = __cpu_to_le16(i);

		if (hfa384x_set_rid(dev, HFA384X_RID_CNFWEPDEFAULTKEYID, &val,
				    2) < 0) {
			printk(KERN_DEBUG "Setting default WEP key id "
			       "failed\n");
			return -EINVAL;
		}
	}


	if (hfa384x_get_rid(dev, HFA384X_RID_CNFWEPFLAGS, &val, 2, 1) < 0) {
		printk(KERN_DEBUG "CNFWEPFLAGS reading failed\n");
		return -EOPNOTSUPP;
	}
	le16_to_cpus(&val);

	if (erq->flags & IW_ENCODE_OPEN)
		val &= ~HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED;
	else
		val |= HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED;

	if ((erq->pointer != (caddr_t) 0 && erq->length == 0) ||
	    erq->flags & IW_ENCODE_DISABLED) {
		val &= ~(HFA384X_WEPFLAGS_PRIVACYINVOKED |
			 HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED);
		local->use_wep = 0;
	} else {
		val |= HFA384X_WEPFLAGS_PRIVACYINVOKED;
		if (local->iw_mode == IW_MODE_MASTER)
			val |= HFA384X_WEPFLAGS_HOSTENCRYPT;
		else
			val &= ~HFA384X_WEPFLAGS_HOSTENCRYPT;
		local->use_wep = 1;
	}

	cpu_to_le16s(&val);
	if (hfa384x_set_rid(dev, HFA384X_RID_CNFWEPFLAGS, &val, 2)) {
		printk("CNFWEPFLAGS writing failed\n");
		return -EOPNOTSUPP;
	}

	if (prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}


static int prism2_ioctl_giwencode(struct net_device *dev, struct iw_point *erq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int i;
	u16 val;
	char keybuf[14];
	int keylen;

	if (erq->pointer == (caddr_t) 0)
		return -EINVAL;

	if (local->iw_mode == IW_MODE_MASTER)
		i = local->wep_tx_key;
	else
		i = prism2_get_key_idx(dev, erq->flags & IW_ENCODE_INDEX);
	if (i < 0 || i >= WEP_KEYS)
		return -EINVAL;

	erq->flags = i + 1;

	/* Reads from HFA384X_RID_CNFDEFAULTKEY* return bogus values, so show
	 * the keys from driver buffer */
	keylen = hfa384x_get_rid(dev, HFA384X_RID_CNFDEFAULTKEY0 + i,
				 keybuf, sizeof(keybuf), 0);
	if (copy_to_user(erq->pointer, local->wep_keys[i],
			 local->wep_key_len[i]))
		return -EFAULT;
	else
		erq->length = local->wep_key_len[i];

	if (hfa384x_get_rid(dev, HFA384X_RID_CNFWEPFLAGS, &val, 2, 1) < 0) {
		printk("CNFWEPFLAGS reading failed\n");
		return -EOPNOTSUPP;
	}
	le16_to_cpus(&val);
	if (val & HFA384X_WEPFLAGS_PRIVACYINVOKED)
		erq->flags |= IW_ENCODE_ENABLED;
	else
		erq->flags |= IW_ENCODE_DISABLED;
	if (val & HFA384X_WEPFLAGS_EXCLUDEUNENCRYPTED)
		erq->flags |= IW_ENCODE_RESTRICTED;
	else
		erq->flags |= IW_ENCODE_OPEN;

	return 0;
}


static int prism2_ioctl_giwspy(struct net_device *dev, struct iw_point *srq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sockaddr addr[IW_MAX_SPY];
	struct iw_quality qual[IW_MAX_SPY];

	if (srq->pointer == NULL) {
		printk(KERN_DEBUG "%s: SIOCGIWSPY: pointer == NULL\n",
		       dev->name);
		srq->length = 0;
		return -EINVAL;
	}

	if (local->iw_mode != IW_MODE_MASTER) {
		printk("SIOCGIWSPY is currently only supported in Host AP "
		       "mode\n");
		srq->length = 0;
		return -EOPNOTSUPP;
	}

	srq->length = prism2_ap_get_sta_qual(local, addr, qual, IW_MAX_SPY, 0);

	if (copy_to_user(srq->pointer, &addr, sizeof(addr[0]) * srq->length))
		return -EFAULT;
	if (copy_to_user(srq->pointer + sizeof(addr[0]) * srq->length, &qual,
			 sizeof(qual[0]) * srq->length))
		return -EFAULT;

	return 0;
}


static int prism2_ioctl_siwrate(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 val;
	int ret = 0;

	if (rrq->fixed) {
		switch(rrq->value) {
		case 11000000:
			val = __constant_cpu_to_le16(HFA384X_RATES_11MBPS);
			break;
		case 5500000:
			val = __constant_cpu_to_le16(HFA384X_RATES_5MBPS);
			break;
		case 2000000:
			val = __constant_cpu_to_le16(HFA384X_RATES_2MBPS);
			break;
		case 1000000:
			val = __constant_cpu_to_le16(HFA384X_RATES_1MBPS);
			break;
		default:
			val = __constant_cpu_to_le16(HFA384X_RATES_1MBPS |
						     HFA384X_RATES_2MBPS |
						     HFA384X_RATES_5MBPS |
						     HFA384X_RATES_11MBPS);
			break;
		}
	} else {
		switch(rrq->value) {
		case 11000000:
			val = __constant_cpu_to_le16(HFA384X_RATES_1MBPS |
						     HFA384X_RATES_2MBPS |
						     HFA384X_RATES_5MBPS |
						     HFA384X_RATES_11MBPS);
			break;
		case 5500000:
			val = __constant_cpu_to_le16(HFA384X_RATES_1MBPS |
						     HFA384X_RATES_2MBPS |
						     HFA384X_RATES_5MBPS);
			break;
		case 2000000:
			val = __constant_cpu_to_le16(HFA384X_RATES_1MBPS |
						     HFA384X_RATES_2MBPS);
			break;
		case 1000000:
			val = __constant_cpu_to_le16(HFA384X_RATES_1MBPS);
			break;
		default:
			val = __constant_cpu_to_le16(HFA384X_RATES_1MBPS |
						     HFA384X_RATES_2MBPS |
						     HFA384X_RATES_5MBPS |
						     HFA384X_RATES_11MBPS);
			break;
		}
	}

	local->tx_rate_control = val;
	ret = (hfa384x_set_rid(dev, HFA384X_RID_TXRATECONTROL, &val, 2) ||
	       prism2_reset_port(dev));
		
	if (ret) {
		printk("%s: TXRateControl setting to %d failed\n",
		       dev->name, __le16_to_cpu(val));
	}
	return ret;
}

static int prism2_ioctl_giwrate(struct net_device *dev, struct iw_param *rrq)
{
	u16 val;
	int ret = 0;

	if (hfa384x_get_rid(dev, HFA384X_RID_TXRATECONTROL, &val, 2, 1) < 0)
		return -EINVAL;

	if ((val & 0x1) && (val > 1))
		rrq->fixed = 0;
	else
		rrq->fixed = 1;

	if (hfa384x_get_rid(dev, HFA384X_RID_CURRENTTXRATE, &val, 2, 1) < 0)
		return -EINVAL;

	switch (val) {
	case HFA384X_RATES_1MBPS:
		rrq->value = 1000000;
		break;
	case HFA384X_RATES_2MBPS:
		rrq->value = 2000000;
		break;
	case HFA384X_RATES_5MBPS:
		rrq->value = 5500000;
		break;
	case HFA384X_RATES_11MBPS:
		rrq->value = 11000000;
		break;
	default:
		/* should not happen */
		rrq->value = 11000000;
		ret = -EINVAL;
		break;
	}

	return ret;
}


static int prism2_ioctl_siwsens(struct net_device *dev, struct iw_param *sens)
{
	u16 val;

	/* Set the desired AP density */
	if (sens->value < 1 || sens->value > 3)
		return -EINVAL;

	val = __cpu_to_le16(sens->value);

	if (hfa384x_set_rid(dev, HFA384X_RID_CNFSYSTEMSCALE, &val, 2) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwsens(struct net_device *dev, struct iw_param *sens)
{
	u16 val;

	/* Get the current AP density */
	if (hfa384x_get_rid(dev, HFA384X_RID_CNFSYSTEMSCALE, &val, 2, 1) < 0)
		return -EINVAL;

	sens->value = __le16_to_cpu(val);
	sens->fixed = 1;

	return 0;
}


static int prism2_ioctl_giwaplist(struct net_device *dev,
				  struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct sockaddr addr[IW_MAX_AP];
	struct iw_quality qual[IW_MAX_AP];

	if (data->pointer == NULL) {
		printk(KERN_DEBUG "%s: SIOCGIWAPLIST: pointer == NULL\n",
		       dev->name);
		data->length = 0;
		return -EINVAL;
	}

	if (local->iw_mode != IW_MODE_MASTER) {
		printk(KERN_DEBUG "SIOCGIWAPLIST is currently only supported "
		       "in Host AP mode\n");
		data->length = 0;
		return -EOPNOTSUPP;
	}

	data->length = prism2_ap_get_sta_qual(local, addr, qual, IW_MAX_AP, 1);

	if (copy_to_user(data->pointer, &addr, sizeof(addr[0]) * data->length))
		return -EFAULT;
	data->flags = 1; /* has quality information */
	if (copy_to_user(data->pointer + sizeof(addr[0]) * data->length, &qual,
			 sizeof(qual[0]) * data->length))
		return -EFAULT;

	return 0;
}


static int prism2_ioctl_siwrts(struct net_device *dev, struct iw_param *rts)
{
	u16 val;

	if (rts->disabled)
		val = __constant_cpu_to_le16(2347);
	else if (rts->value < 0 || rts->value > 2347)
		return -EINVAL;
	else
		val = __cpu_to_le16(rts->value);

	if (hfa384x_set_rid(dev, HFA384X_RID_RTSTHRESHOLD, &val, 2) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwrts(struct net_device *dev, struct iw_param *rts)
{
	u16 val;

	if (hfa384x_get_rid(dev, HFA384X_RID_RTSTHRESHOLD, &val, 2, 1) < 0)
		return -EINVAL;

	rts->value = __le16_to_cpu(val);
	rts->disabled = (rts->value == 2347);
	rts->fixed = 1;

	return 0;
}


static int prism2_ioctl_siwfrag(struct net_device *dev, struct iw_param *rts)
{
	u16 val;

	if (rts->disabled)
		val = __constant_cpu_to_le16(2346);
	else if (rts->value < 256 || rts->value > 2346)
		return -EINVAL;
	else
		val = __cpu_to_le16(rts->value & ~0x1); /* even numbers only */

	if (hfa384x_set_rid(dev, HFA384X_RID_FRAGMENTATIONTHRESHOLD, &val, 2)
	    || prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwfrag(struct net_device *dev, struct iw_param *rts)
{
	u16 val;

	if (hfa384x_get_rid(dev, HFA384X_RID_FRAGMENTATIONTHRESHOLD,
			    &val, 2, 1) < 0)
		return -EINVAL;

	rts->value = __le16_to_cpu(val);
	rts->disabled = (rts->value == 2346);
	rts->fixed = 1;

	return 0;
}
#endif /* WIRELESS_EXT > 8 */


static int prism2_ioctl_giwap(struct net_device *dev, struct sockaddr *ap_addr)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (hfa384x_get_rid(dev, HFA384X_RID_CURRENTBSSID, &ap_addr->sa_data,
			    ETH_ALEN, ETH_ALEN) < 0)
		return -EOPNOTSUPP;

	/* FIX: local->bssid should be updated in a better place;
	 * maybe in info frame LinkStatus reception? */
	memcpy(local->bssid, &ap_addr->sa_data, ETH_ALEN);
	ap_addr->sa_family = ARPHRD_ETHER;

	return 0;
}


#if WIRELESS_EXT > 8
static int prism2_ioctl_siwnickn(struct net_device *dev, struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (!data->pointer)
		return -EINVAL;

	if (data->length > MAX_NAME_LEN + 1)
		return -E2BIG;

	copy_from_user(local->name, data->pointer, data->length);
	local->name[MAX_NAME_LEN] = '\0';
	local->name_set = 1;

	if (prism2_set_string(dev, HFA384X_RID_CNFOWNNAME, local->name) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwnickn(struct net_device *dev, struct iw_point *data)
{
	int len;
	char name[MAX_NAME_LEN + 3];
	u16 val;

	if (!data->pointer)
		return -EINVAL;

	len = hfa384x_get_rid(dev, HFA384X_RID_CNFOWNNAME,
			      &name, MAX_NAME_LEN + 2, 0);
	val = __le16_to_cpu(*(u16 *) name);
	if (len > MAX_NAME_LEN + 2 || len < 0 || val > MAX_NAME_LEN)
		return -EOPNOTSUPP;

	name[val + 2] = '\0';
	data->length = val + 1;
	copy_to_user(data->pointer, name + 2, val);

	return 0;
}
#endif /* WIRELESS_EXT > 8 */


static int prism2_ioctl_siwfreq(struct net_device *dev, struct iw_freq *freq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 val;

	/* freq => chan. */
	if (freq->e == 1 &&
	    freq->m / 100000 >= freq_list[0] &&
	    freq->m / 100000 <= freq_list[FREQ_COUNT - 1]) {
		int ch;
		int fr = freq->m / 100000;
		for (ch = 0; ch < FREQ_COUNT; ch++) {
			if (fr == freq_list[ch]) {
				freq->e = 0;
				freq->m = ch + 1;
				break;
			}
		}
	}

	if (freq->e != 0 || freq->m < 1 || freq->m > FREQ_COUNT ||
	    !(local->channel_mask & (1 << (freq->m - 1))))
		return -EINVAL;

	val = __cpu_to_le16(freq->m);
	local->channel = val; /* channel is used in prism2_setup_rids() */
	if (hfa384x_set_rid(dev, HFA384X_RID_CNFOWNCHANNEL, &val, 2) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwfreq(struct net_device *dev, struct iw_freq *freq)
{
	u16 val;

	if (hfa384x_get_rid(dev, HFA384X_RID_CURRENTCHANNEL, &val, 2, 1) < 0)
		return -EINVAL;

	le16_to_cpus(&val);
	if (val < 1 || val > FREQ_COUNT)
		return -EINVAL;

	freq->m = freq_list[val - 1] * 100000;
	freq->e = 1;

	return 0;
}


#if WIRELESS_EXT > 8
static int prism2_ioctl_siwessid(struct net_device *dev, struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int len;
	char ssid[MAX_SSID_LEN + 2];

	if (!data->pointer)
		return -EINVAL;

	if (data->flags == 0) {
		local->essid[0] = '\0'; /* ANY */
	} else {
		if (data->length > MAX_SSID_LEN + 1)
			return -E2BIG;

		copy_from_user(local->essid, data->pointer, data->length);
		local->essid[MAX_SSID_LEN] = '\0';
	}

	len = strlen(local->essid);
	memset(ssid, 0, sizeof(ssid));
	memcpy(ssid + 2, local->essid, len);
	ssid[0] = len;
	if (hfa384x_set_rid(dev,
			    local->iw_mode == IW_MODE_INFRA ?
			    HFA384X_RID_CNFDESIREDSSID :
			    HFA384X_RID_CNFOWNSSID, &ssid,
			    MAX_SSID_LEN + 2) ||
	    prism2_reset_port(dev))
		return -EINVAL;

	return 0;
}

static int prism2_ioctl_giwessid(struct net_device *dev, struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 val;

	if (!data->pointer)
		return -EINVAL;

	data->flags = 1; /* active */
	if (local->iw_mode == IW_MODE_MASTER) {
		data->length = strlen(local->essid) + 1;
		copy_to_user(data->pointer, local->essid,
			     sizeof(local->essid));
	} else {
		int len;
		char ssid[MAX_SSID_LEN + 3];
		len = hfa384x_get_rid(dev, HFA384X_RID_CURRENTSSID,
				      &ssid, MAX_SSID_LEN + 2, 0);
		val = __le16_to_cpu(*(u16 *) ssid);
		if (len > MAX_SSID_LEN + 2 || len < 0 || val > MAX_SSID_LEN) {
			return -EOPNOTSUPP;
		}
		ssid[val + 2] = '\0';
		data->length = val + 1;
		copy_to_user(data->pointer, ssid + 2, val);
	}

	return 0;
}


static int prism2_ioctl_giwrange(struct net_device *dev, struct iw_point *data)
{
	local_info_t *local = (local_info_t *) dev->priv;
	struct iw_range range;
	u16 val;
	int i;

	if (verify_area(VERIFY_WRITE, data->pointer, sizeof(range)))
		return -EINVAL;

	data->length = sizeof(range);
	memset(&range, 0, sizeof(range));

#if WIRELESS_EXT > 9
	/* TODO: could fill num_txpower and txpower array with
	 * something; however, there are 128 different values.. */

	range.txpower_capa = IW_TXPOW_DBM;
#endif /* WIRELESS_EXT > 9 */

#if WIRELESS_EXT > 10
	range.we_version_compiled = WIRELESS_EXT;
	range.we_version_source = 11;
#endif /* WIRELESS_EXT > 10 */

	range.min_nwid = range.max_nwid = 0;

	range.num_channels = FREQ_COUNT;

	val = 0;
	for (i = 0; i < FREQ_COUNT; i++) {
		if (local->channel_mask & (1 << i)) {
			range.freq[val].i = i + 1;
			range.freq[val].m = freq_list[i] * 100000;
			range.freq[val].e = 1;
			val++;
		}
		if (val == IW_MAX_FREQUENCIES)
			break;
	}
	range.num_frequency = val;

	range.max_qual.qual = 92; /* 0 .. 92 */
	range.max_qual.level = 154; /* 27 .. 154 */
	range.max_qual.noise = 154; /* 27 .. 154 */
	range.sensitivity = 3;

	if (copy_to_user(data->pointer, &range, sizeof(range)))
		return -EFAULT;

	return 0;
}


static int prism2_ioctl_siwmode(struct net_device *dev, __u32 mode)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 val;

	if (mode != IW_MODE_ADHOC && mode != IW_MODE_INFRA &&
	    mode != IW_MODE_MASTER)
		return -EOPNOTSUPP;

	if (mode == local->iw_mode)
		return 0;

	printk(KERN_DEBUG "prism2: %s: operating mode changed "
	       "%d -> %d\n", dev->name, local->iw_mode, mode);
	local->iw_mode = mode;

	val = HFA384X_PORTTYPE_HOSTAP;
	if (local->iw_mode == IW_MODE_ADHOC && local->pseudo_adhoc)
		val = HFA384X_PORTTYPE_PSEUDO_IBSS;
	else if (local->iw_mode == IW_MODE_ADHOC)
		val = HFA384X_PORTTYPE_IBSS;
	else if (local->iw_mode == IW_MODE_INFRA)
		val = HFA384X_PORTTYPE_BSS;

	if (hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &val, 2))
		return -EOPNOTSUPP;

	if (prism2_reset_port(dev))
		return -EINVAL; 

	return 0;
}


static int prism2_ioctl_siwpower(struct net_device *dev, struct iw_param *wrq)
{
	u16 val;
	int ret = 0;

	if (wrq->disabled) {
		val = __constant_cpu_to_le16(0);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPMENABLED, &val, 2);
		return ret;
	}

	switch (wrq->flags & IW_POWER_MODE) {
	case IW_POWER_UNICAST_R:
		val = __constant_cpu_to_le16(0);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFMULTICASTRECEIVE,
				      &val, 2);
		if (ret)
			return ret;
		val = __constant_cpu_to_le16(1);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPMENABLED, &val, 2);
		if (ret)
			return ret;
		break;
	case IW_POWER_ALL_R:
		val = __constant_cpu_to_le16(1);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFMULTICASTRECEIVE,
				      &val, 2);
		if (ret)
			return ret;
		val = __constant_cpu_to_le16(1);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPMENABLED, &val, 2);
		if (ret)
			return ret;
		break;
	case IW_POWER_ON:
		break;
	default:
		return -EINVAL;
	}
		
	if (wrq->flags & IW_POWER_TIMEOUT) {
		val = __constant_cpu_to_le16(1);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPMENABLED, &val, 2);
		if (ret)
			return ret;
		val = __cpu_to_le16(wrq->value / 1024);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPMHOLDOVERDURATION,
				      &val, 2);
		if (ret)
			return ret;
	}
	if (wrq->flags & IW_POWER_PERIOD) {
		val = __constant_cpu_to_le16(1);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPMENABLED, &val, 2);
		if (ret)
			return ret;
		val = __cpu_to_le16(wrq->value / 1024);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFMAXSLEEPDURATION,
				      &val, 2);
		if (ret)
			return ret;
	}

	return ret;
}


static int prism2_ioctl_giwpower(struct net_device *dev, struct iw_param *rrq)
{
	u16 enable, mcast;

	if (hfa384x_get_rid(dev, HFA384X_RID_CNFPMENABLED, &enable, 2, 1) < 0)
		return -EINVAL;

	if (!__le16_to_cpu(enable)) {
		rrq->disabled = 1;
		return 0;
	}

	rrq->disabled = 0;

	if ((rrq->flags & IW_POWER_TYPE) == IW_POWER_TIMEOUT) {
		u16 timeout;
		if (hfa384x_get_rid(dev, HFA384X_RID_CNFPMHOLDOVERDURATION,
				    &timeout, 2, 1) < 0)
			return -EINVAL;

		rrq->flags = IW_POWER_TIMEOUT;
		rrq->value = __le16_to_cpu(timeout) * 1024;
	} else {
		u16 period;
		if (hfa384x_get_rid(dev, HFA384X_RID_CNFMAXSLEEPDURATION,
				    &period, 2, 1) < 0)
			return -EINVAL;

		rrq->flags = IW_POWER_PERIOD;
		rrq->value = __le16_to_cpu(period) * 1024;
	}

	if (hfa384x_get_rid(dev, HFA384X_RID_CNFMULTICASTRECEIVE, &mcast,
			    2, 1) < 0)
		return -EINVAL;

	if (__le16_to_cpu(mcast))
		rrq->flags |= IW_POWER_ALL_R;
	else
		rrq->flags |= IW_POWER_UNICAST_R;

	return 0;
}
#endif /* WIRELESS_EXT > 8 */


#if WIRELESS_EXT > 10
static int prism2_ioctl_siwretry(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;

	if (rrq->disabled)
		return -EINVAL;

	/* setting retry limits is not supported with the current station
	 * firmware code; simulate this with alternative retry count for now */
	if (rrq->flags == IW_RETRY_LIMIT) {
		if (rrq->value < 0) {
			/* disable manual retry count setting and use firmware
			 * defaults */
			local->manual_retry_count = -1;
			local->tx_control &= ~HFA384X_TX_CTRL_ALT_RTRY;
		} else {
			u16 val;
			val = __cpu_to_le16(rrq->value);
			if (hfa384x_set_rid(dev,
					    HFA384X_RID_CNFALTRETRYCOUNT,
					    &val, 2)) {
				printk(KERN_DEBUG "%s: Alternate retry count "
				       "setting to %d failed\n",
				       dev->name, __le16_to_cpu(val));
				return -EOPNOTSUPP;
			}

			local->manual_retry_count = rrq->value;
			local->tx_control |= HFA384X_TX_CTRL_ALT_RTRY;
		}
		return 0;
	}

	return -EOPNOTSUPP;

#if 0
	/* what could be done, if firmware would support this.. */

	if (rrq->flags & IW_RETRY_LIMIT) {
		if (rrq->flags & IW_RETRY_MAX)
			HFA384X_RID_LONGRETRYLIMIT = rrq->value;
		else if (rrq->flags & IW_RETRY_MIN)
			HFA384X_RID_SHORTRETRYLIMIT = rrq->value;
		else {
			HFA384X_RID_LONGRETRYLIMIT = rrq->value;
			HFA384X_RID_SHORTRETRYLIMIT = rrq->value;
		}

	}

	if (rrq->flags & IW_RETRY_LIFETIME) {
		HFA384X_RID_MAXTRANSMITLIFETIME = rrq->value / 1024;
	}

	return 0;
#endif /* 0 */
}

static int prism2_ioctl_giwretry(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 shortretry, longretry, lifetime;

	if (hfa384x_get_rid(dev, HFA384X_RID_SHORTRETRYLIMIT, &shortretry, 2,
			    1) < 0 ||
	    hfa384x_get_rid(dev, HFA384X_RID_LONGRETRYLIMIT, &longretry, 2, 1)
	    < 0 ||
	    hfa384x_get_rid(dev, HFA384X_RID_MAXTRANSMITLIFETIME, &lifetime, 2,
			    1) < 0)
		return -EINVAL;

	le16_to_cpus(&shortretry);
	le16_to_cpus(&longretry);
	le16_to_cpus(&lifetime);

	rrq->disabled = 0;

	if ((rrq->flags & IW_RETRY_TYPE) == IW_RETRY_LIFETIME) {
		rrq->flags = IW_RETRY_LIFETIME;
		rrq->value = lifetime * 1024;
	} else {
		if (local->manual_retry_count >= 0) {
			rrq->flags = IW_RETRY_LIMIT;
			rrq->value = local->manual_retry_count;
		} else if ((rrq->flags & IW_RETRY_MAX)) {
			rrq->flags = IW_RETRY_LIMIT | IW_RETRY_MAX;
			rrq->value = longretry;
		} else {
			rrq->flags = IW_RETRY_LIMIT;
			rrq->value = shortretry;
			if (shortretry != longretry)
				rrq->flags |= IW_RETRY_MIN;
		}
	}
	return 0;
}
#endif /* WIRELESS_EXT > 10 */


#if WIRELESS_EXT > 9
/* Map HFA386x's CR31 to and from dBm with some sort of ad hoc mapping..
 * This version assumes following mapping:
 * CR31 is 7-bit value with -64 to +63 range.
 * -64 is mapped into +20dBm and +63 into -43dBm.
 * This is certainly not an exact mapping for every card, but at least
 * increasing dBm value should correspond to increasing TX power.
 */

static int prism2_txpower_hfa386x_to_dBm(u16 val)
{
	signed char tmp;

	if (val > 255)
		val = 255;

	tmp = val;
	tmp >>= 2;

	return -12 - tmp;
}

static u16 prism2_txpower_dBm_to_hfa386x(int val)
{
	signed char tmp;

	if (val > 20)
		return 128;
	else if (val < -43)
		return 127;

	tmp = val;
	tmp = -12 - tmp;
	tmp <<= 2;
	
	return (unsigned char) tmp;
}


static int prism2_ioctl_siwtxpow(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	char *tmp;
	u16 val;
	int ret = 0;

	if (rrq->disabled) {
		if (local->txpower_type != PRISM2_TXPOWER_OFF) {
			val = 0xff; /* use all standby and sleep modes */
			ret = hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF,
					  HFA386X_CR_A_D_TEST_MODES2,
					  &val, NULL);
			printk(KERN_DEBUG "%s: Turning radio off: %s\n",
			       dev->name, ret ? "failed" : "OK");
			local->txpower_type = PRISM2_TXPOWER_OFF;
		}
		return (ret ? -EOPNOTSUPP : 0);
	}

	if (local->txpower_type == PRISM2_TXPOWER_OFF) {
		val = 0; /* disable all standby and sleep modes */
		ret = hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF,
				  HFA386X_CR_A_D_TEST_MODES2, &val, NULL);
		printk(KERN_DEBUG "%s: Turning radio on: %s\n",
		       dev->name, ret ? "failed" : "OK");
		local->txpower_type = PRISM2_TXPOWER_UNKNOWN;
	}

	if (!rrq->fixed && local->txpower_type != PRISM2_TXPOWER_AUTO) {
		printk(KERN_DEBUG "Setting ALC on\n");
		val = HFA384X_TEST_CFG_BIT_ALC;
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_CFG_BITS << 8), 1, &val, NULL);
		local->txpower_type = PRISM2_TXPOWER_AUTO;
		return 0;
	}

	if (local->txpower_type != PRISM2_TXPOWER_FIXED) {
		printk(KERN_DEBUG "Setting ALC off\n");
		val = HFA384X_TEST_CFG_BIT_ALC;
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_CFG_BITS << 8), 0, &val, NULL);
			local->txpower_type = PRISM2_TXPOWER_FIXED;
	}

	if (rrq->flags == IW_TXPOW_DBM)
		tmp = "dBm";
	else if (rrq->flags == IW_TXPOW_MWATT)
		tmp = "mW";
	else
		tmp = "UNKNOWN";
	printk(KERN_DEBUG "Setting TX power to %d %s\n", rrq->value, tmp);

	if (rrq->flags != IW_TXPOW_DBM) {
		printk("SIOCSIWTXPOW with mW is not supported; use dBm\n");
		return -EOPNOTSUPP;
	}

	local->txpower = rrq->value;
	val = prism2_txpower_dBm_to_hfa386x(local->txpower);
	if (hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF,
			HFA386X_CR_MANUAL_TX_POWER, &val, NULL))
		ret = -EOPNOTSUPP;

	return ret;
}

static int prism2_ioctl_giwtxpow(struct net_device *dev, struct iw_param *rrq)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 resp0;

	rrq->flags = IW_TXPOW_DBM;
	rrq->disabled = 0;
	rrq->fixed = 0;

	if (local->txpower_type == PRISM2_TXPOWER_AUTO) {
		if (hfa384x_cmd(dev, HFA384X_CMDCODE_READMIF,
				HFA386X_CR_MANUAL_TX_POWER,
				NULL, &resp0) == 0) {
			rrq->value = prism2_txpower_hfa386x_to_dBm(resp0);
		} else {
			/* Could not get real txpower; guess 15 dBm */
			rrq->value = 15;
		}
	} else if (local->txpower_type == PRISM2_TXPOWER_OFF) {
		rrq->value = 0;
		rrq->disabled = 1;
	} else if (local->txpower_type == PRISM2_TXPOWER_FIXED) {
		rrq->value = local->txpower;
		rrq->fixed = 1;
	} else {
		printk("SIOCGIWTXPOW - unknown txpower_type=%d\n",
		       local->txpower_type);
	}
	return 0;
}
#endif /* WIRELESS_EXT > 9 */


#if WIRELESS_EXT > 8
static int prism2_ioctl_giwpriv(struct net_device *dev, struct iw_point *data)
{
	struct iw_priv_args priv[] = {
#ifdef PRISM2_MONITOR
		{ PRISM2_IOCTL_MONITOR,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "monitor" },
#endif /* PRISM2_MONITOR */
		{ PRISM2_IOCTL_PRISM2_PARAM,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0,
		  "prism2_param" },
		{ PRISM2_IOCTL_READMIF,
		  IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | 1,
		  IW_PRIV_TYPE_BYTE | IW_PRIV_SIZE_FIXED | 1, "readmif" },
		{ PRISM2_IOCTL_WRITEMIF,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0, "writemif" },
		{ PRISM2_IOCTL_RESET,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "reset" },
		{ PRISM2_IOCTL_INQUIRE,
		  IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 1, 0, "inquire" },
		{ PRISM2_IOCTL_WDS_ADD,
		  IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 18, 0, "wds_add" },
		{ PRISM2_IOCTL_WDS_DEL,
		  IW_PRIV_TYPE_CHAR | IW_PRIV_SIZE_FIXED | 18, 0, "wds_del" }
	};

	if (!data->pointer ||
	    verify_area(VERIFY_WRITE, data->pointer, sizeof(priv)))
		return -EINVAL;

	data->length = sizeof(priv) / sizeof(priv[0]);
	copy_to_user(data->pointer, priv, sizeof(priv));
	return 0;
}
#endif /* WIRELESS_EXT > 8 */
#endif /* WIRELESS_EXT */


static int prism2_ioctl_priv_inquire(struct net_device *dev, int *i)
{
	if (hfa384x_cmd(dev, HFA384X_CMDCODE_INQUIRE, *i, NULL, NULL))
		return -EOPNOTSUPP;

	return 0;
}


static int prism2_ioctl_priv_prism2_param(struct net_device *dev, int *i)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int param = *i;
	int value = *(i + 1);
	int ret = 0;
	u16 val;

	switch (param) {
	case PRISM2_PARAM_PTYPE:
		val = __cpu_to_le16(value);
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &val, 2)) {
			ret = -EOPNOTSUPP;
			break;
		}

		if (prism2_reset_port(dev))
			ret = -EINVAL;
		break;

	case PRISM2_PARAM_TXRATECTRL:
		local->fw_tx_rate_control = value;
		break;

	case PRISM2_PARAM_BEACON_INT:
		val = __cpu_to_le16(value);
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFBEACONINT, &val, 2) ||
		    prism2_reset_port(dev))
			ret = -EINVAL;
		else
			local->beacon_int = value;
		break;

	case PRISM2_PARAM_PSEUDO_IBSS:
		if (value == local->pseudo_adhoc)
			break;

		if (value != 0 && value != 1) {
			ret = -EINVAL;
			break;
		}

		printk(KERN_DEBUG "prism2: %s: pseudo IBSS change %d -> %d\n",
		       dev->name, local->pseudo_adhoc, value);
		local->pseudo_adhoc = value;
		if (local->iw_mode != IW_MODE_ADHOC)
			break;

		if (local->pseudo_adhoc)
			val = __constant_cpu_to_le16(
				HFA384X_PORTTYPE_PSEUDO_IBSS);
		else
			val = __constant_cpu_to_le16(HFA384X_PORTTYPE_IBSS);

		if (hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &val, 2)) {
			ret = -EOPNOTSUPP;
			break;
		}

		if (prism2_reset_port(dev))
			ret = -EINVAL;
		break;

	case PRISM2_PARAM_ALC:
		printk(KERN_DEBUG "%s: %s ALC\n", dev->name,
		       value == 0 ? "Disabling" : "Enabling");
		val = HFA384X_TEST_CFG_BIT_ALC;
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_CFG_BITS << 8),
			    value == 0 ? 0 : 1, &val, NULL);
		break;

	case PRISM2_PARAM_TXPOWER:
		val = value;
		if (hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF,
				HFA386X_CR_MANUAL_TX_POWER, &val, NULL))
			ret = -EOPNOTSUPP;
		break;

	case PRISM2_PARAM_DUMP:
		local->frame_dump = value;
		break;

	case PRISM2_PARAM_OTHER_AP_POLICY:
		if (value < 0 || value > 3) {
			ret = -EINVAL;
			break;
		}
		if (local->ap != NULL)
			local->ap->ap_policy = value;
		break;

	case PRISM2_PARAM_AP_MAX_INACTIVITY:
		if (value < 0 || value > 7 * 24 * 60 * 60) {
			ret = -EINVAL;
			break;
		}
		if (local->ap != NULL)
			local->ap->max_inactivity = value * HZ;
		break;

	case PRISM2_PARAM_AP_BRIDGE_PACKETS:
		if (local->ap != NULL)
			local->ap->bridge_packets = value;
		break;

	case PRISM2_PARAM_DTIM_PERIOD:
		if (value < 0 || value > 65535) {
			ret = -EINVAL;
			break;
		}
		val = __cpu_to_le16(value);
		if (hfa384x_set_rid(dev, HFA384X_RID_CNFOWNDTIMPERIOD, &val,
				    2) ||
		    prism2_reset_port(dev))
			ret = -EINVAL;
		else
			local->dtim_period = value;
		break;

	case PRISM2_PARAM_AP_NULLFUNC_ACK:
		if (local->ap != NULL)
			local->ap->nullfunc_ack = value;
		break;

	case PRISM2_PARAM_MAX_WDS:
		local->wds_max_connections = value;
		break;

	case PRISM2_PARAM_AP_AUTOM_AP_WDS:
		if (local->ap != NULL)
			local->ap->autom_ap_wds = value;
		break;

	case PRISM2_PARAM_AP_AUTH_ALGS:
		if (local->ap != NULL)
			local->ap->auth_algs = value;
		break;

#ifdef PRISM2_MONITOR
	case PRISM2_PARAM_MONITOR_ALLOW_FCSERR:
		local->monitor_allow_fcserr = value;
		break;
#endif /* PRISM2_MONITOR */

	default:
		printk(KERN_DEBUG "%s: prism2_param: unknown param %d\n",
		       dev->name, param);
		ret = -EOPNOTSUPP;
		break;
	}

	return ret;
}


static int prism2_ioctl_priv_readmif(struct net_device *dev, struct iwreq *wrq)
{
	u16 val, resp0;

	val = *((int *) wrq->u.name);
	if (hfa384x_cmd(dev, HFA384X_CMDCODE_READMIF, val, NULL, &resp0))
		return -EOPNOTSUPP;
	else
		*(wrq->u.name) = resp0;

	return 0;
}


static int prism2_ioctl_priv_writemif(struct net_device *dev, int *i)
{
	u16 cr, val;

	cr = *i;
	val = *(i + 1);
	if (hfa384x_cmd(dev, HFA384X_CMDCODE_WRITEMIF, cr, &val, NULL))
		return -EOPNOTSUPP;

	return 0;
}


#ifdef PRISM2_MONITOR
static int prism2_ioctl_priv_monitor(struct net_device *dev, int *i)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int ret = 0;
	u16 tmp;

	if (*i == 0) {
		struct sock *nl = local->nl_monitor;
		printk("Disabling monitor mode\n");
		dev->type = ARPHRD_ETHER;
		dev->hard_header_parse = local->saved_eth_header_parse;
		local->monitor_type = PRISM2_MONITOR_OFF;
		local->nl_monitor = NULL;
		if (nl != NULL) {
			if (nl->socket == NULL)
				printk("nl->socket == NULL\n");

			/* FIX: this seems to crash the kernel, but
			 * something needs to be released.. */
			/* sock_release(nl->socket); */
		}
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_STOP << 8),
			    0, NULL, NULL);
		tmp = __constant_cpu_to_le16(HFA384X_PORTTYPE_HOSTAP);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &tmp, 2);
		prism2_reset_port(dev);
	} else if (*i == 1 || *i == 2) {
		printk("Enabling monitor mode(%i)\n", *i);
		if (*i == 1) {
			dev->type = ARPHRD_ETHER;
			dev->hard_header_parse = local->saved_eth_header_parse;
			local->monitor_type = PRISM2_MONITOR_NL;
		} else {
			dev->type = ARPHRD_IEEE80211;
			dev->hard_header_parse = prism2_80211_header_parse;
			local->monitor_type = PRISM2_MONITOR_DEV;
		}
		if (local->monitor_type == PRISM2_MONITOR_NL &&
		    local->nl_monitor == NULL) {
			local->nl_monitor =
				netlink_kernel_create(NETLINK_USERSOCK,
						      monitor_tx);
			if (local->nl_monitor == NULL)
				printk(KERN_WARNING
				       "netlink_kernel_create failed\n");
		}
#ifdef PRISM2_MONITOR_PACKET_INJECT
		tmp = __constant_cpu_to_le16(HFA384X_PORTTYPE_HOSTAP);
#else /* PRISM2_MONITOR_PACKET_INJECT */
		tmp = __constant_cpu_to_le16(HFA384X_PORTTYPE_PSEUDO_IBSS);
#endif /* PRISM2_MONITOR_PACKET_INJECT */
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFPORTTYPE, &tmp, 2);
		if (ret) {
			printk("Port type setting to pseudo IBSS failed\n");
			return -EOPNOTSUPP;
		}
		/* Host decrypt is needed to get the IV and ICV fields;
		 * however, monitor mode seems to remove WEP flag from frame
		 * control field */
		tmp = __constant_cpu_to_le16(HFA384X_WEPFLAGS_HOSTENCRYPT |
					     HFA384X_WEPFLAGS_HOSTDECRYPT);
		ret = hfa384x_set_rid(dev, HFA384X_RID_CNFWEPFLAGS, &tmp, 2);
		if (ret) {
			printk("WEP flags setting failed\n");
			return -EOPNOTSUPP;
		}
#if 0
		tmp = __constant_cpu_to_le16(1);
		ret = hfa384x_set_rid(dev, HFA384X_RID_PROMISCUOUSMODE,
				      &tmp, 2);
		if (ret) {
			printk("Promiscuous mode setting failed\n");
			return -EOPNOTSUPP;
		}
#endif /* 0 */
		prism2_reset_port(dev);
#ifndef PRISM2_MONITOR_PACKET_INJECT
		hfa384x_cmd(dev, HFA384X_CMDCODE_TEST |
			    (HFA384X_TEST_MONITOR << 8),
			    0, NULL, NULL);
#endif /* !PRISM2_MONITOR_PACKET_INJECT */
	} else
		ret = -EINVAL;

	return ret;
}
#endif /* PRISM2_MONITOR */


static int prism2_ioctl_priv_reset(struct net_device *dev, int *i)
{
	printk(KERN_DEBUG "%s: manual reset request(%d)\n", dev->name, *i);
	switch (*i) {
	case 0:
		/* Disable and enable card */
		prism2_hw_shutdown(dev, 1);
		prism2_hw_config(dev, 0);
		break;

	case 1:
		/* COR sreset */
		prism2_hw_reset(dev);
		break;

	case 2:
		/* Disable and enable port 0 */
		prism2_reset_port(dev);
		break;

	default:
		printk(KERN_DEBUG "Unknown reset request %d\n", *i);
		return -EOPNOTSUPP;
	}

	return 0;
}


static inline int hex2int(char c)
{
	if (c >= '0' && c <= '9')
		return (c - '0');
	if (c >= 'a' && c <= 'f')
		return (c - 'a' + 10);
	if (c >= 'A' && c <= 'F')
		return (c - 'A' + 10);
	return -1;
}

static int macstr2addr(char *macstr, u8 *addr)
{
	int i, val, val2;
	char *pos = macstr;

	for (i = 0; i < 6; i++) {
		val = hex2int(*pos++);
		if (val < 0)
			return -1;
		val2 = hex2int(*pos++);
		if (val2 < 0)
			return -1;
		addr[i] = (val * 16 + val2) & 0xff;

		if (i < 5 && *pos++ != ':')
			return -1;
	}

	return 0;
}


static int prism2_ioctl_priv_wds(struct net_device *dev, int add, char *macstr)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u8 addr[6];

	if (macstr2addr(macstr, addr)) {
		printk(KERN_DEBUG "Invalid MAC address\n");
		return -EINVAL;
	}

	if (add)
		return prism2_wds_add(local, addr, 1);
	else
		return prism2_wds_del(local, addr, 1, 0);
}


static int prism2_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
#ifdef WIRELESS_EXT
	struct iwreq *wrq = (struct iwreq *) ifr;
#endif
#if WIRELESS_EXT > 8
	local_info_t *local = (local_info_t *) dev->priv;
#endif /* WIRELESS_EXT > 8 */
	int ret = 0;

	switch (cmd) {

#ifdef WIRELESS_EXT
	case SIOCGIWNAME:
		strcpy(wrq->u.name, "IEEE 802.11-DS");
		break;

	case SIOCSIWFREQ:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwfreq(dev, &wrq->u.freq);
		break;
	case SIOCGIWFREQ:
		ret = prism2_ioctl_giwfreq(dev, &wrq->u.freq);
		break;

#if WIRELESS_EXT > 8
	case SIOCSIWESSID:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwessid(dev, &wrq->u.data);
		break;
	case SIOCGIWESSID:
		ret = prism2_ioctl_giwessid(dev, &wrq->u.data);
		break;

	case SIOCSIWRATE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwrate(dev, &wrq->u.bitrate);
		break;
	case SIOCGIWRATE:
		ret = prism2_ioctl_giwrate(dev, &wrq->u.bitrate);
		break;

	case SIOCSIWRTS:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwrts(dev, &wrq->u.rts);
		break;
	case SIOCGIWRTS:
		ret = prism2_ioctl_giwrts(dev, &wrq->u.rts);
		break;

	case SIOCSIWFRAG:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwfrag(dev, &wrq->u.rts);
		break;
	case SIOCGIWFRAG:
		ret = prism2_ioctl_giwfrag(dev, &wrq->u.rts);
		break;

	case SIOCSIWENCODE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwencode(dev, &wrq->u.encoding);
		break;
	case SIOCGIWENCODE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_giwencode(dev, &wrq->u.encoding);
		break;
#endif /* WIRELESS_EXT > 8 */

	case SIOCGIWAP:
		ret = prism2_ioctl_giwap(dev, &wrq->u.ap_addr);
		break;

#if WIRELESS_EXT > 8
	case SIOCSIWNICKN:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwnickn(dev, &wrq->u.data);
		break;
	case SIOCGIWNICKN:
		ret = prism2_ioctl_giwnickn(dev, &wrq->u.data);
		break;

	case SIOCGIWSPY:
		ret = prism2_ioctl_giwspy(dev, &wrq->u.data);
		break;

	case SIOCGIWRANGE:
		ret = prism2_ioctl_giwrange(dev, &wrq->u.data);
		break;

	case SIOCSIWSENS:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwsens(dev, &wrq->u.sens);
		break;
	case SIOCGIWSENS:
		ret = prism2_ioctl_giwsens(dev, &wrq->u.sens);
		break;

	case SIOCGIWAPLIST:
		ret = prism2_ioctl_giwaplist(dev, &wrq->u.data);
		break;

	case SIOCSIWMODE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwmode(dev, wrq->u.mode);
		break;
	case SIOCGIWMODE:
		wrq->u.mode = local->iw_mode;
		break;

	case SIOCSIWPOWER:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwpower(dev, &wrq->u.power);
		break;
	case SIOCGIWPOWER:
		ret = prism2_ioctl_giwpower(dev, &wrq->u.power);
		break;
#endif /* WIRELESS_EXT > 8 */

#if WIRELESS_EXT > 9
	case SIOCSIWTXPOW:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwtxpow(dev, &wrq->u.txpower);
		break;
	case SIOCGIWTXPOW:
		ret = prism2_ioctl_giwtxpow(dev, &wrq->u.txpower);
		break;
#endif /* WIRELESS_EXT > 9 */

#if WIRELESS_EXT > 10
	case SIOCSIWRETRY:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_siwretry(dev, &wrq->u.retry);
		break;
	case SIOCGIWRETRY:
		ret = prism2_ioctl_giwretry(dev, &wrq->u.retry);
		break;
#endif /* WIRELESS_EXT > 10 */

#if WIRELESS_EXT > 8
	case SIOCGIWPRIV:
		ret = prism2_ioctl_giwpriv(dev, &wrq->u.data);
		break;
#endif /* WIRELESS_EXT > 8 */

	case PRISM2_IOCTL_INQUIRE:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_inquire(dev, (int *) wrq->u.name);
		break;

	case PRISM2_IOCTL_PRISM2_PARAM:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_prism2_param(dev,
							  (int *) wrq->u.name);
		break;

	case PRISM2_IOCTL_READMIF:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_readmif(dev, wrq);
		break;

#ifdef PRISM2_MONITOR
	case PRISM2_IOCTL_MONITOR:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_monitor(dev, (int *) wrq->u.name);
		break;
#endif /* PRISM2_MONITOR */

	case PRISM2_IOCTL_RESET:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_reset(dev, (int *) wrq->u.name);
		break;

	case PRISM2_IOCTL_WRITEMIF:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_writemif(dev,
						      (int *) wrq->u.name);
		break;

	case PRISM2_IOCTL_WDS_ADD:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_wds(dev, 1,
						 (char *) wrq->u.data.pointer);
		break;

	case PRISM2_IOCTL_WDS_DEL:
		if (!capable(CAP_NET_ADMIN)) ret = -EPERM;
		else ret = prism2_ioctl_priv_wds(dev, 0,
						 (char *) wrq->u.data.pointer);
		break;


	/* not supported wireless extensions */
#ifdef SIOCSIWNAME
	case SIOCSIWNAME:
#endif
	case SIOCSIWNWID:
	case SIOCGIWNWID:
	case SIOCSIWPRIV:
		ret = -EOPNOTSUPP;
		break;

	/* FIX: add support for (at least some of) these: */
	case SIOCSIWRANGE:
	case SIOCSIWSPY:
	case SIOCSIWAP:
		printk(KERN_DEBUG "%s: %s unsupported WIRELESS_EXT "
		       "ioctl(0x%04x)\n", dev_info, dev->name, cmd);
		ret = -EOPNOTSUPP;
		break;

#endif /* WIRELESS_EXT */

	default:
		printk(KERN_DEBUG "%s: %s unsupported ioctl(0x%04x)\n",
		       dev_info, dev->name, cmd);
		ret = -EOPNOTSUPP;
		break;
	}

	return ret;
}


static int prism2_debug_proc_read(char *page, char **start, off_t off,
				  int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	int i;
	unsigned long flags;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	spin_lock_irqsave(&local->txfidlock, flags);
	p += sprintf(p, "next_txfid=%d next_alloc=%d\n",
		     local->next_txfid, local->next_alloc);
	for (i = 0; i < PRISM2_TXFID_COUNT; i++)
		p += sprintf(p, "FID: tx=%04X intransmit=%04X\n",
			     local->txfid[i], local->intransmitfid[i]);
	spin_unlock_irqrestore(&local->txfidlock, flags);
	p += sprintf(p, "FW TX rate control: %d\n", local->fw_tx_rate_control);
	p += sprintf(p, "beacon_int=%d\n", local->beacon_int);
	p += sprintf(p, "dtim_period=%d\n", local->dtim_period);
	p += sprintf(p, "wds_max_connections=%d\n",
		     local->wds_max_connections);

	return (p - page);
}


static int prism2_stats_proc_read(char *page, char **start, off_t off,
				  int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	struct comm_tallies_sums *sums = (struct comm_tallies_sums *)
		&local->comm_tallies;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

	p += sprintf(p, "TxUnicastFrames=%u\n", sums->tx_unicast_frames);
	p += sprintf(p, "TxMulticastframes=%u\n", sums->tx_multicast_frames);
	p += sprintf(p, "TxFragments=%u\n", sums->tx_fragments);
	p += sprintf(p, "TxUnicastOctets=%u\n", sums->tx_unicast_octets);
	p += sprintf(p, "TxMulticastOctets=%u\n", sums->tx_multicast_octets);
	p += sprintf(p, "TxDeferredTransmissions=%u\n",
		     sums->tx_deferred_transmissions);
	p += sprintf(p, "TxSingleRetryFrames=%u\n",
		     sums->tx_single_retry_frames);
	p += sprintf(p, "TxMultipleRetryFrames=%u\n",
		     sums->tx_multiple_retry_frames);
	p += sprintf(p, "TxRetryLimitExceeded=%u\n",
		     sums->tx_retry_limit_exceeded);
	p += sprintf(p, "TxDiscards=%u\n", sums->tx_discards);
	p += sprintf(p, "RxUnicastFrames=%u\n", sums->rx_unicast_frames);
	p += sprintf(p, "RxMulticastFrames=%u\n", sums->rx_multicast_frames);
	p += sprintf(p, "RxFragments=%u\n", sums->rx_fragments);
	p += sprintf(p, "RxUnicastOctets=%u\n", sums->rx_unicast_octets);
	p += sprintf(p, "RxMulticastOctets=%u\n", sums->rx_multicast_octets);
	p += sprintf(p, "RxFCSErrors=%u\n", sums->rx_fcs_errors);
	p += sprintf(p, "RxDiscardsNoBuffer=%u\n",
		     sums->rx_discards_no_buffer);
	p += sprintf(p, "TxDiscardsWrongSA=%u\n", sums->tx_discards_wrong_sa);
	p += sprintf(p, "RxDiscardsWEPUndecryptable=%u\n",
		     sums->rx_discards_wep_undecryptable);
	p += sprintf(p, "RxMessageInMsgFragments=%u\n",
		     sums->rx_message_in_msg_fragments);
	p += sprintf(p, "RxMessageInBadMsgFragments=%u\n",
		     sums->rx_message_in_bad_msg_fragments);
	/* FIX: this may grow too long for one page(?) */

	return (p - page);
}


static int prism2_registers_proc_read(char *page, char **start, off_t off,
				      int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;

	if (off != 0) {
		*eof = 1;
		return 0;
	}

#define SHOW_REG(n) \
p += sprintf(p, "%s=%04x\n", #n, \
hfa384x_read_reg(local->dev, HFA384X_##n##_OFF))

	SHOW_REG(CMD);
	SHOW_REG(PARAM0);
	SHOW_REG(PARAM1);
	SHOW_REG(PARAM2);
	SHOW_REG(STATUS);
	SHOW_REG(RESP0);
	SHOW_REG(RESP1);
	SHOW_REG(RESP2);
	SHOW_REG(INFOFID);
	SHOW_REG(CONTROL);
	SHOW_REG(SELECT0);
	SHOW_REG(SELECT1);
	SHOW_REG(OFFSET0);
	SHOW_REG(OFFSET1);
	SHOW_REG(RXFID);
	SHOW_REG(ALLOCFID);
	SHOW_REG(TXCOMPLFID);
	SHOW_REG(SWSUPPORT0);
	SHOW_REG(SWSUPPORT1);
	SHOW_REG(SWSUPPORT2);
	SHOW_REG(EVSTAT);
	SHOW_REG(INTEN);
	SHOW_REG(EVACK);
	/* Do not read data registers, because they change the state of the
	 * MAC (offset += 2) */
	/* SHOW_REG(DATA0); */
	/* SHOW_REG(DATA1); */
	SHOW_REG(AUXPAGE);
	SHOW_REG(AUXOFFSET);
	/* SHOW_REG(AUXDATA); */

	return (p - page);
}

#define RID(n,t) { HFA384X_RID_##n, #n, t }
enum { RID_HEXDUMP, RID_WORD, RID_HWADDR, RID_STRING, RID_COMPID,
       RID_SUPRANGE };

static struct {
	u16 rid;
	char *name;
	int type;
} rid_table[] = {
	RID(CNFPORTTYPE, RID_WORD),
	RID(CNFOWNMACADDR, RID_HWADDR),
	RID(CNFDESIREDSSID, RID_STRING),
	RID(CNFOWNCHANNEL, RID_WORD),
	RID(CNFOWNSSID, RID_STRING),
	RID(CNFOWNATIMWINDOW, RID_WORD),
	RID(CNFSYSTEMSCALE, RID_WORD),
	RID(CNFMAXDATALEN, RID_WORD),
	RID(CNFWDSADDRESS, RID_HWADDR),
	RID(CNFPMENABLED, RID_WORD),
	RID(CNFPMEPS, RID_WORD),
	RID(CNFMULTICASTRECEIVE, RID_WORD),
	RID(CNFMAXSLEEPDURATION, RID_WORD),
	RID(CNFPMHOLDOVERDURATION, RID_WORD),
	RID(CNFOWNNAME, RID_STRING),
	RID(CNFOWNDTIMPERIOD, RID_WORD),
	RID(CNFWDSADDRESS1, RID_HWADDR),
	RID(CNFWDSADDRESS2, RID_HWADDR),
	RID(CNFWDSADDRESS3, RID_HWADDR),
	RID(CNFWDSADDRESS4, RID_HWADDR),
	RID(CNFWDSADDRESS5, RID_HWADDR),
	RID(CNFWDSADDRESS6, RID_HWADDR),
	RID(CNFMULTICASTPMBUFFERING, RID_WORD),
	RID(UNKNOWN1, RID_WORD),
	RID(UNKNOWN2, RID_WORD),
	RID(CNFWEPDEFAULTKEYID, RID_WORD),
	RID(CNFDEFAULTKEY0, RID_HEXDUMP),
	RID(CNFDEFAULTKEY1, RID_HEXDUMP),
	RID(CNFDEFAULTKEY2, RID_HEXDUMP),
	RID(CNFDEFAULTKEY3, RID_HEXDUMP),
	RID(CNFWEPFLAGS, RID_HEXDUMP),
	RID(CNFWEPKEYMAPPINGTABLE, RID_HEXDUMP),
	RID(CNFAUTHENTICATION, RID_WORD),
	RID(CNFMAXASSOCSTA, RID_WORD),
	RID(CNFTXCONTROL, RID_WORD),
	RID(CNFROAMINGMODE, RID_WORD),
	RID(CNFHOSTAUTHENTICATION, RID_WORD),
	RID(CNFRCVCRCERROR, RID_WORD),
	RID(CNFMMLIFE, RID_WORD),
	RID(CNFALTRETRYCOUNT, RID_WORD),
	RID(CNFBEACONINT, RID_WORD),
	RID(CNFAPPCFINFO, RID_HEXDUMP),
	RID(CNFSTAPCFINFO, RID_HEXDUMP),
	RID(CNFPRIORITYQUSAGE, RID_HEXDUMP),
	RID(CNFTIMCTRL, RID_WORD),
	RID(UNKNOWN3, RID_HEXDUMP),
	RID(CNFTHIRTY2TALLY, RID_WORD),
	RID(CNFENHSECURITY, RID_WORD),
	RID(CNFDBMADJUST, RID_WORD),
	RID(GROUPADDRESSES, RID_HEXDUMP),
	RID(CREATEIBSS, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD, RID_WORD),
	RID(RTSTHRESHOLD, RID_WORD),
	RID(TXRATECONTROL, RID_WORD),
	RID(PROMISCUOUSMODE, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD0, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD1, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD2, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD3, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD4, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD5, RID_WORD),
	RID(FRAGMENTATIONTHRESHOLD6, RID_WORD),
	RID(RTSTHRESHOLD0, RID_WORD),
	RID(RTSTHRESHOLD1, RID_WORD),
	RID(RTSTHRESHOLD2, RID_WORD),
	RID(RTSTHRESHOLD3, RID_WORD),
	RID(RTSTHRESHOLD4, RID_WORD),
	RID(RTSTHRESHOLD5, RID_WORD),
	RID(RTSTHRESHOLD6, RID_WORD),
	RID(TXRATECONTROL0, RID_WORD),
	RID(TXRATECONTROL1, RID_WORD),
	RID(TXRATECONTROL2, RID_WORD),
	RID(TXRATECONTROL3, RID_WORD),
	RID(TXRATECONTROL4, RID_WORD),
	RID(TXRATECONTROL5, RID_WORD),
	RID(TXRATECONTROL6, RID_WORD),
	RID(CNFSHORTPREAMBLE, RID_WORD),
	RID(CNFEXCLUDELONGPREAMBLE, RID_WORD),
	RID(CNFAUTHENTICATIONRSPTO, RID_WORD),
	RID(CNFBASICRATES, RID_HEXDUMP),
	RID(CNFSUPPORTEDRATES, RID_HEXDUMP),
	RID(UNKNOWN5, RID_WORD),
	RID(WEPKEYDISABLE, RID_WORD),
	RID(TICKTIME, RID_WORD),
	RID(SCANREQUEST, RID_HEXDUMP),
	RID(JOINREQUEST, RID_HEXDUMP),
	RID(AUTHENTICATESTATION, RID_HEXDUMP),
	RID(CHANNELINFOREQUEST, RID_HEXDUMP),
	RID(HOSTSCAN, RID_HEXDUMP),

	RID(MAXLOADTIME, RID_WORD),
	RID(DOWNLOADBUFFER, RID_HEXDUMP),
	RID(PRIID, RID_COMPID),
	RID(PRISUPRANGE, RID_SUPRANGE),
	RID(CFIACTRANGES, RID_SUPRANGE),
	RID(NICSERNUM, RID_STRING),
	RID(NICID, RID_COMPID),
	RID(MFISUPRANGE, RID_SUPRANGE),
	RID(CFISUPRANGE, RID_SUPRANGE),
	RID(CHANNELLIST, RID_HEXDUMP),
	RID(REGULATORYDOMAINS, RID_STRING),
	RID(TEMPTYPE, RID_WORD),
	RID(CIS, RID_HEXDUMP),
	RID(STAID, RID_COMPID),
	RID(STASUPRANGE, RID_SUPRANGE),
	RID(MFIACTRANGES, RID_SUPRANGE),
	RID(CFIACTRANGES2, RID_SUPRANGE),
	RID(PORTSTATUS, RID_WORD),
	RID(CURRENTSSID, RID_STRING),
	RID(CURRENTBSSID, RID_HWADDR),
	RID(COMMSQUALITY, RID_HEXDUMP),
	RID(CURRENTTXRATE, RID_WORD),
	RID(CURRENTBEACONINTERVAL, RID_WORD),
	RID(CURRENTSCALETHRESHOLDS, RID_HEXDUMP),
	RID(PROTOCOLRSPTIME, RID_WORD),
	RID(SHORTRETRYLIMIT, RID_WORD),
	RID(LONGRETRYLIMIT, RID_WORD),
	RID(MAXTRANSMITLIFETIME, RID_WORD),
	RID(MAXRECEIVELIFETIME, RID_WORD),
	RID(CFPOLLABLE, RID_WORD),
	RID(AUTHENTICATIONALGORITHMS, RID_HEXDUMP),
	RID(PRIVACYOPTIONIMPLEMENTED, RID_WORD),
	RID(DBMCOMMSQUALITY, RID_HEXDUMP),
	RID(CURRENTTXRATE1, RID_WORD),
	RID(CURRENTTXRATE2, RID_WORD),
	RID(CURRENTTXRATE3, RID_WORD),
	RID(CURRENTTXRATE4, RID_WORD),
	RID(CURRENTTXRATE5, RID_WORD),
	RID(CURRENTTXRATE6, RID_WORD),
	RID(OWNMACADDR, RID_HWADDR),
	RID(SCANRESULTSTABLE, RID_HEXDUMP),
	RID(HOSTSCANRESULTS, RID_HEXDUMP),
	RID(AUTHENTICATIONUSED, RID_HEXDUMP),
	RID(PHYTYPE, RID_WORD),
	RID(CURRENTCHANNEL, RID_WORD),
	RID(CURRENTPOWERSTATE, RID_WORD),
	RID(CCAMODE, RID_WORD),
	RID(SUPPORTEDDATARATES, RID_HEXDUMP),

	RID(BUILDSEQ, RID_HEXDUMP),
	RID(FWID, RID_STRING)
};

#define PROC_LIMIT (PAGE_SIZE - 80)

static int prism2_rids_proc_read(char *page, char **start, off_t off,
				 int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	int i, j, len, strlen, total = 0;
	unsigned char buf[256];
	struct hfa384x_comp_ident *compid;
	struct hfa384x_sup_range *range;

	for (i = 0; i < sizeof(rid_table) / sizeof(rid_table[0]); i++) {
		if (total + (p - page) <= off) {
			total += p - page;
			p = page;
		}
		if (total + (p - page) > off + count)
			break;
		if ((p - page) > PROC_LIMIT)
			break;

		len = hfa384x_get_rid(local->dev, rid_table[i].rid, buf,
				      sizeof(buf), 0);
		if (len < 0)
			continue;

		p += sprintf(p, "%04X=%s=", rid_table[i].rid,
			     rid_table[i].name);

		switch (rid_table[i].type) {
		case RID_HEXDUMP:
			for (j = 0; j < len; j++)
				p += sprintf(p, "<%02x>", buf[j]);
			p += sprintf(p, "\n");
			break;

		case RID_WORD:
			if (len != 2) {
				p += sprintf(p, "<INVALID RID_WORD LEN %d>\n",
					     len);
			} else {
				u16 val = __le16_to_cpu(*(u16 *)buf);
				p += sprintf(p, "%d\n", val);
			}
			break;

		case RID_HWADDR:
			if (len != 6) {
				p += sprintf(p,
					     "<INVALID RID_HWADDR LEN %d>\n",
					     len);
			} else {
				p += sprintf(p, MACSTR "\n", MAC2STR(buf));
			}
			break;

		case RID_STRING:
			strlen = __le16_to_cpu(*(u16 *)buf);
			if (strlen > len)
				strlen = len;
			for (j = 2; j < strlen + 2; j++) {
				if (buf[j] >= 32 && buf[j] < 127)
					p += sprintf(p, "%c", buf[j]);
				else
					p += sprintf(p, "<%02x>", buf[j]);
			}
			p += sprintf(p, "\n");
			break;

		case RID_COMPID:
			if (len != sizeof(*compid)) {
				p += sprintf(p, "<INVALID RID_COMPID LEN "
					     "%d>\n", len);
				break;
			}
			compid = (struct hfa384x_comp_ident *) buf;
			p += sprintf(p, "0x%02x v%d.%d.%d\n",
				     __le16_to_cpu(compid->id),
				     __le16_to_cpu(compid->major),
				     __le16_to_cpu(compid->minor),
				     __le16_to_cpu(compid->variant));
			break;

		case RID_SUPRANGE:
			if (len != sizeof(*range)) {
				p += sprintf(p, "<INVALID RID_SUPRANGE LEN "
					     "%d>\n", len);
				break;
			}
			range = (struct hfa384x_sup_range *) buf;
			p += sprintf(p, "%d 0x%02x %d %d-%d\n",
				     __le16_to_cpu(range->role),
				     __le16_to_cpu(range->id),
				     __le16_to_cpu(range->variant),
				     __le16_to_cpu(range->bottom),
				     __le16_to_cpu(range->top));
			break;

		default:
			p += sprintf(p, "<UNKNOWN TYPE %d>\n",
				     rid_table[i].type);
			break;
		}
	}

	total += (p - page);
	if (total >= off + count)
		*eof = 1;

	if (total < off) {
		*eof = 1;
		return 0;
	}

	len = total - off;
	if (len > (p - page))
		len = p - page;
	*start = p - len;
	if (len > count)
		len = count;

	return len;
}


static int prism2_unknown_rids_proc_read(char *page, char **start, off_t off,
					 int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	u16 rid;
	int pos, rid_entries, len, j, total = 0;
	unsigned char buf[256];

	pos = 0;
	rid_entries = sizeof(rid_table) / sizeof(rid_table[0]);

	for (rid = 0xfc00; rid <= 0xfdff; rid++) {
		if (pos < rid_entries) {
			if (rid_table[pos].rid == rid) {
				pos++;
				continue;
			}
			while (pos < rid_entries && rid_table[pos].rid < rid)
				pos++;
		}

		if (total + (p - page) <= off) {
			total += p - page;
			p = page;
		}
		if (total + (p - page) > off + count)
			break;
		if ((p - page) > PROC_LIMIT)
			break;

		len = hfa384x_get_rid(local->dev, rid, buf, sizeof(buf), 0);

		if (len < 0)
			continue;

		p += sprintf(p, "%04X=", rid);
		for (j = 0; j < len; j++)
			p += sprintf(p, "<%02x>", buf[j]);
		p += sprintf(p, "\n");
	}

	total += (p - page);
	if (total >= off + count)
		*eof = 1;

	if (total < off) {
		*eof = 1;
		return 0;
	}

	len = total - off;
	if (len > (p - page))
		len = p - page;
	*start = p - len;
	if (len > count)
		len = count;

	return len;

}


static int prism2_wds_proc_read(char *page, char **start, off_t off,
				int count, int *eof, void *data)
{
	char *p = page;
	local_info_t *local = (local_info_t *) data;
	prism2_wds_info_t *wds;
	unsigned long flags;

	if (off > PROC_LIMIT) {
		*eof = 1;
		return 0;
	}

	spin_lock_irqsave(&local->wdslock, flags);
	wds = local->wds;
	while (wds != NULL) {
		p += sprintf(p, "%s\t" MACSTR "\n",
			     wds->dev.name, MAC2STR(wds->remote_addr));
		if ((p - page) > PROC_LIMIT) {
			printk(KERN_DEBUG "%s: wds proc did not fit\n",
			       local->dev->name);
			break;
		}
		wds = wds->next;
	}
	spin_unlock_irqrestore(&local->wdslock, flags);

	if ((p - page) <= off) {
		*eof = 1;
		return 0;
	}

	*start = page + off;

	return (p - page - off);
}


static void prism2_init_proc(local_info_t *local)
{
	local->proc = NULL;

	if (proc_net != NULL) {
		if (prism2_proc == NULL)
			prism2_proc = proc_mkdir("prism2", proc_net);
		if (prism2_proc != NULL)
			local->proc = proc_mkdir(local->dev->name,
						 prism2_proc);
	}

	if (local->proc == NULL) {
		printk(KERN_INFO "/proc/net/prism2 creation failed\n");
		return;
	}

	create_proc_read_entry("debug", 0, local->proc,
			       prism2_debug_proc_read, local);
	create_proc_read_entry("stats", 0, local->proc,
			       prism2_stats_proc_read, local);
	create_proc_read_entry("registers", 0, local->proc,
			       prism2_registers_proc_read, local);
	create_proc_read_entry("rids", 0, local->proc,
			       prism2_rids_proc_read, local);
	create_proc_read_entry("unknown_rids", 0, local->proc,
			       prism2_unknown_rids_proc_read, local);
	create_proc_read_entry("wds", 0, local->proc,
			       prism2_wds_proc_read, local);
}


static void prism2_remove_proc(local_info_t *local)
{
	if (local->proc != NULL) {
		remove_proc_entry("wds", local->proc);
		remove_proc_entry("unknown_rids", local->proc);
		remove_proc_entry("rids", local->proc);
		remove_proc_entry("registers", local->proc);
		remove_proc_entry("stats", local->proc);
		remove_proc_entry("debug", local->proc);
		if (local->dev != NULL && local->dev->name != NULL)
			remove_proc_entry(local->dev->name, prism2_proc);
	}
}


static void prism2_setup_dev(struct net_device *dev, local_info_t *local,
			     int main_dev)
{
	ether_setup(dev);

	/* kernel callbacks */
	dev->get_stats = prism2_get_stats;
#ifdef WIRELESS_EXT
	dev->get_wireless_stats = main_dev ? prism2_get_wireless_stats : NULL;
#endif
	dev->open = prism2_open;
	dev->stop = prism2_close;
	dev->hard_start_xmit = prism2_tx;
#ifdef HAVE_MULTICAST
	/* FIX: to be implemented as soon as Prism2 supports GroupAddresses
	 * and correct documentation is available */

	/* dev->set_multicast_list = prism2_set_multicast_list; */
#endif
#ifdef HAVE_PRIVATE_IOCTL
	dev->do_ioctl = main_dev ? prism2_ioctl : NULL;
#endif
#ifdef HAVE_CHANGE_MTU
	dev->change_mtu = prism2_change_mtu;
#endif
#ifdef HAVE_TX_TIMEOUT
	dev->tx_timeout = prism2_tx_timeout;
	dev->watchdog_timeo = TX_TIMEOUT;
#endif

	dev->mtu = mtu;

	init_dev_name(dev, local->node);
	netif_stop_queue(dev);
}


static inline int prism2_wds_special_addr(u8 *addr)
{
	if (addr[0] || addr[1] || addr[2] || addr[3] || addr[4] || addr[5])
		return 0;

	return 1;
}


static int prism2_wds_add(local_info_t *local, u8 *remote_addr,
			  int rtnl_locked)
{
	prism2_wds_info_t *wds, *wds2 = NULL;
	unsigned long flags;
	int i, ret;

	spin_lock_irqsave(&local->wdslock, flags);
	wds = local->wds;
	while (wds != NULL &&
	       memcmp(wds->remote_addr, remote_addr, ETH_ALEN) != 0) {
		if (!wds2 && prism2_wds_special_addr(wds->remote_addr))
			wds2 = wds;
		wds = wds->next;
	}
	if (!wds && wds2) {
		/* take pre-allocated entry into use */
		memcpy(wds2->remote_addr, remote_addr, ETH_ALEN);
	}
	spin_unlock_irqrestore(&local->wdslock, flags);

	if (wds && !prism2_wds_special_addr(remote_addr))
		return -EEXIST;

	if (!wds && wds2) {
		printk(KERN_DEBUG "%s: using pre-allocated WDS netdevice %s\n",
		       local->dev->name, wds2->dev.name);
		return 0;
	}

	if (local->wds_connections >= local->wds_max_connections)
		return -ENOBUFS;

	/* verify that there is room for wds# postfix in the interface name */
	if (strlen(local->dev->name) > IFNAMSIZ - 5) {
		printk(KERN_DEBUG "'%s' too long base device name\n",
		       local->dev->name);
		return -EINVAL;
	}

	wds = (prism2_wds_info_t *) kmalloc(sizeof(*wds), GFP_ATOMIC);
	if (wds == NULL)
		return -ENOMEM;

	memset(wds, 0, sizeof(*wds));

	memcpy(wds->remote_addr, remote_addr, ETH_ALEN);

	prism2_setup_dev(&wds->dev, local, 0);

	wds->dev.priv = local;
	memcpy(wds->dev.dev_addr, local->dev->dev_addr, ETH_ALEN);
	wds->dev.base_addr = local->dev->base_addr;
	wds->dev.irq = local->dev->irq;

	i = 0;
	do {
		sprintf(wds->dev.name, "%swds%d", local->dev->name, i++);
	} while (i < 10000 && dev_get(wds->dev.name));

	if (rtnl_locked)
		ret = register_netdevice(&wds->dev);
	else
		ret = register_netdev(&wds->dev);

	if (ret) {
		printk(KERN_WARNING "%s: registering WDS device '%s' failed\n",
		       local->dev->name, wds->dev.name);
		kfree(wds);
		return -EINVAL;
	}

	spin_lock_irqsave(&local->wdslock, flags);
	local->wds_connections++;
	wds->next = local->wds;
	local->wds = wds;
	spin_unlock_irqrestore(&local->wdslock, flags);

	printk(KERN_DEBUG "%s: registered WDS netdevice %s\n",
	       local->dev->name, wds->dev.name);

	return 0;
}


static int prism2_wds_del(local_info_t *local, u8 *remote_addr,
			  int rtnl_locked, int do_not_remove)
{
	prism2_wds_info_t *wds, *prev = NULL;
	unsigned long flags;

	spin_lock_irqsave(&local->wdslock, flags);
	wds = local->wds;
	while (wds != NULL &&
	       memcmp(wds->remote_addr, remote_addr, ETH_ALEN) != 0) {
		prev = wds;
		wds = wds->next;
	}
	if (wds && !do_not_remove) {
		if (prev != NULL)
			prev->next = wds->next;
		else
			local->wds = wds->next;
	}
	spin_unlock_irqrestore(&local->wdslock, flags);

	if (!wds)
		return -ENODEV;

	if (do_not_remove) {
		memset(wds->remote_addr, 0, ETH_ALEN);
		return 0;
	}

	if (rtnl_locked)
		unregister_netdevice(&wds->dev);
	else
		unregister_netdev(&wds->dev);
	local->wds_connections--;

	printk(KERN_DEBUG "%s: unregistered WDS netdevice %s\n",
	       local->dev->name, wds->dev.name);

	kfree(wds);

	return 0;
}


/* allocate local data and register with CardServices
 * initialize dev_link structure, but do not configure the card yet */
static dev_link_t *prism2_attach(void)
{
	dev_link_t *link;
	local_info_t *local;
	struct net_device *dev;
	client_reg_t client_reg;
	int ret;

	for (link = dev_list; link; link = link->next) {
		if (link->state & DEV_STALE_LINK) {
			printk("%s: flushing stale link\n", dev_info);
			prism2_detach(link);
		}
	}

	link = kmalloc(sizeof(dev_link_t), GFP_KERNEL);
	local = kmalloc(sizeof(local_info_t), GFP_KERNEL);
	if (local) {
		memset(local, 0, sizeof(local_info_t));
		local->ap = kmalloc(sizeof(struct ap_data), GFP_KERNEL);
		if (local->ap)
			memset(local->ap, 0, sizeof(struct ap_data));
	}
	dev = kmalloc(sizeof(struct net_device), GFP_KERNEL);
#ifdef PRISM2_HOSTAPD
	local->apdev = kmalloc(sizeof(struct net_device), GFP_KERNEL);
	if (local->apdev == NULL && !dev) {
		/* kmalloc failed.. set dev to NULL in order to catch this with
		 * following if statement.. */
		kfree(dev);
		dev = NULL;
	}
#endif /* PRISM2_HOSTAPD */
	if (link == NULL || local == NULL || local->ap == NULL || dev == NULL)
	{
		printk(KERN_WARNING "prism2_attach - could not kmalloc memory"
		       "\n");
		if (link) kfree(link);
		if (local) {
			if (local->ap) kfree(local->ap);
			kfree(local);
		}
		if (dev) kfree(dev);
		return NULL;
	}
	memset(link, 0, sizeof(dev_link_t));
	memset(dev, 0, sizeof(struct net_device));

	link->priv = dev;
	dev->priv = local;
	local->link = link;
	local->dev = dev;

	spin_lock_init(&local->txfidlock);
	spin_lock_init(&local->cmdlock);
	spin_lock_init(&local->baplock);
	spin_lock_init(&local->wdslock);

	memcpy(local->essid, essid, sizeof(local->essid));
#ifdef WIRELESS_EXT
	if (iw_mode >= 1 && iw_mode <= 3) {
		local->iw_mode = iw_mode;
	} else {
		printk(KERN_WARNING "prism2: Unknown iw_mode %d; using "
		       "IW_MODE_MASTER\n", iw_mode);
		local->iw_mode = IW_MODE_MASTER;
	}
#endif
	local->channel = channel;
	local->beacon_int = beacon_int;
	local->dtim_period = dtim_period;
	local->wds_max_connections = 16;
	local->tx_control = HFA384X_TX_CTRL_FLAGS;
	local->manual_retry_count = -1;

	skb_queue_head_init(&local->bridge_list);

	/* Initialize task queue structure for bridged packet handling */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	local->bridge_queue.next = NULL;
#else
	INIT_LIST_HEAD(&local->bridge_queue.list);
#endif
	local->bridge_queue.sync = 0;
	local->bridge_queue.routine = handle_bridged_queue;
	local->bridge_queue.data = local;

	prism2_setup_dev(dev, local, 1);

#ifdef PRISM2_HOSTAPD
	memset(local->apdev, 0, sizeof(struct net_device));
	local->apdev->priv = local;
	prism2_setup_dev(local->apdev, local, 0);
	local->apdev->hard_start_xmit = prism2_tx_80211;
	local->apdev->type = ARPHRD_IEEE80211;
	local->apdev->hard_header_parse = prism2_80211_header_parse;
#endif /* PRISM2_HOSTAPD */

#ifdef PRISM2_MONITOR
	local->saved_eth_header_parse = dev->hard_header_parse;
#endif /* PRISM2_MONITOR */

	link->release.function = &prism2_release;
	link->release.data = (u_long)link;

	PDEBUG(DEBUG_HW, "%s: setting Vcc=33 (constant)\n", dev_info);
	link->conf.Vcc = 33;
	link->conf.IntType = INT_MEMORY_AND_IO;

	/* register with CardServices */
	link->next = dev_list;
	dev_list = link;
	client_reg.dev_info = &dev_info;
	client_reg.Attributes = INFO_IO_CLIENT;
	client_reg.EventMask = CS_EVENT_CARD_INSERTION |
		CS_EVENT_CARD_REMOVAL |
		CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
		CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
	client_reg.event_handler = &prism2_event;
	client_reg.Version = 0x0210;
	client_reg.event_callback_args.client_data = link;
	ret = CardServices(RegisterClient, &link->handle, &client_reg);
	if (ret != CS_SUCCESS) {
		cs_error(link->handle, RegisterClient, ret);
		prism2_detach(link);
		return NULL;
	}
	return link;
}


static void prism2_detach(dev_link_t *link)
{
	dev_link_t **linkp;

	PDEBUG(DEBUG_FLOW, "prism2_detach\n");

	for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
		if (*linkp == link)
			break;
	if (*linkp == NULL) {
		printk(KERN_WARNING "%s: Attempt to detach non-existing "
		       "PCMCIA client\n", dev_info);
		return;
	}

	del_timer(&link->release);
	if (link->state & DEV_CONFIG) {
		printk("%s: detach postponed, '%s' still locked\n",
		       dev_info, link->dev->dev_name);
		prism2_release((u_long)link);
		if (link->state & DEV_STALE_CONFIG) {
			link->state |= DEV_STALE_LINK;
			return;
		}
	}

	if (link->handle) {
		int res = CardServices(DeregisterClient, link->handle);
		if (res) {
			printk("CardService(DeregisterClient) => %d\n", res);
			cs_error(link->handle, DeregisterClient, res);
		}
	}

	*linkp = link->next;
	/* release local_info_t struct */
	if (link->priv) {
		struct net_device *dev = (struct net_device *) link->priv;
		local_info_t *local = (local_info_t *) dev->priv;
		prism2_wds_info_t *wds;

		if (local->ap != NULL)
			ap_free_data(local->ap);
		prism2_remove_proc(local);

		wds = local->wds;
		local->wds = NULL;
		while (wds != NULL) {
			unregister_netdev(&wds->dev);
			kfree(wds);
			wds = wds->next;
		}

#ifdef PRISM2_HOSTAPD
		if (local->apdev) {
			unregister_netdev(local->apdev);
			printk(KERN_INFO "%s: Netdevice %s unregistered\n",
			       dev_info, local->apdev->name);
		}
#endif /* PRISM2_HOSTAPD */

		if (link->dev) {
			unregister_netdev(dev);
			printk(KERN_INFO "%s: Netdevice %s unregistered\n",
			       dev_info, dev->name);
		} else
			printk("%s: link->dev == NULL: do not unregister "
			       "netdevice\n", dev_info);
#ifdef PRISM2_MONITOR
		if (local->nl_monitor != NULL) {
			if (local->nl_monitor->socket == NULL)
				printk("nl_monitor->socket == NULL\n");
			/* this seems to crash the kernel.. */
			/* sock_release(local->nl_monitor->socket); */
		}
#endif
		kfree(local);
		kfree(dev);
	}
	kfree(link);
}


/* Called only from hardware IRQ */
static void prism2_alloc_ev(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	int idx;
	u16 fid;

	fid = HFA384X_INW(HFA384X_ALLOCFID_OFF);

	PDEBUG(DEBUG_FID, "FID: interrupt: ALLOC - fid=0x%04x\n", fid);

	idx = local->next_alloc;

	spin_lock(&local->txfidlock);

	do {
		if (local->txfid[idx] == fid) {
			PDEBUG(DEBUG_FID, "FID: found matching txfid[%d]\n",
			       idx);

#ifndef PRISM2_USE_TX_INTERRUPT
			/* Alloc events are used only for TX frames, so
			 * increase TX packet counter here. This will make
			 * separate TX event handling unnecessary. */
			local->stats.tx_packets++;
#endif /* PRISM2_USE_TX_INTERRUPT */

#ifndef final_version
			if (local->intransmitfid[idx] == PRISM2_TXFID_EMPTY)
				printk("Already released txfid found at idx "
				       "%d\n", idx);
			if (local->intransmitfid[idx] == PRISM2_TXFID_RESERVED)
				printk("Already reserved txfid found at idx "
				       "%d\n", idx);
#endif
			local->intransmitfid[idx] = PRISM2_TXFID_EMPTY;
			idx++;
			local->next_alloc = idx >= PRISM2_TXFID_COUNT ? 0 :
				idx;

#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
			if (local->last_txfid_idx == PRISM2_DUMMY_FID &&
			    netif_queue_stopped(dev))
				prism2_netif_wake_queues(dev);
#else /* PRISM2_USE_CMD_COMPL_INTERRUPT */
			if (netif_queue_stopped(dev))
				prism2_netif_wake_queues(dev);
#endif /* PRISM2_USE_CMD_COMPL_INTERRUPT */

			goto out;
		}

		idx++;
		if (idx >= PRISM2_TXFID_COUNT)
			idx = 0;
	} while (idx != local->next_alloc);

	printk(KERN_WARNING "%s: could not found matching txfid (0x%04x) for "
	       "alloc event\n", dev->name, HFA384X_INW(HFA384X_ALLOCFID_OFF));

 out:
	spin_unlock(&local->txfidlock);
}


/* Called only from hardware IRQ */
static void prism2_txexc(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 fid, status, fc;
	int res;
	struct hfa384x_tx_frame txdesc;
	struct sta_info *sta;

	local->stats.tx_errors++;
	fid = HFA384X_INW(HFA384X_TXCOMPLFID_OFF);
	PDEBUG(DEBUG_EXTRA, "%s: TXEXC - fid=0x%04x - ", dev->name, fid);

	res = hfa384x_setup_bap(dev, BAP1, fid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &txdesc, sizeof(txdesc));
	if (res) {
		PDEBUG2(DEBUG_EXTRA, "could not get TXEXC txframe\n");
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}
	status = __le16_to_cpu(txdesc.status);
	PDEBUG2(DEBUG_EXTRA, "status=0x%04x (%s%s%s%s) tx_control=%04x\n",
		status,
		status & HFA384X_TX_STATUS_RETRYERR ? "[RetryErr]" : "",
		status & HFA384X_TX_STATUS_AGEDERR ? "[AgedErr]" : "",
		status & HFA384X_TX_STATUS_DISCON ? "[Discon]" : "",
		status & HFA384X_TX_STATUS_FORMERR ? "[FormErr]" : "",
		__le16_to_cpu(txdesc.tx_control));
	fc = __le16_to_cpu(txdesc.frame_control);
	PDEBUG(DEBUG_EXTRA, "   retry_count=%d tx_rate=%d fc=0x%04x "
	       "(%s%s%s::%d)\n",
	       txdesc.retry_count, txdesc.tx_rate, fc,
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_MGMT ? "Mgmt" : "",
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_CTRL ? "Ctrl" : "",
	       WLAN_FC_GET_TYPE(fc) == WLAN_FC_TYPE_DATA ? "Data" : "",
	       WLAN_FC_GET_STYPE(fc));
	PDEBUG(DEBUG_EXTRA, "   addr1=" MACSTR " addr2=" MACSTR " addr3="
	       MACSTR "\n",
	       MAC2STR(txdesc.addr1), MAC2STR(txdesc.addr2),
	       MAC2STR(txdesc.addr3));

	if (local->iw_mode != IW_MODE_MASTER)
		return;

	spin_lock(&local->ap->sta_table_lock);
	/* FIX: is addr1 correct for all frame types? */
	sta = ap_get_sta(local->ap, txdesc.addr1);
	if (sta) {
		sta->tx_since_last_failure = 0;
		if (sta->tx_rate_idx > 0 &&
		    txdesc.tx_rate <= sta->tx_rate) {
			int rate = sta->tx_rate_idx;
			/* use next lower rate */
			while (rate > 0) {
				rate--;
				if (sta->tx_supp_rates & (1 << rate)) {
					sta->tx_rate_idx = rate;
					break;
				}
			}
			switch (sta->tx_rate_idx) {
			case 0: sta->tx_rate = 10; break;
			case 1: sta->tx_rate = 20; break;
			case 2: sta->tx_rate = 55; break;
			case 3: sta->tx_rate = 110; break;
			default: sta->tx_rate = 0; break;
			}
			PDEBUG(DEBUG_AP, "%s: STA " MACSTR " TX rate lowered "
			       "to %d\n", dev->name, MAC2STR(sta->addr),
			       sta->tx_rate);
		}
	} else {
		PDEBUG(DEBUG_AP, "Could not find STA for this TX error\n");
	}
	spin_unlock(&local->ap->sta_table_lock);
}


static const char* hfa384x_linkstatus_str(u16 linkstatus)
{
	switch (linkstatus) {
	case HFA384X_LINKSTATUS_CONNECTED:
		return "Connected";
	case HFA384X_LINKSTATUS_DISCONNECTED:
		return "Disconnected";
	case HFA384X_LINKSTATUS_AP_CHANGE:
		return "Access point change";
	case HFA384X_LINKSTATUS_AP_OUT_OF_RANGE:
		return "Access point out of range";
	case HFA384X_LINKSTATUS_AP_IN_RANGE:
		return "Access point in range";
	case HFA384X_LINKSTATUS_ASSOC_FAILED:
		return "Association failed";
	default:
		return "Unknown";
	}
}


/* Called only from hardware IRQ */
static void prism2_info(struct net_device *dev)
{
	local_info_t *local = (local_info_t *) dev->priv;
	u16 fid, val, *pos, len;
	int res, i, left;
	struct hfa384x_info_frame info;
	unsigned char buf[1024];
	struct hfa384x_comm_tallies *tallies;
	struct hfa384x_scan_result *scanres;

	fid = HFA384X_INW(HFA384X_INFOFID_OFF);

	res = hfa384x_setup_bap(dev, BAP1, fid, 0);
	if (!res)
		res = hfa384x_from_bap(dev, BAP1, &info, sizeof(info));
	if (res) {
		printk("Could not get info frame\n");
		if (res == -ETIMEDOUT)
			prism2_hw_reset(dev);
		return;
	}

	if (__le16_to_cpu(info.len) & 0x8000) {
		/* data register seems to give 0x8000 in some error cases even
		 * though busy bit is not set in offset register; re-reading
		 * the data seems to fix at least some of the cases */
		printk(KERN_DEBUG "%s: prism2_info: re-reading header to fix "
		       "possibly corrupted BAP read\n", dev->name);
		res = hfa384x_setup_bap(dev, BAP1, fid, 0);
		if (!res)
			res = hfa384x_from_bap(dev, BAP1, &info, sizeof(info));
		if (res)
			printk(KERN_DEBUG "prism2_info: could not re-read "
			       "info header\n");
		if (__le16_to_cpu(info.len) & 0x8000) {
			printk(KERN_DEBUG "prism2_info: re-read did not fix "
			       "header - ignoring this frame\n");
			return;
		}
	}

	le16_to_cpus(&info.len);
	le16_to_cpus(&info.type);

	if (info.type != HFA384X_INFO_COMMTALLIES)
		PDEBUG(DEBUG_EXTRA, "%s: INFO - fid=0x%04x - len=%d "
		       "type=0x%04x\n", dev->name, fid, info.len, info.type);

	switch (info.type) {
	case HFA384X_INFO_COMMTALLIES:
		if (info.len != 22 ||
		    hfa384x_from_bap(dev, BAP1, &buf,
				     sizeof(struct hfa384x_comm_tallies))) {
			printk("  info communication tallies failed\n");
			break;
		}
		tallies = (struct hfa384x_comm_tallies *) buf;
#define ADD_COMM_TALLIES(name) local->comm_tallies.name += tallies->name
		ADD_COMM_TALLIES(tx_unicast_frames);
		ADD_COMM_TALLIES(tx_multicast_frames);
		ADD_COMM_TALLIES(tx_fragments);
		ADD_COMM_TALLIES(tx_unicast_octets);
		ADD_COMM_TALLIES(tx_multicast_octets);
		ADD_COMM_TALLIES(tx_deferred_transmissions);
		ADD_COMM_TALLIES(tx_single_retry_frames);
		ADD_COMM_TALLIES(tx_multiple_retry_frames);
		ADD_COMM_TALLIES(tx_retry_limit_exceeded);
		ADD_COMM_TALLIES(tx_discards);
		ADD_COMM_TALLIES(rx_unicast_frames);
		ADD_COMM_TALLIES(rx_multicast_frames);
		ADD_COMM_TALLIES(rx_fragments);
		ADD_COMM_TALLIES(rx_unicast_octets);
		ADD_COMM_TALLIES(rx_multicast_octets);
		ADD_COMM_TALLIES(rx_fcs_errors);
		ADD_COMM_TALLIES(rx_discards_no_buffer);
		ADD_COMM_TALLIES(tx_discards_wrong_sa);
		ADD_COMM_TALLIES(rx_discards_wep_undecryptable);
		ADD_COMM_TALLIES(rx_message_in_msg_fragments);
		ADD_COMM_TALLIES(rx_message_in_bad_msg_fragments);
		break;

	case HFA384X_INFO_LINKSTATUS:
		if (info.len != 2 || hfa384x_from_bap(dev, BAP1, &val, 2)) {
			printk("  info linkstatus failed\n");
			break;
		}
		le16_to_cpus(&val);
		PDEBUG(DEBUG_EXTRA, "  LinkStatus=%d (%s)\n",
		       val, hfa384x_linkstatus_str(val));
		break;

	case HFA384X_INFO_SCANRESULTS:
		PDEBUG(DEBUG_EXTRA, "  ScanResults:\n");
		if (info.len < 3) {
			printk("    info scanresult failed\n");
			break;
		}
		val = (info.len - 1) * 2;
		if (val > sizeof(buf))
			val = sizeof(buf);
		if (hfa384x_from_bap(dev, BAP1, &buf, val)) {
			printk("    info read failed\n");
			break;
		}

		pos = (u16 *) buf;
		PDEBUG(DEBUG_EXTRA, "    Reserved=0x%04x\n",
		       __le16_to_cpu(*pos));
		pos++;
		PDEBUG(DEBUG_EXTRA, "    ScanReason=%d\n",
		       __le16_to_cpu(*pos));
		pos++;

		left = (info.len - 3) * 2;
		scanres = (struct hfa384x_scan_result *) pos;
		val = 1;
		while (left >= sizeof(struct hfa384x_scan_result)) {
			/* FIX: it might not be that wise to print a lot
			 * from here.. this is called from hw interrupt
			 * handler.. */
			PDEBUG(DEBUG_EXTRA, "    ScanResult %d:\n", val);
			PDEBUG(DEBUG_EXTRA, "      CHID=%d\n",
			       __le16_to_cpu(scanres->chid));
			PDEBUG(DEBUG_EXTRA, "      ANL=%d\n",
			       __le16_to_cpu(scanres->anl));
			PDEBUG(DEBUG_EXTRA, "      SL=%d\n",
			       __le16_to_cpu(scanres->sl));
			PDEBUG(DEBUG_EXTRA, "      bssid=" MACSTR "\n",
			       MAC2STR(scanres->bssid));
			PDEBUG(DEBUG_EXTRA, "      BcnInt=%d\n",
			       __le16_to_cpu(scanres->beacon_interval));
			PDEBUG(DEBUG_EXTRA, "      Capability=0x%04x\n",
			       __le16_to_cpu(scanres->capability));
			PDEBUG(DEBUG_EXTRA, "      SSID='");
			len = __le16_to_cpu(*(u16 *) &scanres->ssid);
			if (len > 32)
				len = 32;
			for (i = 0; i < len; i++) {
				unsigned char c = scanres->ssid[2 + i];
				if (c >= 32 && c < 127)
					PDEBUG2(DEBUG_EXTRA, "%c", c);
				else
					PDEBUG2(DEBUG_EXTRA, "<%02x>", c);
			}
			PDEBUG2(DEBUG_EXTRA, "'\n");
			PDEBUG(DEBUG_EXTRA, "      SupRates=");
			for (i = 0; i < sizeof(scanres->sup_rates); i++)
				PDEBUG2(DEBUG_EXTRA, "%02x ",
					scanres->sup_rates[i]);
			PDEBUG2(DEBUG_EXTRA, "\n");
			PDEBUG(DEBUG_EXTRA, "      Rate=%d\n",
			       __le16_to_cpu(scanres->rate));
			val++;
			left -= sizeof(struct hfa384x_scan_result);
			scanres++;
		}

		if (left > 0) {
			unsigned char *c = (unsigned char *) scanres;
			PDEBUG(DEBUG_EXTRA, "ScanRes extra: left=%d "
			       "(expected %dx):", left,
			       sizeof(struct hfa384x_scan_result));
			for (i = 0; i < left; i++)
				PDEBUG2(DEBUG_EXTRA, " %02x", *c++);
			PDEBUG2(DEBUG_EXTRA, "\n");
		}

		break;

	default:
		val = info.len > 0 ? (info.len - 1) * 2 : 0;
		if (val > sizeof(buf))
			val = sizeof(buf);
		if (val > 0 && hfa384x_from_bap(dev, BAP1, &buf, val)) {
			printk("  info read failed\n");
			break;
		}
		PDEBUG(DEBUG_EXTRA, "Unknown info frame:");
		for (i = 0; i < val; i++)
			PDEBUG2(DEBUG_EXTRA, " %02x", buf[i]);
		PDEBUG2(DEBUG_EXTRA, "\n");
	}
}


/* Called only from hardware IRQ */
static void prism2_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = (struct net_device *) dev_id;
	local_info_t *local = (local_info_t *) dev->priv;
	int events = 0;
	u16 ev;
#ifndef final_version
	static long int last_magic_err = 0;
#endif

	if ((local->link->state & (DEV_PRESENT | DEV_CONFIG)) !=
	    (DEV_PRESENT | DEV_CONFIG)) {
		printk(KERN_DEBUG "%s: Interrupt, but dev not OK\n",
		       dev->name);
		return;
	}

	if (!local->hw_ready || local->hw_resetting) {
		ev = HFA384X_INW(HFA384X_EVSTAT_OFF);
		printk(KERN_DEBUG "%s: prism2_interrupt: hw not ready; "
		       "skipping events 0x%04x\n", dev->name, ev);
		HFA384X_OUTW(ev, HFA384X_EVACK_OFF);
		return;
	}

#ifndef final_version
	if (HFA384X_INW(HFA384X_SWSUPPORT0_OFF) != HFA384X_MAGIC) {
		HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
		if (jiffies - last_magic_err > 10 * HZ)
			printk("%s: Interrupt, but SWSUPPORT0 does not match: "
			       "%04X != %04X - card removed?\n", dev->name,
			       HFA384X_INW(HFA384X_SWSUPPORT0_OFF),
			       HFA384X_MAGIC);
		last_magic_err = jiffies;
		prism2_hw_reset(dev);
		return;
	}
#endif

	do {
		ev = HFA384X_INW(HFA384X_EVSTAT_OFF);
		if ((ev & HFA384X_EVENT_MASK) == 0)
			break;
		if (ev == 0xffff) {
			HFA384X_OUTW(0xffff, HFA384X_EVACK_OFF);
			printk(KERN_DEBUG "%s: prism2_interrupt: ev=0xffff\n",
			       dev->name);
			prism2_hw_reset(dev);
			return;
		}

#ifdef PRISM2_USE_CMD_COMPL_INTERRUPT
		if (ev & HFA384X_EV_CMD) {
			prism2_cmd_ev(dev);
			HFA384X_OUTW(HFA384X_EV_CMD, HFA384X_EVACK_OFF);
		}
#endif /* PRISM2_USE_CMD_COMPL_INTERRUPT */

		if (ev & HFA384X_EV_ALLOC) {
			prism2_alloc_ev(dev);
			HFA384X_OUTW(HFA384X_EV_ALLOC, HFA384X_EVACK_OFF);
		}

#ifdef PRISM2_USE_TX_INTERRUPT
		if (ev & HFA384X_EV_TX) {
			local->stats.tx_packets++;
			PDEBUG(DEBUG_FID, "interrupt: TX - fid=0x%04x\n",
			       HFA384X_INW(HFA384X_TXCOMPLFID_OFF));
			HFA384X_OUTW(HFA384X_EV_TX, HFA384X_EVACK_OFF);
		}
#endif /* PRISM2_USE_TX_INTERRUPT */

		if (ev & HFA384X_EV_TXEXC) {
			prism2_txexc(dev);
			HFA384X_OUTW(HFA384X_EV_TXEXC, HFA384X_EVACK_OFF);
		}

		if (ev & HFA384X_EV_RX) {
			prism2_rx(dev);
			HFA384X_OUTW(HFA384X_EV_RX, HFA384X_EVACK_OFF);
		}

#ifndef final_version
		if (ev & HFA384X_EV_WTERR) {
			PDEBUG(DEBUG_EXTRA, "%s: WTERR event\n", dev->name);
			HFA384X_OUTW(HFA384X_EV_WTERR, HFA384X_EVACK_OFF);
		}
#endif /* final_version */

		if (ev & HFA384X_EV_INFO) {
			prism2_info(dev);
			HFA384X_OUTW(HFA384X_EV_INFO, HFA384X_EVACK_OFF);
		}

		events++;
	} while (events < 20);

	if (events >= 20)
		PDEBUG(DEBUG_EXTRA, "prism2_interrupt: >20 events\n");
}


static void prism2_check_sta_fw_version(local_info_t *local)
{
	struct hfa384x_comp_ident comp;
	int variant, major, minor;

	if (hfa384x_get_rid(local->dev, HFA384X_RID_STAID,
			    &comp, sizeof(comp), 1) < 0)
		return;

	if (__le16_to_cpu(comp.id) != 0x1f)
		return;

	major = __le16_to_cpu(comp.major);
	minor = __le16_to_cpu(comp.minor);
	variant = __le16_to_cpu(comp.variant);

	/* no firmware version specific configuration for the base driver yet,
	 * so just call AP specific parts */
	ap_check_sta_fw_version(local->ap, major, minor, variant);
}


#define CS_CHECK(fn, args...) \
while ((last_ret = CardServices(last_fn = (fn), args)) != 0) goto cs_failed

#define CFG_CHECK(fn, args...) \
if (CardServices(fn, args) != 0) goto next_entry

#define CFG_CHECK2(fn, args...) \
do { int ret = CardServices(fn, args); \
if (ret != 0) { \
	PDEBUG(DEBUG_EXTRA, "CardServices(" #fn ") returned %d\n", ret); \
	cs_error(link->handle, fn, ret); \
	goto next_entry; \
} \
} while (0)


/* run after a CARD_INSERTATION event is received to configure the PCMCIA
 * socket and make the device available to the system */
static int prism2_config(dev_link_t *link)
{
	struct net_device *dev = (struct net_device *) link->priv;
	local_info_t *local = (local_info_t *) dev->priv;
	int ret;
	tuple_t tuple;
	cisparse_t parse;
	int last_fn, last_ret;
	u_char buf[64];
	config_info_t conf;
	win_req_t req;
	memreq_t map;
	cistpl_cftable_entry_t dflt = { 0 };

	PDEBUG(DEBUG_FLOW, "prism2_config()\n");

	tuple.DesiredTuple = CISTPL_CONFIG;
	tuple.Attributes = 0;
	tuple.TupleData = buf;
	tuple.TupleDataMax = sizeof(buf);
	tuple.TupleOffset = 0;
	CS_CHECK(GetFirstTuple, link->handle, &tuple);
	CS_CHECK(GetTupleData, link->handle, &tuple);
	CS_CHECK(ParseTuple, link->handle, &tuple, &parse);
	link->conf.ConfigBase = parse.config.base;
	link->conf.Present = parse.config.rmask[0];

	CS_CHECK(GetConfigurationInfo, link->handle, &conf);
	PDEBUG(DEBUG_HW, "%s: %s Vcc=%d (from config)\n", dev_info,
	       ignore_cis_vcc ? "ignoring" : "setting", conf.Vcc);
	link->conf.Vcc = conf.Vcc;

	tuple.DesiredTuple = CISTPL_MANFID;
	tuple.Attributes = TUPLE_RETURN_COMMON;
	tuple.Attributes = 0;
	tuple.TupleData = buf;
	tuple.TupleDataMax = sizeof(buf);

	if (CardServices(GetFirstTuple, link->handle, &tuple) == CS_SUCCESS &&
	    CardServices(GetTupleData, link->handle, &tuple) == CS_SUCCESS &&
	    CardServices(ParseTuple, link->handle, &tuple, &parse) ==
	    CS_SUCCESS) {
		local->manfid = parse.manfid.manf;
		local->prodid = parse.manfid.card;
		PDEBUG(DEBUG_EXTRA, "CISTPL_MANFID: 0x%04x, 0x%04x\n",
		       local->manfid, local->prodid);

		/* 0x014d,0x0001 Symbol LA4111 Spectrum24
		 * 0x0101,0x0001 3Com 3CRWE737A AirConnect
		 * 0x0089,0x0001 Intel PRO/Wireless 2011 */
		if ((local->manfid == 0x014d && local->prodid == 0x0001) ||
		    (local->manfid == 0x0101 && local->prodid == 0x0001) ||
		    (local->manfid == 0x0089 && local->prodid == 0x0001)) {
			PDEBUG(DEBUG_EXTRA, "Symbol-based card\n");
			local->is_symbol = 1;
		}

		if (local->manfid == 0x0156 && local->prodid == 0x0002) {
			PDEBUG(DEBUG_EXTRA, "Lucent-based card\n");
			local->is_lucent = 1;
		}
	}

	/* Look for an appropriate configuration table entry in the CIS */
	tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
	CS_CHECK(GetFirstTuple, link->handle, &tuple);
	for (;;) {
		cistpl_cftable_entry_t *cfg = &(parse.cftable_entry);
		CFG_CHECK2(GetTupleData, link->handle, &tuple);
		CFG_CHECK2(ParseTuple, link->handle, &tuple, &parse);

		if (cfg->flags & CISTPL_CFTABLE_DEFAULT)
			dflt = *cfg;
		if (cfg->index == 0)
			goto next_entry;
		link->conf.ConfigIndex = cfg->index;
		PDEBUG(DEBUG_EXTRA, "Checking CFTABLE_ENTRY 0x%02X "
		       "(default 0x%02X)\n", cfg->index, dflt.index);
	
		/* Does this card need audio output? */
		if (cfg->flags & CISTPL_CFTABLE_AUDIO) {
			link->conf.Attributes |= CONF_ENABLE_SPKR;
			link->conf.Status = CCSR_AUDIO_ENA;
		}
	
		/* Use power settings for Vcc and Vpp if present */
		/*  Note that the CIS values need to be rescaled */
		if (cfg->vcc.present & (1 << CISTPL_POWER_VNOM)) {
			if (conf.Vcc != cfg->vcc.param[CISTPL_POWER_VNOM] /
			    10000 && !ignore_cis_vcc)
				goto next_entry;
		} else if (dflt.vcc.present & (1 << CISTPL_POWER_VNOM)) {
			if (conf.Vcc != dflt.vcc.param[CISTPL_POWER_VNOM] /
			    10000 && !ignore_cis_vcc)
				goto next_entry;
		}
	    
		if (cfg->vpp1.present & (1 << CISTPL_POWER_VNOM))
			link->conf.Vpp1 = link->conf.Vpp2 =
				cfg->vpp1.param[CISTPL_POWER_VNOM] / 10000;
		else if (dflt.vpp1.present & (1 << CISTPL_POWER_VNOM))
			link->conf.Vpp1 = link->conf.Vpp2 =
				dflt.vpp1.param[CISTPL_POWER_VNOM] / 10000;
	
		/* Do we need to allocate an interrupt? */
		if (cfg->irq.IRQInfo1 || dflt.irq.IRQInfo1)
			link->conf.Attributes |= CONF_ENABLE_IRQ;
		else if (!(link->conf.Attributes & CONF_ENABLE_IRQ)) {
			/* At least Compaq WL200 does not have IRQInfo1 set,
			 * but it does not work without interrupts.. */
			printk("Config has no IRQ info, but trying to enable "
			       "IRQ anyway..\n");
			link->conf.Attributes |= CONF_ENABLE_IRQ;
		}
	
		/* IO window settings */
		PDEBUG(DEBUG_EXTRA, "IO window settings: cfg->io.nwin=%d "
		       "dflt.io.nwin=%d\n",
		       cfg->io.nwin, dflt.io.nwin);
		link->io.NumPorts1 = link->io.NumPorts2 = 0;
		if ((cfg->io.nwin > 0) || (dflt.io.nwin > 0)) {
			cistpl_io_t *io = (cfg->io.nwin) ? &cfg->io : &dflt.io;
			link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
			PDEBUG(DEBUG_EXTRA, "io->flags = 0x%04X, "
			       "io.base=0x%04x, len=%d\n", io->flags,
			       io->win[0].base, io->win[0].len);
			if (!(io->flags & CISTPL_IO_8BIT))
				link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;
			if (!(io->flags & CISTPL_IO_16BIT))
				link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
			link->io.IOAddrLines = io->flags &
				CISTPL_IO_LINES_MASK;
			link->io.BasePort1 = io->win[0].base;
			link->io.NumPorts1 = io->win[0].len;
			if (io->nwin > 1) {
				link->io.Attributes2 = link->io.Attributes1;
				link->io.BasePort2 = io->win[1].base;
				link->io.NumPorts2 = io->win[1].len;
			}
		}

		/* This reserves IO space but doesn't actually enable it */
		CFG_CHECK2(RequestIO, link->handle, &link->io);

		/* Now set up a common memory window, if needed.  There is room
		 * in the dev_link_t structure for one memory window handle,
		 * but if the base addresses need to be saved, or if multiple
		 * windows are needed, the info should go in the private data
		 * structure for this device.
		 *
		 * Note that the memory window base is a physical address, and
		 * needs to be mapped to virtual space with ioremap() before it
		 * is used.
		 */
		if ((cfg->mem.nwin > 0) || (dflt.mem.nwin > 0)) {
			cistpl_mem_t *mem =
				(cfg->mem.nwin) ? &cfg->mem : &dflt.mem;
			req.Attributes = WIN_DATA_WIDTH_16 |
				WIN_MEMORY_TYPE_CM | WIN_ENABLE;
			req.Base = mem->win[0].host_addr;
			req.Size = mem->win[0].len;

			if (local->is_symbol) {
				/* Symbol-based cards seems to require
				 * following settings: */
				printk("Symbol-based card - using different "
				       "common memory window settings\n");
				req.Attributes = WIN_DATA_WIDTH_8 |
					WIN_MEMORY_TYPE_CM |
					WIN_ADDR_SPACE_MEM |
					WIN_ENABLE;
				req.Size = 0;
			}

			req.AccessSpeed = 0;
			PDEBUG(DEBUG_EXTRA, "Requesting common memory window: "
			       "Base=0x%04X, Size=0x%04X\n",
			       (unsigned int) req.Base,
			       req.Size);
			link->win = (window_handle_t)link->handle;
			CFG_CHECK2(RequestWindow, &link->win, &req);
			map.Page = 0; map.CardOffset = mem->win[0].card_addr;
			CFG_CHECK2(MapMemPage, link->win, &map);
		}
		/* This configuration table entry is OK */
		break;

	next_entry:
		CS_CHECK(GetNextTuple, link->handle, &tuple);
	}

	/*
	 * Allocate an interrupt line.  Note that this does not assign a
	 * handler to the interrupt, unless the 'Handler' member of the
	 * irq structure is initialized.
	 */
	if (link->conf.Attributes & CONF_ENABLE_IRQ) {
		int i;
		link->irq.Attributes = IRQ_TYPE_EXCLUSIVE | IRQ_HANDLE_PRESENT;
		link->irq.IRQInfo1 = IRQ_INFO2_VALID | IRQ_LEVEL_ID;
		if (irq_list[0] == -1)
			link->irq.IRQInfo2 = irq_mask;
		else
			for (i = 0; i < 4; i++)
				link->irq.IRQInfo2 |= 1 << irq_list[i];
		link->irq.Handler = (void *) prism2_interrupt;
		link->irq.Instance = dev;
		CS_CHECK(RequestIRQ, link->handle, &link->irq);
	}
	
	/*
	 * This actually configures the PCMCIA socket -- setting up
	 * the I/O windows and the interrupt mapping, and putting the
	 * card and host interface into "Memory and IO" mode.
	 */
	CS_CHECK(RequestConfiguration, link->handle, &link->conf);

	dev->irq = link->irq.AssignedIRQ;
	dev->base_addr = link->io.BasePort1;

	/* Finally, report what we've done */
	printk(KERN_INFO "%s: index 0x%02x: Vcc %d.%d",
	       dev_info, link->conf.ConfigIndex,
	       link->conf.Vcc / 10, link->conf.Vcc % 10);
	if (link->conf.Vpp1)
		printk(", Vpp %d.%d", link->conf.Vpp1 / 10,
		       link->conf.Vpp1 % 10);
	if (link->conf.Attributes & CONF_ENABLE_IRQ)
		printk(", irq %d", link->irq.AssignedIRQ);
	if (link->io.NumPorts1)
		printk(", io 0x%04x-0x%04x", link->io.BasePort1,
		       link->io.BasePort1+link->io.NumPorts1-1);
	if (link->io.NumPorts2)
		printk(" & 0x%04x-0x%04x", link->io.BasePort2,
		       link->io.BasePort2+link->io.NumPorts2-1);
	if (link->win)
		printk(", mem 0x%06lx-0x%06lx", req.Base,
		       req.Base+req.Size-1);
	printk("\n");

	link->state |= DEV_CONFIG;
	link->state &= ~DEV_CONFIG_PENDING;


#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	{
		int i = 0;
		do {
			sprintf(dev->name, "wlan%d", i++);
		} while (dev_get(dev->name));
	}
#else
	memcpy(dev->name, "wlan%d", 7);
#endif
	if (register_netdev(dev)) {
		printk(KERN_WARNING "%s: register_netdev() failed!\n",
		       dev_info);
		prism2_release((u_long)link);
		return 1;
	}
	printk(KERN_INFO "%s: Registered netdevice %s\n", dev_info, dev->name);

#ifdef PRISM2_HOSTAPD
	local->apdev->base_addr = dev->base_addr;
	local->apdev->irq = dev->irq;
	sprintf(local->apdev->name, "%sap", dev->name);
	if (register_netdev(local->apdev)) {
		printk(KERN_WARNING "%s: register_netdev(AP) failed!\n",
		       dev_info);
		prism2_release((u_long)link);
		return 1;
	}
	printk(KERN_INFO "%s: Registered netdevice %s for AP management\n",
	       dev_info, local->apdev->name);
#endif /* PRISM2_HOSTAPD */

	prism2_init_proc(local);
	ap_init_data(local);
	prism2_wep_init(local);

	copy_dev_name(local->node, dev);
	link->dev = &local->node;

	ret = prism2_hw_config(dev, 1);
	prism2_check_sta_fw_version(local);
	return ret;

 cs_failed:
	cs_error(link->handle, last_fn, last_ret);
	prism2_release((u_long)link);
	return 1;
}


static void prism2_release(u_long arg)
{
	dev_link_t *link = (dev_link_t *)arg;
	struct net_device *dev = (struct net_device *) link->priv;
#ifndef PRISM2_HOSTAPD
	local_info_t *local = (local_info_t *) dev->priv;
#endif /* PRISM2_HOSTAPD */

	PDEBUG(DEBUG_FLOW, "prism2_release\n");

#ifndef PRISM2_HOSTAPD
	if (local->hw_ready && local->ap)
		ap_deauth_all_stas(dev, local->ap);
#endif /* PRISM2_HOSTAPD */

	if (link->open) {
		printk("%s: release postponed, '%s' still open\n",
		      dev_info, link->dev->dev_name);
		if (link->state & DEV_PRESENT) {
			link->state |= DEV_STALE_CONFIG;
			return;
		} else {
			/* FIX - this should not happen.. prism2_close()
			 * should be called from somewhere else */
			printk("Card already removed - closing device\n");
			prism2_close(dev);
		}
	}

	PDEBUG(DEBUG_FLOW, "release -> prism2_hw_shutdown\n");
	if (dev != NULL)
		prism2_hw_shutdown(dev, 0);
	PDEBUG(DEBUG_FLOW, "release <- prism2_hw_shutdown\n");


	if (link->win)
		CardServices(ReleaseWindow, link->win);
	CardServices(ReleaseConfiguration, link->handle);
	if (link->io.NumPorts1)
		CardServices(ReleaseIO, link->handle, &link->io);
	if (link->irq.AssignedIRQ)
		CardServices(ReleaseIRQ, link->handle, &link->irq);

	PDEBUG(DEBUG_FLOW, "release: CardServices(Release*) done\n");


	link->state &= ~DEV_CONFIG;

	if (link->state & DEV_STALE_LINK) {
		/* FIX - should this be here? */
		PDEBUG(DEBUG_FLOW, "prism2_release => prism2_detach\n");
		prism2_detach(link);
	}

	PDEBUG(DEBUG_FLOW, "release - done\n");
}



static int prism2_event(event_t event, int priority,
			event_callback_args_t *args)
{
	dev_link_t *link = args->client_data;
	struct net_device *dev = (struct net_device *) link->priv;

	switch (event) {
	case CS_EVENT_CARD_INSERTION:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_CARD_INSERTION\n", dev_info);
		link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
		if (prism2_config(link))
			dev->irq = 0;
		break;

	case CS_EVENT_CARD_REMOVAL:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_CARD_REMOVAL\n", dev_info);
		link->state &= ~DEV_PRESENT;
		if (link->state & DEV_CONFIG) {
			prism2_netif_stop_queues(dev);
			netif_device_detach(dev);
			mod_timer(&link->release, jiffies + HZ / 20);
		}
		break;

	case CS_EVENT_PM_SUSPEND:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_PM_SUSPEND\n", dev_info);
		link->state |= DEV_SUSPEND;
		/* fall through */

	case CS_EVENT_RESET_PHYSICAL:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_RESET_PHYSICAL\n", dev_info);
		if (link->state & DEV_CONFIG) {
			if (link->open) {
				prism2_netif_stop_queues(dev);
				netif_device_detach(dev);
			}
			CardServices(ReleaseConfiguration, link->handle);
		}
		break;

	case CS_EVENT_PM_RESUME:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_PM_RESUME\n", dev_info);
		link->state &= ~DEV_SUSPEND;
		/* fall through */

	case CS_EVENT_CARD_RESET:
		PDEBUG(DEBUG_EXTRA, "%s: CS_EVENT_CARD_RESET\n", dev_info);
		if (link->state & DEV_CONFIG) {
			CardServices(RequestConfiguration, link->handle,
				     &link->conf);
			if (link->open) {
				prism2_hw_shutdown(dev, 1);
				prism2_hw_config(dev, 0);
				netif_device_attach(dev);
				netif_start_queue(dev);
			}
		}
		break;

	default:
		PDEBUG(DEBUG_EXTRA, "%s: prism2_event() - unknown event %d\n",
		       dev_info, event);
		break;
	}
	return 0;
}


static int __init init_prism2(void)
{
	servinfo_t serv;

	printk(KERN_INFO "%s: %s\n"
	       "%s: (c) SSH Communications Security Corp <jkm@ssh.com>\n",
	       dev_info, version, dev_info);
	CardServices(GetCardServicesInfo, &serv);
	if (serv.Revision != CS_RELEASE_CODE) {
		printk(KERN_NOTICE
		       "%s: CardServices release does not match!\n", dev_info);
		return -1;
	}
	register_pccard_driver(&dev_info, &prism2_attach, &prism2_detach);

	return 0;
}


static void __exit exit_prism2(void)
{
	unregister_pccard_driver(&dev_info);
	while (dev_list) {
		PDEBUG(DEBUG_FLOW, "exit_prism2 - detaching device\n");
		del_timer(&dev_list->release);
		if (dev_list->state & DEV_CONFIG)
			prism2_release((u_long)dev_list);
		prism2_detach(dev_list);
	}

	if (prism2_proc != NULL)
		remove_proc_entry("prism2", proc_net);

	printk(KERN_INFO "%s: Driver unloaded\n", dev_info);
}


module_init(init_prism2);
module_exit(exit_prism2);
