/*
 *  linux/arch/arm/mach-sa1100/assabet-i2c.c
 *
 *  Copyright (C) 2001 Russell King
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  This is a combined I2C and L3 bus driver.
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/i2c-algo-bit.h>
#include <linux/l3/algo-bit.h>

#include <asm/hardware.h>
#include <asm/mach-types.h>

#define SDA	GPIO_GPIO15
#define SCL	GPIO_GPIO18
#define L3_MODE	GPIO_GPIO17

static void setsda(void *data, int state)
{
	if (state)
		GPDR &= ~SDA;
	else
		GPDR |= SDA;
}

static void setscl(void *data, int state)
{
	if (state)
		GPDR &= ~SCL;
	else
		GPDR |= SCL;
}

static int getsda(void *data)
{
	return GPLR & SDA;
}

static int getscl(void *data)
{
	return GPLR & SCL;
}

#ifdef CONFIG_I2C_BIT_ASSABET
static struct i2c_algo_bit_data i2c_bit_data = {
	setsda:		setsda,
	setscl:		setscl,
	getsda:		getsda,
	getscl:		getscl,
	udelay:		10,
	mdelay:		10,
	timeout:	100,
};

static struct i2c_adapter i2c_adapter = {
	name:			"Assabet_GPIO",
	algo_data:		&i2c_bit_data,
//	inc_use:		i2c_inc_use,
//	dec_use:		i2c_dec_use,
};

#define LOCK	&i2c_adapter.lock

static int __init i2c_init(void)
{
	return i2c_bit_add_bus(&i2c_adapter);
}

static void __exit i2c_exit(void)
{
	i2c_bit_del_bus(&i2c_adapter);
}

#else
static DECLARE_MUTEX(l3_lock);
#define LOCK		&l3_lock
#define i2c_init()	(0)
#define i2c_exit()	do { } while (0)
#endif

#ifdef CONFIG_L3_BIT_ASSABET
static void setmode(void *data, int state)
{
	if (state)
		GPSR = L3_MODE;
	else
		GPCR = L3_MODE;
}

static struct l3_algo_bit_data l3_bit_data = {
	data:		NULL,
	setdat:		setsda,
	setclk:		setscl,
	setmode:	setmode,
	getdat:		getsda,
	data_hold:	1,
	data_setup:	1,
	clock_high:	1,
	mode_hold:	1,
	mode_setup:	1,
};

static struct l3_adapter l3_adapter = {
	owner:		THIS_MODULE,
	name:		"l3-bit-assabet",
	algo_data:	&l3_bit_data,
	lock:		LOCK,
};

static int __init l3_init(void)
{
	return l3_bit_add_bus(&l3_adapter);
}

static void __exit l3_exit(void)
{
	l3_bit_del_bus(&l3_adapter);
}
#else
#define l3_init()	(0)
#define l3_exit()	do { } while (0)
#endif

static int __init bus_init(void)
{
	int ret;

	/*
	 * Default level for L3 mode is low.
	 * We set SCL and SDA high (i2c idle state).
	 */
	GPCR = L3_MODE | SCL | SDA;
	GPDR &= ~(SCL | SDA);
	GPDR |= L3_MODE;

	/*
	 * Release reset on UCB1300, ADI7171 and UDA1341.  We need
	 * to do this here so that we can communicate on the I2C/L3
	 * buses.
	 */
	BCR_set(BCR_CODEC_RST);
	mdelay(1);
	BCR_clear(BCR_CODEC_RST);
	mdelay(1);
	BCR_set(BCR_CODEC_RST);

	ret = i2c_init();
	if (ret == 0) {
		ret = l3_init();
		if (ret)
			i2c_exit();
	}

	return ret;
}

static void __exit bus_exit(void)
{
	l3_exit();
	i2c_exit();
}

module_init(bus_init);
module_exit(bus_exit);
