/*
* Driver interface to the ASIC Complasion chip on the iPAQ H3800
*
* Copyright 2001 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author:  Andrew Christian
*          <Andrew.Christian@compaq.com>
*          October 2001
*/

#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/mtd/mtd.h>
#include <linux/ctype.h>
#include <linux/spi/spi.h>
#include <linux/delay.h>
#include <linux/serial.h>  /* For bluetooth */

#include <asm/irq.h>
#include <asm/uaccess.h>   /* for copy to/from user space */
#include <asm/arch/hardware.h>
#include <asm/arch/h3600_hal.h>
#include <asm/arch/h3600_asic.h>
#include <asm/arch/serial_h3800.h>

#define H3600_ASIC_PROC_DIR     "asic"
#define H3600_ASIC_PROC_STATS   "stats"
#define REG_DIRNAME "registers"

// Define this to see all suspend/resume init/cleanup messages
#define DEBUG_INIT()  \
        if (0) printk(__FUNCTION__ "\n")

/* Parameters */

static int battery_sample_interval = 2000;  /* Two seconds  */
static int sleep_battery_interval  = 5;     /* Five seconds */
static int owm_sample_delay        = 200;   /* 20 milliseconds? */
static int owm_reset_delay         = 50;    /* 50 milliseconds */

MODULE_PARM(battery_sample_interval,"i");
MODULE_PARM_DESC(battery_sample_interval,"Time between battery samples (milliseconds)");
MODULE_PARM(sleep_battery_interval,"i");
MODULE_PARM_DESC(sleep_battery_interval,"Time betwen battery samples while asleep (seconds)");
MODULE_PARM(owm_reset_delay,"i");
MODULE_PARM_DESC(owm_reset_delay,"Pause time after OWM reset (milliseconds)");
MODULE_PARM(owm_sample_delay,"i");
MODULE_PARM_DESC(owm_sample_delay,"Pause time while waiting for OWM response (milliseconds)");

MODULE_AUTHOR("Andrew Christian");
MODULE_DESCRIPTION("Hardware abstraction layer for the iPAQ H3800");

/* Statistics */
struct asic_statistics {
	u32  spi_bytes;   /* Bytes handled by SPI */
	u32  spi_timeout; /* Timeouts in transmission */
	u32  spi_wip;     /* Write in process */
	u32  owm_timeout;
	u32  owm_reset;
	u32  owm_written;
	u32  owm_read;
	u32  owm_valid_isr[5];
	u32  owm_invalid_isr[5];
	u32  owm_post_isr[5];
} g_statistics;

/***********************************************************************************
 *      Shared resources                                                           
 *                                                                                 
 *      EX1 on Clock      24.576 MHz crystal (OWM,ADC,PCM,SPI,PWM,UART,SD,Audio)       
 *      EX2               33.8688 MHz crystal (SD Controller and Audio)
 ***********************************************************************************/

struct asic_shared {
	spinlock_t        lock;
	int               clock_ex1;    /* 24.765 MHz crystal */
	int               clock_ex2;    /* 33.8688 MHz crystal */
} g_shared;

enum ASIC_SHARED {
	ASIC_SHARED_CLOCK_EX1 = 1,  /* Bit fields */
	ASIC_SHARED_CLOCK_EX2 = 2
};

static void h3600_asic_shared_add( unsigned long *s, enum ASIC_SHARED v )
{
	unsigned long flags;
	if (0) printk(__FUNCTION__ " %lx %d\n",*s,v);

	if ( (*s & v) == v )    /* Already set */
		return;

	spin_lock_irqsave( &g_shared.lock, flags );

	if ( v & ASIC_SHARED_CLOCK_EX1 && !(*s & ASIC_SHARED_CLOCK_EX1)) {
		if ( !g_shared.clock_ex1++ ) {
			if (0) printk(__FUNCTION__ ": enabling EX1\n");
			H3800_ASIC2_CLOCK_Enable |= ASIC2_CLOCK_EX1;
		}
	}

	if ( v & ASIC_SHARED_CLOCK_EX2 && !(*s & ASIC_SHARED_CLOCK_EX2)) {
		if ( !g_shared.clock_ex2++ ) {
			if (0) printk(__FUNCTION__ ": enabling EX2\n");
			H3800_ASIC2_CLOCK_Enable |= ASIC2_CLOCK_EX2;
		}
	}

	spin_unlock_irqrestore( &g_shared.lock, flags );
	*s |= v;
}

static void h3600_asic_shared_release( unsigned long *s, enum ASIC_SHARED v )
{
	unsigned long flags;
	if (0) printk(__FUNCTION__ " %lx %d\n",*s,v);

	spin_lock_irqsave( &g_shared.lock, flags );

	if ( v & ASIC_SHARED_CLOCK_EX1 && !(~*s & ASIC_SHARED_CLOCK_EX1)) {
		if ( !--g_shared.clock_ex1 ) {
			if (0) printk(__FUNCTION__ ": disabling EX1\n");
			H3800_ASIC2_CLOCK_Enable &= ~ASIC2_CLOCK_EX1;
		}
	}

	if ( v & ASIC_SHARED_CLOCK_EX2 && !(~*s & ASIC_SHARED_CLOCK_EX2)) {
		if ( !--g_shared.clock_ex2 ) {
			if (0) printk(__FUNCTION__ ": disabling EX2\n");
			H3800_ASIC2_CLOCK_Enable &= ~ASIC2_CLOCK_EX2;
		}
	}

	spin_unlock_irqrestore( &g_shared.lock, flags );
	*s &= ~v;
}


static int __init h3600_asic_shared_init( void )
{
	DEBUG_INIT();
	spin_lock_init(&g_shared.lock);
	H3800_ASIC2_CLOCK_Enable &= ~(ASIC2_CLOCK_EX1 | ASIC2_CLOCK_EX2);
	return 0;
}

static void __exit h3600_asic_shared_cleanup( void )
{
	DEBUG_INIT();

	if ( H3800_ASIC2_CLOCK_Enable & (ASIC2_CLOCK_EX1 | ASIC2_CLOCK_EX2) ) {
		printk(__FUNCTION__ ": Error - EX still enabled\n");
		H3800_ASIC2_CLOCK_Enable &= ~(ASIC2_CLOCK_EX1 | ASIC2_CLOCK_EX2);
	}
}


/***********************************************************************************
 *      Sleeve ISR
 *
 *   Resources used:     KPIO interface on ASIC2
 *                       GPIO for Power button on SA1110
 ***********************************************************************************/

static void h3600_asic_sleeve_isr(int irq, void *dev_id, struct pt_regs *regs)
{
        int present = (GPLR & GPIO_H3800_NOPT_IND) ? 0 : 1;
	h3600_hal_option_detect( present );
        GEDR = GPIO_H3800_NOPT_IND;    /* Clear the interrupt */
}


/***********************************************************************************
 *      Keyboard ISRs
 *
 *   Resources used:     KPIO interface on ASIC2
 *                       GPIO for Power button on SA1110
 ***********************************************************************************/

#define MAKEKEY(index, down)  ((down) ? (index) : ((index) | 0x80))

static void h3600_asic_power_isr(int irq, void *dev_id, struct pt_regs *regs)
{
        int down = (GPLR & GPIO_H3600_NPOWER_BUTTON) ? 0 : 1;
	h3600_hal_keypress( MAKEKEY( 11, down ) );
}

struct asic_to_button { 
	u32 mask;
	u32 button;
	u8  down;
};

static struct asic_to_button asic_to_button[] = {
	{ 0, 0 },      /* Null */
	{ KPIO_RECORD_BTN_N,   KPIO_RECORD_BTN_N },  // 1:  Record button
	{ KPIO_KEY_LEFT_N,     KPIO_KEY_LEFT_N },    // 2:  Calendar
	{ KPIO_KEY_RIGHT_N,    KPIO_KEY_RIGHT_N },   // 3:  Contacts (looks like Outlook)
	{ KPIO_KEY_AP1_N,      KPIO_KEY_AP1_N },     // 4:  Envelope (Q on older iPAQs)
	{ KPIO_KEY_AP2_N,      KPIO_KEY_AP2_N },     // 5:  Start (looks like swoopy arrow)
	{ KPIO_ALT_KEY_ALL,    KPIO_KEY_5W1_N | KPIO_KEY_5W4_N | KPIO_KEY_5W5_N }, // 6:  Up
	{ KPIO_KEY_AP4_N,      KPIO_KEY_AP4_N },     // 7:  Right
	{ KPIO_KEY_AP3_N,      KPIO_KEY_AP3_N },     // 8:  Left
	{ KPIO_ALT_KEY_ALL,    KPIO_KEY_5W2_N | KPIO_KEY_5W3_N | KPIO_KEY_5W5_N }, // 9:  Down
	{ KPIO_ALT_KEY_ALL,    KPIO_KEY_5W5_N },     // 10: Action
};

static void h3600_asic_key_isr( int irq, void *dev_id, struct pt_regs *regs )
{
	int i;
	u32 keys = H3800_ASIC2_KPIOPIOD;

	if (0) printk(__FUNCTION__ ": %x\n", keys);
	
	for ( i = 1 ; i < (sizeof(asic_to_button)/sizeof(struct asic_to_button)) ; i++ ) {
		int down = ((~keys & asic_to_button[i].mask) == asic_to_button[i].button) ? 1 : 0;
		if ( down != asic_to_button[i].down ) {
			h3600_hal_keypress(MAKEKEY(i,down));
			asic_to_button[i].down = down;
		}
	}

	H3800_ASIC2_KPIINTALSEL  = (KPIO_KEY_ALL & ~keys);
}

static void h3600_asic_key_setup( void ) 
{
	H3800_ASIC2_KPIODIR        =  KPIO_KEY_ALL;  /* Inputs    */
	H3800_ASIC2_KPIOALT        =  KPIO_ALT_KEY_ALL;
	H3800_ASIC2_KPIINTTYPE    &= ~KPIO_KEY_ALL;  /* Level */
	H3800_ASIC2_KPIINTALSEL    =  (KPIO_KEY_ALL & ~H3800_ASIC2_KPIOPIOD);
}

static int h3600_asic_key_suspend( void )
{
	DEBUG_INIT();
	disable_irq( IRQ_H3800_KEY );
	return 0;
}

static void h3600_asic_key_resume( void )
{
	DEBUG_INIT();
	h3600_asic_key_setup();
	enable_irq( IRQ_H3800_KEY );
}

static int __init h3600_asic_key_init( void )
{
	int result;

	DEBUG_INIT();
	h3600_asic_key_setup();

	result = request_irq(IRQ_H3800_KEY, h3600_asic_key_isr, 
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_keyboard", NULL );

	if ( result )
		printk(KERN_CRIT __FUNCTION__ ": unable to grab keyboard virtual IRQ\n");
	return result;
}

static void __exit h3600_asic_key_cleanup( void )
{
	DEBUG_INIT();
	free_irq( IRQ_H3800_KEY, NULL );
}


/***********************************************************************************
 *      SPI interface
 *
 *   Resources used:     SPI interface on ASIC2
 *                       SPI Clock on CLOCK (CX5)
 *                       SPI KPIO interrupt line
 *   Shared resource:    EX1 (24.576 MHz crystal) on CLOCK
 ***********************************************************************************/

static struct h3600_asic_spidev {
	struct semaphore  lock;
	wait_queue_head_t waitq;
} g_spidev;

static void h3600_asic_spi_isr( int irq, void *dev_id, struct pt_regs *regs )
{
	if (0) printk(__FUNCTION__ "\n");

	wake_up_interruptible( &g_spidev.waitq );
	H3800_ASIC2_SPI_Control &= ~SPI_CONTROL_SPIE;   /* Clear the interrupt */
}

/* 
 * This routine both transmits and receives data from the SPI bus.
 * It returns the byte read in the result code, or a negative number
 * if there was an error.
 */

static int h3600_asic_spi_process_byte( unsigned char data )
{
	wait_queue_t  wait;
	signed long   timeout;
	int           result = 0;

	if (0) printk(__FUNCTION__ ": sending=0x%02x ", data);
	g_statistics.spi_bytes++;

	H3800_ASIC2_SPI_Control |= SPI_CONTROL_SPIE;     /* Enable interrupts */
	H3800_ASIC2_SPI_Data = data;                     /* Send data */
	H3800_ASIC2_SPI_Control |= SPI_CONTROL_SPE;      /* Start the transfer */

	/* We're basically using interruptible_sleep_on_timeout */
	/* and waiting for the transfer to finish */
	init_waitqueue_entry(&wait,current);
	add_wait_queue(&g_spidev.waitq, &wait);
	timeout = 100 * HZ / 1000;    /* 100 milliseconds (empirically derived) */

	while ( timeout > 0 ) {
		set_current_state( TASK_INTERRUPTIBLE );
		if ( !(H3800_ASIC2_SPI_Control & SPI_CONTROL_SPE )) {
			result = H3800_ASIC2_SPI_Data;
			break;
		}
		if ( signal_pending(current) ) {
			result = -ERESTARTSYS;
			break;
		}
		timeout = schedule_timeout( timeout );
		if ( timeout <= 0 ) {
			result = -ETIMEDOUT;       /* is this right? */
			g_statistics.spi_timeout++;
		}
	}
	set_current_state( TASK_RUNNING );
	remove_wait_queue(&g_spidev.waitq, &wait);

	H3800_ASIC2_SPI_Control &= ~SPI_CONTROL_SPIE;    /* Disable interrupts (may be timeout) */
	if (0) printk(" result=0x%02x\n", result);
	return result;
}

/* 
   Read from a Microchip 25LC040 EEPROM tied to CS1
   This is the standard EEPROM part on all option packs
*/

#define SPI_25LC040_RDSR        0x05     /* Read status register */
#define SPI_25LC040_RDSR_WIP  (1<<0)     /* Write-in-process (1=true) */
#define SPI_25LC040_RDSR_WEL  (1<<1)     /* Write enable latch */
#define SPI_25LC040_RDSR_BP0  (1<<2)     /* Block protection bit */
#define SPI_25LC040_RDSR_BP1  (1<<3)     /* Block protection bit */

#define SPI_25LC040_READ_HIGH(addr)   (0x03 |((addr & 0x100)>>5))  /* Read command (put A8 in bit 3) */
#define SPI_25LC040_READ_LOW(addr)    (addr&0xff)                  /* Low 8 bits */

/* Wait until the EEPROM is finished writing */
static int h3600_asic_spi_eeprom_ready( void )
{
	int result;
	int i;

	if (0) printk(__FUNCTION__ "\n");

	H3800_ASIC2_SPI_ChipSelectDisabled = 0; /* Disable all chip selects */

	for ( i = 0 ; i < 20 ; i++ ) {
		if ( (result = h3600_asic_spi_process_byte( SPI_25LC040_RDSR )) < 0 )
			return result;

		if ( (result = h3600_asic_spi_process_byte( 0 )) < 0 )
			return result;

		/* Really should wait a bit before giving up */
		if (!(result & SPI_25LC040_RDSR_WIP) )
			return 0;

		g_statistics.spi_wip++;
	}
	return -ETIMEDOUT;
}

static int h3600_asic_spi_eeprom_read( unsigned short address, unsigned char *data, unsigned short len )
{
	int result = 0;
	int i;

	if (0) printk(__FUNCTION__ ": address=%x, len=%d, data=%p\n", address,len,data);

	H3800_ASIC2_SPI_ChipSelectDisabled = 0; /* Disable all chip selects */

	if ( (result = h3600_asic_spi_process_byte( SPI_25LC040_READ_HIGH(address) )) < 0 )
		return result;

	if ( (result = h3600_asic_spi_process_byte( SPI_25LC040_READ_LOW(address) )) < 0 )
		return result;

	for ( i = 0 ; i < len ; i++ ) {
		if ( (result = h3600_asic_spi_process_byte( 0 )) < 0 )
			return result;
		data[i] = result;
	}

	return 0;
}

static int h3600_asic_spi_read(unsigned short address, unsigned char *data, unsigned short len)
{
	int result;
	unsigned long shared = 0;

	if (0) printk(__FUNCTION__ ": address=%x, len=%d, data=%p\n", address,len,data);

	if ( down_interruptible(&g_spidev.lock) )
		return -ERESTARTSYS;

	h3600_asic_shared_add( &shared, ASIC_SHARED_CLOCK_EX1 );

	H3800_ASIC2_SPI_Control = SPI_CONTROL_SPR(3) | SPI_CONTROL_SEL_CS0;
	H3800_ASIC2_CLOCK_Enable |= ASIC2_CLOCK_SPI;

	if ( (result = h3600_asic_spi_eeprom_ready()) < 0 )
		goto read_optionpaq_exit;

	if ( (result = h3600_asic_spi_eeprom_read( address, data, len )) < 0 )
		goto read_optionpaq_exit;

	result = 0;    /* Good return code */

read_optionpaq_exit:
	H3800_ASIC2_SPI_ChipSelectDisabled = 0;
	h3600_asic_shared_release( &shared, ASIC_SHARED_CLOCK_EX1 );

	up(&g_spidev.lock);
	return result;
}

/* 
   Read from the PCMCIA option jacket microcontroller
*/

#define SPI_PCMCIA_HEADER          0xA1     /* STX */
#define	SPI_PCMCIA_ID              0x10
#define SPI_PCMCIA_ID_RESULT       0x13     /* 3 bytes : x.xx */
#define SPI_PCMCIA_BATTERY         0x20
#define SPI_PCMCIA_BATTERY_RESULT  0x24     /* 4 bytes : chem percent flag voltage */

#define SPI_WRITE(_x) do { int _result = h3600_asic_spi_process_byte(_x); \
                           if (_result < 0) return _result; } while (0)
#define SPI_READ(_x)  do { _x = h3600_asic_spi_process_byte(0); \
                           if (_x < 0) return _x; } while (0)

static int h3600_asic_spi_pcmcia_read( unsigned char cmd, unsigned char reply, unsigned char *data )
{
	unsigned char checksum;
	int result;
	int i;

	/* Send message */
	SPI_WRITE( SPI_PCMCIA_HEADER );
	SPI_WRITE( cmd );
	SPI_WRITE( cmd );  /* The checksum */
	
	/* Pause for a jiffie to give the micro time to respond */
	set_current_state( TASK_INTERRUPTIBLE );
	schedule_timeout(1);
	set_current_state( TASK_RUNNING );

	/* Read message here */
	SPI_READ( result );
	if ( result != SPI_PCMCIA_HEADER )
		return -EIO;

	SPI_READ( result );
	if ( result != reply )
		return -EIO;
	checksum = result;

	for ( i = 0 ; i < ( reply & 0x0f ) ; i++ ) {
		SPI_READ( result );
		data[i] = result;
		checksum += result;
	}
	
	SPI_READ( result );
	if ( checksum != result )
		return -EIO;

	return 0;
}

static int h3600_asic_spi_read_pcmcia_battery( unsigned char *chem, unsigned char *percent, unsigned char *flag )
{
	int result;
	unsigned long shared = 0;
	unsigned char data[4];

	if ( down_interruptible(&g_spidev.lock) )
		return -ERESTARTSYS;

	h3600_asic_shared_add( &shared, ASIC_SHARED_CLOCK_EX1 );

	H3800_ASIC2_SPI_Control = SPI_CONTROL_SPR(2) | SPI_CONTROL_SEL_CS1;
	H3800_ASIC2_CLOCK_Enable |= ASIC2_CLOCK_SPI;

	if ( (result = h3600_asic_spi_pcmcia_read( SPI_PCMCIA_BATTERY, SPI_PCMCIA_BATTERY_RESULT, data )) < 0 )
		goto read_optionpaq_exit;

	result = 0;    /* Good return code */
	*chem    = data[0];
	*percent = data[1];
	*flag    = data[2];

read_optionpaq_exit:
	H3800_ASIC2_SPI_ChipSelectDisabled = 0;
	h3600_asic_shared_release( &shared, ASIC_SHARED_CLOCK_EX1 );

	up(&g_spidev.lock);
	return result;
}

static int h3600_asic_spi_suspend( void )
{
	DEBUG_INIT();
	down(&g_spidev.lock);   // Grab the lock, no interruptions
	disable_irq( IRQ_H3800_SPI );
	return 0;
}

static void h3600_asic_spi_resume( void )
{
	DEBUG_INIT();
	enable_irq( IRQ_H3800_SPI );
	up(&g_spidev.lock);
}

static int __init h3600_asic_spi_init( void )
{
	int result;
	DEBUG_INIT();
	init_waitqueue_head( &g_spidev.waitq );
	init_MUTEX( &g_spidev.lock );

	result = request_irq(IRQ_H3800_SPI, h3600_asic_spi_isr, 
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_spi", NULL );
	if ( result )
		printk(KERN_CRIT __FUNCTION__ ": unable to grab SPI virtual IRQ\n");
	return result;
}

static void __exit h3600_asic_spi_cleanup( void )
{
	DEBUG_INIT();
	down(&g_spidev.lock);   // Grab the lock, no interruptions
	free_irq( IRQ_H3800_SPI, NULL );
}


/***********************************************************************************
 *   SPI bus interface
 ***********************************************************************************/

int h3800_asic_spi_xfer(struct spi_adapter *adap, struct spi_msg msgs[], int num)
{
        int count = 0;
        int i;
	for (i = 0; i < num; i++) {
		struct spi_msg *pmsg = &msgs[i];
                if (pmsg->flags & SPI_M_RD) {
                        if (!h3600_spi_read(pmsg->addr, pmsg->buf, pmsg->len))
                                return count;
                        count++;
                } else {
                        if (!h3600_spi_write(pmsg->addr, pmsg->buf, pmsg->len))
                                return count;
                        count++;
                }
        }
        return count;
}

static struct spi_algorithm h3800_asic_spi_algorithm = {
        name: "h3800-asic-spi-algo",
        xfer: h3800_asic_spi_xfer
};

static struct spi_adapter h3800_asic_spi_adapter = {
        name: "h3xxx-spi",
        algo: &h3800_asic_spi_algorithm,
};


/***********************************************************************************
 *   Backlight
 *
 *   Resources used:     PWM_0
 *                       PWM Clock enable on CLOCK (CX7)
 *                       GPIO pin on ASIC1 (for frontlight power)
 ***********************************************************************************/

static unsigned long backlight_shared;

static int h3600_asic_backlight_control( enum flite_pwr power, unsigned char level )
{
	if (0) printk(__FUNCTION__ " power=%d level=%d\n", power, level);
	switch (power) {
	case FLITE_PWR_OFF:
		H3800_ASIC1_GPIO_OUT       &= ~GPIO1_FL_PWR_ON;
		H3800_ASIC2_PWM_0_TimeBase &= ~PWM_TIMEBASE_ENABLE;
		H3800_ASIC2_CLOCK_Enable   &= ~ASIC2_CLOCK_PWM;
		h3600_asic_shared_release( &backlight_shared, ASIC_SHARED_CLOCK_EX1 );
		break;
	case FLITE_PWR_ON:
		h3600_asic_shared_add( &backlight_shared, ASIC_SHARED_CLOCK_EX1 );
		H3800_ASIC2_CLOCK_Enable    |= ASIC2_CLOCK_PWM;
		if ( level < 21 ) level = 21;
		if ( level > 64 ) level = 64;
		H3800_ASIC2_PWM_0_DutyTime = level;
		H3800_ASIC2_PWM_0_TimeBase |= PWM_TIMEBASE_ENABLE;
		H3800_ASIC1_GPIO_OUT       |= GPIO1_FL_PWR_ON;
		break;
	}
	return 0;
}

/* 
   The backlight should automatically be handled through suspend/resume
   by the framebuffer, so no special handling is necessary
 */

static int h3600_asic_backlight_init( void )
{
	DEBUG_INIT();
	H3800_ASIC2_PWM_0_TimeBase   = PWM_TIMEBASE_VALUE(8);
	H3800_ASIC2_PWM_0_PeriodTime = 0x40;
	return 0;
}

static void h3600_asic_backlight_cleanup( void )
{
	DEBUG_INIT();
	H3800_ASIC1_GPIO_OUT       &= ~GPIO1_FL_PWR_ON;
	H3800_ASIC2_PWM_0_TimeBase &= ~PWM_TIMEBASE_ENABLE;
	H3800_ASIC2_CLOCK_Enable   &= ~ASIC2_CLOCK_PWM;
	h3600_asic_shared_release( &backlight_shared, ASIC_SHARED_CLOCK_EX1 );
}


/***********************************************************************************
 *      ADC - Shared resource
 *
 *   Resources used:     ADC 3 & 4
 *                       Clock: ADC (CX4)
 *                       ADC Interrupt on KPIO
 *
 *   Shared resources:   Clock: 24.576MHz crystal (EX1)
 *
 *   The ADC is multiplexed between the touchscreen, battery charger, and light sensor
 ***********************************************************************************/

enum adc_state {
	ADC_STATE_IDLE,
	ADC_STATE_TOUCHSCREEN,  // Servicing the touchscreen
	ADC_STATE_USER          // Servicing a user-level request
};

struct adc_data {
	enum adc_state     state;
	struct semaphore   lock;      // Mutex for access from user-level
	wait_queue_head_t  waitq;     // Waitq for user-level access (waits for interrupt service)
	int                last;      // Return value for user-level acces
	int                user_mux;  // Requested mux for the next user read
	int                ts_mux;    // Requested mux for the next touchscreen read
	int              (*ts_callback)(int);  // Touchscreen callback
	unsigned long      shared;    // Shared resources
} g_adcdev;

static void h3600_asic_start_adc_sample( int mux )
{
	H3800_ASIC2_ADMUX     = mux | ASIC2_ADMUX_CLKEN;
	H3800_ASIC2_ADCSR     = ASIC2_ADCSR_ADPS(4) | ASIC2_ADCSR_INT_ENABLE | ASIC2_ADCSR_ENABLE;
	enable_irq( IRQ_H3800_ADC );
	H3800_ASIC2_ADCSR    |= ASIC2_ADCSR_START;
}

static void h3600_asic_adc_select_next_sample( struct adc_data *adc )
{
	if ( adc->ts_mux > 0 ) {
		adc->state = ADC_STATE_TOUCHSCREEN;
		h3600_asic_start_adc_sample( adc->ts_mux );
	}
	else if ( adc->user_mux >= 0 ) {
		adc->state = ADC_STATE_USER;
		h3600_asic_start_adc_sample( adc->user_mux );
	} else {
		adc->state = ADC_STATE_IDLE;
	}
}

static void h3600_asic_adc_isr( int irq, void *dev_id, struct pt_regs *regs )
{
	struct adc_data *adc = &g_adcdev;
	int data = H3800_ASIC2_ADCDR;

	H3800_ASIC2_ADCSR &= ~ASIC2_ADCSR_INT_ENABLE;   /* Disable the interrupt */
	disable_irq( IRQ_H3800_ADC );

	switch ( adc->state ) {
	case ADC_STATE_IDLE:
		printk(__FUNCTION__ ": Error --- called when no outstanding requests!\n");
		break;
	case ADC_STATE_TOUCHSCREEN:
		if ( adc->ts_callback )
			adc->ts_mux = adc->ts_callback( data );
		else {
			printk(__FUNCTION__ ": Error --- no touchscreen callback\n");
			adc->ts_mux = 0;
		}
		break;
	case ADC_STATE_USER:
		if ( adc->user_mux >= 0 ) {
			adc->last     = data;
			adc->user_mux = -1;
			wake_up_interruptible( &adc->waitq );
		}
		break;
	}
	
	h3600_asic_adc_select_next_sample( adc );
}

/* Call this from touchscreen code (running in interrupt context) */
static void h3600_asic_adc_start_touchscreen( int (*callback)(int), int mux )
{
	struct adc_data *adc = &g_adcdev;

	adc->ts_mux = mux;
	adc->ts_callback = callback;
	if ( adc->state == ADC_STATE_IDLE )
		h3600_asic_adc_select_next_sample(adc);
}

/* Call this from user-mode programs */
static int h3600_asic_adc_read_channel( int mux )
{
	struct adc_data *adc = &g_adcdev;
	int              result;
	unsigned long    flags;

	if ( down_interruptible(&adc->lock) )
		return -ERESTARTSYS;

	// Kick start if we aren't currently running
	save_flags_cli( flags );
	adc->user_mux = mux;
	if ( adc->state == ADC_STATE_IDLE ) 
		h3600_asic_adc_select_next_sample( adc );
	restore_flags( flags );

	result = wait_event_interruptible( adc->waitq, adc->user_mux < 0 );

	adc->user_mux = -1;  // May not be -1 if we received a signal
	if ( result >= 0 ) 
		result = adc->last;
	
	up(&adc->lock);
	return result;
}

static void h3600_asic_adc_up( struct adc_data *adc )
{
	h3600_asic_shared_add( &adc->shared, ASIC_SHARED_CLOCK_EX1 );
	H3800_ASIC2_CLOCK_Enable         |= ASIC2_CLOCK_ADC;
}

static void h3600_asic_adc_down( struct adc_data *adc )
{
	H3800_ASIC2_CLOCK_Enable         &= ~ASIC2_CLOCK_ADC;
	h3600_asic_shared_release( &adc->shared, ASIC_SHARED_CLOCK_EX1 );

	// Clear any current touchscreen requests
	adc->ts_mux = 0;
	adc->state  = ADC_STATE_IDLE;
}

static int h3600_asic_adc_suspend( void )
{
	DEBUG_INIT();
	down(&g_adcdev.lock);  // No interruptions
	h3600_asic_adc_down( &g_adcdev );
	disable_irq( IRQ_H3800_ADC );
	return 0;
}

static void h3600_asic_adc_resume( void )
{
	DEBUG_INIT();
	enable_irq( IRQ_H3800_ADC );
	h3600_asic_adc_up( &g_adcdev );
	up(&g_adcdev.lock);
}

static int h3600_asic_adc_init( void )
{
	int result;

	DEBUG_INIT();
	init_MUTEX(&g_adcdev.lock);
	init_waitqueue_head( &g_adcdev.waitq );
	h3600_asic_adc_up( &g_adcdev );

	result = request_irq(IRQ_H3800_ADC, h3600_asic_adc_isr, 
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_adc", NULL );

	if ( result )
		printk(KERN_CRIT __FUNCTION__ ": unable to grab ADC IRQ %d error=%d\n", 
		       IRQ_H3800_ADC, result);

	return result;
}

static void h3600_asic_adc_cleanup( void )
{
	DEBUG_INIT();
	h3600_asic_adc_down( &g_adcdev );
	free_irq( IRQ_H3800_ADC, NULL );
}


/***********************************************************************************
 *      Touchscreen
 *
 *   Resources           ADC stuff
 *                       Pen interrupt on GPIO2
 *
 ***********************************************************************************/

#define TSMASK  (H3800_ASIC2_GPIOPIOD & ~(GPIO2_IN_Y1_N | GPIO2_IN_X0 | GPIO2_IN_Y0 | GPIO2_IN_X1_N))

struct ts_sample {
	unsigned int   mask;
	unsigned short mux;
	char *         name;
};

/* TODO:  This is ugly, but good for debugging.
   In the future, we'd prefer to use one of the ASIC2 timers
   to toggle ADC sampling, rather than queueing a timer.
   We also should look at settling times for the screen.
*/

const struct ts_sample g_samples[] = {
	{ GPIO2_IN_X1_N | GPIO2_IN_Y0, ASIC2_ADMUX_3_TP_X0, "X" },    /* Measure X */
	{ GPIO2_IN_X1_N | GPIO2_IN_Y0, ASIC2_ADMUX_3_TP_X0, "X" },    /* Measure X */
	{ GPIO2_IN_X1_N | GPIO2_IN_Y0, ASIC2_ADMUX_3_TP_X0, "X" },    /* Measure X */
	{ GPIO2_IN_X1_N | GPIO2_IN_Y0, ASIC2_ADMUX_3_TP_X0, "X" },    /* Measure X */
	{ GPIO2_IN_X1_N | GPIO2_IN_Y0, ASIC2_ADMUX_4_TP_Y1, "XY" },  
	{ GPIO2_IN_Y1_N | GPIO2_IN_X0, ASIC2_ADMUX_4_TP_Y1, "Y" },    /* Measure Y */
	{ GPIO2_IN_Y1_N | GPIO2_IN_X0, ASIC2_ADMUX_4_TP_Y1, "Y" },    /* Measure Y */
	{ GPIO2_IN_Y1_N | GPIO2_IN_X0, ASIC2_ADMUX_4_TP_Y1, "Y" },    /* Measure Y */
	{ GPIO2_IN_Y1_N | GPIO2_IN_X0, ASIC2_ADMUX_4_TP_Y1, "Y" },    /* Measure Y */
	{ GPIO2_IN_Y1_N | GPIO2_IN_X0, ASIC2_ADMUX_3_TP_X0, "YX" },    /* Measure Y */
	{ GPIO2_IN_Y1_N | GPIO2_IN_Y0, ASIC2_ADMUX_3_TP_X0, "CX" },
	{ GPIO2_IN_Y1_N | GPIO2_IN_Y0, ASIC2_ADMUX_4_TP_Y1, "CY" },

	{ GPIO2_IN_Y1_N | GPIO2_IN_X0 | GPIO2_IN_X1_N, 0,   "End" },      /* Go to PEN_IRQ mode */
};

#define TS_SAMPLE_COUNT (sizeof(g_samples)/sizeof(struct ts_sample))

struct touchscreen_data {
	unsigned long          shared;   // Shared resources
	int                    samples[TS_SAMPLE_COUNT];
	int                    index;
	struct timer_list      timer;
} g_touch;

#define ASIC_ADC_DELAY  10  /* Delay 10 milliseconds */

/* Called by ADC ISR routine */
/* Return the number of the next mux to read, or 0 to stop */

static int h3600_asic_touchscreen_record( int data )
{
	struct touchscreen_data *touch = &g_touch;
	const struct ts_sample  *s;

	touch->samples[ touch->index++ ] = data;
	s = &g_samples[ touch->index ];

	H3800_ASIC2_GPIOPIOD = TSMASK | s->mask;    // Set the output pins
	if ( !s->mux )
		mod_timer(&touch->timer, jiffies + (ASIC_ADC_DELAY * HZ) / 1000);

	return s->mux;
}

static void h3600_asic_touchscreen_start_record( struct touchscreen_data *touch )
{
	touch->index = 0;
	H3800_ASIC2_GPIOPIOD = TSMASK | g_samples[0].mask;
	h3600_asic_adc_start_touchscreen( h3600_asic_touchscreen_record, g_samples[0].mux );
}

/* Invoked after a complete series of ADC samples has been taken  */
static void h3600_asic_timer_callback( unsigned long nr )
{
	// The last ADC sample sets us up for "pen" mode
	if ( H3800_ASIC2_GPIOPIOD & GPIO2_PEN_IRQ ) {
		h3600_hal_touchpanel(0,0,0);
		enable_irq( IRQ_H3800_PEN );
	} else {
		unsigned long flags;
		h3600_hal_touchpanel( g_touch.samples[3], g_touch.samples[8], 1 );
		save_flags_cli(flags);
		h3600_asic_touchscreen_start_record( &g_touch );
		restore_flags(flags);
	}
}

static void h3600_asic_pen_isr( int irq, void *dev_id, struct pt_regs *regs )
{
	disable_irq( IRQ_H3800_PEN );
	h3600_asic_touchscreen_start_record( &g_touch );
}

static int h3600_asic_touchscreen_suspend( void )
{
	DEBUG_INIT();
	disable_irq( IRQ_H3800_PEN );
	if (del_timer_sync(&g_touch.timer))
		h3600_hal_touchpanel(0,0,0);
	H3800_ASIC2_GPIOPIOD    = TSMASK | GPIO2_IN_Y1_N | GPIO2_IN_X1_N; /* Normal off state */
	return 0;
}

static void h3600_asic_touchscreen_resume( void )
{
	DEBUG_INIT();

	H3800_ASIC2_GPIOPIOD    = TSMASK | GPIO2_IN_Y1_N | GPIO2_IN_X0 | GPIO2_IN_X1_N;
	enable_irq( IRQ_H3800_PEN );
}

static int h3600_asic_touchscreen_init( void )
{
	int result;
	DEBUG_INIT();

	init_timer(&g_touch.timer);
	g_touch.timer.function = h3600_asic_timer_callback;
	g_touch.timer.data     = (unsigned long) NULL;

	H3800_ASIC2_GPIOPIOD    = TSMASK | GPIO2_IN_Y1_N | GPIO2_IN_X0 | GPIO2_IN_X1_N;

	result = request_irq(IRQ_H3800_PEN, h3600_asic_pen_isr, 
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_touchscreen", NULL );

	if ( result )
		printk(KERN_CRIT __FUNCTION__ ": unable to grab touchscreen virtual IRQ\n");

	return result;
}

static void h3600_asic_touchscreen_cleanup( void )
{
	DEBUG_INIT();

	free_irq( IRQ_H3800_PEN, NULL );
	if (del_timer_sync(&g_touch.timer))
		h3600_hal_touchpanel(0,0,0);

	H3800_ASIC2_GPIOPIOD    = TSMASK | GPIO2_IN_Y1_N | GPIO2_IN_X1_N; /* Normal off state */
}

/***********************************************************************************
 *   One wire interface for talking with batteries
 *
 *   Resources used:     OWM interface on ASIC2
 *                       OWM Clock on CLOCK (CX6)
 *                       OWM KPIO interrupt line
 *   Shared resource:    EX1 (24.576 MHz crystal) on CLOCK
 ***********************************************************************************/

enum owm_state {
	OWM_STATE_IDLE = 0,
	OWM_STATE_RESET,
	OWM_STATE_WRITE,
	OWM_STATE_READ,
	OWM_STATE_DONE
};

static unsigned char *owm_state_names[] = {
	"Idle", "Reset", "Write", "Read", "Done"
};

static struct h3600_asic_owmdev {
	struct semaphore  lock;
	wait_queue_head_t waitq;
	enum owm_state    state;
	unsigned long     shared;
	unsigned char     last;
} g_owmdev;

struct owm_net_address {
	char address[8];
};

static int  h3600_asic_owm_sample( struct h3600_asic_owmdev *dev )
{
	int value = H3800_ASIC2_OWM_Interrupt;

	if ( (value & OWM_INT_PD) && dev->state == OWM_STATE_RESET ) {
		wake_up_interruptible(&dev->waitq);
		return 0;
	}
	
	if ( (value & OWM_INT_TBE) && (value & OWM_INT_RBF) && dev->state == OWM_STATE_WRITE ) {
		dev->last = H3800_ASIC2_OWM_Data;
		wake_up_interruptible(&dev->waitq);
		return 0;
	}

	if ( (value & OWM_INT_RBF) ) {
		dev->last = H3800_ASIC2_OWM_Data;
		if ( dev->state == OWM_STATE_READ ) {
			wake_up_interruptible(&dev->waitq);
			return 0;
		}
	}
	return 1;
}

static void h3600_asic_owm_isr( int irq, void *dev_id, struct pt_regs *regs ) 
{
	if ( h3600_asic_owm_sample(&g_owmdev) )
		g_statistics.owm_invalid_isr[g_owmdev.state]++;
	else {
		g_statistics.owm_valid_isr[g_owmdev.state]++;
		g_owmdev.state = OWM_STATE_DONE;
	}
}

static int one_wire_wait_for_interrupt( int msec )
{
	wait_queue_t  wait;
	signed long   timeout;
	int           result = 0;

	/* We're basically using interruptible_sleep_on_timeout */
	/* and waiting for the transfer to finish */
	init_waitqueue_entry(&wait,current);
	add_wait_queue(&g_owmdev.waitq, &wait);
	timeout = msec * HZ / 1000;

	while ( timeout > 0 ) {
		set_current_state( TASK_INTERRUPTIBLE );
		if ( g_owmdev.state == OWM_STATE_DONE )
			break;
		if ( signal_pending(current) ) {
			result = -ERESTARTSYS;
			break;
		}
		timeout = schedule_timeout( timeout );
		if ( timeout <= 0 ) {
			if ( h3600_asic_owm_sample(&g_owmdev) ) {
				result = -ETIMEDOUT;       /* is this right? */
				g_statistics.owm_timeout++;
			}
			else {
				g_statistics.owm_post_isr[g_owmdev.state]++;
			}
		}
	}
	set_current_state( TASK_RUNNING );
	remove_wait_queue(&g_owmdev.waitq, &wait);

	g_owmdev.state = OWM_STATE_IDLE;
	return result;
}

static int one_wire_reset( void )
{
	int result;

	if (0) printk(__FUNCTION__ "\n");

	g_statistics.owm_reset++;
	g_owmdev.state = OWM_STATE_RESET;
	result = H3800_ASIC2_OWM_Interrupt;    /* Dummy read */

	H3800_ASIC2_OWM_Command |= OWM_CMD_ONE_WIRE_RESET;
	result = one_wire_wait_for_interrupt( owm_sample_delay );

	if ( result ) {
		printk(__FUNCTION__ " OWM reset failed %d (0x%04x)\n", result, H3800_ASIC2_OWM_Interrupt);
		return result;
	}
		
	/* No battery? */
	if ( H3800_ASIC2_OWM_Interrupt & OWM_INT_PDR ) { /* 0 indicates success */
		printk(__FUNCTION__ " OWM reset failed: no battery\n");
		return -ERESTARTSYS;
	}

	udelay(owm_reset_delay);
	return 0;
}

static int one_wire_write( unsigned char data )
{
	int result;

	if (0) printk(__FUNCTION__ ": 0x%02x\n", data);
	g_statistics.owm_written++;
	g_owmdev.state = OWM_STATE_WRITE;

	result = H3800_ASIC2_OWM_Interrupt;    /* Dummy read */

	H3800_ASIC2_OWM_Data = data;
	result = one_wire_wait_for_interrupt( owm_sample_delay );

	if ( result ) 
		printk("OWM write failed %d (0x%04x)\n", 
		       result, H3800_ASIC2_OWM_Interrupt);

	return result;
}

/* To read, we must clock in data and then read the output buffer */
static int one_wire_read( void )
{
	int result;

	if (0) printk(__FUNCTION__ "\n");
	g_statistics.owm_read++;
	g_owmdev.state = OWM_STATE_READ;

	result = H3800_ASIC2_OWM_Interrupt;    /* Dummy read */

	H3800_ASIC2_OWM_Data = 0xff;
	result = one_wire_wait_for_interrupt( owm_sample_delay );

	if ( result ) {
		printk("OWM read failed %d (0x%04x)\n", 
		       result, H3800_ASIC2_OWM_Interrupt);
		return result;
	}

	return g_owmdev.last;
}

/* Higher-level routines */

static int h3600_asic_owm_read_bytes( struct owm_net_address *net, unsigned char address, 
				      unsigned char *data, unsigned short len )
{
	int result = 0;
	int i;

	if ( down_interruptible(&g_owmdev.lock) )
		return -ERESTARTSYS;

	if ( (result = one_wire_reset()) != 0 ) goto owm_read_bytes_fail;

	if ( net ) {
		if ((result = one_wire_write(0x55)) != 0) goto owm_read_bytes_fail;
		for ( i = 0 ; i < 8 ; i++ )
			if ((result = one_wire_write(net->address[i])) != 0 ) goto owm_read_bytes_fail;
	}
	else {
		if ((result = one_wire_write(0xcc)) != 0) goto owm_read_bytes_fail;
	}
	/* Set address */
	if ( (result = one_wire_write(0x69)) < 0 ) goto owm_read_bytes_fail;
	if ( (result = one_wire_write(address)) < 0 ) goto owm_read_bytes_fail;

	/* Read data */
	for ( i = 0 ; i < len ; i++ ) {
		if ((result = one_wire_read()) < 0) goto owm_read_bytes_fail;
		data[i] = result;
	}

	result = 0;

owm_read_bytes_fail:
	up(&g_owmdev.lock);
	return result;
}

/*
static int h3600_asic_owm_write_bytes( struct owm_net_address *net, unsigned char address, 
				      unsigned char *data, unsigned short len )
{
	int result = 0;
	int i;

	if ( down_interruptible(&g_owmdev.lock) )
		return -ERESTARTSYS;

	if ( (result = one_wire_reset())     != 0 ) goto owm_write_bytes_fail;

	if ( net ) {
		if ((result = one_wire_write(0x55)) != 0) goto owm_write_bytes_fail;
		for ( i = 0 ; i < 8 ; i++ )
			if ((result = one_wire_write(net->address[i])) != 0 ) goto owm_write_bytes_fail;
	}
	else {
		if ((result = one_wire_write(0xcc)) != 0) goto owm_write_bytes_fail;
	}
	// Set address 
	if ( (result = one_wire_write(0x6c)) < 0 ) goto owm_write_bytes_fail;
	if ( (result = one_wire_write(address)) < 0 ) goto owm_write_bytes_fail;

	// Read data 
	for ( i = 0 ; i < len ; i++ )
		if ((result = one_wire_write(data[i])) < 0) goto owm_write_bytes_fail;

	result = 0;

owm_write_bytes_fail:
	up(&g_owmdev.lock);
	return result;
}
*/

static void h3600_asic_owm_up( struct h3600_asic_owmdev *owm )
{
	owm->state = OWM_STATE_IDLE;

	h3600_asic_shared_add( &owm->shared, ASIC_SHARED_CLOCK_EX1 );
	H3800_ASIC2_CLOCK_Enable         |= ASIC2_CLOCK_OWM;
	H3800_ASIC2_OWM_ClockDivisor      = 2;
	H3800_ASIC2_OWM_InterruptEnable   = OWM_INTEN_ERBF | OWM_INTEN_IAS | OWM_INTEN_EPD;
	H3800_ASIC2_INTR_MaskAndFlag     |= ASIC2_INTMASK_OWM; /* Turn on ASIC interrupts */
}

static void h3600_asic_owm_down( struct h3600_asic_owmdev *owm )
{
	wake_up_interruptible(&owm->waitq);

	H3800_ASIC2_INTR_MaskAndFlag     &= ~ASIC2_INTMASK_OWM; /* Turn off ASIC interrupts */
	H3800_ASIC2_OWM_InterruptEnable   = 0;
	H3800_ASIC2_CLOCK_Enable         &= ~ASIC2_CLOCK_OWM;
	h3600_asic_shared_release( &owm->shared, ASIC_SHARED_CLOCK_EX1 );
}

static int h3600_asic_owm_suspend( void )
{
	DEBUG_INIT();
	down(&g_owmdev.lock);
	h3600_asic_owm_down( &g_owmdev );
	disable_irq( IRQ_H3800_OWM );
	return 0;
}

static void h3600_asic_owm_resume( void )
{
	DEBUG_INIT();
	enable_irq( IRQ_H3800_OWM );
	h3600_asic_owm_up( &g_owmdev );
	up(&g_owmdev.lock);
}

static int h3600_asic_owm_init( void )
{
	int result;
	DEBUG_INIT();

	init_waitqueue_head( &g_owmdev.waitq );
	init_MUTEX(&g_owmdev.lock);

	h3600_asic_owm_up( &g_owmdev );

	result = request_irq(IRQ_H3800_OWM, h3600_asic_owm_isr, 
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_owm", NULL );

	if ( result )
		printk(KERN_CRIT __FUNCTION__ ": unable to grab OWM virtual IRQ\n");
	return result;
}

static void h3600_asic_owm_cleanup( void )
{
	DEBUG_INIT();
	h3600_asic_owm_down( &g_owmdev );
	free_irq( IRQ_H3800_OWM, NULL );
}


/***********************************************************************************
 *   LED control
 ***********************************************************************************/

enum led_color {
	BLUE_LED,
	GREEN_LED,
	YELLOW_LED
};

static void h3600_asic_set_led( enum led_color color, int tbs, int pts, int dts )
{
	H3800_ASIC2_LED_TimeBase(color) = 0;
	if ( tbs ) {
		H3800_ASIC2_LED_PeriodTime(color) = pts;
		H3800_ASIC2_LED_DutyTime(color)   = dts;
		H3800_ASIC2_LED_TimeBase(color)   = tbs;
	}
}

#define h3600_asic_led_on(color)  \
        h3600_asic_set_led(color, LEDTBS_BLINK | LEDTBS_AUTOSTOP | LEDTBS_ALWAYS, 4, 4 )

#define h3600_asic_led_off(color)  \
        h3600_asic_set_led(color, 0, 0, 0)

#define h3600_asic_led_blink(color,rate,pts,dts)  \
        h3600_asic_set_led(color, LEDTBS_BLINK | LEDTBS_AUTOSTOP | LEDTBS_ALWAYS | rate, pts, dts )


/***********************************************************************************
 *   Battery control
 * 
 *   Interface for talking with batteries and controlling the battery charger.
 *
 *   Resources used:     OWM interface
 *                       AC GPIO line
 *                       ADC converter to read battery charger status
 ***********************************************************************************/

enum charging_state {
	CHARGING_STATE_INIT = 0,
	CHARGING_STATE_NO_AC,      /* No AC */
	CHARGING_STATE_ACTIVE,     /* Actively charging the battery */
	CHARGING_STATE_FULL        /* The battery is fully charged */
};

struct h3600_asic_batdev {
	struct timer_list    timer;
	enum charging_state  state;
} g_batdev;

static void h3600_asic_battery_set_led( struct h3600_asic_batdev *dev, enum charging_state state )
{
	if ( state != dev->state ) {
		switch (state) {
		case CHARGING_STATE_INIT: /* Deliberate fall through */
		case CHARGING_STATE_NO_AC:
			H3800_ASIC2_LED_2_TimeBase = 0;
			break;
		case CHARGING_STATE_ACTIVE:
			H3800_ASIC2_LED_2_TimeBase   = 0;
			H3800_ASIC2_LED_2_PeriodTime = 4;
			H3800_ASIC2_LED_2_DutyTime   = 2;
			H3800_ASIC2_LED_2_TimeBase   = LEDTBS_BLINK | LEDTBS_AUTOSTOP | LEDTBS_ALWAYS;
			break;
		case CHARGING_STATE_FULL:
			H3800_ASIC2_LED_2_TimeBase   = 0;
			H3800_ASIC2_LED_2_PeriodTime = 4;
			H3800_ASIC2_LED_2_DutyTime   = 4;
			H3800_ASIC2_LED_2_TimeBase   = LEDTBS_BLINK | LEDTBS_AUTOSTOP | LEDTBS_ALWAYS;
			break;
		}
		dev->state = state;
	}
}

static int voltage_to_percent( int voltage, int *cal )
{
	int i;

	if ( voltage > cal[0] )
		return 100;

	for ( i = 2 ; cal[i] > 0 ; i+=2 ) 
		if ( voltage > cal[i] ) {
			int p1 = cal[i+1];
			int v1 = cal[i];
			int p2 = cal[i-1];
			int v2 = cal[i-2];
			return (p1 * (v2 - voltage) + p2 * (voltage - v1)) / (v2 - v1);
		}

	return 0;
}

static int lipolymer_calibration[] = {
	881, 100, // 100%
	830,  90, //  90%
	816,  80, //  80%
	805,  70, //  70%
	797,  60, //  60%
	789,  50, //  50%
	783,  40, //  40%
	777,  30, //  30%
	770,  20, //  20%
	764,  10, //  10%
	750,   0, //   0% 
	  0
};

/* 
   The OWM interface call doesn't always work ... sometimes the battery
   just doesn't respond.  We return an error code, but the APM circuitry
   doesn't use error codes (go figure!).  So we also return a structure filled
   out with values equal to the last time we called.  The only value that is
   affected will be the battery voltage level.
*/

static int h3600_asic_battery_read( struct h3600_battery *query )
{
	static int voltage   = 783;         /* DUMMY VALUE - good for first initialization */
	static int chemistry = H3600_BATT_CHEM_UNKNOWN;

	int result;
	unsigned char buf[64];
	int percentage;
        int ac_power = (GPLR & GPIO_H3800_AC_IN) ? 0 : 1;

	result = h3600_asic_owm_read_bytes( NULL, 0, buf, 64 );

	if ( !result && buf[1] ) {
		chemistry = buf[48] >> 6;
		voltage   = (((char) buf[12]) << 8 | buf[13]) >> 5;   /* In units of 4.88 mV per */
	}

	query->ac_status            = (ac_power ? H3600_AC_STATUS_AC_ONLINE : H3600_AC_STATUS_AC_OFFLINE );
	query->battery_count        = 1;
	query->battery[0].chemistry = ( chemistry == 0 ? H3600_BATT_CHEM_LIPOLY : H3600_BATT_CHEM_UNKNOWN );
	query->battery[0].voltage   = voltage;

	query->battery[0].status    = H3600_BATT_STATUS_UNKNOWN;
	switch ( g_batdev.state ) {
	case CHARGING_STATE_INIT:
	case CHARGING_STATE_NO_AC:
		if ( voltage > 782 )  /* About 3.82 volts */
			query->battery[0].status = H3600_BATT_STATUS_HIGH;
		else if ( voltage > 764 ) /* About 3.73 voltags */
			query->battery[0].status = H3600_BATT_STATUS_LOW;
		else
			query->battery[0].status = H3600_BATT_STATUS_CRITICAL;
		break;
	case CHARGING_STATE_ACTIVE:
		query->battery[0].status = H3600_BATT_STATUS_CHARGING;
		voltage -= 12;   /* Adjust for charging effect on battery percentage */
		break;
	case CHARGING_STATE_FULL:
		query->battery[0].status = H3600_BATT_STATUS_FULL;
		break;
	}

	percentage = voltage_to_percent( voltage, lipolymer_calibration );
	query->battery[0].percentage = percentage;
	query->battery[0].life       = 300 * percentage / 100;
	
	/* If a sleeve has been inserted, give it a try */
	if ( !(GPLR & GPIO_H3800_NOPT_IND ) && (H3800_ASIC2_GPIOPIOD & GPIO2_OPT_ON)) {
		unsigned char chem, percent, flag;
		if ( !h3600_asic_spi_read_pcmcia_battery( &chem, &percent, &flag ) ) {
			query->battery_count = 2;
			query->battery[1].chemistry  = chem;
			query->battery[1].voltage    = 0;
			query->battery[1].status     = flag;
			query->battery[1].percentage = percent;
			query->battery[1].life       = 0;
		}
	}

	return result;
}

static void h3600_asic_battery_probe_task_handler( void *nr )
{
	struct h3600_asic_batdev *bat = (struct h3600_asic_batdev *) nr;	
	int result = h3600_asic_adc_read_channel( ASIC2_ADMUX_1_IMIN );

	if ( result < 0 ) {
		printk(__FUNCTION__ " error reading battery channel\n");
	}
	else if ( result < 100 ) {
		h3600_asic_battery_set_led( bat, CHARGING_STATE_FULL );
	}
	else {
		h3600_asic_battery_set_led( bat, CHARGING_STATE_ACTIVE );
	}
}

static struct tq_struct battery_probe_task = { routine: h3600_asic_battery_probe_task_handler };

static void h3600_asic_battery_timer_callback( unsigned long nr )
{
	struct h3600_asic_batdev *bat = (struct h3600_asic_batdev *) nr;
	battery_probe_task.data = bat;
	schedule_task(&battery_probe_task);
	mod_timer(&bat->timer, jiffies + battery_sample_interval);
}

static void h3600_asic_battery_up( struct h3600_asic_batdev *bat )
{
        int ac_power = (GPLR & GPIO_H3800_AC_IN) ? 0 : 1;
	
	if ( ac_power ) {
		h3600_asic_battery_set_led( bat, CHARGING_STATE_ACTIVE );
		H3800_ASIC1_GPIO_OUT |= GPIO1_CH_TIMER;
		mod_timer(&bat->timer, jiffies + battery_sample_interval);
	}
	else {
		h3600_asic_battery_set_led( bat, CHARGING_STATE_NO_AC );
		H3800_ASIC1_GPIO_OUT &= ~GPIO1_CH_TIMER;
		del_timer_sync(&bat->timer);
	}
}

static void h3600_asic_battery_down( struct h3600_asic_batdev *bat )
{
	h3600_asic_battery_set_led( bat, CHARGING_STATE_NO_AC );
	H3800_ASIC1_GPIO_OUT &= ~GPIO1_CH_TIMER;
	del_timer_sync(&bat->timer);
        flush_scheduled_tasks();
}

static void h3600_asic_ac_in_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	h3600_asic_battery_up( &g_batdev );
}

static int h3600_asic_battery_suspend( void )
{
	DEBUG_INIT();
//	h3600_asic_battery_down( &g_batdev );
	del_timer_sync(&g_batdev.timer);
        flush_scheduled_tasks();
	return 0;
}

static void h3600_asic_battery_resume( void )
{
	DEBUG_INIT();
	h3600_asic_battery_up( &g_batdev );
}

static int h3600_asic_battery_init( void )
{
	DEBUG_INIT();

	init_timer(&g_batdev.timer);
	g_batdev.timer.function = h3600_asic_battery_timer_callback;
	g_batdev.timer.data     = (unsigned long) &g_batdev;
	g_batdev.state          = CHARGING_STATE_INIT;

	h3600_asic_battery_up( &g_batdev );
	return 0;
}

static void h3600_asic_battery_cleanup( void )
{
	DEBUG_INIT();

	h3600_asic_battery_down( &g_batdev );
}

/***********************************************************************************
 *   Audio handlers
 ***********************************************************************************/

#define SET_ASIC2_CLOCK(x) \
	H3800_ASIC2_CLOCK_Enable = (H3800_ASIC2_CLOCK_Enable & ~ASIC2_CLOCK_AUDIO_MASK) | (x)

static int h3600_asic_audio_clock( long samplerate )
{
	static unsigned long shared;

	if ( !samplerate ) {
		h3600_asic_shared_release( &shared, ASIC_SHARED_CLOCK_EX1 );
		h3600_asic_shared_release( &shared, ASIC_SHARED_CLOCK_EX2 );
		return 0;
	}

	/* Set the external clock generator */
	switch (samplerate) {
	case 24000:
	case 32000:
	case 48000:
		/* 12.288 MHz - needs 24.576 MHz crystal */
		h3600_asic_shared_add( &shared, ASIC_SHARED_CLOCK_EX1 );
		SET_ASIC2_CLOCK(ASIC2_CLOCK_AUDIO_2);
		h3600_asic_shared_release( &shared, ASIC_SHARED_CLOCK_EX2 );
		break;
	case 22050:
	case 29400:
	case 44100:
		/* 11.2896 MHz - needs 33.869 MHz crystal */
		h3600_asic_shared_add( &shared, ASIC_SHARED_CLOCK_EX2 );
		SET_ASIC2_CLOCK(ASIC2_CLOCK_AUDIO_4);
		h3600_asic_shared_release( &shared, ASIC_SHARED_CLOCK_EX1 );
		break;
	case 8000:
	case 10666:
	case 16000:
		/* 4.096 MHz  - needs 24.576 MHz crystal */
		h3600_asic_shared_add( &shared, ASIC_SHARED_CLOCK_EX1 );
		SET_ASIC2_CLOCK(ASIC2_CLOCK_AUDIO_1);
		h3600_asic_shared_release( &shared, ASIC_SHARED_CLOCK_EX2 );
		break;
	case 10985:
	case 14647:
	case 21970:
		/* 5.6245 MHz  - needs 33.869 MHz crystal */
		h3600_asic_shared_add( &shared, ASIC_SHARED_CLOCK_EX2 );
		SET_ASIC2_CLOCK(ASIC2_CLOCK_AUDIO_3);
		h3600_asic_shared_release( &shared, ASIC_SHARED_CLOCK_EX1 );
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static int h3600_asic_audio_power( long samplerate )
{
	int retval;

	H3800_ASIC1_GPIO_OUT &= ~GPIO1_AUD_PWR_ON;
	retval = h3600_asic_audio_clock(samplerate);

	if ( samplerate > 0 )
		H3800_ASIC1_GPIO_OUT |= GPIO1_AUD_PWR_ON;

	return retval;
}

static void h3600_asic_audio_fix_jack( int mute )
{
	int earphone = H3800_ASIC2_GPIOPIOD & GPIO2_EAR_IN_N ? 0 : 1;

	if ( earphone )
		H3800_ASIC2_GPIINTESEL |= GPIO2_EAR_IN_N; /* Rising */
	else
		H3800_ASIC2_GPIINTESEL &= ~GPIO2_EAR_IN_N; /* Falling */

	if ( mute ) 
		H3800_ASIC1_GPIO_OUT = (H3800_ASIC1_GPIO_OUT | GPIO1_EAR_ON_N) & ~GPIO1_SPK_ON;
	else {
		if ( earphone )
			H3800_ASIC1_GPIO_OUT &= ~(GPIO1_SPK_ON | GPIO1_EAR_ON_N);
		else
			H3800_ASIC1_GPIO_OUT |= GPIO1_SPK_ON | GPIO1_EAR_ON_N;
	}
}

/* Called from HAL or the debounce timer (from an interrupt)*/
static int h3600_asic_audio_mute( int mute )
{
	unsigned long flags;
	local_irq_save(flags);
	h3600_asic_audio_fix_jack(mute);
	local_irq_restore(flags);

	return 0;
}

static void audio_timer_callback( unsigned long nr )
{
	h3600_asic_audio_mute( !(H3800_ASIC1_GPIO_OUT & GPIO1_AUD_PWR_ON) );
}

static struct timer_list g_audio_timer = { function: audio_timer_callback };

static void h3600_asic_ear_in_isr( int irq, void *dev_id, struct pt_regs *regs )
{
	mod_timer( &g_audio_timer, jiffies + (2 * HZ) / 1000 );
}

static int h3600_asic_audio_suspend( void )
{
	DEBUG_INIT();
	del_timer_sync(&g_audio_timer);
	disable_irq( IRQ_H3800_EAR_IN );
	return 0;
}

static void h3600_asic_audio_resume( void )
{
	DEBUG_INIT();
	H3800_ASIC2_GPIINTTYPE |= GPIO2_EAR_IN_N;   /* Set edge-type interrupt */
	h3600_asic_audio_fix_jack(1);
	enable_irq( IRQ_H3800_EAR_IN );
}

static int h3600_asic_audio_init( void )
{
	int result;

	DEBUG_INIT();
	init_timer(&g_audio_timer);
	H3800_ASIC2_GPIINTTYPE |= GPIO2_EAR_IN_N;   /* Set edge-type interrupt */
	h3600_asic_audio_fix_jack(1);

	result = request_irq(IRQ_H3800_EAR_IN, h3600_asic_ear_in_isr, 
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_ear_in", NULL );

	if ( result )
		printk(KERN_CRIT __FUNCTION__ ": unable to grab EAR_IN virtual IRQ\n");
	return result;
}

static void h3600_asic_audio_cleanup( void )
{
	DEBUG_INIT();
	del_timer_sync(&g_audio_timer);
	free_irq( IRQ_H3800_EAR_IN, NULL );
}


/***********************************************************************************
 *   Assets
 ***********************************************************************************/

struct mtd_info *g_asset_mtd;

/*
#define DUMP_BUF_SIZE   16
#define MYPRINTABLE(x) \
    ( ((x)<='z' && (x)>='a') || ((x)<='Z' && (x)>='A') || isdigit(x))

static void dump_mem( struct mtd_info *mtd, loff_t from, size_t count )
{
	size_t retlen;
	u_char buf[DUMP_BUF_SIZE];
	u_char pbuf[DUMP_BUF_SIZE + 1];

	int    ret;
	size_t len;
	int    i;

	printk(__FUNCTION__ ": dumping 0x%08X, len %dd\n", from, count);
	while ( count ) {
		len = count;
		if ( len > DUMP_BUF_SIZE )
			len = DUMP_BUF_SIZE;

		ret = mtd->read( mtd, from, len, &retlen, buf );
		if (!ret) {
			for ( i = 0 ; i < retlen ; i++ ) {
				printk(" %02x", buf[i]);
				pbuf[i] = ( MYPRINTABLE(buf[i]) ? buf[i] : '.' );
				from++;
				count--;
				if ( (from % DUMP_BUF_SIZE) == 0 ) {
					pbuf[i+1] = '\0';
					printk(" %s\n", pbuf);
				}
			}
		}
		else {
			printk(__FUNCTION__ ": error %d reading at %d, count=%d", ret,from,count);
			return;
		}
	}

}
*/

static int copytchar( struct mtd_info *mtd, unsigned char *dest, loff_t from, size_t count )
{
	size_t retlen;

	if (0) printk(__FUNCTION__ ": %p %lld %u\n", dest, from, count );

	while ( count ) {
		int ret = mtd->read( mtd, from, count, &retlen, dest );
		if (!ret) {
			count -= retlen;
			dest += retlen;
			from += retlen;
		}
		else {
			printk(__FUNCTION__ ": error %d reading at %lld, count=%d\n", ret, from, count );
			return ret;
		}
	}
	return 0;
}

static int copyword( struct mtd_info *mtd, unsigned short *dest, loff_t from )
{
	unsigned char buf[2];
	int ret = copytchar( mtd, buf, from, 2 );
	if ( ret ) 
		return ret;
	*dest = (((unsigned short) buf[1]) << 8 | (unsigned short) buf[0]);
	return 0;
}

#define COPYTCHAR(_p,_offset,_len) \
        ( g_asset_mtd ? \
        copytchar( g_asset_mtd, _p, _offset + 0x30000, _len ) \
        : 0)

#define COPYWORD(_w,_offset) \
        ( g_asset_mtd ? \
        copyword( g_asset_mtd, &_w, _offset + 0x30000 ) \
          : 0)

static int h3600_asic_asset_read( struct h3600_asset *asset )
{
	int retval = -1;

	if (0) printk(__FUNCTION__ "\n");

	switch (asset->type) {
	case ASSET_HM_VERSION:
		retval = COPYTCHAR( asset->a.tchar, 0, 10 );
		break;
	case ASSET_SERIAL_NUMBER:
		retval = COPYTCHAR( asset->a.tchar, 10, 40 );
		break;
	case ASSET_MODULE_ID:
		retval = COPYTCHAR( asset->a.tchar, 152, 20 );
		break;
	case ASSET_PRODUCT_REVISION:
		retval = COPYTCHAR( asset->a.tchar, 182, 10 );
		break;
	case ASSET_PRODUCT_ID:
		retval = COPYWORD( asset->a.vshort, 110 );
		break;
	case ASSET_FRAME_RATE:
		retval = COPYWORD( asset->a.vshort, 966 );
		break;
	case ASSET_PAGE_MODE:    
		retval = COPYWORD( asset->a.vshort, 976 );
		break;
	case ASSET_COUNTRY_ID:
		retval = COPYWORD( asset->a.vshort, 980 );
		break;
	case ASSET_IS_COLOR_DISPLAY:
		retval = COPYWORD( asset->a.vshort, 292 );
		break;
	case ASSET_ROM_SIZE:
		retval = COPYWORD( asset->a.vshort, 1514 );
		break;
	case ASSET_RAM_SIZE:
		asset->a.vshort = 0;
		retval = 0;
		break;
	case ASSET_HORIZONTAL_PIXELS:
		retval = COPYWORD( asset->a.vshort, 276 );
		break;
	case ASSET_VERTICAL_PIXELS:
		retval = COPYWORD( asset->a.vshort, 278 );
		break;
	}

	return retval;
}


static int h3600_asic_asset_init( void )
{
	int i;
	struct mtd_info *mtd;

	for ( i = 0 ; i < MAX_MTD_DEVICES ; i++ ) {
		if ( (mtd = get_mtd_device( NULL, i )) != NULL ) {
			if ( !strcmp(mtd->name, "asset" )) {
				g_asset_mtd = mtd;
				return 0;
			}
			else {
				put_mtd_device(mtd);
			}
		}
	}
	return 0;
}

static void h3600_asic_asset_cleanup( void )
{
	if ( g_asset_mtd )
		put_mtd_device(g_asset_mtd);
}


/***********************************************************************************
 *   Bluetooth
 *
 *   Resources used:    
 *   Shared resources:  
 *
 ***********************************************************************************/

struct h3600_asic_bluetooth {
	int line;       // Serial port line
	unsigned long shared;
};

static struct h3600_asic_bluetooth g_bluedev;

static void h3600_asic_bluetooth_up( struct h3600_asic_bluetooth *dev )
{
	H3800_ASIC2_INTR_MaskAndFlag &= ~ASIC2_INTMASK_UART_0;

	H3800_ASIC2_UART_0_RSR = 1;   // Reset the UART
	H3800_ASIC2_UART_0_RSR = 0;

	H3800_ASIC2_CLOCK_Enable = 
		(H3800_ASIC2_CLOCK_Enable & ~ASIC2_CLOCK_UART_0_MASK) | ASIC2_CLOCK_SLOW_UART_0;
	h3600_asic_shared_add( &dev->shared, ASIC_SHARED_CLOCK_EX1 );

	// H3600.c has already taken care of sending the 3.6864 MHz clock to the UART (TUCR) 
	// but we're paranoid...
	GPDR |= GPIO_H3800_CLK_OUT;
	GAFR |= GPIO_H3800_CLK_OUT;

	TUCR = TUCR_3_6864MHzA;

	// Set up the TXD & RXD of the UART to normal operation
	H3800_ASIC2_GPIODIR &= ~GPIO2_UART0_TXD_ENABLE;
	H3800_ASIC2_GPIOPIOD |= GPIO2_UART0_TXD_ENABLE;

	// More black magic
	H3800_ASIC2_KPIODIR  &= ~KPIO_UART_MAGIC;
	H3800_ASIC2_KPIOPIOD |=  KPIO_UART_MAGIC;
}

static void h3600_asic_bluetooth_down( struct h3600_asic_bluetooth *dev )
{
	H3800_ASIC2_INTR_MaskAndFlag   |= ASIC2_INTMASK_UART_0;
	h3600_asic_shared_release( &dev->shared, ASIC_SHARED_CLOCK_EX1 );
}

static int h3600_asic_bluetooth_suspend( void )
{
	h3600_asic_bluetooth_down( &g_bluedev );
	return 0;
}

static void h3600_asic_bluetooth_resume( void )
{
	h3600_asic_bluetooth_up( &g_bluedev );
}

static void serial_fn_pm( struct uart_port *port, u_int state, u_int oldstate )
{
	if (0) printk(__FUNCTION__ ": %p %d %d\n", port, state, oldstate);
}

static int serial_fn_open( struct uart_port *port, struct uart_info *info )
{
	if (0) printk(__FUNCTION__ ": %p %p\n", port, info);
	MOD_INC_USE_COUNT;

	H3800_ASIC1_GPIO_OUT |= GPIO1_BT_PWR_ON;
	h3600_asic_led_blink(BLUE_LED,1,7,1);
	return 0;
}

static void serial_fn_close( struct uart_port *port, struct uart_info *info )
{
	if (0) printk(__FUNCTION__ ": %p %p\n", port, info);
	h3600_asic_led_off(BLUE_LED);
	H3800_ASIC1_GPIO_OUT &= ~GPIO1_BT_PWR_ON;

	MOD_DEC_USE_COUNT;
}

static struct serial_h3800_fns serial_fns = {
	pm    : serial_fn_pm,
	open  : serial_fn_open,
	close : serial_fn_close
};

static int h3600_asic_bluetooth_init( void )
{
	h3600_asic_bluetooth_up( &g_bluedev );

	register_serial_h3800_fns(&serial_fns);
	g_bluedev.line = register_serial_h3800(0);
	if ( g_bluedev.line < 0 ) {
		printk(__FUNCTION__ ": register serial failed\n");
		return g_bluedev.line;
	} 

	return 0;
}

static void h3600_asic_bluetooth_cleanup( void )
{
	printk(__FUNCTION__ " %d\n", g_bluedev.line);
	unregister_serial_h3800( g_bluedev.line );
	register_serial_h3800_fns(NULL);
	h3600_asic_bluetooth_down( &g_bluedev );
}

/***********************************************************************************
 *      Generic IRQ handling
 ***********************************************************************************/

static int __init h3600_asic_init_isr( void )
{
	int result;
	DEBUG_INIT();

	/* Request IRQs */
        set_GPIO_IRQ_edge( GPIO_H3600_NPOWER_BUTTON, GPIO_BOTH_EDGES );
	result = request_irq(IRQ_GPIO_H3600_NPOWER_BUTTON, 
			     h3600_asic_power_isr, 
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3600_suspend", NULL);

	if ( result ) {
		printk(KERN_CRIT __FUNCTION__ ": unable to grab power button IRQ\n");
		return result;
	}

	set_GPIO_IRQ_edge( GPIO_H3800_AC_IN, GPIO_BOTH_EDGES );
	result = request_irq(IRQ_GPIO_H3800_AC_IN,
			     h3600_asic_ac_in_isr,
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_ac_in", NULL );
	if ( result ) {
		printk(KERN_CRIT __FUNCTION__ ": unable to grab AC in IRQ\n");
		free_irq(IRQ_GPIO_H3600_NPOWER_BUTTON, NULL );
		return result;
	}

	set_GPIO_IRQ_edge( GPIO_H3800_NOPT_IND, GPIO_BOTH_EDGES );
	result = request_irq(IRQ_GPIO_H3800_NOPT_IND,
			     h3600_asic_sleeve_isr,
			     SA_INTERRUPT | SA_SAMPLE_RANDOM,
			     "h3800_sleeve", NULL );
	if ( result ) {
		printk(KERN_CRIT __FUNCTION__ ": unable to grab sleeve insertion IRQ\n");
		free_irq(IRQ_GPIO_H3600_NPOWER_BUTTON, NULL );
		free_irq(IRQ_GPIO_H3800_AC_IN, NULL );
		return result;
	}

	return 0;
}

static void __exit h3600_asic_release_isr( void )
{
	DEBUG_INIT();

//	H3800_ASIC2_KPIINTSTAT  = 0;    
//	H3800_ASIC2_GPIINTSTAT  = 0;

	free_irq(IRQ_GPIO_H3600_NPOWER_BUTTON, NULL);
	free_irq(IRQ_GPIO_H3800_AC_IN, NULL);
	free_irq(IRQ_GPIO_H3800_NOPT_IND, NULL);
}


/***********************************************************************************/
/*      Standard entry points for HAL requests                                     */
/***********************************************************************************/

static int h3600_asic_version( struct h3600_ts_version *result )
{
	return -EINVAL;
}

static int h3600_asic_eeprom_read( unsigned short address, unsigned char *data, unsigned short len )
{
	if (0) printk(__FUNCTION__ ": address=%d len=%d\n", address, len);
        return -EINVAL;
}

static int h3600_asic_eeprom_write( unsigned short address, unsigned char *data, unsigned short len )
{
        return -EINVAL;
}

static int h3600_asic_thermal_sensor( unsigned short *result )
{
	return -EINVAL;
}

static int h3600_asic_notify_led( unsigned char mode, unsigned char duration, 
			    unsigned char ontime, unsigned char offtime )
{
	return -EINVAL;
}

static int h3600_asic_spi_write(unsigned short address, unsigned char *data, unsigned short len)
{
	return -EINVAL;
}

static int h3600_asic_read_light_sensor( unsigned char *result )
{
	int value = h3600_asic_adc_read_channel( ASIC2_ADMUX_0_LIGHTSENSOR );
	if ( value >= 0 ) {
		*result = value >> 2;
		return 0;
	}
		
	return -EIO;
}

static int h3600_asic_option_pack( int *result )
{
        int opt_ndet = (GPLR & GPIO_H3800_NOPT_IND);

	if (0) printk(__FUNCTION__ ": opt_ndet=%d\n", opt_ndet);
	*result = opt_ndet ? 0 : 1;
	return 0;
}

/***********************************************************************************
 *   Proc filesystem interface
 ***********************************************************************************/

static int h3600_asic_reset_handler(ctl_table *ctl, int write, struct file * filp,
			     void *buffer, size_t *lenp);

static struct ctl_table h3600_asic_table[] =
{
	{ 1, "owm_reset_delay", &owm_reset_delay, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 2, "owm_sample_delay", &owm_sample_delay, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 3, "battery_sample_interval", &battery_sample_interval, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 4, "sleep_battery_interval", &sleep_battery_interval, sizeof(int),
	  0666, NULL, &proc_dointvec },
	{ 5, "reset", NULL, 0, 0600, NULL, (proc_handler *) &h3600_asic_reset_handler },
	{0}
};

static struct ctl_table h3600_asic_dir_table[] =
{
	{1, "asic", NULL, 0, 0555, h3600_asic_table},
	{0}
};

static struct ctl_table_header *h3600_asic_sysctl_header = NULL;

static struct proc_dir_entry   *asic_proc_dir;
static struct proc_dir_entry   *reg_proc_dir;
static struct proc_dir_entry   *adc_proc_dir;

#define PRINT_DATA(x,s) \
	p += sprintf (p, "%-28s : %d\n", s, g_statistics.x)

static int h3600_asic_proc_stats_read(char *page, char **start, off_t off,
				      int count, int *eof, void *data)
{
	char *p = page;
	int len;
	int i;

	PRINT_DATA(spi_bytes,   "SPI bytes sent/received");
	PRINT_DATA(spi_wip,     "SPI write-in-process delays");
	PRINT_DATA(spi_timeout, "SPI timeouts");
	PRINT_DATA(owm_timeout, "OWM timeouts");
	PRINT_DATA(owm_reset,   "OWM reset");
	PRINT_DATA(owm_written, "OWM written");
	PRINT_DATA(owm_read,    "OWM read");
	p += sprintf(p,         "OWM ISR received   Valid Invalid Post\n");
	for ( i = 0 ; i < 5 ; i++ )
		p += sprintf(p, "     %10s : %4d   %4d   %4d\n", 
			     owm_state_names[i],
			     g_statistics.owm_valid_isr[i],
			     g_statistics.owm_invalid_isr[i],
			     g_statistics.owm_post_isr[i]);

	p += sprintf(p, "%-28s : %d\n", "EX1 usage", g_shared.clock_ex1 );
	p += sprintf(p, "%-28s : %d\n", "EX2 usage", g_shared.clock_ex2 );

	len = (p - page) - off;
	if (len < 0)
		len = 0;

	*eof = (len <= count) ? 1 : 0;
	*start = page + off;

	return len;
}

static int h3600_asic_proc_adc_read(char *page, char **start, off_t off,
				    int count, int *eof, void *data )
{
	int result;
	char *p = page;

	MOD_INC_USE_COUNT;
	result = h3600_asic_adc_read_channel((int) data);
	if ( result < 0 )
		p += sprintf(p, "Error code %d\n", result );
	else
		p += sprintf(p, "0x%04x\n",(unsigned int) result );

	*eof = 1;
	MOD_DEC_USE_COUNT;
	return (p - page);
}

static int h3600_asic_proc_battery_read(char *page, char **start, off_t off,
					int count, int *eof, void *data)
{
	char *p = page;
	int result;
	unsigned char buf[64];
	int i;

	MOD_INC_USE_COUNT;    // Necessary because we can sleep

	result = h3600_asic_owm_read_bytes( NULL, 0, buf, 64 );
	if ( result < 0 || !buf[1] ) {
		p += sprintf(p, "Unable to read battery (%d)\n", result);
	}
	else {
		int v;
		unsigned char chem, percent, flag;
		for ( i = 0 ; i < 64 ; i++ ) {
			p += sprintf(p, " %02x", buf[i]);
			if ( ((i+1) % 8 ) == 0 )
				*p++ = '\n';
		}
		p += sprintf(p, "Protection          : 0x%02x\n", buf[0] );
		p += sprintf(p, "Status              : 0x%02x\n", buf[1] );
		p += sprintf(p, "EEPROM              : 0x%02x\n", buf[7] );
		p += sprintf(p, "Special feature     : 0x%02x\n", buf[8] );
		p += sprintf(p, "Voltage             : 0x%02x%02x", buf[12], buf[13] );
		v = ((char) buf[12]) << 8 | buf[13];
		v = (v >> 5) * 5000 / 1024;
		p += sprintf(p, " (%d mV)\n", v);
		p += sprintf(p, "Current             : 0x%02x%02x\n", buf[14], buf[15] );
		p += sprintf(p, "Accumulated current : 0x%02x%02x\n", buf[16], buf[17] );
		p += sprintf(p, "Temperature         : 0x%02x%02x\n", buf[24], buf[25] );
		p += sprintf(p, "Cell chemistry      : 0x%02x\n", buf[48] >> 6 );
		p += sprintf(p, "Manufacturer        : 0x%02x\n", (buf[48] & 0x3c) >> 2 );
		p += sprintf(p, "Rated cell capacity : %d\n", buf[51] * 10 );
		p += sprintf(p, "Current offset      : 0x%02x\n", buf[51] );
		p += sprintf(p, "Sense resistor      : 0x%02x\n", buf[52] );

		if ( !h3600_asic_spi_read_pcmcia_battery( &chem, &percent, &flag ) ) {
			p += sprintf(p, "Option jacket\n");
			p += sprintf(p, "          chemistry : 0x%02x\n", chem );
			p += sprintf(p, "         percentage : 0x%02x (%d)\n", percent, percent );
			p += sprintf(p, "               flag : 0x%02x\n", flag );
		}
	}

	*eof = 1;
	MOD_DEC_USE_COUNT;
	return (p - page);
}

/* Coded lifted from "registers.c" */

static ssize_t proc_read_reg(struct file * file, char * buf,
		size_t nbytes, loff_t *ppos);
static ssize_t proc_write_reg(struct file * file, const char * buffer,
		size_t count, loff_t *ppos);

static struct file_operations proc_reg_operations = {
	read:	proc_read_reg,
	write:	proc_write_reg
};

typedef struct asic_reg_entry {
	u32   phyaddr;
	char* name;
	char* description;
	unsigned short low_ino;
} asic_reg_entry_t;

static asic_reg_entry_t asic_regs[] =
{
/*	{ virt_addr,   name,     description } */
	{ 0x0000, "GPIODIR",    "GPIO Input/Output direction register" },
	{ 0x0004, "GPIINTTYPE", "GPI Interrupt Type (Edge/Level)"},
	{ 0x0008, "GPIINTESEL", "GPI Interrupt active edge select"},
	{ 0x000c, "GPIINTALSEL","GPI Interrupt active level select"},
	{ 0x0010, "GPIINTFLAG", "GPI Interrupt active flag" },
	{ 0x0014, "GPIOPIOD",   "GPIO Port input/output data" },
	{ 0x0018, "GPOBFSTAT",  "GPO output data in batt_fault" },
	{ 0x001c, "GPIINTSTAT", "GPI Interrupt status" },
	{ 0x003c, "GPIOALT",    "GPIO ALTernate function" },

	{ 0x0200, "KPIODIR",    "KPIO Input/output direction"},
	{ 0x0204, "KPIINTTYP",  "KPI interrupt type (edge/level)" },
	{ 0x0208, "KPIINTESEL", "KPI Interrupt active edge select" },
	{ 0x020c, "KPIINTALSEL","KPI Interrupt active level select" },
	{ 0x0210, "KPIINTFLAG", "KPI Interrupt active flag" },
	{ 0x0214, "KPIOPIOD",   "KPIO Port input/output data" },
	{ 0x0218, "KPOBFSTAT",  "KPO Ouput data in batt_fault status" },
	{ 0x021c, "KPIINTSTAT", "KPI Interrupt status" },
	{ 0x023c, "KALT",       "KIU alternate function" },

	{ 0x0400, "SPICR",      "SPI control register" },
	{ 0x0404, "SPIDR",      "SPI data register" },
	{ 0x0408, "SPIDCS",     "SPI chip select disabled register" },

	{ 0x0600, "PWM0_TBS",   "PWM Time base set register" },
	{ 0x0604, "PWM0_PTS",   "PWM Period time set register" },
	{ 0x0608, "PWM0_DTS",   "PWM Duty time set register" },

	{ 0x0700, "PWM1_TBS",   "PWM Time base set register" },
	{ 0x0704, "PWM1_PTS",   "PWM Period time set register" },
	{ 0x0708, "PWM1_DTS",   "PWM Duty time set register" },

	{ 0x0800, "LED0_TBS",   "LED time base set register" },
	{ 0x0804, "LED0_PTS",   "LED period time set register" },
	{ 0x0808, "LED0_DTS",   "LED duty time set register" },
	{ 0x080c, "LED0_ASTC",  "LED auto stop counter register" },

	{ 0x0880, "LED1_TBS",   "LED time base set register" },
	{ 0x0884, "LED1_PTS",   "LED period time set register" },
	{ 0x0888, "LED1_DTS",   "LED duty time set register" },
	{ 0x088c, "LED1_ASTC",  "LED auto stop counter register" },

	{ 0x0900, "LED2_TBS",   "LED time base set register" },
	{ 0x0904, "LED2_PTS",   "LED period time set register" },
	{ 0x0908, "LED2_DTS",   "LED duty time set register" },
	{ 0x090c, "LED2_ASTC",  "LED auto stop counter register" },

	{ 0x0a00, "UART0_BUF",  "Receive/transmit buffer"},
	{ 0x0a04, "UART0_IER",  "Interrupt enable" },
	{ 0x0a08, "UART0_IIR",  "Interrupt identify" },
	{ 0x0a08, "UART0_FCR",  "Fifo control" },
	{ 0x0a0c, "UART0_LCR",  "Line control" },
	{ 0x0a10, "UART0_MCR",  "Modem control" },
	{ 0x0a14, "UART0_LSR",  "Line status" },
	{ 0x0a18, "UART0_MSR",  "Modem status" },
	{ 0x0a1c, "UART0_SCR",  "Scratch pad" },

	{ 0x0c00, "UART1_BUF",  "Receive/transmit buffer"},
	{ 0x0c04, "UART1_IER",  "Interrupt enable" },
	{ 0x0c08, "UART1_IIR",  "Interrupt identify" },
	{ 0x0c08, "UART1_FCR",  "Fifo control" },
	{ 0x0c0c, "UART1_LCR",  "Line control" },
	{ 0x0c10, "UART1_MCR",  "Modem control" },
	{ 0x0c14, "UART1_LSR",  "Line status" },
	{ 0x0c18, "UART1_MSR",  "Modem status" },
	{ 0x0c1c, "UART1_SCR",  "Scratch pad" },

	{ 0x0e00, "TIMER_0",    "Timer counter 0 register" },
	{ 0x0e04, "TIMER_1",    "Timer counter 1 register" },
	{ 0x0e08, "TIMER_2",    "Timer counter 2 register" },
	{ 0x0e0a, "TIMER_CNTL", "Timer control register (write only)" },
	{ 0x0e10, "TIMER_CMD",  "Timer command register" },

	{ 0x1000, "CDEX",       "Crystal source, control clock" },

	{ 0x1200, "ADMUX",      "ADC multiplixer select register" },
	{ 0x1204, "ADCSR",      "ADC control and status register" },
	{ 0x1208, "ADCDR",      "ADC data register" },
	
	{ 0x1600, "INTMASK",    "Interrupt mask control & cold boot flag" },
	{ 0x1604, "INTCPS",     "Interrupt timer clock pre-scale" },
	{ 0x1608, "INTTBS",     "Interrupt timer set" },

	{ 0x1800, "OWM_CMD",    "OWM command register" },
	{ 0x1804, "OWM_DATA",   "OWM transmit/receive buffer" },
	{ 0x1808, "OWM_INT",    "OWM interrupt register" },
	{ 0x180c, "OWM_INTEN",  "OWM interrupt enable register" },
	{ 0x1810, "OWM_CLKDIV", "OWM clock divisor register" },

	{ 0x1a00, "SSETR",      "Size of flash memory setting register" },

	{ 0x1c00, "MMC_STR_STP_CLK", "MMC Start/Stop Clock" },
	{ 0x1c04, "MMC_STATUS", "MMC Status" },
	{ 0x1c08, "MMC_CLK_RATE", "MMC Clock rate" },

	{ 0x1e60, "GPIO1_MASK",           "GPIO1 mask" },
	{ 0x1e64, "GPIO1_DIR",            "GPIO1 direction" },
	{ 0x1e68, "GPIO1_OUT",            "GPIO1 output level" },
	{ 0x1e6c, "GPIO1_LEVELTRI",       "GPIO1 level trigger (0=edge trigger)" },
	{ 0x1e70, "GPIO1_RISING",         "GPIO1 rising edge trigger" },
	{ 0x1e74, "GPIO1_LEVEL",          "GPIO1 high level trigger" },
	{ 0x1e78, "GPIO1_LEVEL_STATUS",   "GPIO1 level trigger sttaus" },
	{ 0x1e7c, "GPIO1_EDGE_STATUS",    "GPIO1 edge trigger status" },
	{ 0x1e80, "GPIO1_STATE",          "GPIO1 state (read only)" },
	{ 0x1e84, "GPIO1_RESET",          "GPIO1 reset" },
	{ 0x1e88, "GPIO1_SLEEP_MASK",     "GPIO1 sleep mask trigger" },
	{ 0x1e8c, "GPIO1_SLEEP_DIR",      "GPIO1 sleep direction" },
	{ 0x1e90, "GPIO1_SLEEP_OUT",      "GPIO1 sleep level" },
	{ 0x1e94, "GPIO1_STATUS",         "GPIO1 status (read only)" },
	{ 0x1e98, "GPIO1_BATT_FAULT_DIR", "GPIO1 battery fault direction" },
	{ 0x1e9c, "GPIO1_BATT_FAULT_OUT", "GPIO1 battery fault level" },

	{ 0x1f00, "FLASHWP",    "Flash write protect" },
};

#define NUM_OF_ASIC_REG_ENTRY	(sizeof(asic_regs)/sizeof(asic_reg_entry_t))

static int proc_read_reg(struct file * file, char * buf,
		size_t nbytes, loff_t *ppos)
{
	int i_ino = (file->f_dentry->d_inode)->i_ino;
	char outputbuf[15];
	int count;
	int i;
	asic_reg_entry_t* current_reg=NULL;
	if (*ppos>0) /* Assume reading completed in previous read*/
		return 0;
	for (i=0;i<NUM_OF_ASIC_REG_ENTRY;i++) {
		if (asic_regs[i].low_ino==i_ino) {
			current_reg = &asic_regs[i];
			break;
		}
	}
	if (current_reg==NULL)
		return -EINVAL;

	count = sprintf(outputbuf, "0x%08X\n", *((volatile u32 *)(_H3800_ASIC2_Base + current_reg->phyaddr)));
	*ppos+=count;
	if (count>nbytes)  /* Assume output can be read at one time */
		return -EINVAL;
	if (copy_to_user(buf, outputbuf, count))
		return -EFAULT;
	return count;
}

static ssize_t proc_write_reg(struct file * file, const char * buffer,
		size_t count, loff_t *ppos)
{
	int i_ino = (file->f_dentry->d_inode)->i_ino;
	asic_reg_entry_t* current_reg=NULL;
	int i;
	unsigned long newRegValue;
	char *endp;

	for (i=0;i<NUM_OF_ASIC_REG_ENTRY;i++) {
		if (asic_regs[i].low_ino==i_ino) {
			current_reg = &asic_regs[i];
			break;
		}
	}
	if (current_reg==NULL)
		return -EINVAL;

	newRegValue = simple_strtoul(buffer,&endp,0);
	*((volatile u32 *)(_H3800_ASIC2_Base + current_reg->phyaddr))=newRegValue;
	return (count+endp-buffer);
}

static int h3600_asic_proc_wakeup_read(char *page, char **start, off_t off,
				       int count, int *eof, void *data );

static int __init h3600_asic_register_procfs( void )
{
	int i;

	asic_proc_dir = proc_mkdir(H3600_ASIC_PROC_DIR, NULL);
	if ( !asic_proc_dir ) {
		printk(KERN_ALERT __FUNCTION__ 
		       ": unable to create proc entry " H3600_ASIC_PROC_DIR "\n");
		return -ENOMEM;
	}

	create_proc_read_entry(H3600_ASIC_PROC_STATS, 0, asic_proc_dir, 
			       h3600_asic_proc_stats_read, NULL );

	create_proc_read_entry("battery", 0, asic_proc_dir, 
			       h3600_asic_proc_battery_read, NULL );

	create_proc_read_entry("wakeup", 0, asic_proc_dir, 
			       h3600_asic_proc_wakeup_read, NULL );

	adc_proc_dir = proc_mkdir("adc", asic_proc_dir);
	if (adc_proc_dir == NULL) {
		printk(KERN_ERR __FUNCTION__ ": can't create /proc/adc\n");
		return(-ENOMEM);
	}
	for (i=0;i<5;i++) {
		unsigned char name[10];
		sprintf(name,"%d",i);
		create_proc_read_entry(name, S_IWUSR |S_IRUSR | S_IRGRP | S_IROTH, 
					       adc_proc_dir, h3600_asic_proc_adc_read, (void *) i );
	}

	reg_proc_dir = proc_mkdir(REG_DIRNAME, asic_proc_dir);
	if (reg_proc_dir == NULL) {
		printk(KERN_ERR __FUNCTION__ ": can't create /proc/" REG_DIRNAME "\n");
		return(-ENOMEM);
	}

	for(i=0;i<NUM_OF_ASIC_REG_ENTRY;i++) {
		struct proc_dir_entry *entry = create_proc_entry(asic_regs[i].name,
								 S_IWUSR |S_IRUSR | S_IRGRP | S_IROTH,
								 reg_proc_dir);
		
		if ( !entry ) {
			printk( KERN_ERR __FUNCTION__ ": can't create /proc/" REG_DIRNAME
				"/%s\n", asic_regs[i].name);
			return(-ENOMEM);
		}

		asic_regs[i].low_ino = entry->low_ino;
		entry->proc_fops = &proc_reg_operations;
	}

	h3600_asic_sysctl_header = register_sysctl_table(h3600_asic_dir_table, 0 );
	return 0;
}

static void __exit h3600_asic_unregister_procfs( void )
{
	int i;
	if ( asic_proc_dir ) {
		unregister_sysctl_table(h3600_asic_sysctl_header);
		for(i=0;i<NUM_OF_ASIC_REG_ENTRY;i++)
			remove_proc_entry(asic_regs[i].name,reg_proc_dir);
		remove_proc_entry(REG_DIRNAME, asic_proc_dir);
		for(i=0;i<5;i++) {
			char name[10];
			sprintf(name,"%d",i);
			remove_proc_entry(name,adc_proc_dir);
		}
		remove_proc_entry("adc",asic_proc_dir);
		remove_proc_entry("battery", asic_proc_dir);
		remove_proc_entry("wakeup", asic_proc_dir);
		remove_proc_entry(H3600_ASIC_PROC_STATS, asic_proc_dir);
		remove_proc_entry(H3600_ASIC_PROC_DIR, NULL );
		asic_proc_dir = NULL;
	}
}

/***********************************************************************************
 *   Utility routines
 ***********************************************************************************/

struct asic_system_handler { 
	int  (*init)( void );
	void (*cleanup)( void );
	int  (*suspend)( void );
	void (*resume)( void );
};

/* 
   We initialize and resume from top to bottom,
   cleanup and suspend from bottom to top 
*/
   
const struct asic_system_handler asic_system_handlers[] = {
	{ 
		init:    h3600_asic_shared_init,      
		cleanup: h3600_asic_shared_cleanup 
	},{ 
		init:    h3600_asic_adc_init,
		cleanup: h3600_asic_adc_cleanup,
		suspend: h3600_asic_adc_suspend,
		resume:  h3600_asic_adc_resume,
	},{ 
		init:    h3600_asic_key_init,
		cleanup: h3600_asic_key_cleanup,
		suspend: h3600_asic_key_suspend,
		resume:  h3600_asic_key_resume 
	},{ 
		init:    h3600_asic_spi_init,
		cleanup: h3600_asic_spi_cleanup,
		suspend: h3600_asic_spi_suspend,
		resume:  h3600_asic_spi_resume 
	},{ 
		init:    h3600_asic_backlight_init,   
		cleanup: h3600_asic_backlight_cleanup 
	},{ 
		init:    h3600_asic_touchscreen_init, 
		cleanup: h3600_asic_touchscreen_cleanup,
		suspend: h3600_asic_touchscreen_suspend, 
		resume:  h3600_asic_touchscreen_resume 
	},{ 
		init:    h3600_asic_owm_init, 
		cleanup: h3600_asic_owm_cleanup,
		suspend: h3600_asic_owm_suspend, 
		resume:  h3600_asic_owm_resume 
	},{ 
		init:    h3600_asic_battery_init, 
		cleanup: h3600_asic_battery_cleanup,
		suspend: h3600_asic_battery_suspend, 
		resume:  h3600_asic_battery_resume 
	},{ 
		init:    h3600_asic_audio_init, 
		cleanup: h3600_asic_audio_cleanup,
		suspend: h3600_asic_audio_suspend, 
		resume:  h3600_asic_audio_resume  
	},{ 
		init:    h3600_asic_asset_init,   
		cleanup: h3600_asic_asset_cleanup 
	},{
		init:    h3600_asic_bluetooth_init,
		cleanup: h3600_asic_bluetooth_cleanup,
		suspend: h3600_asic_bluetooth_suspend,
		resume:  h3600_asic_bluetooth_resume
	}
};

#define SYS_HANDLER_SIZE   (sizeof(asic_system_handlers)/sizeof(struct asic_system_handler))

static int h3600_asic_suspend_handlers( void )
{
	int i;
	int result;

	for ( i = SYS_HANDLER_SIZE - 1 ; i >= 0 ; i-- ) {
		if ( asic_system_handlers[i].suspend ) {
			if ((result = asic_system_handlers[i].suspend()) != 0 ) {
				while ( ++i < SYS_HANDLER_SIZE )
					if ( asic_system_handlers[i].resume )
						asic_system_handlers[i].resume();
				return result;
			}
		}
	}
	return 0;
}

static void h3600_asic_resume_handlers( void )
{
	int i;

	for ( i = 0 ; i < SYS_HANDLER_SIZE ; i++ )
		if ( asic_system_handlers[i].resume )
			asic_system_handlers[i].resume();
}

static int __init h3600_asic_init_handlers( void )
{
	int i;
	int result;

	for ( i = 0 ; i < SYS_HANDLER_SIZE ; i++ ) {
		if ( asic_system_handlers[i].init ) {
			if ( (result = asic_system_handlers[i].init()) != 0 ) {
				while ( --i >= 0 )
					if ( asic_system_handlers[i].cleanup )
						asic_system_handlers[i].cleanup();
				return result;
			}
		}
	}
	return 0;
}

static void __exit h3600_asic_cleanup_handlers( void )
{
	int i;

	for ( i = SYS_HANDLER_SIZE - 1 ; i >= 0 ; i-- )
		if ( asic_system_handlers[i].cleanup )
			asic_system_handlers[i].cleanup();
}

/***********************************************************************************
 *   Power management
 *
 *   On sleep, if we return anything other than "0", we will cancel sleeping.
 *
 *   On resume, if we return anything other than "0", we will put the iPAQ
 *     back to sleep immediately.
 ***********************************************************************************/

enum {
	REASON_BUTTON      = 0x01,
	REASON_RTC_BATTERY = 0x02,
	REASON_RTC_OTHER   = 0x04,
	REASON_AC_IN       = 0x08,
	REASON_AC_OUT      = 0x10
};

struct wakeup_reason {
	int reason;
	int rcnr;
	int saved_rtar;
};

#define MAX_REASONS  24

struct sleep_state {
	struct wakeup_reason reason_list[MAX_REASONS];
	int count;
	int ac_power;
	u32 saved_rtar;
	enum charging_state state;
};

static struct sleep_state g_sleep;

static void h3600_asic_set_for_sleep( struct sleep_state *s )
{
	u32 rcnr = RCNR;

        s->ac_power   = (GPLR & GPIO_H3800_AC_IN) ? 0 : 1;
	s->saved_rtar = 0;

	if ( s->ac_power ) {
		H3800_ASIC1_GPIO_OUT     |= GPIO1_CH_TIMER;
		if ( s->state == CHARGING_STATE_NO_AC || s->state == CHARGING_STATE_INIT )
			s->state = CHARGING_STATE_ACTIVE;
	}
	else {
		H3800_ASIC1_GPIO_OUT     &= ~GPIO1_CH_TIMER;
		H3800_ASIC2_CLOCK_Enable &= ~ASIC2_CLOCK_ADC;
		if ( s->state == CHARGING_STATE_ACTIVE || s->state == CHARGING_STATE_FULL )
			s->state = CHARGING_STATE_NO_AC;
	}

	h3600_asic_battery_set_led( &g_batdev, s->state );

	if ( (RTSR & RTSR_ALE) && (RTAR > rcnr)) {
		s->saved_rtar = RTAR;   // Save the old interrupt time
		if ( RTAR <= rcnr + sleep_battery_interval + 2)
			return;   // The RTC interrupt occurs very soon; don't set our own
	}

	/* Set up a battery RTC alarm and run the ADC converter */
	if ( s->ac_power ) {
		RTAR  = rcnr + sleep_battery_interval;
		RTSR |= RTSR_ALE;

		H3800_ASIC2_CLOCK_Enable |= ASIC2_CLOCK_ADC;
		H3800_ASIC2_ADMUX         = ASIC2_ADMUX_1_IMIN | ASIC2_ADMUX_CLKEN;
		H3800_ASIC2_ADCSR         = ASIC2_ADCSR_ADPS(4) | ASIC2_ADCSR_ENABLE;
		H3800_ASIC2_ADCSR        |= ASIC2_ADCSR_START;
	}
}

static int h3600_asic_wakeup_reason( struct sleep_state *s )
{
	struct wakeup_reason *r;
	int reason   = 0;
        int ac_power = (GPLR & GPIO_H3800_AC_IN) ? 0 : 1;

	if ( ac_power != s->ac_power ) 
		reason |= ( ac_power ? REASON_AC_IN : REASON_AC_OUT);

	if ( ipaq_model_ops.icpr & IC_RTCAlrm ) {
		if ( s->saved_rtar == 0 || s->saved_rtar > RCNR )
			reason |= REASON_RTC_BATTERY;
		else
			reason |= REASON_RTC_OTHER;
	}

	if ( ipaq_model_ops.icpr & IC_GPIO0 )
		reason |= REASON_BUTTON;

	r = &(s->reason_list[ s->count++ % MAX_REASONS ]);
	r->reason = reason;
	r->rcnr   = RCNR;
	r->saved_rtar = s->saved_rtar;

	return reason;
}


static int h3600_asic_check_wakeup( struct sleep_state *s )
{
	int reason = h3600_asic_wakeup_reason(s);
	u32 rtsr = RTSR;

	/* 
	   Sort out the RTC alarms.  If it was a user-set alarm, we don't want
	   to touch anything.  If it was a battery alarm, we need to acknowledge it.
	   No matter what the wake up reason, we shut off the battery RTC alarm
	   and restore any pending user-set alarms.
	*/

	if ( reason & REASON_RTC_OTHER )  // Better not touch the alarm structures
		return 0;

	if ( reason & REASON_RTC_BATTERY )     // Acknowledge the interrupt
		RTSR = RTSR_AL | (rtsr & RTSR_HZE); 

	if ( s->saved_rtar > RCNR ) {    // Re-enable the user-set RTC alarm
		RTAR = s->saved_rtar;
		RTSR = RTSR_ALE | (rtsr & RTSR_HZE); // Enable ALE
	}
	else                           // Otherwise disable RTC alarms
		RTSR = rtsr & RTSR_HZE;
		
	/* 
	   The RTC alarms have been sorted out.  Decide how to handle
	   the wakeup
	*/
	if ( reason & REASON_BUTTON || !reason )
		return 0;     // Wake up the iPAQ for either a button press or an unexplained event

	if ( reason & REASON_AC_IN )
		s->state = CHARGING_STATE_ACTIVE;

	if ( reason & REASON_RTC_BATTERY )
		s->state = ( H3800_ASIC2_ADCDR < 100 ) ? CHARGING_STATE_FULL : CHARGING_STATE_ACTIVE;

	if ( reason & REASON_AC_OUT )   // Note: it's possible to AC_OUT && RTC_BATTERY at same time
		s->state = CHARGING_STATE_NO_AC;

	h3600_asic_set_for_sleep(s);
	return 1;    // Don't wake up
}

static int h3600_asic_proc_wakeup_read(char *page, char **start, off_t off,
				       int count, int *eof, void *data )
{
	char *p = page;
	int i;
	struct sleep_state *s = &g_sleep;

	for ( i = ( s->count - MAX_REASONS < 0 ? 0 : s->count - MAX_REASONS ) ; i < s->count ; i++ ) {
		struct wakeup_reason *r = &s->reason_list[ i % MAX_REASONS ];
		p += sprintf(p, "[%03d] RCNR=%08x SAVED_RTAR=%08x  %02x ", i, r->rcnr, r->saved_rtar, r->reason);
		if ( r->reason & REASON_BUTTON ) 
			p += sprintf(p, "BUTTON ");
		if ( r->reason & REASON_RTC_BATTERY )
			p += sprintf(p, "RTC_BAT ");
		if ( r->reason & REASON_RTC_OTHER )
			p += sprintf(p, "RTC_OTHER ");
		if ( r->reason & REASON_AC_IN )
			p += sprintf(p, "AC_IN ");
		if ( r->reason & REASON_AC_OUT )
			p += sprintf(p, "AC_OUT ");
		p += sprintf(p, "\n");
	}

	*eof = 1;
	return (p - page);
}



static int h3600_asic_pm_callback(pm_request_t req)
{
	int result = 0;
        if (0) printk(__FUNCTION__ ": req=%d\n", req );

	switch (req) {
	case PM_SUSPEND:
		H3800_ASIC2_INTR_MaskAndFlag &= ~ASIC2_INTMASK_GLOBAL;
		result = h3600_asic_suspend_handlers();
		H3800_ASIC2_KPIINTSTAT  = 0;    
		H3800_ASIC2_GPIINTSTAT  = 0;
		g_sleep.count = 0;
		g_sleep.state = g_batdev.state;
		h3600_asic_set_for_sleep( &g_sleep );
		break;

	case PM_RESUME:
		MSC2 = (MSC2 & 0x0000ffff) | 0xE4510000;  /* Set MSC2 correctly */
		result = h3600_asic_check_wakeup( &g_sleep );
		if ( !result ) {
			/* These probably aren't necessary */
			H3800_ASIC2_KPIINTSTAT     =  0;     /* Disable all interrupts */
			H3800_ASIC2_GPIINTSTAT     =  0;

			H3800_ASIC2_KPIINTCLR      =  0xffff;     /* Clear all KPIO interrupts */
			H3800_ASIC2_GPIINTCLR      =  0xffff;     /* Clear all GPIO interrupts */

			H3800_ASIC2_CLOCK_Enable       |= ASIC2_CLOCK_EX0;   /* 32 kHZ crystal on */
			H3800_ASIC2_INTR_ClockPrescale |= ASIC2_INTCPS_SET;
			H3800_ASIC2_INTR_ClockPrescale  = ASIC2_INTCPS_CPS(0x0e) | ASIC2_INTCPS_SET;
			H3800_ASIC2_INTR_TimerSet       = 1;

			h3600_asic_resume_handlers();
			H3800_ASIC2_INTR_MaskAndFlag  |= ASIC2_INTMASK_GLOBAL; /* Turn on ASIC interrupts */
		}
		break;
	}
	return result;
}

static int h3600_asic_reset_handler(ctl_table *ctl, int write, struct file * filp,
				    void *buffer, size_t *lenp)
{
	MOD_INC_USE_COUNT;
	h3600_asic_pm_callback( PM_SUSPEND );
	h3600_asic_pm_callback( PM_RESUME );
	MOD_DEC_USE_COUNT;

	return 0;
}


/***********************************************************************************
 *   Initialization code
 ***********************************************************************************/

static struct h3600_hal_ops h3800_asic_ops = {
	get_version         : h3600_asic_version,
	eeprom_read         : h3600_asic_eeprom_read,
	eeprom_write        : h3600_asic_eeprom_write,
	get_thermal_sensor  : h3600_asic_thermal_sensor,
	set_notify_led      : h3600_asic_notify_led,
	read_light_sensor   : h3600_asic_read_light_sensor,
	get_battery         : h3600_asic_battery_read,
	spi_read            : h3600_asic_spi_read,
	spi_write           : h3600_asic_spi_write,
	get_option_detect   : h3600_asic_option_pack,
	audio_clock         : h3600_asic_audio_clock,
	audio_power         : h3600_asic_audio_power,
	audio_mute          : h3600_asic_audio_mute,
	backlight_control   : h3600_asic_backlight_control,
	asset_read          : h3600_asic_asset_read,
        owner               : THIS_MODULE,
};

static void __exit h3600_asic_cleanup( void )
{
	h3600_unregister_pm_callback( h3600_asic_pm_callback );
	h3600_asic_unregister_procfs();
        spi_del_adapter( &h3800_asic_spi_adapter);
	h3600_hal_unregister_interface( &h3800_asic_ops );
	h3600_asic_cleanup_handlers();
	h3600_asic_release_isr();
}

int __init h3600_asic_init( void )
{
	int result;

	if ( !machine_is_h3800() ) {
		printk(__FUNCTION__ ": unknown iPAQ model %s\n", h3600_generic_name() );
		return -ENODEV;
	}

	/* We need to set nCS5 without changing nCS4 */
	/* Properly, this should be set in the bootldr */
	MSC2 = (MSC2 & 0x0000ffff) | 0xE4510000;

	result = h3600_asic_init_isr();  
	if ( result ) return result;

	result = h3600_asic_init_handlers();
	if ( result ) goto init_fail;

	result = h3600_hal_register_interface( &h3800_asic_ops );
	if ( result ) goto init_fail;

	result = h3600_asic_register_procfs();
	if ( result ) goto init_fail;

        result = spi_add_adapter( &h3800_asic_spi_adapter);
        if ( result ) goto init_fail;

	h3600_register_pm_callback( h3600_asic_pm_callback );

	H3800_ASIC2_INTR_MaskAndFlag  |= ASIC2_INTMASK_GLOBAL; /* Turn on ASIC interrupts */
	return 0;

init_fail:
	printk(__FUNCTION__ ": FAILURE!  Exiting...\n");
	h3600_asic_cleanup();
	return result;
}

void __exit h3600_asic_exit( void )
{
	h3600_asic_cleanup();
	H3800_ASIC2_INTR_MaskAndFlag &= ~ASIC2_INTMASK_GLOBAL;
}

module_init(h3600_asic_init)
module_exit(h3600_asic_exit)
