/*
 * brh_timer.c: High resolution timer support for the XScale (80200) BRH board
 *
 * Real-Time and Multimedia Systems Laboratory
 * Copyright (c) 2002 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Real-Time and Multimedia Systems Laboratory
 *  Attn: Prof. Raj Rajkumar
 *  Electrical and Computer Engineering, and Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 *  or via email to raj@ece.cmu.edu
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */
#include <rk/rk_linux.h>
#include <rk/rk.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <asm/hardware.h>
#include <asm/arch/irq.h>
#include <asm/mach/irq.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/timex.h>

#if defined (__brh__) ||  defined(CONFIG_ARCH_BRH)          

/* this defines the minimum # of cpu cycles used for the timer */
#define RK_BRH_MIN_TIMER 		8

int timer_val = 0;
cpu_tick_data_t current_ticks=0;


static void rk_brh_rdtsc_overflow(int irq, void *dev_id, struct pt_regs *regs)
{
  extern void rk_tsc_update(void);

  rk_tsc_update();
  *BRH_TIMER1_CTRL = *BRH_TIMER1_CTRL | BRH_TIMER_INTERRUPT;
}

static struct irqaction rk_brh_rdtsc_irq = {
        name: "rk_rdtsc_timer",
        handler: rk_brh_rdtsc_overflow,
        flags: SA_INTERRUPT
};



/* We will use TIMER0 to maintain RK timer and TIMER1 as a RTC */
/* Hence, we set TIMER1 to be a periodic timer with the longest period */

void rk_brh_rdtsc_init(void) {
#ifdef DEBUG_BRH
  printk("rk_brh_rdtsc_init setup TIMERB to be rtc \n");
#endif /* DEBUG_BRH */
  setup_arm_irq(IRQ_BRH_TIMERB, &rk_brh_rdtsc_irq); 
  *BRH_TIMER1_CTRL = *BRH_TIMER1_CTRL | BRH_TIMER_INTERRUPT | BRH_TIMER_CONTINOUS ;
  *BRH_TIMER1_PRELOAD = 0xffffffff;
  // Enable interrupt
  *BRH_TIMER1_CTRL = *BRH_TIMER1_CTRL | BRH_TIMER_INTERRUPT;
  *BRH_TIMER1_CTRL = *BRH_TIMER1_CTRL | BRH_TIMER_ENABLE;

}

void rk_update_hw_timer(struct rk_timer *tmr)
{
  cpu_tick_data_t now;
  unsigned long val;
  //Set match register to expiration time of timer,
  //and check if timer has already expired, and if true, then set to minimum delay
    
  *BRH_TIMER0_CTRL = *BRH_TIMER0_CTRL | BRH_TIMER_INTERRUPT;
  rk_rdtsc(&now);
  if ((tmr->tmr_expire <= now ) && (!(tmr->overflow))) {
    val = RK_BRH_MIN_TIMER;
  }
  else val = (tmr->tmr_expire-now);
  *BRH_TIMER0_PRELOAD = val;
  // Enable interrupt
  *BRH_TIMER0_CTRL = *BRH_TIMER0_CTRL | BRH_TIMER_INTERRUPT;
  *BRH_TIMER0_CTRL = *BRH_TIMER0_CTRL | BRH_TIMER_ENABLE;

}

/* Return the control to Linux, We will set the interrupt at the next jiffy */

void rk_hw_timer_release(void) {
  cpu_tick_data_t now;
  extern cpu_tick_data_t rk_cpu_tick_at_next_jiffy;
  unsigned long val;

  *BRH_TIMER0_CTRL = *BRH_TIMER0_CTRL | BRH_TIMER_INTERRUPT;
  rk_rdtsc(&now);
  if (rk_cpu_tick_at_next_jiffy <= now ) {
      /* set at the next jiffy */
      val = rk_cpu_tick_at_next_jiffy + rk_cpu_ticks_per_jiffy - now;
  }
  else val = (rk_cpu_tick_at_next_jiffy-now);
  *BRH_TIMER0_PRELOAD = val;
  // Enable interrupt
  *BRH_TIMER0_CTRL = *BRH_TIMER0_CTRL | BRH_TIMER_INTERRUPT;
  *BRH_TIMER0_CTRL = *BRH_TIMER0_CTRL | BRH_TIMER_ENABLE;
	

}



#endif /* defined (__brh__) || defined(CONFIG_ARCH_BRH)*/
