/*
 * brh_timer.c: High resolution timer support for the XScale (PXA255) Bitsyx board
 *
 * Real-Time and Multimedia Systems Laboratory
 * Copyright (c) 2002 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Real-Time and Multimedia Systems Laboratory
 *  Attn: Prof. Raj Rajkumar
 *  Electrical and Computer Engineering, and Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 *  or via email to raj@ece.cmu.edu
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */
#include <rk/rk_linux.h>
#include <rk/rk.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <asm/hardware.h>
#include <asm/arch/irq.h>
#include <asm/mach/irq.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/timex.h>

#if defined(__bitsyx__) ||  defined(CONFIG_ARCH_ADSBITSYX)          

/* this defines the minimum # of cpu cycles used for the timer */
#define RK_BITSYX_MIN_TIMER 		30

int timer_val = 0;
cpu_tick_data_t current_ticks=0;
extern int setup_arm_irq(int irq, struct irqaction * new);

#ifdef DEBUG_BITSYX
cpu_tick_data_t then;
#endif

static void rk_bitsyx_rdtsc_overflow(int irq, void *dev_id, struct pt_regs *regs)
{
  extern void rk_tsc_update(void);

/*#ifdef DEBUG_BITSYX
printk("OSSR 0x%x, OSMR0 0x%x, then 0x%llx\n", OSSR, OSMR0, then);
#endif*/
  rk_tsc_update();
  OSMR1 = 0xffffffff;
  OSSR = OSSR_M1;
}

static struct irqaction rk_bitsyx_rdtsc_irq = {
        name: "rk_rdtsc_timer",
        handler: rk_bitsyx_rdtsc_overflow,
        flags: SA_INTERRUPT
};



/* We will use TIMER0 to maintain RK timer and TIMER1 as a RTC */
/* Hence, we set TIMER1 to be a periodic timer with the longest period */

void rk_bitsyx_rdtsc_init(void) {

#ifdef DEBUG_BITSYX
  printk("rk_bitsyx_rdtsc_init setup OSMR1 to be rtc \n");
#endif

  setup_arm_irq(IRQ_OST1, &rk_bitsyx_rdtsc_irq);
  /* set match register : before overflow */
  OSMR1 = 0xffffffff;
  OSSR = OSSR_M1;

  /* Enable interrupt */
  OIER |= OIER_E1;
}

void rk_update_hw_timer(struct rk_timer *tmr)
{
  cpu_tick_data_t now;
  //Set match register to expiration time of timer,
  //and check if timer has already expired, and if true, then set to minimum delay    
//printk("OSSR %u, OIER %u", OSSR, OIER);

  rk_rdtsc(&now);
#ifdef DEBUG_BITSYX
then=now;
#endif

  if ((tmr->tmr_expire <= (now + RK_BITSYX_MIN_TIMER) ) && (!(tmr->overflow))) {
//printk("?");
    OSMR0 = OSCR + RK_BITSYX_MIN_TIMER;
  }
  else OSMR0 = (tmr->tmr_expire & 0xffffffff);

  OSSR = OSSR_M0;
}

/* Return the control to Linux, We will set the interrupt at the next jiffy */

void rk_hw_timer_release(void) {
  cpu_tick_data_t now;
  extern cpu_tick_data_t rk_cpu_tick_at_next_jiffy;

  rk_rdtsc(&now);
  if (rk_cpu_tick_at_next_jiffy <= now ) {
	/* set at the next jiffy */
	OSMR0 = OSCR + (rk_cpu_tick_at_next_jiffy + rk_cpu_ticks_per_jiffy - now);
  }
  else OSMR0 = OSCR + (rk_cpu_tick_at_next_jiffy - now);

  // Enable interrupt
  OSSR = OSSR_M0;
}

#endif /* defined (__bitsyx__) || defined(CONFIG_ARCH_BITSYX)*/
