/*
 * linux/drivers/video/epson13806fb.c
 *      -- Support for the Epson S1S13806 LCD/CRT controller
 *	(Modified by Boris Itkis)
 * based on linux/drivers/video/epson1355fb.c
 *  witch is Copyright (C) 2000 Philipp Rumpf <prumpf@tux.org>
 * based on linux/drivers/video/skeletonfb.c, which was
 *  Created 28 Dec 1997 by Geert Uytterhoeven
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */

#include <asm/io.h>
#include <linux/config.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/string.h>
#include <linux/tty.h>
#include <video/fbcon-cfb16.h>
#include <video/fbcon-cfb8.h>
#include <video/fbcon-cfb4.h>
#include <video/fbcon-mfb.h>
#include <video/fbcon.h>
#include <linux/pm.h>

// New defintion
#define E13806_RevCode			0x00
#define E13806_Misc			0x01
#define E13806_GpioCfg0			0x04
#define E13806_GpioCfg1			0x05
#define E13806_GpioCtrl0		0x08
#define E13806_GpioCtrl1		0x09
#define E13806_MdCfgStatus		0x0c
#define E13806_MemClkCfg		0x10
#define E13806_LcdPclkCfg		0x14
#define E13806_CrtPclkCfg		0x18
#define E13806_MPclkCfg			0x1c
#define E13806_CpuMemWaitSel		0x1e
#define E13806_MemCfg			0x20
#define E13806_DramRefresh		0x21
#define E13806_DramTimingCtrl0		0x2a
#define E13806_DramTimingCtrl1		0x2b
#define E13806_PanelType		0x30
#define E13806_ModRate			0x31
#define E13806_LcdHDP			0x32
#define E13806_LcdHNDP			0x34
#define E13806_TftFplineStart		0x35
#define E13806_TftFplinePulse		0x36
#define E13806_LcdVDP0			0x38
#define E13806_LcdVDP1			0x39
#define E13806_LcdVNDP     		0x3a
#define E13806_TftFpframeStart		0x3b
#define E13806_TftFpframePulse		0x3c
#define E13806_LcdLineCount0		0x3e
#define E13806_LcdLineCount1		0x3f
#define E13806_LcdDispMode		0x40
#define E13806_LcdMisc			0x41
#define E13806_LcdStart0		0x42
#define E13806_LcdStart1		0x43
#define E13806_LcdStart2		0x44
#define E13806_LcdStride0		0x46
#define E13806_LcdStride1		0x47
#define E13806_LcdPixelPan		0x48
#define E13806_LcdFifoHigh		0x4a
#define E13806_LcdFifoLow		0x4b
#define E13806_CrtHDP			0x50
#define E13806_CrtHNDP			0x52
#define E13806_CrtHRTCStart		0x53
#define E13806_CrtHRTCPulse		0x54
#define E13806_CrtVDP0			0x56
#define E13806_CrtVDP1			0x57
#define E13806_CrtVNDP			0x58
#define E13806_CrtVRTCStart		0x59
#define E13806_CrtVRTCPulse		0x5a
#define E13806_TvOutputCtrl		0x5b
#define E13806_CrtLineCount0		0x5e
#define E13806_CrtLineCount1		0x5f
#define E13806_CrtDispMode		0x60
#define E13806_CrtStart0		0x62
#define E13806_CrtStart1		0x63
#define E13806_CrtStart2		0x64
#define E13806_CrtStride0		0x66
#define E13806_CrtStride1		0x67
#define E13806_CrtPixelPan		0x68
#define E13806_CrtFifoHigh		0x6a
#define E13806_CrtFifoLow		0x6b
#define E13806_LcdInkCursCtrl		0x70
#define E13806_LcdInkCursStart		0x71
#define E13806_LcdCursorXpos0		0x72
#define E13806_LcdCursorXpos1		0x73
#define E13806_LcdCursorYpos0		0x74
#define E13806_LcdCursorYpos1		0x75
#define E13806_LcdInkCursBlue0		0x76
#define E13806_LcdInkCursGreen0		0x77
#define E13806_LcdInkCursRed0		0x78
#define E13806_LcdInkCursBlue1		0x7a
#define E13806_LcdInkCursGreen1		0x7b
#define E13806_LcdInkCursRed1		0x7c
#define E13806_LcdInkCursFifo		0x7e
#define E13806_CrtInkCursCtrl		0x80
#define E13806_CrtInkCursStart		0x81
#define E13806_CrtCursorXpos0		0x82
#define E13806_CrtCursorXpos1		0x83
#define E13806_CrtCursorYpos0		0x84
#define E13806_CrtCursorYpos1		0x85
#define E13806_CrtInkCursBlue0		0x86
#define E13806_CrtInkCursGreen0		0x87
#define E13806_CrtInkCursRed0		0x88
#define E13806_CrtInkCursBlue1		0x8a
#define E13806_CrtInkCursGreen1		0x8b
#define E13806_CrtInkCursRed1		0x8c
#define E13806_CrtInkCursFifo		0x8e
#define E13806_BltCtrl0			0x100
#define E13806_BltCtrl1			0x101
#define E13806_BltROP			0x102
#define E13806_BltOperation		0x103
#define E13806_BltSrcStart0		0x104
#define E13806_BltSrcStart1		0x105
#define E13806_BltSrcStart2		0x106
#define E13806_BltDstStart0		0x108
#define E13806_BltDstStart1		0x109
#define E13806_BltDstStart2		0x10a
#define E13806_BltStride0		0x10c
#define E13806_BltStride1		0x10d
#define E13806_BltWidth0		0x110
#define E13806_BltWidth1		0x111
#define E13806_BltHeight0		0x112
#define E13806_BltHeight1		0x113
#define E13806_BltBgColor0		0x114
#define E13806_BltBgColor1		0x115
#define E13806_BltFgColor0		0x118
#define E13806_BltFgColor1		0x119
#define E13806_LutMode			0x1e0
#define E13806_LutAddr			0x1e2
#define E13806_LutData			0x1e4
#define E13806_PwrsaveCfg		0x1f0
#define E13806_PwrsaveStatus		0x1f1
#define E13806_CpuMemWatchdog		0x1f4
#define E13806_DispMode			0x1fc
#define E13806_MediaLCMD		0x1000
#define E13806_MediaReservedLCMD	0x1002
#define E13806_MediaCMD			0x1004
#define E13806_MediaReservedCMD		0x1006
#define E13806_MediaData		0x1008

#ifdef CONFIG_E13806_REG_BASE
#define E13806_REG_BASE		CONFIG_E13806_REG_BASE
#ifdef CONFIG_E13806_FB_BASE
#define E13806_FB_BASE		CONFIG_E13806_FB_BASE
#else
#define E13806_FB_BASE		((CONFIG_E13806_REG_BASE)+0x200000)
#endif
#else
#ifdef CONFIG_ARCH_FORTUNET
#define E13806_REG_BASE		0x50000000
#define E13806_FB_BASE		0x50200000
#else
#error unknown start of s1d13806
#endif
#endif

int remaped_regs = 0;
void * screen_base = NULL;

    u16 cfb16[16];

static inline u8 e13806_read_reg(int index)
{
	if(!remaped_regs)
	{
		printk("Can not read index %03X (no remap)\n",index);
		return(0);
	}
	return __raw_readb(remaped_regs + index);
}

static inline void e13806_write_reg(u8 data, int index)
{
	if(!remaped_regs)
	{
		printk("Can not write index %03X = %02X (no remap)\n",index,data);
		return;
	}
	__raw_writeb(data, remaped_regs + index);
}

static inline u16 e13806_read_reg16(int index)
{
	return (e13806_read_reg(index) & 0xff) | ((e13806_read_reg(index+1) & 0xff) << 8);
}

static inline void e13806_write_reg16(u16 data, int index)
{
	e13806_write_reg((data&0xff), index);
	e13806_write_reg(((data>>8)&0xff), index + 1);
}

struct e13806fb_info {
	struct fb_info_gen gen;
};


static struct display disp;
static int panel_xres = 0;
static int panel_yres = 0;
static int panel_present = 1;

static struct fb_var_screeninfo default_var;

int e13806fb_init(void);
int e13806fb_setup(char*);

#define C_ENABLE 0
#define C_DISABLE 1

static struct tq_struct e13806_tq;

static int e13806_action = -1;

static void e13806_task(void *dummy)
{
	switch (e13806_action) {

	case C_ENABLE:
#ifdef CONFIG_ARCH_ADSAGX
		if (machine_is_adsagx())
			sio_backlite_on();
#endif
		break;

	case C_DISABLE:
#ifdef CONFIG_ARCH_ADSAGX
		if (machine_is_adsagx())
			sio_backlite_off();
#endif
		break;
	}
}

void e13806_run_task(int action)
{
	e13806_action = action;
	schedule_task(&e13806_tq);
}

static int e13806_encode_var(struct fb_var_screeninfo *var, const void *par,
			    struct fb_info_gen *info);
/* ------------------- chipset specific functions -------------------------- */


static void disable_hw_cursor(void)
{
	u8 curs;

	curs = e13806_read_reg(E13806_LcdInkCursCtrl);
	curs &= ~0xFC;
	e13806_write_reg(curs, E13806_LcdInkCursCtrl);

	curs = e13806_read_reg(E13806_CrtInkCursCtrl);
	curs &= ~0xFC;
	e13806_write_reg(curs, E13806_CrtInkCursCtrl);
}

static void e13806_detect(void)
{
	u8 rev;

	e13806_write_reg(0x00, E13806_Misc);

	rev = e13806_read_reg(E13806_RevCode);

	if ((rev & 0xfc) != 0x1C) {
		panel_present = 0;
		printk(KERN_WARNING "Epson 13806 not detected\n");
		return;
	}
	
	/* XXX */
	disable_hw_cursor();

	// e13806_init_regs();

	e13806_encode_var(&default_var, NULL, NULL);

}

struct e13806_par {
	int xres;
	int yres;
	int bpp;
	int mem_bpp;

	u32 panel_xres;
	u32 panel_yres;

	int panel_width;
};

static int e13806_encode_fix(struct fb_fix_screeninfo *fix,
			    const void *raw_par,
			    struct fb_info_gen *info)
{
	const struct e13806_par *par = raw_par;
	unsigned char disp_width;

	disp_width = e13806_read_reg(E13806_LcdHDP);

	memset(fix, 0, sizeof *fix);

	strcpy(fix->id, "SED13806");
	fix->type= FB_TYPE_PACKED_PIXELS;
	fix->line_length = (disp_width + 1) * 8;

	if (!par)
		BUG();

	if (par->bpp == 1) {
		fix->visual = FB_VISUAL_MONO10;
	} else if (par->bpp <= 8) {
		fix->visual = FB_VISUAL_PSEUDOCOLOR;
	} else {
		fix->visual = FB_VISUAL_DIRECTCOLOR;
		fix->line_length *= 2;
	}

	fix->smem_start = E13806_FB_BASE;
	fix->smem_len = 0x140000;
	fix->mmio_start = E13806_REG_BASE;
	fix->mmio_len = 0x200;

	fix->xpanstep = fix->ypanstep = fix->ywrapstep = 0;
	fix->accel = FB_ACCEL_NONE;

	return 0;
}

static int e13806_set_bpp(struct e13806_par *par, int bpp)
{
	int code;
	u8 disp;
	u16 stride;

	switch(bpp) {
	case 4:
		code = 2; break;
	case 8:
		code = 3; break;
	case 16:
		code = 5; break;
	default:
		return -EINVAL; break;
	}

	disp = e13806_read_reg(E13806_LcdDispMode);
	disp &= 0xF8;
	disp |= code;
	e13806_write_reg(disp, E13806_LcdDispMode);

	disp = e13806_read_reg(E13806_CrtDispMode);
	disp &= 0xF8;
	disp |= code;
	e13806_write_reg(disp, E13806_CrtDispMode);

	stride = panel_xres*bpp/16;

	e13806_write_reg16(stride, E13806_LcdStride0);

	e13806_write_reg16(stride, E13806_CrtStride0);

	par->bpp = bpp;

	return 0;
}

static int e13806_decode_var(const struct fb_var_screeninfo *var,
			    void *raw_par,
			    struct fb_info_gen *info)
{
	struct e13806_par *par = raw_par;
	int ret;

	if (!par)
		BUG();

	/*
	 * Don't allow setting any of these yet: xres and yres don't
	 * make sense for LCD panels; xres_virtual and yres_virtual
	 * should be supported fine by our hardware though.
	 */
	if (var->xres != panel_xres ||
	    var->yres != panel_yres ||
	    var->xres != var->xres_virtual ||
	    var->yres != var->yres_virtual ||
	    var->xoffset != 0 ||
	    var->yoffset != 0)
	{
		printk("epson13806fb: "
			"Trying to set display to %d * %d bpp %d is not allowed\n",var->xres,
			var->yres,var->bits_per_pixel);
		printk(	"var->xres = %d, panel_xres = %d\n"
			"var->yres = %d, panel_yres = %d\n"
			"var->xres_virtual = %d\n"
			"var->yres_virtual = %d\n"
			"var->xoffset = %d\n"
			"var->yoffset = %d\n",
			var->xres,panel_xres,var->yres,panel_yres,
			var->xres_virtual,var->yres_virtual,
			var->xoffset,var->yoffset);
		return -EINVAL;
	}

	if(var->bits_per_pixel != par->bpp) {
		ret = e13806_set_bpp(par, var->bits_per_pixel);

		if (ret)
			goto out_err;
	}

	return 0;

 out_err:
	return ret;
}

static void dump_panel_data(void)
{
	u8 panel = e13806_read_reg(E13806_PanelType);
	int width[2][4] = { { 4, 8, 16, -1 }, { 9, 12, 16, -1 } };

	printk("%s %s %s panel, width %d bits\n",
	       panel & 2 ? "dual" : "single",
	       panel & 4 ? "color" : "mono",
	       panel & 1 ? "TFT" : "passive",
	       width[panel&1][(panel>>4)&3]);

	printk("resolution %d x %d\n",
	       ((e13806_read_reg(E13806_LcdHDP)&0x7F) + 1) * 8,
	       ((e13806_read_reg16(E13806_LcdVDP0)&0x3FF) + 1));
}

static int e13806_bpp_to_var(int bpp, struct fb_var_screeninfo *var)
{
	switch(bpp) {
	case 1:
	case 2:
	case 4:
	case 8:
		var->bits_per_pixel = bpp;
		var->red.offset = var->green.offset = var->blue.offset = 0;
		var->red.length = var->green.length = var->blue.length = bpp;
		break;
	case 16:
		var->bits_per_pixel = 16;
		var->red.offset = 11;
		var->red.length = 5;
		var->green.offset = 5;
		var->green.length = 6;
		var->blue.offset = 0;
		var->blue.length = 5;
		break;
	}

	return 0;
}

static int e13806_encode_var(struct fb_var_screeninfo *var, const void *raw_par,
			    struct fb_info_gen *info)
{
	u8 panel, display,display2;
	u32 xres, yres, xres2, yres2;
	struct e13806_par *par = (struct e13806_par *) raw_par;
	static int width[2][4] = { { 4, 8, 16, -1 }, { 9, 12, 16, -1 } };
	static int bpp_tab[8] = { 1, 2, 4, 8, 15, 16 };
	int bpp, hw_bpp;
	int is_color, is_dual, is_tft;
	int lcd_enabled, crt_enabled;

	panel = e13806_read_reg(E13806_PanelType);
	display = e13806_read_reg(E13806_LcdDispMode);
	display2 = e13806_read_reg(E13806_CrtDispMode);

	is_color = (panel & 0x04) != 0;
	is_dual  = (panel & 0x02) != 0;
	is_tft   = (panel & 0x01) != 0;

	bpp = bpp_tab[display&7];
	e13806_bpp_to_var(bpp, var);

	crt_enabled = (display2 & 0x80) != 0;
	lcd_enabled = (display & 0x80) != 0;

	hw_bpp = width[is_tft][(panel>>4)&3];

	xres = ((e13806_read_reg(E13806_LcdHDP)&0x7F) + 1) * 8;
	yres = ((e13806_read_reg16(E13806_LcdVDP0)&0x3FF) + 1);

	xres2 = ((e13806_read_reg(E13806_CrtHDP)&0x7F) + 1) * 8;
	yres2 = ((e13806_read_reg16(E13806_CrtVDP0)&0x3FF) + 1);

	// Use the larger of CRT or LCD for the 'screen' size
	// Note, one day maybe this driver could be configured
	// to support multiple FBs (IE: different stuff on the
	// CRT or on the Panel).
	if (xres < xres2 && yres < yres2) {
		xres = xres2;
		yres = yres2;
	}

	var->xres = xres;
	var->yres = yres;
	var->xres_virtual = xres;
	var->yres_virtual = yres;

	var->xoffset = var->yoffset = 0;

	if(par)
	{
		par->xres = xres;
		par->yres = yres;
	}

	panel_xres = xres;
	panel_yres = yres;

	var->grayscale = !is_color;

	return 0;
}

#define is_dual(panel) (((panel)&3)==2)

static void get_panel_data(struct e13806_par *par)
{
	u8 panel;
	u8 bpp_select;

	int width[2][4] = { { 4, 8, 16, -1 }, { 9, 12, 16, -1 } };
	int bpp[8] = { -1, -1, 4, 8, -1, 16, -1, -1};

	panel = e13806_read_reg(E13806_PanelType);
	par->panel_width = width[panel&1][(panel>>4)&3];
	par->panel_xres = ((e13806_read_reg(E13806_LcdHDP)&0x7F) + 1) * 8;
	par->panel_yres = ((e13806_read_reg16(E13806_LcdVDP0)&0x3FF) + 1);

	bpp_select = e13806_read_reg(E13806_LcdDispMode) & 0x07;
	par->bpp = bpp[bpp_select];
}

static void e13806_get_par(void *raw_par, struct fb_info_gen *info)
{
	struct e13806_par *par = raw_par;

	get_panel_data(par);
}

static void e13806_set_par(const void *par, struct fb_info_gen *info)
{
}

static int e13806_getcolreg(unsigned regno, unsigned *red, unsigned *green,
			   unsigned *blue, unsigned *transp,
			   struct fb_info *info)
{
	u8 r, g, b;

	e13806_write_reg(regno, E13806_LutAddr);
	r = e13806_read_reg(E13806_LutData);
	g = e13806_read_reg(E13806_LutData);
	b = e13806_read_reg(E13806_LutData);

	*red = r << 8;
	*green = g << 8;
	*blue = b << 8;

	return 0;
}

static int e13806_setcolreg(unsigned regno, unsigned red, unsigned green,
			   unsigned blue, unsigned transp,
			   struct fb_info *info)
{
	u8 r = (red >> 8) & 0xf0;
	u8 g = (green>>8) & 0xf0;
	u8 b = (blue>> 8) & 0xf0;

	e13806_write_reg(regno, E13806_LutAddr);
	e13806_write_reg(r, E13806_LutData);
	e13806_write_reg(g, E13806_LutData);
	e13806_write_reg(b, E13806_LutData);

	return 0;
}

static int e13806_pan_display(const struct fb_var_screeninfo *var,
			     struct fb_info_gen *info)
{
	BUG();

	return -EINVAL;
}

static int e13806_blank(int blank_mode, struct fb_info_gen *info)
{
	u8 disp;

	switch (blank_mode) {
	case VESA_NO_BLANKING:
		disp = e13806_read_reg(E13806_LcdDispMode);
		disp &= ~0x80;
		e13806_write_reg(disp, E13806_LcdDispMode);

		disp = e13806_read_reg(E13806_CrtDispMode);
		disp &= ~0x80;
		e13806_write_reg(disp, E13806_CrtDispMode);

		e13806_run_task(C_ENABLE);

		break;

	case VESA_VSYNC_SUSPEND:
	case VESA_HSYNC_SUSPEND:
	case VESA_POWERDOWN:
		disp = e13806_read_reg(E13806_LcdDispMode);
		disp |= 1;
		e13806_write_reg(disp, E13806_LcdDispMode);

		disp = e13806_read_reg(E13806_CrtDispMode);
		disp |= 1;
		e13806_write_reg(disp, E13806_CrtDispMode);

		e13806_run_task(C_DISABLE);

		break;

	default:
		return -EINVAL;
	}

	return 0;
}

static struct display_switch e13806_dispsw;

static void e13806_set_disp(const void *unused, struct display *disp,
			   struct fb_info_gen *info)
{
	struct display_switch *d = NULL;

	if (!disp || !info) {
		printk(KERN_INFO "Framebuffer registers uninitialized.  Probably missing register.txt file\n");
		return;
	}

	disp->screen_base = screen_base;
	info->info.fix.type= FB_TYPE_PACKED_PIXELS;
	strcpy(info->info.fix.id, "SED13806");

	disp->dispsw = &e13806_dispsw;

	switch(disp->var.bits_per_pixel) {

	case 4:
		d = &fbcon_cfb4; break;

	case 8:
		d = &fbcon_cfb8; break;

	case 16:
		d = &fbcon_cfb16;
		disp->dispsw_data = cfb16;
		break;

	default:
		printk(KERN_WARNING
		       "Warning bpp: %d.  Is FB initialized properly?  Check register.txt\n", 
		       disp->var.bits_per_pixel);
		return;
	}

	memcpy(&e13806_dispsw, d, sizeof *d);
}

/* ------------ Interfaces to hardware functions ------------ */


struct fbgen_hwswitch e13806_switch = {
	detect:		e13806_detect,
	encode_fix:	e13806_encode_fix,
	decode_var:	e13806_decode_var,
	encode_var:	e13806_encode_var,
	get_par:	e13806_get_par,
	set_par:	e13806_set_par,
	getcolreg:	e13806_getcolreg,
	setcolreg:	e13806_setcolreg,
	pan_display:	e13806_pan_display,
	blank:		e13806_blank,
	set_disp:	e13806_set_disp,
};

/* ------------ Hardware Independent Functions ------------ */


static struct fb_ops e13806fb_ops = {
	owner:		THIS_MODULE,
     	fb_get_fix:	fbgen_get_fix,
	fb_get_var:	fbgen_get_var,
	fb_set_var:	fbgen_set_var,
	fb_get_cmap:	fbgen_get_cmap,
	fb_set_cmap:	fbgen_set_cmap,
	fb_pan_display:	fbgen_pan_display,
};

static struct e13806fb_info fb_info;

int __init e13806fb_setup(char *str)
{
	return 0;
}

#ifdef CONFIG_PM

static short e13806_pm_reg[] = {  0x001,      0x010,0x014,0x018,0x01C,0x01E,0x021,
                                  0x02A,0x02B,0x020,0x030,0x031,0x032,0x034,0x035,
                                  0x036,0x038,0x039,0x03A,0x03B,0x03C,0x040,0x041,
                                  0x042,0x043,0x044,0x046,0x047,0x048,0x04a,0x04B,
                                  0x050,0x052,0x053,0x054,0x056,0x057,0x058,0x059,
                                  0x05A,0x05B,0x060,0x062,0x063,0x064,0x066,0x067,
                                  0x068,0x06A,0x06B,0x070,0x071,0x072,0x073,0x074,
				  0x075,0x076,0x077,0x078,0x07A,0x07B,0x07C,0x07E,
				  0x080,0x081,0x082,0x083,0x084,0x085,0x086,0x087,
				  0x088,0x08A,0x08B,0x08C,0x08E,0x100,0x101,0x102,
				  0x103,0x104,0x105,0x106,0x108,0x109,0x10A,0x10C,
				  0x10D,0x110,0x111,0x112,0x113,0x114,0x115,0x118,
				  0x119,0x1E0,0x1E2,0x1F0,0x1F1,0x1F4,0x1FC};

static unsigned char e13806_pm_reg_val[sizeof(e13806_pm_reg)/sizeof(e13806_pm_reg[0])];

static void e13806_pm_save_regs(void)
{
	int i;

	for (i=0; i<sizeof(e13806_pm_reg)/sizeof(e13806_pm_reg[0]); i++)
		e13806_pm_reg_val[i] = e13806_read_reg(e13806_pm_reg[i]);
}

static void e13806_pm_restore_regs(void)
{
	int i;

	for (i=0; i<sizeof(e13806_pm_reg)/sizeof(e13806_pm_reg[0]); i++)
		e13806_write_reg(e13806_pm_reg_val[i], e13806_pm_reg[i]);
}

/*
 * Power management hook.  Note that we won't be called from IRQ context,
 * unlike the blank functions above, so we may sleep.
 */
static int
e13806_pm_callback(struct pm_dev *pm_dev, pm_request_t req, void *data)
{
	if (req == PM_SUSPEND || req == PM_RESUME) {
		int state = (int)data;

		if (state == 0) {
			/* Enter D0. */

			e13806_pm_restore_regs();

			e13806_write_reg(0x10 ,0x01F0);                            // disable power save mode
			e13806_write_reg(e13806_read_reg(0x01FC) | 0x01 ,0x01FC);  // enable LCD signals

			e13806_run_task(C_ENABLE);

		} else {

			e13806_pm_save_regs();

			/* Enter D1-D3.  Disable the LCD controller.  */
			e13806_write_reg(0x11 ,0x01F0);                            // enable power save mode
			e13806_write_reg(e13806_read_reg(0x01FC) & 0xFE, 0x01FC);  // disable LCD signals

			e13806_run_task(C_DISABLE);

		}
	}
	return 0;
}
#endif

int __init e13806fb_init(void)
{
	remaped_regs = (int)ioremap(E13806_REG_BASE,0x200);
	screen_base = (void *)ioremap_nocache(E13806_FB_BASE,0x140000);
	printk("Registers remaped to %08X from %08X\n",remaped_regs,E13806_REG_BASE);
	printk("FB remaped to %p from %08X\n",screen_base,E13806_FB_BASE);
	fb_info.gen.fbhw = &e13806_switch;
	fb_info.gen.fbhw->detect();

	if (!panel_present) {
		iounmap((void *) remaped_regs);
		iounmap(screen_base);
		return -ENXIO;
	}

	strcpy(fb_info.gen.info.modename, "SED13806");
	fb_info.gen.info.changevar = NULL;
	fb_info.gen.info.node = -1;
	fb_info.gen.info.fbops = &e13806fb_ops;
	fb_info.gen.info.disp = &disp;
	fb_info.gen.parsize = sizeof(struct e13806_par);
	fb_info.gen.info.switch_con = &fbgen_switch;
	fb_info.gen.info.updatevar = &fbgen_update_var;
	fb_info.gen.info.blank = &fbgen_blank;
	fb_info.gen.info.flags = FBINFO_FLAG_DEFAULT;
	/* This should give a reasonable default video mode */
	fbgen_get_var(&disp.var, -1, &fb_info.gen.info);
	fbgen_do_set_var(&disp.var, 1, &fb_info.gen);
	fbgen_set_disp(-1, &fb_info.gen);
	if (disp.var.bits_per_pixel > 1)
		fbgen_install_cmap(0, &fb_info.gen);
	if (register_framebuffer(&fb_info.gen.info) < 0)
		return -EINVAL;
	printk(KERN_INFO "fb%d: %s frame buffer device\n", GET_FB_IDX(fb_info.gen.info.node),
	       fb_info.gen.info.modename);

	INIT_TQUEUE(&e13806_tq, e13806_task, NULL);

	e13806_run_task(C_ENABLE);

#ifdef CONFIG_PM
	/*
	 * Note that the console registers this as well, but we want to
	 * power down the display prior to sleeping.
	 */
	pm_register(PM_SYS_DEV, PM_SYS_VGA, e13806_pm_callback);
#endif


	return 0;
}


    /*
     *  Cleanup
     */

void e13806fb_cleanup(struct fb_info *info)
{
	/*
	 *  If your driver supports multiple boards, you should unregister and
	 *  clean up all instances.
	 */

	unregister_framebuffer(info);
	/* ... */
}

MODULE_LICENSE("GPL");
