/*
 * drivers/pcmcia/sa1100_adsbitsyx.c
 *
 * PCMCIA implementation routines for ADS BitsyX
 *
 * Created Feb 18, 2003 by Robert Whaley <rwhaley@applieddata.net>
 *                      and Jeff Lackey
 *
 * This file comes from sa1100_adsbitsy.c of Woojung Huh <whuh@applieddata.net>
 *
 * Change log:
 *	29 JAN 03 JML
 *	- Adapted for PXA
 */
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ioport.h>

#include <asm/hardware.h>
#include <asm/hardware/sa1111.h>
#include <asm/irq.h>

#include <pcmcia/ss.h>
#include <asm/arch/pcmcia.h>
#include "../sa1111_generic.h"

#define NIRQ 2

static struct irqs {
	int irq;
	const char *str;
} irqs[PXA_PCMCIA_MAX_SOCK][NIRQ] = {
	{{ S0_CD_VALID,    "SA1111 PCMCIA card detect" },
	 { S0_BVD1_STSCHG, "SA1111 PCMCIA BVD1"        }},
	{{ S1_CD_VALID,    "SA1111 CF card detect"     },
	 { S1_BVD1_STSCHG, "SA1111 CF BVD1"            }},
};

static int adsbitsyx_pcmcia_init(struct pcmcia_init *init)
{
  int ret=0;
  int all_socks_ignored = 1;
  int i=0;
  int sock=0;

  extern int pxa_pcmcia_get_ignore_socket(int sock);

  if (!pxa_pcmcia_get_ignore_socket(0)) {

    /* Set GPIO_A<3:0> to be outputs for PCMCIA power controller: */
    PA_DDR &= ~(GPIO_bit(0) | GPIO_bit(1));
  
    /* Disable Power 3.3V/5V for PCMCIA */
    PA_DWR |= GPIO_bit(0) | GPIO_bit(1);

    INTPOL1 |= SA1111_IRQMASK_HI(S0_READY_NINT) |
	       SA1111_IRQMASK_HI(S0_CD_VALID)   |
               SA1111_IRQMASK_HI(S0_BVD1_STSCHG);

    all_socks_ignored = 0;
  }

  if (!pxa_pcmcia_get_ignore_socket(1)) {

    /* Set GPIO_A<3:0> to be outputs for CF power controller: */
    PA_DDR &= ~(GPIO_bit(2) | GPIO_bit(3));
  
    /* Disable Power 3.3V/5V for CF */
    PA_DWR |= GPIO_bit(2) | GPIO_bit(3);

    INTPOL1 |= SA1111_IRQMASK_HI(S1_READY_NINT) |
               SA1111_IRQMASK_HI(S1_CD_VALID)   |
               SA1111_IRQMASK_HI(S1_BVD1_STSCHG);
    all_socks_ignored = 0;
  }

  if (!all_socks_ignored)
	  if (!request_mem_region(_PCCR, 512, "PCMCIA"))
		  return -1;

  for (sock = 0; sock < PXA_PCMCIA_MAX_SOCK; sock++) {
	  if (pxa_pcmcia_get_ignore_socket(sock))
		  continue;

	  for (i = 0; i < NIRQ; i++) {
		  ret = request_irq(irqs[sock][i].irq, init->handler, SA_INTERRUPT,
				    irqs[sock][i].str, NULL);
		  if (ret) break;
	  }
	  if (ret) break;
  }

  if (ret) {
	  printk(KERN_ERR "sa1111_pcmcia: unable to grab IRQ%d (%d)\n",
		 irqs[sock][i].irq, ret);
	  do {

		  if (pxa_pcmcia_get_ignore_socket(sock))
			  continue;

		  while (i--)
			  free_irq(irqs[sock][i].irq, NULL);

		  i = NIRQ;
	  }
	  while (sock--);

	  release_mem_region(_PCCR, 16);
  }

  return ret ? -1 : 2;
}

static int adsbitsyx_pcmcia_shutdown(void)
{

	if (!pxa_pcmcia_get_ignore_socket(0)) {
		free_irq(S0_CD_VALID, NULL);
		free_irq(S0_BVD1_STSCHG, NULL);
		INTPOL1 &= ~(SA1111_IRQMASK_HI(S0_CD_VALID) | SA1111_IRQMASK_HI(S0_BVD1_STSCHG));
	}

	if (!pxa_pcmcia_get_ignore_socket(1)) {
		free_irq(S1_CD_VALID, NULL);
		free_irq(S1_BVD1_STSCHG, NULL);
		INTPOL1 &= ~(SA1111_IRQMASK_HI(S1_CD_VALID) | SA1111_IRQMASK_HI(S1_BVD1_STSCHG));
	}

	return 0;
}


static int adsbitsyx_pcmcia_socket_state(struct pcmcia_state_array *state)
{
	unsigned long status;

	memset(state->state, 0,
	       (state->size)*sizeof(struct pcmcia_state));

	status = PCSR;

	if (pxa_pcmcia_get_ignore_socket(0)) {
		// it returns not detect/ready/...
		state->state[0].detect = 0;
		state->state[0].ready = 0;
		state->state[0].bvd1 = 0;
		state->state[0].bvd2 = 0;
		state->state[0].wrprot = 0;
		state->state[0].vs_3v = 0;
		state->state[0].vs_Xv = 0;
	}
	else {
		state->state[0].detect = status & PCSR_S0_DETECT ? 0 : 1;
		state->state[0].ready  = status & PCSR_S0_READY  ? 1 : 0;
		state->state[0].bvd1   = status & PCSR_S0_BVD1   ? 1 : 0;
		state->state[0].bvd2   = status & PCSR_S0_BVD2   ? 1 : 0;
		state->state[0].wrprot = status & PCSR_S0_WP     ? 1 : 0;
		state->state[0].vs_3v  = status & PCSR_S0_VS1    ? 0 : 1;
		state->state[0].vs_Xv  = status & PCSR_S0_VS2    ? 0 : 1;
	}

	if (pxa_pcmcia_get_ignore_socket(1)) {
		// it returns not detect/ready/...
		state->state[1].detect = 0;
		state->state[1].ready = 0;
		state->state[1].bvd1 = 0;
		state->state[1].bvd2 = 0;
		state->state[1].wrprot = 0;
		state->state[1].vs_3v = 0;
		state->state[1].vs_Xv = 0;
	}
	else {
		state->state[1].detect = status & PCSR_S1_DETECT ? 0 : 1;
		state->state[1].ready  = status & PCSR_S1_READY  ? 1 : 0;
		state->state[1].bvd1   = status & PCSR_S1_BVD1   ? 1 : 0;
		state->state[1].bvd2   = status & PCSR_S1_BVD2   ? 1 : 0;
		state->state[1].wrprot = status & PCSR_S1_WP     ? 1 : 0;
		state->state[1].vs_3v  = status & PCSR_S1_VS1    ? 0 : 1;
		state->state[1].vs_Xv  = status & PCSR_S1_VS2    ? 0 : 1;
	}
	return 1;
}

static int adsbitsyx_pcmcia_configure_socket(unsigned int sock, socket_state_t *state)
{
  unsigned int pa_dwr_mask, pa_dwr_set, pccr_mask, rst, flt, wait, pse;
  unsigned long flags;

  switch (sock) {
  case 0:
    pa_dwr_mask = GPIO_bit(0) | GPIO_bit(1);
    rst  = PCCR_S0_RST;
    flt  = PCCR_S0_FLT;
    wait = PCCR_S0_PWAITEN;
    pse  = PCCR_S0_PSE;

    switch (state->Vcc) {
    case 0:
      pa_dwr_set = GPIO_bit(0) | GPIO_bit(1);
      pccr_mask  = 0;
      break;
    case 33:
      pa_dwr_set = GPIO_bit(1);
      pccr_mask  = wait;
      break;
    case 50:
      pa_dwr_set = GPIO_bit(0);
      pccr_mask  = pse | wait;
      break;
    default:
      printk(KERN_ERR "%s(): unrecognised VCC %u\n", __FUNCTION__, state->Vcc);
      return -1;
    }
    break;

  case 1:
    pa_dwr_mask = GPIO_bit(2) | GPIO_bit(3);
    rst  = PCCR_S1_RST;
    flt  = PCCR_S1_FLT;
    wait = PCCR_S1_PWAITEN;
    pse  = PCCR_S1_PSE;

    switch (state->Vcc) {
    case 0:
      pa_dwr_set = GPIO_bit(2) | GPIO_bit(3);
      pccr_mask  = 0;
      break;
    case 33:
      pa_dwr_set = GPIO_bit(2);
      pccr_mask  = wait;
      break;
    case 50:
      pa_dwr_set = GPIO_bit(3);
      pccr_mask  = pse | wait;
      break;
    default:
      printk(KERN_ERR "%s(): unrecognised VCC %u\n", __FUNCTION__, state->Vcc);
      return -1;
    }
    break;

  default:
    return -1;
  }

  if (state->Vpp != state->Vcc && state->Vpp != 0) {
    printk(KERN_ERR "%s(): cannot support Vpp %u\n", __FUNCTION__, state->Vpp);
    return -1;
  }

  if (state->flags&SS_RESET)
    pccr_mask |= rst;

  if (state->flags&SS_OUTPUT_ENA)
    pccr_mask |= flt;

  local_irq_save(flags);
  PCCR   = (PCCR & ~(pse | flt | wait | rst)) | pccr_mask;
  PA_DWR = (PA_DWR & ~pa_dwr_mask) | pa_dwr_set;
  local_irq_restore(flags);

  return 0;
}

struct pcmcia_low_level adsbitsyx_pcmcia_ops = {
  init:			adsbitsyx_pcmcia_init,
  shutdown:		adsbitsyx_pcmcia_shutdown,
  socket_state:		adsbitsyx_pcmcia_socket_state,
  get_irq_info:		sa1111_pcmcia_get_irq_info,
  configure_socket:	adsbitsyx_pcmcia_configure_socket,
};
