/*
 * drivers/pcmcia/sa1100_adsagx.c
 *
 * PCMCIA implementation routines for ADS BitsyX
 *
 * Created April 29, 2003 by Robert Whaley <rwhaley@applieddata.net>
 *			and Jeff Lackey 
 *
 * This file comes from sa1100_adsbitsy.c of Woojung Huh <whuh@applieddata.net>
 *
 * Change log:
 */
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ioport.h>

#include <asm/hardware.h>
#include <asm/irq.h>

#include <pcmcia/ss.h>
#include <asm/arch/pcmcia.h>

int pc_debug = 3;

static struct irqs {
	int irq;
	const char *str;
} irqs[] = {
	{ ADSAGX_IRQ_DET_A, "ADSAGX PCMCIA card detect" },
	{ ADSAGX_IRQ_STS_A, "ADSAGX PCMCIA BVD1"	},
	{ ADSAGX_IRQ_DET_B, "ADSAGX CF card detect"	},
	{ ADSAGX_IRQ_STS_B, "ADSAGX CF BVD1"		},
};

static int adsagx_pcmcia_init(struct pcmcia_init *init)
{
	int ret=0;
	int nirq = 4;
	int slots = 2;
	int i;

	/* Disable Power 3.3V/5V for PCMCIA */
	ADSAGX_CPLD_PCMCIA_CR &= ~(ADSAGX_CPLD_PCMCIA_CR_A_3V | ADSAGX_CPLD_PCMCIA_CR_A_3V);

	/* Disable Power 3.3V/5V for CF */
	ADSAGX_CPLD_PCMCIA_CR &= ~(ADSAGX_CPLD_PCMCIA_CR_B_3V | ADSAGX_CPLD_PCMCIA_CR_B_3V);
  
	for (i = ret = 0; i < nirq; i++) {
		ret = request_irq(irqs[i].irq, init->handler, SA_INTERRUPT,
				  irqs[i].str, NULL);
		if (ret)
			break;
	}

	if (i < nirq) {
		printk(KERN_ERR "adsagx_pcmcia: unable to grab IRQ%d (%d)\n",
		       irqs[i].irq, ret);
		while (i--)
			free_irq(irqs[i].irq, NULL);
	}

	return ret ? -1 : slots;
}

static int adsagx_pcmcia_shutdown(void)
{

	free_irq(ADSAGX_IRQ_DET_A, NULL);
	free_irq(ADSAGX_IRQ_STS_A, NULL);

	free_irq(ADSAGX_IRQ_DET_B, NULL);
	free_irq(ADSAGX_IRQ_STS_B, NULL);

	return 0;
}

static int adsagx_pcmcia_socket_state(struct pcmcia_state_array *state)
{
	unsigned long status;

	if(state->size<2) return -1;

	memset(state->state, 0,
	       (state->size)*sizeof(struct pcmcia_state));

	status = ADSAGX_CPLD_PCMCIA_SR;

	state->state[0].detect = status & ADSAGX_CPLD_PCMCIA_STA_A_DET ? 1 : 0;
	state->state[0].ready  = status & ADSAGX_CPLD_PCMCIA_STA_A_IRQ ? 1 : 0;
	state->state[0].vs_3v  = GPLR(7)  & GPIO_bit(7)	 ? 0 : 1;
	state->state[0].vs_Xv  = GPLR(10) & GPIO_bit(10) ? 0 : 1;
	state->state[0].bvd1   = 0;
	state->state[0].bvd2   = 0;
	state->state[0].wrprot = 0;

	state->state[1].detect = status & ADSAGX_CPLD_PCMCIA_STA_B_DET ? 1 : 0;
	state->state[1].ready  = status & ADSAGX_CPLD_PCMCIA_STA_B_IRQ ? 1 : 0;
	state->state[1].vs_3v  = GPLR(11) & GPIO_bit(11) ? 0 : 1;
	state->state[1].vs_Xv  = GPLR(14) & GPIO_bit(14) ? 0 : 1;
	state->state[1].bvd1   = 0;
	state->state[1].bvd2   = 0;
	state->state[1].wrprot = 0;

	return 1;
}

static int adsagx_pcmcia_get_irq_info(struct pcmcia_irq_info *info)
{
	switch(info->sock){
	case 0:
		info->irq=ADSAGX_IRQ_CARD_A;
		break;

	case 1:
		info->irq=ADSAGX_IRQ_CARD_B;
		break;
		
	default:
		return -1;
	}

	return 0;
}

static int adsagx_pcmcia_configure_socket(unsigned int sock, socket_state_t *state)
{
	unsigned char clear_bits = 0;
	unsigned char set_bits = 0;
	unsigned long flags;
	
	switch (sock) {
	case 0:

		switch (state->Vcc) {
		case 0:
			clear_bits = ADSAGX_CPLD_PCMCIA_CR_A_3V | ADSAGX_CPLD_PCMCIA_CR_A_5V;
			break;
		case 33:
			clear_bits = ADSAGX_CPLD_PCMCIA_CR_A_5V;
			set_bits = ADSAGX_CPLD_PCMCIA_CR_A_3V;
			break;
		case 50:
			clear_bits = ADSAGX_CPLD_PCMCIA_CR_A_3V;
			set_bits = ADSAGX_CPLD_PCMCIA_CR_A_5V;
			break;
		default:
			printk(KERN_ERR "%s(): unrecognised VCC %u\n", __FUNCTION__, state->Vcc);
			return -1;
		}
		if (state->flags & SS_RESET)
			set_bits |= ADSAGX_CPLD_PCMCIA_CR_A_RS;
		else
			clear_bits |= ADSAGX_CPLD_PCMCIA_CR_A_RS;
		break;

	case 1:

		switch (state->Vcc) {
		case 0:
			clear_bits = ADSAGX_CPLD_PCMCIA_CR_B_3V | ADSAGX_CPLD_PCMCIA_CR_B_5V;
			break;
		case 33:
			clear_bits = ADSAGX_CPLD_PCMCIA_CR_B_5V;
			set_bits = ADSAGX_CPLD_PCMCIA_CR_B_3V;
			break;
		case 50:
			clear_bits = ADSAGX_CPLD_PCMCIA_CR_B_3V;
			set_bits = ADSAGX_CPLD_PCMCIA_CR_B_5V;
			break;
		default:
			printk(KERN_ERR "%s(): unrecognised VCC %u\n", __FUNCTION__, state->Vcc);
			return -1;
		}
		if (state->flags & SS_RESET)
			set_bits |= ADSAGX_CPLD_PCMCIA_CR_B_RS;
		else
			clear_bits |= ADSAGX_CPLD_PCMCIA_CR_B_RS;
		break;

	default:
		return -1;
	}

	if (state->Vpp != state->Vcc && state->Vpp != 0) {
		printk(KERN_ERR "%s(): cannot support Vpp %u\n", __FUNCTION__, state->Vpp);
		return -1;
	}

	local_irq_save(flags);
	ADSAGX_CPLD_PCMCIA_CR = (ADSAGX_CPLD_PCMCIA_CR | set_bits) & ~clear_bits;
	local_irq_restore(flags);

	return 0;
}

struct pcmcia_low_level adsagx_pcmcia_ops = {
	init:			adsagx_pcmcia_init,
	shutdown:		adsagx_pcmcia_shutdown,
	socket_state:		adsagx_pcmcia_socket_state,
	get_irq_info:		adsagx_pcmcia_get_irq_info,
	configure_socket:	adsagx_pcmcia_configure_socket,
};
