/*
 * Map driver for the ADS PXA Boards.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>


#define WINDOW_ADDR 	0x04000000
#define WINDOW_SIZE 	64*1024*1024
#define BUSWIDTH 	4

static __u8 ads_pxa_flash_read8(struct map_info *map, unsigned long ofs)
{
	return *(__u8 *)(map->map_priv_1 + ofs);
}

static __u16 ads_pxa_flash_read16(struct map_info *map, unsigned long ofs)
{
	return *(__u16 *)(map->map_priv_1 + ofs);
}

static __u32 ads_pxa_flash_read32(struct map_info *map, unsigned long ofs)
{
	return *(__u32 *)(map->map_priv_1 + ofs);
}

static void ads_pxa_flash_copy_from(struct map_info *map, void *to, unsigned long from, ssize_t len)
{
	memcpy(to, (void *)(map->map_priv_1 + from), len);
}

static void ads_pxa_flash_write8(struct map_info *map, __u8 d, unsigned long adr)
{
	*(__u8 *)(map->map_priv_1 + adr) = d;
}

static void ads_pxa_flash_write16(struct map_info *map, __u16 d, unsigned long adr)
{
	*(__u16 *)(map->map_priv_1 + adr) = d;
}

static void ads_pxa_flash_write32(struct map_info *map, __u32 d, unsigned long adr)
{
	*(__u32 *)(map->map_priv_1 + adr) = d;
}

static void ads_pxa_flash_copy_to(struct map_info *map, unsigned long to, const void *from, ssize_t len)
{
	memcpy((void *)(map->map_priv_1 + to), from, len);
}

static struct map_info pxa_map = {
	name: "PXA flash",
	size: WINDOW_SIZE,
	buswidth: BUSWIDTH,
	read8:		ads_pxa_flash_read8,
	read16:		ads_pxa_flash_read16,
	read32:		ads_pxa_flash_read32,
	copy_from:	ads_pxa_flash_copy_from,
	write8:		ads_pxa_flash_write8,
	write16:	ads_pxa_flash_write16,
	write32:	ads_pxa_flash_write32,
	copy_to:	ads_pxa_flash_copy_to,
};

#ifdef CONFIG_ARCH_ADSBITSYX
#define ADSBITSYX_FLASH_SIZE		0x04000000
#define ADSBITSYX_BUS_WIDTH		4
static struct mtd_partition adsbitsyx_partitions[] = {
	{
		name:		"adslinux.rom",
		size:		0x00040000,
		offset:		0,
		mask_flags:	MTD_WRITEABLE,  /* force read-only */
	}, {
		name:		"zImage",
		size:		0x00100000,
		offset:		MTDPART_OFS_APPEND,
		mask_flags:	MTD_WRITEABLE,  /* force read-only */
	}, {
		name:		"ramdisk.gz",
		size:		0x00300000,
		offset:		MTDPART_OFS_APPEND,
		mask_flags:	MTD_WRITEABLE,  /* force read-only */
	}, {
		name:		"flashfs1",
		size:		MTDPART_SIZ_FULL,
		offset:		MTDPART_OFS_APPEND
	}
};
#endif

#ifdef CONFIG_ARCH_ADSAGX
#define ADSAGX_FLASH_SIZE		0x04000000
#define ADSAGX_BUS_WIDTH		4
static struct mtd_partition adsagx_partitions[] = {
	{
		name:		"adslinux.rom",
		size:		0x00040000,
		offset:		0,
		mask_flags:	MTD_WRITEABLE,  /* force read-only */
	}, {
		name:		"zImage",
		size:		0x00100000,
		offset:		MTDPART_OFS_APPEND,
		mask_flags:	MTD_WRITEABLE,  /* force read-only */
	}, {
		name:		"ramdisk.gz",
		size:		0x00300000,
		offset:		MTDPART_OFS_APPEND,
		mask_flags:	MTD_WRITEABLE,  /* force read-only */
	}, {
		name:		"flashfs1",
		size:		MTDPART_SIZ_FULL,
		offset:		MTDPART_OFS_APPEND
	}
};
#endif
#define NB_OF(x)  (sizeof(x)/sizeof(x[0]))

static struct mtd_info *mymtd;
static struct mtd_partition *parsed_parts;

extern int parse_redboot_partitions(struct mtd_info *master, struct mtd_partition **pparts);

static int __init init_pxa_flash(void)
{
	struct mtd_partition *parts;
	int nb_parts = 0;
	int parsed_nr_parts = 0;
	char *part_type = "static";

	printk("Probing PXA flash at physical address 0x%08x\n", WINDOW_ADDR);
	pxa_map.map_priv_1 = (unsigned long)__ioremap(WINDOW_ADDR, WINDOW_SIZE, 0);
	if (!pxa_map.map_priv_1) {
		printk("Failed to ioremap\n");
		return -EIO;
	}
	// what about jedec_probe ?
	mymtd = do_map_probe("cfi_probe", &pxa_map);

	if (!mymtd) {
		iounmap((void *)pxa_map.map_priv_1);
		return -ENXIO;
	}
	mymtd->module = THIS_MODULE;

#ifdef CONFIG_MTD_REDBOOT_PARTS
	if (parsed_nr_parts == 0) {
		int ret = parse_redboot_partitions(mymtd, &parsed_parts);

		if (ret > 0) {
			part_type = "RedBoot";
			parsed_nr_parts = ret;
		}
	}
#endif
#ifdef CONFIG_MTD_CMDLINE_PARTS
	if (parsed_nr_parts == 0) {
		int ret = parse_cmdline_partitions(mymtd, &parsed_parts, "pxa");
		if (ret <= 0) {
			ret = parse_cmdline_partitions(mymtd, &parsed_parts, "sa1100");
		}
		if (ret > 0) {
			part_type = "Command Line";
			parsed_nr_parts = ret;
		}
	}
#endif

	if (parsed_nr_parts > 0) {
		parts = parsed_parts;
		nb_parts = parsed_nr_parts;
	} else {
#ifdef CONFIG_ARCH_ADSBITSYX
		parts = adsbitsyx_partitions;
		nb_parts = NB_OF(adsbitsyx_partitions);
		pxa_map.size = ADSBITSYX_FLASH_SIZE;
		pxa_map.buswidth = ADSBITSYX_BUS_WIDTH;
#endif
#ifdef CONFIG_ARCH_ADSAGX
		parts = adsagx_partitions;
		nb_parts = NB_OF(adsagx_partitions);
		pxa_map.size = ADSAGX_FLASH_SIZE;
		pxa_map.buswidth = ADSAGX_BUS_WIDTH;
#endif
	}
	if (nb_parts) {
		printk(KERN_NOTICE "Using %s partition definition\n", part_type);
		add_mtd_partitions(mymtd, parts, nb_parts);
	} else {
		add_mtd_device(mymtd);
	}
	return 0;
}

static void __exit cleanup_pxa_flash(void)
{
	if (mymtd) {
		del_mtd_partitions(mymtd);
		map_destroy(mymtd);
		if (parsed_parts)
			kfree(parsed_parts);
	}
	if (pxa_map.map_priv_1)
		iounmap((void *)pxa_map.map_priv_1);
}

module_init(init_pxa_flash);
module_exit(cleanup_pxa_flash);

