/*
 * ltc1663.c
 *
 * Device driver for Linear Technology's Micropower DAC LTC1663.
 *
 * Copyright (C) 2003 Applied Data Systems
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>

#include <linux/kernel.h>
#include <linux/poll.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/proc_fs.h>

#include "ltc1663.h"

#define DEBUG 0

#if DEBUG
static unsigned int ltc1663_debug = DEBUG;
#else
#define ltc1663_debug 0	/* gcc will remove all the debug code for us */
#endif

static unsigned short slave_address = LTC1663_I2C_SLAVE_ADDR;

struct i2c_driver ltc1663_driver;
struct i2c_client *ltc1663_i2c_client = 0;

static unsigned short ignore[] = { I2C_CLIENT_END };
static unsigned short normal_addr[] = { LTC1663_I2C_SLAVE_ADDR, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c:		normal_addr,
	normal_i2c_range:	ignore,
	probe:			ignore,
	probe_range:		ignore,
	ignore: 		ignore,
	ignore_range:		ignore,
	force:			ignore,
};

static int ltc1663_probe(struct i2c_adapter *adap);
static int ltc1663_detach(struct i2c_client *client);
static int ltc1663_set_voltage(struct i2c_client *client, unsigned int cmd, void *arg);

struct i2c_driver ltc1663_driver = {
	name:		I2C_NAME_LTC1663,
	id:		I2C_DRIVERID_LTC1663,
	flags:		I2C_DF_NOTIFY,
	attach_adapter: ltc1663_probe,
	detach_client:	ltc1663_detach,
	command:	ltc1663_set_voltage
};

static int
ltc1663_attach(struct i2c_adapter *adap, int addr, unsigned short flags,int kind)
{
	struct i2c_client *c;
	unsigned char ad[1] = { 7 };
	struct i2c_msg msgs[2] = {
		{ addr	, 0,	    1, ad  }
	};
	int ret;

	if (ltc1663_debug > 2)
		printk ("%s\n", __FUNCTION__);

	c = (struct i2c_client *)kmalloc(sizeof(*c), GFP_KERNEL);
	if (!c)
		return -ENOMEM;

	strcpy(c->name, I2C_NAME_LTC1663);
	c->id		= ltc1663_driver.id;
	c->flags	= I2C_CLIENT_ALLOW_USE;
	c->addr 	= addr;
	c->adapter	= adap;
	c->driver	= &ltc1663_driver;
	c->data 	= NULL;

	ret = i2c_transfer(c->adapter, msgs, 1);

	if ( ret != 1 && ltc1663_debug > 0 )
		printk ("%s : i2c_transfer() returned %d\n", __FUNCTION__, ret);

	ltc1663_i2c_client = c;

	return i2c_attach_client(c);
}

static int
ltc1663_probe(struct i2c_adapter *adap)
{
	if (ltc1663_debug > 2)
		printk ("%s\n", __FUNCTION__);

	return i2c_probe(adap, &addr_data, ltc1663_attach);
}

static int
ltc1663_detach(struct i2c_client *client)
{
	if (ltc1663_debug > 2)
		printk ("%s\n", __FUNCTION__);

	i2c_detach_client(client);

	return 0;
}

static int
ltc1663_set_voltage(struct i2c_client *client, unsigned int ignore, void *value)
{
	unsigned char buf[3];
	unsigned int val = * (unsigned int *) value;
	int ret, len = 3;

	if (!client) return -EIO;

	if (ltc1663_debug > 1)
		printk ("%s sending %#04x\n", __FUNCTION__, val);

	buf[0] = LTC1663_BG;
	buf[1] = (val & 0xff);
	buf[2] = (val >> 8);

	ret = i2c_master_send(client, (char *)buf, len);
	if (ret == len)
		ret = 0;
	else if ( ltc1663_debug > 0 )
		printk ("%s : i2c_master_send() returned %d\n", __FUNCTION__, ret);

	return ret;
}

static __init int ltc1663_init(void)
{
	int retval=0;

	if (slave_address != 0xffff)
	{
		if (ltc1663_debug > 2)
			printk ("%s : slave address is %#x\n", __FUNCTION__, slave_address);

		normal_addr[0] = slave_address;
	}

	if (normal_addr[0] == 0xffff)
	{
		printk(KERN_ERR"I2C: Invalid slave address for LTC1663 DAC (%#x)\n",
		       normal_addr[0]);
		return -EINVAL;
	}

	retval = i2c_add_driver(&ltc1663_driver);

	return retval;
}

static __exit void ltc1663_exit(void)
{
	if (ltc1663_debug > 2)
		printk ("%s\n", __FUNCTION__);

	i2c_del_driver(&ltc1663_driver);
}

module_init(ltc1663_init);
module_exit(ltc1663_exit);

MODULE_PARM (slave_address, "i");
MODULE_PARM_DESC (slave_address, "I2C slave address for LTC1663 DAC.");

MODULE_AUTHOR ("Applied Data Systems");
MODULE_LICENSE("GPL");
