/*
 * linux/arch/arm/drivers/char/adssmartio.c
 *
 * Copyright 2002 Applied Data Systems
 *
 * Keyboard & Smartio driver for ADSBITSY
 *
 * Change log:
 *	2 JUL 03 RW
 *	- Add AGX
 *      - Fix problem with BitsyX
 *	29 JAN 03 JML
 *	- Add PXA
 *	28 JUN 02 RW
 *	- Add I2C
 *	08 MAR 02 JML
 *	- Major merge,
 *	   esp. mutex protection and interrupt info handling
 *	- Consider leaving UCB1200 I/O access here.
 *	01 MAR 02 JML
 *	- Fixed AVR turning back light off after write to Port D
 *	- Added hooks for EEPROM I/O
 *	11/5/01 WH
 *	- Add UCB1200 IOs access (read, write and ioctl) functions
 *	   *FIXME* It is temporal. ucb1200 IOs should be places seperate file.
 *	- Add UCB1200 ADC access
 *	10/18/01 Woojung
 *	   fix warning
 *	9/7/01 Woojung
 *	   SmartIO for ADS Bitsy
 *	7-10/6/01 Thomas Thaele <tthaele@papenmeier.de>
 *	- Added Keyboard Sniffer on /dev/sio12 <minor = 12>
 *	- First implementation of PC- compatible Scancodes (thanks to pc_keyb.c)
 *	3/23/01 Woojung Huh
 *	   Power Management added
 *	12/01/00 Woojung Huh
 *	   Bug fixed
 *	11/16/00 Woojung Huh [whuh@applieddata.net]
 *	   Added smartio device driver on it
 */

/*
 * Introduced setkeycode, ketkeycode for the GC+ by Thomas Thaele
 * <tthaele@papenmeier.de> GC+ now performs like a real PC on the keyboard.
 * Warning: this code is still beta! PrntScrn and Pause keys are not
 * completely tested and implemented!!! Keyboard driver can be confused
 * by hacking like crazy on the keyboard. (hardware problem on serial line?)
 */

#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/kbd_ll.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/kbd_kern.h>
#include <linux/ctype.h>

#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/keyboard.h>
#ifdef CONFIG_UCB1200
#  include <asm/ucb1200.h>
#endif
#include <linux/tqueue.h>
#include <linux/proc_fs.h>
#include <linux/pm.h>

#if defined(CONFIG_ARCH_ADSBITSYX)
#  define ADS_AVR_GPIO			5
#  define ADS_AVR_IRQ			IRQ_GPIO(ADS_AVR_GPIO)
#  define ADS_RESETAVR			27 // Rev 1 BitsyX Only
#elif defined(CONFIG_ARCH_ADSAGX)
#  define ADS_AVR_IRQ			ADSAGX_IRQ_SSP
#  define ADS_RESETAVR			27
#elif defined(CONFIG_SA1100_ADSBITSYPLUS)
#  define ADS_AVR_IRQ			IRQ_GPIO13
#  define ADS_RESETAVR			GPIO_GPIO25
#else
#  define ADS_AVR_IRQ			IRQ_GPCIN5
#  define ADS_RESETAVR			GPIO_GPIO25
#endif

#define	SMARTIO_IOCTL_BASES		's'
#define	SMARTIO_KPD_TIMEOUT		_IOW(SMARTIO_IOCTL_BASES, 0, int)
#define	SMARTIO_KPD_SETUP		_IOW(SMARTIO_IOCTL_BASES, 1, short)
#define	SMARTIO_BL_CONTROL		_IOW(SMARTIO_IOCTL_BASES, 2, char)
#define	SMARTIO_BL_CONTRAST		_IOW(SMARTIO_IOCTL_BASES, 3, char)
#define SMARTIO_PORT_CONFIG		_IOW(SMARTIO_IOCTL_BASES, 4, char)
#define SMARTIO_SNIFFER_TIMEOUT		_IOW(SMARTIO_IOCTL_BASES, 5, long)

static char *smartio_version = "1.12";
static char *smartio_date = "2-Jul-2003";

/* Simple translation table for the SysRq keys */

#ifdef CONFIG_MAGIC_SYSRQ
unsigned char pckbd_sysrq_xlate[128] =
	"\000\0331234567890-=\177\t"			/* 0x00 - 0x0f */
	"qwertyuiop[]\r\000as"				/* 0x10 - 0x1f */
	"dfghjkl;'`\000\\zxcv"				/* 0x20 - 0x2f */
	"bnm,./\000*\000 \000\201\202\203\204\205"	/* 0x30 - 0x3f */
	"\206\207\210\211\212\000\000789-456+1"		/* 0x40 - 0x4f */
	"230\177\000\000\213\214\000\000\000\000\000\000\000\000\000\000" /* 0x50 - 0x5f */
	"\r\000/";					/* 0x60 - 0x6f */
#endif

/*
 * Translation of escaped scancodes to keycodes.
 * This is now user-settable.
 * The keycodes 1-88,96-111,119 are fairly standard, and
 * should probably not be changed - changing might confuse X.
 * X also interprets scancode 0x5d (KEY_Begin).
 *
 * For 1-88 keycode equals scancode.
 */

#define E0_KPENTER 96
#define E0_RCTRL   97
#define E0_KPSLASH 98
#define E0_PRSCR   99
#define E0_RALT    100
#define E0_BREAK   101  /* (control-pause) */
#define E0_HOME    102
#define E0_UP      103
#define E0_PGUP    104
#define E0_LEFT    105
#define E0_RIGHT   106
#define E0_END     107
#define E0_DOWN    108
#define E0_PGDN    109
#define E0_INS     110
#define E0_DEL     111

#define E1_PAUSE   119

/*
 * The keycodes below are randomly located in 89-95,112-118,120-127.
 * They could be thrown away (and all occurrences below replaced by 0),
 * but that would force many users to use the `setkeycodes' utility, where
 * they needed not before. It does not matter that there are duplicates, as
 * long as no duplication occurs for any single keyboard.
 */
#define SC_LIM 89

#define FOCUS_PF1 85           /* actual code! */
#define FOCUS_PF2 89
#define FOCUS_PF3 90
#define FOCUS_PF4 91
#define FOCUS_PF5 92
#define FOCUS_PF6 93
#define FOCUS_PF7 94
#define FOCUS_PF8 95
#define FOCUS_PF9 120
#define FOCUS_PF10 121
#define FOCUS_PF11 122
#define FOCUS_PF12 123

#define JAP_86     124
/* tfj@olivia.ping.dk:
 * The four keys are located over the numeric keypad, and are
 * labelled A1-A4. It's an rc930 keyboard, from
 * Regnecentralen/RC International, Now ICL.
 * Scancodes: 59, 5a, 5b, 5c.
 */
#define RGN1 124
#define RGN2 125
#define RGN3 126
#define RGN4 127

static unsigned char high_keys[128 - SC_LIM] = {
  RGN1, RGN2, RGN3, RGN4, 0, 0, 0,                   /* 0x59-0x5f */
  0, 0, 0, 0, 0, 0, 0, 0,                            /* 0x60-0x67 */
  0, 0, 0, 0, 0, FOCUS_PF11, 0, FOCUS_PF12,          /* 0x68-0x6f */
  0, 0, 0, FOCUS_PF2, FOCUS_PF9, 0, 0, FOCUS_PF3,    /* 0x70-0x77 */
  FOCUS_PF4, FOCUS_PF5, FOCUS_PF6, FOCUS_PF7,        /* 0x78-0x7b */
  FOCUS_PF8, JAP_86, FOCUS_PF10, 0                   /* 0x7c-0x7f */
};

/* BTC */
#define E0_MACRO   112
/* LK450 */
#define E0_F13     113
#define E0_F14     114
#define E0_HELP    115
#define E0_DO      116
#define E0_F17     117
#define E0_KPMINPLUS 118
/*
 * My OmniKey generates e0 4c for  the "OMNI" key and the
 * right alt key does nada. [kkoller@nyx10.cs.du.edu]
 */
#define E0_OK	124
/*
 * New microsoft keyboard is rumoured to have
 * e0 5b (left window button), e0 5c (right window button),
 * e0 5d (menu button). [or: LBANNER, RBANNER, RMENU]
 * [or: Windows_L, Windows_R, TaskMan]
 */
#define E0_MSLW	125
#define E0_MSRW	126
#define E0_MSTM	127

static unsigned char e0_keys[128] = {
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x00-0x07 */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x08-0x0f */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x10-0x17 */
  0, 0, 0, 0, E0_KPENTER, E0_RCTRL, 0, 0,	      /* 0x18-0x1f */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x20-0x27 */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x28-0x2f */
  0, 0, 0, 0, 0, E0_KPSLASH, 0, E0_PRSCR,	      /* 0x30-0x37 */
  E0_RALT, 0, 0, 0, 0, E0_F13, E0_F14, E0_HELP,	      /* 0x38-0x3f */
  E0_DO, E0_F17, 0, 0, 0, 0, E0_BREAK, E0_HOME,	      /* 0x40-0x47 */
  E0_UP, E0_PGUP, 0, E0_LEFT, E0_OK, E0_RIGHT, E0_KPMINPLUS, E0_END,/* 0x48-0x4f */
  E0_DOWN, E0_PGDN, E0_INS, E0_DEL, 0, 0, 0, 0,	      /* 0x50-0x57 */
  0, 0, 0, E0_MSLW, E0_MSRW, E0_MSTM, 0, 0,	      /* 0x58-0x5f */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x60-0x67 */
  0, 0, 0, 0, 0, 0, 0, E0_MACRO,		      /* 0x68-0x6f */
  0, 0, 0, 0, 0, 0, 0, 0,			      /* 0x70-0x77 */
  0, 0, 0, 0, 0, 0, 0, 0			      /* 0x78-0x7f */
};

int smartio_kbd_setkeycode(unsigned int scancode, unsigned int keycode)
{
	if (scancode < SC_LIM || scancode > 255 || keycode > 127)
	  return -EINVAL;
	if (scancode < 128)
	  high_keys[scancode - SC_LIM] = keycode;
	else
	  e0_keys[scancode - 128] = keycode;
	return 0;
}

int smartio_kbd_getkeycode(unsigned int scancode)
{
	return
	  (scancode < SC_LIM || scancode > 255) ? -EINVAL :
	  (scancode < 128) ? high_keys[scancode - SC_LIM] :
	    e0_keys[scancode - 128];
}

int smartio_kbd_translate(unsigned char scancode, unsigned char *keycode,
		    char raw_mode)
{
	static int prev_scancode = 0;

	/* special prefix scancodes.. */
	if (scancode == 0xe0 || scancode == 0xe1) {
		prev_scancode = scancode;
		return 0;
	}

	/* 0xFF is sent by a few keyboards, ignore it. 0x00 is error */
	if (scancode == 0x00 || scancode == 0xff) {
		prev_scancode = 0;
		return 0;
	}

	scancode &= 0x7f;

	if (prev_scancode) {
	  /*
	   * usually it will be 0xe0, but a Pause key generates
	   * e1 1d 45 e1 9d c5 when pressed, and nothing when released
	   */
	  if (prev_scancode != 0xe0) {
	      if (prev_scancode == 0xe1 && scancode == 0x1d) {
		  prev_scancode = 0x100;
		  return 0;
	      } else if (prev_scancode == 0x100 && scancode == 0x45) {
		  *keycode = E1_PAUSE;
		  prev_scancode = 0;
	      } else {
#ifdef KBD_REPORT_UNKN
		  if (!raw_mode)
		    printk(KERN_INFO "keyboard: unknown e1 escape sequence\n");
#endif
		  prev_scancode = 0;
		  return 0;
	      }
	  } else {
	      prev_scancode = 0;
	      /*
	       *  The keyboard maintains its own internal caps lock and
	       *  num lock statuses. In caps lock mode E0 AA precedes make
	       *  code and E0 2A follows break code. In num lock mode,
	       *  E0 2A precedes make code and E0 AA follows break code.
	       *  We do our own book-keeping, so we will just ignore these.
	       */
	      /*
	       *  For my keyboard there is no caps lock mode, but there are
	       *  both Shift-L and Shift-R modes. The former mode generates
	       *  E0 2A / E0 AA pairs, the latter E0 B6 / E0 36 pairs.
	       *  So, we should also ignore the latter. - aeb@cwi.nl
	       */
	      if (scancode == 0x2a || scancode == 0x36)
		return 0;

	      if (e0_keys[scancode])
		*keycode = e0_keys[scancode];
	      else {
#ifdef KBD_REPORT_UNKN
		  if (!raw_mode)
		    printk(KERN_INFO "keyboard: unknown scancode e0 %02x\n",
			   scancode);
#endif
		  return 0;
	      }
	  }
	} else if (scancode >= SC_LIM) {
	    /* This happens with the FOCUS 9000 keyboard
	       Its keys PF1..PF12 are reported to generate
	       55 73 77 78 79 7a 7b 7c 74 7e 6d 6f
	       Moreover, unless repeated, they do not generate
	       key-down events, so we have to zero up_flag below */
	    /* Also, Japanese 86/106 keyboards are reported to
	       generate 0x73 and 0x7d for \ - and \ | respectively. */
	    /* Also, some Brazilian keyboard is reported to produce
	       0x73 and 0x7e for \ ? and KP-dot, respectively. */

	  *keycode = high_keys[scancode - SC_LIM];

	  if (!*keycode) {
	      if (!raw_mode) {
#ifdef KBD_REPORT_UNKN
		  printk(KERN_INFO "keyboard: unrecognized scancode (%02x)"
			 " - ignored\n", scancode);
#endif
	      }
	      return 0;
	  }
 	} else
	  *keycode = scancode;
 	return 1;
}

// this table converts the hardware dependent codes of a MF-2 Keyboard to
// the codes normally comming out of a i8042. This table is 128 Bytes too
// big, but for stability reasons it should be kept like it is!
// There is no range checking in the code!
static int mf_two_kbdmap[256] = {
	00, 67, 65, 63, 61, 59, 60, 88, 00, 68, 66, 64, 62, 15, 41, 00,
	00, 56, 42, 00, 29, 16, 02, 00, 00, 00, 44, 31, 30, 17, 03, 00,
	00, 46, 45, 32, 18, 05, 04, 00, 00, 57, 47, 33, 20, 19, 06, 00,
	00, 49, 48, 35, 34, 21,  7, 00, 00, 00, 50, 36, 22,  8,  9, 00,
	00, 51, 37, 23, 24, 11, 10, 00, 00, 52, 53, 38, 39, 25, 12, 00,
	00, 00, 40, 00, 26, 13, 00, 00, 58, 54, 28, 27, 00, 43, 00, 00,
	00, 86, 00, 00, 00, 00, 14, 00, 00, 79, 00, 75, 71, 00, 00, 00,
	82, 83, 80, 76, 77, 72, 01, 69, 87, 78, 81, 74, 55, 73, 70, 00,
	00, 00, 00, 65, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00,
	00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00,
	00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00,
	00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00,
	00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00,
	00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00,
	00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00,
	00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00, 00 };


// some texts displayed by the proc_file_system
static char *kbd_sniff[2] = { "off", "on" };
static char *kbd_sniff_mode[2] = { "passive", "active" };

#define PASSIVE 0
#define ACTIVE  1

// is the sniffer active (1) or inactive (0)
static int  SNIFFER = 0;
// do we get a copy (SNIFFMODE = PASSIVE) or do we get the original data (SNIFFMODE = ACTIVE)
// and have to reinsert the data
static int  SNIFFMODE = PASSIVE;

// we allow only one process to sniff
static int sniffer_in_use = 0;

// timeout for the keyboard sniffer -1 = blocking, otherwise timeout in msecs
static long sniffer_timeout = -1;

// the value we sniffed from the keyboard
static int sniffed_value;

static int sio_reset_flag = 0;

#define EEPROM_BANK_SIZE 16
#define MAX_EEPROM_BANKS 32

typedef struct _sio_bank {
	unchar	Data[EEPROM_BANK_SIZE];
} sio_bank;


// SSP functions vary by architecture

#define MAX_READ_SSP_CYCLES 20;

#ifdef CONFIG_ARCH_SA1100

static void send_SSP_msg(unchar *pBuf, int num)
{
	ushort	tmp;
	int		i;

	for (i=0;i<num;i++) {
		while ((Ser4SSSR & SSSR_TNF) == 0);
		tmp = pBuf[i];
		Ser4SSDR = (tmp << 8);

		// Throw away SSP echo
		while ((Ser4SSSR & SSSR_RNE) == 0);
		tmp = Ser4SSDR;
	}
}

static inline unchar get_SSP_byte(void)
{
	unchar	data;

	// send 0 to trigger a response
	Ser4SSDR = 0x00;

	// wait for rcv not empty status
	while ((Ser4SSSR & SSSR_RNE) == 0);

	// read the data
	data = (unchar) Ser4SSDR;

	return (data);
}

static unchar ReadSSPByte(void)
{
	if (Ser4SSSR & SSSR_ROR) {
		printk(KERN_WARNING "%s() : Overrun\n", __FUNCTION__);
		return 0;
	}

	return get_SSP_byte();
}


static unchar ReadSSPNonZeroByte(void)
{
	unchar	data = 0;
	int loop = MAX_READ_SSP_CYCLES;

	while (!data && loop--)
	{
		// This is just like ReadSSPByte but don't call
		// ReadSSPByte because we would loop after a ROR.
		if (Ser4SSSR & SSSR_ROR) {
			printk(KERN_WARNING "%s() : Overrun\n", __FUNCTION__);
			return 0;
		}

		data = get_SSP_byte();
	}

	return (data);
}


static void send_SSP_EEPROM_msg(sio_bank* ptr)
{
	ushort	tmp;
	int	i;

	for (i=0;i<sizeof(sio_bank);i++) {
		while ((Ser4SSSR & SSSR_TNF) == 0);
		tmp = ptr->Data[i];
		Ser4SSDR = (tmp << 8);

		while ((Ser4SSSR & SSSR_RNE) == 0);
		tmp = Ser4SSDR;

		if (tmp != ptr->Data[i])
		  printk(" sb %02x %04x\n", ptr->Data[i], tmp);
	}

	if (Ser4SSSR & SSSR_ROR)
		printk(KERN_WARNING "%s() : FIFO ROR\n", __FUNCTION__);
}

#elif defined(CONFIG_ARCH_ADSAGX)

static void send_SSP_msg(unchar *pBuf, int num)
{
	unchar	tmp;
	int	i;

	for (i=0;i<num;i++) {
		ADSAGX_CPLD_SPI_TX = pBuf[i];

		ADSAGX_CPLD_SPI_EN = ADSAGX_CPLD_SPI_ENABLE;

		udelay(1);
		// wait for transmission to complete
		while ((ADSAGX_CPLD_SPI_EN & ADSAGX_CPLD_SPI_ENABLE) != 0);

		// Throw away SSP echo
		tmp = ADSAGX_CPLD_SPI_RX;
	}
}


static inline unchar get_SSP_byte(void)
{
	unchar	data;

	// send 0 to trigger a response
	ADSAGX_CPLD_SPI_TX = 0;

	ADSAGX_CPLD_SPI_EN = ADSAGX_CPLD_SPI_ENABLE;

	udelay(1);
	// wait for transmission to complete
	while ((ADSAGX_CPLD_SPI_EN & ADSAGX_CPLD_SPI_ENABLE) != 0);

	// read the data
	data = (unchar) ADSAGX_CPLD_SPI_RX;

	return (data);
}

#define ReadSSPByte get_SSP_byte

static unchar ReadSSPNonZeroByte(void)
{
	unchar	data = 0;
	// int loop = MAX_READ_SSP_CYCLES;
	int loop = 10000;

	while (!data && --loop)
	{
		data = get_SSP_byte();
	}

	if (!loop)
		printk(KERN_WARNING "ReadSSPNonZeroByte: timeout\n");

	return (data);
}

static void send_SSP_EEPROM_msg(sio_bank* ptr)
{
	ushort	tmp;
	int	i;

	for (i=0;i<sizeof(sio_bank);i++) {
		ADSAGX_CPLD_SPI_TX = ptr->Data[i];
		
		ADSAGX_CPLD_SPI_EN = ADSAGX_CPLD_SPI_ENABLE;

		udelay(1);
		// wait for transmission to complete
		while ((ADSAGX_CPLD_SPI_EN & ADSAGX_CPLD_SPI_ENABLE) != 0);

		tmp = ADSAGX_CPLD_SPI_RX;

		if (tmp != ptr->Data[i])
		  printk(" sb %02x %04x\n", ptr->Data[i], tmp);
	}
}

#elif defined(CONFIG_ARCH_ADSBITSYX)

#define MAX_SSP_FIFO_CYCLES 20000;

#  define SSPCR0        SSCR0
#  define SSPCR1	SSCR1
#  define SSPSR		SSSR
#  define SSPDR		SSDR
// FIXME: This is sloppy - JML
#  undef  SSSR_TNF
#  undef  SSSR_RNE
#  undef  SSSR_ROR
#  define SSSR_TNF	0x00000004	// Transmit FIFO Not Full (PXA)
#  define SSSR_RNE	0x00000008	// Receive FIFO Not Empty (PXA)
#  define SSSR_ROR	0x00000080	// Receive FIFO Over-Run (PXA)

static void send_SSP_msg(unchar *pBuf, int num)
{
	ushort	tmp;
	int	i;

	for (i=0;i<num;i++) {

		udelay(50); // the AVR can't keep up with the BitsyX!

		while ((SSPSR & SSSR_TNF) == 0);
		tmp = pBuf[i];
		SSPDR = tmp;

		// Throw away SSP echo
		while ((SSPSR & SSSR_RNE) == 0);
		tmp = SSPDR;
	}
}


static inline unchar get_SSP_byte(void)
{
	unchar	data;

	udelay(50); // the AVR can't keep up with the BitsyX!

	// send 0 to trigger a response
	SSPDR = 0x00;

	// wait for rcv not empty status
	while ((SSPSR & SSSR_RNE) == 0);

	// read the data
	data = (unchar) SSPDR;

	return (data);
}

static unchar ReadSSPByte(void)
{
	if (SSPSR & SSSR_ROR) {
		printk(KERN_WARNING "%s() : Overrun\n", __FUNCTION__);
		return 0;
	}

	return get_SSP_byte();
}


static unchar ReadSSPNonZeroByte(void)
{
	unchar	data = 0;
	int loop = MAX_READ_SSP_CYCLES;

	while (!data && loop--)
	{
		// This is just like ReadSSPByte but don't call
		// ReadSSPByte because we would loop after a ROR.
		if (SSPSR & SSSR_ROR) {
			printk(KERN_WARNING "%s() : Overrun\n", __FUNCTION__);
			return 0;
		}

		data = get_SSP_byte();
	}

	return (data);
}

static void send_SSP_EEPROM_msg(sio_bank* ptr)
{
	ushort	tmp;
	int	i;

	for (i=0;i<sizeof(sio_bank);i++) {

		// this one might not be needed
		udelay(50); // the AVR can't keep up with the BitsyX!

		while ((SSPSR & SSSR_TNF) == 0);
		tmp = ptr->Data[i];

		SSPDR = tmp;

		while ((SSPSR & SSSR_RNE) == 0);
		tmp = SSPDR;

		if (tmp != ptr->Data[i])
		  printk(" sb %02x %04x\n", ptr->Data[i], tmp);
	}

	if (SSPSR & SSSR_ROR)
		printk(KERN_WARNING "%s() : FIFO ROR\n", __FUNCTION__);
}

#endif

// SSP functions common to all boards

static void flush_SSP(void)
{
         int i;
	 unchar result;
	 
	 // read 4 zeros in a row to verify that nothing is pending
	 for (i=0; i < 4; ) {
	   result = ReadSSPByte();
	   if (!result) i++;
	   else
	     printk(KERN_WARNING "SMARTIO expected nothing got: %#x instead\n", result);
	 }
}


static unchar skip_SSP_command_echo(unchar group, unchar code)
{
         int i;
	 unchar result;
	 
	 for (result=i=0; (result == 0) && (i < 64); i++) {
	   result = ReadSSPByte();
	 }

	 if (result != group) {
	   printk(KERN_WARNING "SMARTIO expected group %#x, got %#x instead (code: %#x).\n", group, result, code);
	   return result;
	 }

         result = ReadSSPByte();

	 if (result != code) {
	   printk(KERN_WARNING "SMARTIO expected code %#x, got %#x instead (group: %#x).\n", code, result, group);
	   return result;
	 }

	 return 0;
}
static ulong read_SSP_response(int num)
{
	int	i;
	ulong	ret;

	// This should really check the values of the zero, group & code
	// like i2c does.  See skip_SSP_command_echo above.

	// discard leading 0x00 and command echo 0 (command group value)
	ReadSSPByte();

	ReadSSPByte();

	// discard command echo 1 (command code value)
	ReadSSPByte();

	// data from SMARTIO
	// It assumes LSB first.
	// NOTE:Some command uses MSB first order
	ret = 0;
	for (i=0;i<num;i++) {
		ret |= ReadSSPByte() << (8*i);
	}

	return ret;
}


static ulong read_SSP_EEPROM_response(sio_bank* ptr)
{
	int     i;
	
	// discard leading 0x00 and command echo 0 (command group value)
	ReadSSPByte();
	ReadSSPByte();

	// discard command echo 1 (command code value)
	ReadSSPByte();

	// data from SMARTIO
	for (i=0;i<EEPROM_BANK_SIZE;i++) {
		ptr->Data[i] = ReadSSPByte();
	}

	return 0;
}


#define EEPROM_BANK_SIZE 16
#define MAX_EEPROM_BANKS 32

typedef	struct	t_SMARTIO_CMD {
	unchar	Group;
	unchar	Code;
	unchar  Opt[2];
}	SMARTIO_CMD;

static	SMARTIO_CMD RD_INT_CMD = { 0x83, 0x01, { 0x00, 0x00 } };
static	SMARTIO_CMD RD_KBD_CMD = { 0x83, 0x02, { 0x00, 0x00 } };
static	SMARTIO_CMD RD_XIRQ_CMD = { 0x83, 0x03, { 0x00, 0x00 } };
static	SMARTIO_CMD RD_ADC_CMD = { 0x83, 0x28, { 0x00, 0x00 } };
static	SMARTIO_CMD RD_KPD_CMD = { 0x83, 0x04, { 0x00, 0x00 } };
static	SMARTIO_CMD INIT_KBD_CMD = { 0x83, 0x31, { 0x00, 0x00 } };

static	volatile ushort	adc_value;
static int adc_completion;
static	volatile unchar	kpd_value;
static	unsigned int	kpd_timeout = 10000;			// 10000 msec

#define SIO_INT_0        0
#define SIO_INT_ADC      1
#define SIO_INT_KEYBOARD 2
#define SIO_INT_KEYPAD   3
#define SIO_INT_SPI      4
#ifdef CONFIG_SA1100_ADSBITSYPLUS
#define SIO_INT_I2CDATA  5
#define SIO_INT_I2CFAULT 6
#define SIO_INT_I2CDONE  7
#endif
#define SIO_INT_5        5
#define SIO_INT_6        6
#define SIO_INT_7        7

static  struct 	_smartio_interrupt_count {
	char	*name[8];
	ulong	bit[8];
} interrupt = {
#if defined(CONFIG_SA1100_ADSBITSYPLUS)
	{ "EXTENDED", "ADC", "KEYBOARD", "KEYPAD", "SPI", "I2CDATA", "I2CFAULT", "I2CDONE" },
#else
	{ "EXTENDED", "ADC", "KEYBOARD", "KEYPAD", "SPI", "NONE", "NONE", "NONE" },
#endif
	{ 0, 0, 0, 0, 0, 0, 0, 0 }
};

static void smartio_interrupt_task(void *data);
static void new_smartio_interrupt_task(void *data);

static struct tq_struct tq_smartio = {
		{ NULL,	NULL },		// struct list_head
		0,			// unsigned long sync
		smartio_interrupt_task,	// void (*routine)(void *)
		NULL,			// void *data
};

static struct tq_struct new_tq_smartio = {
		{ NULL,	NULL },				// struct list_head
		0,							// unsigned long sync
		new_smartio_interrupt_task,	// void (*routine)(void *)
		NULL,						// void *data
};

DECLARE_WAIT_QUEUE_HEAD(smartio_queue);
DECLARE_WAIT_QUEUE_HEAD(smartio_adc_queue);
DECLARE_WAIT_QUEUE_HEAD(smartio_kpd_queue);
DECLARE_WAIT_QUEUE_HEAD(sniffer_queue);

static DECLARE_MUTEX(keypad_sem);
static DECLARE_MUTEX(adc_sem);

#ifdef CONFIG_SA1100_ADSBITSYPLUS
DECLARE_WAIT_QUEUE_HEAD(smartio_i2c_queue);
static DECLARE_MUTEX(i2c_sem);
static int i2c_completion;
static int i2c_read_pending = 0;
#endif

static spinlock_t smartio_busy_lock = SPIN_LOCK_UNLOCKED;
static atomic_t	smartio_busy = ATOMIC_INIT(0);
static struct tq_struct *intr_task_needs_to_run = NULL;

static int f_five_pressed = 0;
static int f_seven_pressed = 0;
//static int e_null_counter = 0;
//static int f_null_counter = 0;
//static int keydown = 0;
static unchar previous_code = 0;
//static int e0 = 0;

extern void kbd_reset_kdown(void);
extern int adsbitsyplus_connector_board_rev(void);

static void smartio_extended_interrupt(void)
{
	unchar ext_intr_code;
	send_SSP_msg((unchar *) &RD_XIRQ_CMD, 2);
	ext_intr_code = read_SSP_response(1);

	switch (ext_intr_code) {

	case 0x00:
		// ignore 0 return
		break;

	case 0x81:
		// keyboard overflow inside AVR

	case 0x82:
		// keyboard overflow inside keyboard

		kbd_reset_kdown();
		send_SSP_msg((unchar *) &INIT_KBD_CMD, 2);
		skip_SSP_command_echo(INIT_KBD_CMD.Group, INIT_KBD_CMD.Code);

		break;
	}
}	
		
static void smartio_interrupt_task(void *arg)
{
	unchar	code;
	unchar	kcode;
	unsigned long flags;
	unchar  dummy;

	spin_lock_irqsave(&smartio_busy_lock, flags);
	if (atomic_read(&smartio_busy) == 1) {
		intr_task_needs_to_run = &tq_smartio;
		spin_unlock_irqrestore(&smartio_busy_lock, flags);
		return;
	}
	else {
		atomic_set(&smartio_busy, 1);
		intr_task_needs_to_run = NULL;
		spin_unlock_irqrestore(&smartio_busy_lock, flags);
	}

	/* Read SMARTIO Interrupt Status to check which Interrupt is occurred 
	 * and Clear SMARTIO Interrupt */
	send_SSP_msg((unchar *) &RD_INT_CMD, 2);

	code = (unchar) (read_SSP_response(1) & 0xFF);

	if (code & 0x01) interrupt.bit[SIO_INT_0] ++;

	// ADC resolution is 10bit (0x000 ~ 0x3FF)
	if (code & 0x02) {					// ADC Complete Interrupt
		interrupt.bit[SIO_INT_ADC] ++;
		send_SSP_msg((unchar *) &RD_ADC_CMD, 2);
		adc_value = (ushort) (read_SSP_response(2) & 0x3FF);
		adc_completion = SIO_INT_ADC;
		wake_up_interruptible(&smartio_adc_queue);
	}

#ifdef CONFIG_VT
	if (code  & 0x04) {					// Keyboard Interrupt
		interrupt.bit[SIO_INT_KEYBOARD] ++;
		/* Read Scan code */
		send_SSP_msg((unchar *) &RD_KBD_CMD, 2);
		kcode = (unchar) (read_SSP_response(1) & 0xFF);
		dummy = kcode & 0x80;
		if ((kcode == 0xE0) || (kcode == 0xE1) || (kcode == 0xF0)) {	// combined kcode
			if (kcode == 0xF0) {
				if (!previous_code) {
					kcode = 0xE0;
					previous_code = 0xF0;
				} else {
					kcode = mf_two_kbdmap[kcode & 0x7F] | dummy;
					previous_code = 0;
				}
			} else if (kcode == 0xE0) {
				if (previous_code != 0) {
					kcode = mf_two_kbdmap[kcode & 0x7F] | dummy;
					previous_code = 0;
				} else previous_code = kcode;
			} else {						// 0xE1
				if (!previous_code) {
					kcode = mf_two_kbdmap[kcode &0x7F] | dummy;
					previous_code = 0;
				} else {
					previous_code = kcode;
				}
			}
		} else {
			if (kcode == 0x03) {
				f_five_pressed = 1;
			} else if (kcode == 0x83) {
				if (f_five_pressed != 0) {
					f_five_pressed = 0;
					kcode = 0x03;
				} else if (f_seven_pressed == 0) {
					f_seven_pressed = 1;
					kcode = 2;
					dummy = 0;
				} else {
					f_seven_pressed = 0;
					kcode = 2;
				}
			}
			previous_code = 0;
			kcode &= 0x7F;
			kcode = mf_two_kbdmap[kcode] | dummy;
		}
		sniffed_value = (ushort)kcode;
		if (SNIFFER) wake_up_interruptible(&sniffer_queue);
		if (SNIFFMODE == PASSIVE) {
			handle_scancode( kcode, (kcode & 0x80) ? 0 : 1 );
			if (kcode & 0x80) {
				mdelay(10);		// this makes the whole thing a bit more stable
							// keyboard handling can be corrupted when hitting
							// thousands of keys like crazy. kbd_translate might catch up
							// with irq routine? or there is simply a buffer overflow on
							// the serial device? somehow it looses some key sequences.
							// if a break kcode is lost or coruppted the keyboard starts
							// to autorepeat like crazy and appears to hang.
							// this needs further investigations! Thomas
			}
		}
	}
#else
	if (code & 0x04) interrupt.bit[SIO_INT_KEYBOARD] ++;
#endif

	if (code & 0x08) { 					// Keypad interrupt
		interrupt.bit[SIO_INT_KEYPAD] ++;
		send_SSP_msg((unchar *) &RD_KPD_CMD, 2);
		kpd_value = (unchar) (read_SSP_response(1) & 0xFF);
		wake_up_interruptible(&smartio_kpd_queue);
	}

	if (code & 0x10) interrupt.bit[SIO_INT_SPI] ++;

	if (code & 0x20) interrupt.bit[SIO_INT_5] ++;

	if (code & 0x40) interrupt.bit[SIO_INT_6] ++;

	if (code & 0x80) interrupt.bit[SIO_INT_7] ++;

	spin_lock_irqsave(&smartio_busy_lock, flags);
	atomic_set(&smartio_busy, 0);
	spin_unlock_irqrestore(&smartio_busy_lock, flags);

	enable_irq(ADS_AVR_IRQ);

	wake_up_interruptible(&smartio_queue);
}

static void new_smartio_interrupt_task(void *arg)
{
	ulong	code;
	ulong	kcode;
	unsigned long flags;
	unchar  key[4];
	int		pressed, size, i;

	spin_lock_irqsave(&smartio_busy_lock, flags);
	if (atomic_read(&smartio_busy) == 1) {
		intr_task_needs_to_run = &new_tq_smartio;
		spin_unlock_irqrestore(&smartio_busy_lock, flags);
		return;
	}
	else {
		atomic_set(&smartio_busy, 1);
		intr_task_needs_to_run = NULL;
		spin_unlock_irqrestore(&smartio_busy_lock, flags);
	}
	/* flush any characters waiting to come out */
	flush_SSP();

	/* Read SMARTIO Interrupt Status to check which Interrupt is occurred 
	 * and Clear SMARTIO Interrupt */
	send_SSP_msg((unchar *) &RD_INT_CMD, 2);

	skip_SSP_command_echo(RD_INT_CMD.Group, RD_INT_CMD.Code);

	code = ReadSSPByte();

	if (code & 0x01) {
		interrupt.bit[SIO_INT_0] ++;                    // Check extended interrupt status word
		smartio_extended_interrupt();
	}

	// ADC resolution is 10bit (0x000 ~ 0x3FF)
	if (code & 0x02) {					// ADC Complete Interrupt
		interrupt.bit[SIO_INT_ADC] ++;
		send_SSP_msg((unchar *) &RD_ADC_CMD, 2);
		adc_value = (ushort) (read_SSP_response(2) & 0x3FF);
		adc_completion = SIO_INT_ADC;
		wake_up_interruptible(&smartio_adc_queue);
	}

#ifdef CONFIG_VT
	if (code  & 0x04) {					// Keyboard Interrupt
	  static ulong read_SSP_KBD_response(int *num);
		pressed = 1;
		interrupt.bit[SIO_INT_KEYBOARD] ++;
		/* Read Scan code */
		send_SSP_msg((unchar *) &RD_KBD_CMD, 2);
		kcode = read_SSP_KBD_response(&size);
		memcpy(key, &kcode, sizeof(kcode));
		for (i=0;i<size;i++) {
			kcode = (ulong)key[i];
			// key[] can have one, two, or three bytes
			// Trim leading 0x00 bytes
			if (kcode == 0x00) 
				continue;
			if ((kcode == 0xE0)|| (kcode == 0xE1) || (kcode == 0xF0)) {
				if (kcode == 0xF0) {
					pressed = 0;
				} else {
					kcode = mf_two_kbdmap[kcode &0x7F];
				}
			} else {
				kcode = mf_two_kbdmap[kcode];
			}
			handle_scancode( kcode, pressed );
			if (pressed) {
				mdelay(10);		// this makes the whole thing a bit more stable
							// keyboard handling can be corrupted when hitting
							// thousands of keys like crazy. kbd_translate might catch up
							// with irq routine? or there is simply a buffer overflow on
							// the serial device? somehow it looses some key sequences.
							// if a break kcode is lost or coruppted the keyboard starts
							// to autorepeat like crazy and appears to hang.
							// this needs further investigations! Thomas
			}
		} // end of for loop
	}
#else
	if (code & 0x04) interrupt.bit[SIO_INT_KEYBOARD] ++;
#endif

	if (code & 0x08) { 					// Keypad interrupt
		interrupt.bit[SIO_INT_KEYPAD] ++;
		send_SSP_msg((unchar *) &RD_KPD_CMD, 2);
		kpd_value = (unchar) (read_SSP_response(1) & 0xFF);
		wake_up_interruptible(&smartio_kpd_queue);
	}

	if (code & 0x10) {
		interrupt.bit[SIO_INT_SPI] ++;
        }

#ifdef CONFIG_SA1100_ADSBITSYPLUS
	/* I2C interrupt */
	if (code & 0x20) {
                static void read_i2c_response(void);
		interrupt.bit[SIO_INT_I2CDATA] ++;
		// clear the interrupt
		read_i2c_response();
		wake_up_interruptible(&smartio_i2c_queue);
	}

	if (code & 0x40) {
	  interrupt.bit[SIO_INT_I2CFAULT] ++;
	  i2c_completion = SIO_INT_I2CFAULT;
	  wake_up_interruptible(&smartio_i2c_queue);
	  printk(KERN_WARNING "i2c fault intr\n");
	}

	if (code & 0x80) {
	  interrupt.bit[SIO_INT_I2CDONE] ++;
	  i2c_completion = SIO_INT_I2CDONE;
	  wake_up_interruptible(&smartio_i2c_queue);
	}
#else
	if (code & 0x20) interrupt.bit[SIO_INT_5] ++;

	if (code & 0x40) interrupt.bit[SIO_INT_6] ++;

	if (code & 0x80) interrupt.bit[SIO_INT_7] ++;
#endif

	spin_lock_irqsave(&smartio_busy_lock, flags);
	atomic_set(&smartio_busy, 0);
	spin_unlock_irqrestore(&smartio_busy_lock, flags);

	enable_irq(ADS_AVR_IRQ);

	wake_up_interruptible(&smartio_queue);
}

static void smartio_sio_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
#ifdef CONFIG_VT
	kbd_pt_regs = regs;
#endif

	// *NOTE*
	// ADS SMARTIO interrupt is cleared after reading interrupt status
	// from smartio.
	// disable SMARTIO IRQ here and re-enable at samrtio_bh.
	// 11/13/00 Woojung
	disable_irq(ADS_AVR_IRQ);

	queue_task(&tq_smartio, &tq_immediate);
	mark_bh(IMMEDIATE_BH);
}

static ulong read_SSP_KBD_response(int *num)
{
	int		i;
	ulong	ret;
	unchar	byte;

	skip_SSP_command_echo(RD_KBD_CMD.Group, RD_KBD_CMD.Code);

	ret = i = 0;
	while (1) {
		byte = ReadSSPNonZeroByte();
		ret |= byte << (8*i);
		if ((byte == 0xF0) || (byte == 0xE0)) {
			i ++;
			if (i > 3) {
				// Error
				i = -1;
				ret = 0xFFFFFFFF;
				break;
			}
			else {
				continue;
			}
		}
		else
			break;
	}
	*num = (i+1);			// bytes received

	return ret;
}


// This routine is for handling new SmartIO PS/2 Keyboard driver
static void new_smartio_sio_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
#ifdef CONFIG_VT
	kbd_pt_regs = regs;
#endif

	// *NOTE*
	// ADS SMARTIO interrupt is cleared after reading interrupt status
	// from smartio. 
	// disable SMARTIO IRQ here and re-enable at samrtio_bh.
	// 11/13/00 Woojung
	disable_irq(ADS_AVR_IRQ);

	queue_task(&new_tq_smartio, &tq_immediate);
	mark_bh(IMMEDIATE_BH);
}

char smartio_kbd_unexpected_up(unsigned char keycode)
{
	/* unexpected, but this can happen: maybe this was a key release for a
	   FOCUS 9000 PF key; if we want to see it, we have to clear up_flag */
	if (keycode >= SC_LIM || keycode == 85)
		return 0;

	return	0200;
}

static ushort control_sio_backlite(int cmd, int value);
static int backlite_control_value = 0;
static int backlite_contrast_value = 0;

static inline void smartio_sio_init(void)
{
#if defined(CONFIG_ARCH_ADSBITSYX)
	// Configure the edge and direction and disable
	// any alternate function for the AVR interrupt
	set_GPIO_IRQ_edge(ADS_AVR_GPIO, GPIO_RISING_EDGE); // calls set_GPIO_mode( (ADS_AVR_GPIO | GPIO_IN) );

	// Configure the Alternate Function and 
	// Direction Registers for the SSP Pins
	set_GPIO_mode(GPIO23_SCLK_md);
	set_GPIO_mode(GPIO24_SFRM_MD);
	set_GPIO_mode(GPIO25_STXD_MD);
	set_GPIO_mode(GPIO26_SRXD_MD);

	// Enable SSP Unit Clock
	CKEN |= CKEN3_SSP;

	// Configure the SSP Serial Port
	SSPCR0  = SSCR0_Motorola | SSCR0_SerClkDiv(336) | SSCR0_DataSize(8);
	SSPSR   = SSSR_ROR;
	SSPCR1  = SSCR1_SP;
	SSPCR0 |= SSCR0_SSE;

	// Reset SMARTIO
	if (ADSBITSYX_REV < 2) {
		set_GPIO_mode( (ADS_RESETAVR | GPIO_OUT) );
		GPDR(ADS_RESETAVR) |= GPIO_bit(ADS_RESETAVR);
		GPCR(ADS_RESETAVR) |= GPIO_bit(ADS_RESETAVR);
		mdelay(300);
		GPSR(ADS_RESETAVR) |= GPIO_bit(ADS_RESETAVR);
	} else {
		ADSBITSYX_CPLD_SUPPC2 &= ~ADSBITSYX_SUPPC2_RESETAVR;
		mdelay(300);
		ADSBITSYX_CPLD_SUPPC2 |= ADSBITSYX_SUPPC2_RESETAVR;
	}

#elif defined(CONFIG_ARCH_ADSAGX)

	// Reset SMARTIO
	set_GPIO_mode( (ADS_RESETAVR | GPIO_OUT) );
	GPDR(ADS_RESETAVR) |= GPIO_bit(ADS_RESETAVR);
	GPCR(ADS_RESETAVR) |= GPIO_bit(ADS_RESETAVR);
	mdelay(300);
	GPSR(ADS_RESETAVR) |= GPIO_bit(ADS_RESETAVR);

#elif defined(CONFIG_SA1100_ADSBITSYPLUS)
        /* make sure the interrupt is input and not alternate function */
	GPDR &= ~GPIO_GPIO13; 	// Input
	GAFR &= ~GPIO_GPIO13;
	set_GPIO_IRQ_edge(GPIO_GPIO13, GPIO_RISING_EDGE);
	SSPCR0 = 0xA707;
	SSPSR  = SSSR_ROR;
	SSPCR1 = 0x0010;
	SSPCR0 = 0xA787;

	// Reset SMARTIO
	GPDR |= ADS_RESETAVR;
	GPCR |= ADS_RESETAVR;
	mdelay(300);
	GPSR |= ADS_RESETAVR;

#elif defined(CONFIG_SA1100_ADSBITSY)
        /* the Bitsy uses the GPIO Alternate pins for the SPI interfact */
	GPDR |= (GPIO_GPIO10 | GPIO_GPIO12 | GPIO_GPIO13); 	// Output
	GPDR &= ~GPIO_GPIO11;

	// Alternative Function
	GAFR |= (GPIO_GPIO10 | GPIO_GPIO11 | GPIO_GPIO12 | GPIO_GPIO13);
	SSPCR0 = 0xA707;
	SSPSR  = SSSR_ROR;
	SSPCR1 = 0x0010;
	SSPCR0 = 0xA787;

	// Reset SMARTIO
	GPDR |= ADS_RESETAVR;
	GPCR |= ADS_RESETAVR;
	mdelay(300);
	GPSR |= ADS_RESETAVR;

#endif

	// Give a delay to be stable
	mdelay(300);

	// Enable backlite control
	control_sio_backlite(SMARTIO_BL_CONTROL, backlite_control_value);
	control_sio_backlite(SMARTIO_BL_CONTRAST, backlite_contrast_value);
}

typedef struct _sio_ver {
	uint	DevVer;
	uint	DevType;
	uint	FwLevel;
} sio_ver;

static ushort read_sio_version(sio_ver *ptr);

#ifdef CONFIG_PC_KEYMAP
#  if defined(CONFIG_ARCH_ADSBITSYX)
static void smartio_power_keyboard(int power)
{
	if (power) {
		// power on PS2 keyboard
		ADSBITSYX_CPLD_PCON |= ADSBITSYX_PCON_CONN_B_PE1;
	} else {
		// power off PS2 keyboard
		ADSBITSYX_CPLD_PCON &= ~ADSBITSYX_PCON_CONN_B_PE1;
	}
}
#  elif defined(CONFIG_SA1100_ADSBITSYPLUS)
static void smartio_power_keyboard(int power)
{
	if (power) {
		if (adsbitsyplus_connector_board_rev() == 0x02) {
			// power on PS2 keyboard
			ADS_CPLD_PCON |= ADS_PCON_CONN_B_PE2;
		}
		else {
			// power on PS2 keyboard
			ADS_CPLD_PCON |= ADS_PCON_CONN_B_PE1;
		}
	} else {
		if (adsbitsyplus_connector_board_rev() == 0x02) {
			// power off PS2 keyboard
			ADS_CPLD_PCON &= ~ADS_PCON_CONN_B_PE2;
		}
		else {
			// power off PS2 keyboard
			ADS_CPLD_PCON &= ~ADS_PCON_CONN_B_PE1;
		}
	}
}
#  endif
#endif

void __init smartio_kbd_init_hw(void)
{
	sio_ver version;

#if defined(CONFIG_SA1100_ADSBITSYPLUS) || defined(CONFIG_ARCH_ADSBITSYX)
#ifdef CONFIG_PC_KEYMAP
	smartio_power_keyboard(1);
#endif
#endif

	k_setkeycode	= smartio_kbd_setkeycode;
	k_getkeycode	= smartio_kbd_getkeycode;
	k_translate	= smartio_kbd_translate;
	k_unexpected_up	= smartio_kbd_unexpected_up;
#ifdef CONFIG_MAGIC_SYSRQ
	k_sysrq_key	= 0x54;
	/* sysrq table??? --rmk */
#endif

	smartio_sio_init();

	read_sio_version(&version);

	printk(KERN_INFO "ADSmartIO ID     : %#06x\n", version.FwLevel);
	if (isprint(version.DevVer) && !isspace(version.DevVer))
		printk(KERN_INFO "Firmware Version : %c\n", version.DevVer);
	else
		printk(KERN_INFO "Firmware Version : %#x\n", version.DevVer);
	printk(KERN_INFO "Device Type      : %#x\n", version.DevType);

#if defined(CONFIG_SA1100_ADSBITSYPLUS) || defined(CONFIG_ARCH_ADSBITSYX) || defined(CONFIG_ARCH_ADSAGX)

	// setup IRQ
	if (request_irq(ADS_AVR_IRQ, new_smartio_sio_interrupt,
		0,"smartio",NULL) != 0)
		printk(KERN_WARNING "Could not allocate SMARTIO IRQ!\n");
#else
	if (version.FwLevel == 0x1011 || version.FwLevel == 0x1020 || version.FwLevel == 0x1022) {
		printk (KERN_INFO "ADS Bitsy compatible kbd driver v%s\n", smartio_version);
		if (request_irq(ADS_AVR_IRQ, smartio_sio_interrupt,
			0,"smartio",NULL) != 0)
			printk(KERN_WARNING "Could not allocate SMARTIO IRQ!\n");
	}
	else {
		printk (KERN_INFO "ADS Bitsy keyboard driver v%s\n", smartio_version);
		if (request_irq(ADS_AVR_IRQ, new_smartio_sio_interrupt,
			0,"smartio",NULL) != 0)
			printk(KERN_WARNING "Could not allocate SMARTIO IRQ!\n");
	}
#endif
	sio_reset_flag = 1;
}

/* SMARTIO ADC Interface */
#define SMARTIO_VERSION				0
#define SMARTIO_PORT_A				1
#define SMARTIO_PORT_B				2
#define SMARTIO_PORT_C				3
#define SMARTIO_PORT_D				4
#define SMARTIO_SELECT_OPTION			5
#define SMARTIO_BACKLITE			6
#define SMARTIO_KEYPAD				7
#define SMARTIO_ADC				8
#define	SMARTIO_VEE_PWM				9
#define SMARTIO_SLEEP				11
#define SMARTIO_KBD_SNIFFER			12
#define SMARTIO_EEPROM				13
#define SMARTIO_THERMISTOR			14
#define UCB1200_IO				20
#define	UCB1200_ADC0				30
#define	UCB1200_ADC1				31
#define	UCB1200_ADC2				32
#define	UCB1200_ADC3				33
#define	SMARTIO_SPI_EXT1			110			// for SPI Init
#define	SMARTIO_SPI_EXT2			111			// for SPI Write
#define	SMARTIO_SPI_EXT3			112			// for SPI Block Write
#define	SMARTIO_SPI_EXT4			113			// for SPI Block Read

static	SMARTIO_CMD CONV_ADC_CMD = { 0x80, 0x28, { 0x00, 0x00 } };
static	SMARTIO_CMD READ_PORT_CMD = { 0x82, 0x00, { 0x00, 0x00 } };

static	SMARTIO_CMD READ_DEVVER_CMD = { 0x82, 0x05, { 0x00, 0x00 } };
static	SMARTIO_CMD READ_DEVTYPE_CMD = { 0x82, 0x06, { 0x00, 0x00 } };
static	SMARTIO_CMD READ_FWLEVEL_CMD = { 0x82, 0x07, { 0x00, 0x00 } };

static	SMARTIO_CMD READ_EEPROM_CMD = { 0x82, 0x10, { 0x00, 0x00 } };

static int lock_smartio(unsigned long *flags)
{
	spin_lock_irqsave(&smartio_busy_lock, *flags);
	if (atomic_read(&smartio_busy) == 1) {
		spin_unlock_irqrestore(&smartio_busy_lock, *flags);
		interruptible_sleep_on(&smartio_queue);
	}
	else {
		atomic_set(&smartio_busy, 1);
		spin_unlock_irqrestore(&smartio_busy_lock, *flags);
	}

	return 1;
}

static int unlock_smartio(unsigned long *flags)
{
	spin_lock_irqsave(&smartio_busy_lock, *flags);
	atomic_set(&smartio_busy, 0);
	if (intr_task_needs_to_run != NULL)
	  queue_task(intr_task_needs_to_run, &tq_timer);
	spin_unlock_irqrestore(&smartio_busy_lock, *flags);

	return 1;
}

static ushort read_sio_adc(int channel)
{
	unsigned long	flags;

	if ((channel < 0) || (channel > 7))
		return 0xFFFF;

	down(&adc_sem);
	CONV_ADC_CMD.Opt[0] = (unchar) channel;

	lock_smartio(&flags);
	adc_completion = -1;

	send_SSP_msg((unchar *) &CONV_ADC_CMD, 3);
	unlock_smartio(&flags);

	wait_event_interruptible(smartio_adc_queue, adc_completion != -1);
	up(&adc_sem);

	return adc_value & 0x3FF;
}

static ushort read_sio_port(int port)
{
	unsigned long	flags;
	ushort			ret;

	if ((port < SMARTIO_PORT_A) || (port > SMARTIO_PORT_D))
		return 0xFFFF;

	READ_PORT_CMD.Code = (unchar) port;

	lock_smartio(&flags);
	send_SSP_msg((unchar *) &READ_PORT_CMD, 2);
	ret = read_SSP_response(1);
	unlock_smartio(&flags);

	return ret;
}

static ushort read_sio_kpd(void)
{
	long	timeout;

	down(&keypad_sem);
	// kpd_timeout is mSec order
	// interrupt_sleep_on_timeout is based on 10msec timer tick
	if (kpd_timeout == -1) {
		interruptible_sleep_on(&smartio_kpd_queue);
		up(&keypad_sem);
	}
	else {
		timeout = interruptible_sleep_on_timeout(&smartio_kpd_queue,
							kpd_timeout/10);
		up(&keypad_sem);
		if (timeout == 0) {
			// timeout without keypad input
			return 0xFFFF;
		}
	}
	return kpd_value;
}

static ushort read_sio_sniff(void)
{
        long    timeout;

        // kpd_timeout is mSec order
        // interrupt_sleep_on_timeout is based on 10msec timer tick
        if (sniffer_timeout == -1) {
                interruptible_sleep_on(&sniffer_queue);
        }
        else {
                timeout = interruptible_sleep_on_timeout(&sniffer_queue,
                                                                sniffer_timeout/10);
                if (timeout == 0) {
                        // timeout without keypad input
                        return -1;
                }
        }
        return (ushort)sniffed_value;
}

static ushort read_sio_version(sio_ver *ptr)
{
	unsigned long	flags;
	ushort		ret;

	// Read Device Version
	lock_smartio(&flags);
	send_SSP_msg((unchar *) &READ_DEVVER_CMD, 2);
	ret = read_SSP_response(1);
	unlock_smartio(&flags);
	ptr->DevVer = (uint)ret;
	// Read Device Type
	lock_smartio(&flags);
	send_SSP_msg((unchar *) &READ_DEVTYPE_CMD, 2);
	ret = read_SSP_response(2);
	unlock_smartio(&flags);
#if defined(CONFIG_ARCH_ADSBITSYX)
	// swap MSB & LSB
	ret = ((ret & 0xFF) << 8) | ((ret & 0xFF00) >> 8);
#endif
	ptr->DevType = (uint)ret;
	// Read Firmware Level
	lock_smartio(&flags);
	send_SSP_msg((unchar *) &READ_FWLEVEL_CMD, 2);
	ret = read_SSP_response(2);
	unlock_smartio(&flags);
	// swap MSB & LSB
	ret = ((ret & 0xFF) << 8) | ((ret & 0xFF00) >> 8);
	ptr->FwLevel = (uint)ret;

	return 0;
}

static ushort read_sio_eeprom(sio_bank *ptr, loff_t *ppos)
{
	SMARTIO_CMD     read_bank = READ_EEPROM_CMD;
	unsigned long	flags;

	// Set bank number
	if (*ppos % sizeof(sio_bank))
		return 0; // Unaligned boundary
	if (*ppos / sizeof(sio_bank) > MAX_EEPROM_BANKS)
		return 0; // EOF
	read_bank.Opt[0] = (unchar)(*ppos / sizeof(sio_bank));

	// Read EEPROM bank 
	lock_smartio(&flags);
	send_SSP_msg((unchar *) &read_bank, 3);
	read_SSP_EEPROM_response(ptr);
	unlock_smartio(&flags);

	// Move position to next bank
	*ppos += sizeof(sio_bank);

	return sizeof(sio_bank);
}

static ssize_t sio_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
	struct inode *inode = file->f_dentry->d_inode;
	unsigned int minor = MINOR(inode->i_rdev);
	ushort *ret = (ushort *)buf;
#ifdef CONFIG_UCB1200
	int             si_rtn;
#endif
#if defined(CONFIG_SA1100_ADSBITSYPLUS)
	unsigned int    ui_rtn;
#endif

	switch (minor) {
	case	SMARTIO_ADC:
			if ((*ret = read_sio_adc(buf[0])) != 0xFFFF)
				return sizeof(ushort);					// 2 bytes
			break;
	case	SMARTIO_PORT_A:
	case	SMARTIO_PORT_B:
	case	SMARTIO_PORT_C:
	case	SMARTIO_PORT_D:
			if ((*ret = read_sio_port(minor)) != 0xFFFF)
				return sizeof(ushort);
			break;
	case	SMARTIO_VERSION:
			if (count < sizeof(sio_ver))
				return 0;
			if ((read_sio_version((sio_ver *)buf)) != 0xFFFF)
				return sizeof(sio_ver);
			break;
	case	SMARTIO_KEYPAD:
			if ((*ret = read_sio_kpd()) != 0xFFFF)
				return sizeof(ushort);
			break;
	case	SMARTIO_KBD_SNIFFER:
			if ((*ret = read_sio_sniff()) != (ushort)-1)
				return 1;
			break;
	case	SMARTIO_EEPROM:
			if (count < sizeof(sio_bank))
				return 0;
			return read_sio_eeprom((sio_bank *)buf, ppos);
#if defined(CONFIG_SA1100_ADSBITSYPLUS)
			/* we have UCB compatible digital IOs on Bitsy Plus */
	case	UCB1200_IO:
                        ui_rtn = ADS_IO1_MASK & ADS_CPLD_IO1;
			ui_rtn |= (ADS_IO2_MASK & ADS_CPLD_IO2) << 8;
			*ret = (ushort)ui_rtn;
			return sizeof(ushort); // unsigned int ?
#endif
#ifdef CONFIG_UCB1200
	case	UCB1200_IO:
			*ret = (ushort)ucb1200_read_io();
			return sizeof(ushort); // unsigned int ?
#ifdef CONFIG_ADC_UCB1200
	case	UCB1200_ADC0:
	case	UCB1200_ADC1:
	case	UCB1200_ADC2:
	case	UCB1200_ADC3:
			if (file->f_flags & O_NONBLOCK)
				return -EAGAIN;
			si_rtn = ucb1200_multi_adc_read(minor - UCB1200_ADC0);
			// Error
			if (si_rtn < 0)
				return si_rtn;
			*ret = (ushort)si_rtn;
			return sizeof(*ret);
#endif
#endif
	default :
			return -ENXIO;
	}
	return -ENXIO;
}

static	SMARTIO_CMD WRITE_PORT_CMD = { 0x81, 0x00, { 0x00, 0x00 } };
static	SMARTIO_CMD SELECT_OPT_CMD = { 0x80, 0x00, { 0x00, 0x00 } };
static	SMARTIO_CMD CONTROL_BL_CMD = { 0x80, 0x00, { 0x00, 0x00 } };
static	SMARTIO_CMD CONTRAST_BL_CMD = { 0x80, 0x21, { 0x00, 0x00 } };
static	SMARTIO_CMD CONTROL_KPD_CMD = { 0x80, 0x27, { 0x00, 0x00 } };
static	SMARTIO_CMD CONTROL_VEE_CMD = { 0x80, 0x22, { 0x00, 0x00 } };
static	SMARTIO_CMD WRITE_EEPROM_CMD = { 0x81, 0x10, { 0x00, 0x00 } };
static	SMARTIO_CMD CONTROL_THERMISTOR_CMD = { 0x81, 0x09, { 0x00, 0x00 } };

static ushort write_sio_port(int port, unchar value)
{
	unsigned long	flags;

	if ((port < SMARTIO_PORT_A) || (port > SMARTIO_PORT_D)) 
		return 0xFFFF;

	WRITE_PORT_CMD.Code = (unchar) port;
	WRITE_PORT_CMD.Opt[0] = (unchar) value;

	lock_smartio(&flags);
	send_SSP_msg((unchar *) &WRITE_PORT_CMD, 3);
	unlock_smartio(&flags);

	// There's a problem with older AVR f/w that causes
	// the back light to go off when writing to Port D.
	// To fix it, the back light control value is saved
	// in the control function and restored here.
	if (SMARTIO_PORT_D == port)
		control_sio_backlite(SMARTIO_BL_CONTROL, backlite_control_value);

	return 0;
}

static ushort write_sio_select(unchar select)
{
	unsigned long	flags;

	if ((select < 1) || (select > 3))
		return 0xFFFF;

	SELECT_OPT_CMD.Code = (unchar) (select + 0x28);

	lock_smartio(&flags);
	send_SSP_msg((unchar *) &SELECT_OPT_CMD, 2);
	unlock_smartio(&flags);

	return 0;
}

static ushort control_sio_backlite(int cmd, int value)
{
#ifdef CONFIG_ARCH_ADSAGX
	if (cmd == SMARTIO_BL_CONTRAST) {
		value &= 0xFF;
		PWM_PWDUTY1 = 0xFF - value;
		backlite_contrast_value = value;
	}
	else if (cmd == SMARTIO_BL_CONTROL) {
                if (value) {
		  ADSAGX_CPLD_CR3 |= ADSAGX_CPLD_CR3_BLON;
		}
		else {
		  ADSAGX_CPLD_CR3 &= ~ADSAGX_CPLD_CR3_BLON;
		}
		backlite_control_value = value;
	}
	else
		return 0xFFFF;

	return 0;
#else
	unsigned long	flags;

	if (cmd == SMARTIO_BL_CONTRAST) {
		value &= 0xFF;
		CONTRAST_BL_CMD.Opt[0] = (unchar) value;

		lock_smartio(&flags);
		send_SSP_msg((unchar *) &CONTRAST_BL_CMD, 3);
		unlock_smartio(&flags);
		backlite_contrast_value = value;
	}
	else if (cmd == SMARTIO_BL_CONTROL) {
		if (value == 0x00) {
			// Backlite OFF
			CONTROL_BL_CMD.Code = 0x24;
		}
		else {
			// Backlite ON
			CONTROL_BL_CMD.Code = 0x23;
		}
		lock_smartio(&flags);
		send_SSP_msg((unchar *) &CONTROL_BL_CMD, 2);
		backlite_control_value = value;
		unlock_smartio(&flags);
	}
	else
		return 0xFFFF;

	return 0;
#endif
}

void sio_backlite_off(void)
{
	if (sio_reset_flag) {
		int save_contrast_value = backlite_contrast_value;
		control_sio_backlite(SMARTIO_BL_CONTROL, 0 );
		control_sio_backlite(SMARTIO_BL_CONTRAST, 0xFF );
		backlite_contrast_value = save_contrast_value;
	}
	else {
		backlite_control_value = 0;
		backlite_contrast_value = 0xFF;
	}
}

void sio_backlite_on(void)
{
	if (sio_reset_flag) {
		control_sio_backlite(SMARTIO_BL_CONTROL, 1);
		control_sio_backlite(SMARTIO_BL_CONTRAST, backlite_contrast_value );
	}
	else {
		backlite_control_value = 1;
	}
}

static ushort control_sio_keypad(int x, int y)
{
	unsigned long	flags;

	if ( (x<1) || (x>8) || (y<1) || (y>8)) {
		return 0xFFFF;
	}

	CONTROL_KPD_CMD.Opt[0] = (unchar) x;
	CONTROL_KPD_CMD.Opt[1] = (unchar) y;

	lock_smartio(&flags);
	send_SSP_msg((unchar *) &CONTROL_KPD_CMD, 4);
	unlock_smartio(&flags);

	return 0;
}

static ushort control_sio_vee(int value)
{
	unsigned long	flags;

	value &= 0xFF;

#ifdef CONFIG_ARCH_ADSAGX
	PWM_PWDUTY0 = 0xFF - value;
#else
	CONTROL_VEE_CMD.Opt[0] = (unchar) value;

	lock_smartio(&flags);
	send_SSP_msg((unchar *) &CONTROL_VEE_CMD, 3);
	unlock_smartio(&flags);
	value &= 0xFF;
#endif

	return 0;
}

#ifndef CONFIG_ARCH_ADSAGX
static ushort control_sio_thermistor(int value)
{
	unsigned long	flags;

	value &= 0xFF;
	CONTROL_THERMISTOR_CMD.Opt[0] = (unchar) value;

	lock_smartio(&flags);
	send_SSP_msg((unchar *) &CONTROL_THERMISTOR_CMD, 3);
	unlock_smartio(&flags);

	return 0;
}
#endif

static ushort write_sio_eeprom(sio_bank *ptr, loff_t *ppos)
{
	SMARTIO_CMD write_bank = WRITE_EEPROM_CMD;
	unsigned long	flags;

	// Set bank number
	if (*ppos % sizeof(sio_bank))
		return 0; // Unaligned boundary
	if (*ppos / sizeof(sio_bank) > MAX_EEPROM_BANKS)
		return 0; // EOF
	write_bank.Opt[0] = (unchar)(*ppos / sizeof(sio_bank));

	// Write EEPROM bank 
	lock_smartio(&flags);
	send_SSP_msg((unchar *) &write_bank, 3);
	send_SSP_EEPROM_msg(ptr);
	unlock_smartio(&flags);

	// Move position to next bank
	*ppos += sizeof(sio_bank);

	return sizeof(sio_bank);
}

#ifdef CONFIG_SA1100_ADSBITSYPLUS

/* This stuff is related to the RTC connected to the AVR via I2C
   Ideally there would be a separate I2C driver for the AVR and 
   then the RTC could be layered over that.  We aren't doing that
   yet... */

#include <linux/time.h>

#define I2C_MAX_DATA 32

typedef	struct	t_SMARTIO_I2C_CMD {
	unchar	Group;
	unchar	Code;
        unchar  Device;
        unchar  Count;
        unchar  data[I2C_MAX_DATA];
}	SMARTIO_I2C_CMD;

static	SMARTIO_CMD INIT_I2C_CMD =  { 0x83, 0x0A };
static	SMARTIO_CMD WRITE_I2C_CMD = { 0x83, 0x0B };
static	SMARTIO_CMD READ1_I2C_CMD = { 0x83, 0x0C };
static	SMARTIO_CMD READ2_I2C_CMD = { 0x83, 0x0D };

static int i2c_buffer_count;
static unchar i2c_buffer[I2C_MAX_DATA];

#define DS1307_DEVICE_ID 0x68

static int ads_sio_i2c_present = 1;

int ads_sio_probe(void)
{
        return ads_sio_i2c_present;
}

void ads_sio_initialize_i2c(void)
{
  send_SSP_msg((unchar *) &INIT_I2C_CMD, 2);
}

int ads_sio_write_i2c(unchar device_id, int count, unchar *data)
{
        int i, ret;
	unsigned long	flags;
	SMARTIO_I2C_CMD buffer;

	// This is the only device connected, smartio does not
	// give any error indication if we address a device that
        // does not exist.  So check here...
	if (device_id != DS1307_DEVICE_ID)
	  return -EREMOTEIO;

	// check for too big buffer
	if (count > I2C_MAX_DATA)
	  return -EMSGSIZE;

	buffer.Group =  WRITE_I2C_CMD.Group;
	buffer.Code =  WRITE_I2C_CMD.Code;
	buffer.Device = device_id;
	buffer.Count = count;

	for (i=0; i<count; i++) {
	  buffer.data[i] = data[i];
	}

	down(&i2c_sem);
	lock_smartio(&flags);
	i2c_completion = -1;
        send_SSP_msg((unchar *) &buffer, (&(buffer.data[count]) - (unchar *)(&buffer)));

	unlock_smartio(&flags);

	wait_event_interruptible(smartio_i2c_queue, i2c_completion != -1);

	if (i2c_completion == SIO_INT_I2CDONE) ret = count;
	else ret = -1;

	up(&i2c_sem);
	return ret;
}

static void read_i2c_response(void)
{
  int i;

  if (!i2c_read_pending) {
    printk(KERN_WARNING "i2c got read response interrupt, but no read was pending.\n");
    return;
  }

  send_SSP_msg((unchar *) &READ2_I2C_CMD, 2);

  skip_SSP_command_echo(READ2_I2C_CMD.Group, READ2_I2C_CMD.Code);

  i2c_buffer_count = ReadSSPByte();

  if (i2c_buffer_count > sizeof(i2c_buffer))
    printk(KERN_INFO "i2c_buffer_count too big: %d\n", i2c_buffer_count);

  for (i=0; i < i2c_buffer_count; i++) 
    if (i > sizeof(i2c_buffer))
      ReadSSPByte();
    else
      i2c_buffer[i] = ReadSSPByte();

  i2c_read_pending = 0;
  i2c_completion = SIO_INT_I2CDATA;
}

int ads_sio_read_i2c(unchar device_id, int count, unchar *data)
{
        int i;
	int ret;
	unsigned long	flags;	SMARTIO_I2C_CMD buffer;

	// This is the only device connected, smartio does not
	// give any error indication if we address a device that
        // does not exist.  So check here...
	if (device_id != DS1307_DEVICE_ID)
	  return -EREMOTEIO;

	// check for too big buffer or less than zero
	if (count > I2C_MAX_DATA + 1 || count < 0)
	  return -EMSGSIZE;

	// check for zero buffer size
	if (count == 0)
	  return 0;

	buffer.Group =  READ1_I2C_CMD.Group;
	buffer.Code =  READ1_I2C_CMD.Code;
	buffer.Device = device_id;
	buffer.Count = count;

	down(&i2c_sem);

	/* send the read command */

	lock_smartio(&flags);

	i2c_buffer_count = -1;
	i2c_completion = -1;
        i2c_read_pending = 1;
        send_SSP_msg((unchar *) &buffer, 4);

	unlock_smartio(&flags);

	/* wait for interrupt */
	wait_event_interruptible(smartio_i2c_queue, i2c_completion != -1);

	if (i2c_buffer_count != count)
	  printk(KERN_INFO "requested %d bytes, got %d bytes from I2C\n", count, i2c_buffer_count);
	for (i=0; i < i2c_buffer_count; i++) 
	  if (i < count) 
	    data[i] = i2c_buffer[i];

	if (i2c_completion == SIO_INT_I2CDATA)
	  if (count > i2c_buffer_count) ret = i2c_buffer_count;
	  else ret = count;
	else
	  ret = -1;
	
	up(&i2c_sem);
	return ret;
}

#endif

static ssize_t sio_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	struct inode *inode = file->f_dentry->d_inode;
	unsigned int minor = MINOR(inode->i_rdev);
#if defined(CONFIG_SA1100_ADSBITSYPLUS)
	unsigned int data;
#endif

	switch (minor) {
	case	SMARTIO_PORT_A:
	case	SMARTIO_PORT_B:
	case	SMARTIO_PORT_C:
	case	SMARTIO_PORT_D:
			if (write_sio_port(minor, buf[0]) != 0xFFFF)
				return 1;
			break;
	case	SMARTIO_SELECT_OPTION:
			if (write_sio_select(buf[0]) != 0xFFFF)
				return 1;
			break;
	case	SMARTIO_BACKLITE:
                        /* on off */
			if (control_sio_backlite(SMARTIO_BL_CONTROL, buf[0]) != 0xFFFF)
				return 1;
			break;
	case	SMARTIO_KEYPAD:
			if (control_sio_keypad(buf[0], buf[1]) != 0xFFFF)
				return 2;
			break;
	case	SMARTIO_VEE_PWM:
                        /* passive panel contrast */
			if (control_sio_vee(buf[0]) != 0xFFFF)
				return 1;
			break;
	case	SMARTIO_KBD_SNIFFER:
			// here are the scancodes injected
			handle_scancode((unchar)buf[0], (buf[0] & 0x80) ? 0 : 1);
			// give some time to process! File IO is a bit faster than manual typing ;-)
			mdelay(10);
			return 1;
	case	SMARTIO_EEPROM:
			if (count < sizeof(sio_bank))
				return 0;
			return write_sio_eeprom((sio_bank *)buf, ppos);
#ifndef CONFIG_ARCH_ADSAGX
	case	SMARTIO_THERMISTOR:
                        /* thermistor conversion */
			if (control_sio_thermistor(buf[0]) != 0xFFFF)
				return 1;
			break;
#endif
#if defined(CONFIG_SA1100_ADSBITSYPLUS)
			/* we have UCB compatible digital IOs on Bitsy Plus */
	case	UCB1200_IO:
			data = *(unsigned int *)buf;
			ADS_CPLD_IO1 = ADS_IO1_MASK & data;
			ADS_CPLD_IO2 = ADS_IO2_MASK & (data >> 8);
			return 1;
#endif
#ifdef CONFIG_UCB1200
	case	UCB1200_IO:
			ucb1200_write_io(*(unsigned short *)buf);
			return sizeof(unsigned short);
#endif
	default:
		return -ENXIO;
	}
	return -ENXIO;
}

static unsigned int sio_poll(struct file *file, struct poll_table_struct *wait)
{
	return 0;
}

static	SMARTIO_CMD IOCTL_PORT_CMD = { 0x81, 0x00, { 0x00, 0x00 } };

static ushort ioctl_sio_port(int port, unchar value)
{
	unsigned long	flags;

	if ((port < SMARTIO_PORT_A) || (port > SMARTIO_PORT_D))
		return 0xFFFF;

	IOCTL_PORT_CMD.Code = (unchar) port + 0x04;		// 0x05 ~ 0x08
	if (port == SMARTIO_PORT_B) {
		// Port B has 4 bits only
		IOCTL_PORT_CMD.Opt[0] = (unchar) value & 0x0F;
	}
	else
		IOCTL_PORT_CMD.Opt[0] = (unchar) value;

	lock_smartio(&flags);
	send_SSP_msg((unchar *) &IOCTL_PORT_CMD, 3);
	unlock_smartio(&flags);

	return 0;
}

static int sio_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	unsigned int minor = MINOR(inode->i_rdev);
	unchar	*buf = (unchar *)arg;
#if defined(CONFIG_SA1100_ADSBITSYPLUS)
	unsigned int dir;
#endif

	switch (minor) {
	case	SMARTIO_PORT_A:
	case	SMARTIO_PORT_B:
	case	SMARTIO_PORT_C:
	case	SMARTIO_PORT_D:
			if (cmd == SMARTIO_PORT_CONFIG) {
				if (ioctl_sio_port(minor, buf[0]) != 0xFFFF)
					return 0;
			}
			return -EINVAL;
	case	SMARTIO_SELECT_OPTION:
			if (write_sio_select(buf[0]) != 0xFFFF)
				return 0;
			return -EINVAL;
	case	SMARTIO_BACKLITE:
                        /* on off */
			if (cmd == SMARTIO_BL_CONTROL) {
				if (control_sio_backlite(SMARTIO_BL_CONTROL, buf[0]) != 0xFFFF)
					return 0;
			}
                        /* backlight brightness */
			else if (cmd == SMARTIO_BL_CONTRAST) {
				if (control_sio_backlite(SMARTIO_BL_CONTRAST, buf[0]) != 0xFFFF)
					return 0;
			}
			return -EINVAL;
	case	SMARTIO_KEYPAD:
			if (cmd == SMARTIO_KPD_TIMEOUT) {
				kpd_timeout = *(long*)buf;
				return 0;
			}
			else if (cmd == SMARTIO_KPD_SETUP) {
				if (control_sio_keypad(buf[0], buf[1]) != 0xFFFF)
					return 0;
			}
			return -EINVAL;
	case	SMARTIO_VEE_PWM:
                        /* passive panel contrast */
			if (control_sio_vee(buf[0]) != 0xFFFF)
				return 0;
			return -EINVAL;
	case	SMARTIO_KBD_SNIFFER:
			if (cmd == SMARTIO_SNIFFER_TIMEOUT) {
				sniffer_timeout = *(long*)buf;
				if (sniffer_timeout < 0) sniffer_timeout = -1;
				// the value will be devided by 10 later on
				if (!sniffer_timeout) sniffer_timeout = 10;
				return 0;
			}
			return -EINVAL;
#ifndef CONFIG_ARCH_ADSAGX
	case	SMARTIO_THERMISTOR:
                        /* thermistor conversion */
			if (control_sio_thermistor(buf[0]) != 0xFFFF)
				return 0;
			return -EINVAL;
#endif
#if defined(CONFIG_SA1100_ADSBITSYPLUS)
			/* we have UCB compatible digital IOs on Bitsy Plus */
	case	UCB1200_IO:
			dir = *(unsigned int *)buf;
			ADS_CPLD_IODR1 = ADS_IO1_MASK & dir;
			ADS_CPLD_IODR2 = ADS_IO2_MASK & (dir >> 8);
			return 0;
#endif
#ifdef CONFIG_UCB1200
	case	UCB1200_IO:
			ucb1200_init_io_direction(*(unsigned short *)buf);
			return 0;
#endif
	default:
		return -ENXIO;
	}
}

static int sio_open(struct inode *inode, struct file *file)
{
        unsigned int minor = MINOR(inode->i_rdev);

	// we open all by default. we only have a special handler for the kbd sniffer
	switch (minor) {
		case SMARTIO_KBD_SNIFFER:
			if (sniffer_in_use) return -EBUSY;
			sniffer_in_use = 1;
			SNIFFER = 1;
			// sniff in active or passive mode
			if ((file->f_flags & O_RDWR) == O_RDWR) SNIFFMODE = 1; else SNIFFMODE = 0;
			// do we have a blocking or non blocking sniffer?
			if ((file->f_flags & O_NONBLOCK) == O_NONBLOCK) sniffer_timeout = 100; else sniffer_timeout = -1;
			break;
		default:
			break;
	}
	return 0;
}

static int sio_close(struct inode *inode, struct file *file)
{
        unsigned int minor = MINOR(inode->i_rdev);

	switch (minor) {
		case SMARTIO_KBD_SNIFFER:
			SNIFFER = 0;
			SNIFFMODE = 0;
			sniffer_in_use = 0;
			break;
		default:
			break;
	}
	return 0;
}

/*
loff_t sio_llseek(struct file *file, loff_t offset, int origin)
{
	long long retval;

	switch (origin) {
		case 2:
			offset += file->f_dentry->d_inode->i_size;
			break;
		case 1:
			offset += file->f_pos;
	}
	retval = -EINVAL;
	if (offset >= 0) {
		if (offset != file->f_pos) {
			file->f_pos = offset;
			file->f_reada = 0;
			file->f_version = ++event;
		}
		retval = offset;
	}
	return retval;
}
*/

static struct file_operations sio_fops = {
	read: 		sio_read,
	write:		sio_write,
	poll:		sio_poll,
	ioctl:		sio_ioctl,
	open:		sio_open,
	release:	sio_close,
//	llseek:		sio_llseek,
};

static struct proc_dir_entry *sio_dir, *parent_dir = NULL;

#define	SMARTIO_MAJOR	58
#define	MAJOR_NR	SMARTIO_MAJOR

#define	PROC_NAME	"sio"

static int sio_read_proc(char *buf, char **start, off_t pos, int count, int *eof, void *data)
{
	char	*p = buf;
	int	i;

	p += sprintf(p, "ADS SMARTIO Status: \n");
	for (i=0;i<8;i++) {
		p += sprintf(p, "\tBit%d %8s Interrupt : %lu\n",
			i, interrupt.name[i], interrupt.bit[i]);
	}
	p += sprintf(p, "\tKeyboard Sniffer : %s, mode : %s\n", kbd_sniff[ SNIFFER ], kbd_sniff_mode [ SNIFFMODE ]);

	return (p-buf);
}

#ifdef	CONFIG_PM
static int pm_smartio_callback(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst) {
		case	PM_RESUME:
#if defined(CONFIG_SA1100_ADSBITSYPLUS) || defined(CONFIG_ARCH_ADSBITSYX)
#ifdef CONFIG_PC_KEYMAP
                        // power on PS2 keyboard
			smartio_power_keyboard(1);
#endif
#endif
			smartio_sio_init();
			enable_irq(ADS_AVR_IRQ);
			break;

		case	PM_SUSPEND:
			disable_irq(ADS_AVR_IRQ);
			// The /RESET line to the AVR
			// should stay HIGH during sleep
#ifdef CONFIG_ARCH_ADSBITSYX
			if (ADSBITSYX_REV < 2)
				PGSR0 |= GPIO_bit(ADS_RESETAVR);
#elif defined(CONFIG_SA1100)
			PGSR |= ADS_RESETAVR;
#endif

#if defined(CONFIG_SA1100_ADSBITSYPLUS) || defined(CONFIG_ARCH_ADSBITSYX)
#ifdef CONFIG_PC_KEYMAP
                        // power off PS2 keyboard
			smartio_power_keyboard(0);
#endif
#endif
			// stop any pending interrupts
			atomic_set(&smartio_busy, 0);
			intr_task_needs_to_run = NULL;

			// reset key down array
			kbd_reset_kdown();
			break;
	}

	return 0;
}
#endif

int __init sio_init(void)
{
	if (register_chrdev(MAJOR_NR, "sio", &sio_fops)) {
		printk(KERN_WARNING "smartio : unable to get major %d\n", MAJOR_NR);
		return -EINVAL;
	}
	else {
		printk(KERN_INFO "smartio driver initialized. version %s, date: %s\n",
				smartio_version, smartio_date);

		// if no keyboard is configured then the irqs still need to be set up... */
		if (sio_reset_flag != 1) {
#if defined(CONFIG_SA1100_ADSBITSYPLUS) || defined(CONFIG_ARCH_ADSBITSYX)  || defined(CONFIG_ARCH_ADSAGX)
			smartio_sio_init();

			if (request_irq(ADS_AVR_IRQ, new_smartio_sio_interrupt,0,"sio",NULL) != 0) {
				printk(KERN_WARNING "smartio : Could not allocate IRQ %d!\n", ADS_AVR_IRQ);
				return -EINVAL;
			}
#else
			sio_ver version;

			smartio_sio_init();
			read_sio_version(&version);

			if (version.FwLevel == 0x1011 || version.FwLevel == 0x1020 || version.FwLevel == 0x1022) {
			  if (request_irq(ADS_AVR_IRQ, smartio_sio_interrupt,0,"sio",NULL) != 0) {
			    printk(KERN_WARNING "smartio : Could not allocate IRQ!\n");
			    return -EINVAL;
			  }
			}
			else {
			  if (request_irq(ADS_AVR_IRQ, new_smartio_sio_interrupt,0,"sio",NULL) != 0) {
			    printk(KERN_WARNING "smartio : Could not allocate IRQ!\n");
			    return -EINVAL;
			  }
			}
#endif
		}

		if ((sio_dir = create_proc_entry(PROC_NAME, 0, parent_dir)) == NULL) {
			printk(KERN_INFO "smartio : Unable to create /proc entry\n");
			return -EINVAL;
		}
		else {
			sio_dir->read_proc = sio_read_proc;
#ifdef	CONFIG_PM
			pm_register(PM_SYS_DEV, PM_SYS_KBC, pm_smartio_callback);
#endif
		}
#ifdef CONFIG_SA1100_ADSBITSYPLUS
		ads_sio_initialize_i2c();
#endif		
	}

	return 0;
}

module_init(sio_init);
