/*
* Driver for the ADS784 touch screen drivers
*
* adapted from:
*
* Driver for the H3100/H3600/H3700 Touch Screen and other Atmel controlled devices.
*
* Copyright 2000,2001 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Original author: Charles Flynn.
*
* Substantially modified by: Andrew Christian
*                            September, 2001
*
* Changes:
*
*     12/01 Andrew Christian      Added new touchscreen filter routines
*      7/03 Robert Whaley         ADS784 touch screen driver
*/

#include <linux/module.h>
#include <linux/version.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>        /* get_user,copy_to_user */
#include <linux/string.h>
#include <linux/interrupt.h>
#include <linux/sysctl.h>
#include <linux/console.h>
#include <linux/devfs_fs_kernel.h>

#include <linux/tqueue.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/proc_fs.h>

#include <linux/kbd_ll.h>
#include <linux/apm_bios.h>
#include <linux/kmod.h>

#include <asm/hardware.h>

#include <asm/hardware/ssp.h>
#ifdef CONFIG_SA1111
#include <asm/hardware/sa1111.h>
#endif

#include <asm/irq.h>

static struct ssp_ops *sspops;
static int ads784x_irq;
static int ads784x_ignore_irq = 0;
static int ssp_baud_rate;

#ifdef	CONFIG_PM
static struct pm_dev *pm;
#endif

#ifdef CONFIG_DEVFS_FS
static devfs_handle_t devfs_ts, devfs_ts_dir, devfs_tsraw;
static int gMajor;		/* Dynamic major for now */
#endif

#define ADS784X_TS_MODULE_NAME "ads784x-ts"
#define TS_MAJOR 11

#define INCBUF(x,mod) (((x)+1) & ((mod) - 1))

#define X_SETTLE_COUNT 8
#define X_DATA_COUNT 1
#define Y_SETTLE_COUNT 4
#define Y_DATA_COUNT 1
#define DOWN_SETTLE_COUNT 0

#define ADS784X_SETTLE_X        0xd3 // S, A2, A0, D1, D0
#define ADS784X_SAMPLE_X	0xd3 // S, A2, A0, D1, D0
#define ADS784X_SETTLE_Y        0x93 // S, A0, D1, D0
#define ADS784X_SAMPLE_Y	0x90 // S, A0

#define IOC_ads784x_TS_MAGIC  'f'

#define TS_GET_CAL		_IOR(IOC_ads784x_TS_MAGIC, 10, struct ads784x_ts_calibration)
#define TS_SET_CAL		_IOW(IOC_ads784x_TS_MAGIC, 11, struct ads784x_ts_calibration)

enum ads784x_ts_minor_devices {
	TS_MINOR    = 0,
	TSRAW_MINOR = 1,
	KEY_MINOR   = 2
};

typedef struct ads784x_ts_calibration {
        int xscale;
        int xtrans;
        int yscale;
        int ytrans;
        int xyswap;
} TS_CAL;

typedef struct ads784x_ts_event {
        unsigned short down;
        unsigned short x;
        unsigned short y;
        unsigned short pad;
} TS_EVENT;

struct ads784x_ts_general_device {
	unsigned int          head, tail;        /* Position in the event buffer */
	struct fasync_struct *async_queue;       /* Asynchronous notification    */
	wait_queue_head_t     waitq;             /* Wait queue for reading       */
	struct semaphore      lock;              /* Mutex for reading            */
	unsigned int          usage_count;       /* Increment on each open       */
	unsigned int          total;             /* Total events                 */
	unsigned int          processed;
	unsigned int          dropped;  
};

#define MOUSEBUF_SIZE 32
struct ads784x_ts_mouse_device {
	struct ads784x_ts_general_device d;
	struct ads784x_ts_event          buf[MOUSEBUF_SIZE];
};

enum pen_state {
	PEN_UP = 0,
	PEN_DISCARD,
	PEN_DOWN
};

#define TS_FILTER_LENGTH 8
struct pen_data {
	enum pen_state state;
	unsigned short x[TS_FILTER_LENGTH];  // Unfiltered data points
	unsigned short y[TS_FILTER_LENGTH];
	unsigned short count;   // Number of points recorded in this "DOWN" or "DISCARD" series
	unsigned short index;   // Location in ring buffer of last stored data value
	int            last_cal_x;  // Last reported X value to the user
	int            last_cal_y;  // Last reported Y value to the user
};

struct ads784x_ts_device {
	struct pen_data               pen;
        struct ads784x_ts_calibration   cal;          /* ts calibration parameters */
	struct ads784x_ts_mouse_device  raw;
	struct ads784x_ts_mouse_device  filtered;     /* and calibrated */
};

struct ads784x_ts_device  g_touchscreen;

/* Global values */
static int            touchScreenSilenced       = 0;
static int            touchScreenSilenceOnBlank = 1;
static int            discard_initial_touch     = 1;
static int            touch_filter_delay        = 0;
static int            jitter_threshold          = 1;
static int            touch_filter[TS_FILTER_LENGTH] = { 1, 1, 1, 1, 0, 0, 0, 0 };

/* Parameters */
MODULE_PARM(discard_initial_touch, "i");
MODULE_PARM_DESC(discard_initial_touch,"Number of initial touchscreen events to discard");
MODULE_PARM(touch_filter_delay, "i");
MODULE_PARM_DESC(touch_filter_delay,"Number of values to accumulate before sending filtered events");
MODULE_PARM(touch_filter, "8i");
MODULE_PARM_DESC(touch_filter,"Filter values for touchscreen coordinates (most recent first)");
MODULE_PARM(jitter_threshold, "i");
MODULE_PARM_DESC(jitter_threshold,"Anti-jitter threshold");

MODULE_AUTHOR("Robert Whaley");
MODULE_DESCRIPTION("ADS784x Touch Screen driver");
MODULE_LICENSE("Dual BSD/GPL");

void ads784x_ts_touchpanel_event( unsigned short x, unsigned short y, int down );

static int inline pen_is_down_p(void)
{
#ifdef CONFIG_ARCH_ADSBITSYX
	if (machine_is_adsbitsyx())
		return (PB_DRR & GPIO_bit(0)) ? 0 : 1;
#endif
#ifdef CONFIG_ARCH_ADSAGX
	if (machine_is_adsagx())
		return (GPLR(4) & GPIO_bit(4)) ? 0 : 1;
#endif
	return 0;
}

static int get_ssp_sample(void)
{
	int sx = 0;
	int sy = 0;
	int down = 0;
	static int down_count = 0;
	static int last_sx = 0;
	static int last_sy = 0;
	int i;

	for (i=0; i<X_SETTLE_COUNT; i++) {
		sspops->ssp_write_word(ADS784X_SETTLE_X);
		sspops->ssp_read_word();
	}

	for (i=0; i<X_DATA_COUNT; i++) {
		sspops->ssp_write_word(ADS784X_SAMPLE_X);
		sx += (sspops->ssp_read_word() >> 4);
	}

	for (i=0; i<Y_SETTLE_COUNT; i++) {
		sspops->ssp_write_word(ADS784X_SETTLE_Y);
		sspops->ssp_read_word();
	}

	for (i=0; i<Y_DATA_COUNT; i++) {
		sspops->ssp_write_word(ADS784X_SAMPLE_Y);
		sy += (sspops->ssp_read_word() >> 4);
	}

	for (i=0; i<DOWN_SETTLE_COUNT; i++) {
		sspops->ssp_write_word(ADS784X_SAMPLE_X);
		sspops->ssp_read_word();
	}

	down = pen_is_down_p();

	if (down) {

		down_count++;
		sx = sx / X_DATA_COUNT;
		sy = sy / Y_DATA_COUNT;

		if (down_count > 2) {
			sx = last_sx + (sx - last_sx) / 2;
			sy = last_sy + (sy - last_sy) / 4;
			ads784x_ts_touchpanel_event(last_sx, last_sy, down);
		}

		last_sx = sx;
		last_sy = sy;
	}

	else {
		ads784x_ts_touchpanel_event(0, 0, 0);
		last_sx = 0;
		last_sy = 0;
		down_count = 0;
	}
		
	return down;
}

#define TS_FREQ		(HZ / 40)
	
static struct timer_list timer;

static void sample_timer(unsigned long data);

static int start_sample_timer(void)
{
	init_timer(&timer);
	timer.function = sample_timer;
	timer.expires = jiffies + TS_FREQ;
	add_timer(&timer);

	return 0;
}

static void sample_timer(unsigned long data)
{
	int down;

	// Get a sample and queue it
	down = get_ssp_sample();

	// If the pen is not up, continue sampling
	if (down)
		start_sample_timer();
	else {
		ads784x_ignore_irq = 0;
		enable_irq(ads784x_irq);
	}
		
}


// INTERRUPT SERVICE ROUTINE ---------------------------------------------------

static void ads784x_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	if ((!ads784x_ignore_irq) && pen_is_down_p()) {
		ads784x_ignore_irq = 1;
		disable_irq(ads784x_irq);
		start_sample_timer();
	}
}

static void ads784x_ts_add_queue( struct ads784x_ts_mouse_device *dev, 
				unsigned short x, unsigned short y, int down )
{
	struct ads784x_ts_event *event = &dev->buf[dev->d.head];
	unsigned int          nhead = INCBUF( dev->d.head, MOUSEBUF_SIZE );
	
	/* Store the character only if there is room */
	dev->d.total++;
	if ( nhead != dev->d.tail ) {
		event->x        = x;
		event->y        = y;
		event->down = down;

		dev->d.head = nhead;

		if ( dev->d.async_queue )
			kill_fasync( &dev->d.async_queue, SIGIO, POLL_IN );

		wake_up_interruptible( &dev->d.waitq );
		dev->d.processed++;
	}
	else
		dev->d.dropped++;
}


int ads784x_ts_apply_filter( unsigned short data[] )
{
	struct pen_data *pen = &g_touchscreen.pen;

	int i;
	unsigned long data_sum = 0;
	unsigned long filter_sum = 0;
	
	for ( i = 0 ; i < TS_FILTER_LENGTH && i < pen->count; i++ ) {
		int index = ( pen->index - i + TS_FILTER_LENGTH ) % TS_FILTER_LENGTH;
		filter_sum += touch_filter[i];
		data_sum += touch_filter[i] * data[index];
	}
	if ( filter_sum <= 0 ) {
		printk(KERN_ERR "%s: unable to apply imaginary filter\n", __FUNCTION__);
		return data[pen->index];
	}
	return data_sum / filter_sum;
}


void ads784x_ts_touchpanel_event( unsigned short x, unsigned short y, int down )
{
	struct pen_data *pen = &g_touchscreen.pen;
	struct ads784x_ts_calibration *cal = &g_touchscreen.cal;

	if ( touchScreenSilenced )
		return;

	switch ( pen->state ) {
	case PEN_UP: 
		if ( !down )
			return;        // No events to report

		if ( discard_initial_touch > 0 ) {
			pen->state = PEN_DISCARD;
			pen->count = 1;
			return;        // No events to report
		}

		pen->state = PEN_DOWN;
		pen->count = 1;
		break;

	case PEN_DISCARD:
		if ( !down ) {
			pen->state = PEN_UP;
			return;        // No events to report
		}

		pen->count++;
		if ( pen->count <= discard_initial_touch )
			return;

		pen->state = PEN_DOWN;
		pen->count = 1;
		break;

	case PEN_DOWN:
		if ( !down )
			pen->state = PEN_UP;
		else
			pen->count++;
		break;
	}

	// If I get this far, I need to record an UP or DOWN raw event
	// (and possibly an up/down filtered event)
	ads784x_ts_add_queue( &g_touchscreen.raw, x, y, down ); 

	pen->index = (pen->index + 1) % TS_FILTER_LENGTH;
	pen->x[pen->index] = x;
	pen->y[pen->index] = y;

	// Only process and record filtered pen up/down events if we have exceeded our delay
	// This could be an up or a down event
	if ( pen->count > touch_filter_delay ) {
		int x1 = ((ads784x_ts_apply_filter(pen->x) * cal->xscale)>>8) + cal->xtrans;
		int y1 = ((ads784x_ts_apply_filter(pen->y) * cal->yscale)>>8) + cal->ytrans;

		x = ( x1 < 0 ? 0 : x1 );
		y = ( y1 < 0 ? 0 : y1 );
		
		// Anti-jitter function
		if ( down ) {
			if ( pen->count > touch_filter_delay + 1 ) {
				int dx = (x < pen->last_cal_x ? pen->last_cal_x - x : x - pen->last_cal_x);
				int dy = (y < pen->last_cal_y ? pen->last_cal_y - y : y - pen->last_cal_y);
				if ( (dx + dy) <= jitter_threshold )
					return;
			}
			pen->last_cal_x = x;
			pen->last_cal_y = y;
		}

		ads784x_ts_add_queue( &g_touchscreen.filtered, x, y, down );
	}
}

/*    
      Called whenever the touchscreen is blanked
*/

static void ads784x_ts_reset_filters( void )
{
	if (0) printk("%s\n", __FUNCTION__);
	ads784x_ts_touchpanel_event(0,0,0);   // Send a "mouse up"
}

void ads784x_ts_blank_helper(int blank)
{
	if (0) printk("  " "%s: blank=%d interrupt=%d\n", __FUNCTION__, blank, in_interrupt());

	if ( blank ) {
		touchScreenSilenced = touchScreenSilenceOnBlank;
	}
	else { 
		if ( touchScreenSilenced ) {
			touchScreenSilenced = 0;
			ads784x_ts_reset_filters();
		}
	}
}

/***********************************************************************************/
/*      File operations interface                                                  */
/***********************************************************************************/

static int ads784x_ts_ioctl(struct inode * inode, struct file *filp,
		       unsigned int cmd , unsigned long arg)
{
	int retval = 0;

        if (0) printk("%s: cmd=%x\n", __FUNCTION__,cmd);

        switch (cmd) {

	case TS_GET_CAL:
                if ( copy_to_user((void *)arg, &g_touchscreen.cal, 
				  sizeof(struct ads784x_ts_calibration)))
			retval = -EFAULT;
                break;
	case TS_SET_CAL:
		if ( copy_from_user(&g_touchscreen.cal, (void *) arg, 
				    sizeof(struct ads784x_ts_calibration)))
			retval = -EFAULT;
		break;
	default:
		retval = -ENOIOCTLCMD;
		break;
	}

        return retval;
}


#define ADS784X_READ_WAIT_FOR_DATA \
   do { \
	if (down_interruptible(&dev->d.lock))      \
		return -ERESTARTSYS;                 \
	while ( dev->d.head == dev->d.tail ) {   \
		up(&dev->d.lock);                  \
		if ( filp->f_flags & O_NONBLOCK )    \
			return -EAGAIN;              \
		if ( wait_event_interruptible( dev->d.waitq, (dev->d.head != dev->d.tail) ) )  \
			return -ERESTARTSYS;                \
		if ( down_interruptible(&dev->d.lock))    \
			return -ERESTARTSYS;                \
	} \
   } while (0)


static ssize_t ads784x_ts_read_touchscreen(struct file *filp, char *buf, size_t count, loff_t *f_pos)
{
	struct ads784x_ts_mouse_device *dev = (struct ads784x_ts_mouse_device *) filp->private_data;

	if (count < sizeof(struct ads784x_ts_event))
		return -EINVAL;

	ADS784X_READ_WAIT_FOR_DATA;

	if ( copy_to_user(buf, &dev->buf[dev->d.tail], sizeof(struct ads784x_ts_event)) ) {
		up(&dev->d.lock);
		return -EFAULT;
	}

	dev->d.tail = INCBUF(dev->d.tail, MOUSEBUF_SIZE);
	
	up(&dev->d.lock);
	return sizeof(struct ads784x_ts_event);
}

static int ads784x_ts_fasync(int fd, struct file *filp, int mode)
{
	struct ads784x_ts_general_device *dev = (struct ads784x_ts_general_device *) filp->private_data;
	if (0) printk("%s: mode %x\n", __FUNCTION__, mode );
	return fasync_helper(fd, filp, mode, &dev->async_queue);
}

static unsigned int ads784x_ts_poll( struct file * filp, poll_table *wait )
{
	struct ads784x_ts_general_device *dev = (struct ads784x_ts_general_device *) filp->private_data;
	poll_wait(filp, &dev->waitq, wait);
	return (dev->head == dev->tail ? 0 : (POLLIN | POLLRDNORM));
}

static int ads784x_ts_open( struct inode * inode, struct file * filp)
{
        unsigned int minor = MINOR( inode->i_rdev );   /* Extract the minor number */
	struct ads784x_ts_general_device *dev;

#ifndef CONFIG_DEVFS_FS
	if ( !filp->private_data ) {
		switch (minor) {
		case TS_MINOR:
			filp->private_data = &g_touchscreen.filtered;
			break;
		case TSRAW_MINOR:
			filp->private_data = &g_touchscreen.raw;
			break;
		}
	}
#endif

	dev = (struct ads784x_ts_general_device *) filp->private_data;

	if ( dev->usage_count++ == 0 )  /* We're the first open - clear the buffer */
		dev->tail = dev->head;

	if (0) printk("%s: usage=%d\n", __FUNCTION__, dev->usage_count);

	MOD_INC_USE_COUNT;
	return 0;
}

static int ads784x_ts_release(struct inode * inode, struct file * filp)
{
	struct ads784x_ts_general_device *dev = (struct ads784x_ts_general_device *) filp->private_data;

	dev->usage_count--;

	if (0) printk("%s: usage=%d\n", __FUNCTION__, dev->usage_count);

	filp->f_op->fasync( -1, filp, 0 );  /* Remove ourselves from the async list */
	MOD_DEC_USE_COUNT;
        return 0;
}

struct file_operations ts_fops = {
	read:           ads784x_ts_read_touchscreen,
        poll:           ads784x_ts_poll,
	ioctl:		ads784x_ts_ioctl,
        fasync:         ads784x_ts_fasync,
	open:		ads784x_ts_open,
	release:	ads784x_ts_release,
};

#ifdef CONFIG_DEVFS_FS
static int ads784x_ts_open_generic(struct inode * inode, struct file * filp)
{
        unsigned int minor = MINOR( inode->i_rdev );   /* Extract the minor number */
	int result;

	if ( minor > 2 ) {
		printk("%s: bad minor = %d\n", __FUNCTION__, minor );
		return -ENODEV;
	}

        if (0) printk("%s: minor=%d\n", __FUNCTION__,minor);

	if ( !filp->private_data ) {
		switch (minor) {
		case TS_MINOR:
			filp->private_data = &g_touchscreen.filtered;
			filp->f_op = &ts_fops;
			break;
		case TSRAW_MINOR:
			filp->private_data = &g_touchscreen.raw;
			filp->f_op = &ts_fops;
			break;
		}
	}

	result = filp->f_op->open( inode, filp );
	if ( !result ) 
		return result;

	return 0;
}

struct file_operations generic_fops = {
	open:     ads784x_ts_open_generic
};
#endif

/***********************************************************************************/
/*   Proc filesystem interface                                                     */
/***********************************************************************************/

static struct ctl_table ads784x_ts_table[] = 
{
	{2, "calibration", &g_touchscreen.cal, sizeof(g_touchscreen.cal), 
	 0600, NULL, &proc_dointvec },
        {4, "silenced", &touchScreenSilenced, sizeof(touchScreenSilenced), 
	 0600, NULL, &proc_dointvec },
        {5, "silenceOnBlank", &touchScreenSilenceOnBlank, sizeof(touchScreenSilenceOnBlank), 
	 0600, NULL, &proc_dointvec },
	{7, "discard_initial_touch", &discard_initial_touch, sizeof(discard_initial_touch),
	 0666, NULL, &proc_dointvec },
	{8, "touch_filter_delay", &touch_filter_delay, sizeof(touch_filter_delay),
	 0666, NULL, &proc_dointvec },
	{9, "touch_filter", &touch_filter, sizeof(touch_filter),
	 0666, NULL, &proc_dointvec },
	{10, "jitter_threshold", &jitter_threshold, sizeof(jitter_threshold),
	 0666, NULL, &proc_dointvec },
	{0}
};

static struct ctl_table ads784x_ts_dir_table[] =
{
	{11, "ts", NULL, 0, 0555, ads784x_ts_table},
	{0}
};

static struct ctl_table_header *ads784x_ts_sysctl_header = NULL;

#define exproc(x) g_touchscreen.x.d.total, g_touchscreen.x.d.processed, g_touchscreen.x.d.dropped

static int ads784x_ts_proc_read(char *page, char **start, off_t off,
			      int count, int *eof, void *data)
{
	char *p = page;
	int len;

	p += sprintf(p, "                 Total  Processed Dropped\n");
	p += sprintf(p, "TS Raw      : %8d %8d %8d\n", exproc(raw));
	p += sprintf(p, "   Filtered : %8d %8d %8d\n", exproc(filtered));

	len = (p - page) - off;
	if (len < 0)
		len = 0;

	*eof = (len <= count) ? 1 : 0;
	*start = page + off;

	return len;
}

/***********************************************************************************/
/*       Initialization                                                            */
/***********************************************************************************/

static int ads784x_ts_init_calibration( void )
{
	if ( machine_is_adsbitsyx() ) {
                g_touchscreen.cal.xscale = (1 << 8);
                g_touchscreen.cal.xtrans = 0;
                g_touchscreen.cal.yscale = (1 << 8);
                g_touchscreen.cal.ytrans = 0;
        }

	if ( machine_is_adsagx() ) {
                g_touchscreen.cal.xscale = (1 << 8);
                g_touchscreen.cal.xtrans = 0;
                g_touchscreen.cal.yscale = (1 << 8);
                g_touchscreen.cal.ytrans = 0;
        }

        return 0;
}


void ads784x_ts_init_device( struct ads784x_ts_general_device *dev )
{
	dev->head = 0;
	dev->tail = 0;
	init_waitqueue_head( &dev->waitq );
	init_MUTEX( &dev->lock );
	dev->async_queue = NULL;
}

static int ads784x_ts_hw_init(void)
{
	int ret;

#ifdef CONFIG_ARCH_ADSBITSYX
	if (machine_is_adsbitsyx()) {
		ADSBITSYX_CPLD_SUPPC &= ~ADSBITSYX_SUPPC_CB_SPI_SEL;
		ADSBITSYX_CPLD_SUPPC |= ADSBITSYX_SUPPC_TS_SPI_SEL;
		SKPEN0 &= ~1;
		PB_DDR |= GPIO_bit(0);
		sspops = &sa1111_ssp_ops;
		ads784x_irq = IRQ_GPBIN0;
		ssp_baud_rate = 450000;
	}
#endif
#ifdef CONFIG_ARCH_ADSAGX
	if (machine_is_adsagx()) {
		sspops = &pxa_ssp_ops;
		set_GPIO_IRQ_edge(4, GPIO_FALLING_EDGE);
		ads784x_irq = IRQ_GPIO(4);
		ssp_baud_rate = 450000;
	}
#endif

	ret = sspops->ssp_init(0, 16, ssp_baud_rate, SSCR0_National);
	if (ret) {
		printk("%s: ssp_init failed %d\n", __FUNCTION__, ret);
		return ret;
	}

	if ((ret = request_irq (ads784x_irq, ads784x_interrupt, 0, ADS784X_TS_MODULE_NAME, NULL))) {
		printk("%s: request_irq failed. irq: %d\n", __FUNCTION__, ads784x_irq);
		sspops->ssp_exit();
		return ret;
	}

	return ret;
}

#ifdef	CONFIG_PM
static int pm_ads784x_callback(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	switch (rqst)
	{
		case	PM_RESUME:

			ads784x_ts_hw_init();
			// ads784x_ts_init_device(&g_touchscreen.raw.d);
			// ads784x_ts_init_device(&g_touchscreen.filtered.d);
			ads784x_ts_reset_filters();
			
			break;

		case	PM_SUSPEND:

			free_irq(ads784x_irq, NULL);
			del_timer(&timer);
			sspops->ssp_exit();

			break;
	}
	return 0;
}
#endif



int __init ads784x_ts_init_module(void)
{
	int ret;

	ret = ads784x_ts_hw_init();

        printk("%s: registering char device\n", __FUNCTION__);

#ifdef CONFIG_DEVFS_FS
        gMajor = devfs_register_chrdev(0, ADS784X_TS_MODULE_NAME, &generic_fops);
        if (gMajor < 0) {
                printk("%s: can't get major number\n", __FUNCTION__);
                return gMajor;
        }

        devfs_ts_dir = devfs_mk_dir(NULL, "touchscreen", NULL);
	if ( !devfs_ts_dir ) return -EBUSY;

        devfs_ts     = devfs_register( devfs_ts_dir, "0", DEVFS_FL_DEFAULT,
				       gMajor, TS_MINOR, 
				       S_IFCHR | S_IRUSR | S_IWUSR, 
				       &ts_fops, &g_touchscreen.filtered );
        devfs_tsraw  = devfs_register( devfs_ts_dir, "0raw", DEVFS_FL_DEFAULT,
				       gMajor, TSRAW_MINOR, 
				       S_IFCHR | S_IRUSR | S_IWUSR, 
				       &ts_fops, &g_touchscreen.raw );
#else
	register_chrdev(TS_MAJOR, ADS784X_TS_MODULE_NAME, &ts_fops);
#endif

#ifdef	CONFIG_PM
	pm = pm_register(PM_SYS_DEV, PM_SYS_UNKNOWN, pm_ads784x_callback);
#endif

	ads784x_ts_init_device(&g_touchscreen.raw.d);
	ads784x_ts_init_device(&g_touchscreen.filtered.d);

	ads784x_ts_sysctl_header = register_sysctl_table(ads784x_ts_dir_table, 0);
	create_proc_read_entry("ts", 0, NULL, ads784x_ts_proc_read, NULL);

	ads784x_ts_init_calibration();
	ads784x_ts_reset_filters();

	return 0;
}

void ads784x_ts_cleanup_module(void)
{
	printk("%s: shutting down touchscreen\n", __FUNCTION__);

	del_timer(&timer);

	free_irq(ads784x_irq, NULL);

        flush_scheduled_tasks();

	remove_proc_entry("ts", NULL);
        unregister_sysctl_table(ads784x_ts_sysctl_header);

#ifdef CONFIG_DEVFS_FS
        devfs_unregister(devfs_ts);
        devfs_unregister(devfs_tsraw);
	devfs_unregister(devfs_ts_dir);

	devfs_unregister_chrdev(gMajor, ADS784X_TS_MODULE_NAME);
#else
	unregister_chrdev(TS_MAJOR, ADS784X_TS_MODULE_NAME);
#endif

#ifdef CONFIG_PM
	pm_unregister(pm);
#endif
	sspops->ssp_exit();
}

module_init(ads784x_ts_init_module);
module_exit(ads784x_ts_cleanup_module);
