/*
 *  linux/arch/arm/mach-pxa/sa1111_ssp.c
 *
 *  Copyright (C) 2003 Russell King.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Generic SSP driver.  This provides the generic core for simple
 *  IO-based SSP applications.
 *
 *  Adapted for SA1111 SSP by Robert Whaley July, 2003
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/init.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/hardware/ssp.h>
#include <asm/hardware/sa1111.h>

#undef SSCR0
#undef SSCR1
#undef SSSR
#undef SSDR
#undef SSSR_ROR
#undef SSSR_TNF
#undef SSSR_RNE
#undef SSSR_BSY

#define SSCR0  __CCREG( 0x0800 )
#define SSCR1  __CCREG( 0x0804 )
#define SSSR   __CCREG( 0x0810 )
#define SSDR   __CCREG( 0x0840 )
#define SSSR_ROR 0x80
#define SSSR_TNF 0x04
#define SSSR_RNE 0x08
#define SSSR_BSY 0x10

static void sa1111_ssp_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned int status = SSSR;

	if (status & SSSR_ROR) {
		printk(KERN_WARNING "SSP: receiver overrun\n");
	}

	SSSR = SSSR_ROR;
}

/**
 * sa1111_ssp_write_word - write a word to the SSP port
 * @data: 16-bit, MSB justified data to write.
 *
 * Wait for a free entry in the SSP transmit FIFO, and write a data
 * word to the SSP port.
 *
 * The caller is expected to perform the necessary locking.
 *
 * Returns:
 *   %-ETIMEDOUT	timeout occurred (for future)
 * 0 success */
static int sa1111_ssp_write_word(u16 data)
{
	while (!(SSSR & SSSR_TNF))
		cpu_relax();

	SSDR = data;

	return 0;
}

/**
 * sa1111_ssp_read_word - read a word from the SSP port
 *
 * Wait for a data word in the SSP receive FIFO, and return the
 * received data.  Data is LSB justified.
 *
 * Note: Currently, if data is not expected to be received, this
 * function will wait for ever.
 *
 * The caller is expected to perform the necessary locking.
 *
 * Returns:
 *   %-ETIMEDOUT	timeout occurred (for future)
 *   16-bit data	success
 */
static int sa1111_ssp_read_word(void)
{
	while (!(SSSR & SSSR_RNE))
		cpu_relax();

	return SSDR;
}

/**
 * sa1111_ssp_flush - flush the transmit and receive FIFOs
 *
 * Wait for the SSP to idle, and ensure that the receive FIFO
 * is empty.
 *
 * The caller is expected to perform the necessary locking.
 */
static void sa1111_ssp_flush(void)
{
	do {
		while (SSSR & SSSR_RNE) {
			(void) SSDR;
		}
	} while (SSSR & SSSR_BSY);
}

/**
 * sa1111_ssp_enable - enable the SSP port
 *
 * Turn on the SSP port.
 */
static void sa1111_ssp_enable(void)
{
	SSCR0 |= SSCR0_SSE;
}

/**
 * sa1111_ssp_disable - shut down the SSP port
 *
 * Turn off the SSP port, optionally powering it down.
 */
static void sa1111_ssp_disable(void)
{
	SSCR0 &= ~SSCR0_SSE;
}

/**
 * sa1111_ssp_save_state - save the SSP configuration
 * @ssp: pointer to structure to save SSP configuration
 *
 * Save the configured SSP state for suspend.
 */
static void sa1111_ssp_save_state(struct ssp_state *ssp)
{
	ssp->cr0 = SSCR0;
	ssp->cr1 = SSCR1;

	SSCR0 &= ~SSCR0_SSE;
}

/**
 * sa1111_ssp_restore_state - restore a previously saved SSP configuration
 * @ssp: pointer to configuration saved by ssp_save_state
 *
 * Restore the SSP configuration saved previously by sa1111_ssp_save_state.
 */
static void sa1111_ssp_restore_state(struct ssp_state *ssp)
{
	SSSR = SSSR_ROR;

	SSCR0 = ssp->cr0 & ~SSCR0_SSE;
	SSCR1 = ssp->cr1;
	SSCR0 = ssp->cr0;
}

/**
 * sa1111_ssp_init - setup the SSP port
 *
 * initialise and claim resources for the SSP port.
 *
 * Returns:
 *   %-ENODEV	if the SSP port is unavailable
 *   %-EBUSY	if the resources are already in use
 *   %0		on success
 */
static int sa1111_ssp_init(int unit, int data_size, int baud_rate, int format)
{
	int ret;
	int div = 3686400 / baud_rate;

	// Enable SSP Controller Clock
	SKPCR |= SKPCR_SCLKEN;

	// Configure the SSP Serial Port
	// National Microwire - 450k baud - 16 bit
	SSCR0  = format | SSCR0_SerClkDiv(div) | SSCR0_DataSize(data_size);
	SSSR   = SSSR_ROR;
	SSCR1 &= ~(SSCR1_RIE | SSCR1_TIE | SSCR1_LBM | SSCR1_ECS);
	SSCR1 |= (0x77<<7);
	SSCR0 |= SSCR0_SSE;

	ret = request_irq(SSPROR, sa1111_ssp_interrupt, 0, "SA1111-SSP", NULL);
	if (ret)
		printk("%s failed request_irq\n", __FUNCTION__);

	return ret;
}

/**
 * sa1111_ssp_exit - undo the effects of sa1111_ssp_init
 *
 * release and free resources for the SSP port.
 */
static void sa1111_ssp_exit(void)
{
	SSCR0 &= ~SSCR0_SSE;

	SKPCR &= ~SKPCR_SCLKEN;

	free_irq(SSPROR, NULL);
}


struct ssp_ops sa1111_ssp_ops = {
	ssp_write_word: sa1111_ssp_write_word,
	ssp_read_word: sa1111_ssp_read_word,
	ssp_flush: sa1111_ssp_flush,
	ssp_enable: sa1111_ssp_enable,
	ssp_disable: sa1111_ssp_disable,
	ssp_save_state: sa1111_ssp_save_state,
	ssp_restore_state: sa1111_ssp_restore_state,
	ssp_init: sa1111_ssp_init,
	ssp_exit: sa1111_ssp_exit};
		
MODULE_AUTHOR("Robert Whaley");
MODULE_DESCRIPTION("SA1111 SSP PIO driver");
MODULE_LICENSE("GPL");

EXPORT_SYMBOL(sa1111_ssp_ops);
