/*
 *  linux/arch/arm/mach-pxa/pxa_ssp.c
 *
 *  Copyright (C) 2003 Russell King.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Generic SSP driver.  This provides the generic core for simple
 *  IO-based SSP applications.
 *
 *  Adapted for PXA SSP by Robert Whaley July, 2003
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/init.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/hardware/ssp.h>

static void pxa_ssp_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned int status = SSSR;

	if (status & SSSR_ROR) {
		printk(KERN_WARNING "SSP: receiver overrun\n");
	}

	SSSR = SSSR_ROR;
}

/**
 * pxa_ssp_write_word - write a word to the SSP port
 * @data: 16-bit, MSB justified data to write.
 *
 * Wait for a free entry in the SSP transmit FIFO, and write a data
 * word to the SSP port.
 *
 * The caller is expected to perform the necessary locking.
 *
 * Returns:
 *   %-ETIMEDOUT	timeout occurred (for future)
 * 0 success */
static int pxa_ssp_write_word(u16 data)
{
	while (!(SSSR & SSSR_TNF))
		cpu_relax();
	SSDR = data;

	return 0;
}

/**
 * pxa_ssp_read_word - read a word from the SSP port
 *
 * Wait for a data word in the SSP receive FIFO, and return the
 * received data.  Data is LSB justified.
 *
 * Note: Currently, if data is not expected to be received, this
 * function will wait for ever.
 *
 * The caller is expected to perform the necessary locking.
 *
 * Returns:
 *   %-ETIMEDOUT	timeout occurred (for future)
 *   16-bit data	success
 */
static int pxa_ssp_read_word(void)
{
	while (!(SSSR & SSSR_RNE))
		cpu_relax();

	return SSDR;
}

/**
 * pxa_ssp_flush - flush the transmit and receive FIFOs
 *
 * Wait for the SSP to idle, and ensure that the receive FIFO
 * is empty.
 *
 * The caller is expected to perform the necessary locking.
 */
static void pxa_ssp_flush(void)
{
	do {
		while (SSSR & SSSR_RNE) {
			(void) SSDR;
		}
	} while (SSSR & SSSR_BSY);
}

/**
 * pxa_ssp_enable - enable the SSP port
 *
 * Turn on the SSP port.
 */
static void pxa_ssp_enable(void)
{
	SSCR0 |= SSCR0_SSE;
}

/**
 * pxa_ssp_disable - shut down the SSP port
 *
 * Turn off the SSP port, optionally powering it down.
 */
static void pxa_ssp_disable(void)
{
	SSCR0 &= ~SSCR0_SSE;
}

/**
 * pxa_ssp_save_state - save the SSP configuration
 * @ssp: pointer to structure to save SSP configuration
 *
 * Save the configured SSP state for suspend.
 */
static void pxa_ssp_save_state(struct ssp_state *ssp)
{
	ssp->cr0 = SSCR0;
	ssp->cr1 = SSCR1;

	SSCR0 &= ~SSCR0_SSE;
}

/**
 * pxa_ssp_restore_state - restore a previously saved SSP configuration
 * @ssp: pointer to configuration saved by ssp_save_state
 *
 * Restore the SSP configuration saved previously by pxa_ssp_save_state.
 */
static void pxa_ssp_restore_state(struct ssp_state *ssp)
{
	SSSR = SSSR_ROR;

	SSCR0 = ssp->cr0 & ~SSCR0_SSE;
	SSCR1 = ssp->cr1;
	SSCR0 = ssp->cr0;
}

/**
 * pxa_ssp_init - setup the SSP port
 *
 * initialise and claim resources for the SSP port.
 *
 * Returns:
 *   %-ENODEV	if the SSP port is unavailable
 *   %-EBUSY	if the resources are already in use
 *   %0		on success
 */
static int pxa_ssp_init(int unit, int data_size, int baud_rate, int format)
{
	int ret;
	int div = 3686400 / baud_rate;

	// Enable SSP Controller Clock
	set_GPIO_mode(GPIO23_SCLK_md);
	set_GPIO_mode(GPIO24_SFRM_MD);
	set_GPIO_mode(GPIO25_STXD_MD);
	set_GPIO_mode(GPIO26_SRXD_MD);

	// Enable SSP Unit Clock
	CKEN |= CKEN3_SSP;

	// Configure the SSP Serial Port
	// National Microwire - 450k baud - 16 bit
	SSCR0  = format | SSCR0_SerClkDiv(div) | SSCR0_DataSize(data_size);
	SSSR   = SSSR_ROR;
	SSCR1 &= ~(SSCR1_RIE | SSCR1_TIE | SSCR1_LBM | SSCR1_MWDS);
	SSCR1 |= (0x77<<7);
	SSCR0 |= SSCR0_SSE;

	ret = request_irq(IRQ_SSP, pxa_ssp_interrupt, 0, "PXA-SSP", NULL);
	if (ret)
		printk("%s failed request_irq\n", __FUNCTION__);
	return ret;
}

/**
 * pxa_ssp_exit - undo the effects of pxa_ssp_init
 *
 * release and free resources for the SSP port.
 */
static void pxa_ssp_exit(void)
{
	SSCR0 &= ~SSCR0_SSE;

	CKEN &= ~CKEN3_SSP;

	free_irq(IRQ_SSP, NULL);
}


struct ssp_ops pxa_ssp_ops = {
	ssp_write_word: pxa_ssp_write_word,
	ssp_read_word: pxa_ssp_read_word,
	ssp_flush: pxa_ssp_flush,
	ssp_enable: pxa_ssp_enable,
	ssp_disable: pxa_ssp_disable,
	ssp_save_state: pxa_ssp_save_state,
	ssp_restore_state: pxa_ssp_restore_state,
	ssp_init: pxa_ssp_init,
	ssp_exit: pxa_ssp_exit};
		
MODULE_AUTHOR("Robert Whaley");
MODULE_DESCRIPTION("PXA SSP PIO driver");
MODULE_LICENSE("GPL");

EXPORT_SYMBOL(pxa_ssp_ops);
