/*
 *  linux/arch/arm/mach-pxa/pxa_nssp.c
 *
 *  Copyright (C) 2003 Russell King.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Generic SSP driver.  This provides the generic core for simple
 *  IO-based SSP applications.
 *
 *  Adapted for PXA SSP by Robert Whaley Sept, 2003
 *
 *  This driver does not provide the extra features of this SSP port
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/init.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/hardware/ssp.h>

static void pxa_nssp_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned int status = NSSSR;

	if (status & NSSSR_ROR) {
		printk(KERN_WARNING "SSP: receiver overrun\n");
	}

	NSSSR = NSSSR_ROR;
}

/**
 * pxa_nssp_write_word - write a word to the SSP port
 * @data: 16-bit, MSB justified data to write.
 *
 * Wait for a free entry in the SSP transmit FIFO, and write a data
 * word to the SSP port.
 *
 * The caller is expected to perform the necessary locking.
 *
 * Returns:
 *   %-ETIMEDOUT	timeout occurred (for future)
 * 0 success */
static int pxa_nssp_write_word(u16 data)
{
	while (!(NSSSR & NSSSR_TNF))
		cpu_relax();
	NSSDR = data;

	return 0;
}

/**
 * pxa_nssp_read_word - read a word from the SSP port
 *
 * Wait for a data word in the SSP receive FIFO, and return the
 * received data.  Data is LSB justified.
 *
 * Note: Currently, if data is not expected to be received, this
 * function will wait for ever.
 *
 * The caller is expected to perform the necessary locking.
 *
 * Returns:
 *   %-ETIMEDOUT	timeout occurred (for future)
 *   16-bit data	success
 */
static int pxa_nssp_read_word(void)
{
	while (!(NSSSR & NSSSR_RNE))
		cpu_relax();

	return NSSDR;
}

/**
 * pxa_nssp_flush - flush the transmit and receive FIFOs
 *
 * Wait for the SSP to idle, and ensure that the receive FIFO
 * is empty.
 *
 * The caller is expected to perform the necessary locking.
 */
static void pxa_nssp_flush(void)
{
	do {
		while (NSSSR & NSSSR_RNE) {
			(void) NSSDR;
		}
	} while (NSSSR & NSSSR_BSY);
}

/**
 * pxa_nssp_enable - enable the NSSP port
 *
 * Turn on the NSSP port.
 */
static void pxa_nssp_enable(void)
{
	NSSCR0 |= NSSCR0_SSE;
}

/**
 * pxa_nssp_disable - shut down the SSP port
 *
 * Turn off the SSP port, optionally powering it down.
 */
static void pxa_nssp_disable(void)
{
	NSSCR0 &= ~NSSCR0_SSE;
}

/**
 * pxa_nssp_save_state - save the SSP configuration
 * @ssp: pointer to structure to save SSP configuration
 *
 * Save the configured SSP state for suspend.
 */
static void pxa_nssp_save_state(struct ssp_state *ssp)
{
	ssp->cr0 = NSSCR0;
	ssp->cr1 = NSSCR1;

	NSSCR0 &= ~NSSCR0_SSE;
}

/**
 * pxa_nssp_restore_state - restore a previously saved SSP configuration
 * @ssp: pointer to configuration saved by ssp_save_state
 *
 * Restore the SSP configuration saved previously by pxa_nssp_save_state.
 */
static void pxa_nssp_restore_state(struct ssp_state *ssp)
{
	NSSSR = NSSSR_ROR;

	NSSCR0 = ssp->cr0 & ~NSSCR0_SSE;
	NSSCR1 = ssp->cr1;
	NSSCR0 = ssp->cr0;
}

/**
 * pxa_nssp_init - setup the SSP port
 *
 * initialise and claim resources for the SSP port.
 *
 * Returns:
 *   %-ENODEV	if the SSP port is unavailable
 *   %-EBUSY	if the resources are already in use
 *   %0		on success
 */
static int pxa_nssp_init(int unit, int data_size, int baud_rate, int format)
{
	int ret;
	int div = 3686400 / baud_rate;

	// Enable SSP Controller Clock
	set_GPIO_mode(GPIO81_NSSPCLK_OUT_MD);
	set_GPIO_mode(GPIO82_NSSPSFRM_OUT_MD);
	set_GPIO_mode(GPIO83_NSSPTXD_OUT_MD);
	set_GPIO_mode(GPIO84_NSSPRXD_IN_MD);

	// Enable SSP Unit Clock
	CKEN |= CKEN9_NSSP;

	// Configure the SSP Serial Port
	// National Microwire - 450k baud - 16 bit
	NSSCR0  = format | NSSCR0_SerClkDiv(div) | NSSCR0_DataSize(data_size);
	NSSSR   = NSSSR_ROR;
	NSSCR1 &= ~(SSCR1_RIE | NSSCR1_TIE | NSSCR1_LBM | NSSCR1_MWDS);
	NSSCR1 |= (0x77<<7);
	NSSCR0 |= NSSCR0_SSE;

	ret = request_irq(IRQ_NSSP, pxa_nssp_interrupt, 0, "PXA-NSSP", NULL);
	if (ret)
		printk("%s failed request_irq\n", __FUNCTION__);
	return ret;
}

/**
 * pxa_nssp_exit - undo the effects of pxa_nssp_init
 *
 * release and free resources for the SSP port.
 */
static void pxa_nssp_exit(void)
{
	NSSCR0 &= ~SSCR0_SSE;

	CKEN &= ~CKEN9_NSSP;

	free_irq(IRQ_NSSP, NULL);
}


struct ssp_ops pxa_nssp_ops = {
	ssp_write_word: pxa_nssp_write_word,
	ssp_read_word: pxa_nssp_read_word,
	ssp_flush: pxa_nssp_flush,
	ssp_enable: pxa_nssp_enable,
	ssp_disable: pxa_nssp_disable,
	ssp_save_state: pxa_nssp_save_state,
	ssp_restore_state: pxa_nssp_restore_state,
	ssp_init: pxa_nssp_init,
	ssp_exit: pxa_nssp_exit};
		
MODULE_AUTHOR("Robert Whaley");
MODULE_DESCRIPTION("PXA SSP PIO driver");
MODULE_LICENSE("GPL");

EXPORT_SYMBOL(pxa_nssp_ops);
