/*
 * PXA250/210 Power Management Routines
 *
 * Original code for the SA11x0:
 * Copyright (c) 2001 Cliff Brake <cbrake@accelent.com>
 *
 * Modified for the PXA250 by Nicolas Pitre:
 * Copyright (c) 2002 Monta Vista Software, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/pm.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/sysctl.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/cpufreq.h>

#include <asm/hardware.h>
#ifdef CONFIG_SA1111
#include <asm/hardware/sa1111.h>
#endif
#include <asm/memory.h>
#include <asm/system.h>
#include <asm/leds.h>
#include <asm/irq.h>

#ifdef CONFIG_SYSCTL

#  if defined(CONFIG_ARCH_ADSBITSYX) || defined(CONFIG_ARCH_ADSAGX)

#  define SLEEP_ON_IRQ

#  endif
#endif

/*
 * Debug macros
 */
#undef DEBUG

extern void pxa_cpu_suspend(void);
extern void pxa_cpu_resume(void);

#ifdef CONFIG_ARCH_ADSBITSYX
extern void adsbitsyx_sa1111_wake(unsigned long pa_dwr);
extern void adsbitsyx_avr_wake(void);
extern void adsbitsyx_set_sleep_state(void);
#endif

#ifdef CONFIG_ARCH_ADSAGX
extern void adsagx_set_sleep_state(void);
#endif

#define SAVE(x)		sleep_save[SLEEP_SAVE_##x] = x
#define RESTORE(x)	x = sleep_save[SLEEP_SAVE_##x]

/*
 * List of global PXA peripheral registers to preserve.
 * More ones like CP and general purpose register values are preserved
 * with the stack pointer in sleep.S.
 */
enum {	SLEEP_SAVE_START = 0,

	SLEEP_SAVE_OSCR, SLEEP_SAVE_OIER,
	SLEEP_SAVE_OSMR0, SLEEP_SAVE_OSMR1, SLEEP_SAVE_OSMR2, SLEEP_SAVE_OSMR3,

	SLEEP_SAVE_GPDR0, SLEEP_SAVE_GPDR1, SLEEP_SAVE_GPDR2,
	SLEEP_SAVE_GRER0, SLEEP_SAVE_GRER1, SLEEP_SAVE_GRER2,
	SLEEP_SAVE_GFER0, SLEEP_SAVE_GFER1, SLEEP_SAVE_GFER2,
	SLEEP_SAVE_GAFR0_L, SLEEP_SAVE_GAFR1_L, SLEEP_SAVE_GAFR2_L,
	SLEEP_SAVE_GAFR0_U, SLEEP_SAVE_GAFR1_U, SLEEP_SAVE_GAFR2_U,

	SLEEP_SAVE_FFIER, SLEEP_SAVE_FFLCR, SLEEP_SAVE_FFMCR,
	SLEEP_SAVE_FFSPR, SLEEP_SAVE_FFISR,
	SLEEP_SAVE_FFDLL, SLEEP_SAVE_FFDLH,

	SLEEP_SAVE_BTIER, SLEEP_SAVE_BTLCR, SLEEP_SAVE_BTMCR,
	SLEEP_SAVE_BTSPR, SLEEP_SAVE_BTISR,
	SLEEP_SAVE_BTDLL, SLEEP_SAVE_BTDLH,

	SLEEP_SAVE_STIER, SLEEP_SAVE_STLCR, SLEEP_SAVE_STMCR,
	SLEEP_SAVE_STSPR, SLEEP_SAVE_STISR,
	SLEEP_SAVE_STDLL, SLEEP_SAVE_STDLH,

	SLEEP_SAVE_ICMR,
	SLEEP_SAVE_CKEN,

#ifdef CONFIG_SA1111
	SLEEP_SAVE_INTEN0,SLEEP_SAVE_INTEN1, SLEEP_SAVE_INTPOL0,
	SLEEP_SAVE_INTPOL1,
#endif

	SLEEP_SAVE_CKSUM,

	SLEEP_SAVE_SIZE
};

#ifdef SLEEP_ON_IRQ

static wait_queue_head_t sleep_queue;
static int irq_present = 0;
static int irq_active = 0;

static void sleep_irq( int irq, void *dev_id, struct pt_regs *regs )
{
	if (irq_active) {
		irq_active = 0;
		wake_up(&sleep_queue);
	}
}

static int sysctl_pm_suspend_catch(ctl_table *ctl, int write, struct file * filp, void *buffer, size_t *lenp)
{
	while (1) {

		init_waitqueue_head(&sleep_queue);

		// Set GPIO1 as non Alternative Function and Input
		// and setup the IRQ (set it up each time since sleeping
		// fools around with GPIO1 quite a bit)

		set_GPIO_mode(0 | GPIO_IN);
		if (irq_present) {
			free_irq(IRQ_GPIO(0), NULL);
			irq_present=0;
		}

		// register IRQ
		set_GPIO_IRQ_edge(0, GPIO_RISING_EDGE);
		if (request_irq(IRQ_GPIO(0), sleep_irq, 0,"sleep",NULL)){ 
			return -EINVAL;
		}
		irq_active = 1;
		irq_present = 1;

		interruptible_sleep_on(&sleep_queue);

		// debounce GPIO0 a little bit so we don't wake right up!
		current->state = TASK_INTERRUPTIBLE;
		schedule_timeout(HZ / 2);
		
		if (!irq_active)
			return 0;
	}
}

#endif

static int pm_sleep_timeout = 0;
#if defined(CONFIG_ARCH_ADSBITSYX)
static unsigned long pa_dwr = 0x0f;
#endif

int pm_do_suspend(void)
{
	unsigned long sleep_save[SLEEP_SAVE_SIZE];
	unsigned long checksum = 0;
	unsigned long rcnr;
	int i;

	cli();
	clf();

	leds_event(led_stop);

	/* preserve current time */
	RCNR = xtime.tv_sec;

	/* 
	 * Temporary solution.  This won't be necessary once
	 * we move pxa support into the serial driver
	 */
	// Save the FF UART
	SAVE(FFIER);
	SAVE(FFLCR);
	SAVE(FFMCR);
	SAVE(FFSPR);
	SAVE(FFISR);
	FFLCR |= 0x80;
	SAVE(FFDLL);
	SAVE(FFDLH);
	FFLCR &= 0xef;

	// Save the BT UART
	SAVE(BTIER);
	SAVE(BTLCR);
	SAVE(BTMCR);
	SAVE(BTSPR);
	SAVE(BTISR);
	BTLCR |= 0x80;
	SAVE(BTDLL);
	SAVE(BTDLH);
	BTLCR &= 0xef;

	// Save the ST UART
	SAVE(STIER);
	SAVE(STLCR);
	SAVE(STMCR);
	SAVE(STSPR);
	SAVE(STISR);
	STLCR |= 0x80;
	SAVE(STDLL);
	SAVE(STDLH);
	STLCR &= 0xef;

	if (machine_is_adsbitsyx() || machine_is_adsagx()) {
		// Don't let FF UART inputs float
		set_GPIO_mode(34 | GPIO_IN); // has pull up
		set_GPIO_mode(35 | GPIO_OUT);
		set_GPIO_mode(36 | GPIO_OUT);
		set_GPIO_mode(37 | GPIO_OUT);
		set_GPIO_mode(38 | GPIO_OUT);

		// Don't let SSP SFrm or MMCCD float either
		set_GPIO_mode(12 | GPIO_IN); // has pull up
		set_GPIO_mode(24 | GPIO_IN); // has pull up
	}

	/* save vital registers */
	SAVE(OSCR);
	SAVE(OSMR0);
	SAVE(OSMR1);
	SAVE(OSMR2);
	SAVE(OSMR3);
	SAVE(OIER);

	SAVE(GPDR0); SAVE(GPDR1); SAVE(GPDR2);
	SAVE(GRER0); SAVE(GRER1); SAVE(GRER2);
	SAVE(GFER0); SAVE(GFER1); SAVE(GFER2);
	SAVE(GAFR0_L); SAVE(GAFR0_U);
	SAVE(GAFR1_L); SAVE(GAFR1_U);
	SAVE(GAFR2_L); SAVE(GAFR2_U);

#ifdef CONFIG_SA1111
	SAVE(INTEN0);
	SAVE(INTEN1);
	SAVE(INTPOL0);
	SAVE(INTPOL1);
#endif

	SAVE(ICMR);
	ICMR = 0;

	SAVE(CKEN);
	CKEN = 0;

	/* Note: wake up source are set up in each machine specific files */

	/* clear GPIO transition detect  bits */
	GEDR0 = GEDR0; GEDR1 = GEDR1; GEDR2 = GEDR2;

	/* Clear previous reset status */
	RCSR = RCSR_HWR | RCSR_WDR | RCSR_SMR | RCSR_GPR;

	/* PGSRx gets blown away when we wake up so set it every time! */
#ifdef CONFIG_ARCH_ADSBITSYX	
	if (machine_is_adsbitsyx())
		adsbitsyx_set_sleep_state();
#endif

#ifdef CONFIG_ARCH_ADSAGX
	if (machine_is_adsagx())
		adsagx_set_sleep_state();
#endif

	if (pm_sleep_timeout != 0) {
		// Set RTC Alarm
		ICMR &= ~PWER_RTC;
		RTSR = RTSR_AL + RTSR_HZ;

		ICLR &= ~PWER_RTC;
		rcnr = xtime.tv_sec;
		RCNR = rcnr;
		rcnr += pm_sleep_timeout;
		RTAR = rcnr;

		PWER |= PWER_RTC;
		RCSR = 0xF;

		RTSR = RTSR_AL + RTSR_HZ + RTSR_ALE;
		ICIP &= ~PWER_RTC;
		ICMR |= PWER_RTC;
	}

	/* set resume return address */
	PSPR = virt_to_phys(pxa_cpu_resume);

	/* before sleeping, calculate and save a checksum */
	for (i = 0; i < SLEEP_SAVE_SIZE - 1; i++)
		checksum += sleep_save[i];
	sleep_save[SLEEP_SAVE_CKSUM] = checksum;

	/* *** go zzz *** */
	pxa_cpu_suspend();

	/* after sleeping, validate the checksum */
	checksum = 0;
	for (i = 0; i < SLEEP_SAVE_SIZE - 1; i++)
		checksum += sleep_save[i];

#ifdef CONFIG_ARCH_LUBBOCK
	/* if invalid, display message and wait for a hardware reset */
	if (checksum != sleep_save[SLEEP_SAVE_CKSUM]) {
		LUB_HEXLED = 0xbadbadc5;
		while (1);
	}
#endif

	/* restore registers */
	RESTORE(GPDR0); RESTORE(GPDR1); RESTORE(GPDR2);
	RESTORE(GRER0); RESTORE(GRER1); RESTORE(GRER2);
	RESTORE(GFER0); RESTORE(GFER1); RESTORE(GFER2);
	RESTORE(GAFR0_L); RESTORE(GAFR0_U);
	RESTORE(GAFR1_L); RESTORE(GAFR1_U);
	RESTORE(GAFR2_L); RESTORE(GAFR2_U);

	// JML - At this point, only PSSR_SSS is set - at least normally it is.
	PSSR = PSSR_PH; // JML - What about PSSR_RDH PSSR_VFS PSSR_BFS PSSR_SSS ?

	RESTORE(OSMR0);
	RESTORE(OSMR1);
	RESTORE(OSMR2);
	RESTORE(OSMR3);
	RESTORE(OSCR);
	RESTORE(OIER);

	RESTORE(CKEN);

	ICLR = 0;
	ICCR = 1;
	RESTORE(ICMR);

	/* 
	 * Temporary solution.  This won't be necessary once
	 * we move pxa support into the serial driver.
	 */
	if (machine_is_adsbitsyx() || machine_is_adsagx()) {
		// Restore FF UART input lines
		set_GPIO_mode(GPIO34_FFRXD_MD);
		set_GPIO_mode(GPIO35_FFCTS_MD);
		set_GPIO_mode(GPIO36_FFDCD_MD);
		set_GPIO_mode(GPIO37_FFDSR_MD);
		set_GPIO_mode(GPIO38_FFRI_MD);

		// Restore SSP SFrm output, too
		set_GPIO_mode(GPIO24_SFRM_MD);
	}

	// Restore the ST UART
	RESTORE(STMCR);
	RESTORE(STSPR);
	RESTORE(STLCR);
	STLCR |= 0x80;
	RESTORE(STDLH);
	RESTORE(STDLL);
	RESTORE(STLCR);
	RESTORE(STISR);
	STFCR = FCR_RESETTF | FCR_RESETRF | FCR_TRFIFOE;
	RESTORE(STIER);

	// Restore the BT UART
	RESTORE(BTMCR);
	RESTORE(BTSPR);
	RESTORE(BTLCR);
	BTLCR |= 0x80;
	RESTORE(BTDLH);
	RESTORE(BTDLL);
	RESTORE(BTLCR);
	RESTORE(BTISR);
	BTFCR = FCR_RESETTF | FCR_RESETRF | FCR_TRFIFOE;
	RESTORE(BTIER);

	// Restore the FF UART
	RESTORE(FFMCR);
	RESTORE(FFSPR);
	RESTORE(FFLCR);
	FFLCR |= 0x80;
	RESTORE(FFDLH);
	RESTORE(FFDLL);
	RESTORE(FFLCR);
	RESTORE(FFISR);
	FFFCR = 0x07;
	RESTORE(FFIER);

#ifdef CONFIG_SA1111
	if (machine_is_adsbitsyx()) {
		// This assumes the PCMCIA/CF power state (pa_dwr) is the same after sleeping - what else can we do?
		// Inserting a card during sleep seems to be OK but I wouldn't swap one for another!
		adsbitsyx_sa1111_wake(pa_dwr);
	}

	RESTORE(INTEN0);
	RESTORE(INTEN1);
	RESTORE(INTPOL0);
	RESTORE(INTPOL1);
#endif

#ifdef CONFIG_ARCH_ADSBITSYX	
	if (machine_is_adsbitsyx()) {
		if (RTSR & RTSR_AL)
			adsbitsyx_avr_wake();
	}
#endif

	/* restore current time */
	xtime.tv_sec = RCNR;

#ifdef DEBUG
	printk(KERN_DEBUG "*** made it back from resume\n");
#endif

	leds_event(led_start);

	sti();

	return 0;
}

unsigned long sleep_phys_sp(void *sp)
{
	return virt_to_phys(sp);
}

#ifdef CONFIG_SYSCTL
/*
 * ARGH!  ACPI people defined CTL_ACPI in linux/acpi.h rather than
 * linux/sysctl.h.
 *
 * This means our interface here won't survive long - it needs a new
 * interface.  Quick hack to get this working - use sysctl id 9999.
 */
#warning ACPI broke the kernel, this interface needs to be fixed up.
#define CTL_ACPI 9999
#define ACPI_S1_SLP_TYP 19
#define ACPI_S1_SLP_TIME 20

/*
 * Send us to sleep.
 */
static int sysctl_pm_do_suspend(void)
{
	int retval;
#ifdef CONFIG_CPU_FREQ
	unsigned int restore_khz = cpufreq_get(0);      // preserve current speed
#endif

#ifdef CONFIG_ARCH_ADSBITSYX	
	if (machine_is_adsbitsyx())
		pa_dwr = PA_DWR; // Save SA1111 Port A output state before we suspend all
#endif

	retval = pm_send_all(PM_SUSPEND, (void *)3);

	if (retval == 0) {
#ifdef CONFIG_CPU_FREQ
		cpufreq_set(0); // minimize power during sleep by lowering voltage
#endif
		retval = pm_do_suspend();  // bootloader will set freq to max
		pm_send_all(PM_RESUME, (void *)0);

#ifdef SLEEP_ON_IRQ		
		wake_up(&sleep_queue);
#endif

#ifdef CONFIG_CPU_FREQ
		cpufreq_set(restore_khz);  // now restore the CPU speed
#endif
	}
	return retval;
}

static struct ctl_table pm_table[] =
{
	{ACPI_S1_SLP_TYP, "suspend", NULL, 0, 0600, NULL, (proc_handler *)&sysctl_pm_do_suspend},
	{ACPI_S1_SLP_TIME, "suspend_timeout", &pm_sleep_timeout, sizeof(pm_sleep_timeout), 0600, NULL, &proc_dointvec},
#ifdef SLEEP_ON_IRQ
	{ACPI_S1_SLP_TYP, "suspend_catch", NULL, 0, 0600, NULL, (proc_handler *)&sysctl_pm_suspend_catch},
#endif
	{0}
};

static struct ctl_table pm_dir_table[] =
{
	{CTL_ACPI, "pm", NULL, 0, 0555, pm_table},
	{0}
};

/*
 * Initialize power interface
 */
static int __init pm_init(void)
{
	register_sysctl_table(pm_dir_table, 1);
	return 0;
}

__initcall(pm_init);

#endif
