/*
 * linux/arch/arm/mach-pxa/adsbitsy.c
 *
 * Pieces specific to the ADS BitsyX
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/init.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>

#include <asm/hardware.h>
#include <asm/hardware/sa1111.h>
#include <asm/setup.h>
#include <asm/irq.h>

#include <asm/mach/irq.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>

#include <asm/arch/irq.h>

#include "generic.h"
#include "sa1111.h"

int adsbitsy_probe_for_smc91x(void);
void adsbitsyx_avr_wake(void);

static int __init adsbitsyx_init(void)
{
	int ret;
	int cpld_rev;
	int sa1111_irq;

	if (!machine_is_adsbitsyx())
		return -ENODEV;

	/*
	 * Enable 32 kHz Clock
	 */
	OSCC = 0x02;

	/*
	 * Ensure that the memory bus request/grant signals are setup,
	 * and the grant is held in its inactive state.
	 */
	pxa_mb_disable();

	/*
	 * Wake up AVR
	 */
	adsbitsyx_avr_wake();

	/*
	 * Reset SA1111
	 */
	GPCR0 = GPIO_bit(17);
	udelay(1000);
	GPSR0 = GPIO_bit(17);

	/*
	 * Probe for SA1111.
	 */
	ret = sa1111_probe(ADSBITSYX_SA1111_BASE);
	if (ret < 0)
		return ret;

	/*
	 * We found it.  Wake the chip up.
	 */
	sa1111_wake();

	/*
	 * The SDRAM configuration of the CPU and companion chip  must
	 * match.  This is very important to ensure that SA1111 accesses
	 * don't corrupt the SDRAM.  Note that this ungates the SA1111's
	 * MBGNT signal, so we must have called pxa_mb_disable()
	 * beforehand.
	 */
	sa1111_configure_smc(1,
			     FExtr(MDCNFG, MDCNFG_PXA_DRAC0),
			     FExtr(MDCNFG, MDCNFG_PXA_DTC0));

	/*
	 * We only need to turn on DCLK whenever we want to use the
	 * DMA.  It can otherwise be held firmly in the off position.
	 */
	SKPCR |= SKPCR_DCLKEN;

	/*
	 * Enable the CPU memory bus request and grant signals.
	 */
	pxa_mb_enable();


	cpld_rev =  (ADSBITSYX_CPLD_REV & ADSBITSYX_CPLD_REV_MASK) >> 4;

	if (cpld_rev > 5) {
		sa1111_irq = 2;
	}
	else {
		sa1111_irq = 1;
	}

	set_GPIO_IRQ_edge(sa1111_irq, GPIO_RISING_EDGE);
	sa1111_init_irq(IRQ_GPIO(sa1111_irq));

	printk("ADS CPLD REV: %#x\n", cpld_rev);

#ifdef CONFIG_SMC91X
	adsbitsy_probe_for_smc91x();
#endif
	return 0;
}

__initcall(adsbitsyx_init);

static void __init adsbitsyx_init_irq(void)
{
	/* First the standard IRQs */
	pxa_init_irq();
}

/*
 * Resume SA1111 when system wakes up (pm.c)
 */
void adsbitsyx_sa1111_wake(unsigned long pa_dwr)
{
	// Turn ON SA1111
	GPCR0 = GPIO_bit(17);
	mdelay(1);
	GPSR0 = GPIO_bit(17);

	set_GPIO_mode(GPIO11_3_6MHz_MD);

	SBI_SKCR = SKCR_PLL_BYPASS | SKCR_RDYEN | SKCR_OE_EN;
	udelay(100);
	SBI_SKCR = SKCR_PLL_BYPASS | SKCR_RCLKEN | SKCR_RDYEN | SKCR_OE_EN;

	GAFR(0) |= (GPIO13_MBGNT | GPIO14_MBREQ);
	GPDR(0) |= GPIO13_MBGNT;
	GPDR(0) &= ~GPIO14_MBREQ;

	set_GPIO_mode(GPIO13_MBGNT_MD);
	set_GPIO_mode(GPIO14_MBREQ_MD);

	sa1111_configure_smc(1,
			     FExtr(MDCNFG, MDCNFG_PXA_DRAC0),
			     FExtr(MDCNFG, MDCNFG_PXA_DTC0));

	SKPCR |= SKPCR_DCLKEN;

	// Reset PCMCIA
	PCCR = 0xFF;
	mdelay(100);
	PA_DDR = 0x00;
	PA_DWR = pa_dwr; // This "restores" the PCMCIA/CF power state
	PCCR = ~(PCCR_S0_RST | PCCR_S1_RST);

#ifdef CONFIG_USB_OHCI_SA1111
	// Turn ON clock
	SKPCR |= SKPCR_UCLKEN;
	udelay(100);

	// force a RESET
	USB_RESET = 0x01;
	USB_RESET |= 0x02;
	udelay(100);

	// Set Power Sense and Control Line
	USB_RESET = 0;
	USB_RESET = USB_RESET_PWRSENSELOW;
	USB_STATUS = 0;
	udelay(10);
#endif
}

void adsbitsyx_avr_wake(void)
{
	// Signal the AVR on boot up and when the RTC Alarm wakes us up.
	// (The AVR gets woke up by the CPLD when /RQONOFF wakes us up.)
	ADSBITSYX_CPLD_SUPPC2 &= ~ADSBITSYX_SUPPC2_AVR_WKP; // wake up active low
	mdelay(100);
	ADSBITSYX_CPLD_SUPPC2 |= ADSBITSYX_SUPPC2_AVR_WKP; // leave in high state
	mdelay(100);
}

void adsbitsyx_set_sleep_state(void)
{
	/*
	 * Setup BitsyX sleep states.
	 */

	// Negate chip select 1-5 during sleep
	PGSR0 |= GPIO_bit(GPIO15_nCS_1);
	PGSR2 |= (GPIO_bit(GPIO78_nCS_2) | GPIO_bit(GPIO79_nCS_3));
	// Chip selects 4 & 5 are not used on the BitsyX.
	//	PGSR1 |= GPIO_bit(GPIO33_nCS_5);
	//	PGSR2 |= GPIO_bit(GPIO80_nCS_4);

	// Drive the MMC outputs low during sleep
	PGSR0 &= ~(GPIO_bit(GPIO6_MMCCLK) | GPIO_bit(GPIO8_MMCCS0) | GPIO_bit(GPIO9_MMCCS1) | GPIO_bit(12));
	// Drive the 3.6 MHz Clock low during sleep
	PGSR0 &= ~GPIO_bit(GPIO11_3_6MHz);
	// Drive the Memory Bus Grant line low during sleep
	PGSR0 &= ~GPIO_bit(GPIO13_MBGNT);
	// Drive the AVR /RESET line high during sleep
	if (ADSBITSYX_REV < 2)
		PGSR0 |= GPIO_bit(27);
	// Negate PCMCIA/CF Interface output and enable lines during sleep
	PGSR1 |= (GPIO_bit(48) | GPIO_bit(49) | GPIO_bit(50) | GPIO_bit(51) | GPIO_bit(52) | GPIO_bit(53));

	// Don't try to clear RTS & DTR during sleep - they have pull down resistors.
	// UARTs 1 & 3 are buffered and we shut the buffers off during sleep.
	//   Clear Serial port 1 RTS and DTR during sleep
	//	PGSR1 |= (GPIO_bit(40) | GPIO_bit(41));
	//   Clear Serial port 3 RTS during sleep
	//	PGSR1 |= GPIO_bit(45);

	// Drive the LCD outputs low during sleep
	PGSR1 &= ~(0x3f << 26); // GPIO 58-63
	PGSR2 &= ~(0x000003ff); // GPIO 64-73

	if (ADSBITSYX_REV > 1) {
		GPDR0 |= 0x00790498;
		GPCR0 = 0x00790498;
		PGSR0 &= ~0x00790498;
	}

	// Wake up from GPIO0 interrupt (/RQONOFF)
	set_GPIO_mode(0 | GPIO_IN);
	GPDR(0) &= ~GPIO_bit(0); // (1 = output)
	GEDR0 = GFER0 = GRER0 = GPIO_bit(0);
	GEDR1 = GFER1 = GRER1 = 0;
	GEDR2 = GFER2 = GRER2 = 0;
	// PCFR = PCFR_OPDE | PCFR_FS;
	PCFR = PCFR_OPDE | PCFR_FP | PCFR_FS;
	// PWER = PEDR = PFER = PRER = GPIO_bit(0);
	PEDR = PFER = PRER = GPIO_bit(0); // Don't touch PWER

	// Enable wakeup from sleep
	if (ADSBITSYX_REV > 1)
		ADSBITSYX_CPLD_SUPPC2 |= ADSBITSYX_SUPPC2_RESETAVR;
}

void adsbitsyx_turn_audio_on(void)
{
	if (ADSBITSYX_REV < 2)
		return;

	ADSBITSYX_CPLD_PCON |= ADSBITSYX_PCON_AUDIOPA_ON;
	udelay(100);
	ADSBITSYX_CPLD_PCON &= ~ADSBITSYX_PCON_AUDIO_ON;
}

void adsbitsyx_turn_audio_off(void)
{
	if (ADSBITSYX_REV < 2)
		return;

	ADSBITSYX_CPLD_PCON &= ~ADSBITSYX_PCON_AUDIOPA_ON;
	ADSBITSYX_CPLD_PCON |= ADSBITSYX_PCON_AUDIO_ON;
}

void adsbitsyx_lcd_on(void)
{
	ADSBITSYX_CPLD_PCON &= ~ADSBITSYX_PCON_PANEL_ON;
	ADSBITSYX_CPLD_SUPPC |= ADSBITSYX_SUPPC_VEE_ON;
}

void adsbitsyx_lcd_off(void)
{
	ADSBITSYX_CPLD_PCON |= ADSBITSYX_PCON_PANEL_ON;
	ADSBITSYX_CPLD_SUPPC &= ~ADSBITSYX_SUPPC_VEE_ON;
	// make the LCD drivers high impedence (saves 0.5 mA)
	GPDR1 &= ~(0x3f << 26); // GPIO 58-63
	GPDR2 &= ~(0x000003ff); // GPIO 64-73
}

void adsbitsyx_serial_on(void)
{
	ADSBITSYX_CPLD_PCON |= ADSBITSYX_PCON_COM1_3_ON;
#ifndef CONFIG_PXA_FIR
	ADSBITSYX_CPLD_PCON |= ADSBITSYX_PCON_CONN_B_PE2;
#endif
}

void adsbitsyx_serial_off(void)
{
	ADSBITSYX_CPLD_PCON &= ~ADSBITSYX_PCON_COM1_3_ON;
	ADSBITSYX_CPLD_PCON &= ~ADSBITSYX_PCON_CONN_B_PE2;
}

void adsbitsyx_set_cpu_voltage(unsigned int freq)
{
	static struct i2c_client *i2c_dac = NULL;

	/* this table should be read: above 331Mhz, use 57, above 235Mhz use 162, ... */
	static struct {int freq; int value;} dac_table[] = {{331000,  57},  // 1.3V
							    {235000, 162},  // 1.1V
							    {118000, 215},  // 1.0V
							    {     0, 294}}; // 0.85V
	int dac_value = 0;
	int i;

	if (!i2c_dac)
		i2c_dac = i2c_get_client(I2C_DRIVERID_LTC1663, 0 ,NULL);
	if (!i2c_dac) {
		printk(KERN_WARNING " %s: can't find dac to set cpu voltage\n", __FUNCTION__);
		return;
	}
	else
		i2c_use_client(i2c_dac);

	for (i=0; !dac_value; i++)
		if (dac_table[i].freq <= freq) dac_value = dac_table[i].value;

	i2c_dac->driver->command(i2c_dac, 0, &dac_value);

}


#ifdef CONFIG_SMC91X

static int adsbitsy_smc91x_present_flag = 0;

extern void pxa_pcmcia_set_ignore_socket(int sock);

#define REV_REG		        0x000A
#define BANK_SELECT		14
#define SMC_IOADDR		(ADSBITSYX_ETH_BASE + 0x300)
#define SMC_CURRENT_BANK()	SMC_inw( ioaddr, BANK_SELECT )
#define SMC_SELECT_BANK(x)	SMC_outw( x, ioaddr, BANK_SELECT )
#define SMC_GET_REV()		SMC_inw( ioaddr, REV_REG )
#define	SMC_outw(v, a, r)	(*((volatile unsigned short *)((a) + (r))) = (v))
#define	SMC_inw(a, r)		(*((volatile unsigned short *)((a) + (r))))

int __init adsbitsy_probe_for_quad_uart(void)
{
}

int __init adsbitsy_probe_for_smc91x(void)
{
	int ioaddr = ADSBITSYX_ETH_VBASE;
	int bank;
	int rev;
	int ret = 0;

	printk("Probing for adsbitsyx smc91x ");

	//  Enable Power 3.3V only
	PA_DDR &= ~(GPIO_bit(2) | GPIO_bit(3));
	// Turn off 5V
	PA_DWR |= GPIO_bit(3);
	// Enable Power 3.3V for Compactflash (this powers the chip on REV A, powers pullup on REV 2 connector boards
	PA_DWR &= ~GPIO_bit(2);

	// reset chip
	PCCR |= PCCR_S1_RST | PCCR_S1_PSE | PCCR_S1_FLT | PCCR_S1_PWAITEN;
	mdelay(10);
	PCCR &= ~PCCR_S1_RST;
	mdelay(50);

	SMC_SELECT_BANK(0);

	bank = SMC_CURRENT_BANK();
	if ( (bank & 0xFF00 ) == 0x3300 ) {
	        SMC_SELECT_BANK(3);
		rev = SMC_GET_REV();
		if (((rev >> 4) & 0x0f) == 9)
        		ret = 1;
	}

        if ( ret ) {

		adsbitsy_smc91x_present_flag = 1;

		// Set memory speed 
//		MCMEM1 = 0x00004001;
		MCMEM1 = 0x00000000;

		pxa_pcmcia_set_ignore_socket(1);

		printk("found it!\n");
	}
	else
		printk("not found.\n");


	return ret;
}

int adsbitsy_smc91x_present(void)
{
        return adsbitsy_smc91x_present_flag;
}
#endif

static struct map_desc adsbitsyx_io_desc[] __initdata = {
 /* virtual               physical               length      domain     r  w  c  b */
  { ADSBITSYX_CFBUS_VBASE,ADSBITSYX_CFBUS_BASE,  0x00004000, DOMAIN_IO, 0, 1, 0, 0 }, /* 91C1111 and Quad Uart */
  { SA1111_VBASE,         ADSBITSYX_SA1111_BASE, 0x00800000, DOMAIN_IO, 0, 1, 0, 0 }, /* SA1111 */
  { ADSBITSYX_CPLD_VBASE, ADSBITSYX_CPLD_BASE,   0x00001000, DOMAIN_IO, 0, 1, 0, 0 }, /* CPLD Controller */
  LAST_DESC
};

static void __init adsbitsyx_map_io(void)
{
	pxa_map_io();
	iotable_init(adsbitsyx_io_desc);

	adsbitsyx_serial_on();
#ifndef CONFIG_PXA_FIR
	ADSBITSYX_CPLD_SUPPC2 |= ADSBITSYX_SUPPC2_IRDAON; // active low
#endif
}

MACHINE_START(ADSBITSYX, "ADS BitsyX")
	BOOT_PARAMS(0xa000003c)
	BOOT_MEM(0xa0000000, 0x40000000, io_p2v(0x40000000))
	MAPIO(adsbitsyx_map_io)
	INITIRQ(adsbitsyx_init_irq)
MACHINE_END
