/*
 * linux/arch/arm/mach-pxa/adsagx.c
 *
 * Pieces specific to the ADS AGX
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/init.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/ptrace.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>

#include <asm/hardware.h>
#include <asm/setup.h>
#include <asm/irq.h>

#include <asm/mach/irq.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>

#include <asm/arch/irq.h>

#include "generic.h"

void adsagx_set_sleep_state(void)
{
	/*
	 * Setup AGX sleep states.
	 */

	// high chip select 12,3 & 5 during sleep
	PGSR0 |= GPIO_bit(GPIO15_nCS_1);
	PGSR2 |= (GPIO_bit(GPIO78_nCS_2) | GPIO_bit(GPIO79_nCS_3));
	PGSR1 |= GPIO_bit(GPIO33_nCS_5);

	// high chip select 4 during sleep
	PGSR2 |= GPIO_bit(GPIO80_nCS_4);

	// ac97 bits low during sleep
	PGSR0 &= ~(GPIO_bit(GPIO28_BITCLK) | GPIO_bit(GPIO24_SFRM));

	// backlight pwm bits low during sleep
	PGSR0 &= ~GPIO_bit(GPIO17_PWM1);

	// Drive the MMC outputs low during sleep
	PGSR0 &= ~(GPIO_bit(GPIO6_MMCCLK) | GPIO_bit(GPIO8_MMCCS0) | GPIO_bit(GPIO9_MMCCS1) | GPIO_bit(GPIO12_32KHz));
	// Drive the AVR /RESET line high during sleep
	PGSR0 |= GPIO_bit(27);
	// Negate PCMCIA/CF Interface output and enable lines during sleep
	PGSR1 |= (GPIO_bit(48) | GPIO_bit(49) | GPIO_bit(50) | GPIO_bit(51) | GPIO_bit(52) | GPIO_bit(53));

	// Drive the LCD outputs low during sleep
	PGSR1 &= ~(0x3f << 26); // GPIO 58-63
	PGSR2 &= ~(0x000003ff); // GPIO 64-73

	// Make "free" GPIOs output low
	GPDR0 |= 0x00780000;
	GPCR0 = 0x00780000;
	PGSR0 &= ~0x00780000;

	// Wake up from GPIO0 interrupt (/RQONOFF)
	set_GPIO_mode(0 | GPIO_IN);
	GPDR(0) &= ~GPIO_bit(0); // (1 = output)
	GEDR0 = GFER0 = GRER0 = GPIO_bit(0);
	GEDR1 = GFER1 = GRER1 = 0;
	GEDR2 = GFER2 = GRER2 = 0;
	// PCFR = PCFR_OPDE | PCFR_FS;
	PCFR = PCFR_OPDE | PCFR_FP | PCFR_FS;
	// PWER = PEDR = PFER = PRER = GPIO_bit(0);
	PEDR = PFER = PRER = GPIO_bit(0); // Don't touch PWER
}

static int __init adsagx_init(void)
{
	if (!machine_is_adsagx())
		return -ENODEV;

	/*
	 * Enable 32 kHz Clock
	 */
	OSCC = 0x02;

	/*
	 * Reset AVR
	 */
	GPSR(27) = GPIO_bit(27);
	mdelay(100);
	GPCR(27) = GPIO_bit(27);

	/*
	 * Enable PWM control for VEE
	 */
	PWM_CTRL0 = 0x0;
	PWM_PWDUTY0 = 0xFF;
	PWM_PERVAL0 = 0xFE;
	CKEN |= CKEN0_PWM0;

	/*
	 * Enable PWM control for Backlight
	 */

	PWM_CTRL1 = 0x0;
	PWM_PWDUTY1 = 0xFF;
	PWM_PERVAL1 = 0xFE;
	CKEN |= CKEN1_PWM1;

	printk("ADS CPLD REV: %#x-%#x\n", 
	       (ADSAGX_CPLD_REV & ADSAGX_CPLD_REV_HW_MSK) >> 4,
	       ADSAGX_CPLD_REV & ADSAGX_CPLD_REV_FW_MSK);

	return 0;
}

__initcall(adsagx_init);

#define ADSAGX_N_IRQ (ADSAGX_IRQ_MAX + 1 - ADSAGX_IRQ_START)

static void ADSAGX_IRQ_demux( int irq, void *dev_id, struct pt_regs *regs )
{
	int i;

	while( (irq = ADSAGX_CPLD_IRQ_FL | ((ADSAGX_CPLD_PCMCIA_IRQ_MASK & ADSAGX_CPLD_PCMCIA_IRQ_FL) << 8)) ){
		for( i = 0; i < ADSAGX_N_IRQ; i++ )
			if( irq & (1<<i) ) {
				do_IRQ( ADSAGX_IRQ_START + i, regs );
			}
	}

	
}

static struct irqaction ADSAGX_cpld_irq = {
	.name		= "ADSAGX_CPLD_IRQ",
	.handler	= ADSAGX_IRQ_demux,
	.flags		= SA_INTERRUPT
};

static void ADSAGX_mask_and_ack_irq0(unsigned int irq)
{
	int mask = (1 << (irq - ADSAGX_IRQ_START));

	ADSAGX_CPLD_IRQ_EN &= ~mask;
	// can't clear interrupt here - need to clear device itself
}

static void ADSAGX_mask_irq0(unsigned int irq)
{
	int mask = (1 << (irq - ADSAGX_IRQ_START));

	ADSAGX_CPLD_IRQ_EN &= ~mask;
}

static void ADSAGX_unmask_irq0(unsigned int irq)
{
	int mask = (1 << (irq - ADSAGX_IRQ_START));

	ADSAGX_CPLD_IRQ_EN |= mask;
}

static void ADSAGX_mask_and_ack_irq1(unsigned int irq)
{
	int mask = (1 << (irq - (ADSAGX_IRQ_STS_B)));

	ADSAGX_CPLD_PCMCIA_IRQ_EN &= ~mask;
	// can't clear interrupt here - need to clear device itself
	// except for CARD DETECT
	if (irq == ADSAGX_IRQ_DET_B || irq == ADSAGX_IRQ_DET_A)
		ADSAGX_CPLD_PCMCIA_IRQ_FL = mask;
}

static void ADSAGX_mask_irq1(unsigned int irq)
{
	int mask = (1 << (irq - (ADSAGX_IRQ_STS_B)));

	ADSAGX_CPLD_PCMCIA_IRQ_EN &= ~mask;
}

static void ADSAGX_unmask_irq1(unsigned int irq)
{
	int mask = (1 << (irq - (ADSAGX_IRQ_STS_B)));

	ADSAGX_CPLD_PCMCIA_IRQ_EN |= mask;
}

static void __init adsagx_init_irq(void)
{
	int irq;

	/* First the standard IRQs */
	pxa_init_irq();

	/* disable all CPLD IRQs */
	ADSAGX_CPLD_IRQ_EN = 0;
	ADSAGX_CPLD_PCMCIA_IRQ_EN = 0;

	/* setup descriptors for all CPLD IRQs */
	for (irq = ADSAGX_IRQ_START; irq < ADSAGX_IRQ_STS_B; irq++) {
		irq_desc[irq].valid	= 1;
		irq_desc[irq].probe_ok	= 1;
		irq_desc[irq].mask_ack	= ADSAGX_mask_and_ack_irq0;
		irq_desc[irq].mask	= ADSAGX_mask_irq0;
		irq_desc[irq].unmask	= ADSAGX_unmask_irq0;
	}
	for (irq = ADSAGX_IRQ_STS_B; irq <= ADSAGX_IRQ_MAX; irq++) {
		irq_desc[irq].valid	= 1;
		irq_desc[irq].probe_ok	= 1;
		irq_desc[irq].mask_ack	= ADSAGX_mask_and_ack_irq1;
		irq_desc[irq].mask	= ADSAGX_mask_irq1;
		irq_desc[irq].unmask	= ADSAGX_unmask_irq1;
	}

	set_GPIO_IRQ_edge(1, GPIO_RISING_EDGE);
	setup_arm_irq( IRQ_GPIO1, &ADSAGX_cpld_irq );	
}

void adsagx_turn_audio_on(void)
{
	ADSAGX_CPLD_CR1 &= ~ADSAGX_CPLD_CR1_AMP;
	udelay(100);
	ADSAGX_CPLD_CR1 |= ADSAGX_CPLD_CR1_CODEC;
	ADSAGX_CPLD_CR1 &= ~ADSAGX_CPLD_CR1_BTL;
}

void adsagx_turn_audio_off(void)
{
	ADSAGX_CPLD_CR1 &= ~ADSAGX_CPLD_CR1_CODEC;
	ADSAGX_CPLD_CR1 |= (ADSAGX_CPLD_CR1_BTL | ADSAGX_CPLD_CR1_AMP);
}

#ifdef CONFIG_I2C_LTC1663
void adsagx_set_cpu_voltage(unsigned int freq)
{
	static struct i2c_client *i2c_dac = NULL;

	/* this table should be read: above 331Mhz, use 57, above 235Mhz use 162, ... */
	static struct {int freq; int value;} dac_table[] = {{331000,  57},  // 1.3V
							    {235000, 162},  // 1.1V
							    {118000, 215},  // 1.0V
							    {     0, 294}}; // 0.85V
	int dac_value = 0;
	int i;
	

	if (!i2c_dac)
		i2c_dac = i2c_get_client(I2C_DRIVERID_LTC1663, 0 ,NULL);
	if (!i2c_dac) {
		printk(KERN_WARNING " %s: can't find dac to set cpu voltage\n", __FUNCTION__);
		return;
	}
	else
		i2c_use_client(i2c_dac);

	for (i=0; !dac_value; i++)
		if (dac_table[i].freq <= freq) dac_value = dac_table[i].value;

	i2c_dac->driver->command(i2c_dac, 0, &dac_value);
}
#else
void adsagx_set_cpu_voltage(unsigned int freq)
{
	;
}
#endif

static struct map_desc adsagx_io_desc[] __initdata = {
 /* virtual            physical          length      domain     r  w  c  b */
  { ADSAGX_ETH_VBASE,  PXA_CS4_PHYS,     0x00004000, DOMAIN_IO, 0, 1, 0, 0 }, /* 91C1111 */
  { ADSAGX_CPLD_VBASE, ADSAGX_CPLD_BASE, 0x00100000, DOMAIN_IO, 0, 1, 0, 0 }, /* CPLD Controller */
  LAST_DESC
};

void adsagx_serial_on(void)
{
	ADSAGX_CPLD_CR1 |= ADSAGX_CPLD_CR1_COM3EN;
#ifdef CONFIG_PXA_FIR
	ADSAGX_CPLD_CR1 |= ADSAGX_CPLD_CR1_IRDAON;
#endif
}

void adsagx_serial_off(void)
{
	ADSAGX_CPLD_CR1 &= ~ADSAGX_CPLD_CR1_COM3EN;
	ADSAGX_CPLD_CR1 &= ~ADSAGX_CPLD_CR1_IRDAON;
}

static void __init adsagx_map_io(void)
{
	pxa_map_io();
	iotable_init(adsagx_io_desc);
	adsagx_serial_on();
}

MACHINE_START(ADSAGX, "ADS AGX")
	BOOT_PARAMS(0xa000003c)
	BOOT_MEM(0xa0000000, 0x40000000, io_p2v(0x40000000))
	MAPIO(adsagx_map_io)
	INITIRQ(adsagx_init_irq)
MACHINE_END
