#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <rk/rk.h>
#include <rk/rk_error.h>
#include <signal.h>
#include <linux/unistd.h>

unsigned int get_us(struct timespec *t);

void Usage(char *com)
{
    fprintf(stderr, "Usage: %s\n", com);
    exit(-1);
}

int main(int argc, char *argv[])
{
    int i, k, ncount, count1 = 0, count2 = 0, nprocs, procsize;
    char name[64];
    rk_resource_set_t *rsets;
    rk_reserve_t rsv;
    cpu_reserve_attr_data_t params;
    pid_t *procs;

    if (argc > 1) {
	Usage(argv[0]);
    }

    while (1) {
	count1 = rk_resource_sets_get_num();
	printf("system has %d resource sets\n", count1);

	rsets =
	    (rk_resource_set_t *) malloc(count1 *
					 sizeof(rk_resource_set_t));
	if (
	    (count2 =
	     rk_resource_sets_get_list(rsets,
				       count1 *
				       sizeof(rk_resource_set_t))) ==
	    count1) {
	    /* count not changing anymore! */
	    break;
	} else {
	    if (count2 < 0) {
		fprintf(stderr,
			"%s: could not get rset information! (error = %d)\n",
			argv[0], count2);
		exit(-2);
	    }
	    fprintf(stderr, "new return value: %d, original count: %d\n",
		    count2, count1);
	}
	free(rsets);
    }

    /* print header */
    printf
	("   RSET        NAME          CPU_RSV     C (ms)   P (ms)    D (ms)   Mode Procs\n");

    /* print details of each rset */
    for (i = 0; i < count1; i++) {
	rk_resource_set_get_name(rsets[i], name);
	printf("%p %s ", rsets[i], name);

	/* pad with spaces to accommodate the max name length */
	{
	    int i;
	    for (i = 0; i < (16 - strlen(name)); i++) {
		printf(" ");
	    }
	}

	/* print CPU reserve */
	rsv = rk_resource_set_get_cpu_rsv(rsets[i]);
	printf("%p ", rsv);

	if (rsv != NULL) {
	    if (rk_cpu_reserve_get_attr(rsv, &params) != RK_SUCCESS) {
		fprintf(stderr,
			"%s: failed to get attributes of CPU reserve %p\n",
			argv[0], rsv);
	    } else {
		printf("% 9.3f % 9.3f % 9.3f ",
		       (float) (get_us(&(params.compute_time)) / 1000.0),
		       (float) (get_us(&(params.period)) / 1000.0),
		       (float) (get_us(&(params.deadline)) / 1000.0));
		switch (params.reserve_type.enf_mode) {
		case RSV_HARD:
		    printf("HARD ");
		    break;
		case RSV_FIRM:
		    printf("FIRM ");
		    break;
		case RSV_SOFT:
		    printf("SOFT ");
		    break;
		}
	    }
	}

	/* display attached process ids */
	if ((nprocs = rk_resource_set_get_num_procs(rsets[i])) <= 0) {
	    printf("<none>\n");
	    continue;
	}

	/* allocate space to get attached processes */
	procsize = nprocs * sizeof(pid_t);
	if ((procs = (pid_t *) malloc(10 * procsize)) == NULL) {
	    printf("%s: malloc failed\n", argv[0]);
	    exit(-1);
	}

	/* get attached process ids */
	if ((ncount = rk_resource_set_get_proclist(rsets[i], procs))
	    > 0) {
	    printf("<%d>", ncount);
	    for (k = 0; k < ncount; k++) {
		printf("*%d*", (pid_t) procs[k]);
		if (k < (ncount - 1))
		    printf(",");
	    }
	} else {
	    printf("<error>");
	}

	/* free allocated space for next iteration */
	free(procs);

	printf("\n");
    }

    return 0;
}

unsigned int get_us(struct timespec *t)
{
    unsigned int us;
    us = t->tv_sec * 1000000 + t->tv_nsec / 1000;
    return us;
}
