#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <string.h>
#include <limits.h>
#include <assert.h>

#include <rk/rk.h>

#define DEBUG

void print_error(char *emsg)
{
    fprintf(stderr, "%s", emsg);
    exit(-1);
}

void print_usage(char *pname)
{
    printf
	("Usage: %s [--rset n] [--newrset] [--c_period T] [--cpu C] [--exec 'progname arguments']\n\n",
	 pname);
    printf("\t--rset n\t specifies existing resource set in hex.\n");
    printf("\t--newrset\t create new resource set\n");
    printf
	("\t--cpu C\t\t make CPU reservation with comp. time of C microsecs\n");
    printf
	("\t--c_period T\t make CPU reservation with period of T microsecs\n");
    printf
	("\t--c_deadline D\t make CPU reservation with deadline of D microsecs\n");
    printf("\t--start-time S\t start CPU reservation at S microsecs\n");
    printf("\t[--soft]\tmake reservation to be soft\n");
    printf("\t[--hard]\tmake reservation to be hard");
    printf("\t--help\t\t this message\n\n");
    printf
	("\t--inherit\t\t children of the process will inherit the reserve\n");
    printf("some examples...\n");
    printf
	("\t %s --newrset --cpu 2500 --period 10000 --exec /usr/X11R6/lib/xscreensaver/kumppa\n",
	 pname);
    printf
	("\t %s --newrset --cpu 5000 --period 10000 --deadline 8000 --hard --exec '/usr/X11R6/lib/xscreensave/attraction -glow -points 500 -threshold 200 -delay 10'\n",
	 pname);
    exit(1);
}



/*
 * rkexec -exec netscape ...
 */
int main(int argc, char *argv[])
{
    int c, status;
    char *cmd = NULL;
    char newrset = 0x0;
    char cpu_reserve_mode = RSV_HARD;
    int period_us = 100000;	/* 100 ms by default */
    int deadline_us = 100000;	/* 100 ms by default */
    int start_us = 0;		/* start asap        */
    int cpu_us = 0;
    int inherit = 0;
    pid_t pid;
    char name[RSET_NAME_LEN];

    rk_reserve_t cpu_rsv;
    rk_resource_set_t rset = NULL;

    while (1) {
	int option_index = 0;
	static struct option long_options[] = {
	    {"exec", 1, 0, 0},
	    {"rset", 1, 0, 0},
	    {"newrset", 0, 0, 0},
	    {"name", 1, 0, 0},
	    {"cpu", 1, 0, 0},
	    {"period", 1, 0, 0},
	    {"deadline", 1, 0, 0},
	    {"start-time", 1, 0, 0},
	    {"help", 0, 0, 0},
	    {"hard", 0, 0, 0},
	    {"firm", 0, 0, 0},
	    {"soft", 0, 0, 0},
	    {"inherit", 0, 0, 0},
	    {0, 0, 0, 0}
	};

#define OPT_EXEC 	0
#define OPT_RSET 	1
#define OPT_NEWRSET	2
#define OPT_NAME	3
#define OPT_CPU		4
#define OPT_PERIOD	5
#define OPT_DEADLINE	6
#define OPT_START	7
#define OPT_HELP	8
#define OPT_HARD	9
#define OPT_FIRM	10
#define OPT_SOFT	11
#define OPT_INHERIT 12


	c = getopt_long_only(argc, argv, "", long_options, &option_index);

	/* bunch of option handling... */
	if (c == -1)
	    break;
	switch (c) {
	case 0:

#ifdef DEBUG
	    printf("option[%d] %s", option_index,
		   long_options[option_index].name);
	    if (optarg)
		printf(" with arg %s", optarg);
	    printf("\n");
#endif

	    switch (option_index) {
	    case OPT_EXEC:
		cmd = optarg;
		break;

	    case OPT_RSET:
		/* we want base 16 */
		rset = (rk_resource_set_t) strtoul(optarg, NULL, 16);
		if (rset == (rk_resource_set_t) ULONG_MAX) {
		    perror("ULONG_MAX");
		    exit(-1);
		}
		break;

	    case OPT_NEWRSET:
		newrset = 0x1;
		break;

	    case OPT_NAME:
		strncpy(name, optarg, RSET_NAME_LEN);
		break;

	    case OPT_CPU:
		cpu_us = atoi(optarg);
		break;

	    case OPT_PERIOD:
		period_us = atoi(optarg);
		break;

	    case OPT_DEADLINE:
		deadline_us = atoi(optarg);
		break;

	    case OPT_START:
		start_us = atoi(optarg);
		break;

	    case OPT_HELP:
		print_usage(argv[0]);
		break;

	    case OPT_HARD:
		cpu_reserve_mode = RSV_HARD;
		break;

	    case OPT_FIRM:
		cpu_reserve_mode = RSV_FIRM;
		break;

	    case OPT_SOFT:
		cpu_reserve_mode = RSV_SOFT;
		break;

	    case OPT_INHERIT:
		inherit = 1;
		break;
	    }
	    break;

	case '?':
	    break;
	default:
	    printf("?? getopt returned character code 0%o ??\n", c);
	    print_usage(argv[0]);
	}
    }

    if (optind < argc) {
	printf("undefined arguments....ignoring   ");
	while (optind < argc)
	    printf("%s ", argv[optind++]);
	printf("\n");
    }


    if (newrset == 0x0 && rset == NULL && cmd == NULL)
	print_usage(argv[0]);

    rk_inherit_mode(inherit);

    if (name[0] == '\0')
	sprintf(name, "Rsrc Set x");
    /* resource set */
    if (newrset == 0x1) {
	if (rset != NULL) {
	    print_error("--rset and --newrset cannot be both set\n");
	    exit(-1);
	}
	if ((rset = rk_resource_set_create(name)) == NULL) {
	    fprintf(stderr,
		    "Could not create resource set, quitting...\n");
	    exit(-2);
	}
#ifdef DEBUG
	printf("Resource Set Created (%p)=(%d)\n", rset, (int) rset);
#endif

	/* CPU reserve */
	{
	    rk_reserve_param_data_t p;
	    cpu_reserve_attr_data_t attr;
	    long int csec = cpu_us / 1000000;
	    long int tsec = period_us / 1000000;
	    long int dsec = deadline_us / 1000000;

	    assert(rset != NULL);

	    p.enf_mode = cpu_reserve_mode;
	    p.sch_mode = cpu_reserve_mode;
	    p.rep_mode = cpu_reserve_mode;

	    attr.compute_time.tv_sec = csec;
	    attr.compute_time.tv_nsec = (cpu_us - (csec * 1000000)) * 1000;
	    attr.period.tv_sec = tsec;
	    attr.period.tv_nsec = (period_us - (tsec * 1000000)) * 1000;
	    if (dsec <= tsec && deadline_us <= period_us) {
		attr.deadline.tv_sec = dsec;
		attr.deadline.tv_nsec =
		    (deadline_us - (dsec * 1000000)) * 1000;
	    } else {
		attr.deadline.tv_sec = tsec;
		attr.deadline.tv_nsec =
		    (period_us - (tsec * 1000000)) * 1000;
	    }
	    attr.blocking_time.tv_sec = (long int) 0;
	    attr.blocking_time.tv_nsec = (long int) 0;
	    attr.start_time.tv_sec = (long int) 0;
	    attr.start_time.tv_nsec = (long int) 0;
	    attr.reserve_type = p;

	    if ((cpu_rsv = rk_cpu_reserve_create(rset, &attr)) == NULL) {
		fprintf(stderr,
			"Could not create cpu reserve, quitting...\n");
		rk_resource_set_destroy(rset);
		exit(-2);
	    }
	}
    }

    /* execute the new program */
    if (cmd != NULL) {
	char *progname;
	printf("cmd = %s\n", cmd);
	progname = strsep(&cmd, (const char *) " ");
	printf("progname = %s\n", progname);
	if (cmd != NULL)
	    printf("argument = %s\n", cmd);

	if ((pid = fork()) < 0) {
	    perror("fork");
	    exit(1);
	}
#define MAXARG 20

	if (pid == 0) {
	    /* child process */
	    int i;
	    char const *argument[MAXARG];
	    argument[0] = progname;

	    /* attach to a resource set */
	    /* TODO
	     * unfortunately, I cannot attach to rset at this point.
	     * should look into the code.....
	     */
	    if (rk_resource_set_attach_process(rset, getpid()) != 0) {
		fprintf(stderr, "Could not attach process %d to reserve\n",
			getpid());
		rk_resource_set_destroy(rset);
		exit(-3);
	    }

	    /* build the argument vector */
	    for (i = 1; i < MAXARG; i++) {
		argument[i] = strsep(&cmd, (const char *) " ");
		if (argument[i] == NULL)
		    break;
		/* skip spaces */
		if (*argument[i] == 0)
		    i--;
	    }
	    execvp(progname, (char **const) argument);
	    perror(progname);
	    exit(1);
	}


	/* rk_resource_set_attach_process(rset, pid); */
	while (wait(&status) != pid);

	/* destroy resourceset */
	rk_resource_set_destroy(rset);
    }

    exit(0);
}
