#!/usr/bin/perl

use strict;
use Getopt::Long;
use File::Basename;
use Pod::Usage;
use POSIX qw(setsid _exit);

Getopt::Long::Configure('bundling', 'permute', 'no_getopt_compat' ,'no_ignore_case');

# This script converts from LSF format to PBS format and
# launches the job in the corresponding queue

my (@variables, @resources, @deps);
my ($junk, $efile, $ofile, $lsfdeps, $qtype, $name, $rerun, $excl);
my ($runlocal, $foreground);
GetOptions('J=s' => sub { my ($opt, $arg) = @_; $name = substr $arg, 0, 12 },
	   'm=s' => sub { warn "You cannot specify machines.  -m option ignored" },
	   'R=s' => \$junk,
	   'e=s' => \$efile,
	   'o=s' => \$ofile,
	   'q=s' => \$qtype,
	   'W=s' => \@deps,
	   'w=s' => \$lsfdeps,
	   'v=s' => \@variables,
	   'l=s' => \@resources,
	   'N=s' => \$name,
	   'local' => \$runlocal,
	   'foreground' => \$runlocal,
	   'r' => \$rerun,
	   'x' => \$excl,
	   'help|?' => sub { pod2usage(1) })
    or pod2usage(2);

if (defined($lsfdeps)) {
    $lsfdeps =~ s/\s*&&\s*/:/g;
    $lsfdeps = "depend=afterok:$lsfdeps";
}
$efile = "/dev/null" unless defined($efile);
$ofile = "/dev/null" unless defined($ofile);

unless (@ARGV) {
    pod2usage(2);
}

my $jobID;
if ($runlocal) {
    my $pid = fork;
    die "fork() failed: $!" unless defined($pid);
    if ($pid == 0) {
	setsid() unless $foreground;
	close STDIN;
	open STDERR, ">$efile" or die "Failed to open $efile: $!";
	open STDOUT, ">$ofile" or die "Failed to open $ofile: $!";

	if (defined($lsfdeps)) {
	    $lsfdeps =~ s/^depend=afterok://;
	    my @pids = split /:/, $lsfdeps;
	    my $alive = @pids;
	    while ($alive > 0) {
		$alive = kill(0, @pids);
		sleep(1);
	    }
	}

	exec @ARGV;
	_exit(127);
    }
    else {
	$jobID = $pid;
	if ($foreground) {
	    waitpid $pid, 0;
	    die "Job $pid failed with status $?" if $?;
	}
    }
}
else {
    # Make temporary script file containing job (UNSAFE!)
    my $prog = shift;
    my $fnhead = basename($prog);
    my $fn = "/tmp/$fnhead.$$.sh";
    #$fn = "./$fnhead.$$.sh";
    open FN, ">$fn"
	or die "Failed to open temporary file $fn: $!\n";
    END { unlink $fn }
    $SIG{__DIE__} = $SIG{INT} = sub { unlink $fn };
    print FN "#!/bin/sh\n";
    print FN "if \[ -n \"\$PBS_ENVIRONMENT\" \]; then\n";
    print FN "echo \" \"\n";
    print FN "echo \"This job was submitted by user: \$PBS_O_LOGNAME\"\n";
    print FN "echo \"This job was submitted to host: \$PBS_O_HOST\"\n";
    print FN "echo \"This job was submitted to queue: \$PBS_O_QUEUE\"\n";
    print FN "echo \"PBS working directory: \$PBS_O_WORKDIR\"\n";
    print FN "echo \"PBS job id: \$PBS_JOBID\"\n";
    print FN "echo \"PBS job name: \$PBS_JOBNAME\"\n";
    print FN "echo \"PBS environment: \$PBS_ENVIRONMENT\"\n";
    print FN "echo \" \"\n";
    print FN "echo \"This script is running on \`hostname\` \"\n";
    print FN "echo \" \"\n";
    print FN "fi\n";
    my $currentDir = dirname($prog);
    print FN "cd \"$currentDir\"\n";
    print FN "\"$prog\" \\\n";
    foreach (@ARGV) {
	print FN "\t\"$_\" \\\n";
    }
    print FN "\nexit 0\n";
    close FN or die "Failed to close temporary file $fn: $!\n";
    chmod $fn, 0777;

    # Make argument list for qsub
    my @Qargs = (-e => $efile, -o => $ofile);
    push(@Qargs, -W => $lsfdeps) if defined($lsfdeps);
    push(@Qargs, -N => $name) if defined($name);
    push(@Qargs, -q => $qtype) if defined($qtype);
    push(@Qargs, -r => ($rerun ? 'y' : 'n'));
    foreach (@deps) {
	push @Qargs, -W => $_;
    }
    foreach (@resources) {
	push @Qargs, -l => $_;
    }
    foreach (@variables) {
	push @Qargs, -v => $_;
    }
    push(@Qargs, -l => 'nodes=1') if $excl;

    # Quote them nicely
    foreach (@Qargs) {
	s/\"/\\\"/g;
	$_ = "\"$_\"";
    }
    #print STDERR "/usr/bin/qsub @Qargs $fn\n";
    $jobID = `/usr/bin/qsub @Qargs $fn`;
    # To make output compatible with script, the output has to be
    # Job <jobID>
    chomp $jobID;
    $jobID =~ s/\..*//g;
}
print "Job \<$jobID\>\n";

__END__

=head1 NAME

qsub.pl - Generic interface to queues (emulates LSF queue)

=head1 SYNOPSIS

 qsub.pl <-l resources> <-W dependencies> <-e errfile>
	 <-o outfile> program arguments

=head1 ARGUMENTS

=over 4

=item B<-J> I<job_name>

=item B<-e> I<err_file>

=item B<-o> I<output_file>

=item B<-W> I<PBS_attributes>

=item B<-w> I<dependencies>

=item B<-v> I<variable> ...

=item B<-l> I<resource> ...

=item B<-N> I<job_name>

=item B<--local>

=item B<--foreground>

=item B<-r>

=item B<-x>

=back

=head1 DESCRIPTION

=head1 AUTHORS

Written by someone, maybe Rita Singh?  Modified and documented by
David Huggins-Daines <dhuggins@cs.cmu.edu>

=cut
