//===========================================================================
/*!
 *  \file LineSearch.h
 *
 *  \brief LineSearch
 *
 *  \author O. Krause, S. Dahlgaard
 *  \date 2010-2013
 *
 *  \par Copyright (c) 1998-2007:
 *      Institut f&uuml;r Neuroinformatik<BR>
 *      Ruhr-Universit&auml;t Bochum<BR>
 *      D-44780 Bochum, Germany<BR>
 *      Phone: +49-234-32-25558<BR>
 *      Fax:   +49-234-32-14209<BR>
 *      eMail: Shark-admin@neuroinformatik.ruhr-uni-bochum.de<BR>
 *      www:   http://www.neuroinformatik.ruhr-uni-bochum.de<BR>
 *      <BR>
 *
 *
 *  <BR><HR>
 *  This file is part of Shark. This library is free software;
 *  you can redistribute it and/or modify it under the terms of the
 *  GNU General Public License as published by the Free Software
 *  Foundation; either version 3, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this library; if not, see <http://www.gnu.org/licenses/>.
 *
 */
//===========================================================================

#ifndef SHARK_ALGORITHMS_GRADIENTDESCENT_LINESEARCH_H
#define SHARK_ALGORITHMS_GRADIENTDESCENT_LINESEARCH_H

#include <shark/LinAlg/Base.h>
#include <shark/LinAlg/arrayoptimize.h>
#include <shark/Core/IConfigurable.h>
#include <shark/Core/ISerializable.h>
#include "Impl/ObjectiveFunctionWrapper.h"
#include "Impl/wolfecubic.inl"
#include <shark/Algorithms/GradientDescent/LineSearch.h>

namespace shark {
///\brief Wrapper for the linesearch class of functions in the linear algebra library.
///
///This class is a wrapper for the linesearch class of functions of the linear algebra library.
///The class is used for example in CG or BFGS for their internal linesearch learning steps.
///It is NOT an Optimizer on its own, since it needs the Newton direction to be specified.
class LineSearch:public IConfigurable, public ISerializable {
public:
	enum LineSearchType {
	    Dlinmin,
	    Linmin,
	    WolfeCubic
	};
	typedef AbstractObjectiveFunction<VectorSpace<double>,double> ObjectiveFunction;
	typedef AbstractObjectiveFunction<VectorSpace<double>,double>::FirstOrderDerivative Derivative;

	///Initializes the internal variables of the class to useful default values.
	///Dlinmin is used as default
	LineSearch() {
		m_minInterval=0;
		m_maxInterval=1;
		m_lineSearchType=Dlinmin;
	}

	LineSearchType lineSearchType()const {
		return m_lineSearchType;
	}
	LineSearchType &lineSearchType() {
		return m_lineSearchType;
	}
	///minInterval sets the minimum initial bracket
	double minInterval()const {
		return m_minInterval;
	}
	///minInterval sets the minimum initial bracket
	double &minInterval() {
		return m_minInterval;
	}
	///maxInterval sets the maximum initial bracket
	double maxInterval()const {
		return m_maxInterval;
	}
	///maxInterval sets the maximum initial bracket
	double &maxInterval() {
		return m_maxInterval;
	}

	///initializes the internal state of the LineSearch class and sets the function on which the lineSearch is to be evaluated
	void init(const ObjectiveFunction &objectiveFunction) {
		m_function.function()=&objectiveFunction;
	}

	///performs a linesearch on the objectiveFunction given the starting point, its value the newton direction and optionally the derivative at the starting point
	///@param searchPoint the point where the linesearch start
	///@param pointValue the value of the function at searchPoint
	///@param newtonDirection the search direction of the line search
	///@param derivative the derivative of the funktion at searchPoint
	///@param stepLength initial step length guess for guiding the line search
	virtual void operator()(RealVector &searchPoint,double &pointValue,const RealVector &newtonDirection, RealVector &derivative, double stepLength = 1.0)const {
		switch (m_lineSearchType) {
		case Dlinmin:
			dlinmin(searchPoint, newtonDirection, pointValue, m_function, m_minInterval, m_maxInterval);
			m_function(searchPoint, derivative);
			break;
		case Linmin:
			linmin(searchPoint, newtonDirection, pointValue, m_function, m_minInterval, m_maxInterval);
			m_function(searchPoint, derivative);
			break;
		case WolfeCubic:
			wolfecubic(searchPoint, newtonDirection, pointValue, m_function, derivative, stepLength);
			break;
		}
	}
	///performs a linesearch on the objectiveFunction given the starting point, its value the newton direction
	///when cubic line search is used, this method first evaluates the derivative at the initial searchPoint. So, if available, use allready available derivative information and call the
	///other version!
	///@param searchPoint the point where the linesearch start
	///@param pointValue the value of the function at searchPoint
	///@param newtonDirection the search direction of the line search
	///@param stepLength initial step length guess for guiding the line search
	virtual void operator()(RealVector &searchPoint,double &pointValue,const RealVector &newtonDirection, double stepLength = 1.0)const {
		switch (m_lineSearchType) {
		case Dlinmin:
			dlinmin(searchPoint, newtonDirection, pointValue, m_function, m_minInterval, m_maxInterval);
			break;
		case Linmin:
			linmin(searchPoint, newtonDirection, pointValue, m_function, m_minInterval, m_maxInterval);
			break;
		case WolfeCubic:
			throw SHARKEXCEPTION("[LineSearch] WolfeCubic requires derivative");
		}
	}

	//IConfigurable
	void configure(const PropertyTree &node) {
		m_lineSearchType=static_cast<LineSearchType>(node.get("searchtype",(unsigned int)Dlinmin));
		m_minInterval=node.get("minInterval",0.0);
		m_maxInterval=node.get("maxInterval",1.0);
	}

	//ISerializable
	virtual void read(InArchive &archive) {
		archive>>m_minInterval;
		archive>>m_maxInterval;
		archive>>m_lineSearchType;
	}

	virtual void write(OutArchive &archive) const {
		archive<<m_minInterval;
		archive<<m_maxInterval;
		archive<<m_lineSearchType;
	}


protected:
	///initial [min,max] bracket for linesearch
	double m_minInterval;
	///initial [min,max] bracket for linesearch
	double m_maxInterval;

	LineSearchType m_lineSearchType;

	///function to optimize
	ObjectiveFunctionDerivativeWrapper m_function;
};
}

#endif
