// PPR.H header file

// The Robotics Institute
// Carnegie Mellon University
// Copyright (C) 2000 by RI.
// All Rights Reserved.

// PalmPilot Robot header file
// By Greg Reshko
// Last modified: 3/15/2000


#define PILOT_PRECOMPILED_HEADERS_OFF

#include <Common.h>
#include <Pilot.h>
#include <SysAll.h>
#include <UIAll.h>
#include <FeatureMgr.h>
#include <SerialMgr.h>

#include <TimeMgr.h>
#include <StringMgr.h>
#include <SoundMgr.h>

extern "C" {
#include "MathLib\src\MathLib.h"
}

#include "vecmath.h"

#include "Robot2.h"

#define Form1 1000


//void Drive(int sv1, int sv2, int sv3);
//void Servo(int number, int speed);
//int Sensor(int number);
//int Sensor_To_Range(int x);
//void Display(int number);
//void DisplayTxt(const Char* str);
//void DisplayIR(int sensor, int value);
//void Wait(int delay_ticks);
//void StopRobot(void);
//int  Vel_To_Value(double vel);
//void Vector_Drive(vector V, double omega);
//int  SmartWait(int delay_ticks);


UInt portRef;      // port reference, set by serial library, needed for communication with port
double k=0;
int TURN_DIR=-1;


// find and open MathLib library
// find and open serial port
// open Form1
static int StartApplication() {
  Err error;
	error = SysLibFind(MathLibName, &MathLibRef);
	if (error)
    error = SysLibLoad(LibType, MathLibCreator, &MathLibRef);
	ErrFatalDisplayIf(error, "Can't find MathLib");
	error = MathLibOpen(MathLibRef, MathLibVersion);
	ErrFatalDisplayIf(error, "Can't open MathLib");
  SysLibFind("Serial Library", &portRef);
  SerOpen(portRef, 0, 9600);
  FrmGotoForm(Form1);
	return 0;
}

// Reads and displays value of given analog sensor
int Sensor(int number) {
	// send request for analog value to SSC board
	Err error;
	char buf[10];
	char incoming[]="       ";
	int value=20;
	int len;
	int l;	// loop variable
	StrPrintF(buf, "AD%d \r", number);
	len = StrLen(buf);
	SerSendWait(portRef, -1); // wait for port to clear
	SerSend(portRef, buf, len, &error); // send buffer
	// receive from portRef into incoming buffer using 5 bytes and 3ms timeout
	SerReceive(portRef, incoming, 4, 10, &error);
	// buffer correction -- replace all ASCII non-numbers with spaces
	for (l=0; l<=2; l++)
		if ( incoming[l] == 10 )
			incoming[l]=' ';
	incoming[3] = '\0';
	if (incoming[0] != '1')	// two digit number cut-off
		incoming[2] = '\0';
	SerReceiveFlush(portRef, 5); // flush buffers
	SerSendFlush(portRef);
  value = StrAToI(incoming);
	return value;
}



// Sends value to given servo
void Servo(int number, int speed) {
  Err error;
  char buf[10];
  int len;
  if (speed != 0) {	// make sure speed is not zero -> zero is used for stop and does not change
	speed = 128 + (speed-128)*TURN_DIR; }	// adjust for different bases
																			// WARNING: TURN_DIR can be only +1/-1 only!
  StrPrintF(buf, "SV%d M%d \r", number, speed);
  len = StrLen(buf);
  SerSendWait(portRef, -1); // wait for port to clear
  SerSend(portRef, buf, len, &error); // send buffer
}
// ***** END OF HARDWARE FUNCTIONS *****



// Update all 3 drive servos
void Drive(int sv1, int sv2, int sv3) {
	Servo(1, sv1);
	Servo(2, sv2);
	Servo(3, sv3);
}



// ***** USER INTERFACE FUNCTIONS *****
// Display string (outputs it to Message Field)
void DisplayTxt(const Char* str) {
	FormPtr frm = FrmGetActiveForm();
	FrmCopyLabel(frm, Form1MesLabel, "              ");
	FrmCopyLabel(frm, Form1MesLabel, str);
}

// Displays number (outputs it to Message Field)
void Display(int number) {
	FormPtr frm = FrmGetActiveForm();
	char numbertxt[]="              ";
	FrmCopyLabel(frm, Form1MesLabel, numbertxt);
	StrIToA(numbertxt, number);
	FrmCopyLabel(frm, Form1MesLabel, numbertxt);
}

// Displays IR 1 or 2
void DisplayIR(int sensor, int value) {
	FormPtr frm = FrmGetActiveForm();
	char numbertxt[]="              ";
	StrIToA(numbertxt, value);
	switch(sensor) {
		case 1:
			FrmCopyLabel(frm, Form1IRoneLabel, numbertxt);
			FrmCopyLabel(frm, Form1IRoneLabel, numbertxt);
			break;
		case 2:
			FrmCopyLabel(frm, Form1IRtwoLabel, numbertxt);
			FrmCopyLabel(frm, Form1IRtwoLabel, numbertxt);
			break; }
}

// ***** END OF USER INTERFACE FUNCTIONS *****

// Stops robot.
void StopRobot(void) {
	Drive(0,0,0);
}


// ***** CONVERTER FUNCTIONS *****
// Calculates distance using Power Regression y=a*x^b
// returns value in cm ranging from 10 to 80
int Sensor_To_Range(int x)
{
	double a=2141.72055;
	double b=-1.078867;
	int y=100;
	if (x>0) 	 y = a*pow(x,b);
	if (y<10)  y=10;
	if (y>100) y=100;
	return y;
}

// Calculates servo value using Cubic Regression y=a*x^3+b*x^2+c*x+d
// Converts velocity in rad per second to servo value (0-255)
// vel can only be [-6,6] rad/sec
// x can only be [-1,1] rev/sec
int Vel_To_Value(double vel) {
	double x;
	x = vel / PI / 2;	// convert rad/sec to rev/sec
	if ( (vel>-6) && (vel<6) )
		return (39.8053*x*x*x - 12.6083*x*x + 22.1197*x + 128.4262);
	else
		return 0;
}

// ***** PROGRAM FUNCTIONS *****

// Suspends application for specified number of ticks
void Wait(int delay_ticks) {
	ULong start_time;
	start_time = TimGetTicks();
	while ( TimGetTicks()	< (start_time + delay_ticks));
}

// Pauses application for specified number of ticks
// Can be interrupted by penDown event
// Returns 1 if it was interrupted by penDown
// Returns 0 if delay ended with no interrupts
int SmartWait(int delay_ticks) {
	ULong start_time;
	EventPtr quit_event;
	start_time = TimGetTicks();
	while ( TimGetTicks()	< (start_time + delay_ticks)) {
		EvtGetEvent(quit_event, 1);	// get event using 1ms timeout
		if (quit_event->eType ==	penDownEvent)	// stop if screen was touched anywhere
			return 1; }
	return 0;
}
// ***** END OF PROGRAM FUNCTIONS *****

// Drives robot in a given direction vector V with given angular
// velocity omega.
// r - wheel radius in meters
// b - wheel baseline in meters
void Vector_Drive(vector V, double omega) {
	vector F1(-1.000,0.000), F2(0.500,-0.866), F3(0.866,0.500);
	double omega1, omega2, omega3, b=0.090, r=0.020, h=0;
	omega1 = ( F1*V + b*omega ) / r;	// F1*V is overloaded dot product
	omega2 = ( F2*V + b*omega ) / r;
	omega3 = ( F3*V + b*omega ) / r;
	// makes sure that given path is physically possible
	if ( (omega1>6) || (omega2>6) || (omega3>6) || (omega1<-6) || (omega2<-6) || (omega3<-6) )
		DisplayTxt("Vectors: ERROR!");
	else
		Drive(Vel_To_Value(omega1),Vel_To_Value(omega2),Vel_To_Value(omega3));
}

