(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")


#|
=================================================
Relator interface: Functions called from outside.
================================================

These functions are available to the user and Prodigy for using the
Relator algorithm. Other algorithms usually do not call any other
of Relator's functions directly.

For the details of the data structures used by Relator, see the
"alpine/struct.lisp" and "relator/struct.lisp" files. For the details
of the PDL syntax for describing relevance graphs, see the comment in
the beginning of the "Converting the Relevance Graph" part in the
"relator/build.lisp" file.

The following main functions are for the access by the user and
outside functions (even though some auxiliary functions may also
accessed from outside, because of the code spagettiness):

show-rel-graph      print out the relevane graph, in PDL
build-rel-graph     build a relevance graph
show-rel-tables     print out relevance tables
update-rel-tables   update relevance tables for a new goal
build-rel-tables    build and store positive and negative relevance tables
relevant-vared-p    determine if a PDL-form predicate is relevant
relevant-literal-p  determine if a literal is relevant
|#

(export '(show-rel-graph build-rel-graph show-rel-tables 
  update-rel-tables build-rel-tables relevant-vared-p relevant-literal-p))

;;-------------------------------------------------------------------------
;; Builiding and accessing a relevance graph.

;; Print out the relevance graph for "pspace" in the PDL language.
;; The function either generates a new relevance graph or reads the
;; graph from the plist of "pspace". If a new relevance graph is
;; generated, it is not stored.
;;
;; pspace        the problem space
;; new           takes one of the following two values:
;;                 T  build a new abstraction graph for *pspace*
;;                 nil  read the abstraction graph from the abs-graph slot
;; use-primary   nil (do not use), :side (treat rest like side), 
;;   :prim (... like prim), or :default (as alredy constructed;
;;   if no rel-graph is constructed, as specified in pspace)

(defun show-rel-graph (&key (pspace *current-problem-space*)
		            (new nil) (use-primary nil))
  (declare (type problem-space pspace) (symbol use-primary))
  (if (or new (not (valid-rel-p pspace use-primary)))
    ;; Generate and show a new relevance graph (without storing it).
    (let ((*pspace* pspace)
	  (graph (build-rel-graph :pspace pspace 
                    :use-primary use-primary :store-p nil)))
      (declare (type problem-space *pspace*) (special *pspace*))
      (unless new
        (format t 
          "~&;; The current graph is for :USE-PRIMARY = ~S.
           ~%;; Computing the relevance graph for ~S."
          (problem-space-rel-graph-for pspace) use-primary))
      (if (eq graph :collapse)
        (format t "~&;; Relevance graph has collapsed to a single component.")
        (format t "~&~A" (relator-encode-graph graph))))  ;; Print graph.
    ;; Show a relevance graph in the plist of the problem space.
    (let ((*pspace* pspace)
	  (graph (problem-space-rel-graph pspace)))
      (declare (type problem-space *pspace*) (special *pspace*))
      (cond
        ((null graph) (format t "~&;; Relevance graph is not computed."))
	((eq graph :collapse)
          (format t "~&;; Relevance graph has only one component, ~
            for :USE-PRIMARY = ~S." (problem-space-rel-graph-for pspace)))
	((abs-graph-p graph)
          (format t "~&;; Relevance graph is for :USE-PRIMARY = ~S."
            (problem-space-rel-graph-for pspace))
	  (format t "~%~A" (relator-encode-graph graph)))  ;; Print graph.
	(t
	  (format t
	    "~&Warning: The `:rel-graph' slot of the problem space's plist ~
            contains `~A', which is not a relevance graph." graph))))))


;; Return the relevance graph from the plist of a problem space. If the
;; plist does not have a relevance graph, generate and store a graph.
;;
;; pspace        problem space
;; use-primary   nil (do not use), :side (treat rest like side), 
;;   :prim (... like prim), or :default (as alredy constructed;
;;   if no rel-graph is constructed, as specified in pspace)
;;
;; returned value: the relevance graph of the problem space

(defun get-rel-graph (pspace &key (use-primary nil))
  (declare (type problem-space pspace) (symbol use-primary))
  (if (and (problem-space-rel-graph pspace)
           (valid-rel-p pspace use-primary))
    (problem-space-rel-graph pspace)
    (build-rel-graph :pspace pspace :use-primary use-primary)))


;; Build a relevance graph.
;;
;; pspace        the problem space (by default, *current-problem-space*)
;; name          name of a relevance graph
;; use-primary   nil (do not use). :side (use primary, rest like side), 
;;                or :rest (use primary, rest like primary)
;; store-p       if not nil, store the graph in the ":rel-graph" slot 
;;                 of the problem space
;;
;; returned value: 
;; the resulting relevance graph, if the graph has more than one level;
;; :collapse otherwise

(defun build-rel-graph (&key pspace name (use-primary nil) (store-p t))
  (declare (symbol use-primary)
           (special *current-problem-space*)
           (type problem-space *current-problem-space*))
  (let* ((*pspace* (or pspace *current-problem-space*)) graph)
    (declare (type problem-space *pspace*) (special *pspace*))
    (case use-primary
      ((nil :prim :side) )
      (:default (setf use-primary (any-space-property :use-primary *pspace*)))
      (t (error "build-rel-graph: `~S' not a valid value of :use-primary")))
    (setf graph (initial-rel-graph :name name :use-primary use-primary))
    (unless (final-graph graph)  ;; one level?
      (setf graph :collapse))
    (when store-p
      (setf (problem-space-rel-graph *pspace*) graph)
      (setf (problem-space-rel-graph-for *pspace*) use-primary)
      (set-nil (problem-space-rel-tables-for *pspace*)))
    graph))


;; Determine whether the rel-graph (if any) of the current problem space
;; is valid for the specified use of primary effects (the :default value
;; of "use-primary" means the, whatever rel-graph, it is valid).

(defun valid-rel-p (pspace use-primary)
  `(or (eq ,use-primary :default)
       (eq ,use-primary (problem-space-rel-graph-for ,pspace))))

;;-------------------------------------------------------------------------
;; Building and accessing relevance tables.

;; Print out the list of (positive and negative) relevant predicates.
;; The function either generates new relevance tables or reads them from the 
;; plist of "pspace". If new tables are generated, they are not stored.
;;
;; pspace        problem space
;; new           takes one of the following two values:
;;                 T  build new relevance tables, using rel-graph in pspace
;;                 nil  read the rel-tables from pspace
;; goal-exp      goal expression, used only when generating new rel-tables
;;                 (by default, the goal of the current problem in pspace)
;; decs          declarations of the goal expression (if goal is default, this
;;                 argument is ignored and declarations are for default goal;
;;                 note that the epty list of declarations, for an 
;;                 instantiatied goal, must be explicitly specified)
;; use-primary   nil, :prim, :side, or :default

(defun show-rel-tables (&key (pspace *current-problem-space*) (new nil)
         (goal-exp :default) (decs :default) (use-primary nil))
  (declare (type problem-space pspace) (symbol use-primary))
  (cond
    ;; Generate and show new rel-tables (without storing them).
    (new
       (let ((rel-tables 
               (build-rel-tables :goal-exp goal-exp :decs decs :pspace pspace 
                                 :use-primary use-primary :store-p nil)))
         (format t "~&Relevance tables are for :USE-PRIMARY = ~S.~%~%"
           (problem-space-use-primary-for pspace))
         (show-rel-table (first rel-tables) :negated-p nil)
         (format t "~%")
         (show-rel-table (second rel-tables) :negated-p t)))
    ;; Show rel-tables in the plist of the problem space.
    ((valid-rel-p pspace use-primary)
       (format t "~&Relevance tables are for :USE-PRIMARY = ~S.~%~%"
         (problem-space-use-primary-for pspace))
       (show-rel-table (problem-space-pos-rel-table pspace) :negated-p nil)
       (format t "~%")
       (show-rel-table (problem-space-neg-rel-table pspace) :negated-p t))
    ;; The rel-graph is for a different value of :use-primary.
    (t (format t "~&The current rel-graph is for :USE-PRIMARY = ~S, not ~S."
         (problem-space-use-primary-for pspace) use-primary))))


;; If relevance table is a hash-table, print the list of the stored
;; predicates. Otherwise, print whether all predicates are relevant,
;; all irrelevant, or the table is undefined.
;;
;; rel-table   positive or negative relevance table
;; negated-p   nil of negative table; not nil for positive table

(defun show-rel-table (rel-table &key (negated-p nil))
  (let ((pos-or-neg-string (if negated-p "negative" "positive")))
    (cond
      ((null rel-table) 
         (format t "~%Relevance table for ~A predicates is not computed."
           pos-or-neg-string))
      ((eq rel-table :all)
         (format t "~%All ~A predicates are relevant." pos-or-neg-string))
      ((eq rel-table :none)
         (format t "~%No ~A predicates are relevant." pos-or-neg-string))
      ((hash-table-p rel-table)
         (show-relevant-predicates rel-table :negated-p negated-p))
      (t (format t
	   "~&Warning: The value `~A' for ~A relevance table is ~
            not a legal relevance table." rel-table pos-or-neg-string)))))


;; Print the list of predicates in a relevance table.
;;
;; rel-table   positive or negative relevance table
;; negated-p   nil for negative table; not nil for positive table

(defun show-relevant-predicates (rel-table &key (negated-p nil))
  (declare (type hash-table rel-table))
  (dohash (pred-name entry rel-table)
    (typecase entry
      (symbol  ;; the entry in the table is ":any"
        (if negated-p (format t "~%  ~~ ") (format t "~%  "))
        (format t "~A" (string-downcase pred-name)))
      (list  ;; the entry in the table is a list of argument numbers
        (print-nums pred-name entry :negated-p negated-p))
      (rel-entry  ;; the entry is Rel-Entry (list of typed-argument lists)
        (print-nums pred-name (rel-entry-nums entry) :negated-p negated-p)
        (print-arglists pred-name (rel-entry-arglists entry) 
          :negated-p negated-p)))))


;; Print the list of predicates with given name and given argument numbers.
;;
;; pred-name   predicate name (symbol), e.g. 'on
;; nums        list of argument numbers (integers), e.g. (0 2)
;; negated-p   nil for negative predicate; not nil for positive predicate

(defun print-nums (pred-name nums &key (negated-p nil))
  (declare (type symbol pred-name))
  (dolist (num nums)
    (if negated-p (format t "~%  ~~ ") (format t "~%  "))
    (format t "(~A" (string-downcase pred-name))
    (dotimes (i num) (format t " *"))
    (format t ")")))


;; Print the list of predicates with given name and given argument lists.
;; Every argument list is a list of typed arguments, where every typed
;; argument is a Prodigy object, simple type, or type disjunction. Note
;; that we use objects and types themsleves, not their names.
;;
;; pred-name   predicate name (symbol)
;; nums        list of argument lists
;; negated-p   nil for negative predicate; not nil for positive predicate

(defun print-arglists (pred-name arglists &key (negated-p nil))
  (declare (type symbol pred-name))
  (dolist (arglist arglists)
    (if negated-p (format t "~%  ~~ ") (format t "~%  "))
    (format t "(~A" (string-downcase pred-name))
    (dolist (arg-type arglist)
      (print-arg-type arg-type))
    (format t ")")))


;; Print the argument type, which may be a Prodigy object (e.g. peg-1), 
;; a simple type (e.g. Peg), or a type disjunction (e.g. (or Disk Peg)).
;;
;; arg-type   Prodigy object or type (type itself, not its name)

(defun print-arg-type (arg-type)
  (let ((*encoding* (make-array 0 :fill-pointer t :adjustable t 
                                  :element-type 'excl::string-char)))
    (declare (type string *encoding*) (special *encoding*))
    (alpine-encode-arg arg-type)
    (format t " ~A" *encoding*)))


;; Get positive or negative relevance table. If the relevance tables
;; have not been constructed, create them.
;;
;; pos-or-neg    :pos for table of positive literals, :neg for negative lits
;; graph         relevance graph used in building the relevance table
;;                 (if not specified, then taken from pspace)
;; pspace        problem space (by defatul, *current-problem-space*)
;; use-primary   nil, :prim, :side, or :default

(defun get-rel-table (pos-or-neg &key graph pspace (use-primary nil))
  (declare (symbol use-primary)
           (special *current-problem-space*)
           (type problem-space *current-problem-space*))
  (unless pspace 
    (setf pspace *current-problem-space*))
  ;; If either positive or negative table has not been constructed,
  ;; create both tables (normally, either both tables are created or
  ;; both are not; if only one table is present, something is wrong).
  (unless (and (valid-rel-p pspace use-primary)
               (problem-space-pos-rel-table pspace) 
               (problem-space-neg-rel-table pspace))
    (build-rel-tables :graph graph :pspace pspace :use-primary use-primary))
  ;; Return the table.
  (case pos-or-neg
    (:pos (problem-space-pos-rel-table pspace))
    (:neg (problem-space-neg-rel-table pspace))
    (otherwise 
     (error "get-rel-table: ~S not valid value of `pos-or-neg'" pos-or-neg))))


;; Generate relevance tables for the given goal expression, or for the
;; current problem (if not yet generated).
;;
;; goal-exp   goal experession for which abstraction is generated;
;;   T or :default  if relevance tables are for the current problem;
;;   nil if all literals are relevant (for problem-independent situatuations)
;; decs       declarations of the goal expression (if goal is T, :default, or
;;   nil, this argument is ignored;note that the epmty list of declarations,
;;   of declarations, for an instantiatied goal, must be explicitly specified)
;; pspace     problem space (if nil, current problem space)
;; use-primary   nil, :prim, :side, or :default

(defun update-rel-tables (&key goal-exp (decs :default) pspace 
                               (use-primary nil))
  (declare (symbol use-primary))
  (case goal-exp
      ((nil)  ;; everything is relevant
        (setf (problem-space-pos-rel-table pspace) :all)
        (setf (problem-space-neg-rel-table pspace) :all)
        (set-nil (problem-space-rel-tables-for pspace)))
      ((t :default)  ;; relevance for the current problem
        (unless (and (valid-rel-p pspace use-primary)
                     (eq (problem-space-rel-tables-for pspace)
                         (current-problem pspace)))
          (build-rel-tables :pspace pspace :use-primary use-primary)))
      (otherwise  ;; relevance for specified goals
        (build-rel-tables :goal-exp goal-exp :decs decs 
          :pspace pspace :use-primary use-primary))))


;; Build positive and negative relevance tables for a specific goal
;; and store them in the problem space's plist.
;;
;; If all positive or all negative predicates are relevant, store
;; ":all" instead of a relevance table. If the problem space does
;; not have a relevance graph, generate and store a graph.
;;
;; goal-exp      goal expression (by default, goal of the current problem)
;; decs          declarations of the goal expression (if goal is default, this
;;                 argument is ignored and declarations are for default goal;
;;                 note that the epmty list of declarations, for an 
;;                 instantiatied goal, must be explicitly specified)
;; graph         relevance graph used in building the relevance table
;;                 (if not specified, then taken from pspace)
;; pspace        problem space (by defatul, *current-problem-space*)
;; use-primary   nil (do not use), :side (treat rest like side), 
;;   :prim (... like prim), or :default (as alredy constructed;
;;   if no rel-graph is constructed, as specified in pspace)
;; store-p       if not nil, store the rel-tables in the ":pos-rel-table"
;;                 and ":neg-rel-table" slots of the problem space
;;
;; returned value:
;; two-element list, where the first element is rel-table of positive
;; predicates and second is rel-table of negative predicates

(defun build-rel-tables (&key (goal-exp :default) (decs :default) graph
                              pspace (use-primary nil) (store-p t))
  (declare (symbol use-primary)
           (special *current-problem-space*)
           (type problem-space *current-problem-space*))
  (let* ((*pspace* (or pspace *current-problem-space*))
         (graph (or graph (get-rel-graph *pspace* :use-primary use-primary)))
         current-problem current-goal rel-goal-preds rel-tables)
    (declare (type problem-space *pspace*) (special *pspace*))
    (if (eq graph :collapse)  ;; one-component relevance graph?
      (setf rel-tables '(:all :all))
      (progn
        ;; Determine the goal expression.
        (cond 
          ((eq goal-exp :default)
            (setf current-problem
              (getf (problem-space-plist *pspace*) :current-problem))
            (unless current-problem
              (error "build-rel-tables: The ':goal-exp' keyword is ~
                not specified and the problem space has no current problem."))
            (unless (eq decs :default)
              (format t "Warning: build-rel-tables: The ':decs' keyword ~
                is specified for the current goal; ignoring the value of ~
                ':decs' and taking the declarations of the current problem."))
            (setf current-goal (cdr (problem-goal current-problem)))
            (case (length current-goal)
              (1 (setf goal-exp (first current-goal)) 
                 (setf decs nil))
              (2 (setf goal-exp (second current-goal)) 
                 (setf decs (first current-goal)))
              (otherwise
                 (error "Wrong number of subfields in goal the ~
                   specification (should be 2 or 3)~%"))))
          ((and (eq decs :default) 
                (some #'is-variable-p (leaves-to-elts goal-exp)))
            (error "build-rel-tables: The ':goal-exp' keyword is explicitly ~
              specified, but the ':decs' keyord is not specified."))
          ((eq decs :default)
            (set-nil decs)))
        ;; Find relevant typed predicates.
        (setf rel-goal-preds (find-rels goal-exp decs graph))
        ;; Generate relevance tables.
        (setf rel-tables (hash-rels graph rel-goal-preds))))
    ;; Store the relevance tables in the problem space's plist.
    (when store-p
      (setf (problem-space-pos-rel-table *pspace*) (first rel-tables))
      (setf (problem-space-neg-rel-table *pspace*) (second rel-tables))
      (setf (problem-space-rel-tables-for *pspace*)
        (if (eq goal-exp :default) (current-problem *pspace*))))
    rel-tables))

;;-------------------------------------------------------------------------
;; Determining the relevance of predicates and literals.

;; Determine if a predicate from a PDL expression is relevant,
;; according to relevance tables in the plist of pspace.
;;
;; Predicate is represented as a list, where the first element is the
;; predicate name and the other elements are arguments. The arguments
;; are objects and variables, represented by their names (rather than
;; pointers to the objects themselves). For example, (on small <peg>).
;;
;; predicate   predicate from a PDL expression
;; decs        declarations of the variables
;; pspace      problem space
;; negated-p   T for a negated predicate; nil for a positive predicate;
;;               :either for an unsigned predicate, which is considered
;;               relevant if either its asserting or negation is relevant
;; use-primary   nil (do not use), :side (treat rest like side), 
;;   :prim (... like prim), or :default (as alredy constructed;
;;   if no rel-graph is constructed, as specified in pspace)
;;
;; returned values: non-nil if the predicate is relevant; nil otherwise

(defun relevant-vared-p 
    (predicate decs pspace &key (negated-p nil) (use-primary nil))
  (declare (type problem-space pspace))
  (let ((*pspace* pspace))
    (declare (special *pspace*))
    (relevant-typed-p 
      (car predicate)
      (mapcar #'(lambda (arg-name) 
                        (arg-name-to-type arg-name decs))
                    (cdr predicate))
      pspace
      :negated-p negated-p
      :use-primary use-primary)))


;; Determine if a literal is relevant, according to relevance tables
;; in the plist of pspace.
;;
;; A literal is represented as a list, where the first element is the
;; predicate name and the other elements are object names; for example,
;; (on small peg-1).
;;
;; The "relevant-vared-p" can determine the relevance of a literal, but
;; "relevant-literal-p" should work faster.
;;
;; literal       literal, represented as a list
;; pspace        problem space
;; negated-p     T for a negated literal; nil for a positive literal;
;;                 :either for an unsigned literal, which is considered
;;                 relevant if either its asserting or negation is relevant
;; use-primary   nil, :prim, :side, or :default
;;
;; returned values: non-nil if the predicate is relevant; nil otherwise

(defun relevant-literal-p 
    (literal pspace &key (negated-p nil) (use-primary nil))
  (declare (type problem-space pspace))
  (relevant-typed-p
    (car literal)
    (mapcar #'(lambda (object-name)
                (or (and (symbolp object-name) 
                         (object-name-to-object object-name pspace))
                    (type-name-to-type :Top-Type pspace)))
            (cdr literal))
    pspace
    :negated-p negated-p
    :use-primary use-primary))


;; Determine if a typed predicate is relevant, according to relevance
;; tables in the plist of pspace. If the problem space does not have
;; relevance tables, create them.
;;
;; name          predicate name (without "~")
;; args          predicate's argument list (:any, number, or typed arguments)
;; pspace        problem space
;; negated-p     T for a negated predicate; nil for a positive predicate;
;;                 :either for an unsigned predicate, which is considered
;;                 relevant if either its asserting or negation is relevant
;; use-primary   nil, :prim, :side, or :default
;;
;; returned values: non-nil if the predicate is relevant; nil otherwise

(defun relevant-typed-p 
    (name args pspace &key (negated-p nil) (use-primary nil))
  (declare (symbol name) (type problem-space pspace))
  (if (eq negated-p :either)
    (or (relevant-typed-p name args pspace :negated-p nil)
        (relevant-typed-p name args pspace :negated-p t))
    (let ((rel-table 
            (get-rel-table (if negated-p :neg :pos) 
              :pspace pspace :use-primary use-primary)))
      (cond
        ((null rel-table) (error "relevant-typed-p: There is no rel-table"))
        ((eq rel-table :all) t)
        ((eq rel-table :none) nil)
        (t (relevance-test name args rel-table))))))

#|
==================================================
Run-time functions for using the relevance tables.
==================================================

We call these functions during problem solving or learning, not only
when building the relevance graph or relevance tables.
|#

;; Given a predicate and a (positive or negative) relevance table,
;; determine whether the predicate is relevant.
;;
;; name        predicate name (without "~")
;; args        predicate's argument list (:any, number, or typed arguments)
;; rel-table   relevance table (must not be ":all" or :none")
;;
;; returned value: non-nil if the predicate is relevant; nil otherwise

(defun relevance-test (name args rel-table)
  (declare (symbol name))
  (let ((entry (gethash name rel-table)))
    (typecase entry
      (null nil)  ;; predicate name is not in the table
      (symbol t)  ;; the entry in the table is ":any"
      (list  ;; the entry in the table is a list of argument numbers
        (relevance-nums-test args entry))
      (rel-entry  ;; the entry is Rel-Entry (list of typed-argument lists)
        (or (relevance-nums-test args (rel-entry-nums entry))
            (relevance-arglist-test args entry))))))


;; Determine whether predicate arguments match the list of argument
;; numbers. That is, the number of arguments of a predicate equals one
;; of the numbers in the argument-number list.
;;
;; args   argument list (:any, number, or typed arguments)
;; nums   list of argument numbers (note that this list is sorted,
;;          but the current implementation does use the order)
;;
;; returned value:
;; non-nil if arguments match the list (i.e. the predicate is relevant);
;; nil otherwise

(defmacro relevance-nums-test (args nums)
  `(if ,nums
     (typecase ,args
       (symbol t)  ;; args is ":any"
       (number (member ,args ,nums))  ;; args represent the argument number
       (otherwise  ;; args is a list of typed arguments
         (member (length ,args) ,nums :test #'=)))))


;; Determine whether predicate arguments match "arglists" of rel-entry.
;; That is, the arguments intersect with some element of "arglists".
;;
;; If we have already tested these particular args against this
;; rel-entry, then they are stored in the hash table in the "tested"
;; slot of rel-entry.  We lookup the table to see if they are
;; relevant. If we have not yet tested these args, we store the
;; results of matching in the "tested" hash table, so that we do not
;; need to match them in the future.
;;
;; Matching of args with arglists is time-consuming, because it calls a
;; time-consuming function "arg-intersection-p" (which is in the
;; "alpine/build" file) and the number of calls is linear in the
;; number of elements of "arglists".
;;
;; args    argument list (:any, number, or typed arguments)
;; entry   rel-entry, against which we match "args"
;;
;; non-nil if arguments match the rel-entry (i.e. the predicate is relevant);
;; nil otherwise

(defun relevance-arglist-test (args entry)
  (declare (type rel-entry entry))
  (typecase args
    (symbol t)  ;; args is ":any"
    (number  ;; args represent the argument number
      (member args (rel-entry-tested entry) :key #'length))
    (otherwise  ;; args is a list of typed arguments
      (let ((relevance (gethash args (rel-entry-tested entry))))
        (cond
          ;; "args" have already been tested and, thus, stored in hash-table.
          (relevance (eq relevance 'relevant))
          ;; Matching againts "arglists" shows that the predicate is relevant.
          ;; Store this fact in the hash-table.
          ((member args (rel-entry-arglists entry) 
                    :test #'args-intersection-p)
             (setf (gethash args (rel-entry-tested entry)) 'relevant) 
             t)
          ;; Predicate is not relevant; store this fact in the hash-table.
          (t (setf (gethash args (rel-entry-tested entry)) 'irrelevant) 
             nil))))))
