#-clisp
(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")


#|
=========================================================
Converting the PDL description of relevance into a graph.
=========================================================

These functions convert a PDL description of a relevance graph into an
abs-graph structure. The functions here are very similar to the functions 
for converting abstraction into an abs-graph, in the "alpine/load" file.

The input of the top-level conversion function is the list that 
encode the relevance graph, without the word "Relevance". The 
input graph must not be collapsed.

The functions do not check the syntax.  The syntax-check function,
"relator-syntax-graph", must be called before convering the
description into an abs-graph. The functions, however, provide some
semantic checking.
|#


;; Convert a description of a relevance graph into into the graph
;; itself. The input of the function does not include the word
;; "Relevnace"; it begins with the name of the relevance graph.
;;
;; The description is a list of relevance-graph components and
;; ordering constraints, with the keyword :order between.
;;
;; graph-code   PDL encoding of a relevance hierarchy
;;
;; returned values: the relevance graph

(defun relator-decode-graph (graph-code)
  (unless (listp graph-code)
    (error "relator-decode-graph: Relevance-graph description is not list."))
  (let ((graph (create-empty-graph)))
    (declare (type abs-graph graph))
    ;; Decode the graph name.
    (if (symbolp (first graph-code))
      (setf (abs-graph-name graph) (pop graph-code)))
    ;; Decode components.
    (do* ((comp-code (pop graph-code) (pop graph-code)))
         ((or (null comp-code) (member comp-code '(user::order :order))))
      (push (relator-decode-component comp-code) 
            (abs-graph-components graph)))
    (setf (abs-graph-components graph) 
          (nreverse (abs-graph-components graph)))
    (alpine-enumerate-components graph)
    ;; Decode ordering constraints.
    (cond 
      (graph-code  ;; ordering constraints explicitly described?
        (dolist (order graph-code)
          (alpine-decode-order order graph))
        (dolist (component (abs-graph-components graph))
          (del-duplicates (abs-component-out component))
          (del-duplicates (abs-component-in component)))
        (alpine-check-top-sort graph))
      (t  ;; otherwise, the components totally ordered
        (alpine-total-order graph)))
    ;; Create the list of abstraction nodes, mark subset relations,
    ;; and tolopolgically sort the list of nodes on subset relations.
    (setf (abs-graph-nodes graph)
      (apply #'append 
             (mapcar #'abs-component-nodes (abs-graph-components graph))))
    (relator-subsets-and-intersections graph)
    (alpine-subset-sort graph)
    graph))


;; Convert a relevance-graph component description into the component
;; itself. Set the name of the component (if any), the list of nodes,
;; and the "loop-p" slot (showing whether the component is looped).
;;
;; A component description is a list of typed predicates. The 
;; first two elements may be a component name and "="; the last two 
;; elements may be "- no-loop" or "- :no-loop", showing that the
;; component is not looped; for example:
;; "(Component-1 = (on 'large Peg) - no-loop)" or
;; "(Component-0 = (on 'small Peg) (on 'medium Peg))".
;;
;; comp-code   description of a component
;;
;; returned value: the resulting component

(defun relator-decode-component (comp-code)
  (unless (listp comp-code)
    (error "relator-decode-component: ~S is not a component." comp-code))
  (let ((comp-copy comp-code)
        (component (make-abs-component))
        (no-loop-p nil))
    (declare (type abs-component component))
    ;; Decode the component name.
    (when (eq (second comp-copy) 'user::=)
      (setf (abs-component-name component) (car comp-copy))
      (setf comp-copy (cddr comp-copy)))
    (setf comp-copy (reverse comp-copy))
    ;; Check whether the component is looped.
    (when (and (member (first comp-copy) '(user::no-loop :no-loop))
               (eq (second comp-copy) 'user::-))
      (setf no-loop-p t)
      (setf comp-copy (cddr comp-copy)))
    ;; Decode the nodes (note that the nodes returned by the
    ;; "relator-decode-node" function are looped).
    (setf (abs-component-nodes component)
      (mapcar #'relator-decode-node (reverse comp-copy)))
    (cond 
      ((and no-loop-p (short-p (abs-component-nodes component)))
        (setf (abs-node-loop-p (car (abs-component-nodes component))) nil))
      (no-loop-p
        (error "relator-decode-component: Component `~S' has more than ~
          one predicate; therefore, it must have the loop." comp-code)))
    ;; Set the "component" filed of the nodes.
    (dolist (node (abs-component-nodes component))
      (setf (abs-node-component node) component))
  component))


;; Convert a (possibly negated) typed predicate into a node of the
;; relevance graph. Set the "name" and "args" fields of the node,
;; and set the "loop-p" filed to T (by default, nodes are looped).
;;
;; A typed predicate is a predicate name (e.g. 'on), a
;; predicate name with the number of arguments (e.g. (on . 2), a
;; list, the first element of which is the predicate name and the
;; other elements are argument types (e.g. (on Disk Peg)), or any
;; of these negated (e.g. (~ on) or (~ (on Disk Peg))).
;;
;; If the predicate is negated, cons "~" to its name, e.g. (~ . on).
;; If only a predicate name is specified, set the "args" field of the
;; node to :any. If only the argument number is specified, set "args"
;; to this number.
;;
;; pred   possibly negated typed predicate
;;
;; returned value: the node

(defun relator-decode-node (pred)
  (let ((node (make-abs-node :loop-p t))
        (negated-p nil))
    ;; Check whether the predicate is negated.
    (when (and (consp pred) (eq (car pred) 'user::~))
      (setf negated-p t)
      (setf pred (second pred)))
    ;; Determine the predicate name and arguments.
    (cond 
      ((symbolp pred)
        (setf (abs-node-name node) pred)
        (setf (abs-node-args node) :any))
      ((and (listp pred) (symbolp (car pred)) (natural-p (cdr pred)))
        (setf (abs-node-name node) (car pred))
        (setf (abs-node-args node) (cdr pred)))
      ((listp pred)
        (setf (abs-node-name node) (car pred))
        (setf (abs-node-args node)
              (mapcar #'alpine-decode-arg (cdr pred))))
      (t
        (error "relator-decode-node: ~S is not a predicate." pred)))
    ;; If negated, cons "~" to the name.
    (if negated-p
      (push '~ (abs-node-name node)))
    node))

;;-------------------------------------------------------------------------

;; Find all subset-relations between nodes and mark them by subset-links.
;;
;; If the same node occurs twice in the graph, or if two nodes from
;; different components intersect and neither of them is a subset of the 
;; other, signal an error.
;;
;; graph   the relevance graph
;;
;; local variables (parameters of the main loop):
;; node1         a node from the graph's list (goes through all nodes)
;; other-nodes   the list of nodes following node1 in the graph's list

(defun relator-subsets-and-intersections (graph)
  (do ((node1 (car (abs-graph-nodes graph)) (car other-nodes))
       (other-nodes (cdr (abs-graph-nodes graph)) (cdr other-nodes)))
      ((null other-nodes))
    (dolist (node2 other-nodes)
      (cond 
        ((same-node-p node1 node2)
          (error "Predicate `~S' occurs more then once in relevance graph."
                 (alpine-node-to-encoding node1)))
        ((node-subset-p node1 node2)
          (add-subset-link node1 node2))
        ((node-subset-p node2 node1)
          (add-subset-link node2 node1))
        ((and (diff (abs-node-component node1) (abs-node-component node2))
              (node-intersection-p node1 node2))
          (error "In relevance graph, set of literals corresponding to ~
            predicate `~S' intersects with set of literals corresponding ~
            to predicate `~S', these sets are in different components, and ~
            neither of them is a subset of the other." 
                 (alpine-node-to-encoding node1)
                 (alpine-node-to-encoding node2)))))))


#|
==========================================================
Check the syntax of the relevance-graph PDL description.
==========================================================

These functions check the syntactic validity of the relevance-graph
description in PDL. The functions here are very similar to the 
functions for checking the syntax of abstraction, in the 
"alpine/load" file.

The input of the top-level syntax-checking function is the list that
encode the relevance graph, without the word "Relevance". The functions 
only check the syntax; they do not create any data structures.
|#


;; Check the validity of the description of a relevance graph.
;; The input of the function does not include the word "Relevance".
;; The description is valid if it is a list of relevance-graph components
;;   and ordering constraints, with the keyword :order between.
;; For more details on the description syntax, see the comment in the 
;;   beginning of the "Converting the Relevance Graph" part in the
;;   "build.lisp" file.
;;
;; graph   description of the relevance graph
;;
;; dynamically-scoped global variables:
;; *component-names*   list of the names of the already-checked components
;; *infinite-types*    list of infinite-type names used in typed predicates

(defun relator-syntax-graph (graph *pspace*)
  (declare (type problem-space *pspace*) (special *pspace*))
  (let ((*component-names* nil)
        (*infinite-types* nil))
    (declare (special *component-names* *infinite-types*))
    ;; Check that the relevance-graph description is a list.
    (unless (listp graph)
      (error "Relevance-graph description is not a list."))
    ;; Check that the relevance graph is not collapsed.
    (if (or (member (first graph) '(user::collapse :collapse))
            (member (second graph) '(user::collapse :collapse)))
      (error "Relevance graph is collapsed."))
    ;; Check the validity of the relevance-graph name.
    (if (symbolp (car graph))
      (relator-syntax-graph-name (pop graph)))
    ;; Check that the relevance graph has at least two components.
    (if (or (null graph) (member (first graph) '(user::order :order)))
      (error "Relevance graph has no components."))
    (if (or (singleton-p graph) (member (second graph) '(user::order :order)))
      (error "Relevance graph has only one component."))
    ;; Check the validity of components.
    (do* ((component (pop graph) (pop graph)))
         ((or (null component) (member component '(user::order :order))))
      (unless (listp component)
        (error "`~S' in relevance graph is not a component." component))
      (relator-syntax-component component))
    ;; If typed predicates contain infinite types,
    ;; print a warning with the list of these types.
    (when *infinite-types*
      (format t "~&Warning: Relevance-graph description contains infinite ~
                types (`~S'~{, `~S'~}).~%These types will be replaced with ~
                :TOP-TYPE." (car *infinite-types*) (cdr *infinite-types*)))
    ;; Check the validity of ordering constraints.
    (when graph  ;; any ordering constraints?
      (dolist (order graph)
        (unless (listp order)
          (error "`~S' in relevance-graph description is not an ~
                 ordering constraint." order))
        (relator-syntax-order order)))))


;; Check the validity of the relevance-graph name.
;; "graph-name" must be a symbol.
;;
;; graph-name    name of the relevance graph
;; graph-names   names of other relevance graphs

(defun relator-syntax-graph-name (graph-name &optional graph-names)
  (declare (symbol graph-name))
  (unless graph-name
     (error "Relevance-graph name may not be NIL."))
  (if (member graph-name graph-names)
     (error "More than one relevance graph has name `~S'." graph-name)))


;; Check the validity of a relevance-graph component.
;;
;; A component description is valid if it is a list, the first element 
;; of which is component name, the second element is "=", and the
;; other elements are (possibly negated) typed predicates; for example, 
;; (Component-0 = (on 'small Peg) (~ (on 'medium Peg))).
;;
;; The first two elements, the name and "=", may be absent; for example, 
;; ((on 'small Peg) (~ (on 'medium Peg))). The last two elements may 
;; be "- no-loop" or "- :no-loop", which means that the component
;; is not looped. For example, (Component-0 = (on 'small Peg) - no-loop).
;;
;; "component" must be a list.
;;
;; component   description of a component of the relevance graph

(defun relator-syntax-component (component)
  (let ((comp-copy component))
    ;; Check the name of the component.
    (when (eq (second component) 'user::=)
      (unless (symbolp (car component))
        (error "`~S' is not a component name." (car component)))
      (relator-syntax-component-name (car component))
      (setf comp-copy (cddr component)))
    (setf comp-copy (reverse comp-copy))
    ;; Check the "- :no-loop" constract in the end of the description.
    (when (eq (second component) 'user::-)
      (unless (member (car component) '(user::no-loop :no-loop))
        (error "The last symbol, after `-', in the component description ~
          `~S' is not `no-loop' or `:no-loop'." component))
      (setf comp-copy (cddr comp-copy))
      (unless (short-p comp-copy)
        (error "Component `~S' has more than one predicate; therefore, ~
          it must have the loop." component)))
    ;; Verify that the component has at least one predicate.
    (unless comp-copy
      (error "Component `~S' has no predicates." component))
    ;; Check (possibly negated) typed predicates.
    (dolist (pred component)
      (if (eq (car pred) 'user::~)  ;; negated?
        (alpine-syntax-pred (second pred))
        (alpine-syntax-pred pred)))))


;; Check the validity of the name of a component.
;; The name is valid if it is not nil and differs from the
;;   names of the previously checked components.
;; After checking the name, add it to the list of names, *component-names*.
;; "component-name" must be a symbol.
;;
;; component-name   name of a component of the relevance graph

(defun relator-syntax-component-name (component-name)
  (declare (symbol component-name) (special *component-names*))
  (unless component-name
    (error "`NIL' is not a valid name for a relevance-graph component."))
  (if (member component-name *component-names*)
    (error "More than one relevance-graph component has name `~S'." 
           component-name))
  (push component-name *component-names*))


;; Check the validity of a component ordering.
;;
;; The ordering element is valid if its first element is an existing 
;; component name, the second element is ">", and the other elements
;; are the existing component names distinct from the first component
;; name. There must be at least one component name after ">"; 
;; for example (Component-2 > Component-0 Component-1).
;;
;; "order" must be a list.
;;
;; order   component ordering

(defun relator-syntax-order (order)
  (declare (special *component-names*))
  (unless (and (eq (second order) 'user::>) (cddr order))
    (error "`~S' in relevance-graph description is not a valid ordering." 
           order))
  (dolist (component-name (cons (car order) (cddr order)))
    (unless (symbolp component-name)
      (error "`~S' in relevance-graph ordering description is not a ~
             component name." component-name))
    (unless (member component-name *component-names*)
      (error "`~S' in relevance-graph ordering description is not a ~
             valid component name." component-name)))
  (dolist (component-name (cddr order))
    (if (eq component-name (car order))
      (error "Component `~S' is in the left and right side of the same
             ordering in the relevance-graph description." component-name))))
