#|
============================
General Comments to Relator.
============================
Required files.
---------------
The use of Relator with Prodigy requires to load all the files of my
version of Prodigy (from the "planner" subdirectory), all Alpine files 
(from the "alpine" subdirectory), and the four Relator files 
("alpine/struct", "alpine/build", "alpine/load", and "alpine/use").

Comparison with Alpine.
-----------------------
Construction of the relevanve graph is very similary to constructing
the abstraction graph in Alpine (see the "alpine/build" file). The
implementation closely follows that of the Alpine; we use all of the
Alpine's data structures (see the "alpine/struct" file) and many of
Alpine's procedures. 

We represent Prodigy objects and types by the pointers to the objects
and types themselves, not by their names, in the same way as an Alpine.
For example, a typed predicate may be (on #<TYPE: disk> #<TYPE: peg>).
This representation is different from Prim, where we represent objects
and types by the names; for example, (on Disk Peg).

The following are the main differences from Alpine:

(1) The constraints (edges) between the preconditions and effects of
operators and control rules are different from the Alpine's constraints.
Relator (unlike Alpine) treats opetors in the same way as inference rules.

(2) We do not distinguish between static and nonstatic predicates. The
relevance graph contains both statics and nonstatics.

(3) We distinguish between a predicate and its negation; a positive 
and negative predicate make two different nodes of a relevance graph.
Since the abs-node data structure does not have a field for negation,
we use different names to distinguish between a positive and negative
predicate. The name of a positive-predicate node is just the predicate's
name, say "pred-name", and the name of a negative-predicate node is
"(~ . pred-name)". 

(4) When outputing a relevance graph to a file, we store the information
which of its components are looped. A component is looped if there is
an edge from this component to itself in the transitive closure of the
relevanve graph. We use the information on which components are looped
in generating a relevance table.

(5) The graph does not contain predicates that are always irrelevant, which 
are the predicates that occur only side effects, preconditions of rules 
without primary effects, and  preconditions of static inference rules.
|#


#-clisp
(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")


#|
================
Data structures.
================

We use all data structures from the "alpine/struct" file" and also the
following data structures:

rel-entry: 
Entry of a relevance table, for a predicate name, which shows which
arguments make the predicate relevant.

We do not use global variables in generating a relevance table from a
relevanve graph. We, however, use the following global variables in
generating a relevance graph. The Relator-interface functions (see the
"relator/use" file) define these globale variables before calling the
graph-generating functions:

*pspace*: 
problem space for which we generate the relevance graph
|#

;; Entry of a relevance table, which corresponds to a predicate name;
;; it shows which arguments make the predicate relevant.

(defstruct (rel-entry (:print-function print-rel-entry))
  (nums nil)     ;; matching argument numbers, e.g. (0 2 3); if the number of
                 ;;   arguments of a pred is in the list, then it is relevant
  (arglists nil) ;; matching lists of typed arguments, e.g. 
                 ;;   ((Small Peg) (Medium Peg)); if the typed arguments of
                 ;;   a pred intersect one of these lists, then it is relevant
  (tested nil))  ;; hash-table of the typed argument lists of predicates
                 ;;   that has been matched with "arglists"


;; Print out an entry, of the type "rel-entry", of a relevance table;
;; for example, #<RelEntry: [ON (0 2) ..]>.

(defun print-rel-entry (entry stream z)
  (declare (type rel-entry entry) (stream stream) (ignore z))
  (princ "#<RelEntry: " stream)
  (print-short-rel-entry entry stream)
  (princ ">" stream))


;; Print out a shorter version of a relevance entry, which includes the
;; values of the "nums" field (if not nil) and a dot for each element
;; of "arglists"; for example, [(0 2) ....].

(defun print-short-rel-entry (entry stream)
  (declare (type rel-entry entry) (stream stream))
  (princ "[" stream)
  (when (rel-entry-nums entry) 
    (princ (rel-entry-nums entry) stream)
    (princ " " stream))
  (dotimes (i (length (rel-entry-arglists entry))) (princ "." stream))
  (princ "]" stream))


;; Show the data in a relevance entry.
;;
;; Print the argument number and the list of nodes.

(defun show-rel-entry (entry &optional (stream t))
  (declare (type rel-entry entry) (stream stream))
  (let ((hash-size (hash-table-count (rel-entry-tested entry))))
    (format stream "~&- Relevance Entry -")
    (when (rel-entry-nums entry)
      (format stream "~%Matching argument numbers:~%  ~{ ~D~}" 
        (rel-entry-nums entry)))
    (when (rel-entry-arglists entry)
      (format stream "~%Matching argument lists:")
      (dolist (arglist (rel-entry-arglists entry))
        (format stream "~%  (")
        (print-args arglist stream)
        (format stream ")")))
    (when (> hash-size 0)
      (format stream "~%~D entries in the table of tested lists" hash-size))))
