#-clisp
(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")


#|
=========================================
Creating the initial graph of predicates.
=========================================

We convert preconditions and effect of the operators and inference rules
into nodes of the relevance graph and imposing constraints.

The relevance graph, unlike an abstraction graph, contains static
predicates. If a predicate does not have any constraints, we do not
add it to the graph. A predicate does not have constraints only if it
may never become a subgoal (e.g., a side effect).
|#

;; Create the initial relevance graph.
;;
;; Add all the preconditions and effects of operators and nonstatic
;; inference rules to the graph and impose constraints between
;; preconditions and effects of every rule.
;;
;; name          name of the relevance graph
;; operators     list of operators (by default, all operators in the domain)
;; eagers        list of eager inference rules (by default, all eager rules)
;; lazys         list of lazy inference rules (by default, all lazy rules)
;; use-primary   nil (do not use), :side (treat rest like side),
;;                 or :rest (treat rest like primary)
;;
;; returned value: the initial relevance graph

(defun initial-rel-graph (&key (name nil)
                               (operators :default) 
                               (eagers :default)
                               (lazys :default)
                               (use-primary nil))
  (declare (type abs-graph graph) (symbol use-primary)
           (special *pspace*) (type problem-space *pspace*))
  ;; Set default lists of operators and inference rules.
  (if (eq operators :default)
    (setf operators (problem-space-operators *pspace*)))
  (if (eq eagers :default)
    (setf eagers (problem-space-eager-inference-rules *pspace*)))
  (if (eq lazys :default)
    (setf lazys (problem-space-lazy-inference-rules *pspace*)))
  (if (eq use-primary :default)
    (setf use-primary (any-space-property :use-primary *pspace*)))
  ;; Create the initial relevance graph.
  (let ((graph (create-empty-graph name)))
    (dolist (operator operators)
      (unless (eq (rule-name operator) '*finish*)
        (add-rule-to-rel-graph operator graph use-primary)))
    (dolist (inf-rule (append eagers lazys))
      (unless (static-inference-rule-p inf-rule)
        (add-rule-to-rel-graph inf-rule graph use-primary)))
    graph))

;;-------------------------------------------------------------------------

;; Convert all preconditions and primary effects of a rule into nodes
;; of a relevance graph, add these nodes to the graph (unless they are
;; already in the graph), and add all necessary edges.
;;
;; We add edges from every simple primary effect and every primary
;; action of a conditional effect to every precondition. For every
;; conditional effect, we add edges from all its primary actions to
;; all its conditions (see the "add-cond-eff-to-rel-graph" function).
;;
;; rule          an operator or inference rule
;; graph         the abstraction graph
;; use-primary   nil (do not use), :side (treat rest like side),
;;                 or :rest (treat rest like primary)

(defun add-rule-to-rel-graph (rule graph use-primary)
  (declare (type rule rule) (type abs-graph graph) (symbol use-primary))
  (let* ((precond-decs (second (rule-precond-exp rule)))
         (effect-decs (append (second (rule-effects rule)) precond-decs))
         precond-nodes prim-eff-nodes)
    ;; Add precondition nodes to the graph.
    (setf precond-nodes (delete-duplicates (add-precond-exp-to-rel-graph 
      (third (rule-precond-exp rule)) precond-decs graph)))
    ;; Add effect nodes to the graph.
    (setf prim-eff-nodes
      (add-effs-to-rel-graph (rule-add-list rule) (rule-del-list rule)
        effect-decs graph use-primary))
    ;; Add edges from effects to preconditions.
    (dolist (precond-node precond-nodes)
      (dolist (prim-eff-node prim-eff-nodes)
        (add-directed-edge prim-eff-node precond-node)))))


;; Convert all primary effects of a rule into nodes of a relevance
;; graph and add these nodes to the graph (unless they are already in
;; the graph).
;;
;; For conditional primary effects, add their conditions to the graph
;; and add edges from actions to conditions.
;;
;; add-list      rule's add-list
;; del-list      rule's del-list
;; decs          declarations from the precond-exp of the rule
;; graph         the relevance graph
;; use-primary   way of using primary effects (nil, :side, or :prim)
;;
;; returned value:
;; the list of abs-nodes corresponding to the simple primary effects
;; and actions of the conditional primary effects of the rule; if
;; several effects correspond to the same node, then this node occurs
;; several times in the list; if the rule has no primary effects (that
;; is, the effect list is empty), return nil

(defun add-effs-to-rel-graph (add-list del-list decs graph use-primary)
  (declare (type abs-graph graph) (symbol use-primary))
  (let ((del-remains (copy-list del-list))
        (prim-eff-nodes nil) del-effs)
    ;; Add the nodes for the add-list to the graph. If a conditional
    ;; effect have both add and del effects, we call the
    ;; "add-eff-list-to-rel-graph" function on both of them, and
    ;; remove the corresponding effect list from del-remains (which is
    ;; a copy of del-list). This coupling of the add-actions and
    ;; del-actions of a conditional effect prevents a double attemp of
    ;; adding the effect's conditions.
    (dolist (add-effs add-list)
      ;; Find del-actions of the same conditional effect.
      (setf del-effs
        (find (effect-cond-conditional (car add-effs)) del-remains
         :key #'(lambda (eff-list) (effect-cond-conditional (car eff-list)))))
      ;; Remove them from the "del-remains" list.
      (if del-effs
        (del del-effs del-remains))
      ;; Add the conditional effect to the graph.
      (n-push-list (add-eff-list-to-rel-graph 
                     add-effs del-effs decs graph use-primary)
                   prim-eff-nodes))
    ;; Add nodes for the del-list to the graph. Note that we here only add the
    ;; del-actions of conditional effects that do not have any add-actions.
    (dolist (del-effs del-remains)
      (n-push-list (add-eff-list-to-rel-graph 
                     nil del-effs decs graph use-primary)
                   prim-eff-nodes))
    prim-eff-nodes))


;; Convert all unconditional primary effects or all primary actions of
;; a conditional effect into abstraction nodes and add these nodes to
;; the relevance graph (unless they are already in the graph). If the
;; effect is conditional and has primary actions, add its conditions
;; to the abstraction graph and then add edges from every action to
;; every condition.
;;
;; Note that we add only primary effects to the graph; we ignore side
;; effects. The function is time-consuming, because it calls the
;; time-consuming function "add-predicate-to-graph".
;;
;; add-effs      list of unconditional add-effects or contional effect's 
;;   add-actions (from add-list of a rule); may be empty
;; del-effs      list of unconditional add-effects or contional effect's 
;;   add-actions (from add-list of a rule); may be empty, but then 
;;   add-effs must be nonempty; the condition is the same as in add-effs
;;   (or, alternatively, both add-effs and del-effs are unconditional)
;; decs          declarations of the variables in the effect
;; graph         the abstraction graph
;; use-primary   way of using primary effects (nil, :side, or :prim)
;;
;; returned value:
;; the list of abs-nodes corresponding to the primary actions;
;; if several actions correspond to the same node, then this node
;; occurs several times in the list

(defun add-eff-list-to-rel-graph (add-effs del-effs decs graph use-primary)
  (declare (type abs-graph graph) (ignore use-primary))
  (let ((prim-eff-nodes nil) condition-nodes conditional)
    ;; Add the primary add-action nodes to the graph.
    (dolist (effect add-effs)
      (push (add-predicate-to-graph 
              (effect-cond-effect effect) decs graph :negated-p nil)
            prim-eff-nodes))
    ;; Add the primary del-action nodes to the graph.
    (dolist (effect del-effs)
      (push (add-predicate-to-graph 
              (effect-cond-effect effect) decs graph :negated-p t)
            prim-eff-nodes))
    ;; If the effect is conditional and some actions are primary, add 
    ;; condition nodes and edges to them from all primary actions. 
    (setf conditional 
      (effect-cond-conditional (or (car add-effs) (car del-effs))))
      ;; We use "(or ..)" because either add-effs or del-effs may be empty.
    (when (and conditional prim-eff-nodes)
      (setf condition-nodes
        (delete-duplicates 
          (add-precond-exp-to-rel-graph (second conditional) decs graph)))
      (dolist (eff-node prim-eff-nodes)
        (dolist (condition-node condition-nodes)
          (add-directed-edge eff-node condition-node))))
    prim-eff-nodes))


;; Convert all predicates in a precondition expression into nodes of a
;; relevance graph and add these nodes to the graph (unless they are
;; already in the graph).
;; 
;; This function is time-consuming, because it calls the time-consuming
;; function "add-predicate-to-graph" for every literal in the precondition
;; expression.
;;
;; precond-exp   precondition expression (to be converted into abs-nodes)
;; decs          declarations of the variables in precond-exp
;; graph         the relevance graph
;; negated-p     T if the predicate is negated; nil otherwise
;;
;; returned value:
;; the list of abs-nodes corresponding to the predicates of precond-exp;
;; if several predicates correspond to the same node, then this node
;; occurs several times in the list

(defun add-precond-exp-to-rel-graph 
    (precond-exp decs graph &key (negated-p nil))
  (declare (type abs-graph graph))
  (case (car precond-exp)
    (user::~
      (add-precond-exp-to-rel-graph (second precond-exp) decs graph 
                                    :negated-p (not negated-p)))
    ((user::and user::or)
      (let ((nodes nil))
        (dolist (sub-exp (cdr precond-exp))
          (n-push-list 
            (add-precond-exp-to-rel-graph sub-exp decs graph 
                                          :negated-p negated-p) 
            nodes))
        nodes))
    ((user::exists user::forall)
      (add-precond-exp-to-rel-graph (third precond-exp) 
        (append (second precond-exp) decs) graph :negated-p negated-p))
    (otherwise  ;; it is a predicate
      (list (add-predicate-to-graph precond-exp decs graph
                                    :negated-p negated-p)))))


#|
========================================
Converting the relevance graph into PDL.
========================================

These functions convert the relevance graph, stored in the memory as
an abs-graph, into its in the PDL language.

The relevance graph is represented in PDL in the form shown below. The
upper-case words are the fixed words of PDL and the lower-case words
should be replaced with specific instances. Square brackets denote the
optional parts of the description (they are not a part of the syntax).

(RELEVANCE [rel-graph-name]
  ([component-name =] signed-typed-pred signed-typed-pred ... 
    [- NO-LOOP | - :NO-LOOP])
  ([component-name =] sighed-typed-pred signed-typed-pred ... 
    [- NO-LOOP | - :NO-LOOP])
  ...
[{ORDER | :ORDER}
  (component-name > component-name component-name ...)
  (component-name > component-name component-name ...)
  ...)]

Every "signed-typed-pred" is a possibly negated predicate name,
predicate name with the number of arguments, or predicate with
argument type; each argument type is a prodigy-object name, preceded
by "'", a type name, or a disjunction of type-names:

signed-typed-pred ::= typed-pred | (~ typed-pred)
typed-pred ::= pred-name | (pred-name . arg-number) 
               | (pred-name arg-type arg-type ...)
arg-type ::= 'object-name | type-name | (or type-name type-name ...)

"- :no-loop" means that there is no edge from a graph component to
itself, which may happen only if a component contains one element. 
"- :no-loop" in a component with two or more typed predicates is an error.

If the abstraction graph contains only one component, we encode it as
(RELEVANCE [rel-graph-name] {COLLAPSE | :COLLAPSE}).

When converting the internal representation of the graph into PDL, we
write the corresponding PDL statement into the dynamically-scoped
global variable *encoding*. Only the functions called by the
"alpine-encode-graph" function have the access to this variable.
|#


;; Encode the relevance graph. If the components are totally ordered,
;; do not encode the order of the components; that is, do not encode the 
;; part beginning from the word :order (see the detailed comment above.
;;
;; The graph of components must be topologically sorted. If the
;; components are not totally ordered by the constraints, then every component
;; must have a name (otherwise, we would not be able to encode the order).
;;
;; Before encoding the relevance graphe, go to a new line. The word "Relevance"
;; (in the beginning of the encoding), the graph name, the component names,
;; and the type names are capitalized. The predicate names and the object
;; names are not capitalized.
;;
;; Returned value: PDL encoding of the graph.

(defun relator-encode-graph (graph)
  (declare (type abs-graph graph)
           (special *pspace*) (type problem-space *pspace*))
  (if (singleton-p (abs-graph-components graph))
    (relator-encode-collapse graph)
    (let ((*encoding* (make-array 0 :fill-pointer t :adjustable t 
                                    :element-type 'excl::string-char)))
      (declare (type string *encoding*) (special *encoding*))
      (format *encoding* "~%(Relevance")
      (if (abs-graph-name graph)
        (format *encoding* " ~A" (string-cap (abs-graph-name graph))))
      (name-rel-components graph)
      (dolist (component (abs-graph-components graph))
        (relator-encode-component component))
      (unless (total-order-p graph)
        (unless (every #'abs-component-name (abs-graph-components graph))
          (error "relator-encode-graph: Graph is not total order and some ~
                 components have no names."))
        (format *encoding* "~% :order")
        (dolist (component (abs-graph-components graph))
          (if (abs-component-out component)
            (alpine-encode-order component))))
      (format *encoding* ")")
      *encoding*)))


;; Encode a one-component graph, in the form 
;; (Relevance [rel-graph-name] :collapse). 
;;
;; Before encoding the relevance graph, go to a new line. 
;; The word "Abstraction" and the graph name are capitalized.
;;
;; Returned value: PDL encoding of the graph.

(defun relator-encode-collapse (graph)
  (declare (type abs-graph graph))
  (if (abs-graph-name graph)
    (concatenate 'string "(Relevance " 
      (string-cap (abs-graph-name graph)) " :collapse)")
    "(Relevance :collapse)"))


;; Name all components in the relevance graph (unless they are already named).
;;
;; The name of a component is Component-N, where N is the component's 
;; number from the end of the component list, beginning from 0.
;; For example, if there are three components, they will be
;; named Component-2, Component-1, and Component-0.
;;
;; graph   the relevance graph

(defun name-rel-components (graph)
  (let ((num (length (abs-graph-components graph))))
    (dolist (component (abs-graph-components graph))
      (decf num)
      (unless (abs-component-name component)
        (setf (abs-component-name component) 
              (format nil "Component-~D" num))))))


;; Encode a component of a relevance graph; that is, encode the name
;; of the component and the nodes in the component.  Each node
;; (predicate) begins from a new line; for example,
;;   (Component-1 = 
;;     (~ (on 'small Peg))
;;     (on 'medium Peg))
;;
;; Before encoding the component, go to a new line and leave two
;; spaces from the beginning. The encoding of every node begins from a
;; new line. The component name is capitalized.

(defun relator-encode-component (component)
  (declare (type abs-component component) 
           (special *encoding*) (type string *encoding*))
  (format *encoding* "~%  (")
  (if (abs-component-name component)
    (format *encoding* "~A =" 
            (string-cap (abs-component-name component))))
  (dolist (node (abs-component-nodes component))
    (format *encoding* "~%    ")
    (relator-encode-node node))
  (if (abs-component-loop-p component)
    (format *encoding* ")")
    (format *encoding* " - :no-loop)")))


;; Encode a node of the relevance graph; that is, encode negation (if
;; necessary), the name of the predicate in the node and the types of
;; its arguments; for example, "(~ (on 'small Peg))".
;;
;; If the "args" field of the node is set to :any, then the node represents
;; the predicate with the specified name and any arguments. In this case,
;; encode only the predicate name (without brackets); for example, "(~ on)".
;;
;; If "args" is an integer, then the node represents the predicate
;; with the specified name and this number arguments of any type. In
;; this case, encode the predicate name and the number of arguments;
;; for example, "(on . 2)".

(defun relator-encode-node (node)
  (declare (type abs-node node) 
           (special *encoding*) (type string *encoding*))
  (let* ((negated-p (consp (abs-node-name node)))
           ;; T if the predicate is negated; nil otherwise.
         (predicate-name (if negated-p 
                           (cdr (abs-node-name node))
                           (abs-node-name node))))
           ;; Predicate name.
    (if negated-p (format *encoding* "(~~ "))
    (typecase (abs-node-args node)
      (symbol  ;; ":any"
         (format *encoding* "~A" (string-downcase predicate-name)))
      (integer  ;; argument number
        (format *encoding* "(~A . ~D)" 
          (string-downcase (abs-node-name node)) (abs-node-args node)))
      (list  ;; typed argument list
        (format *encoding* "(~A" (string-downcase predicate-name))
        (dolist (arg-type (abs-node-args node))
          (format *encoding* " ")
          (alpine-encode-arg arg-type))
        (format *encoding* ")")))
    (if negated-p (format *encoding* ")"))))


#|
==========================
Creating relevance tables.
==========================

Given a goal expression and a relevance greaph with declarations, we
build the positive relevance table and negative relevance table for
this goal.

The relevance table (rel-table) for positive predicates is completely
separate from that for negative predicates. If all positive (negative)
predicates are relevant, we store ":all" instead of a rel-table, if no
predicates are relevant (which often happens for negative predicates),
we store ":none".

A relevance table is a hash-table, hashed on predicate names. An entry
of the table may be in one of the followint four forms:
    (1) ":any", which means that all predicates with this name are relevant
(this often happend for positive predicates).
    (2) nil (that is, the predicate name is not hashed), which means
that no predicate with this name is relevant (this happens especially
often for negative predicates).
    (3) A sorted list of natural numbers, e.g. (0 2 3), which means that 
if the number of arguments of a predicate is one of the numbers on the list, 
it is relevant, and if the number of arguments is not on the list, then it 
is irrelevant; argument types do not matter. For the list in our example, 
(on-table) would be relevant and (on-table block-1) irrelevant.
    (4) A Rel-Entry structure, which has three slots, "num", "arglists",
and "tested". The "num" slot is an oredred list of natural numbers, with
the same meaining as in Case 3. The "arglists" slot is a list of typed
argument lists, e.g. ((Small Peg) (Medium Peg)). If the number of arguments
of a predicate is one of the numbers of the "num" list, then it is relevant.
    If not, we match its arguments againt argument lists in "arglists". If 
the intersection of the predicate's typed arguments with one of the argument
lists is not empty, then this predicate is aslo relevant. This matching is
expensive, and we want to avoid it. 
    For this reason, after matching, we store a predicate in the hash table 
of already tested predicates, in the "tested" slot. This slot is a hash
table, hashed on the typed argument lists. The entry for each list is
either 'relevant or 'irrelevant. When we need to match a predicate against
"arglists", we first check whether it is in hash table. If it is, we know
whether it is relevant or not and, thus, we do not need to perform the
expensive matching with "arglists".

We use the relevance graph to find relevant predicates for a give problem. 
We color the graph with black and white, where black color is for relevant 
predicates and white is for irrelevant. Initially, all nodes are white. We 
then search for relevant nodes and paint them black.
|#

;;-------------------------------------------------------------------------
;; Finding relevant predicates.

;; Given a goal (PDL expression with declarations), paint all relevant
;; nodes of a relevance graph black and irrelevant nodes white. A node
;; is relevant in one of the following cases:
;; - it is a subset of some typed goal predicate
;; - it is looped and it interesects some goal predicate
;; - it is a child of a node that is not looped and intersects goal predicate
;; - it is a successor of a relevant node
;;
;; Note that if a node intersects a goal predicate and is not looped,
;; then it may not be relevant. In this case, we have to add the goal
;; predicate itself, which is not a node of the relevance graph, to
;; the list of relevant nodes. The function returns the list of all
;; such goal predicates.
;;
;; goal-exp   goal expression 
;; decs       declarations of the goal expression
;; graph      relevance graph
;; fast-p     takes on of the following two values:
;;              T     use the "fast-add-goal-to-rels" function for finding
;;                      nodes of rel-graph that intesect with goal predicates
;;              nil   use the the "slow-add-goal-to-rels" function instead
;;
;; returned value: 
;; list of goal predicates, represented as nodes, that are not a
;; subset of relevant nodes in the relevance graph; this contains
;; does not have duplicates

(defun find-rels (goal-exp decs graph &key (fast-p t))
  (declare (type abs-graph graph))
  (paint-white graph)
  (delete-duplicates (add-goals-to-rels goal-exp decs graph :fast-p fast-p)
    :test #'same-node-p))


;; Paint all components and all nodes of a relevance gaph white.
;;
;; Other functions will paint some nodes as relevant by painting them
;; black; we use only black and white color.
;;
;; graph   relevance graph

(defun paint-white (graph)
  (declare (type abs-graph graph))
  (dolist (node (abs-graph-nodes graph))
    (setf (abs-node-color node) 'white))
  (dolist (component (abs-graph-components graph))
    (setf (abs-component-color component) 'white)))


;; Find relevant components in the relevance graph for a goal (PDL
;; expression with declarations) and paint relevant components and
;; their nodes black.
;;
;; The function recursively descends to the predicates and negated
;; predicates of the goal expression and, for each (negated) predicate
;; finds relevant nodes in the relevance graph.
;;
;; goal-exp   goal expression 
;; decs       declarations of the goal expression
;; graph      relevance graph
;; negated-p  T if the goal is negated (i.e. inside a negation); nil otherwise
;; fast-p     takes on of the following two values:
;;              T     use the "fast-add-goal-to-rels" function for finding
;;                      nodes of rel-graph that intesect with goal predicates
;;              nil   use the the "slow-add-goal-to-rels" function instead
;;
;; returned value:
;; list of goal predicates, represented as nodes, that are not a subset
;; of relevant nodes in the relevance graph

(defun add-goals-to-rels (goal-exp decs graph &key (negated-p nil) (fast-p t))
  (declare (type abs-graph graph))
  (case (car goal-exp)
    (user::~
      (add-goals-to-rels (second goal-exp) decs graph 
        :negated-p (not negated-p) :fast-p fast-p))
    ((user::and user::or)
      (mapcan #'(lambda (sub-exp)
                  (add-goals-to-rels sub-exp decs graph 
                    :negated-p negated-p :fast-p fast-p))
              (cdr goal-exp)))
    ((user::exists user::forall)
      (add-goals-to-rels (third goal-exp) 
        (append (second goal-exp) decs) graph
          :negated-p negated-p :fast-p fast-p))
    (otherwise  ;; it is a predicate
      (if fast-p
        (fast-add-goal-to-rels 
          (predicate-to-node goal-exp decs :negated-p negated-p) graph)
        (slow-add-goal-to-rels 
          (predicate-to-node goal-exp decs :negated-p negated-p) graph)))))


;; Find relevant components in the relevance graph for a specific goal
;; predicate and paint relevant components and their nodes black.
;;
;; The function first tries to find a node in the graph that matches
;; the goal predicate exactly (that is, rerpesents the same
;; predicate).  If such node is not found, it calls the function
;; "slow-add-goal-to-rels" to find all nodes that intersect with the
;; goal predicate; in this case, the function is time-consuming.
;;
;; node   typed goal predicate, encoded as an abs-node
;; graph  relevance graph
;;
;; returned values:
;; a one-element list with the node that encodes the goal, if this
;; node is not a subset of any node in rel-graph; nil otherwise

(defun fast-add-goal-to-rels (node graph)
  (declare (type abs-node node) (type abs-graph graph))
  (let ((node-in-graph
          (find node (abs-graph-nodes graph) :test #'same-node-p)))
    (if node-in-graph
      (progn (add-component-to-rels (abs-node-component node-in-graph)) nil)
      (slow-add-goal-to-rels node graph))))


;; Find relevant components in the relevance graph for a specific goal
;; predicate and paint relevant components and their nodes black.
;;
;; This function is type consuming, because it call the time-consuming
;; function "node-intersection-p" (which is in the "alpine/build" file)
;; for each node of the relevance graph.
;;
;; node   typed goal predicate, encoded as an abs-node
;; graph  relevance graph
;;
;; returned values:
;; a one-element list with the node that encodes the goal, if this
;; node is not a subset of any node in rel-graph; nil otherwise
  
(defun slow-add-goal-to-rels (node graph)
  (declare (type abs-node node) (type abs-graph graph))
  (let ((add-goal-pred-p nil) intersection-p)
    (dolist (component (abs-graph-components graph))
      (setf intersection-p 
        (member node (abs-component-nodes component) 
          :test #'node-intersection-p))
      (cond
        ((and intersection-p (abs-component-loop-p component))
           (add-component-to-rels component))
        (intersection-p
           (add-component-to-rels component :only-children t)
           (setf add-goal-pred-p t))))
    (when add-goal-pred-p 
      (setf (abs-node-color node) 'black)
      (list node))))


;; Given a component of a relevance graph, paint all its successors
;; black (that is, mark as relevant), using DFS.
;;
;; The function uses the fact that, if some component is already
;; black, then all its sucessors are are also black. When painting
;; a component black, the function also paints all its nodes.
;;
;; Note that every component and every node in the graph is either
;; white or black (we do not use gray).
;;
;; component       component of a relevance graph
;; only-children   takes one of the following two values:
;;                   T  do not paint the component itself
;;                   nil  paint the component as well as its successors

(defun add-component-to-rels (component &key (only-children nil))
  (declare (type abs-component component))
  (when (eq (abs-component-color component) 'white)
    ;; Recursively paint the children and their succerssors.
    (dolist (child (abs-component-out component))
      (add-component-to-rels child))
    ;; Paint the component itself, if necessary.
    (unless only-children
      (dolist (node (abs-component-nodes component))
        (setf (abs-node-color node) 'black))
      (setf (abs-component-color component) 'black))))
          
;;-------------------------------------------------------------------------
;; Simplifying relevant predicates and building relevance tables.

;; Construct two hash tables of relevant predicates, one for positive
;; predicates and one for negative predicates.
;;
;; The relevance is determined by the color of nodes in the abs-graph: 
;; blacks are relevant and whites are not relevant.
;;
;; graph            relevance graph, where the relevant nodes are black
;; rel-goal-preds   goal predicates that are not subsets of relevant nodes
;;                    in the relevance graph (they all are relevant);
;;                    this list must not have duplicates
;;
;; returned value:
;; two-element list, where the first element is a table of relevant positive
;; predicates and a sencod element is a table of relevant negative preidcates

(defun hash-rels (graph rel-goal-preds)
  (declare (type abs-graph graph))
  (let (;; Encode all predicates as a list of Pred structures.
        (preds (build-preds graph rel-goal-preds)))
    ;; For each predicate name, determine whether all nodes with this
    ;; name are of the same color.
    (dolist (pred preds)
      (color-for-pred pred))
    ;; Construct relevance tables of positive and negative predicates.
    (list (preds-to-rel-table preds :negated-p nil)
          (preds-to-rel-table preds :negated-p t))))


;; Encode all predicates that need to be stored in relevance tables
;; (rel-tables) as a list of Pred structures.
;;
;; The predicates come from two sources: the nodes in the relevance
;; graph (graph) and the goal predicates that are not subsets of
;; relevant nodes in the graph (rel-goal-preds).
;; 
;; The predicates in the list of pred structure are arranged by their
;; names and the number of arguments. For the details on representing
;; predicates by Pred structures, see the "alpine/struct" file.
;;
;; A relevant predicate must not be a subset of another relevant
;; predicate (since storing a superset is sufficient), so we remove
;; all subsets from the list of relevant predicate before converting
;; this list into rel-tables.
;;
;; This function is time consuming, because it calls the time-consuming
;; function "node-subset-p" (which is in the "alpine/build.lisp" file)
;; and the number of calls is quadratic on the number of predicates.
;;
;; graph            relevance graph, where the relevant nodes are black
;; rel-goal-preds   goal predicates that are not subsets of relevant nodes
;;                    in the relevance graph (they all are relevant);
;;                    this list must not have duplicates.
;;
;; local variable:
;; *preds*   list of Pred structures, where each structure is for some node; 
;;             we use this list to arrange the nodes by their names and the
;;             number of arguments; this variable is dynamically scoped and
;;             used in the "get-pred" function (which is in "alpine/build")
;;
;; returned value: the resulting list of Pred structures

(defun build-preds (graph rel-goal-preds)
  (let ((black-nodes nil)
        (white-nodes nil)
        (*preds* nil))
    ;; Build the list of all black and all white nodes in the graph.
    ;; If a black node has a black superset node, do not add it to the
    ;; list, since we do not store subsets of relevant predicates.
    ;; Note that a white node cannot be a subset of a black node,
    ;; because subsets of a relevant node are relevant.
    (declare (special *preds*))
    (dolist (node (abs-graph-nodes graph))
      (cond
        ((abs-node-white-p node) 
           (push node white-nodes))
        ((every #'abs-node-white-p (abs-node-super node))
           (push node black-nodes))))
    ;; Add all relevant (black) nodes, except subsets of goal predicates,
    ;; to the list of Pred structures.
    (dolist (black-node black-nodes)
      (unless (member black-node rel-goal-preds :test #'node-subset-p)
        (add-node-to-preds black-node)))
    ;; Add goal predicates, except subsets of other goal predicates
    ;; or relevant graph nodes, to the list of Pred structures.
    ;; Note that if "rel-goal-preds" contained some predicate twice,
    ;; it would not be added, because it a subset of itself.
    (dolist (rel-goal-pred rel-goal-preds)
      (unless (or (member rel-goal-pred 
                          (remove rel-goal-pred rel-goal-preds) 
                          :test #'node-subset-p)
                  (member rel-goal-pred black-nodes :test #'node-subset-p))
        (add-node-to-preds rel-goal-pred)))
    ;; Add all irrelevant (white) nodes to the list of Pred structures.
    (dolist (white-node white-nodes)
      (add-node-to-preds white-node))
    *preds*))
    

;; Determine if all nodes with some name, which are stored in a Pred
;; structure, are have the same color (where color determines
;; relevance). If they do, store their color in the "color" field of
;; Pred; if not, store nil there.
;;
;; The function also stores the color for each Argnum in Pred, by
;; calling "color-for-argnum". The function assumes that the list
;; of Argnums in Pred is not empty.
;;
;; pred   Pred structure, containing all nodes with some name
;; 
;; returned value: 
;; the color, if all nodes are of the same color; nil otherwise

(defun color-for-pred (pred)
  (declare (type pred pred))
  (let* ((argnums (pred-argnums pred))
         (color (color-for-argnum (car argnums))))
    (dolist (argnum (cdr argnums))
      (unless (eq color (color-for-argnum argnum))
        (setf color nil)))
    (setf (pred-color pred) color)))


;; Determine if all nodes with some name and argument number, which
;; are stored in an Argnum structure, have the same color (where color
;; determines relevance). If they do, store their color in the
;; "color" field of Argnum; if not, store nil there.
;;
;; The function assumes that the list of nodes in Argnum is not empty.
;;
;; argnum   Argnum structure, containing all nodes with some name
;;            and argument number
;; 
;; returned value: 
;; the color, if all nodes are of the same color; nil otherwise

(defun color-for-argnum (argnum)
  (declare (type argnum argnum))
;;D  (format t "~%~%----------") ;;D
  (if (same-p (argnum-nodes argnum) :test #'eq :key #'abs-node-color)
    (setf (argnum-color argnum)
          (abs-node-color (car (argnum-nodes argnum))))
    (setf (argnum-color argnum) nil)))
;;D  (format t "~%Argnum-Color: ~S" (argnum-color argnum)) ;;D
;;D  (format t "~%Node-Color: ~S~%~%" 
;;D    (mapcar #'abs-node-color (argnum-nodes argnum)))) ;;D


;; Given a list of all predicates, encoded by Pred structures (a Pred
;; structure for each predicate name), construct a hash table of
;; positive or negative relevant predicates (rel-table).
;;
;; The relevance of a predicate is determined by color, stored in the
;; Pred structure: blacks are relevant; whites are not. The predicates
;; in the "preds" list are arranged by their names and the number of
;; arguments. Each element of the list, a Pred structure, encodes all
;; predicates with some name. For the details on representing
;; predicates by Pred structures, see the "alpine/struct" file.
;;
;; preds       all predicates, encoded by Pred structures
;; negated-p   T for constructing a table of relevant negative predicates;
;;             nil for constraction a table of relevant positive predicates
;;
;; returned value:
;; :all, if all predicates are relevant; :none if no predicates are relevant;
;; a hast table of all relevant predicates among "pred" otherwise

(defun preds-to-rel-table (preds &key (negated-p nil))
  (let ((pos-or-neg-preds
          (if negated-p
            (remove-if #'atom preds :key #'pred-name) ;; negative preds
            (remove-if-not #'atom preds :key #'pred-name)))) ;; positive preds
;;D    (format t "~%~S" (mapcar #'pred-color pos-or-neg-preds)) ;;D
    (cond 
      ;; All predicates are relevant.
      ((same-as 'black pos-or-neg-preds :key #'pred-color) :all)
      ;; All predicates are not relevant.
      ((same-as 'white pos-or-neg-preds :key #'pred-color) :none)  
      ;; Some predicates are relevant and some are not relevant.
      (t (let ((rel-table (make-hash-table :test #'equal)))
           (dolist (pred pos-or-neg-preds)
             (add-pred-to-rel-table 
               (if negated-p (cdr (pred-name pred)) (pred-name pred))
               (rels-for-pred pred) rel-table))
           (if (zerop (hash-table-count rel-table))
             (error "preds-to-rel-table: The generated relevance table has ~
               no entries, which must not happen; investigate it."))
           rel-table)))))


;; Generate the relevant typed-argument lists for some predicate name and
;; argument number. 
;;
;; The resulting list of argument lists is as follows:
;; - If all predicates with this name and argument number are relevant,
;;   then return the list containing the number of arguments; e.g. (3).
;; - If such predicates are never relevant, return nil.
;; - If such predicates are sometimes relevant and sometimes not, return the
;;   lists of typed arguments for which the nodes are relevant; for example,
;;   ((Small Peg) (Medium Peg)).
;;
;; The relevance is determined by color: blacks are relevant; whites are not.
;;
;; argnum   Argnum structure, containing all nodes with some name
;;            and argument number
;;
;; returned value: 
;; list of the type-argument lists that make the predicate relevant

(defun rels-for-pred (pred)
  (declare (type pred pred))
;;D  (format t "~%~S" (pred-color pred)) ;;D
  (case (pred-color pred)
    (black :any)
    (white nil)
    (otherwise (mapcan #'rels-for-argnum (pred-argnums pred)))))


;; Generate the relevant typed-argument lists for some predicate name and
;; argument number. 
;;
;; The resulting list of argument lists is as follows:
;; - If all predicates with this name and argument number are relevant,
;;   then return the list containing the number of arguments; e.g. (3).
;; - If such predicates are never relevant, return nil.
;; - If such predicates are sometimes relevant and sometimes not, return the
;;   lists of typed arguments for which the nodes are relevant; for example,
;;   ((Small Peg) (Medium Peg)).
;;
;; The relevance is determined by color: blacks are relevant; whites are not.
;; The function uses the fact that the "num" field of argnum cannot be "-1"
;; (meaning any number of arguments), because in this case the corresponding
;; "Pred" structure would have the black color and the "rels-for-pred"
;; function would not call this function.
;;
;; argnum   Argnum structure, containing all nodes with some name
;;            and argument number
;;
;; returned value: 
;; list of the type-argument lists that make the predicate relevant

(defun rels-for-argnum (argnum)
  (declare (type argnum argnum))
;;D  (format t "~%~S  ~S" (argnum-color argnum) (argnum-nodes argnum)) ;;D
;;D  (format t "~%~S" (mapcar #'abs-node-color (argnum-nodes argnum))) ;;D
  (case (argnum-color argnum)
    (black (list (argnum-num argnum)))  ;; All such predicates are relevant.
    (white nil)  ;; Such predicates are never relevant.
    (otherwise  ;; Such predicates are sometime relevant and sometimes not.
      (let ((rels nil))
        (dolist (node (argnum-nodes argnum))
          (if (abs-node-black-p node) (push (abs-node-args node) rels)))
        rels))))


;; Store a predicate (name and possible typed arguments) in a rel-table.
;;
;; The table is hashed on a predicate name (without "~") and the entry
;; for a name is determined as follows:
;; - If all predicates with this name are relevant, store ":any".
;; - If predicates with this name are never relevant, the function
;;   is not called and the name is not added to the table.
;; - If the relevance of a predicate depends only on the number of 
;;   arguments (for example, all predicates with this name and two
;;   arguments are relevant), then store all argument numbers that
;;   make predicate relevant, as a list; for example, if predicates
;;   with this name and two or three arguments are relevant, store (2 3).
;; - If the relevance may depend on the number of arguments for some
;;   number and on specific arguments for some numbers for arguments,
;;   store the list of numbers for which it depends only on numbers
;;   and the typed arguments when it depends on types. In this case,
;;   use the "rel-entry" structure to store them.
;;       For example, if all predicates with zero and one arguments are 
;;   relevant, and the predicates with two arguments of types Small and Peg, 
;;   or Medium and Peg are also relevant, then store (0 1) in "rel-entry-nums"
;;   and ((Small Peg) (Medium Peg)) is "rel-entry-arglists)".
;;
;; name        name of a predicate (without "~")
;; rels        list of typed argument lists that make the predicate with
;;               this name relevat; e.g. '(0 1 (Small Peg) (Medium Peg))
;; rel-table   rel-table of positive or negative predicates

(defun add-pred-to-rel-table (name rels rel-table)
  (declare (type symbol name) (type hash-table rel-table))
  (if rels
    (setf (gethash name rel-table)
      (cond
        ;; Predicate is always relevant.
        ((eq rels :any) :any)
        ;; Predicate relevance depends only on the number of arguments.
        ((notany #'consp rels) (sort rels #'<))
        ;; Predicate relevance depends on the types of arguments.
        (t (let ((nums (sort (remove-if #'consp rels) #'<))
                    ;; Relevance that depends only on argument number.
                 (arglists (remove-if-not #'consp rels))
                    ;; Relevance that depends on argument types.
                 (tested (make-hash-table :test #'equal)))
                    ;; Already tested typed arguments for this predicate;
                    ;; (they will accumulate as we test more type predicates).
             (dolist (args arglists)
               (setf (gethash args tested) 'relevant))
             (make-rel-entry :nums nums :arglists arglists 
                             :tested tested)))))))

;;-------------------------------------------------------------------------
;; Miscellaneous operations on abs-components and abs-nodes.

;; Determine if there is an edge from an abs-component into itself.
;;
;; The component is looped if its nodes are looped. A component that
;; has more than one node must always be looped.

(defmacro abs-component-loop-p (component)
  `(or (long-p (abs-component-nodes ,component))
       (abs-node-loop-p (car (abs-component-nodes ,component)))))


;; Determine if the "color" field of an abs-node is 'white.
;;
;; returned value: T if 'white; nil otherwise

(defun abs-node-white-p (node)
  (declare (type abs-node node))
  (eq (abs-node-color node) 'white))


;; Determine if the "color" field of an abs-node is 'black.
;;
;; returned value: T if 'black; nil otherwise

(defun abs-node-black-p (node)
  (declare (type abs-node node))
  (eq (abs-node-color node) 'black))
