#-clisp
(unless (find-package "PRODIGY4")
  (make-package "PRODIGY4" :nicknames '("P4") :use '("LISP")))
(in-package "PRODIGY4")

(export '(aliasing show detail show-hash-table show-list where 
  defaults knobs))

#|
================================
Miscellaneous Prodigy functions.
================================

I write here small functions and macros for general operations on the
Prodigy system. Some of these functions may already be coded by others,
but it is easier to write them than to find. The functions include:

static-predicate-p    check if a predicate is static
inf-type-name-p       determine if the name is of an infinite type
set-pspace-property   set a slot in the plist of *current-problem-space*
consed-to-listed      convert ((<x> . A) (<y> . B) ..) into (A B nil ..)
quote-consed-binding  add quotes in front of the values in a consed binding
consed-named-to-objected   replace object names in a binding with the objects
consed-objected-to-named   replace objects in a bidning with their names
name-to-object        convert the name of a fin or inf object into the object
object-to-name        return the name of a finite or infinite object
problem-space-real-ops          return operators not including *finish*
problem-space-use-abstraction   return :use-abstraction of pspace's plist
problem-space-abs-graph-for  problem and primty for which abs-graph computed
problem-space-abs-graph-name name of the current abstraction graph
problem-space-description-name  name of the current descrition
problem-space-crules-name    name of the current set of control rules
problem-space-prim-sel-name  name of the current selection of primary effects
problem-space-objects-for    problem for which objects were computed
problem-space-problem-for    problem for which state and goals computed
problem-space-relevance-p    return non-nil if relevance table is current
problem-space-use-primary    return :user-primary of pspace's plist
problem-space-rel-graph      return :rel-graph of pspace's plist
problem-space-rel-graph-for  primty for which rel-graph computed
problem-space-rel-tables-for return problem for which rel-tables computed
problem-space-pos-rel-table  return :positive-rel-table of pspace's plist
problem-space-neg-rel-table  return :negative-rel-table of pspace's plist
problem-space-select-apply-op  control rules to select an applicable operator
problem-space-reject-apply-op  control rules to reject an applicable operator
problem-space-running-mode, problem-space-smart-apply-p,
  problem-space-analogical-replay, problem-space-incremental-pending-goals,
  problem-space-apply-order-p, problem-space-use-protection,
  problem-space-no-danglers
    problem-space properties used in place of old global variables
rule-add-and-del-list        return all add and del effects of a rule
operator-add-and-del-list    return all add and del effects of an operator
operator-min-cost            return :min-cost slot from an operator's plist
operator-max-cost            return :max-cost slot from an operator's plist
rule-effects-all, rule-effects-prim, rule-effects-side, rule-effects-rest
                      get all, prim, side, or rest effects of a rule
effects-all, effects-prim, effects-side, effects-rest
                      get all, prim, side, or rest effects from effect list
prim-effect-p         determine whether an effect is primary in current space
effect-cond-node, f-effect-cond-node       :node slot of effect-cond's plist
effect-cond-out, f-effect-cond-out         :out slot if effect-cond's plist
abs-node-inter, f-abs-node-inter           :inter slot from abs-node's plist
abs-node-temp-p, f-abs-node-temp-p         :temp-p slot from abs-node's plist
abs-graph-maybes      return :maybes slot from an abs-graph's plist
abs-graph-pasts       return :pasts slot from an abs-graph's plist
domain-names-assert   output domain names that do not correspond to domains
domain-name-p         verify that there is a domain with a specified name
problem-names-assert  output problem names that do not correspond to problems
problem-name-p        verify that there is a problem with a specified name
prefix-problems       return problem names beginning with given prefix
|#


;; Determine whether a predicate is static.
;; The predicate is represented as a list (e.g. "(on small <peg>)").
;;
;; Returned value: T if the predicate is static; nil otherwise

(defun static-predicate-p 
    (predicate &optional (pspace *current-problem-space*))
  (declare (type problem-space pspace))
  (if (member (car predicate) (problem-space-static-preds pspace)) t))


;; Determine whether the type with a given name is infinite.
;;
;; returned value:
;; T if the type with the given name is infinite; nil otherwise

(defun inf-type-name-p (type-name &optional (pspace *current-problem-space*))
  (declare (type problem-space pspace))
  (functionp (type-name-to-type type-name pspace)))


;; Set a new value of the specified slot in the plist of the current problem
;; space (the slot name usually begins with ":", like ":search-depth").
;;
;; property   slot name
;; value      new value of the slot

(defmacro set-pspace-property (property value)
  `(setf (getf (problem-space-plist *current-problem-space*) ,property)
         ,value))


;; Convert a consed binding for a rule (of the form "((<x> . A) (<y>
;; . B) ..)") into the listed binding (of the form "(A B C nil ...)",
;; where the values are listed in the order of the operator variables
;; and nils are for unbound vars).
;;
;; consed-binding   a consed binding for a rule
;; rule             the rule (operator or inference rule)
;;
;; returned value: the corresponding listed bidnings

(defmacro consed-to-listed (consed-binding rule)
  `(substitute-if nil #'is-variable-p 
    (sublis ,consed-binding (rule-vars ,rule))))

;; Add quotes in front of the values (object names) in a consed binding;
;; for example, ((<x> . A) (<y> . B)) --> ((<x> . 'A) (<y> . 'B)).
;; Note that  ((<x> . 'A) (<y> . 'B)) is identical to 
;; ((<x> quote A) (<y> quote B)).

(defun quote-consed-binding (binding)
  (let ((quoted-binding nil))
    (dolist (var-value-pair binding)
      (push (list (car var-value-pair) 'quote (cdr var-value-pair))
        quoted-binding))
    (reverse quoted-binding)))


;; Convert a consed binding (of the form "((<x> . A) (<y> . B) ..)"),
;; where A, B, .. are object names, into a consed binding where
;; A, B, .. are objects themsleves.

(defun consed-named-to-objected 
    (named-binding &optional (pspace *current-problem-space*))
  (declare (type problem-space pspace))
  (let ((objected-binding (copy-alist named-binding)))
    (dolist (var-value-pair objected-binding)
      (setf (cdr var-value-pair)
        (name-to-object (cdr var-value-pair) pspace)))
    objected-binding))


;; Convert a consed binding (of the form "((<x> . A) (<y> . B) ..)"),
;; where A, B, .. are objects, into a consed binding where
;; A, B, .. are the names of the objects.

(defun consed-objected-to-named (consed-binding)
  (dolist (var-value-pair consed-binding)
    (setf (cdr var-value-pair)
      (object-to-name (cdr var-value-pair)))))


;; Convert an object name into the object. If the name is not a symbol
;; or does not correspond to any object, assume that this is an
;; object of an infinite type and return it unchanged.

(defmacro name-to-object (object-name pspace)
  `(or (and (symbolp ,object-name) 
            (object-name-to-object ,object-name ,pspace))
       ,object-name))


;; Return the name of an object. If the argument is not a Prodigy
;; object, assume that it is an object of an infinite type and return
;; it unchanged.

(defmacro object-to-name (object)
  `(if (prodigy-object-p ,object)
     (object-name ,object)
     ,object))


;; Return the list of operators, in a problem space, that does
;; not include the "*finish*" operator.

(defmacro problem-space-real-ops (pspace)
  `(remove '*finish* (problem-space-operators ,pspace) :key #'operator-name))


;; The following FOURTEEN macros access fields in the problem space's plist.

;; T if we use abstraction; nil otherwise.
(defmacro problem-space-use-abstraction (pspace)
  `(getf (problem-space-plist ,pspace) :use-abstraction))

;; Points to the primty and problem for which abs-graph was computed, in
;; the form (primty . prob); (primty . T) if it is problem-indep;
;; nil if it is out-of-date or not computed.
(defmacro problem-space-abs-graph-for (pspace)
  `(getf (problem-space-plist ,pspace) :abs-graph-for))

;; Name of the current abstraction graph; NIL if no name.
(defmacro problem-space-abs-graph-name (pspace)
  `(abs-graph-name (problem-space-abs-graph ,pspace)))

;; Name of the current description, which includes the current control
;; rules, selection of primary effects, and abstraction.
(defmacro problem-space-description-name (pspace)
  `(getf (problem-space-plist ,pspace) :description-name))
;; Name of the current set of control rules; NIl if no name.
(defmacro problem-space-crules-name (pspace)
  `(getf (problem-space-plist ,pspace) :crules-name))

;; Name of the current selection of primary effects; NIL if no name.
(defmacro problem-space-prim-sel-name (pspace)
  `(getf (problem-space-plist ,pspace) :prim-sel-name))

;; Points to the problem for which objects, initial state, and goals
;; were processed; nil if they were not processed for any problem.
(defmacro problem-space-objects-for (pspace)
  `(getf (problem-space-plist ,pspace) :objects-for))

;; Points to the problem for which initial state and goals were
;; processed; nil if they were not processed for any problem.
(defmacro problem-space-problem-for (pspace)
  `(getf (problem-space-plist ,pspace) :problem-for))

;; Returns non-nil if the relevance table correponds to the current
;; selection of primary effects, and nil otherwise. If it is nil, then
;; the relevance table must be updated before planning.
(defmacro problem-space-relevance-p (pspace)
  `(getf (problem-space-plist ,pspace) :relevance-new-p))

;; T if we use primary effects; nil otherwise.
(defmacro problem-space-use-primary (pspace)
  `(getf (problem-space-plist ,pspace) :use-primary))

;; Relevance graph.
(defmacro problem-space-rel-graph (pspace)
  `(getf (problem-space-plist ,pspace) :rel-graph))

;; Pirmty (nil, :prim, or :side) for which rel-graph is computed.
(defmacro problem-space-rel-graph-for (pspace)
  `(getf (problem-space-plist ,pspace) :rel-graph-for))

;; Points to the problem for which rel-tables were computed;
;; nil if rel-tables are out-of-date or not computed.
(defmacro problem-space-rel-tables-for (pspace)
  `(getf (problem-space-plist ,pspace) :rel-tables-for))

;; Positive relevance table for a problem.
(defmacro problem-space-pos-rel-table (pspace)
  `(getf (problem-space-plist ,pspace) :pos-rel-table))

;; Negative relevance table for a problem.
(defmacro problem-space-neg-rel-table (pspace)
  `(getf (problem-space-plist ,pspace) :neg-rel-table))

;; Control rules to select an applicable operator.
(defmacro problem-space-select-apply-op (pspace)
  `(getf (problem-space-plist ,pspace) :select-apply-op))

;; Control rules to reject an applicable operator.
(defmacro problem-space-reject-apply-op (pspace)
  `(getf (problem-space-plist ,pspace) :reject-apply-op))


;; The following SIX macros access the fields of a problem space that I
;; am using in place of old global veriables with the same names.

;; Running mode, which is Savta, Saba, or Eager.
;; The initial default value is 'Savta.
(defmacro problem-space-running-mode (pspace)
  `(getf (problem-space-plist ,pspace) :problem-space-running-mode))

;; Determines whether we use the smart-apply option;
;; it is used with Saba, but not with Savta.
;; The initial default value is nil.
(defmacro problem-space-smart-apply-p (pspace)
  `(getf (problem-space-plist ,pspace) :problem-space-smart-apply-p))

;; Determines whether the analogical replay is used.
;; The initial default value is nil.
(defmacro problem-space-analogical-replay (pspace)
  `(getf (problem-space-plist ,pspace) :problem-space-analogical-replay))

;; If this value is nonnil, Prodigy calculates pending goals
;; incrementally, which is slightly faster.
;; The initial default value is nil.
(defmacro problem-space-incremental-pending-goals (pspace)
  `(getf (problem-space-plist ,pspace) 
     :incremental-pending-goals))

;; If this value is nonnil, Prodigy always applies operators in the
;; reverse order of selecting them; I think it is no longer used.
;; The initial default value is T.
(defmacro problem-space-apply-order-p (pspace)
  `(getf (problem-space-plist ,pspace) :apply-order-p))

;; If this value is nonnil, Prodigy adds goal-protection branches to the
;; search tree, which is necessary for completeness.
;; The initial default value is nil.
(defmacro problem-space-use-protection (pspace)
  `(getf (problem-space-plist ,pspace) :use-protection))

;; If this value is nonnil, Prodigy removes dangling literals.
;; The initial value is T.
(defmacro problem-space-no-danglers (pspace)
  `(getf (problem-space-plist ,pspace) :no-danglers))


;; The following FOUR macros access fields in a rule's structure;
;; in particular, in its plist.

;; All add and del effects of a rule.
(defmacro rule-add-and-del-list (rule)
  `(append (rule-add-list ,rule) (rule-del-list ,rule)))

;; All add and del effects of an operator.
(defmacro operator-add-and-del-list (operator)
  `(append (operator-add-list ,operator) (operator-del-list ,operator)))

;; Minimal known cost of an instantiation of the operator.
(defmacro operator-min-cost (operator)
  `(getf (operator-plist ,operator) :min-cost))

;; Maximal known cost of an instantiation of the operator.
(defmacro operator-max-cost (operator)
  `(getf (operator-plist ,operator) :max-cost))


;; The following SEVEN macros access primary, side, and rest effects
;; in the rule-effects field of a rule.
;; 
;; Note that this information is a copy of the user's initial
;; specification and it may be different from the primality
;; information in the Add-list and Del-list, which changes during the
;; automatic selection of primary effects.

;; Get all effects from a rule.
(defmacro rule-effects-all (rule)
  `(append (cdr (third (rule-effects ,rule))) 
           (cdr (fourth (rule-effects ,rule))) 
           (cdr (fifth (rule-effects ,rule)))))

;; Get primary effects from a rule.
(defmacro rule-effects-prim (rule)
  `(cdr (third (rule-effects ,rule))))

;; Get side effects from a rule.
(defmacro rule-effects-side (rule)
  `(cdr (fourth (rule-effects ,rule))))

;; Get rest effects from a rule.
(defmacro rule-effects-rest (rule)
  `(cdr (fifth (rule-effects ,rule))))

;; Get all effects from a rule-effect list.
(defmacro effects-all (effects)
  `(append (cdr (third ,effects)) 
           (cdr (fourth ,effects)) 
           (cdr (fifth ,effects))))

;; Get primary effects from a rule-effects list.
(defmacro effects-prim (effects)
  `(cdr (third ,effects)))

;; Get side effects from a rule-effects list.
(defmacro effects-side (effects)
  `(cdr (fourth ,effects)))

;; Get rest effects from a rule-effects list.
(defmacro effects-rest (effects)
  `(cdr (fifth ,effects)))


;; Determine whether the effect is primary in the current problem space.
(defun prim-effect-p (effect &optional (use-primary :default))
  (declare (type effect-cond effect))
  (case (effect-cond-primty effect)
    (:prim t)
    (:side (null 
             (if (eq use-primary :default)
               (problem-space-property :use-primary) 
               use-primary)))
    (:rest (eq :prim
             (if (eq use-primary :default)
               (problem-space-property :use-primary) 
               use-primary)))))


;; The following TWO macros and two functions access fields in an
;; effect-cond's plist.

;; Node in the abstraction graph that corresponds to this effect.
(defmacro effect-cond-node (effect)
  `(getf (effect-cond-plist ,effect) :node))
(defun f-effect-cond-node (effect)
  (declare (type effect-cond effect))
  (getf (effect-cond-plist effect) :node))

;; Nodes that would be connected by edges outcoming from the effect's
;; node if the effect were primary. We use this field to store
;; "potential" edges outcoming for the node of a "rest" effect.
(defmacro effect-cond-out (effect)
  `(getf (effect-cond-plist ,effect) :out))
(defun f-effect-cond-out (effect)
  (declare (type effect-cond effect))
  (getf (effect-cond-plist effect) :out))


;; The following TWO macros access fields in an abs-node's plist;
;; these fields are used by Margie (not by Alpine).

;; List of maybe-nodes that intersect this maybe-node without being
;; its subsets or supersets. This field is set only for maybe-nodes.
(defmacro abs-node-inter (node)
  `(getf (abs-node-plist ,node) :inter))
(defun f-abs-node-inter (node)
  (declare (type abs-node node))
  (getf (abs-node-plist node) :inter))

;; Temporary boolean value used in some algorithms
(defmacro abs-node-temp-p (node)
  `(getf (abs-node-plist ,node) :temp-p))
(defun f-abs-node-temp-p (node)
  (declare (type abs-node node))
  (getf (abs-node-plist node) :temp-p))


;; The following TWO macros access fields in an abs-graph's plist;
;; these fields are used by Margie (not by Alpine). Note that the
;; node lists called "nodes", "maybes", and "pasts" are disjoint.

;; List of nodes that may be inserted into the graph in the future.
(defmacro abs-graph-maybes (graph)
  `(getf (abs-graph-plist ,graph) :maybes))

;; List of nodes removed from the graph.
(defmacro abs-graph-pasts (graph)
  `(getf (abs-graph-plist ,graph) :pasts))


;; If some names do not correspond to domains, terminate with an
;; error and list these names.

(defun domain-names-assert (names)
  (let ((wrong-names (remove-if #'domain-name-p names)))
    (if wrong-names
      (error "The following domains do not exist:~{ ~S~}." 
        wrong-names))))


;; Verify that there is a domain with a specified name.
;; The name may be a symbol or sring (it is downcased).

(defun domain-name-p (name)
  (declare (special user::*world-path* user::*directory-separator*))
  (coerce-bool (probe-file
    (format nil "~A~C~A~Cdomain.lisp" user::*world-path*
      user::*directory-separator* (string-downcase name)
      user::*directory-separator*))))


;; If some names do not correspond to problems, terminate with an 
;; error and list these names.

(defun problem-names-assert (names)
  (let ((wrong-names (remove-if #'problem-name-p names)))
    (if wrong-names
      (error "The following problems do not exist:~{ ~S~}." 
        wrong-names))))


;; Verify that there is a problem with a specified name.
;; The name may be a symbol or sring (it is downcased).

(defun problem-name-p 
    (name &optional (problem-path (problem-space-property :problem-path)))
  (declare (type string problem-path)
           (special user::*directory-separator*))
  (unless *current-problem-space*
    (error "problem-p: Domain is not specified"))
  (coerce-bool (probe-file 
    (format nil "~Aprobs~C~A.lisp" problem-path
      user::*directory-separator* (string-downcase name)))))


;; Return the list of problem names (symbols) that begin with given prefix.
;; The prefix may be a symbol or sring (it is downcased).

(defun prefix-problems (prefix)
  (declare (special *current-problem-space* user::*directory-separator*))
  (let* ((user-package (find-package 'user)) 
         (paths (directory 
                  (format nil "~Aprobs~C~A*.lisp" 
                    (problem-space-property :problem-path)
                    user::*directory-separator* (string-downcase prefix))))
         (problem-names nil)
         file-name)
    (declare (type package user-package))
    (dolist (path paths)
      (setf file-name (file-namestring path))
      (push 
        (intern (string-upcase (subseq file-name 0 (- (length file-name) 5)))
                user-package)
        problem-names))
    (reverse problem-names)))


#|
==================================
Different miscellaneous functions.
==================================

I write here small general-purpose functions and macros that I find
convenient, which are missing in Common Lisp. I have coded the
following functions:

xor              return exclusive or of two boolean values
equiv            test equivalence of boolean values
coerce-bool      convert any value to T/nil
all-eq           test eq of all values
same-p           check whether all elements of a list are identical
same-as          check if all elements of a list are equal to some value
same-subelts-as  check if all subelements of a list are equal to some value
short-p          check that a list is empty or a one-element list
long-p           check that a list has more than one element
singleton-p      check whether a list has exactly one element
quote-value      add a quote in front of a variable's value
setn             silent setf (does not return any value)
set-nil          set a variable to nil
set-bool         set a varaible to t or nil
multi-cons       consing several elements to a list
push-second      like push, but new element is added after the first one
multi-push       pushing several elements onto a list
push-list, n-push-list   like push, but add a list rather than one element
multi-pop        repeat pop several times, and return the last element
pop-list         repeat pop several times, and return list of popped elements
remove-subs      remove every element that is a subset of some other element
del-subs         modify a list by removing subsets of other elements
del-duplicates   modify a list by deleting duplicates
del              modify a list by removing a certain element
set-dif          modify a list by removing elements given in another list
cartesian        find cartesian product of the given sets (lists)
find-all         find all elements equal to the given element
find-duplicate   find an element that occurs two or more times in a list
listify, f-listify, set-listify  if value is atom, convert it to one-elt list
subelts-to-elts  concatenate all sublists of a list
leaves-to-elts   build a list of the leaves of a tree
subset           determine whehter one of two lists is a subset of the other
same-plist-p     determine if to plists are equal
sub-plist-p      determine if one of two plist is a sub-plist of the other
list-random-order     generate a random permuation of the elements of a list
list-random-element   select a random element of a list
weighted-random       select a random element according to specified weights
log-normal            generate a random number, with log-normal distribution
normal                generate a random number, with normal distribution
log-uniform           generate a random number, with log-uniform distribution
uniform               generate a random number, with uniform distribution
random-p              returns t with given probability, and nil otherwise
same-position         find element with the same position as in another list
find-min, find-max    find the minimal and maximal element of a list
leaf             like find, but for a tree
some-leaf        check if some element of a list is a leaf of a tree
dosublists       like dolist, but for all sublists of a list
until            iterate until a condition is met
while            iterate while a condition is satisfied
sqr              square a number
cube             cube a number
diagonal         find the root of the square sum
small-diff-p     check that two values are within a rounding error
sqrt-real        replace negative value with 0 in computing square root
mean-dev, mean   find the mean and standard deviation
regress          perform the regression
hash-member-p    check if a hash-table has an entry with this key
add-hash, push-hash   add a new element to a hash-table
clean-hash       remove all nil entries from a hash-table
hash-to-keys, hash-to-entries   return list of keys and entries in hash-table
dohash, dohash+  like dolist, but for a hash-table
create-dhash-table, getdhash, add-dhash, push-dhash, getsub-dhash,
  remdhash, dodhash, dodhash+  operations on a hash-table of hash-tables
string-up, string-down, string-cap   upcase/downcase/capitalize a keyword
negative-up      return 0 for a negative number, and the number itself ow
min-with-nil     like min, but return nil if no arguments
set-min-with-nil if a variable is smaller than value, set it to the value
min-with-nils, n-min-with-nils  like min, but arguments may be nil
min-sublist      return all minimal elements of a list
set-t            set the variables to the result and time of the execution
function-name    determine the name of a function
timer            return the time taken by executed statements
push-timer, read-timer, pop-timer   use of time stack to get elapsed time
if-not           execute second statement if condition holds, first otherwise
mapcar-macro     analog of mapcar for macros
applies          apply a sequence of functions and/or macros
applies+         apply a sequence of functions until nil is encountered
message          if the first argument is nonnil, print to the standard output
print-nonnil-symbol   print a symbol if it is nonnil
print-names      print a list of names with commas
print-list       print a specified number of elements of a list
print-maybe-list use "print-list" or "princ", depending on the argument
re-package-tree  re-package all leaves in a tree
re-package       return symbols with the same name in another package
*user-package*, *key-package*, *p4-package*   global vars pointing to packages
|#

;; Find "exclusive or" of two boolean values.
;;
;; p1, p2   boolean values
;;
;; returned value: T if exactly one of the arguments in nil; nil otherwise

(defmacro xor (p1 p2)
  `(if ,p1 
       (not ,p2)
       (and ,p2 t)))


;; Determine whether all boolean values are identical.
;;
;; bool    boolean value
;; bools   list of boolean values
;;
;; returned value: 
;; T if all values are nil or all values are not nil; nil otherwise

(defmacro equiv (&optional bool &body bools)
  `(if ,bool
     (and ,@bools t)
     (not (or ,@bools))))


;; If "t-or-nil" is nil, return nil; otherwise, return t.

(defmacro coerce-bool (t-or-nil)
  `(and ,t-or-nil t))


;; Determine whether all values are eq.
;;
;; returned value: T if all values are eq; nil otherwise

(defmacro all-eq (&optional val &body vals)
  `(every #'(lambda (other-val) (eq ,val other-val)) ',vals))


;; Check whether all elements of "lis" are identical.
;;
;; returned value: T if all values are identical; nil otherwise

(defmacro same-p (lis &key (test #'eql) (key #'identity))
  `(or (short-p ,lis)
       (let ((keyed-elt (funcall ,key (car ,lis))))
         (flet ((same-elt (elt) (funcall ,test keyed-elt (funcall ,key elt))))
            (every #'same-elt (cdr ,lis))))))


;; Check whether all elements of "lis" are equal to "elt".
;;
;; returned value: T if all values are equal to "elt"; nil otherwise

(defmacro same-as (elt lis &key (test #'eql) (key #'identity))
  `(or (null ,lis)
       (flet ((same-elt (elt) (funcall ,test ,elt (funcall ,key elt))))
          (every #'same-elt ,lis))))


;; Check whether all subelements (i.e. elements of elements) of lis are
;; equal to "elt".
;;
;; returned value: T if all values are equal to "elt"; nil otherwise

(defmacro same-subelts-as (elt lis &key (test #'eql) (key #'identity))
  `(or (null ,lis)
       (flet ((same-elt (elt) (funcall ,test ,elt (funcall ,key elt)))
              (same-elts (sublis) (every #'same-elt sublis)))
          (every #'same-elts ,lis))))


;; Check that "lis" (a list) is empty or a one-element list.
;;
;; returned value: T if "lis" empty or has one element; nil otherwise

(defun short-p (lis)
  (unless (listp lis)
    (error "singleton-p: `~S' is not a list." lis))
  (null (cdr lis)))


;; Check that "lis" (a list) has more than one element.
;;
;; returned value: T if "lis" has more than one element; nil otherwise

(defun long-p (lis)
  (unless (listp lis)
    (error "singleton-p: `~S' is not a list." lis))
  (coerce-bool (cdr lis)))


;; Check whether "lis" (a list) is a one-element list.
;;
;; returned value: T if "lis" has exactly one element; nil otherwise

(defun singleton-p (lis)
  (unless (listp lis)
    (error "singleton-p: `~S' is not a list." lis))
  (and lis (null (cdr lis))))


;; Check whether the value is a natural number.

(defmacro natural-p (value)
  `(and (integerp ,value) (>= ,value 0)))


;; Return the value of a variable with quote in front of it.  
;; For example, if the variable X evaluates to A, then (quote-value x)
;; evaluates to 'A.

(defmacro quote-value (var)
  `(list 'quote ,var))


;; Silent setf (does not return any value).

(defmacro setn (&body args)
  `(progn (setf ,@args) (values)))


;; Set the variable to nil.

(defmacro set-nil (var)
  `(setf ,var nil))


;; Set the variable to T or nil, depending on the value.

(defmacro set-bool (var value)
  `(setf ,var (and ,value t)))


;; Consing several elements to a list:
;; (multi-cons 'x 'y ... '(z)) --> (cons x (cons y ... '(z))) --> (x y ... z).

(defmacro multi-cons (&body args)
  (let* ((rev-args (reverse args))
         (expression (car rev-args))
         (elts (cdr rev-args)))
    (dolist (elt elts)
      (setf expression (list 'cons elt expression)))
    expression))
          

;; This macro is like "push", but is adds the new element after the
;; first element of the list; for example, "(setf lis '(1 3)) 
;; (push-second 2 '(1 3)" makes lis = (1 2 3). The list must be nonepty.
;;
;; The important propery of this macro is that it allows to modify the
;; values of function arguments. (If we apply "push" to a function
;; argument, the result is lost after the return from the function).
;;
;; returned value: cdr of the new list

(defmacro push-second (elt lis)
  `(push ,elt (cdr ,lis)))


;; Pushing several elements onto a list:
;; (multi-push 'x ... 'y lis) --> (progn (push 'y lis) ... (push 'x lis))
;; results in push modifying lis by adding x ... y to the beginning.

(defmacro multi-push (&body args)
  (let* ((rev-args (reverse args))
         (lis (car rev-args))
         (elts (cdr rev-args))
         (expression '(progn)))
    (dolist (elt elts)
      (push (list 'push elt lis) expression))
    (reverse expression)))


;; The following TWO macros modify the value of "lis2" by adding all 
;; elements of "lis1" in the beginning of "lis2."
;;
;; "push-list" preserves the value of "lis1"
;; "n-push-list" destroys "lis1"
;;
;; lis1, lis2   lists
;;
;; returned value: 
;; new value of "lis", i.e. concatenation of "lis1" and initial "lis2"

(defmacro push-list (lis1 lis2)
  `(progn
     (unless (and (listp ,lis1) (listp ,lis2))
       (error "push-list: `~S' or `~S' is not a list." ,lis1 ,lis2))
     (setf ,lis2 (append ,lis1 ,lis2))))

(defmacro n-push-list (lis1 lis2)
  `(progn
     (unless (and (listp ,lis1) (listp ,lis2))
       (error "n-push-list: `~S' or `~S' is not a list." ,lis1 ,lis2))
     (setf ,lis2 (nconc ,lis1 ,lis2))))


;; Repeat the pop operation seveal times.
;;
;; lis   list, elements of which are popped
;; num   number of pops
;;
;; returned value: the last popped element

(defmacro multi-pop (lis &optional (num 1))
  `(dotimes (i (1- ,num) (pop ,lis)) (pop ,lis)))


;; Repeat the pop operation several times, 
;; and return the list of popped elements.

(defmacro pop-list (lis &optional (num 1))
  `(let ((popped-lis nil))
    (dotimes (i ,num (reverse popped-lis))
      (push (pop ,lis) popped-lis))))


;; First elt is removed.

;; Given a list, remove every element that is a subset of some other
;; element and return the resulting new list.
;;
;; The subset relationship, defined by the ":test" key, should be 
;; transitive (otherwise, the operation is not well-defined). If
;; two elements are "identical" (i.e. each is subset of the other),
;; then the one earlier in the sequence is removed. The remaining
;; elements will appear in their original order.

(defmacro remove-subs (lis &key (test #'subsetp) (key #'identity))
  `(cond 
     ((short-p ,lis) ,lis)
     ((eq ,key #'identity)
        (do* ((new-lis nil)
              (old-lis ,lis (cdr old-lis)))
             ((null old-lis) (reverse new-lis))
          (unless (member (car old-lis) 
                    (append new-lis (cdr old-lis)) :test ,test)
            (push (car old-lis) new-lis))))
     (t (do* ((new-lis nil)
              (old-lis ,lis (cdr old-lis))
              (new-keyed-lis nil)
              (old-keyed-lis (mapcar ,key ,lis) (cdr old-keyed-lis)))
             ((null old-lis) (reverse new-lis))
          (unless (member (car old-keyed-lis) 
                    (append new-keyed-lis (cdr old-keyed-lis)) :test ,test)
            (push (car old-lis) new-lis)
            (push (car old-keyed-lis) new-keyed-lis))))))


;; Modify the value of "lis" by removing subsets of other elements.

(defmacro del-subs (lis &body body)
  `(progn
     (unless (listp ,lis)
       (error "del-duplicates: `~S' is not a list." ,lis))
     (setf ,lis (remove-subs ,lis ,@body))))


;; Modify the value of "lis" by deleting duplicates.
;;
;; lis   list

(defmacro del-duplicates (lis)
  `(progn
     (unless (listp ,lis)
       (error "del-duplicates: `~S' is not a list." ,lis))
     (setf ,lis (delete-duplicates ,lis))))


;; Modify the value of "lis" by deleting the element eql to "elt".
;;
;; elt   (possibly) element of lis
;; lis   list
;; key   any keywords that "delete" takes

(defmacro del (elt lis &body keys)
  `(progn 
     (unless (listp ,lis)
       (error "del: `~S' is not a list." ,lis))
     (setf ,lis (delete ,elt ,lis ,@keys))))


;; Modify the value of "lis1" by deleting all elements eql to
;; any element of "lis2".
;;
;; lis1, lis2    lists

(defmacro set-dif (lis1 lis2)
  `(setf ,lis1 (set-difference ,lis1 ,lis2)))


;; Find the cartesian product of given sets (lists); for example,
;; (cartesian '(1 2) '(a b) '(x y)) --> ((1 a x) (1 a y) (1 b x) (1 b y) ...).
;; If the list of arguments is empty, return (nil).

(defun cartesian (&rest lists)
  (let* ((old-product (list nil))
         new-product)
    (dolist (lis (reverse lists) old-product)
      (setf new-product nil)
      (dolist (lis-elt lis)
        (dolist (old-product-elt old-product)
          (push (cons lis-elt old-product-elt) new-product)))
      (setf old-product (reverse new-product)))))


;; Find all elements equal to the given element (opposite to remove)
;;
;; elt    element
;; lis    list
;; body   the optional arguments identical to that of remove

(defmacro find-all (elt lis &body body)
  `(remove ,elt ,lis :test-not #'eq ,@body))


;; Find an element that occurs two or more times in the list.
;;
;; lis    list
;; test   function used two test the identity of elements
;; key    applied to the elements before testing identity
;;
;; returned value: found duplicate element; nil if no duplicate elements found

(defun find-duplicate (lis &key (test #'eql) (key #'identity))
  (do* ((lis-rest lis (cdr lis))
        (keyed-elt (funcall key (car lis)))
        (identical-elt (find keyed-elt (cdr lis) :test test :key key)))
    ((or identical-elt (null (cddr lis))) identical-elt)))


;; If an element is an atom, convery it into a one-element list.
;; Note that nil is not listified (since it is a list). The second
;; function sets "elt" to the listified value.

(defmacro listify (elt)
  `(if (listp ,elt) ,elt (list ,elt)))
(defmacro f-listify (elt)
  (if (listp elt) elt (list elt)))
(defmacro set-listify (elt)
  `(if (listp ,elt) ,elt (setf ,elt (list ,elt))))


;; Concatenate all sublists of "lis" into a list;
;; for example, (sublets-to-elt '((a b) (c d))) --> (a b c d).

(defmacro subelts-to-elts (lis)
  `(apply #'append ,lis))


;; Build a list of the leaves of a given tree; for example,
;; (leaves-to-elts '(a (((b c) (d) e) (f (g)))) --> (g c b d e f a).
;; I use ugly do-loop instead of a neat recursion, for effeciency;
;; as a result, the order of leaves is not pereserved.

(defun leaves-to-elts (tree)
  (do ((old-tree tree)
       (new-tree nil)
       (elts nil))
      ((null old-tree) elts)
    (dolist (elt old-tree)
      (if (atom elt)
        (push elt elts)
        (push-list elt new-tree)))
    (setf old-tree new-tree)
    (set-nil new-tree)))


;; Determine whether one of the lists is a subset of the other.
;; Return :same if the lists represent the same set, :less if
;; the first list is a subset of the second, :more if the second
;; list is a subset of the first, and :diff otherwise.

(defun subset (lis1 lis2 &key (test #'eql) (key #'identity))
  (let ((subset1-p (subsetp lis1 lis2 :test test :key key))
        (subset2-p (subsetp lis2 lis1 :test test :key key)))
    (cond
      ((and subset1-p subset2-p) :same)
      (subset1-p :less)
      (subset2-p :more)
      (t :diff))))


;; Determine whether to plists are equal; that is, for every slot
;; in one of the plists there is an identically named slot in the
;; other, and the slot's value is the same. The implementation is 
;; not particularly efficient. See comments to the "sub-plist-p"
;; for the explanation of the arguments.

(defmacro same-plist-p (plist1 plist2 &body keys)
  `(and (sub-plist-p ,plist1 ,plist2 ,@keys)
        (sub-plist-p ,plist2 ,plist1 ,@keys)))


;; Determine if one plist is a sub-plist of another; that is, for
;; every slot in the first plist, there is identically named slot in
;; the second plist and the slot's value is the same. For example, 
;; (:a 1 :c 3) is a subplist of (:a 1 :b 2 :c 3).
;;
;; The value of the "none" is argument is considered equivalent to the
;; absence of this slot. For example, if "none" is nil, then (.. :d nil ..)
;; is equivalent to the absence of the :d slot, and (:a 1 :c 3 :d nil)
;; is a sub-plist of (:a 1 :b 2 :c 3).

(defun sub-plist-p (plist1 plist2 &key (test #'eql) (none (gensym)))
  (do* ((key (pop plist1) (pop plist1))
        (entry (pop plist1) (pop plist1))
        (member-p (funcall test entry (getf plist2 key none))))
       ((or (not member-p) (null plist1)) member-p)))
    

;; Generate a random permuation of the elements of a list; destructive.
;; Note that the time is quadratic (may be optimized in the future).
;;
;; lis   list
;;
;; returned value: a list built by a random permuation

(defun list-random-order (lis)
  (do* ((new-lis nil (cons elt new-lis))
        (old-lis lis (delete elt old-lis :count 1))
        (elt (list-random-element old-lis) (list-random-element old-lis)))
    ((null old-lis) new-lis)))


;; Select a random element of a list; all elements have an equal chance
;; of being selected. If the list is empty, return nil.
;;
;; lis   list
;;
;; returned value: a random element if the list is not empty; nil otherwise

(defmacro list-random-element (lis)
  `(if ,lis
    (nth (random (length ,lis)) ,lis)))


;; Select a random element of a list, where each element's probability of
;; being selected is proportional to a specified (nonnegative) weight. The
;; length of the probability list and of list from which we select an
;; element must be the same.
;;
;; probs   probabilities (weights)
;; lis     list from which we select an element
;;
;; returned value:
;; selected element; if the list is not given, then the element's position

(defun weighted-random (probs &optional lis)
  (unless (or (null lis) (eq (length probs) (length lis)))
    (error "weighted-random-element: ~S and ~S are not of the same length"
      probs lis))
  (let ((random-value (random (apply #'+ probs)))
        (num 0))
    (declare (real probs))
    ;; Finding the number of a randomly selected element.
    (block do-block
      (dolist (prob probs)
        (if (< random-value prob)
          (return-from do-block))
        (decf random-value prob)
        (incf num)))
    ;; Avoid the case when num gets to big because of rounding errors.
    (if (eq num (length probs)) (decf num))
    (if lis (nth num lis) num)))


;; Given the mean and standard deviation of a unifrom distribution,
;; select a random number according to the corresponding log-normal
;; distribution (the log of this number is normally distributed).

(defun log-normal (mean deviation)
  (exp (normal mean deviation)))


;; Given the mean and standard deviation of a normal distribution,
;; select a random number according to this distribution.

(defun normal (mean deviation)
  (declare (real mean deviation))
  (let ((distance (* deviation (chance-to-z (random 0.5)))))
    (if (random-p 0.5)
      (- mean distance)
      (+ mean distance))))


;; Give the probability that the variable is between 0 and a
;; z-value, find the corresponding z-value (this operation is
;; converse of the z-test.

(defun chance-to-z (chance)
  (let ((num (z-search chance)))
    (declare (integer num)
             (special *z-flat*) (type (array real (310)) *z-flat*))
    (if (= num 309)
      3.09
      (linear-interpolation chance (aref *z-flat* num) 
        (aref *z-flat* (1+ num)) (* .01 num) (* .01 (1+ num))))))


;; Given two points, (x1,y1) and (x2,y2), find the y-value for the
;; point with the given x-coordinate on the same line.

(defun linear-interpolation (x x1 x2 y1 y2)
  (declare (real y x1 x2 y1 y2))
  (cond
    ((and (= x1 x2) (small-diff-p y1 y2)))
    ((= x1 x2) 
      (error "linear-interpolation: x1 = x2 = ~S, whereas y1 = ~S and y2 = ~S"
        x1 y2 y2 ))
    (t (let ((slope (/ (- y2 y1) (- x2 x1))))
         (declare (real slope))
         (+ y1 (* (- x x1) slope))))))


;; Given a probability value, chance, find the index "from" in the
;; array *z-flat* of the z-test probabilities such that
;; (aref *z-flat* from) <= chance < (aref *z-flat* (1+ from).
;; The array is sorted, and we use a binary search.

(defun z-search (chance)
  (declare (real chance) (integer from to)
           (special *z-flat*) (type (array real (310)) *z-flat*))
  (let ((from 0) (to 310) mid)
    (while (< from to)
      (setf mid (floor (+ from to) 2))
      (if (< chance (aref *z-flat* mid))
        (setf to mid)
        (setf from (1+ mid))))
    (1- from)))

;; Table of the probablities for different values of z,
;; in the form of a one-dimensional array.

(if (boundp '*z-flat*) (makunbound '*z-flat*))
(defconstant *z-flat*
;       .00   .01   .02   .03   .04   .05   .06   .07   .08   .09  ; z
     #(.0000 .0040 .0080 .0120 .0160 .0199 .0239 .0279 .0319 .0359 ; 0.0
       .0398 .0438 .0478 .0517 .0557 .0596 .0636 .0675 .0714 .0753 ; 0.1
       .0793 .0832 .0871 .0910 .0948 .0987 .1026 .1064 .1103 .1141 ; 0.2
       .1179 .1217 .1255 .1293 .1331 .1368 .1406 .1443 .1480 .1517 ; 0.3
       .1554 .1591 .1628 .1664 .1700 .1736 .1772 .1808 .1884 .1879 ; 0.4
       .1915 .1950 .1985 .2019 .2054 .2088 .2123 .2157 .2190 .2224 ; 0.5
       ; ---------------------------------------------------------
       .2257 .2291 .2324 .2357 .2389 .2422 .2454 .2486 .2517 .2549 ; 0.6
       .2580 .2611 .2642 .2673 .2704 .2734 .2764 .2794 .2823 .2852 ; 0.7
       .2881 .2910 .2939 .2967 .2995 .3023 .3051 .3078 .3106 .3133 ; 0.8
       .3159 .3186 .3212 .3238 .3264 .3289 .3315 .3340 .3365 .3389 ; 0.9
       .3413 .3438 .3461 .3485 .3508 .3531 .3554 .3577 .3599 .3621 ; 1.0
       ; ---------------------------------------------------------
       .3643 .3665 .3686 .3708 .3729 .3749 .3770 .3790 .3810 .3830 ; 1.1
       .3849 .3869 .3888 .3907 .3925 .3944 .3962 .3980 .3997 .4015 ; 1.2
       .4032 .4049 .4066 .4082 .4099 .4115 .4131 .4147 .4162 .4177 ; 1.3
       .4192 .4207 .4222 .4236 .4251 .4265 .4279 .4292 .4306 .4319 ; 1.4
       .4332 .4345 .4357 .4370 .4382 .4394 .4406 .4418 .4429 .4441 ; 1.5
       ; ---------------------------------------------------------
       .4452 .4463 .4474 .4484 .4495 .4505 .4515 .4525 .4535 .4545 ; 1.6
       .4554 .4564 .4573 .4582 .4591 .4599 .4608 .4616 .4625 .4633 ; 1.7
       .4641 .4649 .4656 .4664 .4671 .4678 .4686 .4693 .4699 .4706 ; 1.8
       .4713 .4719 .4726 .4732 .4738 .4744 .4750 .4756 .4761 .4767 ; 1.9
       .4772 .4778 .4783 .4788 .4793 .4798 .4803 .4808 .4812 .4817 ; 2.0
       ; ---------------------------------------------------------
       .4821 .4826 .4830 .4834 .4838 .4843 .4846 .4850 .4854 .4857 ; 2.1
       .4861 .4864 .4868 .4871 .4875 .4878 .4881 .4884 .4887 .4890 ; 2.2
       .4893 .4896 .4898 .4901 .4904 .4906 .4909 .4911 .4913 .4916 ; 2.3
       .4918 .4920 .4922 .4925 .4927 .4929 .4931 .4932 .4934 .4936 ; 2.4
       .4938 .4940 .4941 .4943 .4945 .4946 .4948 .4949 .4951 .4952 ; 2.5
       ; ---------------------------------------------------------
       .4953 .4955 .4956 .4957 .4959 .4960 .4961 .4962 .4963 .4964 ; 2.6
       .4965 .4966 .4967 .4968 .4969 .4970 .4971 .4972 .4973 .4974 ; 2.7
       .4974 .4975 .4976 .4977 .4977 .4978 .4979 .4979 .4980 .4981 ; 2.8
       .4981 .4982 .4982 .4983 .4984 .4984 .4985 .4985 .4986 .4986 ; 2.9
       .4987 .4987 .4987 .4988 .4988 .4989 .4989 .4989 .4990 .4990)); 3.0


;;  Select random number in the given interval, with log-uniform distribution.

(defun log-uniform (min-value max-value)
  (declare (real min-value max-value))
  (exp (uniform (log min-value) (log max-value))))


;; Select a random number in the given interval, with uniform distribution.
;; The selected number is real (even for the ingerer interval boumds).
;;
;; min-value, man-value   interval bounds
;;
;; returned value: the selected number

(defun uniform (min-value max-value)
  (declare (real min-value max-value))
  (+ min-value (random (float (- max-value min-value)))))


;; Returns t with probability chance/max-chance and nil otherwaise.

(defun random-p (chance &optional (max-chance 1.0))
  (declare (real chance max-chance))
  (< (random max-chance) chance))


;; Given two lists and an element in the first list, return the element
;; of the second list that has the same position.

(defun same-position (elt lis other-lis &key (key #'identity) (test #'eql))
  (declare (type function key))
  (nth (position elt lis :key key :test test) other-lis))


;; The following two functions find the minimal and maximal element of
;; the list, respectively; if there are several maximal elements,
;; return the first of them.
;;
;; lis   list
;; pred  the predicate for comparing two elements (must provide a total order)
;; key   the function applied to the element before comparison
;;
;; returned value: the first of the maximal elements

(defmacro find-min (lis &key key)
  `(find-max ,lis :pred #'< :key ,key))

(defun find-max (lis &key (pred #'>) key)
  (declare (type function pred))
  (unless (and lis (listp lis))
    (error "find-max: `~S' is not a nonempty list" lis))
  (if key
    (let* ((max-elt (car lis))
           (max-value (funcall key max-elt)))
      (dolist (elt (cdr lis))
        (when (funcall pred (funcall key elt) max-value)
          (setf max-elt elt)
          (setf max-value (funcall key elt))))
      max-elt)
    (let ((max-elt (car lis)))
      (dolist (elt (cdr lis))
        (when (funcall pred elt max-elt)
          (setf max-elt elt)))
      max-elt)))

;; Determine whether "elt" is eql to some leaf of "tree". 
;;
;; This function is similar to "member", but it goes recursively to 
;; sublists. For example, 'c is a leaf of '(a ((b c) d)).
;;
;; This function is presently INEFFICIENT. If used often, it needs a
;; more efficient implementation.
;;
;; returned value:
;; nil if "elt" is not eql to any leaf of "tree";
;; something different from nil (undetermined) otherwise

(defun leaf (elt tree)
  (if (atom tree)
    (eql elt tree)
    (member elt tree :test #'leaf)))


;; Determine whether the "lis" list contains some leaf of "tree"; that
;; is, some element of "lis" is eql to some leaf of "tree".
;;
;; For example, '(1 c) contains a leaf of '(a ((b c) d)); this leaf is 'c.
;;
;; This function is presently INEFFICIENT. If used often, it needs a
;; more efficient implementation.
;;
;; returned value:
;; nil if no element of "lis" is eql to any leaf of "tree";
;; something different from nil (undetermined) otherwise

(defun some-leaf (lis tree)
  (if (listp tree)
    (member lis tree :test #'some-leaf)
    (member tree lis)))


;; Execute a program for every element of every element of a list.
;;
;; The syntax is (dosublists (elt list [result]) statement*), where
;; "elt" loops throught the elements of the elments of the list;
;; for example, (dosublists (elt '((a b) (c d))) ...) is equivalent to
;; (dolist (elt '(a b c d)) ...).
;;
;; elt            loop variable
;; lis            list, throught elements of whose sublists elt loops
;; result         expression who value is return after the execution
;; statementets   program to be executed for every value of elt
;;
;; returned value: evaluation of "result"

(defmacro dosublists ((elt lis &optional result) &body statements)
  `(dolist (sublis ,lis ,result) (dolist ,elt sublis ,@statements)))


;; Iterate until a condition is met, e.g. (until (> x 239) (incf x)).

(defmacro until (condition &body statements)
  `(do () (,condition) ,@statements))

;; Iterate while a condition is satsified, e.g. (while (< x 239) (incf x)).

(defmacro while (condition &body statements)
  `(do () ((not ,condition)) ,@statements))


;; Square a number.

(defmacro sqr (x)
  `(* ,x ,x))

;; Cube a number.

(defmacro cube (x)
  `(* ,x ,x ,x))


;; Find the root of the square sum.

(defun diagonal (&rest args)
  (if (listp (car args))
    (setf args (car args)))
  (let ((sum 0))
    (declare (real sum))
    (dolist (arg args)
      (incf sum (* arg arg)))
    (sqrt-real sum)))


;; Check that two values are within a rounding error from each other;
;; the maximal relative rounding error is 0.0001.

(defmacro small-diff-p (value1 value2)
  `(or (= ,value1 ,value2)
       (< (/ (abs (- ,value1 ,value2)) 
             (+ (abs ,value1) (abs ,value2)))
          0.0001)))


;; If a value is negative, replace it with zero when computing square 
;; root, thus avoiding complex numbers due to rounding errors.

(defun sqrt-real (value)
  (declare (real value))
  (if (< value -0.5)
    (error "sqrt-real: The value `~S' is to far from 0" value))
  (if (<= value 0)
    0.0
    (sqrt value)))


;; Find the mean and standard deviation,
;; and return them as a list.

(defmacro mean-dev (&rest args)
  `(multiple-value-list (mean ,@args)))


;; Find the mean and standard deviation,
;; and return them as multiple values.

(defun mean (&rest args)
  (if (listp (car args))
    (setf args (car args)))
  (cond
    ((null args) nil)
    ((short-p args) (car args))
    (t (let ((sum 0)
             (sqr-sum 0)
             (num (length args)))
         (declare (real sum sqr-sum) (integer num))
         (dolist (arg args)
           (incf sum arg)
           (incf sqr-sum (* arg arg)))
         (values (float (/ sum num))
                 (sqrt-real (/ (- sqr-sum (/ (* sum sum) num)) (1- num))))))))


;; Perform the least-squares regression.
;;
;; args   argument list in the form ((x1 y1) (x2 y2) ...); if an argument
;;   has more than two values, the remaining values are ignored
;; 
;; returned value:
;; (alpha beta dev cor), where dependency of y on x is y = alpha + beta * x,
;; dev is the deviation of y, and cor is the coefficient of correlation

(defun regress (&rest args)
  (if (singleton-p args)
    (setf args (car args)))
  (let ((sum-x 0) (sum-y 0) (sum-xx 0) (sum-yy 0) (sum-xy 0)
        (num (length args))
        alpha beta ss-x ss-y ss-xy sse)
    (if (< num 3)
      (error "regress: Argument list is too short"))
    (dolist (arg args)
      (incf sum-x (first arg))
      (incf sum-y (second arg))
      (incf sum-xx (sqr (first arg)))
      (incf sum-yy (sqr (second arg)))
      (incf sum-xy (* (first arg) (second arg))))
    (setf ss-x (- sum-xx (/ (* sum-x sum-x) num)))
    (setf ss-y (- sum-yy (/ (* sum-y sum-y) num)))
    (setf ss-xy (- sum-xy (/ (* sum-x sum-y) num)))
    (setf beta (/ ss-xy ss-x))
    (setf alpha (- (/ sum-y num) (* beta (/ sum-x num))))
    (setf sse (- ss-y (* beta ss-xy)))
    (list (float alpha) (float beta) (sqrt (/ sse (- num 2))) 
          (/ ss-xy (sqrt (* ss-x ss-y))))))


;; Determine if a hash-table has an entry with the specified key.
;;
;; Returned value: T if it has a value with this key; nil otherwise

(defmacro hash-member-p (key table)
  `(second (multiple-value-list (gethash ,key ,table))))


;; The following TWO macros add a new entry to a hashtable.
;;
;; "add-hash" errs if a non-nil entry with this key is already in the table
;; "push-hash" overwrites the old value in such case
;;
;; key, entry   hashing key and entry
;; table        hashtable

(defmacro add-hash (key entry table)
  `(if (hash-member-p ,key ,table)
     (error "add-hash: An entry with key `~S' is already in hash-table `~S'"
        ,key ,table)
     (setf (gethash ,key ,table) ,entry)))

(defmacro push-hash (key entry table)
  `(setf (gethash ,key ,table) ,entry))


;; Remove from a hash-table all entries whose value is nil.

(defmacro clean-hash (table)
  `(maphash #'(lambda (key entry) (unless entry (remhash key ,table))) 
            ,table))


;; The following TWO functions return the list of all keys and all entries
;; in a hash table. If the same entry is hashed on several keys, than the
;; list of entries contains multiple copies of this entry.

(defun hash-to-keys (table)
  (declare (type hash-table table))
  (let ((keys nil))
    (maphash #'(lambda (key entry) (ignore entry) (push key keys)) table)
    table))

(defun hash-to-entries (table)
  (declare (type hash-table table))
  (let ((entries nil))
    (maphash #'(lambda (key entry) (ignore key) (push entry entries)) table)
    table))


;; The following TWO macros execute a program for every entry of a
;; hash table.  The "dohash+" macro skips the entries whose value is nil.
;;
;; The syntax of this loop is similar to the sintax of "dolist":
;;   (dohash (key entry hashtable [result]) statement*),
;; where "key" loops through the keys of the hastable and "entry"
;; loops through the corresponding entries.
;;
;; key            variable that loops through the keys of the hashtable
;; entry          variable that loops through the entries of the hashtable
;; hashtable      the hashtable
;; result         expression whos value is returned after the execution
;; statementets   program to be executed for every value of the table
;;                  (it may contain the "key" amd "entry" variables)
;;
;; returned value: evaluation of "result"

(defmacro dohash ((key entry table &optional result) &body statements)
  `(progn (maphash #'(lambda (,key ,entry) ,@statements) ,table) ,result))

(defmacro dohash+ ((key entry table &optional result) &body statements)
  `(progn (maphash #'(lambda (,key ,entry) (when ,entry ,@statements)) ,table) 
     ,result))


;; The following TEN functions and macros are operations on a hash-table
;; of hash-tables, called dhash-table (where "d" is for "double").
;;
;; The operations are the same as Lisp hash-table operations and hash-table
;; operations in this file with similar names.

(defstruct (dhash-table (:print-function print-dhash-table))
  (table nil)  ;; The hash table of hash-tables.
  (test 'eql)  ;; The test in creating lower-level hash-tables.
  (size 0))    ;; The total number of entries in all lower-level hash-tables.

;; The output shows the tests, the number of subtables, and the total number
;; of their entries, #<EQL EQUAL dhash-table with 2 subtables and 6 entries>.
(defun print-dhash-table (table stream z)
  (declare (type dhash-table table) (stream stream) (ignore z))
  (princ "#<" stream)
  (princ (hash-table-test (dhash-table-table table)) stream)
  (princ " " stream)
  (princ (dhash-table-test table) stream)
  (princ " dhash-table with " stream)
  (princ (hash-table-test (dhash-table-count table)) stream)
  (princ "subtables and " stream)
  (princ (hash-table-size (dhash-table-count table)) stream)
  (princ "entries>"))  

;; Similar to make-hash-table.
(defun create-dhash-table (&key (test1 'eql) (test2 'eql))
  (make-dhash-table
    :table (make-hash-table :test test1) 
    :test (case test2
            ((eq #'eq) 'eq)
            ((eql #'eql) 'eql)
            ((equal #'equal) 'equal)
            (otherwise 
              (error "make-dhash-table: `~S' is not a valid test." test2)))
    :size 0))

;; Unlike gethash, cannot be used with setf; the second of the 
;; returned values is T if an entry is found; nil otherwise.
(defun getdhash (key1 key2 table &optional default)
  (declare (type dhash-table table))
  (let ((subtable (gethash key1 (dhash-table-table table))))
    (if subtable (gethash key2 subtable default) default)))

;; If this key is alread in the table, error; returns the new size.
(defun add-dhash (key1 key2 entry table)
  (declare (type dhash-table table))
  (let ((subtable (getsub-dhash key1 table)))
    (unless (hash-member-p key2 subtable)
      (error "add-hash: An entry with keys `~S' and `~S' is already in~
        dhash-table `~S'" key1 key2 table))
    (setf (gethash key2 subtable) entry)
    (incf (dhash-table-size table))))

;; Returns t if an entry with these key did not exist before; nil otherwise.
(defun push-dhash (key1 key2 entry table)
  (declare (type dhash-table table))
  (let* ((subtable (getsub-dhash key1 table))
         (existed-p (hash-member-p key2 subtable)))
    (setf (gethash key2 subtable) entry)
    (unless existed-p (incf (dhash-table-size table)) t)))

;; Returns a subtable for this key; if not found, creates it.
(defun getsub-dhash (key1 table)
  (declare (type dhash-table table))
  (or (gethash key1 (dhash-table-table table))
      (setf (gethash key1 (dhash-table-table table))
            (make-hash-table :test (dhash-table-test table)))))

;; Returns T if such an entry is found (and deleted); nil otherwise.
(defun remdhash (key1 key2 table)
  (declare (type dhash-table table))
  (let ((subtable (gethash key1 (dhash-table-table table))))
    (if subtable
      (let ((removed-p (remhash key2 subtable)))
        (if removed-p (decf (dhash-table-size table)))
        (if (zerop (hash-table-cound subtable))
          (remhash key1 (dhash-table-table table)))
        removed-p))))

;; Do for all entries.
(defmacro dodhash ((key1 key2 entry table &optional result) &body statements)
  `(progn 
     (maphash #'(lambda (,key1 subtable)
                  (maphash #'(lambda (,key2 ,entry) ,@statements) subtable))
              ,table)
     ,result))

;; Skip entries whose value is nil.
(defmacro dodhash+ ((key1 key2 entry table &optional result) &body statements)
  `(progn 
     (maphash #'(lambda (,key1 subtable)
                  (maphash #'(lambda (,key2 ,entry) (when ,entry ,@statements) 
                           subtable)))
              ,table)
     ,result))


;; The following TWO macros execute a program for every entry of a
;; property list.  The "doplist+" macro skips the entries whose value is nil.
;;
;; The syntax of this loop is similar to the sintax of "dolist":
;;   (doplist (key entry plist [result]) statement*),
;; where "key" loops through the keys of the property list and "entry"
;; loops through the corresponding entries.
;;
;; key            variable that loops through the keys of the property list
;; entry          variable that loops through the entries of the property list
;; plist          the property list
;; result         expression whos value is returned after the execution
;; statementets   program to be executed for every value of the table
;;                  (it may contain the "key" amd "entry" variables)
;;
;; returned value: evaluation of "result"

(defmacro doplist ((key entry plist &optional result) &body statements)
  `(do* ((plist-rest ,plist))
         ((null plist-rest) ,result)
     (let* ((,key (pop plist-rest))
            (,entry (pop plist-rest)))
       ,@statements)))

(defmacro doplist+ ((key entry plist &optional result) &body statements)
  `(do* ((plist-rest ,plist))
         ((null plist-rest) ,result)
     (let* ((,key (pop plist-rest))
            (,entry (pop plist-rest)))
       (when ,entry ,@statements))))


;; The following THREE functions convert a symbol into upcased string, 
;; downcased string, and capitalized string. If the argument is a symbol 
;; from the "keyword" package, the resulting string begins with ":"; 
;; for example ":Top-Type".
;;
;; strin   string or symbol
;;
;; returned value: the resulting string (upcased, downcased, or capitalized)

(defun string-up (sym)
  (declare (symbol sym))
  (if (keywordp sym)
    (concatenate 'string ":" (string-upcase sym))
    (string-upcase sym)))


(defun string-down (sym)
  (declare (symbol sym))
  (if (keywordp sym)
    (concatenate 'string ":" (string-downcase sym))
    (string-downcase sym)))


(defun string-cap (sym)
  (declare (symbol sym))
  (if (keywordp sym)
    (concatenate 'string ":" (string-capitalize sym))
    (string-capitalize sym)))


;; Return 0 for a negative number, and the number itself otherwise.

(defmacro negative-up (value)
  `(max ,value 0))


;; This macro is like "min" if it gets one or more arguments; 
;; it returns "nil" if it gets no arguments.

(defmacro min-with-nil (&body numbers)
  `(if ,numbers (min ,@numbers)))


;; If the varaible is smaller than the value, set it to the value.
;; nil is considered infinity.

(defun set-min-with-nil (var value)
  `(unless (and ,var (<= ,var ,value))
    (setf ,var ,value)))


;; This function determines the minimum of a list of numbers and nils;
;; it treats "nil" as +infinity. In partcular, if the input list is
;; either empty or contains only nil's, then the function returns nil.

(defun min-with-nils (lis)
  (let ((numerical-lis (remove nil lis)))
    (if numerical-lis (apply #'min numerical-lis))))


;; The destructive version of the "min-with-nils" function;
;; it destructively deletes nil's from the input sequence.

(defun n-min-with-nils (lis)
  (let ((numerical-lis (delete nil lis)))
    (if numerical-lis (apply #'min numerical-lis))))


;; Return all minimal elements of the list, counting "nil" for +infty; 
;; for example, (min-sublist 2 1 2 1 nil) -> (1 1). If the list is empty
;; or contains only nil's, return nil.
;;
;; If the ":test<" function is specified, it is used instead of #'< in
;; finding minimum; for example, (min-sublist 2 1 2 1 :test< #'>) ->
;; (2 2).  This function must take two arguments and provide a total
;; order on the elements of the list. If the ":test=" is specified, it
;; is used instead of #'= in determining the equivalence of two elements.
;; 
;; If the ":key" function is specified, it is applied to each element of the
;; list for determining its value; for example,
;; (min-sublist '((a a) (b) (c c) (d) :key #'length)) -> ((b) (d)).

(defun min-sublist (lis &key (test< #'<) (test= #'=) (key #'identity))
  (declare (type function key))
  (let ((min-lis nil) min-value value)
    (dolist (elt lis)
      (setf value (funcall key elt))
      (cond
        ((null value))
        ((or (null min-lis) (funcall test< value min-value))
          (setf min-lis (list elt))
          (setf min-value value))
        ((funcall test= value min-value)
          (push elt min-lis))))
    min-lis))


;; Execute the statements and set the given variables to the value
;; returned by the last statement and the time taken by the execution.
;; Note that time inlcludes executing not only the statements
;; but also the "(setf ,value ...)", which limits the accuracy.
;;
;; value        variable set to the value returned by the last statement
;; time         variable set to the execution time
;; units        the units of time, :units (internal units), :msec, or :sec
;; statements   statments to be exectued

(defmacro set-t (value time units &body statements)
  `(setf ,time (timer ,units (setf ,value (progn ,@statements)))))


;; Determine the name of a function and convert it into a 
;; symbol in the specified package.
;;
;; returned value:
;; the resulting name; nil if the function is unnamed

(defun function-name (func &optional (package *user-package*))
  (declare (type function func))
  (let* ((func-print (format nil "~S" func))
         (start (+ (position #\F func-print) 9))
         (end (position #\Space func-print :start start))
         (func-name (subseq func-print start end))
         (func-symbol 
           (unless (equal func-name "(unnamed)")
             (intern (string-upcase func-name) package))))
    func-symbol))


;; Execute the statements and return the running time that they took.
;;
;; units       the units of time, :units (internal units), :msec, or :sec
;; statements  statments to be exectued

(defmacro timer (units &body statements)
  `(let ((start-time (get-internal-run-time)))
    ,@statements
    (case ,units
      (scale-time (- (get-internal-run-time) start-time) units))))


;; Push the current time on the top of the timer stack.

(defun push-timer ()
  (declare (special *times*))
  (push (get-internal-run-time) *times*))


;; Determine the time since we pushed time on the timer stack.
;; units are :units (internal units), :msec, or :sec.

(defun read-timer (units)
  (declare (symbol units) (special *times*))
  (scale-time (- (get-internal-run-time) (car *times*)) units))


;; Pop the timer stack top and return the time since it has been pushed.
;; units are :units (internal units), :msec, or :sec.

(defun pop-timer (&optional (units :units))
  (declare (symbol units))
  (scale-time (- (get-internal-run-time) (pop *times*)) units))


;; The timer stack (a list).

(defvar *times* nil)


;; Convert the internal units of time into the specified units
;; and return the resulting value.
;;
;; time    number of the internal units of time
;; units   :units (internal units), :msec, or :sec

(defmacro scale-time (time units)
  `(case ,units
    (:units ,time)
    (:msec (* 1000 (/ ,time internal-time-units-per-second)))
    (:sec (/ ,time internal-time-units-per-second))
    (otherwise (error "timer: ~S are illegal units" ,units))))


;; If condition holds, execute the second statement; otherwise,
;; execute the first one.

(defmacro if-not (condition statement1 &optional statement2)
  `(if ,condition ,statement2 ,statement1))


;; Similar to mapcar, but for mapping macros rather than functions;
;; note that the macro name should NOT be preceded by #'. For example,
;; if we have a macro named "simple", we may apply it to the elements
;; of the '(a b c) list by (mapcar-macro simple '(a b c)).

(defmacro mapcar-macro (func lis)
  `(mapcar #'(lambda (elt) (,func elt)) ,lis))


;; Apply a sequence of functions and/or macros to an arguments; for
;; example, (applies sqr 1+ - 4) computes (sqr (1+ (- 4))). The last
;; function may have several arguments, in which case they must be in
;; a list; for example (applies sqr 1+ - (4 2)) computes (sqr (1+ (- 4 2))).

(defmacro applies (&rest args)
  (let* ((rev-args (reverse args))
         (expression (cons (second rev-args) (listify (first rev-args)))))
    (dolist (func (cddr rev-args) expression)
      (setf expression (list func expression)))))


;; The same as applies, but applies the functions or macros only until
;; it encounters nil. If it never envounters nil, then it returns the
;; result of all applications; otherwise, it returns nil. For example,
;; (applies sqr 1+ - or (4 nil)) computs (sqr (1+ (- (or 4 nil)))). On
;; the other hand, (applies sqr 1+ - and (4 nil)) get nil when computing
;; (and 4 nil) and does not apply the functions sqr, 1+, and -.

(defun applies+ (&rest args)
  (let* ((rev-args (reverse args))
         (var (gensym))
         (expression 
           (list (list 'setf var 
                   (cons (second rev-args) (listify (first rev-args)))))))
    (dolist (func (cddr rev-args))
      (push (list 'setf var (list func var)) expression))
    (list 'let (list var) (cons 'and (reverse expression)))))


;; If the "output" is not nil, use format print to the standard output.

(defmacro message (output &body args)
  `(if ,output (format t ,@args)))

;; Print a symbol if it is not nil. We use this macro for outputting
;; names in print-... functions for structures.
;;
;; sym      symbol to be printed
;; space    position of the space: :before, :after, or :none
;; stream   output stream

(defmacro print-nonnil-symbol (sym space &optional (stream t))
  `(when ,sym 
     (if (eq ,space :before) (princ " " ,stream))
     (princ ,sym ,stream)
     (if (eq ,space :after) (princ " " ,stream))))


;; Print a list of names (symbols), separated by commas, and the
;; number of unnamed elements (nils) in the list.

(defun print-names (lis &optional (stream t) (print-case *print-case*))
  (declare (stream stream) (symbol print-case))
  (let ((nil-num (count nil lis))
        (nonnils (remove nil lis))
        (*print-case* print-case))
    (declare (integer nil-num) (symbol *print-case*))
    (if nonnils
      (princ (car nonnils) stream))
    (dolist (name (cdr nonnils))
      (princ ", " stream)
      (princ name stream))
    (if (and nil-num (not (zerop print-case)))
      (princ ", and " stream))
    (unless (zerop print-case)
      (princ (zerop print-case) stream)
      (princ " unnamed" stream))))


;; Given a list, print the specified number of its elements. If the list
;; has fewer elements, print all elements. If the list has more elements,
;; print ellipsis afte the printed elements. Do NOT print parentheses
;; around the list. If the list is empty, do not print anything.
;;
;; If some elements of the list are lists, print the same number of
;; elements of each of these lists, with parentheses arount them.
;;
;; For example, (print-list '(a b c d) 3) gives "a b c ...";
;; (print-list '((a b c d) (e f)) 3) gives "(a b c ...) (e f)".
;;
;; lis      list to be printed
;; num      the number of elements (must be a positive integer)
;; stream   output stream

(defun print-list (lis num &optional (stream t))
  (declare (integer num) (stream stream))
  (when lis
    (print-maybe-list (pop lis) num stream)
    (do ((i (1- num) (1- i)))
        ((or (zerop num) (null lis)))
      (princ " " stream)
      (princ (pop lis) stream))
    (if lis (princ " ..." stream))))


;; If an element is a nonempty list, use "print-list" to print it;
;; enclosed the printed list in parentheses. Otherwise, use "princ".

(defmacro print-maybe-list (elt num stream)
  `(if (and ,elt (listp ,elt))
    (progn 
      (princ "(" ,stream) 
      (print-list ,elt ,num ,stream) 
      (princ ")" ,stream))
    (princ ,elt ,stream)))


;; For every leaf in a tree, if it is a symbol, replace it by the
;; symbol with the same name in another package; if a leaf is 
;; not a symbol, it remains unchanged.
;;
;; returned value: the resulting tree

(defun re-package-tree (tree &optional (package *user-package*))
  (declare (type package package))
  (cond 
    ((listp tree)
      (mapcar #'(lambda (subtree) (re-package-tree subtree package)) tree))
    ((symbolp tree) (intern (symbol-name tree) package))
    (t tree)))


;; Return a symbol or list of symbols with the same name in another package.

(defun re-package (sym-or-lis &optional (package *user-package*))
  (declare (type package package))
  (cond
    ((listp sym-or-lis)
       (mapcar #'(lambda (sym) 
                   (if (symbol-p sym) 
                     (intern (symbol-name sym) package)))
         sym-or-lis))
    ((symbolp sym-or-lis)
       (intern (symbol-name sym-or-lis) package))
    (t sym-or-lis)))


;; Pointers to packages.

(if (boundp '*user-package*) (makunbound '*user-package*))
(defconstant *user-package* (find-package "COMMON-LISP-USER"))

(if (boundp '*key-package*) (makunbound '*key-package*))
(defconstant *key-package* (find-package "KEYWORD"))

(if (boundp '*p4-package*) (makunbound '*p4-package*))
(defconstant *p4-package* (find-package "PRODIGY4"))

#|
=======================================
Debugging and user-interface functions.
=======================================

I write here functions that help me in debugging. These functions are
exported to other packages. I have coded the following functions:

aliases    define new global variables that point to other variables
show       inspect the value of some data
detail     give a detailed description of the contents of some data
show-hash-table  show the data in a hash-table
show-list  show data in a list, in the format of the Unix ls
|#


;; Define several new global variables, which point to other global
;; variables. I use it when I need to give several different names to
;; the same variable.

(defun aliasing ()
  (declare (special user::space p4::*pspace*))
  (setf user::space *current-problem-space*)
  (setf p4::*pspace* *current-problem-space*))


;; -------------------------------------------------------------------------
;; Show the content of data structures.

;; Inspecting the value of some data, usually a structure. The
;; function determines the type of "data", prefixes it with "show-",
;; and calls the resulting "show-" function. This "show-" function is
;; supposed to print the value of the data.
;;
;; For example, if "data" is of the type "matching", then (show data)
;; will call (show-matching data); similarly, if "data" is of the type
;; "prodigy-object", it will call (show-prodigy-object data).

(defun show (data &rest args)
  (declare (special *p4-package*) (type package *p4-package*))
  (let* ((data-type (symbol-name (type-of data)))
         (show-data (find-symbol (concatenate 'string "SHOW-" data-type) 
           *p4-package*)))
    (if (fboundp show-data)
      (apply (symbol-function show-data) (cons data args))
      (format t "~&`SHOW-' for type `~S' is not defined." data-type))))


;; The "detail" function is used in the same way as "show" function.
;; We may use it if the "detail-" function is defined for a data type.
;; This function usually give more detailed description than "show-".

(defun detail (data &rest args)
  (declare (special *p4-package*) (type package *p4-package*))
  (let* ((data-type (symbol-name (type-of data)))
         (detail-data (find-symbol (concatenate 'string "DETAIL-" data-type) 
           *p4-package*)))
    (if (fboundp detail-data)
      (apply (symbol-function detail-data) (cons data args))
      (format t "~&`DETAIL-' for type `~S' is not defined." data-type))))


;; Show the data in a hash-table.
;;
;; Print the keys and the entries of the hash-table.

(defun show-hash-table (table &optional (stream t))
  (declare (type hash-table table) (stream stream))
  (format stream "~&Hash-Table (keys -- entries):")
  (dohash (key entry table)
    (print key stream)
    (princ "-- " stream)
    (prin1 entry stream)))


;; Show the contents of a list in the format of the Unix ls. If the
;; print-out of some elements of the list is longer than 78
;; characters, the output may look ugly. This function is not meant
;; for showing lists with long sublists.

(defun show-list (lis &optional (stream t) (print-case *print-case*))
  (declare (stream stream) (symbol print-case))
  (let* ((*print-case* print-case)  ;; :upcase, :downcase, or :capitalize
         (strings (mapcar #'(lambda (elt) (format nil "~S" elt)) lis))
           ;; String representation of the elemnents, for printing.
         (max-length (+ (find-max (mapcar #'length strings)) 2))
           ;; Length of a field for each element when printing.
         (per-line (floor 78 max-length)) 
           ;; Number of elements per line.
         (num-rows (unless (zerop per-line) (ceiling (length lis) per-line)))
           ;; Number of lines.
         (lines (if num-rows (make-array num-rows))))
           ;; Array element contains list elts for printing on the same line.
    (declare (symbol *print-case*) 
             (integer max-length per-line num-rows)
             (type vector lines))
    (if (< per-line 2)
      (format stream "~&~{~A~%~}" strings)
      (progn
        ;; Determine which list element is printed on which line.
        (do ((string (pop strings) (pop strings))
             (row 0 (if (eq (1+ row) num-rows) 0 (1+ row))))
            ((null strings))
          (push string (aref lines row)))
        ;; Print the list elements.
        (format stream "~&")
        (dotimes (row num-rows)
          (dolist (string (reverse (aref lines row)))
            (format stream "~A" string)
            (dotimes (i (- max-length (length string))) (princ " " stream)))
          (terpri stream))))))
