// The Robotics Institute
// Carnegie Mellon University
// Copyright (C) 2000 by RI.
// All Rights Reserved.

// robot9.cpp
// PalmPilot Robot source code (Robot1 project)
// By Greg Reshko
// Last modified: 3/15/2000

#include "ppr.h"


#define Forward 3
#define Stop 5

#define True 1
#define False 0

#define Fast 1
#define Slow 2
#define Left      1
#define Right     2
#define ToServo1  1
#define ToServo2  2
#define ToServo3  3
#define RToServo1 14
#define RToServo2 15
#define RToServo3 16

static int StartApplication();
static void EventLoop();
static void StopApplication();
static Boolean Form1Handle(EventPtr event);

void SpinAlgorithm(void);
void WallFollowAlgorithm(void);
void TravelerAlgorithm(void);
void TargetFollowAlgorithm(void);
//void ContWallFollowAlgorithm(void);

void Turn(int direction, int speed);
void Move(int direction);
void PMove(int direction);
void Matrix_Traveler(void);



// main function; required for all applications
DWord  PilotMain (Word cmd, Ptr cmdPBP, Word launchFlags)
{
	short error;
	if (cmd == sysAppLaunchCmdNormalLaunch)
	{
		error = StartApplication();
		if (error) return error;
		EventLoop();
		StopApplication ();
	}
	return 0;
}


// ************* ROBOT'S FUNCTIONS *************

// ***** ALGORITHM FUNCTIONS *****

// This algorithm locks onto target and follows it.
// The robot adjusts its position and orientation so that
// sensor #3 always faces the target from 25...35cm distance.
// For faster target location, put target next to wheel #1.
// All objects further than 70cm are ignored.
// Functions used:
// Turn, Move, Drive, SmartWait, Sensor_To_Range, StopRobot

void TargetFollowAlgorithm(void) {
	ULong start_time;
	int pen_dist=30, dist, detected = False, lost = False, infi = 60, played = False;
	// put pen facing wheel #1
	// turn robot to the right until it finds the center of the pen
	Turn(Right, Slow);
	DisplayTxt("Scanning...");
	do {
		dist = Sensor_To_Range(Sensor(3));
		if (dist <= pen_dist)
			detected = True;
	} while ( (detected == False) && (SmartWait(1)==0) );
	StopRobot();	// stop robot
	// begin tracking loop
	do {
		// ADJUST ANGULAR POSITION
		dist = Sensor_To_Range(Sensor(3));
		if (dist > infi) {	// target lost turn left/right
			played = False;
			Turn(Right,Slow); Wait(50); StopRobot(); // turn right for awhile
			dist = Sensor_To_Range(Sensor(3));
			if (dist > infi) { // target is not to the right,
				Turn(Left,Slow);// turn left
				start_time = TimGetTicks();
				do {
					dist = Sensor_To_Range(Sensor(3));
				} while ( (TimGetTicks() < (start_time + 1000)) && (dist > infi) && (SmartWait(1)==0) );	// turn until found
				dist = Sensor_To_Range(Sensor(3));
				if (dist > infi) {
					lost = True;
					break; }
				StopRobot(); } }
		dist = Sensor_To_Range(Sensor(3));
		// ADJUST LINEAR POSITION
		if ( (dist < infi) && (dist < (pen_dist-5)) )	{ // too close to target
			played = False;
			Move(RToServo3);
			do {
				dist = Sensor_To_Range(Sensor(3));
				if (dist > infi) 	// lost angular tracking
					break;
			} while (dist < pen_dist);	// get dist to pen_dist
			StopRobot(); }
		if ( (dist < infi) && (dist > (pen_dist+5)) )	{ // too far
			played = False;
			Move(ToServo3);
			do {
				dist = Sensor_To_Range(Sensor(3));
				if (dist > infi)	// lost angular tracking
					break;
			} while (dist > pen_dist);	// get dist to pen_dist
			StopRobot(); }
		if ( (dist < (pen_dist+5)) && (dist > (pen_dist-5)) && (played == False) ) {
			SndPlaySystemSound(sndConfirmation);
			played = True; }
	} while ( (lost == False) && (SmartWait(10)==0) );
	StopRobot();
}

// "DRIVE & SPIN" SmartAlgorithm
// Robot moves forward while continously rotating.
// The loop is infinite and can be interrupted by penDown event
// at any time.
// Functions used:
// Vector_Drive, SmartWait, StopRobot
void SpinAlgorithm(void) {
	vector V(0,0);
	double omega=0.20, h=0;		  // body angular velocity in rad/sec
	do {
		V.x = 	0.07*cos(h);				// body x velocity vector in meters
		V.y = 	0.07*sin(h);				// body y velocity vector in meters
		h += omega;									// turn velocity vector by omega
		Vector_Drive(V, omega);			// drive robot using direction vector and angular velocity
	}	while (SmartWait(90) == 0); // loop will be interrupted by penDown event
	StopRobot();
}


// "WALL FOLLOW 2" SmartAlgorithm
// The robot follows a wall facing it with wheel #3 and
// using IR sensors #1 and #2.
// dist1 is back, dist2 is front
// Functions used:
// Drive, SmartWait, Sensor_To_Range, StopRobot
void WallFollowAlgorithm(void) {
	vector Z(0,0);
	int dist1, dist2, Tdist1=50, Tdist2=50, obstacle_dist=25;
	int adj_low=4, adj_high=10;	// servo's increase/decrease speed
	int t=5;	// tolerance
	Drive(136,136,50);						// initially drive straight
	while ( SmartWait(2)==0 ) {
		dist1 = Sensor_To_Range(Sensor(1));
		dist2 = Sensor_To_Range(Sensor(2));
		if (dist2 < obstacle_dist) // forward sensors < obstacle -> interior corner
			Drive(100,100,100);			 // turn in place
		else
			if (dist1 > (dist2+t))
				Drive(136-adj_low,136-adj_low,50); 		// dist1 > dist2 (angled left) -> turn right
			else
				if ((dist1 > 60) && (dist2 > 80))
					Drive(136+adj_high,136+adj_high,50);
				else
			
				if (dist1 < (dist2-t)) { // if angled right
				  if (dist2 < 90)  // if front not way long
					  Drive(136+adj_high,136+adj_high,50);	//	dist2 > dist1 (angled right) -> turn left
					else  // front is way long
					  // convex corner! - front way long but back short
					  Drive(136,136,50);	// drive straight
				} else
					if ( (dist1 > (Tdist1+t/2)) && (dist2 > (Tdist2+t/2)) )  // both too far
						// dist1 > Tdist1 and dist2 > Tdist2 -> move close
	 					Drive(136+adj_high,136-adj_low,50);
					else
						if ( (dist1 < (Tdist1-t/2)) && (dist2 < (Tdist2-t/2)) ) // both too close
							// dist1 < Tdist1 and dist2 < Tdist2 -> move far
							Drive(136-adj_low,136+adj_high,50);
						else
							Drive(136,136,50); }	// Drive straight
	StopRobot();
}



// "Wall Follow" Continuous SmartAlgorithm
// motion is a function of sensors
// s3 is front, s2 is back
// ------
// do not turn if front is way too long -- continue forward
// turning is always more important, gradually stop the robot if
// extreme turn is required
// difference between s1 and s2 controls rotation
// value of S1+S2 controls far/close motion
/*
void ContWallFollowAlgorithm(void) {
	vector D;
	double s1, s2, s3, adj, trn;
	while ( SmartWait(1)==0 ) {						// loop will be interrupted by touching the screen
		Scan(&s1,&s2,&s3);									// get vector lengths
		adj = -0.02*( (40 - (s2+s3)/2)/5 );	// adjust position
		trn = ( (s3-s2)/50 - ( (s3>15)?0:1 ) * ( (s3>90)?0:1 );	// adjust rotation
		if (trn > 1)	trn = 1; if (trn < -1)	trn = -1;							// make sure trn is in [-1,1]
		if (adj > 0.05)	adj = 0.05; if (adj < -0.05)	adj = -0.05;	// make sure adj is in [-0.05,0.05]
		D.Set(0.10, adj);	// set x and y components of the drive vector
		Vector_Drive(D*(1-abs(trn)), trn);	}	// drive using given direction vector and angular velocity 'trn'
	StopRobot();
}
*/


void TravelerAlgorithm(void) {
	vector Z(0,0);
	double vel=0.07;	// vel is top speed of servos
	int chosen_dir=0, old_dir=0, quit_loop=0;
	int sensor[6], sensor_obstacle=30, sensor_clearview=40;
//	Move(chosen_dir);
	
	do {
		do { 		// drive until robot sees an obstacle using a current sensor
			quit_loop = SmartWait(2);  	// obstacle detected
		} while ( (Sensor_To_Range(Sensor(chosen_dir)) > sensor_obstacle) && (quit_loop == 0) && (chosen_dir!=0) );
		sensor[1] = Sensor_To_Range(Sensor(1));	// get sensors
		sensor[2] = Sensor_To_Range(Sensor(2));
		sensor[3] = Sensor_To_Range(Sensor(3));
		sensor[4] = sensor[1];
		sensor[5] = sensor[2];
		// sensor+1 -> obstalce & sensor+2 -> obstalce
		if ( (sensor[chosen_dir+2] < sensor_obstacle) && (sensor[chosen_dir+1] < sensor_obstacle) ) {
			Vector_Drive(Z, 0.500);	// turn in place
			chosen_dir=0; }
		else {
			// sensor+1 -> clear  & sensor+2 -> obstacle | go to +1
			if ( (sensor[chosen_dir+1] > sensor_clearview) && (sensor[chosen_dir+2] < sensor_obstacle) )
				chosen_dir+=1;
			// sensor+1 -> obstalce & sensor+2 -> clear | go to +2
			if ( (sensor[chosen_dir+1] < sensor_obstacle) && (sensor[chosen_dir+2] > sensor_clearview) )
				chosen_dir+=2;
			// sensor+1 -> clear & sensor+2 -> clear | go to +1 or +2 -- pick random
			if ( (sensor[chosen_dir+2] > sensor_clearview) && (sensor[chosen_dir+1] > sensor_clearview) )
				// pick 2 for now, change to random later
				chosen_dir+=2;
			if (chosen_dir > 3)
				chosen_dir-=3;	// prevent overflow of current direction
			if (old_dir > 3)
				old_dir-=3;			// prevent overflow of previous direction

//			Move(chosen_dir); }
			switch(chosen_dir) {
				case 1:			Drive(0,50,200); break;
				case 2:			Drive(200,0,50); break;
				case 3:			Drive(50,200,0); break; }}
		Display(chosen_dir);
	} while (quit_loop==0);
	StopRobot();
}

// 'Matrix' obstacle avoidance algorithm
// The matrix represents wheel directions for all possible {s1,s2,s3} combinations 
// of sensor readings. Sn is 1 if there is an obstacle and 0 if it is clear.
void Matrix_Traveler(void) {
	int s1, s2, s3, row, threshold=40;
	int StateM[8][6] = {{0,0,0,  0, 0, 0},
									 	  {0,0,1, -1, 1, 0},
									    {0,1,0,  1, 0,-1},
									    {0,1,1,  0, 1,-1},									  									  
									    {1,0,0,  0,-1, 1},									  
									    {1,0,1, -1, 0, 1},
									    {1,1,0,  1,-1, 0},
									    {1,1,1,  1, 1, 1}};									  
	do {
		// Sn is 1 if there is an obstacle, 0 if it is clear.
		s1 = ( (Sensor_To_Range(Sensor(1))<threshold) ? 1:0 );
		s2 = ( (Sensor_To_Range(Sensor(2))<threshold) ? 1:0 );
		s3 = ( (Sensor_To_Range(Sensor(3))<threshold) ? 1:0 );	
		row = s1 + s2*2 + s3*4;
		if (row>0) {
			Servo(1, 128 + 100*StateM[row][3]);
			Servo(2, 128 + 100*StateM[row][4]);
			Servo(3, 128 + 100*StateM[row][5]);	}
	} while (SmartWait(2) == 0);
	StopRobot();
}


// ***** END OF ALGORITHM FUNCTIONS *****

// ***** HARDWARE FUNCTIONS *****

void Turn(int direction, int speed) {
	switch(speed) {
		case Fast:
			switch(direction) {
				case Right:	Drive(200,200,200); break;
				case Left:	Drive(50,50,50); 		break; }
			break;
		case Slow:
			switch(direction) {
				case Right:	Drive(140,140,140); break;
				case Left:	Drive(100,100,100); break; }
			break; }
}

// Drives robot perpendicular to given servo
// (Given servo can be sued for steering)
void Move(int direction) {
	switch(direction) {
		case RToServo1: 		Drive(0,200,50); break;
		case RToServo2: 		Drive(50,0,200); break;
		case RToServo3: 		Drive(200,50,0); break;
		case (ToServo1+3): 	Drive(0,50,200); break;
		case (ToServo2+3): 	Drive(200,0,50); break;
		case ToServo1: 			Drive(0,50,200); break;
		case ToServo2:			Drive(200,0,50); break;
		case ToServo3:			Drive(50,200,0); break; }
}

// Drives robot parallel to given servo
void PMove(int direction) {
	switch(direction) {
		case ToServo1:
			Drive(200,113,113);
			break;
		case ToServo2:
			Drive(50,0,200);
			break;
		case ToServo3:
			Drive(200,50,0);
			break;
		case RToServo1:
			Drive(50,136,136);
			break;
		case RToServo2:
			Drive(200,0,50);
			break;
		case RToServo3:
			Drive(50,200,0);
			break;
			 }
}

void IR_Scanner(void) {
	do {
	DisplayTxt("Reading dis IR #1");
	DisplayIR(1, Sensor_To_Range(Sensor(1)));
	DisplayTxt("Reading val IR #2");
	DisplayIR(2, Sensor(2));
	} while (SmartWait(2) == 0);
}


// ***** END OF CONVERTER FUNCTIONS *****



// ************* END OF ROBOT'S FUNCTIONS *************

// main event loop
static void EventLoop() {
  EventType e;
  unsigned short err, formID;
  FormType* form;
  do {
		EvtGetEvent(&e,-1);
    if (SysHandleEvent(&e)) continue;
    if (MenuHandleEvent(0, &e, &err)) continue;
    if (e.eType == frmLoadEvent) {
      formID = e.data.frmLoad.formID;
      form=FrmInitForm(formID);
      FrmSetActiveForm(form);
      switch (formID) {
        case Form1: FrmSetEventHandler(form, (FormEventHandlerPtr) Form1Handle); break; } }
    FrmDispatchEvent(&e);
  } while(e.eType != appStopEvent);
}

// Stop Application:
// close MathLib library and close serial port
static void StopApplication () {
	Err error;
  UInt usecount;
	error = MathLibClose(MathLibRef, &usecount);
	ErrFatalDisplayIf(error, "Can't close MathLib");
	if (usecount == 0)
    SysLibRemove(MathLibRef); 
  SerClose(portRef);
}

static Boolean Form1Handle(EventPtr event) {
	Boolean Handled=false;
    switch (event->eType) {
			case frmOpenEvent:
    		Handled=true;
        FrmDrawForm(FrmGetActiveForm());
				break;
      case nilEvent: 
       	Handled=true;
        break;
	    case penDownEvent:   break;
      case penUpEvent:     break;
      case penMoveEvent:   break;
      case keyDownEvent:   break;
      case winEnterEvent:  break;
      case winExitEvent:   break;
      case frmGotoEvent:   break;
      case frmUpdateEvent: break;
      case frmSaveEvent:   break;
      case frmCloseEvent:  break;
      case ctlEnterEvent:  break;
      case ctlExitEvent:   break; 
      case ctlSelectEvent:
        switch (event->data.ctlSelect.controlID) {
         	// Click: 'Forward' Button
					case Form1FwdButton:
						SndPlaySystemSound(sndAlarm);
						Drive(200,200,200);												
						while ( SmartWait(2)==0 ) {}
						StopRobot();
						SndPlaySystemSound(sndAlarm);												
 						break;
					// Click: "Stop" button
					case Form1StopButton:
						DisplayTxt("Stop            ");
						SndPlaySystemSound(sndConfirmation);
						Drive(0,0,0);
						break;
					// Click: 'IRScan' button
					case Form1IRScanButton:
						SndPlaySystemSound(sndConfirmation);
						IR_Scanner();	
						DisplayTxt("IR Scan complete");												
						SndPlaySystemSound(sndConfirmation);
						break;
					// Click: "Move & Spin" button
					case Form1MoveSpinButton:
						SndPlaySystemSound(sndAlarm);																	
						SpinAlgorithm();
						SndPlaySystemSound(sndAlarm);																		
						break;
					// Click: "Target Follow" button
					case Form1TargetFollowButton:
						SndPlaySystemSound(sndAlarm);																	
						TargetFollowAlgorithm();
						SndPlaySystemSound(sndAlarm);																		
						break;
					// Click: "Wall Follow" button
					case Form1WFollowButton:
						SndPlaySystemSound(sndAlarm);																	
						WallFollowAlgorithm();
						SndPlaySystemSound(sndAlarm);																		
						break;
					// Click: "Wandering" button
					case Form1DriveWTButton:
						SndPlaySystemSound(sndAlarm);												
						TravelerAlgorithm();
						SndPlaySystemSound(sndAlarm);												
						break;
					// Click: "Dir" button
					case Form1DirButton:
						TURN_DIR *= -1;	// toggle servo direction for different servos
						if (TURN_DIR == 1)
							DisplayTxt("DEF = 1 SRD = 1");
						else
							DisplayTxt("DEF = 1 SRD = -1");
						SndPlaySystemSound(sndConfirmation);							
						break; }
	      Handled=true;
        break;
	    default:
	    	FrmHandleEvent(FrmGetActiveForm(), event);
	    	Handled=true;
	    	break;
  }
  return Handled;
}

