// PalmPilot Robot source code
// Part of Robot1 project
// Modified by Greg Reshko 1/19/2000.




#define PILOT_PRECOMPILED_HEADERS_OFF

#include <Common.h>
#include <Pilot.h>
#include <SysAll.h>
#include <UIAll.h>
#include <FeatureMgr.h>
#include <SerialMgr.h>

#include <TimeMgr.h>
#include <StringMgr.h>
#include <SoundMgr.h>

extern "C" {
#include "MathLib\src\MathLib.h"
}

#include "vecmath.h"

#include "Robot2.h"


#define Form1 1000

#define Forward 3
#define Stop 5

#define True 1
#define False 0

#define Fast 1
#define Slow 2
#define Left      1 
#define Right     2
#define ToServo1  1
#define ToServo2  2
#define ToServo3  3
#define RToServo1 14
#define RToServo2 15
#define RToServo3 16

//#define PI 3.14159265359


static int StartApplication();
static void EventLoop();
static void StopApplication();
static Boolean Form1Handle(EventPtr event);


void Drive(int sv1, int sv2, int sv3);
void Servo(int number, int speed);
int Sensor(int number);
int Sensor_To_Range(int x);
void Display(int number);
void DisplayTxt(const Char* str);
void DisplayIR(int sensor, int value);

void SpinAlgorithm(void);
void WallFollowAlgorithm(void);
void TravelerAlgorithm(void);
void PenFollowAlgorithm(void);

void Turn(int direction, int speed);
void Move(int direction);
void PMove(int direction);
void Wait(int delay_ticks);
void StopRobot(void);
int Vel_To_Value(double vel);
void Vector_Drive(vector V, double omega);
int SmartWait(int delay_ticks);

UInt portRef;      // port reference, set by serial library, needed for communication with port
double k=0;

// main function; required for all applications
DWord  PilotMain (Word cmd, Ptr cmdPBP, Word launchFlags)
{
  short error;
  if (cmd == sysAppLaunchCmdNormalLaunch)
  {
    error = StartApplication();
    if (error) return error;
    EventLoop();
    StopApplication ();
  }
  return 0;
}


// find and open MathLib library
// find and open serial port
// open Form1
static int StartApplication() {
  Err error;
	error = SysLibFind(MathLibName, &MathLibRef);
	if (error)
    error = SysLibLoad(LibType, MathLibCreator, &MathLibRef);
	ErrFatalDisplayIf(error, "Can't find MathLib");
	error = MathLibOpen(MathLibRef, MathLibVersion);
	ErrFatalDisplayIf(error, "Can't open MathLib"); 
  SysLibFind("Serial Library", &portRef);
  SerOpen(portRef, 0, 9600);
  FrmGotoForm(Form1);
  return 0;
}



// ************* ROBOT'S FUNCTIONS *************


// This algorithm locks onto target an follows it.
// The robot adjusts its position and orientation so that
// sensor #3 always faces the target from 25...35cm distance.
// For faster target location, put target next to wheel #1.
// All objects further than 70cm are ignored.
void PenFollowAlgorithm(void) {
	ULong start_time;
	int pen_dist=30, dist, detected = False, lost = False, infi = 60, played = False;
	// put pen facing wheel #1
	// turn robot to the right until it finds the center of the pen
	Turn(Right, Slow);
	DisplayTxt("Scanning...");
	do {
		dist = Sensor_To_Range(Sensor(3));
		if (dist <= pen_dist) 
			detected = True;
	} while (detected == False);
	StopRobot();	// stop robot
	// begin tracking loop
	do {
		// ADJUST ANGULAR POSITION
		dist = Sensor_To_Range(Sensor(3));
		if (dist > infi) {	// target lost turn left/right
			played = False;
			Turn(Right,Slow); Wait(50); StopRobot(); // turn right for awhile
			dist = Sensor_To_Range(Sensor(3));	
			if (dist > infi) { // target is not to the right,
				Turn(Left,Slow);// turn left
				start_time = TimGetTicks();
				do { 
					dist = Sensor_To_Range(Sensor(3));
				} while ( (TimGetTicks() < (start_time + 1000)) && (dist > infi) );	// turn until found
				dist = Sensor_To_Range(Sensor(3));
				if (dist > infi) {
					lost = True;
					break; }
				StopRobot(); } }
		dist = Sensor_To_Range(Sensor(3));		
		// ADJUST LINEAR POSITION
		if ( (dist < infi) && (dist < (pen_dist-5)) )	{ // too close to target
			played = False;
			Move(RToServo3);
			do {
				dist = Sensor_To_Range(Sensor(3));
				if (dist > infi) 	// lost angular tracking
					break;
			} while (dist < pen_dist);	// get dist to pen_dist
			StopRobot(); }
		if ( (dist < infi) && (dist > (pen_dist+5)) )	{ // too far 
			played = False;
			Move(ToServo3);
			do {
				dist = Sensor_To_Range(Sensor(3));
				if (dist > infi)	// lost angular tracking
					break;
			} while (dist > pen_dist);	// get dist to pen_dist
			StopRobot(); }
		if ( (dist < (pen_dist+5)) && (dist > (pen_dist-5)) && (played == False) ) {
			SndPlaySystemSound(sndConfirmation);												
			played = True; }
	} while ( (lost == False) && (SmartWait(10)==0) );
	StopRobot();
}

// Drive&Spin algorithm
// Robot moves forward while continously rotating.
// The loop is infinite and can be interrupted by penDown event
// at any time.
void SpinAlgorithm(void) {
	vector V(0,0);
	double omega=0.20;		  // body angular velocity in rad/sec
	double h=0;
	DisplayTxt("Move&Spin running");
	do {	
		V.x = 	0.07*cos(h);		// body x velocity vector in meters
		V.y = 	0.07*sin(h);		// body y velocity vector in meters
		h += omega;							// turn velocity vector by omega
		Vector_Drive(V, omega);	// drive robot using direction vector and angular velocity
	}	while (SmartWait(90) == 0); // loop will be interrupted by penDown event
	StopRobot();
}

// Drives robot in a given direction vector V with given angular 
// velocity omega.
// r - wheel radius in meters
// b - wheel baseline in meters
void Vector_Drive(vector V, double omega) {
	vector F1(-1.000,0.000), F2(0.500,-0.866), F3(0.866,0.500);
	double omega1, omega2, omega3, b=0.090, r=0.020, h=0;
	omega1 = ( F1*V + b*omega ) / r;	// F1*V is overloaded dot product
	omega2 = ( F2*V + b*omega ) / r;
	omega3 = ( F3*V + b*omega ) / r;
	// makes sure that given path is physically possible
	if ( (omega1>6) || (omega2>6) || (omega3>6) || (omega1<-6) || (omega2<-6) || (omega3<-6) )
		DisplayTxt("Vectors: ERROR!");
	else 
		Drive(Vel_To_Value(omega1),Vel_To_Value(omega2),Vel_To_Value(omega3));
}



// Calculates distance using Power Regression y=a*x^b
// returns value in cm ranging from 10 to 80
int Sensor_To_Range(int x)
{
	double a=2141.72055;
	double b=-1.078867;
	int y=100;
	if (x>0) 	 y = a*pow(x,b);
	if (y<10)  y=10;
	if (y>100) y=100;
	return y;
}

// Calculates servo value using Cubic Regression y=a*x^3+b*x^2+c*x+d
// Converts velocity in rad per second to servo value (0-255)
// vel can only be [-6,6] rad/sec
// x can only be [-1,1] rev/sec
int Vel_To_Value(double vel) {
	double x;
	x = vel / PI / 2;	// convert rad/sec to rev/sec
	if ( (vel>-6) && (vel<6) )
		return (39.8053*x*x*x - 12.6083*x*x + 22.1197*x + 128.4262);
	else
		return 0;
}


void TravelerAlgorithm(void) {
	vector wheel[3], Z(0,0);
	double omega, vel=0.07;	// vel is top speed of servos
	int chosen_dir = 1, quit_loop=0;
	int sensor[6], sensor_obstacle=30, sensor_clearview=40;
	wheel[1].Set(-1.000*vel,0.000*vel);
	wheel[2].Set(0.500*vel,-0.866*vel);
	wheel[3].Set(0.866*vel,0.500*vel);

	Vector_Drive(wheel[chosen_dir], 0);
	
	do {
		do { 		// drive until robot sees an obstacle using a current sensor
			quit_loop = SmartWait(20);  	// obstacle detected
		} while ( (Sensor_To_Range(Sensor(chosen_dir)) > sensor_obstacle) && (quit_loop == 0) );
		sensor[1] = Sensor_To_Range(Sensor(1));	// get sensors
		sensor[2] = Sensor_To_Range(Sensor(2));
		sensor[3] = Sensor_To_Range(Sensor(3));		
		sensor[4] = sensor[1];		
		sensor[5] = sensor[2];		
		// sensor+1 -> obstalce & sensor+2 -> obstalce
		if ( (sensor[chosen_dir+2] < sensor_clearview) && (sensor[chosen_dir+1] < sensor_clearview) ) 
			Vector_Drive(Z, 0.500); 
		else {
			// sensor+1 -> clear  & sensor+2 -> obstacle
			if ( (sensor[chosen_dir+1] > sensor_clearview) && (sensor[chosen_dir+2] < sensor_clearview) ) 
				chosen_dir+=1;
			// sensor+1 -> obstalce & sensor+2 -> clear
			if ( (sensor[chosen_dir+2] > sensor_clearview) && (sensor[chosen_dir+1] < sensor_clearview) ) 
				chosen_dir+=2;
			// sensor+1 -> clear & sensor+2 -> clear
			if ( (sensor[chosen_dir+2] > sensor_clearview) && (sensor[chosen_dir+1] > sensor_clearview) ) 
				// pick 2 for now, change to random later
				chosen_dir+=2;
			if ((chosen_dir) > 3)
				chosen_dir-=3;	// prevent overflow
			Move(chosen_dir); }
			Display(chosen_dir);
	} while (quit_loop==0);
	StopRobot();
}

/*
			case 1:
				// if sensor+1 and sensor+2 are clear, pick random
				if ((sensor[dir+1] > sensor_clearview) && (sensor[dir+2] > sensor_clearview)) {
					tt = TimGetTicks();
					t = tt/2;
					t = t*2;
					if (tt == t) // even time tick
						dir+= 1;	// add random here
					else
						dir+= 2;	
					Move(dir);
					Display(dir);
					chosen=1;						
					break; }  

*/

void Turn(int direction, int speed) {
	switch(speed) {
		case Fast:
			switch(direction) {
				case Right:	Drive(200,200,200); break;
				case Left:	Drive(50,50,50); 		break; }
			break;
		case Slow:
			switch(direction) {
				case Right:	Drive(140,140,140); break;
				case Left:	Drive(100,100,100); break; }
			break; }
}

// Drives robot perpendicular to given servo
// (Given servo can be sued for steering)
void Move(int direction) {
	switch(direction) {
		case RToServo1: 		Drive(0,200,50); break;
		case RToServo2: 		Drive(50,0,200); break;
		case RToServo3: 		Drive(200,50,0); break;
		case (ToServo1+3): 	Drive(0,50,200); break;
		case (ToServo2+3): 	Drive(200,0,50); break;
		case ToServo1: 			Drive(0,50,200); break;
		case ToServo2:			Drive(200,0,50); break;
		case ToServo3:			Drive(50,200,0); break; }
}


// Drives robot parallel to given servo
void PMove(int direction) {
	switch(direction) {
		case ToServo1:
			Drive(200,113,113);
			break;
		case ToServo2:
			Drive(50,0,200);
			break;
		case ToServo3:
			Drive(200,50,0);
			break;
		case RToServo1:
			Drive(50,136,136);
			break;
		case RToServo2:
			Drive(200,0,50);
			break;
		case RToServo3:
			Drive(50,200,0);
			break;
			 }

}


// Suspends application for specified number of ticks
void Wait(int delay_ticks) {
	ULong start_time;
	start_time = TimGetTicks();
	while ( TimGetTicks()	< (start_time + delay_ticks));
}

// Pauses application for specified number of ticks
// Can be interrupted by penDown event
// Returns 1 if it was interrupted by penDown
// Returns 0 if delay ended with no interrupts
int SmartWait(int delay_ticks) {
	ULong start_time;
	EventPtr quit_event;
	start_time = TimGetTicks();
	while ( TimGetTicks()	< (start_time + delay_ticks)) {
		EvtGetEvent(quit_event, 1);	// get event using 1ms timeout
		if (quit_event->eType ==	penDownEvent)	// stop if screen was touched anywhere
			return 1; }
	return 0;
}


// "Wall follow 2" algorithm
// Align the robot to a wall using any 2 sensors
void WallFollowAlgorithm(void) {
	vector Z(0,0);
	int infi=80, facing_wheel=0;
	int dist1, dist2, dist3;
	int t=10;	// tolerance
	
	Vector_Drive(Z, 0.300);	// turn robot
	
	do {
		dist1 = Sensor_To_Range(Sensor(1));
		dist2 = Sensor_To_Range(Sensor(2));
		dist3 = Sensor_To_Range(Sensor(3));			
		if ( (dist2<(dist3+t))&&(dist2>(dist3-t))&&(dist2<infi) ) 
			facing_wheel=1;
		if ( (dist1<(dist3+t))&&(dist1>(dist3-t))&&(dist1<infi) ) 
			facing_wheel=2;
		if ( (dist1<(dist2+t))&&(dist1>(dist2-t))&&(dist1<infi) ) 
			facing_wheel=3;
		Display(k);
		k++;
	} while(facing_wheel==0);
	// similar sensors found 
	Display(facing_wheel);
	Wait(200);
	StopRobot();
}





/*

// "Wall follow" algorithm
// Drives 'sideways' using Sensor #1 to track distance
void WallFollowAlgorithm(void) {
	ULong start_time;
	int distance, target_dist, ok=1, l;
	SndPlaySystemSound(sndAlarm);													
	for (l=1; l<=3; l++) {

	StopRobot();
	Wait(100);
	target_dist = Sensor_To_Range(Sensor(1)); 	// lock target distance
	start_time = TimGetTicks();
	DisplayTxt("DistLocked");
	while ((ok == 1) && (TimGetTicks()	< (start_time + 1000)) ) {
		distance = Sensor_To_Range(Sensor(1));
		if (distance > (target_dist+1))	// too far -> turn left
			Drive(200,113+5,113+5);
		if (distance < (target_dist-1))	// too close -> turn right
			Drive(160,113-5,113-5);
		if ( (distance < (target_dist+1)) && (distance > (target_dist-1)) )	// continue straight
			Drive(200,113,113);
		if (distance > 120)	// lift the robot to stop it
			break; }
	if (ok == 1) {
	StopRobot();
	Move(RToServo1);
	Wait(50); }

	StopRobot();
	Wait(100);
	target_dist = Sensor_To_Range(Sensor(1)); 	// lock target distance
	start_time = TimGetTicks();
	DisplayTxt("DistLocked");
	while ((ok == 1) && (TimGetTicks()	< (start_time + 1000)) ) {
		distance = Sensor_To_Range(Sensor(1));
		if (distance > (target_dist+1))	// too far -> turn right
			Drive(50,136-5,136-5);
		if (distance < (target_dist-1))	// too close -> turn left
			Drive(50,136+3,136+3);
		if ( (distance < (target_dist+1)) && (distance > (target_dist-1)) )	// continue straight
			Drive(50,136,136);
		if (distance > 120)	// lift the robot to stop it
			ok=0; }
	if (ok == 1) {
	StopRobot();
	Move(RToServo1);
	Wait(50); }
	}
	
	StopRobot();
}

*/



// Reads and displays value of given analog sensor
int Sensor(int number) {
	// send request for analog value to SSC board
	Err error;
	char buf[10];
	char incoming[]="       ";
	int value=20;
	int len;
	int l;	// loop variable
	StrPrintF(buf, "AD%d \r", number);
	len = StrLen(buf);
	SerSendWait(portRef, -1); // wait for port to clear
	SerSend(portRef, buf, len, &error); // send buffer
	// receive from portRef into incoming buffer using 5 bytes and 3ms timeout
	SerReceive(portRef, incoming, 4, 10, &error);
	// buffer correction -- replace all ASCII non-numbers with spaces
	for (l=0; l<=2; l++)
		if ( incoming[l] == 10 )
			incoming[l]=' ';
	incoming[3] = '\0';
	if (incoming[0] != '1')	// two digit number cut-off
		incoming[2] = '\0';
	SerReceiveFlush(portRef, 5); // flush buffers
	SerSendFlush(portRef);
  value = StrAToI(incoming);
	return value;
}


// Display string (outputs it to Message Field)
void DisplayTxt(const Char* str) {
	FormPtr frm = FrmGetActiveForm();
	FrmCopyLabel(frm, Form1MesLabel, "              ");
	FrmCopyLabel(frm, Form1MesLabel, str);
}

// Displays number (outputs it to Message Field)
void Display(int number) {
	FormPtr frm = FrmGetActiveForm();
	char numbertxt[]="              ";
	FrmCopyLabel(frm, Form1MesLabel, numbertxt);
	StrIToA(numbertxt, number);
	FrmCopyLabel(frm, Form1MesLabel, numbertxt);
}


// Displays IR 1 or 2
void DisplayIR(int sensor, int value) {
	FormPtr frm = FrmGetActiveForm();
	char numbertxt[]="              ";
	StrIToA(numbertxt, value);
	switch(sensor) {
		case 1:
			FrmCopyLabel(frm, Form1IRoneLabel, numbertxt);
			FrmCopyLabel(frm, Form1IRoneLabel, numbertxt);
			break;
		case 2:
			FrmCopyLabel(frm, Form1IRtwoLabel, numbertxt);
			FrmCopyLabel(frm, Form1IRtwoLabel, numbertxt);
			break; }
}

// Stops robot.
void StopRobot(void) {
	Drive(0,0,0);
	DisplayTxt("Stop");	
}

// Update all 3 drive servos
void Drive(int sv1, int sv2, int sv3) {
  Servo(1, sv1);
  Servo(2, sv2);
  Servo(3, sv3);
}

// Sends value to given servo
void Servo(int number, int speed) {
  Err error;
  char buf[10];
  int len;
  StrPrintF(buf, "SV%d M%d \r", number, speed);
  len = StrLen(buf);
  SerSendWait(portRef, -1); // wait for port to clear
  SerSend(portRef, buf, len, &error); // send buffer
}

// ************* END OF ROBOT'S FUNCTIONS *************

// main event loop
static void EventLoop() {
  EventType e;
  unsigned short err, formID;
  FormType* form;
  do {
    EvtGetEvent(&e,-1);
    if (SysHandleEvent(&e)) continue;
    if (MenuHandleEvent(0, &e, &err)) continue;
    if (e.eType == frmLoadEvent) {
      formID = e.data.frmLoad.formID;
      form=FrmInitForm(formID);
      FrmSetActiveForm(form);
      switch (formID) {
        case Form1: FrmSetEventHandler(form, (FormEventHandlerPtr) Form1Handle); break; } }
    FrmDispatchEvent(&e);
  } while(e.eType != appStopEvent);
}

// Stop Application:
// close MathLib library and close serial port
static void StopApplication () {
	Err error;
  UInt usecount;
	error = MathLibClose(MathLibRef, &usecount);
	ErrFatalDisplayIf(error, "Can't close MathLib");
	if (usecount == 0)
    SysLibRemove(MathLibRef); 
  SerClose(portRef);
}

static Boolean Form1Handle(EventPtr event) {
	Boolean Handled=false;
    switch (event->eType) {
    	case frmOpenEvent:
    		Handled=true;
        FrmDrawForm(FrmGetActiveForm());
				break;
      case nilEvent: 
       	Handled=true;
        break;
	    case penDownEvent:   break;
      case penUpEvent:     break;
      case penMoveEvent:   break;
      case keyDownEvent:   break;
      case winEnterEvent:  break;
      case winExitEvent:   break;
      case frmGotoEvent:   break;
      case frmUpdateEvent: break;
      case frmSaveEvent:   break;
      case frmCloseEvent:  break;
      case ctlEnterEvent:  break;
      case ctlExitEvent:   break; 
      case ctlSelectEvent:
        switch (event->data.ctlSelect.controlID) {
         	// Click: Forward Button
					case Form1FwdButton:
//						SndPlaySystemSound(sndAlarm);												
						vector K;
						K.Set(0.55*0.07,0.866*0.07);
						Vector_Drive(K,0);
 						break;
					// Click: "Stop" button
					case Form1StopButton:
						DisplayTxt("Stop");				
						SndPlaySystemSound(sndConfirmation);												
						Drive(0,0,0);
						break;
					// Click: IRScan button
					case Form1IRScanButton:
						DisplayTxt("Reading IR #1   ");
						DisplayIR(1, Sensor_To_Range(Sensor(1)));
						DisplayTxt("Reading IR #2   ");
						DisplayIR(2, Sensor_To_Range(Sensor(2)));
						DisplayTxt("IR Scan complete");						
						break;
					// Click: "Move & Spin" button
					case Form1MoveSpinButton:
						SpinAlgorithm();
						break;
					// Click: "Target Follow" button
					case Form1TargetFollowButton:
						PenFollowAlgorithm();
						break;
					// Click: "Wall Follow" button
					case Form1WFollowButton:
						DisplayTxt("Wall Follow: running");
						WallFollowAlgorithm();
						break;
					// Click: "Wandering" button
					case Form1DriveWTButton:
						SndPlaySystemSound(sndAlarm);												
						TravelerAlgorithm();
						break;  }
	      Handled=true;
        break;
	    default:
	    	FrmHandleEvent(FrmGetActiveForm(), event);
	    	Handled=true;
	    	break;
  }
  return Handled;
}

