
#include <iostream>
#include <strstream>
#include <sstream>
#include <iomanip>
#include "test.h"
#include "misc.h"
#include "Geometry.h"
#include "Formation.h"
#include "FieldImage.h"
#include "FixedRWMI.h"
#include "msgstore.h"
#include "PlayerOccupancy.h"
#include "AbstractStateElements.h"
#include "MarkovChain.h"
#include "LibSeaGraphWriter.h"
#include "AbstractStateDifference.h"
#include "SoccerMDP.h"
#include "MDPConversion.h"
#include "QTableFlex.h"
#include "ServerParam.h"
#include "Logger.h"
#include "StateValue.h"
#include "AdviceTree.h"
#include "BinaryFile.h"

using namespace std;
using namespace spades;

void
testError()
{
  double d;
  int prec;

  d=123; prec = 3;
  cout << "Val " << d << " (precision " << prec <<") has error +/- " << getErrorFor(d, prec) << endl;
  d=123.456; prec = 6;
  cout << "Val " << d << " (precision " << prec <<") has error +/- " << getErrorFor(d, prec) << endl;
  d=.01234; prec = 4;
  cout << "Val " << d << " (precision " << prec <<") has error +/- " << getErrorFor(d, prec) << endl;
  d=1.1; prec = 2;
  cout << "Val " << d << " (precision " << prec <<") has error +/- " << getErrorFor(d, prec) << endl;
  d=0; prec = 2;
  cout << "Val " << d << " (precision " << prec <<") has error +/- " << getErrorFor(d, prec) << endl;
}

void
performAllErrBoundOps(const ErrBoundValue& v1, const ErrBoundValue& v2)
{
  ErrBoundValue tmp;
  cout << "\nVals: " << v1 << "\t" << v2 << endl;
  cout << "-(unary): " << (-v1) << endl;
  cout << "multInverse: " << (v1.multInverse()) << endl;
  tmp = v1; tmp += v2;
  cout << "+ :" << (v1+v2) << "\t+=: " << tmp << endl;
  tmp = v1; tmp -= v2;
  cout << "- :" << (v1-v2) << "\t-=: " << tmp << endl;
  tmp = v1; tmp *= v2;
  cout << "* :" << (v1*v2) << "\t*=: " << tmp  << endl;
  tmp = v1; tmp /= v2;
  cout << "/ :" << (v1/v2) << "\t/=: " << tmp << endl;
  cout << "== :" << (v1==v2) << endl;
  cout << "!= :" << (v1!=v2) << endl;
  cout << "< :" << (v1<v2) << endl;
  cout << "<=:" << (v1<=v2) << endl;
  cout << "> :" << (v1>v2) << endl;
  cout << ">=:" << (v1>=v2) << endl;
}

void
testErrBoundValue()
{
  ErrBoundValue v1(5, 1, 3);
  ErrBoundValue v2(5, 1, 3);
  v1.setPrintVerbose(true);
  v2.setPrintVerbose(true);
  v1.setVal(5);
  v2.setVal(5);
  performAllErrBoundOps(v1, v2);
  v1.setVal(-10);
  v2.setVal(5);
  performAllErrBoundOps(v1, v2);
  v1.setVal(-10);
  v2.setVal(-5);
  performAllErrBoundOps(v1, v2);
  v1.setVal(5);
  v2.setVal(-10);
  performAllErrBoundOps(v1, v2);
  v1.setVal(0);
  v2.setVal(-5);
  performAllErrBoundOps(v1, v2);
  v1.setVal(0);
  v2.setVal(5);
  performAllErrBoundOps(v1, v2);
  v1.setVal(5);
  v2.setVal(0.5);
  performAllErrBoundOps(v1, v2);
  v1.setVal(5);
  v2.setVal(-0.5);
  performAllErrBoundOps(v1, v2);
  v1.setVal(5);
  v2.setVal(10); v2.setLowerOffset(0); v2.setUpperOffset(0);
  performAllErrBoundOps(v1, v2);
}

void
testVectorIO()
{
  istrstream in1("(2.1, 3.1)");
  istrstream in2("(2 3)");
  istrstream in3("22.0, 33.0");
  istrstream in4("11.1 22.2");
  istrstream in5("BLAH");
  istrstream in6("(2.0 BlAH, 3.0)");
  istrstream in7("BLAH");
  istrstream in8("(2.0 ");
  VecPosition v;

  in1 >> v; cout << "fail: " << in1.fail() << "\tvalue: " << v << endl;
  in2 >> v; cout << "fail: " << in2.fail() << "\tvalue: " << v << endl;
  in3 >> v; cout << "fail: " << in3.fail() << "\tvalue: " << v << endl;
  in4 >> v; cout << "fail: " << in4.fail() << "\tvalue: " << v << endl;
  in5 >> v; cout << "fail: " << in5.fail() << "\tvalue: " << v << endl;
  in6 >> v; cout << "fail: " << in6.fail() << "\tvalue: " << v << endl;
  in7 >> v; cout << "fail: " << in7.fail() << "\tvalue: " << v << endl;
  in8 >> v; cout << "fail: " << in8.fail() << "\tvalue: " << v << endl;
  
}

void
testFormationIO()
{
  ifstream infile("Formations/simple.form");
  Formation f;
  
  infile >> f;

  cout << "Test Formation: " << endl
       << f << endl;

  f.permuteForGoalie(11);
  f.setHomeRegion(11, new rcss::clang::RegNull);
  cout << "Test Formation (goalie = 11): " << endl
       << f << endl;
}


void
testFieldImage()
{
  std::vector<VecPosition> v;
  std::vector<VecPosition> vHalo;

  /* this tests basic memory management stuff and
     tests the basic points (with halos) */
  
  FieldImage fi1(5, 8);
  fi1.addFieldLines();

  fi1.addPoint(VecPosition(-10, -10));
  fi1.addPoint(VecPosition(10, 10), FieldImage::COLOR_RED), 3;
  
  FieldImage fi2(fi1);

  v.push_back(VecPosition(-20, -10));
  v.push_back(VecPosition(-20, -5));
  v.push_back(VecPosition(-20, 0));
  v.push_back(VecPosition(-20, 5));
  v.push_back(VecPosition(-20, 10));
  fi2.addPoints(v);

  FieldImage::Color fillc(FieldImage::COLOR_RED);
  fillc.setAlpha(0.9);
  fi2.addPointWithHalo(VecPosition(40, 0), 5, 10, FieldImage::COLOR_RED, fillc);
  
  FieldImage fi3(2, 2);
  fi3 = fi1;

  vHalo.push_back(VecPosition(1, 10));
  vHalo.push_back(VecPosition(3, 8));
  vHalo.push_back(VecPosition(5, 6));
  vHalo.push_back(VecPosition(7, 4));
  vHalo.push_back(VecPosition(9, 2));
  fi3.addPointsWithHalo(v, vHalo, FieldImage::COLOR_GREEN, fillc);
  
  fi1.writeTo("testfi1");
  fi2.writeTo("testfi2");
  fi3.writeTo("testfi3");

  //This tests the arrows and lines
  FieldImage fi4(0, 8);
  std::vector< std::pair<VecPosition, VecPosition> > vLines;

  fi4.addFieldLines();
  
  fi4.addLine(VecPosition(-10, 10), VecPosition(10, -10));

  vLines.push_back(make_pair(VecPosition(-20, -10), VecPosition(-10, -15)));
  vLines.push_back(make_pair(VecPosition(-20, -5), VecPosition(-10, -10)));
  vLines.push_back(make_pair(VecPosition(-20, 0), VecPosition(-10, -5)));
  vLines.push_back(make_pair(VecPosition(-20, 5), VecPosition(-10, 0)));
  vLines.push_back(make_pair(VecPosition(-20, 10), VecPosition(-10, 5)));
  fi4.addLines(vLines, FieldImage::COLOR_BLUE, 4);

  fi4.addArrow(VecPosition(20, 20), VecPosition(30, 30));
  vLines.clear();
  vLines.push_back(make_pair(VecPosition(20, -10), VecPosition(10, -15)));
  vLines.push_back(make_pair(VecPosition(20, -5), VecPosition(10, -10)));
  vLines.push_back(make_pair(VecPosition(20, 0), VecPosition(10, -5)));
  vLines.push_back(make_pair(VecPosition(20, 5), VecPosition(10, 0)));
  vLines.push_back(make_pair(VecPosition(20, 10), VecPosition(10, 5)));
  fi4.addArrows(vLines, FieldImage::COLOR_BLUE, 4);
  
  fi4.writeTo("testfi4");

  FieldImage fi5(3, 6);
  fi5.addFieldLines();
  fi5.addText("I am a test", VecPosition(10, 10));
  fi5.addLegendLine("First legend line");
  //fi5.addLegendLine("Second legend line", FieldImage::COLOR_RED);
  fi5.addLegendLineWithDot("Second legend line", FieldImage::COLOR_RED);
  //fi5.addLegendLine("Third legend line", FieldImage::COLOR_GREEN);
  fi5.addLegendLineWithDot("Third legend line", FieldImage::COLOR_GREEN, FieldImage::COLOR_BLUE);
  fi5.writeTo("testfi5");
  
  
  for (int i=1; i<10; i++)
    {
      FieldImage fi(0, i);
      fi.addFieldLines();
      fi.addPoints(v);
      fi.addPointWithHalo(VecPosition(40, 0), 5, 10, FieldImage::COLOR_RED, fillc);
      fi.addArrows(vLines, FieldImage::COLOR_BLUE, 4);
      ostrstream fn;
      fn << "testfi-scale" << i << ends;
      fi.writeTo(fn.str());
      fn.freeze(0);
    }
  
}

void
testDrawPoints()
{
  FieldImage fi;
  VarBindings varbind;
  FixedRWMI wmi;
  bool verbose = true;

  fi.addFieldLines();
  
  rcss::clang::PointSimple ps1(-10, -5);
  ps1.draw(&fi, verbose, FieldImage::COLOR_GREEN, &wmi, varbind);
  rcss::clang::PointSimple ps2(20, 15);
  ps2.draw(&fi, verbose, FieldImage::COLOR_GREEN, &wmi, varbind);

  rcss::clang::PointRel pr1(6, 10, std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall));
  pr1.draw(&fi, verbose, FieldImage::COLOR_GREEN, &wmi, varbind);
  rcss::clang::PointRel pr2(6, 10, std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(30, 20)));
  pr2.draw(&fi, verbose, FieldImage::COLOR_GREEN, &wmi, varbind);

  rcss::clang::PointBall pb;
  pb.draw(&fi, verbose, FieldImage::COLOR_GREEN, &wmi, varbind);

  for (int i=1; i<=11; i++)
    {
      rcss::clang::PointPlayer pp(true, rcss::clang::UNum(i));
      pp.draw(&fi, verbose, FieldImage::COLOR_RED, &wmi, varbind);
    }
  for (int i=1; i<=11; i++)
    {
      rcss::clang::PointPlayer pp(false, rcss::clang::UNum(i));
      pp.draw(&fi, verbose, FieldImage::COLOR_BLUE, &wmi, varbind);
    }

  rcss::clang::PointArith pa(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall),
			     std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall),
			     rcss::util::ArithOp::plus());
  pa.draw(&fi, verbose, FieldImage::COLOR_GREEN, &wmi, varbind);

  fi.writeTo("testdrawpt");
}

void
testDrawRegions()
{
  FieldImage fi;
  VarBindings varbind;
  FixedRWMI wmi;
  bool verbose = true;

  FieldImage::Color color_fill(0, 0, 255, 0.8);
  
  fi.addFieldLines();
  fi.addGrid(4);
  
  rcss::clang::RegArc rarc1(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-30, -25)),
			    0, 10, 0, 360);
  rarc1.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc2(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-15, -25)),
			    0, 10, 120, 360);
  rarc2.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc3(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(5, -25)),
			    5, 15, 120, 360);
  rarc3.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc4(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-30, 0)),
			    0, 10, 120, 90);
  rarc4.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc5(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-10, 0)),
			    0, 10, 0, 130);
  rarc5.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc6(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(5, 0)),
			    0, 10, -50, 300);
  rarc6.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc7(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-30, 25)),
			    8, 10, 120, -90);
  rarc7.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc8(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-10, 25)),
			    5, 10, 0, -130);
  rarc8.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc9(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(5, 25)),
			    5, 10, -50, 300);
  rarc9.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc10(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(40, -20)),
			    0, 5, -90, 180);
  rarc10.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegArc rarc11(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(40, 20)),
			    0, 5, 180, 180);
  rarc11.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  
  fi.writeTo("testdraw_arc");

  fi.erase();
  fi.addFieldLines();
  fi.addGrid(4);
  rcss::clang::RegRec rrec1(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-20, -20)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-10, -10)));
  rrec1.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegRec rrec2(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0, 0)));
  rrec2.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegRec rrec3(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(20, 42)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(21, 30)));
  rrec3.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);

  fi.writeTo("testdraw_rec");

  fi.erase();
  fi.addFieldLines();
  fi.addGrid(4);

  rcss::clang::RegTri rtri1(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-20, -20)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-10, -20)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0, 0)));
  rtri1.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegTri rtri2(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(30, 30)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(35, 30)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(20, 10)));
  rtri2.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  rcss::clang::RegTri rtri3(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(20, -10)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0, 0)));
  rtri3.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);

  fi.writeTo("testdraw_tri");

  fi.erase();
  fi.addFieldLines();
  fi.addGrid(4);
  rcss::clang::RegUnion runi1;
  runi1.getRegions().push_front(new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-20, -20))));
  runi1.getRegions().push_front(new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall)));
  runi1.getRegions().push_front(new rcss::clang::RegPoint(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-20, 20))));
  runi1.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, varbind);
  fi.writeTo("testdraw_uni");

  /* SMURF: these are broken now that the thingys maintain links to their own referant
     I'll worry about it later 
  fi.erase();
  fi.addFieldLines();
  fi.addGrid(4);
  MsgStore<rcss::clang::Region*> msgstore;
  msgstore.add("bar",
	       new rcss::clang::RegArc(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(5, 25)),
				       5, 10, -50, 300));
  rcss::clang::RegNamed rnam1("foo");
  rnam1.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, &msgstore, varbind);
  rcss::clang::RegNamed rnam2("bar");
  rnam2.draw(&fi, verbose, FieldImage::COLOR_RED, color_fill, &wmi, &msgstore, varbind);
  fi.writeTo("testdraw_named");
  */
}


  
void
testGD()
{
  gdImagePtr im = gdImageCreateTrueColor(200, 200);
  int black;
  int red;
  int blue;
  
  /* Points of polygon */
  gdPoint points[3];
  im = gdImageCreate(100, 100);
  /* Background color (first allocated) */
  black = gdImageColorAllocate(im, 0, 0, 0);	
  red = gdImageColorAllocate(im, 255, 0, 0);	
  blue = gdImageColorAllocate(im, 0, 255, 0);	
  /* Draw a triangle. */
  points[0].x = 50;
  points[0].y = 0;
  points[1].x = 99;
  points[1].y = 99;
  points[2].x = 0;
  points[2].y = 99;
  gdImageFilledPolygon(im, points, 3, blue);
  gdImagePolygon(im, points, 3, red);

  FILE* out;
  out = fopen("testgd.png", "wb");
  gdImagePngEx(im, out, 9);
  
  /* Destroy it */
  gdImageDestroy(im);
}

void
testPlayerOccupancy()
{
  VarBindings varbind;
  FixedRWMI wmi;
  bool verbose = true;
  
  PlayerOccupancyElement poelem;
  std::ifstream infile("testin.poelem");

  infile >> poelem;
  if (infile.fail())
    errorlog << "Failed reading element" << ende;
  else
    cout << "Read element: " << poelem << endl;

  std::strstream sstr;
  sstr << poelem;
  PlayerOccupancyElement poelem2;
  sstr >> poelem2;

  cout << "Write/Read: " << poelem2 << endl;

  PlayerOccupancyElement poelem3(poelem);
  PlayerOccupancyElement poelem4;

  poelem4 = poelem;
  
  cout << "Copy construct: " << poelem3 << endl;
  cout << "Op=: " << poelem4 << endl;

  PlayerOccupancySet poset;
  POSetFileReader reader(&poset);
  std::ofstream outposet("testout.poset");
  reader.readFile("testin.poset");
  
  cout << poset;
  outposet << poset;

  FieldImage fi(3, 8);
  fi.addFieldLines();

  poset.draw(&fi, verbose, &wmi, varbind);

  fi.writeTo("testdraw_poset");
  
  PlayerOccupancyState postate(&poset);
  for (int idx = 0; idx < poset.getNumIdx(); idx++)
    {
      postate.setOverallIdx(idx);
      cout << idx << "\t" << postate << "\t" << postate.getOverallIdx() << endl;

      fi.erase();
      fi.addFieldLines();
      postate.draw(&fi, verbose, &wmi, varbind);
      std::ostringstream fn;
      fn << "testdraw_postate_" << idx;
      fi.writeTo(fn.str().c_str());
    }
}

void
testRegions()
{
  VarBindings varbind;
  FixedRWMI wmi;
  VecPosition v;

  rcss::clang::RegRec rrec1(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-20, -20)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(-10, -10)));
  rcss::clang::RegRec rrec2(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointBall),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0, 0)));
  rcss::clang::RegRec rrec3(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(20, 42)),
			    std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointRel(1, -12, std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(20, 42)))));

  v = VecPosition(-15, -15);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;
  v = VecPosition(5, 5);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;
  v = VecPosition(20.5, 35);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;
  v = VecPosition(15, 15);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;
  v = VecPosition(-5, -5);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;

  cout << "Switching sidedness" << endl;
  wmi.setUseRightSideCoords(true);
  
  v = VecPosition(-15, -15);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;
  v = VecPosition(5, 5);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;
  v = VecPosition(20.5, 35);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;
  v = VecPosition(15, 15);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;
  v = VecPosition(-5, -5);
  cout << v << " in " << rrec1 << ": " << rrec1.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec2 << ": " << rrec2.isPtIn(v, &wmi, varbind) << endl;
  cout << v << " in " << rrec3 << ": " << rrec3.isPtIn(v, &wmi, varbind) << endl;

#define	TEST_POLAR(r,th, reg) v = VecPosition::getVecPositionFromPolar(r, th); cout << "Vec(polar):" << r << ',' << th << ": " << reg << ": " << reg.isPtIn(v, &wmi, varbind) << endl;
  rcss::clang::RegArc rarc1(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    0, 10, 0, 360);
  rcss::clang::RegArc rarc2(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    0, 10, 120, 360);
  rcss::clang::RegArc rarc3(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    5, 15, 120, 360);
  rcss::clang::RegArc rarc4(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    0, 10, 120, 90);
  rcss::clang::RegArc rarc5(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    0, 10, 0, 130);
  rcss::clang::RegArc rarc6(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    0, 10, -50, 300);
  rcss::clang::RegArc rarc7(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    8, 10, 120, -90);
  rcss::clang::RegArc rarc8(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    5, 10, 0, -130);
  rcss::clang::RegArc rarc9(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(0,0)),
			    5, 10, -50, 300);

  cout << "Switching sidedness back to left" << endl;
  wmi.setUseRightSideCoords(false);
  
  TEST_POLAR(3, 90, rarc1); //1
  TEST_POLAR(3, 90, rarc2); //1
  TEST_POLAR(3, 359, rarc1); //1
  TEST_POLAR(3, 359, rarc2); //1
  TEST_POLAR(3, 90, rarc3); //0
  TEST_POLAR(9, 90, rarc3); //1
  TEST_POLAR(16, 90, rarc3); //0
  cout << endl;
  
  TEST_POLAR(5, 90, rarc4); //0
  TEST_POLAR(5, 125, rarc4); //1
  TEST_POLAR(5, 209, rarc4); //1
  TEST_POLAR(5, 211, rarc4); //0
  TEST_POLAR(5, 305, rarc4); //0
  cout << endl;

  TEST_POLAR(5, -55, rarc6); //0
  TEST_POLAR(5, -45, rarc6); //1
  TEST_POLAR(5, 0, rarc6); //1
  TEST_POLAR(5, 45, rarc6); //1
  TEST_POLAR(5, 255, rarc6); //0
  cout << endl;

  TEST_POLAR(9, 125, rarc7); //0
  TEST_POLAR(9, 115, rarc7); //1
  TEST_POLAR(9, 35, rarc7); //1
  TEST_POLAR(9, 25, rarc7); //0
  TEST_POLAR(9, 270, rarc7); //0
  cout << endl;

  /* pfr: this is all wrong now. The arc should not actually rotate because the
     isPtIn method expects a right hand coordinate point
  cout << "Switching sidedness to right" << endl;
  wmi.setUseRightSideCoords(true);

  TEST_POLAR(5, 270, rarc4); //0
  TEST_POLAR(5, 305, rarc4); //1
  TEST_POLAR(5, 389, rarc4); //1
  TEST_POLAR(5, 591, rarc4); //0
  TEST_POLAR(5, 485, rarc4); //0
  cout << endl;

  TEST_POLAR(5, 125, rarc6); //0
  TEST_POLAR(5, 135, rarc6); //1
  TEST_POLAR(5, 180, rarc6); //1
  TEST_POLAR(5, 225, rarc6); //1
  TEST_POLAR(5, 435, rarc6); //0
  cout << endl;

  TEST_POLAR(9, 305, rarc7); //0
  TEST_POLAR(9, 295, rarc7); //1
  TEST_POLAR(9, 215, rarc7); //1
  TEST_POLAR(9, 205, rarc7); //0
  TEST_POLAR(9, 450, rarc7); //0
  cout << endl;
  */
  
}

void
testBallGridFactor()
{
  BallGridFactor* pfactor = new BallGridFactor(10, 10);
  WorldState state;
  AbstractStateDescription desc(pfactor);
  AbstractState abs_state (&desc);
  
  for (double y = -ServerParam::instance()->getSPPitchWidth()/2;
       y < ServerParam::instance()->getSPPitchWidth()/2;
       y += 5)
    {
      for (double x = -ServerParam::instance()->getSPPitchLength()/2;
	   x < ServerParam::instance()->getSPPitchLength()/2;
	   x += 5)
	{
	  BallInfo ball(VecPosition(x, y), VecPosition(0,0));
	  state.setBall(ball);

	  pfactor->setStateFor(&abs_state, 0, state, TS_Left);
	  cout << setw(3) << abs_state.getFactorIdx(0);
	}
      cout << endl;
    }
}

void
testMarkovChain()
{
  MarkovChain chain(3);

  chain.addTransition(0, 1, 10);
  chain.addTransition(0, 2, 10);
  chain.addTransition(1, 0, 0.5);
  chain.addTransition(1, 2, 0.5);

  bool conn = chain.checkMinimalConnectivity(true);
  cout << "Connectivity (should be false): " << conn << endl;
  
  chain.addTransition(2, 0, 1);

  cout << "#### chain" << endl;
  cout << chain;
  cout << "TranCount: " << chain.getNumTransitions() << endl;
  
  std::strstream sstr;
  sstr << chain;
  MarkovChain chain2;
  sstr >> chain2;

  cout << "#### chain2: read/write test" << endl;
  cout << chain2;

  chain2.normalize();
  cout << "#### chain2: normalized" << endl;
  cout << chain2;
  cout << "TranCount: " << chain2.getNumTransitions() << endl;
  
  chain2 = chain;
  chain2.incrementTransition(1, 2, 1.0);
  chain2.incrementTransition(2, 2, 1.0);
  cout << "#### chain2: op=, then incr" << endl;
  cout << chain2;
  cout << "Connect: " << chain2.checkMinimalConnectivity(true) << endl;
  cout << "TranCount: " << chain2.getNumTransitions() << endl;
}

void
testMarkovChainReIdx()
{
  MarkovChain chain;

  ifstream inmc("test-reidx.mc");
  if (!inmc)
    errorlog << "Could not open test-reidx.mc" << ende;
  inmc >> chain;
  if (inmc.fail())
    errorlog << "Failed reading markov chain" << ende;
  inmc.close();

  chain.clearReIdx();
  chain.identifyLiveStates();
  cout << "Reindex for live states: " ;
  for (int i=0; i < chain.getNumStates(); i++)
    cout << chain.getReIdxStateVal(i) << " ";
  cout << endl;
}

void
testMarkovChainValue()
{
  MarkovChain chain;
  MarkovChainValue value(NULL, 0.9);

  ifstream inmcv("test.mcv");
  if (!inmcv)
    errorlog << "Could not open test.mcv" << ende;

  inmcv >> value;
  if (inmcv.fail())
    errorlog << "Failed reading markov chain value" << ende;
  inmcv.close();

  ifstream inmc("test.mc");
  if (!inmc)
    errorlog << "Could not open test.mc" << ende;

  inmc >> chain;
  if (inmc.fail())
    errorlog << "Failed reading markov chain" << ende;
  inmc.close();

  cout << chain;
  cout << value;

  std::strstream sstr;
  sstr << value;
  MarkovChainValue value2(&chain, 0.9);
  sstr >> value2;

  cout << "#### value2: read/write test" << endl;
  cout << value2;

  double delta;
  double per_cap_delta;
  value.setChain(&chain);

  for (int i = 0; i < 100; i++)
    {
      cout << "Value iterating: " << flush;
      value.valueIterateInPlace(1, &delta, &per_cap_delta);
      cout << endl << value << endl
	   << "Change: " << delta << ' ' << per_cap_delta << endl;
    }
}


void
testMarkovChainValueLearn()
{
  MarkovChain chain;
  MarkovChainValue value(NULL, 0.9);

  ifstream inmcv("test.mcv");
  if (!inmcv)
    errorlog << "Could not open test.mcv" << ende;

  inmcv >> value;
  if (inmcv.fail())
    errorlog << "Failed reading markov chain value" << ende;
  inmcv.close();

  ifstream inmc("test.mc");
  if (!inmc)
    errorlog << "Could not open test.mc" << ende;

  inmc >> chain;
  if (inmc.fail())
    errorlog << "Failed reading markov chain" << ende;
  inmc.close();

  value.setChain(&chain);

  cout << "Learning with iter limit 10" << endl;
  MarkovChainValue value2(value);
  if (!value2.learnValues(1, -1, -1, 10))
    errorlog << "Error in learning values" << ende;
  cout << value2;
  
  cout << "Learning with per cap limit 1e-2" << endl;
  MarkovChainValue value3(value);
  if (!value3.learnValues(1, -1, 1e-2, -1))
    errorlog << "Error in learning values" << ende;
  cout << value3;
  
  cout << "Learning with global limit 1e-3" << endl;
  MarkovChainValue value4(value);
  if (!value4.learnValues(1, 1e-3, -1, -1))
    errorlog << "Error in learning values" << ende;
  cout << value4;
  
}


void
performAbstractStateDescTests(AbstractStateDescription* pdesc)
{
  AbstractState state(pdesc);
  WorldState ws;
  
  cout << *pdesc << endl;
  cout << "NumStates: " << pdesc->getNumStates() << endl;

  if (!pdesc->getStateForWorldState(ws, TS_Left, &state))
    errorlog << "getState failed" << ende;

  cout << state << endl;
  cout << "Valid: " << state.isValid() << endl;
  cout << "Idx: " << state.getStateIdx() << endl;
}

void
testFactorOr()
{
  ConstantFactor* pfac1 = new ConstantFactor(4, 1);
  ConstantFactor* pfac2 = new ConstantFactor(4, 1);
  ConstantFactor* pfac3 = new ConstantFactor(4, 1);
  AbstractStateFactorOr *por = new AbstractStateFactorOr();
  por->addFactor(pfac1);
  por->addFactor(pfac2);
  por->addFactor(pfac3);

  AbstractStateDescription desc(por);

  performAbstractStateDescTests(&desc);

  pfac2->setConstantReturn(3);
  por->update();
  performAbstractStateDescTests(&desc);

  pfac2->setNumStates(10);
  por->update();
  performAbstractStateDescTests(&desc);

  pfac1->setConstantReturn(-1);
  por->update();
  performAbstractStateDescTests(&desc);

  pfac3->setConstantReturn(-1);
  por->update();
  performAbstractStateDescTests(&desc);

  pfac2->setConstantReturn(-1);
  pfac3->setConstantReturn(2);
  por->update();
  performAbstractStateDescTests(&desc);

}

void
testGraphWriter()
{
  // This tries to duplicate the simple.graph provided with walrus
  ofstream os("test.graph");
  if (!os)
    errorlog << "Could not open test.graph out file" << ende;
  LibSeaGraphWriter writer(os);

  writer.setVerbose(LibSeaGraphWriter::VL_All);
  
  writer.startGraph();

  writer.writeMetaData("Binary Tree",
		       "A complete binary tree of height 3 with some cross links.",
		       15, 17, 0, 0);
  //// Strutural Data
  writer.startStructuralData();
  
  writer.startLinks();
  writer.addLink(0, 1); 
  writer.addLink(0, 2); 
  writer.addLink(1, 3); 
  writer.addLink(1, 4); 
  writer.addLink(2, 5); 
  writer.addLink(2, 6); 
  writer.addLink(3, 7); 
  writer.addLink(3, 8); 
  writer.addLink(4, 9); 
  writer.addLink(4, 10); 
  writer.addLink(5, 11); 
  writer.addLink(5, 12); 
  writer.addLink(6, 13); 
  writer.addLink(6, 14); 
  writer.addLink(7, 0); 
  writer.addLink(14, 0); 
  writer.addLink(1, 2); 
  writer.endLinks();
  
  writer.startPaths();
  writer.endPaths();
  
  writer.endStructuralData();

  //// Attribute data
  writer.startAttributeData();
  writer.writeEnumerations();
  writer.startAttributeDefs();

  writer.startAttribute("$root", "bool", "|| false ||");
  writer.startNodeValues();
  writer.addAttrValue(0, "T");
  writer.endNodeValues();
  writer.startLinkValues();
  writer.endLinkValues();
  writer.startPathValues();
  writer.endPathValues();
  writer.endAttribute();

  writer.startAttribute("$tree_link", "bool", "|| false ||");
  writer.startNodeValues();
  writer.endNodeValues();
  writer.startLinkValues();
  writer.addAttrValue(0, "T");
  writer.addAttrValue(1, "T");
  writer.addAttrValue(2, "T");
  writer.addAttrValue(3, "T");
  writer.addAttrValue(4, "T");
  writer.addAttrValue(5, "T");
  writer.addAttrValue(6, "T");
  writer.addAttrValue(7, "T");
  writer.addAttrValue(8, "T");
  writer.addAttrValue(9, "T");
  writer.addAttrValue(10, "T");
  writer.addAttrValue(11, "T");
  writer.addAttrValue(12, "T");
  writer.addAttrValue(13, "T");
  writer.endLinkValues();
  writer.startPathValues();
  writer.endPathValues();
  writer.endAttribute();
  
  writer.endAttributeDefs();

  writer.startQualifiers();
  writer.writeSpanningTree("sample_spanning_tree", "EMPTY", 0, 1);
  writer.endQualifiers();
  
  writer.endAttributeData();

  writer.writeVisualizationHints();
  writer.writeInterfaceHints();
  
  writer.endGraph();
  
}

AbstractStateDescription*
getTestAbstractStateDescription()
{
  // This chunk of code was taken from ModAbstract
  AbstractStateFactorOr* pOr = new AbstractStateFactorOr();
  pOr->addFactor(new GoalFactor());

  AbstractStateFactorAnd* pDeadBall = new AbstractStateFactorAnd();
  pDeadBall->addFactor(new DeadBallFactor());
  pDeadBall->addFactor(new BallGridFactor(10, 6));
  pOr->addFactor(pDeadBall);
      
  AbstractStateFactorAnd* pAnd = new AbstractStateFactorAnd();
  pAnd->addFilter(new PlayModeFilter(PM_PlayOn));
  //pAnd->addFilter(new BallKickableFilter(pmFeatures));
  PlayerOccupancySet poset;
  //POSetFileReader reader(&poset);
  //reader.readFile(CoachParam::instance()->getAbstractStatePOSetFN().c_str());
  pAnd->addFactor(new POSetFactor(poset));
  pAnd->addFactor(new BallGridFactor(10, 6));
  //pAnd->addFactor(new BallOwnerFactor(pmFeatures));

  pOr->addFactor(pAnd);

  return new AbstractStateDescription(pOr);
}


void
tryASDiffMatch(ASDiffPatternElement* pelement, AbstractStateFactor* pfac, bool exp_res)
{
  int res = pelement->matchTo(pfac);
  if (res != exp_res)
    errorlog << "tryASDiffMatch: did not get expected result " << exp_res
	     << ": " << *pelement << "; " << *pfac << ende;
  if (res)
    cout << "Success: " << *pelement << endl;
  else
    cout << "No match: " << *pelement << endl;
}

void
testASDiffPattern()
{
  AbstractStateDescription* pdesc = getTestAbstractStateDescription();
  
  // pattern: OR --> Goal: TRUE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    p1->addChild(new ASDiffPatternSimple(ASF_Goal));
    tryASDiffMatch(p1, pdesc->getFactor(), true);
  }
     
  // pattern: OR --> Goal, DeadBall: FALSE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    p1->addChild(new ASDiffPatternSimple(ASF_Goal));
    p1->addChild(new ASDiffPatternSimple(ASF_DeadBall));
    tryASDiffMatch(p1, pdesc->getFactor(), false);
  }
  
  // pattern: OR --> DeadBall, Goal: FALSE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    p1->addChild(new ASDiffPatternSimple(ASF_DeadBall));
    p1->addChild(new ASDiffPatternSimple(ASF_Goal));
    tryASDiffMatch(p1, pdesc->getFactor(), false);
  }

  // pattern: AND --> Goal: FALSE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_And);
    p1->addChild(new ASDiffPatternSimple(ASF_Goal));
    tryASDiffMatch(p1, pdesc->getFactor(), false);
  }

  // pattern: OR --> AND --> DeadBall: TRUE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    ASDiffPatternChildren* p2 = new ASDiffPatternChildren(ASF_And);
    p2->addChild(new ASDiffPatternSimple(ASF_DeadBall));
    p1->addChild(p2);
    tryASDiffMatch(p1, pdesc->getFactor(), true);
  }
  
  // pattern: OR --> AND --> BallGrid: TRUE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    ASDiffPatternChildren* p2 = new ASDiffPatternChildren(ASF_And);
    p2->addChild(new ASDiffPatternSimple(ASF_BallGrid));
    p1->addChild(p2);
    tryASDiffMatch(p1, pdesc->getFactor(), true);
  }

  // pattern: OR --> AND --> POSet: TRUE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    ASDiffPatternChildren* p2 = new ASDiffPatternChildren(ASF_And);
    p2->addChild(new ASDiffPatternSimple(ASF_POSet));
    p1->addChild(p2);
    tryASDiffMatch(p1, pdesc->getFactor(), true);
  }

  // pattern: OR --> Goal, AND --> POSet: TRUE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    ASDiffPatternChildren* p2 = new ASDiffPatternChildren(ASF_And);
    p2->addChild(new ASDiffPatternSimple(ASF_POSet));
    p1->addChild(new ASDiffPatternSimple(ASF_Goal));
    p1->addChild(p2);
    tryASDiffMatch(p1, pdesc->getFactor(), true);
  }

  // pattern: OR --> Goal, AND --> POSet, DeadBall: FALSE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    ASDiffPatternChildren* p2 = new ASDiffPatternChildren(ASF_And);
    p2->addChild(new ASDiffPatternSimple(ASF_POSet));
    p2->addChild(new ASDiffPatternSimple(ASF_DeadBall));
    p1->addChild(new ASDiffPatternSimple(ASF_Goal));
    p1->addChild(p2);
    tryASDiffMatch(p1, pdesc->getFactor(), false);
  }

  // pattern: OR --> Goal, AND --> DeadBall, AND --> POSet: FALSE
  {
    ASDiffPatternChildren* p1 = new ASDiffPatternChildren(ASF_Or);
    ASDiffPatternChildren* p2 = new ASDiffPatternChildren(ASF_And);
    ASDiffPatternChildren* p3 = new ASDiffPatternChildren(ASF_And);
    p1->addChild(new ASDiffPatternSimple(ASF_Goal));
    p1->addChild(p2);
    p1->addChild(p3);
    p2->addChild(new ASDiffPatternSimple(ASF_DeadBall));
    p2->addChild(new ASDiffPatternSimple(ASF_POSet));
    tryASDiffMatch(p1, pdesc->getFactor(), false);
  }
  
}


void
tryFeatureMatch(AbstractStateDescription* pdesc,
		int first_state_idx,
		int second_state_idx,
		ASDiffFeature* pfeature,
		bool exp_res)
{
  AbstractState first_state(pdesc);
  AbstractState second_state(pdesc);

  first_state.setStateIdx(first_state_idx);
  second_state.setStateIdx(second_state_idx);
  
  AbstractStateCompare comp(&first_state, &second_state);

  actionlog(50) << "test: trying to match state " << first_state_idx
		<< " to " << second_state_idx << " for feature: "
		<< *pfeature << ende;
  actionlog(50) << "test: states are " << first_state << "; " << second_state << ende;
  cout << "test: trying to match state " << first_state_idx
       << " to " << second_state_idx << " for feature: "
       << *pfeature << endl;
  cout << "test: states are " << first_state << "; " << second_state << endl;

  bool res = pfeature->match(&comp);
  actionlog(60) << "Matched: " << res << ende;
  cout << "\tMatched: " << res << endl;
  if (res != exp_res)
    errorlog << "This match failed!" << ende;
}


void
testASDiffFeature()
{
  AbstractStateDescription* pdesc = getTestAbstractStateDescription();
  
  ASDiffFeature* pFeature;
  std::istringstream istr("MyGoal");
  pFeature = ASDiffFeature::createFromStream(istr);
  if (pFeature == NULL)
  {
    errorlog << "Did not create feature" << ende;
    return;
  }

  tryFeatureMatch(pdesc, 3, 0, pFeature, true);
  tryFeatureMatch(pdesc, 3, 1, pFeature, false);
  tryFeatureMatch(pdesc, 22, 0, pFeature, true);
  tryFeatureMatch(pdesc, 22, 1, pFeature, false);
  tryFeatureMatch(pdesc, 12, 88, pFeature, false);
  tryFeatureMatch(pdesc, 12, 12, pFeature, false);
  tryFeatureMatch(pdesc, 12, 44, pFeature, false);
  
}

void
tryASDiffClassify(AbstractStateDescription* pdesc,
		  int first_state_idx, int second_state_idx,
		  ASDiffClassifierSet* pset)
{
  AbstractState first_state(pdesc);
  AbstractState second_state(pdesc);

  first_state.setStateIdx(first_state_idx);
  second_state.setStateIdx(second_state_idx);
  
  AbstractStateCompare comp(&first_state, &second_state);

  int res = pset->classify(&comp);

  cout << "State " << first_state_idx << " to " << second_state_idx
       << ": " << res << " "
       << ((res < 0) ? "NONE" : pset->getClassifier(res)->getName())
       << endl;
}


void
testASDiffClassifier()
{
  ASDiffClassifierSet cset;
  ASDCSetFileReader reader(&cset);
  reader.readFile("test.cset");

  cout << cset;

  AbstractStateDescription* pdesc = getTestAbstractStateDescription();

  tryASDiffClassify(pdesc, 3, 0, &cset);
  tryASDiffClassify(pdesc, 3, 1, &cset);
  tryASDiffClassify(pdesc, 22, 0, &cset);
  tryASDiffClassify(pdesc, 22, 1, &cset);
  tryASDiffClassify(pdesc, 12, 88, &cset);
  tryASDiffClassify(pdesc, 12, 12, &cset);
  tryASDiffClassify(pdesc, 12, 44, &cset);
}


void
testMDP()
{
  SoccerMDP mdp;
  std::ifstream is("test.mdp");
  if (!is)
    errorlog << "Could not open test.mdp" << ende;
  is >> mdp;
  if (is.fail())
    errorlog << "Error reading MDP: " << is.rdstate() << ende;
  std::cout << "Basic MDP:" << endl;
  std::cout << mdp;

  std::stringstream sstr;
  sstr << mdp;
  SoccerMDP mdp2;
  sstr >> mdp2;

  cout << "#### mdp2: read/write test" << endl;
  cout << mdp2;
}

void
testTransitionSorter()
{
  TransitionSorter sorter;
  std::ifstream is("basic_ball.tsort");
  if (!is)
    errorlog << "Could not open basic_ball.tsort" << ende;
  is >> sorter;
  if (is.fail())
    errorlog << "Error reading sorter: " << is.rdstate() << ende;
  std::cout << "Transition Sorter:" << endl;
  std::cout << sorter;

  std::stringstream sstr;
  sstr << sorter;
  TransitionSorter sorter2;
  sstr >> sorter2;

  cout << "#### sorter2: read/write test" << endl;
  if (sstr.fail())
    cout << "## Reading failed!" << endl;
  cout << sorter2;
}

void
testQTable()
{
  SoccerMDP mdp;
  std::ifstream is("test.mdp");
  if (!is)
    errorlog << "Could not open test.mdp" << ende;
  is >> mdp;
  if (is.fail())
    errorlog << "Error reading MDP: " << is.rdstate() << ende;

  QTableFlex qt(mdp, 0.9);
  cout << "## Empty QT" << endl << qt;
  
  qt.observedUpdate(0, 1, 1, 10);
  qt.observedUpdate(1, 0, 2, 0);
  qt.observedUpdate(2, 0, 0, 0);

  cout << "## After a couple updates" << endl << qt;

  std::stringstream sstr;
  sstr << qt;
  QTableFlex qt2(mdp, 0.9);
  sstr >> qt2;

  cout << "#### qt2: read/write test" << endl;
  if (sstr.fail())
    cout << "## Reading failed!" << endl;
  cout << qt2 << endl;

  int max_act, min_act;
  for (int i=0; i<=2; i++)
    cout << i << ": "
	 << "V=" << qt2.getV(i, &max_act) << "(act=" << max_act << ")\t"
	 << "worstQ=" << qt2.getWorstQ(i, &min_act) << "(act=" << min_act << ")"
	 << endl;
}

void
testRectHelper(BallGridFactor* pfactor, int idx, int idx1, int idx2, bool exp_res)
{
  bool res = pfactor->isLocBetween(idx, idx1, idx2);

  cout << "idx " << idx << " between " << idx1 << " and " << idx2 << endl;
  cout << pfactor->getGridRectangle(idx) << " between "
       << pfactor->getGridRectangle(idx1) << pfactor->getGridRectangle(idx2)
       << ": " << res << endl;
  if (res != exp_res)
    errorlog << "******************* NOT WHAT I EXPECTED *******************" << ende;
}

void
testRectBetween()
{
  BallGridFactor* pfactor = new BallGridFactor(10, 10);

  // simple horix and vertical
  testRectHelper(pfactor, 1, 0, 2, true);
  testRectHelper(pfactor, 0, 1, 3, false);
  testRectHelper(pfactor, 4, 1, 3, false);
  testRectHelper(pfactor, 5, 0, 2, false);
  testRectHelper(pfactor, 5, 0, 9, true);
  testRectHelper(pfactor, 10, 0, 90, true);
  testRectHelper(pfactor, 90, 0, 50, false);
  testRectHelper(pfactor, 20, 0, 90, true);

  // origin points
  testRectHelper(pfactor, 10, 10, 90, true);
  testRectHelper(pfactor, 90, 10, 90, true);

  //random other stuff
  testRectHelper(pfactor, 49, 0, 99, false);
  testRectHelper(pfactor, 45, 0, 99, true);
  testRectHelper(pfactor, 1, 0, 91, true);
  testRectHelper(pfactor, 2, 0, 91, false);
  testRectHelper(pfactor, 2, 0, 95, false);
  testRectHelper(pfactor, 42, 0, 95, true);
}

void
testStateValue()
{
  StateValue<double> sval(5);

  sval.setValue(0, 5.0);
  sval.setValue(1, 10.0);
  sval.setValue(2, -5.0);
  sval.setValue(3, 8.0);
  sval.setValue(4, -100.0);

  cout << "# First time" << endl;
  cout << sval << endl;
  
  std::stringstream sstr;
  sstr << sval;
  StateValue<double> sval2;
  sstr >> sval2;

  cout << "#### sval2: read/write test" << endl;
  if (sstr.fail())
    cout << "## Reading failed!" << endl;
  cout << sval2 << endl;
  
}

void
testAdviceTree()
{
  AdviceTree tree;

  tree.addLevel(AdviceTree::LevelInfo(new GoalFactor, 0));
  tree.addLevel(AdviceTree::LevelInfo(new BallGridFactor(2,2), 1));

  cout << "## Tree before creation" << std::endl;
  cout << tree;

  tree.createTreeFromLevels();
  cout << "## Tree after creation" << std::endl;
  cout << tree;
}

void
testAdviceTreeCreation()
{
  AbstractStateFactorAnd* pDeadBall = new AbstractStateFactorAnd();
  pDeadBall->addFactor(new DeadBallFactor());
  pDeadBall->addFactor(new BallGridFactor(2, 6));

  AbstractStateDescription* pdesc = new AbstractStateDescription(pDeadBall);

  AdviceTree* ptree = pdesc->createAdviceTree();

  cout << "## Tree created from description" << std::endl;
  if (ptree == NULL)
    cout << "READING FAILED" << std::endl;
  else
    cout << *ptree;

  AbstractState state(pdesc);

  state.setFactorIdx(0, 0);
  state.setFactorIdx(1, 0);
  ptree->addAction(&state, new AdviceTreeAction);
  state.setFactorIdx(0, 0);
  state.setFactorIdx(1, 3);
  ptree->addAction(&state, new AdviceTreeAction);
  state.setFactorIdx(0, 1);
  state.setFactorIdx(1, 1);
  ptree->addAction(&state, new AdviceTreeAction);

  cout << "## Tree after adding some actions" << endl;
  cout << *ptree;

  cout << "## Now adding with logging going to std::cout" << endl;

  CoachMessageQueue queue("Tester");
  ptree->createAdvice(&queue, "A", &std::cout);

  cout << "## Message queue: " << endl;
  cout << queue;
  
  delete ptree;
}

void
testCircleIntersect()
{
  VecPosition sol1, sol2;
  int num_sol;
  Line off_line;
  
  off_line.setVertical(20.1024);
  num_sol = off_line.getCircleIntersectionPoints(Circle(VecPosition( 18.0311, -24 ), 8),
						 &sol1, &sol2);

  std::cout << "testCircleIntersect: " << num_sol
	    << " " << sol1 
	    << " " << sol1 
	    << std::endl;
}

void
testCRIHelper(const Circle& c, const Rectangle& r, int exp_num)
{
  static int call_count = 0;

  FieldImage fi(2, 6);
  fi.addFieldLines();
  std::ostringstream os;
  os << c << "\t" << r << std::ends;
  fi.addLegendLine(os.str().c_str());
  fi.addCircle(c, FieldImage::COLOR_GREEN, FieldImage::COLOR_CLEAR);
  fi.addRectangle(r, FieldImage::COLOR_BLUE, FieldImage::COLOR_CLEAR);
  std::ostringstream os_name;
  os_name << "testcri_" << call_count << ends;
  if (!fi.writeTo(os_name.str().c_str()))
    errorlog << "testCRIHelper: write failed" << ende;
  
  std::cout << c << "\t" << r << std::endl;
  std::vector<VecPosition> vsol = r.circleIntersect(c);
  if (vsol.size() != (unsigned)exp_num)
    std::cout << "\tERROR";
  std::cout << "\tSol: " << vsol.size() << "\t";
  std::copy(vsol.begin(), vsol.end(), std::ostream_iterator<VecPosition>(std::cout, " "));
  std::cout << std::endl;

  call_count++;
}

void
testCircleRectIntersect()
{
  testCRIHelper(Circle(VecPosition(0,0), 5),
		Rectangle(VecPosition(-10, -10),
			  VecPosition(10, 10)),
		0);
  testCRIHelper(Circle(VecPosition(9,9), 5),
		Rectangle(VecPosition(-10, -10),
			  VecPosition(10, 10)),
		2);
  testCRIHelper(Circle(VecPosition(5,0), 11),
		Rectangle(VecPosition(-10, -10),
			  VecPosition(10, 10)),
		6);
  testCRIHelper(Circle(VecPosition(12,0), 11),
		Rectangle(VecPosition(-10, -10),
			  VecPosition(10, 10)),
		2);
  testCRIHelper(Circle(VecPosition(0,0), 13),
		Rectangle(VecPosition(-10, -10),
			  VecPosition(10, 10)),
		8);
}


void
testBinaryFile()
{
  std::cout << "Testing binary file handling" << std::endl;
  
  const char* CONST_MAGIC = "PFRTEST";
  const char CONST_CHAR = 124;
  const short CONST_SHORT = 1111;
  const int CONST_INT = 322222;
  const long CONST_LONG = 7777777;
  const float CONST_FLOAT = 12345e10;
  const double CONST_DOUBLE = 9.87654e-4;
  const char* CONST_STR = "Hello World";

  const char* FILENAME = "/tmp/pfrtest.bin";
  
  {
    BinaryFileWriter writer(FILENAME);
    writer.writeMagicHeader(CONST_MAGIC);
    writer.writeChar(CONST_CHAR);
    writer.writeShort(CONST_SHORT);
    writer.writeInt(CONST_INT);
    writer.writeLong(CONST_LONG);
    writer.writeFloat(CONST_FLOAT);
    writer.writeDouble(CONST_DOUBLE);
    // we write it twice to test both input methods
    writer.writeString(CONST_STR);
    writer.writeString(CONST_STR);
  }
  
  {
    BinaryFileReader reader(FILENAME);

    char c;
    short s;
    int i;
    long l;
    float f;
    double d;
    char cstr[100];
    std::string str;

    reader.checkMagicHeader(CONST_MAGIC);
    
    if (!(reader.readChar(&c) &&
	  reader.readShort(&s) &&
	  reader.readInt(&i) &&
	  reader.readLong(&l) &&
	  reader.readFloat(&f) &&
	  reader.readDouble(&d) &&
	  reader.readString(cstr, 100) &&
	  reader.readString(&str)))
      errorlog << "Some reading failed" << ende;

    if (c != CONST_CHAR)
      errorlog << "char failed " << c << ' ' << CONST_CHAR << ende;
    if (s != CONST_SHORT)
      errorlog << "short failed " << s << ' ' << CONST_SHORT << ende;
    if (i != CONST_INT)
      errorlog << "int failed " << i << ' ' << CONST_INT << ende;
    if (l != CONST_LONG)
      errorlog << "long failed " << l << ' ' << CONST_LONG << ende;
    if (f != CONST_FLOAT)
      errorlog << "float failed " << f << ' ' << CONST_FLOAT << ende;
    if (d != CONST_DOUBLE)
      errorlog << "double failed " << d << ' ' << CONST_DOUBLE << ende;
    if (strcmp(cstr, CONST_STR) != 0)
      errorlog << "cstr failed " << cstr << ' ' << CONST_STR << ende;
    if (strcmp(str.c_str(), CONST_STR) != 0)
      errorlog << "str failed " << str << ' ' << CONST_STR << ende;
  }

  std::cout << "Testing complete. No errors above then it's ok" << std::endl;
}

void
testBinQTable()
{
  std::cout << "Testing binary read/write of QTable" << std::endl;

  struct timeval tv_start, tv_end;
  struct timezone tz;
  QTableFlex qtorig(0, 0.9);
  std::ifstream inorig("mc_convert.qtable");
  if (!inorig)
    errorlog << "Failed to open initial qtable file" << ende;

  const char* TMP_FILENAME = "/tmp/pfrtest.binqt" ;
  gettimeofday(&tv_start, &tz);
  inorig >> qtorig;
  gettimeofday(&tv_end, &tz);
  if (inorig.fail())
    errorlog << "Failed to load initial qtable" << ende;

  std::cout << "Info: time to read ascii version: " << tv2seconds(tv_end - tv_start) << std::endl;
  std::cout << "Info: original qtable has " << qtorig.getNumStates() << " states" << std::endl;
    
  {
    BinaryFileWriter writer(TMP_FILENAME);
    if (!qtorig.writeTo(writer))
      errorlog << "Failed writing qtorig to bin" << ende;
  }

  QTableFlex qtnew(0, 0.9);
  
  {
    BinaryFileReader reader(TMP_FILENAME);
    gettimeofday(&tv_start, &tz);
    if (!qtnew.readFrom(reader))
      errorlog << "Failed reading binary file back in" << ende;
    gettimeofday(&tv_end, &tz);
  }	

  std::cout << "Info: time to read bin version: " << tv2seconds(tv_end - tv_start) << std::endl;
  std::cout << "Info: reread qtable has " << qtnew.getNumStates() << " states" << std::endl;

  if (!qtorig.isEqualTo(qtnew, &std::cerr))
    errorlog << "Reread qable not equal to original" << ende;

  std::cout << "Testing the bin or text read..." << std::endl;

  QTableFlex qt(0, 0.9);
  std::ifstream in_text("mc_convert.qtable");
  if (!in_text)
    errorlog << "Could not open text file" << ende;
  
  if (!qt.readTextOrBinary(in_text))
    errorlog << "Failed reading text in readTextOrBinary" << ende;

  if (!qtorig.isEqualTo(qt, &std::cerr))
    errorlog << "Text read qtable not equal to original" << ende;
  else
    std::cout << "Text read successful" << std::endl;

  in_text.close();
  
  std::ifstream in_bin(TMP_FILENAME);
  if (!in_bin)
    errorlog << "Could not open bin file" << ende;
  
  if (!qt.readTextOrBinary(in_bin))
    errorlog << "Failed reading bin in readTextOrBinary" << ende;

  if (!qtorig.isEqualTo(qt, &std::cerr))
    errorlog << "Bin read qtable not equal to original" << ende;
  else
    std::cout << "Bin read successful" << std::endl;

  in_bin.close();
  
  
  std::cout << "Testing complete. No errors above then it's ok" << std::endl;
}


void
testPOElemHelper(PlayerOccupancySet& poset, WorldState& ws, int exp_val)
{
  PlayerOccupancyState* pstate =
    poset.getPOStateForWorldState(ws, TS_Left, NULL);

  int val = pstate->getOverallIdx();

  if (val == exp_val)
    {
      std::cout << "correct " << val << ": " << ws << std::endl;
    }
  else
    {
      std::cout << "ERROR: got " << val << ", exp " << exp_val << ": " << ws << std::endl;
    }
  
  delete pstate;
}

void
testPOElem()
{
  PlayerOccupancySet poset;
  POSetFileReader reader(&poset);

  reader.readFile("BallPathFor5.poset");

  WorldState ws;

  ws.setTime(9999);
  ws.setPlayMode(PM_PlayOn);
  ws.setBall(BallInfo(VecPosition(0,0), VecPosition(0,0)));

  testPOElemHelper(poset, ws, 0);
  
  ws.setPlayer(PlayerInfo(TS_Left, 1,
			  VecPosition(0, -5), VecPosition(0,0),
			  VecPosition(0,0), VecPosition(0,0),
			  0, 0));

  testPOElemHelper(poset, ws, 1);

  ws.setPlayer(PlayerInfo(TS_Left, 2,
			  VecPosition(0, 5), VecPosition(0,0),
			  VecPosition(0,0), VecPosition(0,0),
			  0, 0));

  testPOElemHelper(poset, ws, 3);

  ws.setPlayer(PlayerInfo(TS_Right, 1,
			  VecPosition(0, -6), VecPosition(0,0),
			  VecPosition(0,0), VecPosition(0,0),
			  0, 0));

  testPOElemHelper(poset, ws, 2);

  ws.setPlayer(PlayerInfo(TS_Right, 2,
			  VecPosition(0, 3), VecPosition(0,0),
			  VecPosition(0,0), VecPosition(0,0),
			  0, 0));

  testPOElemHelper(poset, ws, 0);

}

void
testBinMDP()
{
  std::cout << "Testing binary read/write of MDP" << std::endl;

  struct timeval tv_start, tv_end;
  struct timezone tz;
  SoccerMDP mdporig;
  std::ifstream inorig("mc_convert.mdp");
  if (!inorig)
    errorlog << "Failed to open initial qtable file" << ende;

  const char* TMP_FILENAME = "/tmp/pfrtest.binmdp" ;
  gettimeofday(&tv_start, &tz);
  inorig >> mdporig;
  gettimeofday(&tv_end, &tz);
  if (inorig.fail())
    errorlog << "Failed to load initial mdp" << ende;

  std::cout << "Info: time to read text version: " << tv2seconds(tv_end - tv_start) << std::endl;
  std::cout << "Info: original mdp has " << mdporig.getNumStates() << " states" << std::endl;
    
  {
    BinaryFileWriter writer(TMP_FILENAME);
    if (!mdporig.writeTo(writer))
      errorlog << "Failed writing mdporig to bin" << ende;
  }

  SoccerMDP mdpnew;
  
  {
    BinaryFileReader reader(TMP_FILENAME);
    gettimeofday(&tv_start, &tz);
    if (!mdpnew.readFrom(reader))
      errorlog << "Failed reading binary file back in" << ende;
    gettimeofday(&tv_end, &tz);
  }	

  std::cout << "Info: time to read bin version: " << tv2seconds(tv_end - tv_start) << std::endl;
  std::cout << "Info: reread mdp has " << mdpnew.getNumStates() << " states" << std::endl;

  errorlog << "Not yet checking equality" << ende;
  /*
  if (!mdporig.isEqualTo(mdpnew, &std::cerr))
    errorlog << "Reread qable not equal to original" << ende;
  */

  SoccerMDP mdp;
  std::ifstream in_text("mc_convert.mdp");
  if (!in_text)
    errorlog << "Could not open text file" << ende;
  
  if (!mdp.readTextOrBinary(in_text))
    errorlog << "Failed reading text in readTextOrBinary" << ende;

  //if (!mdporig.isEqualTo(mdp, &std::cerr))
  if (mdporig.getNumStates() != mdp.getNumStates())
    errorlog << "Text read mdp not equal to original" << ende;
  else
    std::cout << "Text read successful" << std::endl;

  in_text.close();
  
  std::ifstream in_bin(TMP_FILENAME);
  if (!in_bin)
    errorlog << "Could not open bin file" << ende;
  
  if (!mdp.readTextOrBinary(in_bin))
    errorlog << "Failed reading bin in readTextOrBinary" << ende;

  //if (!mdporig.isEqualTo(mdp, &std::cerr))
  if (mdporig.getNumStates() != mdp.getNumStates())
    errorlog << "Bin read mdp not equal to original" << ende;
  else
    std::cout << "Bin read successful" << std::endl;

  in_bin.close();

  
  std::cout << "Testing complete. No errors above then it's ok" << std::endl;
}

void
testTypeInfo()
{
  #define TESTTYPE(X) { X val; std::cout << "Type " << #X << ": " << typeid(val).name() << std::endl; }

  TESTTYPE(int);
  TESTTYPE(short);
  TESTTYPE(double);
}

void
testIntBucketHelper(std::vector<int>& vals)
{
  IntBucket bucket;

  std::cout << "vals: ";
  
  for (std::vector<int>::iterator iter = vals.begin();
       iter != vals.end();
       ++iter)
    {
      std::cout << *iter << ' ';
      bucket.addPoint(*iter);
    }
  std::cout << std::endl;

  std::cout << "Mean(0) = " << bucket.getMean(0) << std::endl;
  std::cout << "Mean(1) = " << bucket.getMean(1) << std::endl;
  std::cout << "Mean(2) = " << bucket.getMean(2) << std::endl;
  std::cout << bucket;
}

void
testIntBucket()
{
  std::cout << "Testing IntBucket" << std::endl;

  std::vector<int> vals;

  vals.clear();
  vals.push_back(2);
  vals.push_back(4);
  testIntBucketHelper(vals);

  vals.push_back(1);
  vals.push_back(1);
  testIntBucketHelper(vals);

  vals.push_back(0);
  testIntBucketHelper(vals);

  vals.push_back(4);
  vals.push_back(4);
  testIntBucketHelper(vals);
}


//This is a funny test thing. I used it to generate stuff for my thesis
void
testDrawMisc()
{
  FieldImage fi(0, 4);
  FixedRWMI wmi;
  VarBindings bindings;
  
  fi.addFieldLines();

  if (0)
    {
      std::auto_ptr<rcss::clang::Point> pt(new rcss::clang::PointSimple(-25, -10));
      rcss::clang::RegArc reg(pt, 0, 5, 0, 360);
      
      reg.draw(&fi, false,
               FieldImage::COLOR_BLACK,
               FieldImage::COLOR_GREY,
               &wmi, bindings);
    }

  if (0)
    {
      VecPosition pos;
      pos = VecPosition(36.5 -21.16);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 2);
      fi.addText("4", pos + VecPosition(2, 0));
      pos = VecPosition(35, 8);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 2);
      fi.addText("10", pos + VecPosition(2, 0));
      pos = VecPosition(35, -9.16);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 2);
      fi.addText("11", pos + VecPosition(2, 0));
    }

  if (0)
    {
      VecPosition pos;
      VecPosition disp(2, 0);
      VecPosition ballpos(0,0);
      
      pos = VecPosition(-37,-10.84);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 3);
      fi.addText("2", pos + disp);
      pos = VecPosition(-38,-6);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 3);
      fi.addText("5", pos + disp);
      pos = ballpos;
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 3);
      fi.addText("6", pos + disp);
      pos = VecPosition(-38, 6);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 3);
      fi.addText("7", pos + disp);
      pos = VecPosition(-46,-8.84);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 3);
      fi.addText("8", pos + disp);
      pos = VecPosition(-44.5,-0);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 3);
      fi.addText("9", pos + disp);
      pos = VecPosition(14.1916,-10.84);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 3);
      fi.addText("11", pos + disp);

      fi.addArrow(ballpos, VecPosition(-46, -8.84));
    }

  if (1)
    {
      VecPosition pos;
      VecPosition disp(2, 0);
      VecPosition ballpos(0,0);
      
      pos = VecPosition(0,0);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 2);
      fi.addText("4", pos + disp);
      pos = VecPosition(-5,7);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 2);
      fi.addText("9", pos - disp);
      pos = VecPosition(-1,30.999);
      fi.addPoint(pos, FieldImage::COLOR_GREEN, 2);
      fi.addText("10", pos + disp);

      fi.addArrow(ballpos, VecPosition(-5, 7), FieldImage::COLOR_RED, 2);
    }

  //(definer "BG100658" (rec (pt 31.8 23) (pt 42.4 34.5)))
  //(do our {0} (pass (rec (pt 36 -20.16) (pt 52.5 20.16)))) 
  if (0)
    {
      rcss::clang::RegRec reg1(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(31.8, 23)),
                               std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(42.4, 34.5)));

      reg1.draw(&fi, false,
                FieldImage::COLOR_BLACK,
                FieldImage::COLOR_GREY,
                &wmi, bindings);
      fi.addText("BG100658", VecPosition(16, 24));
      
      rcss::clang::RegRec reg2(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(36, -20.16)),
                               std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(52.5, 20.16)));

      reg2.draw(&fi, false,
               FieldImage::COLOR_BLACK,
               FieldImage::COLOR_GREY,
               &wmi, bindings);
      fi.addText("A0000158", VecPosition(30, -25));
    }

  //(definer "PR4" (arc (pt ball) 1.5 15.05 15.1 52.52)) 
  if (0)
    {
      rcss::clang::RegArc reg1(std::auto_ptr<rcss::clang::Point>(new rcss::clang::PointSimple(15, 0)),
                               1.5, 15.05, 15.1, 52.52);

      reg1.draw(&fi, false,
                FieldImage::COLOR_BLACK,
                FieldImage::COLOR_GREY,
                &wmi, bindings);
    }
  

  fi.writeTo("drawmisc");
}

