/* -*- Mode: C++ -*- */

/* Misc utility types, classes, and functions */

#include "soccer_utils.h"
#include "misc.h"
#include "CoachParam.h"
#include "Logger.h"

using namespace spades;

/*************************************************************************/
static const char* TEAM_SIDE_STRINGS[] = {"None", "Left", "Right", "Both"};

std::ostream& operator<< (std::ostream & o, const TeamSide& ts)
{
  o << TEAM_SIDE_STRINGS[(int)ts];
  return o;
}

std::istream &
operator >> (std::istream & is, TeamSide& ts)
{
  std::string str;
  is >> str;
  if (is.fail())
    return is;
  for (int i=0; i<4; i++)
    {
      if (strcasecmp(str.c_str(),TEAM_SIDE_STRINGS[i]) == 0)
	{
	  ts = (TeamSide)i;
	  return is;
	}
    }
  is.setstate(std::ios::failbit);
  return is;
}

static const char* RELATIVE_TEAM_SIDE_STRINGS[] = {"None", "Mine", "Theirs"};

std::ostream&
operator<< (std::ostream & o, const RelativeTeamSide& rts)
{

  o << RELATIVE_TEAM_SIDE_STRINGS[(int)rts];
  return o;
}

std::istream&
operator >> (std::istream & is, RelativeTeamSide& ts)
{
  std::string str;
  is >> str;
  if (is.fail())
    return is;
  for (int i=0; i<4; i++)
    {
      if (strcasecmp(str.c_str(), RELATIVE_TEAM_SIDE_STRINGS[i]) == 0)
	{
	  ts = (RelativeTeamSide)i;
	  return is;
	}
    }
  is.setstate(std::ios::failbit);
  return is;
}



/*************************************************************************/
TeamSide
relativeTeamSideToAbs( RelativeTeamSide rts, TeamSide myside)
{
  switch (rts)
    {
    case RTS_None:
      return TS_None;
    case RTS_Mine:
      return myside;
    case RTS_Theirs:
      switch (myside)
	{
	case TS_Left: return TS_Right;
	case TS_Right: return TS_Left;
	default:
	  errorlog << "What is myside in relative->abs? " << myside << ende;
	}
      break;
    default:
      errorlog << "What is the relative side in relative->abs? " << rts << ende;
      break;
    }
  return TS_None;
}

RelativeTeamSide
absTeamSideToRelative( TeamSide ts, TeamSide myside)
{
  switch (ts)
    {
    case TS_None:
      return RTS_None;

    case TS_Left:
      switch (myside)
	{
	case TS_Left: return RTS_Mine;
	case TS_Right: return RTS_Theirs;
	default:
	  errorlog << "What is myside in abs->relative? " << myside << ende;
	}
      break;
      
    case TS_Right:
      switch (myside)
	{
	case TS_Left: return RTS_Theirs;
	case TS_Right: return RTS_Mine;
	default:
	  errorlog << "What is myside in abs->relative? " << myside << ende;
	}
      break;

    default:    
      errorlog << "What is the side in abs->relative? " << ts << ende;
    }
  return RTS_None;
}


/*************************************************************************/

VecPosition
getGoalPosForSide(TeamSide ts)
{
  switch(ts)
    {
    case TS_Left:
      return VecPosition(ServerParam::instance()->getSPFieldRectangle().getPosLeftTop().getX(), 0.0);
      break;
    case TS_Right:
      return VecPosition(ServerParam::instance()->getSPFieldRectangle().getPosRightBottom().getX(), 0.0);
      break;
    default:
      errorlog << "getGoalPosForSide: invalid side " << ts << ende;
      break;
    }
  return VecPosition(0,0);
}

/*************************************************************************/

TeamSide
getOppositeSide(TeamSide ts)
{
  switch (ts)
    {
    case TS_None: return TS_Both;
    case TS_Left: return TS_Right;
    case TS_Right: return TS_Left;
    case TS_Both: return TS_None;
    default:
      errorlog << "getOppositeSide: What is team side? " << ts << ende;
    }
  return TS_None;
}


/*************************************************************************/

/* this assumes smaller x is towards our goal */
bool inferRolesFromRegion(const std::vector<rcss::clang::Region*>& vReg,
			  std::vector<PlayerRole>& vRole,
			  int goalie_num)
{
#ifdef TO_BE_UPDATED
  //we will work with 0 based stuff all through here
  int *aNums = new int[Mem->SP_team_size];
  float *aXPos = new float[Mem->SP_team_size];
  int num_elems=0;
  for (int num=0; num<Mem->SP_team_size; num++) {
    if (num+1 == goalie_num) {
      aRole[num] = PT_Goaltender;
      continue;
    }
    if (!aReg[num]) {
      aRole[num] = PT_None;
      continue;
    }
    aNums[num_elems] = num;
    aXPos[num_elems] = aReg[num]->getSomePointIn().x;
    num_elems++;
  }

  BubbleSort(num_elems, aNums, aXPos);

  /* the split value says that everything < is before and >= is after */
  /* we split into defender, midfielder, and forward */
  /* there must be at least 1 defender and 1 forward */
  int best_split1=-1, best_split2=-1;
  float best_sep = -1;
  for (int split1=1; split1<num_elems; split1++) {
    for (int split2=split1; split2<num_elems; split2++) {
      int num_def = split1;
      int num_mid = split2 - split1;
      int num_fwd = num_elems - split2;
      if (! (Mem->FP_min_defenders <= num_def   && num_def <= Mem->FP_max_defenders &&
	     Mem->FP_min_midfielders <= num_mid && num_mid <= Mem->FP_max_midfielders &&
	     Mem->FP_min_forwards <= num_fwd    && num_fwd <= Mem->FP_max_forwards))
	continue;

      float sep = aXPos[split1] - aXPos[split1-1] + aXPos[split2] - aXPos[split2-1];
      if (sep > best_sep) {
	best_sep = sep;
	best_split1 = split1;
	best_split2 = split2;
      }
    }
  }

  if (best_split1 == -1 || best_split2 == -1)
    my_error("inferRoles could find no valid split");

  for (int num=0; num<num_elems; num++) {
    PlayerRole role;
    if (num < best_split1)
      role = PT_Defender;
    else if (num < best_split2)
      role = PT_Midfielder;
    else
      role = PT_Forward;
    aRole[aNums[num]] = role;
  }
  
  delete [] aNums;
  delete [] aXPos;
#endif	
  return true;
}

/*************************************************************************/

// determines which side the side_str and team_str are talking about
// side_str is checked first, if it is 'left' or 'right', we are done
// otherwise, it should be empty or 'none'
// otherwise, the team_str is compared.
// The team names are allowed to not be filled in
// If a match is found, TS_Left or TS_RIght is returned
// If no match can be found, TS_None is returned
// If no match because of empty team names, TS_Both is returned
TeamSide
determineSide(const char* side_str, const char* team_str,
	      const char* left_team_name, const char* right_team_name)
{
  /* first check to see if we have an explicit side argument */
  if (!side_str || (strcasecmp(side_str, "none")==0) || side_str[0] == 0)
      ;
  else if (strcasecmp(side_str, "left")==0)
    return TS_Left;
  else if (strcasecmp(side_str, "right")==0)
    return TS_Right;
  else if (strcasecmp(side_str, "both")==0)
    return TS_Both;
  else 
    errorlog << "Bad value for side string: " << side_str << ende;

    
  /* then check to see if we have a team name arg */
  if (team_str && team_str[0] != 0)
    {
      /* we have a team name.
      	 Let's try and compare it to the team names we have */
      /* we use strncmp so that we can specify just a beginning string */
      int num_names_set = 0;
      if (left_team_name[0] != 0)
	{
	  num_names_set++;
	  if (strncasecmp(left_team_name, team_str, strlen(team_str))==0)
	    return TS_Left;
	} /* left team name set */
      if (right_team_name[0] != 0)
	{
	  num_names_set++;
	  if (strncasecmp(right_team_name, team_str, strlen(team_str))==0) {
	    return TS_Right;
	  }
	} /* right team name set */

      if (num_names_set == 2)
	{
	  /* both names are set and we are here. Ack! No match! */
	  errorlog << "Given team name '" << team_str << "'  does not match with any of '"
		   << left_team_name << "' '" << right_team_name << "'" << ende;

	  return TS_None;
	} 
      return TS_Both;; /* waiting for more info */
  } /* team_str set */

  errorlog << "Given no information to determine side "
	   << side_str << "\t" << team_str
	   << ende;
  return TS_None;
}



/*************************************************************************/

float VelAtPt2VelAtFoot_sh(VecPosition pt, float targ_vel_at_pt, 
			   VecPosition mypos, float ball_decay)
{
  if (targ_vel_at_pt < EPSILON) {
    return Geometry::getFirstInfGeomSeries((pt - mypos).getMagnitude(), ball_decay );
  } else {
    float ball_steps =
      Geometry::getLengthGeomSeries(targ_vel_at_pt, 1/ball_decay, (pt - mypos).getMagnitude() );
    return targ_vel_at_pt * pow(1/ball_decay, ball_steps);
  }  
}

int CyclesForBallToPoint(VecPosition pt, float targ_vel_at_pt, float max_kick_speed, 
			 VecPosition mypos, float ball_decay)
{
  if (targ_vel_at_pt < EPSILON) 
    errorlog << "Shouldn't call CyclesForBallToPoint with ~0 targ_vel: " << targ_vel_at_pt << ende;
  float kick_spd = Min(max_kick_speed,
		       VelAtPt2VelAtFoot_sh(pt, targ_vel_at_pt, mypos, ball_decay));
  float steps = Geometry::getLengthGeomSeries(kick_spd, ball_decay, mypos.getDistanceTo(pt));
  return (steps < 0 ? -1 : roundToInt(steps));  
}

//SMURF: there may be a better way to write this....
//HETTYPE: we really need to not use SP_player_decay
int UpperBForPlayerToPoint(VecPosition start, VecPosition end)
{
  int num_cycles = 0;
  float dist = start.getDistanceTo(end);
  float speed = 0;

  if (CoachParam::instance()->getTeamMinTimeTurnWhenGoing() <= 1)
    errorlog << "UpperBForPlayerToPoint: I don't like this value of team_min_time_turn_when_going="
	     << CoachParam::instance()->getTeamMinTimeTurnWhenGoing()
	     << ", I'll probably infinte loop" << ende;
  
  num_cycles += 2; //some turns at the beginning
  for (int i= 1; true; i++) {
    if (dist < 0)
      break; // we got there

    if (i % CoachParam::instance()->getTeamMinTimeTurnWhenGoing() != 0) 
      //do a dash
      speed += ServerParam::instance()->getSPMaxPower() * ServerParam::instance()->getSPDashPowerRate();
    speed = Min(speed, ServerParam::instance()->getSPPlayerSpeedMax());
    dist -= speed;
    speed *= ServerParam::instance()->getSPPlayerDecay();
    num_cycles++;
  }
  
  return num_cycles;  
}

int kick_time_helper(VecPosition start, VecPosition end, float targ_vel_at_pt,
		     float max_kick_speed, int cycles_to_kick)
{
  float init_vel = VelAtPt2VelAtFoot_sh(end, targ_vel_at_pt, start, 
					ServerParam::instance()->getSPBallDecay());
  init_vel = Min(init_vel, max_kick_speed);
  int num_cycles = roundToInt(ceil(Geometry::getLengthGeomSeries(init_vel, 
								 ServerParam::instance()->getSPBallDecay(), 
								 start.getDistanceTo(end))));
  num_cycles += cycles_to_kick;
  
  return num_cycles;
}


int UpperBForKickToPoint(VecPosition start, VecPosition end, float targ_vel_at_pt)
{ 
  return kick_time_helper(start,end,targ_vel_at_pt,CoachParam::instance()->getTeamMinKickSpeed(),
			  CoachParam::instance()->getTeamMaxCyclesToKick());
}

int AvgTimeForKickToPoint(VecPosition start, VecPosition end, float targ_vel_at_pt)
{ 
  return kick_time_helper(start,end,targ_vel_at_pt,CoachParam::instance()->getTeamAvgKickSpeed(),
			  CoachParam::instance()->getTeamAvgCyclesToKick());
}

int MinTimeForKickToPoint(VecPosition start, VecPosition end, float targ_vel_at_pt)
{
  return kick_time_helper(start,end,targ_vel_at_pt,CoachParam::instance()->getTeamMaxKickSpeed(),
			  CoachParam::instance()->getTeamMinCyclesToKick());
}


/***************************************************************************************/

//cycles left in game
int CyclesRemaining(int curr_cycle)
{
  return 2*ServerParam::instance()->getSPHalfCycles() - curr_cycle;  
}

//cycles left in current half
int CyclesRemainingHalf(int curr_cycle)
{
  if (curr_cycle <= ServerParam::instance()->getSPHalfCycles())
    return ServerParam::instance()->getSPHalfCycles() - curr_cycle;
  else
    return 2*ServerParam::instance()->getSPHalfCycles() - curr_cycle;
}

/***************************************************************************************/

