/* opp_models.h: this defines a bunch of opponent models for the coach to use in planning */

/* NOTE: predictMovement needs to be able to handle the case where pd and returner
   point to the same object! */

#include "opp_models.h"
#include "ServerParam.h"

void OM_Stones::predictMovement(PlayerDistribution* pd,
				PlayerDistribution* returner,
				BallMovement* bm)
{
  returner->copyFrom(pd);
}

OM_SpreadingStones::OM_SpreadingStones(float var_drift_tmp) : OppModel() 
{
  var_drift = var_drift_tmp;
  sprintf(name, "SpreadingStones[%.4f]", var_drift);
}

void OM_SpreadingStones::predictMovement(PlayerDistribution* pd,
					 PlayerDistribution* returner,
					 BallMovement* bm)
{
  int total_time = bm->getTotalTime();
  float var_spread = ((float)total_time) * var_drift;
  returner->copyFrom(pd);
  for (int num=1; num <= ServerParam::instance()->getSPTeamSize(); num++) {
    returner->getPlayDist(num)->
      SetStdev(Max(.5, pd->getPlayDist(num)->GetXStdevPlus() + var_spread));
  }
}

void OM_AlwaysLeft::predictMovement(PlayerDistribution* pd,
				    PlayerDistribution* returner,
				    BallMovement* bm)
{
  int total_time = bm->getTotalTime();
  
  returner->copyFrom(pd);

  float drift = ((float)total_time) * 0.5;
  float var_spread = ((float)total_time) / 20;

  for (int num=1; num <= ServerParam::instance()->getSPTeamSize(); num++) {
    Variable2DGaussian* pstart = pd->getPlayDist(num);
    Variable2DGaussian* pret = returner->getPlayDist(num);
    pret->SetMean(ServerParam::instance()->getSPFieldRectangle().adjustToWithin(pstart->GetMean() - drift, 0));
    pret->SetStdev(Max(.5, pstart->GetXStdevPlus() + var_spread));
  }
}

void OM_AlwaysRight::predictMovement(PlayerDistribution* pd,
				     PlayerDistribution* returner,
				     BallMovement* bm)
{
  int total_time = bm->getTotalTime();
  
  returner->copyFrom(pd);

  float drift = ((float)total_time) * 0.5;
  float var_spread = ((float)total_time) / 20;

  for (int num=1; num <= ServerParam::instance()->getSPTeamSize(); num++) {
    Variable2DGaussian* pstart = pd->getPlayDist(num);
    Variable2DGaussian* pret = returner->getPlayDist(num);
    pret->SetMean(ServerParam::instance()->getSPFieldRectangle().adjustToWithin(pstart->GetMean() + drift, 0));
    pret->SetStdev(Max(.5, pstart->GetXStdevPlus() + var_spread));
  }
}


OM_NGoToBall::OM_NGoToBall(int n, float speed_tmp, float var_drift_tmp)
{
  num_to_ball = n;
  speed = speed_tmp;
  var_drift = var_drift_tmp;

  pRecBest = new RecordBestN(num_to_ball);

  sprintf(name, "NGoToBall[%d at %.2f; %.4f]", num_to_ball, speed, var_drift);
}

void OM_NGoToBall::predictMovement(PlayerDistribution* pd,
				   PlayerDistribution* returner,
				   BallMovement* bm)
{
  int num;
  int total_time = bm->getTotalTime();
  
  returner->copyFrom(pd);

  //first find out who is close to the ball's end
  VecPosition end = bm->getFinalPosition();
  pRecBest->Clear();
  for (num=1; num <= ServerParam::instance()->getSPTeamSize(); num++) {
    //we have to use minus because higher is better
    pRecBest->AddPoint((void*)num, -pd->getPlayDist(num)->GetMean().getDistanceTo(end));
  }
  
  //now move those players closer to the ball
  float dist_to_move = total_time*speed;
  for (int i=0; i < num_to_ball; i++) {
    num = (int)pRecBest->GetBest(i);
    VecPosition curr_mean = pd->getPlayDist(num)->GetMean();
    VecPosition play_traj = end - curr_mean;
    float curr_dist = play_traj.getMagnitude();
    play_traj = play_traj * (Min(dist_to_move, curr_dist) / curr_dist);
    returner->getPlayDist(num)->SetMean(curr_mean + play_traj);
  }

  //now spread everyone's variance
  float var_spread = ((float)total_time) * var_drift;
  for (num=1; num <= ServerParam::instance()->getSPTeamSize(); num++) {
    Variable2DGaussian* pG = returner->getPlayDist(num);
    pG->SetXStdev(pG->GetXStdevPlus() + var_spread, 
		  pG->GetXStdevMinus() + var_spread);
    pG->SetYStdev(pG->GetYStdevPlus() + var_spread, 
		  pG->GetYStdevMinus() + var_spread);
    
  }
}



