/* -*- Mode: C++ -*- */

/* This file contains some planner modules to be used in the FourLevelPlanner (see 
   FourLevelPlanner.[Ch] that are not specific to the coach or player */

#include "SharePlannerModules.h"
#include "soccer_utils.h"
#include "ServerParam.h"
#include "CoachParam.h"
#include "Logger.h"
using namespace spades;

/************************* Level -: Getting Initial Position **************/

/************************* Level 1: Waypoint ****************************/
/************************* Level 2: RoleMaker ****************************/

/************************* Level 3: STNCompilation ****************************/
bool AddBallOrderEdges(Plan* p)
{
  int last_ball_idx = -1; //the index of the last end node for what the ball did  
  for (int i=1; i<p->GetNumNodes(); i++) {
    PlanNode* pPN = p->GetPlanNode(i);
    switch (pPN->GetType()) {
    case PN_StartPass:
    case PN_SendBall: 
    case PN_StartDribble:
      //connect the last end and this start
      if (last_ball_idx != -1) {
	p->SetEdgeLowerBound(last_ball_idx, i, 0);	
	actionlog(240) << "Setting ball order edge from " << last_ball_idx << " to " << i << ende;
      }
      break;
    case PN_EndPass:
    case PN_EndDribble:
      //set this as the last end
      last_ball_idx = i;      
      break;      

    case PN_StartGoto:
    case PN_EndGoto:
      break;
      
    default:
      errorlog << "What kind of plan node do I have: " << pPN->GetType() << ende;
      return false;      
    }
  }
  return true;
}

bool AddPlayerOrderEdges(Plan* p)
{
  for (int pnum = 1; pnum < ServerParam::instance()->getSPTeamSize(); pnum++) {
    int last_player_idx = 0; //start from initial pos

    for (int i=1; i<p->GetNumNodes(); i++) {
      if (!p->IsPlayerInvolved(pnum, i))
	continue; //if the player's not involved, we have nothing to do
    
      PlanNode* pPN = p->GetPlanNode(i);
      switch (pPN->GetType()) {
      case PN_StartPass:
      case PN_SendBall: 
      case PN_StartDribble:
      case PN_StartGoto:
	//connect the last end and this start
	if (last_player_idx != -1) {
	  p->SetEdgeLowerBound(last_player_idx, i, 0);	
	  p->SetEdgeUpperBound(last_player_idx, i, 
			       CoachParam::instance()->getTeamMaxCyclesToKick() + 
			       CoachParam::instance()->getSppCyclesToKickBuffer());
	  actionlog(240) << "Setting player order edge from "
			 << last_player_idx << " to " << i << ende;
	}
	break;
      case PN_EndPass:
      case PN_EndDribble:
      case PN_EndGoto:
	//set this as the last end
	last_player_idx = i;      
	break;      
      
      default:
	errorlog << "What kind of plan node do I have: " << pPN->GetType() << ende;
	return false;      
      }
    } //node loop
  } //player loop

  return true;
}


bool OnlyInitialPosSTNC::run(AgentBallMovements* pMoves, Problem* theProb, Plan* output)
{
  if (pTimeLeft && *pTimeLeft == TL_Abort) {
    actionlog(60) << "IPSTNC: aborting at beginning of computation" << ende;
    return false;
  }

  output->Clear();
  
  InitialPosPN* pIP = new InitialPosPN;  
  for (int i=0; i < MAX_LOCATIONS; i++) {
    int num = pMoves->GetObjectNum(i);
    if (num == pMoves->ball_num) 
      break; //
    pIP->SetPlayerPos(num, pMoves->GetObjectLoc(i));
    actionlog(210) << "IPSTNC: Set player " << num << " pos at " << pMoves->GetObjectLoc(i) << ende;
  }
  output->SetPlanNode(0, pIP);
  
  return true;
}

bool FullSTNC::run(AgentBallMovements* pMoves, Problem* theProb, Plan* output)
{
  if (pTimeLeft && *pTimeLeft == TL_Abort) {
    actionlog(60) << "FSTNC: aborting at beginning of computation" << ende;
    return false;
  }

  int pnidx = 0;
  int moveidx;  
  int fromPlayer;  
  VecPosition* currentLoc;
  //the previous start and end pass or send nodes
  int prevSidx = -1;
  int prevEidx = -1; 
  int prevprevEidx = -1;
  
  currentLoc = new VecPosition[ServerParam::instance()->getSPTeamSize()+1]; //0 is for the ball
  
  output->Clear();
  
  // ** First handle the InitialPos node **
  InitialPosPN* pIP = new InitialPosPN;  
  for (moveidx=0; moveidx < MAX_LOCATIONS; moveidx++) {
    int num = pMoves->GetObjectNum(moveidx);
    if (num == pMoves->ball_num) {
      currentLoc[0] = pMoves->GetObjectLoc(moveidx);
      break; 
    }
    pIP->SetPlayerPos(num, pMoves->GetObjectLoc(moveidx));
    currentLoc[num] =  pMoves->GetObjectLoc(moveidx);
    actionlog(210) << "FSTNC: Set player " << num << " pos at " << pMoves->GetObjectLoc(moveidx) << ende;
  }
  output->SetPlanNode(pnidx++, pIP);

  //Figure out which player is starting this thing
  fromPlayer = 0;
  for (int i=0; i<moveidx; i++) {
    actionlog(230) << "FSTNC: Finding starting agent, trying "
		   << pMoves->GetObjectNum(i)
		   << pMoves->GetObjectLoc(i)
		   << pMoves->GetObjectLoc(moveidx)
		   << ende;
    if (pMoves->GetObjectLoc(i).getDistanceTo(pMoves->GetObjectLoc(moveidx)) < 1) {
      fromPlayer = pMoves->GetObjectNum(i);
      break;
    }
  }
  if (fromPlayer == 0) {
    errorlog << "FullSTNC: could not find a starting agent!" << ende;
    delete [] currentLoc;
    return false;
  }
  moveidx++; //advance to one past the ball location
  
  // ** Now do the real work of adding passes and goto **
  /* at the beginning of this loop, the following should be true:
     -moveidx points to the node after the last ball location
     -fromPlayer indicates the player who should currently be controlling the ball
     -ballLoc is the current last location of the ball
     -prevSidx is the previous start pass/send/dribble
     -prevEidx is the previous end pass/send/dribble
     -prevprevEidx is the previous previous end pass/send/dribble
  */
  while(moveidx < pMoves->GetNumLoc()) {
    actionlog(230) << "FSTNC: starting for moveidx " << moveidx << ende;
    
    if (pMoves->GetObjectNum(moveidx) == pMoves->ball_num) {
      actionlog(230) << "FSTNC: starting processing send " << moveidx << ende;
      //Do a send
      SendBallPN* pSB = new SendBallPN;
      pSB->SetPlayer(fromPlayer);
      pSB->SetToPos(pMoves->GetObjectLoc(moveidx));

      output->SetPlanNode(pnidx++, pSB);

      currentLoc[0] = pMoves->GetObjectLoc(moveidx);

      actionlog(80) << "FSTNC: Adding a send from " << fromPlayer << " to pt "
		    << pMoves->GetObjectLoc(moveidx) << ende; 

      moveidx++; //advance past the node
      
    } else {
      /* if this is the last player before a ball loc (at the player is at the 
	 right location), set up a pass, 
	 otherwise set up a Start/End Goto */
      if (pMoves->GetObjectNum(moveidx+1) == pMoves->ball_num &&
	  Sqr(pMoves->GetObjectLoc(moveidx).getDistanceTo(pMoves->GetObjectLoc(moveidx+1))) 
	  < CoachParam::instance()->getSppAtPointBuffer()) {
	if (fromPlayer == pMoves->GetObjectNum(moveidx)) {
	  //set up a dribble
	  actionlog(230) << "FSTNC: starting processing dribble " << moveidx << ende;
	  errorlog << "dribble not impl" << ende;
	  delete [] currentLoc;
	  return false;
	} else {
	  //set up a pass
	  actionlog(230) << "FSTNC: starting processing pass " << moveidx << ende;
	  StartPassPN* pSP = new StartPassPN;
	  StartGotoPN* pSG = new StartGotoPN;
	  EndPassPN*   pEP = new EndPassPN;
	  VecPosition targ = pMoves->GetObjectLoc(moveidx+1);
	  int toPlayer = pMoves->GetObjectNum(moveidx);
	  int SPidx, SGidx, EPidx;

	  pSP->SetPlayer(fromPlayer);
	  pSP->SetToPos(targ);
	  pSG->SetPlayer(toPlayer);
	  pSG->SetToPos(targ);

	  //put the nodes is
	  SPidx = pnidx;
	  output->SetPlanNode(pnidx++, pSP);
	  SGidx = pnidx;
	  output->SetPlanNode(pnidx++, pSG);
	  EPidx = pnidx;
	  output->SetPlanNode(pnidx++, pEP);
	
	  //set appropriate edge lens
	  //StartPass must precede EndPass
	  output->SetEdgeLowerBound(SPidx, EPidx, 0);
	  output->SetEdgeUpperBound(SPidx, EPidx, 
				    UpperBForKickToPoint(currentLoc[0], targ, 
							 CoachParam::instance()->getSppPassTargetSpeed()));

	  //StartGoto must precede EndPass
	  output->SetEdgeLowerBound(SGidx, EPidx, 0);
	  /* we don't want this edge because player may have to wait for ball 
	  output->SetEdgeUpperBound(SGidx, EPidx, 
				    UpperBForPlayerToPoint(currentLoc[toPlayer], targ));
	  */

	  //Now add edge lens relating to previous start and end nodes
	  //we only have to worry about when the receiver can start going
	  actionlog(90) << "FSTNC: Receiver " << toPlayer << " order value: "
			<< pMoves->GetObjectOrder(moveidx) << ende;
	  switch (pMoves->GetObjectOrder(moveidx)) {
	  case ABMO_NoConstraint:
	    //nothing to do
	    break;
	  case ABMO_AfterLastLastEnd:
	    if (prevprevEidx != -1)
	      output->SetEdgeLowerBound(prevprevEidx, SGidx, 0);
	    break;
	  case ABMO_AfterLastEnd:
	    if (prevEidx != -1)
	      output->SetEdgeLowerBound(prevEidx, SGidx, 0);
	    break;
	  case ABMO_AfterLastStart:
	    if (prevSidx != -1)
	      output->SetEdgeLowerBound(prevSidx, SGidx, 0);
	    break;
	  case ABMO_AfterThisStart:
	    output->SetEdgeLowerBound(SPidx, SGidx, 0);
	    break;
	  case ABMO_NoValue:
	    errorlog << "Why does player have ABMO_NoValue?" << ende;
	    break;
	  default:
	    errorlog << "What is player's relative ordering: "
		     <<  pMoves->GetObjectOrder(moveidx) << ende;
	  }

	  //set currentLoc
	  currentLoc[0] = targ;
	  currentLoc[toPlayer] = targ;

	  actionlog(80) << "FSTNC: Adding a pass from " << fromPlayer << " to " << toPlayer
			<< " to pt " << targ << ende;		

	  //set new fromPlayer and prev?idx
	  fromPlayer = toPlayer;	
	  prevSidx = SPidx;
	  prevprevEidx = prevEidx;
	  prevEidx = EPidx;
	  //advance moveidx, once over player node, once over ball node
	  moveidx += 2;	
	}
      } else {
	//set up a goto
	actionlog(230) << "FSTNC: starting processing goto " << moveidx << ende;
	if (pnidx == 1)
	  errorlog << "You are trying to add a Goto node before the first kick. This will not work" << ende;
	//create the nodes
	StartGotoPN* pSG = new StartGotoPN;
	EndGotoPN* pEG = new EndGotoPN;
	int player = pMoves->GetObjectNum(moveidx);
	VecPosition targ = pMoves->GetObjectLoc(moveidx);
	int SGidx, EGidx;

	actionlog(230) << "FSTNC: starting processing goto: initial setup done " << moveidx << ende;

	pSG->SetPlayer(player);
	pSG->SetToPos(targ);

	actionlog(230) << "FSTNC: starting processing goto: node info set " << moveidx << ende;

	//put them in the network
	SGidx = pnidx;
	output->SetPlanNode(pnidx++, pSG);
	EGidx = pnidx;
	output->SetPlanNode(pnidx++, pEG);

	actionlog(230) << "FSTNC: starting processing goto: put node in network " << moveidx << ende;

	//add the edges between start end
	output->SetEdgeLowerBound(SGidx, EGidx, 0);
	output->SetEdgeUpperBound(SGidx, EGidx, 
				  UpperBForPlayerToPoint(currentLoc[player], targ));
	
	actionlog(230) << "FSTNC: starting processing goto: added edge len " << moveidx << ende;

	//add other edges
	actionlog(90) << "FSTNC: Extra Goto " << player << " order value: "
		      << pMoves->GetObjectOrder(moveidx) << ende;
	switch (pMoves->GetObjectOrder(moveidx)) {
	case ABMO_NoConstraint:
	  //nothing to do
	  break;
	case ABMO_AfterLastLastEnd:
	  if (prevprevEidx != -1)
	    output->SetEdgeLowerBound(prevprevEidx, SGidx, 0);
	  break;
	case ABMO_AfterLastEnd:
	  if (prevEidx != -1)
	    output->SetEdgeLowerBound(prevEidx, SGidx, 0);
	  break;
	case ABMO_AfterLastStart:
	  if (prevSidx != -1)
	    output->SetEdgeLowerBound(prevSidx, SGidx, 0);
	  break;
	case ABMO_AfterThisStart:
	  errorlog << "For extra goto, can't have ABMO_AfterThisStart" << ende;
	  break;
	case ABMO_NoValue:
	  errorlog << "Why does player have ABMO_NoValue?" << ende;
	  break;
	default:
	  errorlog << "What is player's relative ordering: "
		   << pMoves->GetObjectOrder(moveidx) << ende;
	}

	//set currentLoc
	currentLoc[player] = targ;

	actionlog(80) << "FSTNC: Adding an extra goto for " << player << " to pt " << targ << ende;		

	//advance moveidx, once over player node
	moveidx += 1;	
      }
    } 
  }

  //Now add edges forcing the ball to only be doing one thing at a time
  if (!AddBallOrderEdges(output)) {
    actionlog(80) << "FSTNC: Adding ball order edges failed" << ende;
    delete [] currentLoc;
    return false;
  }
  
  //Now add edges forcing a single players action ordering
  if (!AddPlayerOrderEdges(output)) {
    actionlog(80) << "FSTNC: Adding player order edges failed" << ende;
    delete [] currentLoc;
    return false;
  }


  delete [] currentLoc;
  return true;
}


/************************* Level 4: AgentInstantiation ****************************/
bool SequentialAI::run(Plan* thePlan, Problem* theProb)
{
  if (pTimeLeft && *pTimeLeft == TL_Abort) {
    actionlog(60) << "SeqAI: aborting at beginning of computation" << ende;
    return false;
  }

  Unum arrRep[NUM_PLAYERS+1];

  if (!thePlan)
    return true;

  int player_num = start_num;
  
  for (int i=1; i<=NUM_PLAYERS; i++) {
    if (thePlan->IsPlayerInvolved(i))
      arrRep[i] = player_num++;
    else
      arrRep[i] = -1;    
  }

  actionlog(230) << "SequentialAI: about to call replace" << ende;  
  thePlan->ReplacePlayerNums(arrRep);  
  return true;  
}
