/* -*- Mode: C++ -*- */

/* This file contains some planner modules to be used in the FourLevelPlanner (see 
   FourLevelPlanner.[Ch] that are specific to the coach */

#ifndef _PLANNER_MODULES_H_	
#define _PLANNER_MODULES_H_ 

#include "FourLevelPlanner.h"
#include "OppModel.h"
#include "SharePlannerModules.h"
#include "ModFormation.h"
#include "WorldHistory.h"
class ModSetplayPlan;

/************************* Helper types, classes, and fucntions *********************/

//Note that the pt_list is *relative to this previous point* (except for the
//first of course, but that should be where the ball currently is anyway
struct ball_path_t {
  int num_pts;
  VecPosition* pt_list;
  float* offsides_line;
  float value;
} ;

void print_path(ball_path_t *pbp);

/* returns a value between 0 and 1, where 1 is good */
class PathEvaluator {
public:
  PathEvaluator(TeamSide my_side);

  float evaluate_path(ball_path_t* pBallPath, PlayerDistribution* pStartPd, 
		      OppModel* pOM, float* new_offsides);

#ifdef OLD_CODE
  float evaluate_path_given_occ(VecPosition* path, int num_pts, float avg_occ, float max_occ);
#endif

  //private:
  float eval_end_send(VecPosition end);
  float eval_end_pass(VecPosition end);
  float eval_end_loc(VecPosition start_loc, VecPosition end_loc, bool is_pass);
  float eval_bad_wp(ball_path_t* pBallPath);
  void eval_occup(ball_path_t* pBallPath, PlayerDistribution* pStartPd, 
		  OppModel* pOM, float* pAvg, float* pMax, float* new_offsides);
  void eval_occup_with_keepaway(ball_path_t* pBallPath, PlayerDistribution* pStartPd, 
		  OppModel* pOM, float* pAvg, float* pMax, float* new_offsides);
  float adjust_prob_to_occup(float prob, int time);

  float standard_length_values[10];
  
  Rectangle shrunkField;

  TeamSide getMySide() const { return my_side; }
  void setMySide(TeamSide ts) { my_side = ts; }

  //some useful variables to improve speed of computation
  float side_sign; //-1 for left, 1 for right
  float goal_targ_dist;
  float corner_targ_dist;
  float send_baseline_max_dist;
  float send_baseline_factor;
  float send_penalty_area_max_dist;
  float send_penalty_area_factor;
  float send_in_pen_area_max_dist2;
  float send_in_pen_area_factor;
  float bad_wp_factor;
  float bad_wp_line_dist;
  float max_pass_dist2;
  float half_pitch_len;
  float half_pitch_wid;
  float half_penalty_area_wid;
  float incr_dist;

  double eval_loc_weight;
  double eval_bad_wp_weight;
  double eval_length_weight;
  double eval_pass_weight;
  double eval_avg_occ_weight;
  double eval_max_occ_weight;
  
private:
    TeamSide my_side;

} ;

/************************* Level -: Getting Initial Position **************/
/* this isn't really a level, but in order to make this usable for
   both coach and player we need to have a function which compilies the
   current position into a PlayerDistribution */

class GetCurrentGPD : public GetPlayerDistribution
{
public:
  GetCurrentGPD(TeamSide ts, const WorldHistory& history)
    : GetPlayerDistribution(), my_side(ts), history(history)  {}
  void SetPositions(PlayerDistribution* pd);  

  TeamSide getMySide() const { return my_side; }
  void setMySide(TeamSide ts) { my_side = ts; }
private:
  TeamSide my_side;
  const WorldHistory& history;
} ;

class RandomGPD : public GetPlayerDistribution
{
public:
  void SetPositions(PlayerDistribution* pd);  
} ;


/************************* Level 1: Waypoint ****************************/
class RandomWP : public Waypoint 
{
 public:
  RandomWP() : Waypoint() {}

  bool run(Problem* theProb, PlayerDistribution* pDist, GetPlayerDistribution* pGetPD, 
	   WaypointList* output);
    
} ;

class HillClimbWP : public Waypoint 
{
public:
  HillClimbWP(PathEvaluator* pe, TeamSide my_side, ModSetplayPlan* pmSetplayPlan);
  ~HillClimbWP();

  bool run(Problem* theProb, PlayerDistribution* pDist, GetPlayerDistribution* pGetPD, 
	   WaypointList* output);

  void PrintStartingPaths(bool show_absolute = TRUE);

  void DrawStartingPaths(const char* dir);
    
  TeamSide getMySide() const { return my_side; }
  void setMySide(TeamSide ts) { my_side = ts; }

private:
  
  PathEvaluator* path_eval;
  TeamSide my_side;
  ModSetplayPlan* pmSetplayPlan;

  ball_path_t* paths;
  //curent bests
  int bestPath;
  float bestPathValue;

  //use 0 in num_pts to say that we don't have a path here
  //use -1 in num_pts to indicate not allocated yet
  // note that offsides_line will be unallocated here
  ball_path_t* starting_paths[num_SpecSetPlayMode];

  Rectangle shrunkField;
  PlayerDistribution WorkerPD;
  
  //some things for efficiency
  float min_pass_dist2;
  float max_pass_dist2;
  float max_send_dist2;

  //some useful functions

  //loads the given seeds into the paths variable
  //returns # of paths loaded
  void SetOffsidesLines(Problem* theProb, PlayerDistribution* pDist, 
			OppModel* pOM, int num_paths);
  void EvalAllPaths(Problem* theProb, PlayerDistribution* pDist, 
		    OppModel* pOM, int num_paths);
  void MoveAllPointsToValid(int path_idx);
  int LoadPathsForProblem(Problem* theProb, PlayerDistribution* pDist, OppModel* pOM);
  void print_all_paths();
  
} ;


/************************* Level 2: RoleMaker ****************************/
class SimpleAtPointRM : public RoleMaker
{
 public:
  SimpleAtPointRM() : RoleMaker() {}
  
  bool run(WaypointList* pWaypoints, Problem* theProb, AgentBallMovements* output);
  
} ;

//I'd like to combine the next couple into a really good rolemaker

//Like AtPoint, but will resuse agent's if possible
class AtPointWithReuseRM : public RoleMaker
{
 public:
  AtPointWithReuseRM() : RoleMaker() {}
  
  bool run(WaypointList* pWaypoints, Problem* theProb, AgentBallMovements* output);
  
} ;

//Starts players off their points, taking into account offsides
class OffPointOffsidesRM : public RoleMaker
{
public:
  OffPointOffsidesRM(TeamSide my_side) : RoleMaker(), my_side(my_side) { ClearLocInfo(); }
  
  bool run(WaypointList* pWaypoints, Problem* theProb, AgentBallMovements* output);

  TeamSide getMySide() const { return my_side; }
  void setMySide(TeamSide ts) { my_side = ts; }

private:  
  enum loc_type_t {
    LT_Empty, //no actual value set
    LT_ReceiverOnside, //receiver whose target is onsides
    LT_ReceiverOffside, //receiver whose target is offsides
    LT_Onside, //backing up a player to be onsides
    LT_JustOff, //backing up just to get player away from target point
  } ;
  
  struct loc_t  {
    VecPosition pos;
    loc_type_t type;
    bool first_pos_set;
  } ;

  loc_t loc_info[MAX_WAYPOINTS][NUM_PLAYERS];
  
  TeamSide my_side;

  void ClearLocInfo( int num_WP = MAX_WAYPOINTS );
  bool IsRoleUsed(int role, int num_wp);
} ;

/************************* Level 3: STNCompilation ****************************/

/************************* Level 4: AgentInstantiation ****************************/
class FixedPlanAI : public AgentInstantiation
{
 public:
  FixedPlanAI(TeamSide my_side, int goalie_num);
  ~FixedPlanAI();

  bool run(Plan* thePlan, Problem* theProb);

  TeamSide getMySide() const { return my_side; }
  void setMySide(TeamSide ts) { my_side = ts; }

  int getGoalieNum() const { return goalie_num; }
  void setGoalieNum(TeamSide ts) { goalie_num = ts; }

 private:
  TeamSide my_side;
  int goalie_num;
  Plan* fixedPlan;
  
} ;

class GreedyByRoleAI : public AgentInstantiation
{
public:
  GreedyByRoleAI(ModFormation* p, TeamSide my_side, int goalie_num);
  ~GreedyByRoleAI();

  bool run(Plan* thePlan, Problem* theProb);

  TeamSide getMySide() const { return my_side; }
  void setMySide(TeamSide ts) { my_side = ts; }

  int getGoalieNum() const { return goalie_num; }
  void setGoalieNum(TeamSide ts) { goalie_num = ts; }

private:
  TeamSide my_side;
  int goalie_num;
  SequentialAI* pSeqAI;
  ModFormation* pmFormation;
  int* arrRep;  
  bool* posAssigned;
  
} ;



#endif // _PLANNER_MODULES_H
