/* -*- Mode: C++ -*- */

/* This file contains some planner modules to be used in the FourLevelPlanner (see 
   FourLevelPlanner.[Ch] that are specific to the coach */

#include "ModSetplayPlan.h"
#include "soccer_utils.h"
#include "data.h"
#include "PlannerModules.h"
#include "plan_utils.h"
#include "ModFormation.h"
#include "ModSetplayPlan.h"
#include "FieldImage.h"
#include "ServerParam.h"
#include "CoachParam.h"
#include "Logger.h"
using namespace spades;

#define DEBUG_MSG(x) 

/************************* Level -: Getting Initial Position **************/
/* this isn't really a level, but in order to make this usable for
   both coach and player we need to have a function which compilies the
   current position into a PlayerDistribution */


void GetCurrentGPD::SetPositions(PlayerDistribution* pd)
{
  /* This is almost unsafe threadwise; another thread is probably updating the WorldHistory,
     so we could get some weird effects. However, getting a reference to the WorldState should
     be atomic enough. Once we have the WorldState, we just hope that it doesn't
     get deleted while we are messing with it. In all reasonable cases, this shouldn't be
     a problem */
  const WorldState& ws = history.getWorldState();
  VecPosition player_pos[ServerParam::instance()->getSPTeamSize()];
  for (int num = 1; num <= ServerParam::instance()->getSPTeamSize(); num++) {
    const PlayerInfo* pi = ws.getPlayer(getOppositeSide(my_side), num);
    if (pi)
      player_pos[num-1] = pi->getPos();
    else
      player_pos[num-1] = VecPosition(0, -1000);
  }

  pd->setInitial(player_pos);
}

void RandomGPD::SetPositions(PlayerDistribution* pd)
{
  VecPosition player_pos[ServerParam::instance()->getSPTeamSize()];
  for (int num = 1; num <= ServerParam::instance()->getSPTeamSize(); num++) {
    //there should really be a Rectangle::random, but my random funcs aren't in clanglib
    player_pos[num-1].setX(range_random(ServerParam::instance()->getSPFieldRectangle().getPosLeftTop().getX(),
					ServerParam::instance()->getSPFieldRectangle().getPosRightBottom().getX()));
    player_pos[num-1].setY(range_random(ServerParam::instance()->getSPFieldRectangle().getPosLeftTop().getY(),
					ServerParam::instance()->getSPFieldRectangle().getPosRightBottom().getY()));
  }

  pd->setInitial(player_pos);
}


/************************* Helper classes **************************/

void print_path(ball_path_t *pbp)
{
  for (int i=0; i<pbp->num_pts; i++)
    cout << pbp->pt_list[i] << " ";
  cout << endl;
}


/*********** This is stuff from keepaway.C ***********/
//this is 1/(1-ball_decay)!
#define ball_time_constant 20

float interception_time(VecPosition ballpt, VecPosition ball_velocity, VecPosition recpt, float player_speed, float player_delay, float time_bound){
  /***
      I will work in ball coordinates --- y is along the ball path with positive y in the direction
      the ball is moving and x is orthoganal with positive x in direction is toward the receiver.
      We let rx and ry be the intial receiver x and y coordinates in this system.  Note
      that rx >= 0 but ry can be negative.

      ball_y(t) is the y coordinate of the ball (which equals the distance traveled by the ball)
      at time t.  ball_y(t) = ymax(1-exp(t/tau)) where tau is the velocity decay time constant
      of the ball.

      define u(t) to be time taken by the receiver to reach the y axis
      at y = ball_y(t).  Given that the receiver runs with velocity 1
      we have u(t) = sqrt(rx^2 + (bally - ry)^2).

      We as seeking the smallest t such that u(t) = t which we will write as g(t) = 0
      where g(t) is defined to by u(t)-t.  We will maintain the invariant
      that u(t) >= t or equivalently g(t) > 0.
      Note that u(t) >= rx so we can initial t to be rx.
  ***/

  float ball_speed = ball_velocity.getMagnitude();
  VecPosition ball_dir = ball_velocity/ball_speed;
  float rx = fabs(VecPosition::crossProduct((recpt - ballpt), ball_dir));
  float ry = VecPosition::dotProduct((recpt - ballpt), ball_dir);
  float tau = ball_time_constant;
  float ymax = ball_speed*tau;

  float t = Max(0.0,rx/player_speed + player_delay);

  int counter = 0;
  while(1){
    counter++;
    float decay = exp(-t/tau);
    float by = ymax*(1-decay);
    float deltay = by - ry;
    float d = sqrt(rx*rx + deltay*deltay);
    float u = d/player_speed + player_delay;
    float g = u-t;
    if (counter > 100) {
      actionlog(90) << "compute_pass_prob: counter is too big, so returning "
		    << counter << " " << g << " " << t << ende;
      return t;
    }
    if(g < .1 || t > time_bound){
      //printf("--- %d iterations\n",counter);
      return t;
    }
    float dd_dy = deltay/d;
    float du_dy = dd_dy/player_speed;
    float dy_dt = ball_speed*decay;
    float du_dt = du_dy*dy_dt;
    float dg_dt = du_dt-1;
    if(dg_dt > 0){
      t = u;
    } else if(deltay < 0){
      t -= g/dg_dt; /*** this increments t ***/
    } else {
      float overshoot_deltat = -g/dg_dt;
      float lower_dy_dt = dy_dt*(1-overshoot_deltat/tau);
      lower_dy_dt = Max(0.0,lower_dy_dt);
      float lower_du_dt = du_dy*lower_dy_dt;
      float lower_dg_dt = lower_du_dt - 1; /*** lower_dg_dt <= dg_dt <= 0 ***/
      t -= g/lower_dg_dt;
      t = Max(t,u);
    }
  }
}

VecPosition ball_position(VecPosition ball_start, VecPosition ball_vel, float t){
  float ball_speed = ball_vel.getMagnitude();

  float tau = ball_time_constant;
  float dist_max = ball_speed*tau;
  float decay = exp(-t/tau);
  float dist = dist_max*(1-decay);
  return ball_start + VecPosition::getVecPositionFromPolar(dist,ball_vel.getDirection());
}

float compute_pass_prob(VecPosition start, VecPosition end, PlayerDistribution* pDist, int mytime) {
  VecPosition velocity = (end-start).scaleTo(CoachParam::instance()->getTeamAvgKickSpeed());
  //Unum receiver = Unum_Unknown;
  Unum interceptor = Unum_Unknown;

  float t;
  //VecPosition pt;
    
  float theirtime = 10000;
  for (Unum PlayerNum = 1; PlayerNum <= ServerParam::instance()->getSPTeamSize(); PlayerNum++){
      
    t = interception_time(start,velocity,pDist->getPlayDist(PlayerNum)->GetMean(),
			  CoachParam::instance()->getSppEvalOppSpeed(), CoachParam::instance()->getSppEvalOppDelay(),
			  theirtime);
	
    if ( t < theirtime ){
      theirtime = t;
      interceptor = PlayerNum;
    }
  }

  float reception_delta = soft_if( soft_less(mytime,7.0,2.0),1.0,5.0 );
  //float reception_delta = 1;
  float reception_prob = soft_if( soft_less(mytime,theirtime,reception_delta),
				  .9,0 );

  //harder to receive if ball moving fast
  float speed_at_reception = velocity.getMagnitude()*pow(ServerParam::instance()->getSPBallDecay(),mytime);
  reception_prob *= soft_if( soft_less(speed_at_reception,1.3,1), 1, .1 );

  float dist2edge = ServerParam::instance()->getSPFieldRectangle().distanceToEdge(end);
  // If the reception point's more than 5 from the edge of the field,
  //   it's safe -- out of bounds forget it -- smooth in between
  reception_prob *= MinMax(0.0,dist2edge/CoachParam::instance()->getSppSidelineBuffer(),1.0);

#if 0

  //local_advantage = local_advantage_at_pt(reception_point);
  next_pass_val = ball_position_value(reception_point);
  next_pass_val *= soft_if ( soft_less(reception_point.getX(),Mem->their_offside_line+5,5),
			     .5,1);

  float reception_delta = soft_if( soft_less(mytime,7,2),1,5 );
  //float reception_delta = 1;
  reception_prob = soft_if( soft_less(mytime,theirtime,reception_delta),
			    .9,0 );

  //harder to receive if ball moving fast
  speed_at_reception = velocity.getMagnitude()*pow(ServerParam::instance()->getSPBallDecay(),mytime);
  reception_prob *= soft_if( soft_less(speed_at_reception,1.3,1), 1, .1 );

  float field_buffer = 5;
  // If the reception point's more than 5 from the edge of the field,
  //   it's safe -- out of bounds forget it -- smooth in between
  reception_prob *= MinMax(0,dist2edge/field_buffer,1);

  if ( TeammateGoodShot[receiver] ){
    shot_option op;
    op.compute_value(Mem->ShotTarget(reception_point),reception_point);
    next_pass_val+=op.value; 
  }

  value = next_pass_val*reception_prob;
#endif

  return reception_prob;

}


/*****************************/


PathEvaluator::PathEvaluator(TeamSide my_side)
  : my_side(my_side)
{
  if (my_side == TS_Left)
    side_sign = -1.0;
  else if (my_side == TS_Right)
    side_sign =  1.0;
  else
    errorlog << "PathEval: what is MySide? " << my_side << ende;

  //VAR: play with the length values
  standard_length_values[0] = 0.0;
  standard_length_values[1] = 0.4;
  standard_length_values[2] = 0.8;
  standard_length_values[3] = 1.0;
  standard_length_values[4] = 0.5;
  standard_length_values[5] = 0.2;
  standard_length_values[6] = 0.1;
  standard_length_values[7] = 0.0;
  
  //for end location evaulations
  //(VecPosition(ServerParam::instance()->getSPPitchLength()/2.0,ServerParam::instance()->getSPPitchWidth()/2.0).getMagnitude() - 
  //ServerParam::instance()->getSPPenaltyAreaLength()));
  goal_targ_dist = ServerParam::instance()->getSPPenaltyAreaLength();
  corner_targ_dist = 6.0;
  send_baseline_max_dist = ServerParam::instance()->getSPPenaltyAreaLength() * 2.0;
  send_baseline_factor = CoachParam::instance()->getSppEvalSendBaseMax() / send_baseline_max_dist;
  send_penalty_area_max_dist = ServerParam::instance()->getSPPitchLength() / 2.0 - 
    ServerParam::instance()->getSPPenaltyAreaLength();
  send_penalty_area_factor = CoachParam::instance()->getSppEvalSendPenMax() / send_penalty_area_max_dist;
  send_in_pen_area_max_dist2 = 
    VecPosition(ServerParam::instance()->getSPPenaltyAreaLength(), ServerParam::instance()->getSPPenaltyAreaWidth()/2.0).getMagnitude2();
  send_in_pen_area_factor = .3 / send_in_pen_area_max_dist2;

  bad_wp_line_dist = 6;
  bad_wp_factor = 1.0 / bad_wp_line_dist;

  max_pass_dist2 = Sqr(CoachParam::instance()->getSppMaxPassDist());
  half_pitch_len = ServerParam::instance()->getSPPitchLength() / 2.0;
  half_pitch_wid = ServerParam::instance()->getSPPitchWidth() / 2.0;
  half_penalty_area_wid = ServerParam::instance()->getSPPenaltyAreaWidth() / 2.0;

  shrunkField = ServerParam::instance()->getSPFieldRectangle().shrink(CoachParam::instance()->getSppSidelineBuffer());

  //scale the evaluation weights
  //notice that we do not add in the eval_bad_wp_weight because it should be negative
  float sum = 
    CoachParam::instance()->getSppEvalLocWeight() +
    CoachParam::instance()->getSppEvalLengthWeight() +
    CoachParam::instance()->getSppEvalPassWeight() +
    CoachParam::instance()->getSppEvalAvgOccWeight() +
    CoachParam::instance()->getSppEvalMaxOccWeight();
  eval_loc_weight = CoachParam::instance()->getSppEvalLocWeight() / sum;
  eval_bad_wp_weight = CoachParam::instance()->getSppEvalBadWpWeight() / sum;
  eval_length_weight = CoachParam::instance()->getSppEvalLengthWeight() / sum;
  eval_pass_weight = CoachParam::instance()->getSppEvalPassWeight()    / sum;
  eval_avg_occ_weight = CoachParam::instance()->getSppEvalAvgOccWeight() / sum;
  eval_max_occ_weight = CoachParam::instance()->getSppEvalMaxOccWeight() / sum;

  incr_dist = sqrt(1.0/CoachParam::instance()->getSppEvalSampleDensity());
  
}

#ifdef OLD_CODE
float PathEvaluator::evaluate_path_given_occ(VecPosition* path, int num_pts, 
					     float avg_occ_value, float max_occ_value)
{
  bool is_last_pass = (path[num_pts-2].getDistanceTo2(path[num_pts-1]) <= max_pass_dist2);
  float loc_value = eval_end_loc(path[0], path[num_pts-1], is_last_pass);
  float length_value = standard_length_values[num_pts-1]; 
  float pass_value = (is_last_pass ? 1.0 : 0.0);

  actionlog(260) << "Eval: "
		 << loc_value << " " 
		 << length_value << " " 
		 << pass_value << " " 
		 << avg_occ_value << " " 
		 << max_occ_value << ende;

  return 
    eval_loc_weight * loc_value +
    eval_length_weight * length_value +
    eval_pass_weight * pass_value +
    eval_avg_occ_weight * avg_occ_value +
    eval_max_occ_weight * max_occ_value;
}
#endif

  
float PathEvaluator::eval_end_send(VecPosition end_loc)
{
  float loc_value;
  if (ServerParam::instance()->getSPPenaltyAreaRectangle(my_side == TS_Left).isInside(end_loc)) {
    //we want to make this tilt out to the sides to get hillclimbing to go that way
    loc_value = 0.2 * fabs(end_loc.getY()) / half_penalty_area_wid;
  } else {
    float corner_loc_value;
    float pen_area_loc_value;
    float x_pos_value = Max(0.0, (-side_sign*end_loc.getX()) / (half_pitch_len));
    float d;
    VecPosition corner_pos(-side_sign*half_pitch_len, signf(end_loc.getY())*half_pitch_wid);
    d = end_loc.getDistanceTo(corner_pos);
    
    if (d > corner_targ_dist) {
      corner_loc_value = Max(0.0, 1.0 - Sqr(d - corner_targ_dist) * Sqr(1.0/18));
    } else {
      corner_loc_value = Max(0.0, 1.0 - Sqr(d - corner_targ_dist) * 
			     Sqr(1.0/corner_targ_dist));
    }

    d = end_loc.getDistanceTo(VecPosition(-side_sign*32.0, 0));
    pen_area_loc_value = 0.6 * Max(0.0, 1.0 - d / 15.0);

    loc_value = weightedSum(corner_loc_value + pen_area_loc_value, 
			    Max(0.0,x_pos_value-.2), 0.7 * x_pos_value);
  }

  return loc_value;
#ifdef OLD_CODE
  float loc_value;
  if (ServerParam::instance()->getSPPenaltyAreaRectangle(my_side == TS_Left).isInside(end_loc)) {
    //we want to make this tilt out to the sides to get hillclimbing to go that way
    loc_value = send_in_pen_area_factor * getGoalPosForSide(getOppositeSide(my_side)).getDistanceTo2(end_loc);
  } else {
    if (fabs(end_loc.getX()) > half_pitch_len - ServerParam::instance()->getSPPenaltyAreaLength()) {
      //use the end line segment
      //VAR: the 5.0 is a buffer that you can play with 
      float seg_x = -side_sign * (half_pitch_len - 5.0);
      float seg_top_y, seg_bot_y;
      if (end_loc.getY() > 0) {
	seg_top_y = half_pitch_wid;
	seg_bot_y = half_pitch_wid - half_penalty_area_wid;
      } else { 
	seg_top_y = -(half_pitch_wid - half_penalty_area_wid);
	seg_bot_y = -half_pitch_wid;
      }
      loc_value = send_baseline_factor * 
	(send_baseline_max_dist - 
	 getDistToVerticalSegment(seg_x, seg_top_y, seg_bot_y, end_loc));
      actionlog(240) << "LocValue (base line): "
		     << loc_value
		     << "\tmax="<< send_baseline_max_dist
		     << ", dist=" << GetDistToVerticalSegment(seg_x, seg_top_y, seg_bot_y, end_loc)
		     << ", segx=" << seg_x
		     << ende;
    } else {
      //use the top of penalty box segment
      float seg_x = -side_sign * (half_pitch_len - ServerParam::instance()->getSPPenaltyAreaLength());
      loc_value = send_penalty_area_factor * 
	(send_penalty_area_max_dist - 
	 GetDistToVerticalSegment(seg_x, half_penalty_area_wid, 
				  -half_penalty_area_wid, end_loc));
      actionlog(240) << "LocValue (pen area): "
		     << loc_value
		     << "\tmax=" << send_penalty_area_max_dist
		     << ", dist=" << GetDistToVerticalSegment(seg_x, half_penalty_area_wid, 
							      -half_penalty_area_wid, end_loc)
		     << ", segx=" << seg_x
		     << ende;
    }
    loc_value = Max(loc_value, 0.0);
  }
  return loc_value;
#endif
}

float PathEvaluator::eval_end_pass(VecPosition end_loc)
{
  float x_pos_value = Max(0.0, (-side_sign*end_loc.getX()) / (half_pitch_len));

  float goal_loc_value;
  float d = end_loc.getDistanceTo(getGoalPosForSide(getOppositeSide(my_side)));
  if (d > goal_targ_dist) {
    goal_loc_value = Max(0.0,
		    1.0 - Sqr(d - goal_targ_dist) * Sqr(1.0/17));
  } else {
    goal_loc_value = Max(0.0,
		    1.0 - Sqr(d - goal_targ_dist) * Sqr(1.0/17));
  }
  goal_loc_value *= .8;
  
  float corner_loc_value;
  VecPosition corner_pos(-side_sign*half_pitch_len, signf(end_loc.getY())*half_pitch_wid);
  d = end_loc.getDistanceTo(corner_pos);

  if (d > corner_targ_dist) {
    corner_loc_value = Max(0.0, 1.0 - Sqr(d - corner_targ_dist) * Sqr(1.0/15));
  } else {
    corner_loc_value = Max(0.0, 1.0 - Sqr(d - corner_targ_dist) * 
			   Sqr(1.0/corner_targ_dist));
  }

  float loc_value = 
  weightedSum(corner_loc_value, fabs(end_loc.getY()) / half_pitch_wid, goal_loc_value);
  loc_value = weightedSum(loc_value, x_pos_value, x_pos_value);
  return loc_value;
  //return goal_loc_value;
  //return weightedSum(corner_loc_value, 0.0, goal_loc_value);
  //return weightedSum(corner_loc_value, 1.0, goal_loc_value);

#ifdef PETER_CODE  
  x_pos_comp = weightedSum ( soft_less(end_loc.getDistanceTo(corner_pos),30,  20),
			     100, x_pos_comp );

  // between 0 and 30 
  
  float x_pos_comp = ((end_loc.getX() + half_pitch_len)/(ServerParam::instance()->getSPPitchLength());
  x_pos_comp += soft_greater( end_loc.getX(), 40, 10 );
  return x_pos_comp;
  

  // add a value for going to the corner
  
  //return x_pos_comp;

  float goal_dist = end_loc.getDistanceTo(getGoalPosForSide(getOppositeSide(my_side))); 
 //x_pos_comp should be less than value of being near the goal
  return weightedSum( soft_less(goal_dist,17,5), 100, x_pos_comp );
#endif
#ifdef OLD_CODE
  float loc_value;
  float d = end_loc.getDistanceTo(getGoalPosForSide(getOppositeSide(my_side)));
  if (d > ServerParam::instance()->getSPPenaltyAreaLength()) {
    loc_value = 1.0 - 
      Sqr((end_loc.getDistanceTo(getGoalPosForSide(getOppositeSide(my_side))) - ServerParam::instance()->getSPPenaltyAreaLength())) * 
      pass_outer_scale_factor;
  } else {
    loc_value = 1.0 - 
      Sqr((end_loc.getDistanceTo(getGoalPosForSide(getOppositeSide(my_side))) - ServerParam::instance()->getSPPenaltyAreaLength())) * 
      pass_inner_scale_factor;
  }
  return loc_value;  
#endif
}


float PathEvaluator::eval_end_loc(VecPosition start_loc, VecPosition end_loc, bool is_pass)
{
  float just_dist_weight = 
    MinMax(0.0, (start_loc.getX() * side_sign) / half_pitch_len, 1.0);

  //VAR: the .75 is something you can play with
  float just_dist_value = -side_sign*(end_loc.getX() - start_loc.getX()) / 
    (.75 * ServerParam::instance()->getSPPitchLength());
  float loc_value;

  if (is_pass) {
    loc_value = eval_end_pass(end_loc);
  } else {
    loc_value = eval_end_send(end_loc);
  }

  actionlog(250) << "End Loc " << end_loc << " just_dist: " << just_dist_value
  << "; real_loc: " << loc_value
  << "; weight:" <<  just_dist_weight << ende;

  return weightedSum(just_dist_value, just_dist_weight, loc_value);
}

float PathEvaluator::eval_bad_wp(ball_path_t* pBallPath) 
{
  float badness = 0.0;
  for (int i=1; i<pBallPath->num_pts; i++) {
    badness += 
      Max(0.0, bad_wp_factor * 
	  (bad_wp_line_dist - getDistToVerticalSegment(side_sign * (half_pitch_len - ServerParam::instance()->getSPPenaltyAreaLength() - 5),
						       -half_penalty_area_wid, half_penalty_area_wid,
						       pBallPath->pt_list[i])));
  }
  return MinMax(0.0, badness, 1.0);
}

/* This function moves the probability towards 1 to get an occupancy value
   As time increase, the values get closer to 1 */
inline float PathEvaluator::adjust_prob_to_occup(float prob, int time)
{
  // this is the max_exp / CoachParam::instance()->getSppTimeLimit
  const float time_factor =  25.0 / 150.0;
  
  return 1.0 - pow((double)(1.0-prob), (double) (time * time_factor));
}



//if you don't define this, we'll just sample the trajectory itself, which is faster
// but not as accurate
#define SAMPLE_CONE
//this is used in the next TWO functions
//#define SAMPLE_NOTHING
inline void PathEvaluator::eval_occup(ball_path_t* pBallPath, PlayerDistribution* pStartPd, 
			       OppModel* pOM, float* pAvg, float* pMax, 
			       float* new_offsides)
{
  MaxMeanSummary summOcc;
  BallMovement bm;
  PlayerDistribution playLoc;
  int total_movement_time = 0;

  playLoc.copyFrom(pStartPd);
  bm.addBallPosition(pBallPath->pt_list[0], 0);
  
  VecPosition curr_pos = pBallPath->pt_list[0];
  VecPosition new_pos;
  for (int pt_idx=0; pt_idx < pBallPath->num_pts-1; pt_idx++) {

    new_pos = curr_pos+pBallPath->pt_list[pt_idx+1];
    //First, step forward the player distribution
    if (new_offsides) {
      //set new_offsides
      //NOTE: we wll not set the num_pts-1 location!
      new_offsides[pt_idx] = GetOffsidesLine(my_side, &playLoc, curr_pos);
    }
    //now use the opponent model for the player locations
    int movement_time = 
      AvgTimeForKickToPoint(curr_pos, new_pos, CoachParam::instance()->getSppPassTargetSpeed());
    total_movement_time += movement_time;
    bm.addBallPosition(new_pos, movement_time);
    
    pOM->predictMovement(&playLoc, &playLoc, &bm);

    //this will shift over the thing in location 1
    bm.removeBallPosition(0);

#ifndef SAMPLE_NOTHING

    VecPosition traj = pBallPath->pt_list[pt_idx+1];
    float traj_len = traj.getMagnitude() + CoachParam::instance()->getSppEvalSampleExtraLen();
#ifdef SAMPLE_CONE
    /* Here's the idea: We store two increment vectors: One along the path (traj_incr)
       and one perpindicular for sampling as the cone widens (cone_incr). We track 
       the number of times to sample above and below the trajectory line (num_cone_incr)
       That value is incremented (by num_cone_incr_incr) as we step along the 
       trajectory */
    VecPosition traj_incr = (traj / traj_len) * incr_dist;
    VecPosition cone_incr = traj_incr.rotate90();
    float num_traj_incr = ceil(traj_len / CoachParam::instance()->getSppEvalSampleDensity());
    float num_cone_incr_at_end = 
      traj_len * CoachParam::instance()->getSppEvalSampleWdRatio() / incr_dist;
    float num_cone_incr_incr = num_cone_incr_at_end / num_traj_incr;
    float num_cone_incr = 0;

    actionlog(270) << "About to sample cone: " << curr_pos << " to "
		   << (curr_pos+pBallPath->pt_list[pt_idx+1])
		   << "(off samples: " << num_cone_incr_at_end << ")"
		   << ende;
    VecPosition traj_sample_pt = curr_pos;
    for (int traj_samp=0; traj_samp < num_traj_incr; traj_samp++) {
      VecPosition upp_cone_sample_pt = traj_sample_pt;
      VecPosition low_cone_sample_pt = traj_sample_pt;
      //SMURF: this samples the trajectory twice, but it's not the end of the world
      for (int cone_samp = (int)floor(num_cone_incr); cone_samp >= 0; cone_samp--) {
	actionlog(270) << "cone sampling " << traj_samp
		       << upp_cone_sample_pt << low_cone_sample_pt << ende;
	summOcc.addPoint(adjust_prob_to_occup(playLoc.getSumOcc(upp_cone_sample_pt), 
					      total_movement_time));
	summOcc.addPoint(adjust_prob_to_occup(playLoc.getSumOcc(low_cone_sample_pt),
					      total_movement_time));
	upp_cone_sample_pt += cone_incr;
	low_cone_sample_pt += cone_incr;
      }
      num_cone_incr += num_cone_incr_incr;
      traj_sample_pt += traj_incr;
    }

#else
    VecPosition incr = (traj / traj_len) * sqrt(1.0/CoachParam::instance()->getSppEvalSampleDensity());
    VecPosition sample_pt = curr_pos;
    for (int i = closest_int(traj_len / CoachParam::instance()->getSppEvalSampleDensity());
	 i>=0; i--) {
      //sample this point
      float sample = playLoc.getSumOcc(sample_pt);
      actionlog(270) << "eval sampled pt: " << sample << ende;
      summOcc.addPoint(adjust_prob_to_occup(sample));
      sample_pt += incr;
    }
#endif
#endif //!SAMPLE_NOTHING
    
    curr_pos = new_pos;
  }

#ifndef SAMPLE_NOTHING
  *pAvg = summOcc.getMean();
  *pMax = summOcc.getMax();
#else
  *pAvg = 0.0;
  *pMax = 0.0;
#endif

}

void PathEvaluator::eval_occup_with_keepaway(ball_path_t* pBallPath, PlayerDistribution* pStartPd, 
					     OppModel* pOM, float* pAvg, float* pMax, 
					     float* new_offsides)
{
  MaxMeanSummary summOcc;
  BallMovement bm;
  PlayerDistribution playLoc;
  int total_movement_time = 0;

  playLoc.copyFrom(pStartPd);
  bm.addBallPosition(pBallPath->pt_list[0], 0);
  
  VecPosition curr_pos = pBallPath->pt_list[0];
  VecPosition new_pos;

  for (int pt_idx=0; pt_idx < pBallPath->num_pts-1; pt_idx++) {

    new_pos = curr_pos+pBallPath->pt_list[pt_idx+1];

    if (new_offsides) {
      //set new_offsides
      //NOTE: we wll not set the num_pts-1 location!
      new_offsides[pt_idx] = GetOffsidesLine(my_side, &playLoc, curr_pos);
    }
    //now use the opponent model for the player locations
    int movement_time = 
      AvgTimeForKickToPoint(curr_pos, new_pos, CoachParam::instance()->getSppPassTargetSpeed());
    total_movement_time += movement_time;

#ifndef SAMPLE_NOTHING

    actionlog(240) << "4LPworker: about to compute pass prob" << ende;
    float pass_prob = compute_pass_prob(curr_pos, new_pos, &playLoc, movement_time);
    actionlog(240) << "4LPworker: done with pass prob" << ende;

    //we use 1.0 minus because 0 is good and 1 is bad
    summOcc.addPoint(1.0 - pass_prob);

#endif //!SAMPLE_NOTHING

    bm.addBallPosition(new_pos, movement_time);
    
    pOM->predictMovement(&playLoc, &playLoc, &bm);

    //this will shift over the thing in location 1
    bm.removeBallPosition(0);
    
  }

#ifndef SAMPLE_NOTHING
  *pAvg = summOcc.getMean();
  *pMax = summOcc.getMax();
#else
  *pAvg = 0.0;
  *pMax = 0.0;
#endif
  
}


float PathEvaluator::evaluate_path(ball_path_t* pBallPath,
				   PlayerDistribution* pStartPd, 
				   OppModel* pOM, float* new_offsides)
{
  bool is_last_pass = (pBallPath->pt_list[pBallPath->num_pts-1].getMagnitude2() <= 
		       max_pass_dist2);
  /* now we need to find the ending point; everything is in relative coords */
  //If we find a point off the field, return 0.0
  VecPosition pos = pBallPath->pt_list[0]; 
  for (int i=1; i<pBallPath->num_pts; i++) {
    VecPosition newpos = pos + pBallPath->pt_list[i];
    if (!shrunkField.isInside(newpos)) {
      //warninglog(20) << "Eval: pt " << i << " of " << pBallPath->num_pts << ": " << newpos << " not in field " << shrunkField << ende;
      actionlog(240) << "Eval: pt " << i << " of " << pBallPath->num_pts << " not in field " << newpos << ende;
      return 0.0;
    }

    if (is_last_pass || i < pBallPath->num_pts - 1) {
      //note that we use the offsides line for when the pass *started*
      if (IsPassTooFarOffsides(my_side, pos, newpos, pBallPath->offsides_line[i-1])) {
	actionlog(240) << "Eval: pt " << i << " too far offsides "
		       << pos << " -> " << newpos
		       << pBallPath->offsides_line[i-1] << ende;
	return 0.0;
      }
    }
    pos = newpos;
  }

  float loc_value = eval_end_loc(pBallPath->pt_list[0], pos, is_last_pass);
  float bad_wp_value = eval_bad_wp(pBallPath);
  float length_value = standard_length_values[pBallPath->num_pts-1]; 
  float pass_value = (is_last_pass ? 1.0 : 0.0);
  float avg_occ_value = -1.0;
  float max_occ_value = -1.0;
  //eval_occup(pBallPath, pStartPd, pOM, &avg_occ_value, &max_occ_value, new_offsides);
  actionlog(230) << "4LPworker: about to eval with keepaway" << ende;
  eval_occup_with_keepaway(pBallPath, pStartPd, pOM, &avg_occ_value, &max_occ_value, new_offsides);
  //avg_occ_value = max_occ_value = 0;
  avg_occ_value = 1 - avg_occ_value;
  max_occ_value = 1 - max_occ_value;

  actionlog(240) << "Eval: "
		 << loc_value << " " 
		 << bad_wp_value << " " 
		 << length_value << " " 
		 << pass_value << " " 
		 << avg_occ_value << " " 
		 << max_occ_value << ende;

  return 
    eval_loc_weight * loc_value +
    eval_bad_wp_weight * bad_wp_value +
    eval_length_weight * length_value +
    eval_pass_weight * pass_value +
    eval_avg_occ_weight * avg_occ_value +
    eval_max_occ_weight * max_occ_value;
}



/************************* Level 1: Waypoint ****************************/

/*********** RandomWP *********/

bool RandomWP::run(Problem* theProb, PlayerDistribution* pDist, 
		   GetPlayerDistribution* pGetPD, WaypointList* output)
{
  if (pTimeLeft && *pTimeLeft == TL_Abort) {
    actionlog(60) << "RandomWP: aborting at beginning of computation" << ende;
    return false;
  }

  PlayerDistribution pd;
  const int num_waypoints = 5;
  const float waypoint_dist = 
    (CoachParam::instance()->getSppMaxPassDist() + CoachParam::instance()->getSppMinPassDist()) / 2.0;

  pd.clear();
  output->Clear();

  output->SetWaypoint(0, theProb->ball_pos, &pd);
  
  for (int i=1; i<num_waypoints-1; i++) {
    output->SetWaypoint(i, 
			ServerParam::instance()->getSPFieldRectangle().adjustToWithin(output->GetPt(i-1) + 
							   VecPosition::getVecPositionFromPolar(waypoint_dist, int_random(360))),
			&pd);    
  }

  //Add a send at the end
  output->SetWaypoint(num_waypoints-1, 
		      ServerParam::instance()->getSPFieldRectangle().adjustToWithin(output->GetPt(num_waypoints-2) + 
							 VecPosition::getVecPositionFromPolar(CoachParam::instance()->getSppMaxSendDist()-5.0, 
								      int_random(360))),
		      &pd);    
  

  return true;  
}

/*********** HillClimbWP *********/

HillClimbWP::HillClimbWP(PathEvaluator* pe, TeamSide my_side, ModSetplayPlan* pmSetplayPlan)
  : Waypoint(), my_side(my_side), pmSetplayPlan(pmSetplayPlan)
{
  //allocate the starting paths 
  for (int i=0; i<num_SpecSetPlayMode; i++) {
    switch ((SpecSetPlayMode)i) {
    case SSPM_No_Mode:
    case SSPM_Their_Kick_Off:
    case SSPM_Their_Kick_In_Def:
    case SSPM_Their_Kick_In_Mid:
    case SSPM_Their_Kick_In_Off:
    case SSPM_Their_Corner_Kick:
    case SSPM_Their_Goal_Kick:
    case SSPM_Their_Free_Kick_Upper:
    case SSPM_Their_Free_Kick_Mid:
    case SSPM_Their_Free_Kick_Lower:
    case SSPM_Their_Goalie_Catch:
      starting_paths[i] = NULL;
      break;
       
    case SSPM_My_Kick_Off:
    case SSPM_My_Kick_In_Def:
    case SSPM_My_Kick_In_Mid:
    case SSPM_My_Kick_In_Off:
    case SSPM_My_Corner_Kick:
    case SSPM_My_Goal_Kick:
    case SSPM_My_Free_Kick_Upper:
    case SSPM_My_Free_Kick_Mid:
    case SSPM_My_Free_Kick_Lower:
    case SSPM_My_Goalie_Catch:
      //cout << "allocating mode: " << i << endl;
      starting_paths[i] = new ball_path_t[CoachParam::instance()->getSppHillclimbNumPaths()];
      for (int path_idx=0; path_idx < CoachParam::instance()->getSppHillclimbNumPaths(); path_idx++)
	starting_paths[i][path_idx].num_pts = -1; //-1 because nothing is allocated	
      break;
     
    default:
      errorlog << "What kind of set play mode is this? " << i << ende;
      break;
    }
    
  }
  

  //read in the starting paths
  std::ifstream infile(CoachParam::instance()->getSppHillclimbDefFile().c_str());
  if (!infile) {
    errorlog << "Could not open default waypoint file: '"
	     << CoachParam::instance()->getSppHillclimbDefFile()
	     << "'" << ende;
    return;
  }

  while (infile.good()) {
    if (!skip_white_space(infile))
      break;
    if (!skip_to_non_comment(infile))
      break;
    if (!skip_white_space(infile))
      break;
    //read a line and add it to the appropriate path
    char mode[100];
    int sp_idx, path_idx;
    
    infile.getline(mode, 100, ':');
    if (!infile) {
      errorlog << "Failed reading the set play mode from " << CoachParam::instance()->getSppHillclimbDefFile() << ende;
      break;
    }

    sp_idx = (int)GetSpecSetPlayMode(mode);
    if (sp_idx < 0) {
      errorlog << "Didn't understand SetPlayMode: '" << mode << "' skipping line" << ende;
      skip_line(infile);
      continue;
    } 
    if (sp_idx == SPM_No_Mode) {
      errorlog << "Why is NoMode here! '" << mode << "' " << sp_idx << ende;
    }
    if (starting_paths[sp_idx] == NULL) {
      errorlog << "I didn't allocate for that set play mode! '" << mode << "' " << sp_idx << ende;
      skip_line(infile);
      continue;
    } 

    //find the first unallocated path
    for (path_idx = 0; path_idx < CoachParam::instance()->getSppHillclimbNumPaths(); path_idx++) {
      if (starting_paths[sp_idx][path_idx].num_pts == -1)
	break;
    }
      
    if (path_idx > CoachParam::instance()->getSppHillclimbNumPaths()) {
      errorlog << "Too many paths for mode '" << mode << "' " << sp_idx << ende;
      skip_line(infile);
      continue;
    }

    infile >> starting_paths[sp_idx][path_idx].num_pts;
    if (starting_paths[sp_idx][path_idx].num_pts > CoachParam::instance()->getSppHillclimbMaxLen()) {
      errorlog << "Path for mode '" << mode << "'(" << sp_idx << ") too long, "
	       << starting_paths[sp_idx][path_idx].num_pts << ende;
      continue;
    }
    starting_paths[sp_idx][path_idx].pt_list = 
      new VecPosition[starting_paths[sp_idx][path_idx].num_pts];
    starting_paths[sp_idx][path_idx].offsides_line = NULL; //don't allocate this!
    for (int i=0; i < starting_paths[sp_idx][path_idx].num_pts; i++) {
      infile >> starting_paths[sp_idx][path_idx].pt_list[i] ;
      if (my_side == TS_Right) {
	starting_paths[sp_idx][path_idx].pt_list[i] = 
	  starting_paths[sp_idx][path_idx].pt_list[i].flipCoords(true, false) ;	
      }
    }
    //now let's insure that all the interior points are passes so we don't confuse
    //our hillcimbing steps
    for (int i=1; i < starting_paths[sp_idx][path_idx].num_pts; i++) {
      float len = starting_paths[sp_idx][path_idx].pt_list[i].getMagnitude();
      if (len > CoachParam::instance()->getSppMaxPassDist()) {
	if (i == starting_paths[sp_idx][path_idx].num_pts-1) {
	  //this is okay because this is the send at the end
	  //let's make sure that send is not too long
	  if (len > CoachParam::instance()->getSppMaxSendDist()) {
	    actionlog(50) << "Starting path point " << i << " for path " << path_idx
			  << " for setplay " << sp_idx << " is too long a send " << len
			  << " shortening" << ende;
	    //we substract 1 just to avoid floating point errors
	    starting_paths[sp_idx][path_idx].pt_list[i] = 
	      starting_paths[sp_idx][path_idx].pt_list[i].scaleTo(CoachParam::instance()->getSppMaxSendDist() - 1.0);
	  }
	} else {
	  errorlog << "Starting path point " << i << " for path " << path_idx
		   << " for setplay " << sp_idx << " is not a pass(long)! " << len << ende;
	  //remove this path so we can use it for the next path
	  delete [] starting_paths[sp_idx][path_idx].pt_list;
	  starting_paths[sp_idx][path_idx].num_pts = -1;
	  i = 1000; //break out of the loop
	}
      } else if (len < CoachParam::instance()->getSppMinPassDist()) {
	errorlog << "Starting path point " << i << " for path " << path_idx
		 << " for setplay " << sp_idx << " is not a pass(short)! " << len << ende;
	//remove this path so we can use it for the next path
	delete [] starting_paths[sp_idx][path_idx].pt_list;
	starting_paths[sp_idx][path_idx].num_pts = -1;
	i = 1000; //break out of the loop
      }
    }

    actionlog(150) << "HillClimbWP: Loaded a default path for _mode " << sp_idx << " "  << mode << ende;
    
  } //while (infile)
  
  //allocate the paths for hillclimbing
  paths = new ball_path_t[CoachParam::instance()->getSppHillclimbNumPaths()];
  for (int path_idx = 0; path_idx < CoachParam::instance()->getSppHillclimbNumPaths(); path_idx++) {
      paths[path_idx].num_pts = -1;
      paths[path_idx].pt_list = new VecPosition[CoachParam::instance()->getSppHillclimbMaxLen()];
      paths[path_idx].offsides_line = new float[CoachParam::instance()->getSppHillclimbMaxLen()];
  }
  
  //copy the path_eval pointer
  path_eval = pe;

  //set the shrunkField var
  shrunkField = ServerParam::instance()->getSPFieldRectangle().shrink(CoachParam::instance()->getSppSidelineBuffer());

  //set the efficiency vars
  min_pass_dist2 = Sqr(CoachParam::instance()->getSppMinPassDist());
  max_pass_dist2 = Sqr(CoachParam::instance()->getSppMaxPassDist());
  max_send_dist2 = Sqr(CoachParam::instance()->getSppMaxSendDist());

}

HillClimbWP::~HillClimbWP()
{
  for (int sp_idx=0; sp_idx<num_SpecSetPlayMode; sp_idx++) {
    if (!starting_paths[sp_idx])
      continue;
    for (int path_idx=0; path_idx < CoachParam::instance()->getSppHillclimbNumPaths(); path_idx++) {
      if (starting_paths[sp_idx][path_idx].num_pts >= 0)
	delete [] starting_paths[sp_idx][path_idx].pt_list;
    }
    delete [] starting_paths[sp_idx];
  }
  
  if (paths != NULL) {
    for (int i=0; i<CoachParam::instance()->getSppHillclimbNumPaths(); i++) {
      delete [] paths[i].pt_list;
      delete [] paths[i].offsides_line;
    }
  } else {
    errorlog << "HillClimbWP: destructor: paths is NULL" << ende;
  }
  
  delete [] paths;
  delete path_eval;
}

void HillClimbWP::SetOffsidesLines(Problem* theProb, PlayerDistribution* pDist, 
				   OppModel* pOM, int num_paths)
{
  BallMovement bm;
  
  for (int path_idx = 0; path_idx < num_paths; path_idx++) {
    WorkerPD.copyFrom(pDist);
    VecPosition curr_pos = paths[path_idx].pt_list[0];
    bm.clear();
    bm.addBallPosition(paths[path_idx].pt_list[0], 0);
    for (int pt_idx = 0; 
	 pt_idx < paths[path_idx].num_pts;
	 pt_idx++) {
      
      //This is really the work we want done
      if (theProb->mode == SPM_My_Kick_Off && pt_idx == 0) {
	//force players to start on the correct half of the field for kickoffs
	paths[path_idx].offsides_line[pt_idx] = 0;
      } else {
	// here we move the offsides line in the field to avoid getting confused
	// by later adjustments
	paths[path_idx].offsides_line[pt_idx] =
	  shrunkField.adjustToWithin(VecPosition(GetOffsidesLine(my_side, &WorkerPD, curr_pos),0)).getX();
	actionlog(250) << "Set path " << path_idx << " offsides line " << pt_idx << " to "
		       << paths[path_idx].offsides_line[pt_idx] << ende;
      }
      
      if (pt_idx == paths[path_idx].num_pts - 1)
	break;
      
      //This is just the pos and player dist increment
      VecPosition new_pos = curr_pos + paths[path_idx].pt_list[pt_idx+1];
      int movement_time = 
	AvgTimeForKickToPoint(curr_pos, new_pos,
			      CoachParam::instance()->getSppPassTargetSpeed());

      bm.addBallPosition(new_pos, movement_time);
      pOM->predictMovement(&WorkerPD, &WorkerPD, &bm);
      //this will shift over the thing in location 1
      bm.removeBallPosition(0);

      curr_pos = new_pos;
    }
  }
}

void HillClimbWP::EvalAllPaths(Problem* theProb, PlayerDistribution* pDist, 
			       OppModel* pOM, int num_paths)
{
  bestPath = -1; 
  bestPathValue = -1.0;
  
  for (int path_idx = 0; path_idx < num_paths; path_idx++) {
    //check to see if this is the best path so far
    float pathVal = path_eval->evaluate_path(&paths[path_idx], pDist, pOM, NULL);
    paths[path_idx].value = pathVal;
    if (bestPathValue < pathVal) {
      bestPathValue = pathVal;
      bestPath = path_idx;
    }
  }
  actionlog(200) << "HCCHECK: Eval all best is " << bestPath << " with val "
		 << bestPathValue << ende;
}


void HillClimbWP::MoveAllPointsToValid(int path_idx)
{
  VecPosition curr_pos = paths[path_idx].pt_list[0];
  for (int pt_idx = 1; 
       pt_idx < paths[path_idx].num_pts;
       pt_idx++) {

    Rectangle onside_rect(shrunkField);
    //we don't want to adjust sends onside
    if (paths[path_idx].pt_list[pt_idx].getMagnitude2() > max_pass_dist2) {
      actionlog(240) << "Not shifting onsides; pt " << pt_idx << " of path " << path_idx
		     << ", send " << paths[path_idx].pt_list[pt_idx].getMagnitude2()
		     << " > " << max_pass_dist2
		     << ende;
    } else {
      if (my_side == TS_Left)
	onside_rect.setRight(paths[path_idx].offsides_line[pt_idx-1]);
      else
	onside_rect.setLeft(paths[path_idx].offsides_line[pt_idx-1]);
    }
    
    VecPosition orig_pos = curr_pos + paths[path_idx].pt_list[pt_idx];
    
    // we shrink this very slightly to move all the points cleanly inside the field
    // this prevents weirdo rounding errors  resulting in errors like:
    // Path 1 still has 0 value after shift!
    onside_rect = onside_rect.shrink(.001);
    VecPosition new_pos = onside_rect.adjustToWithin(orig_pos);
    
    VecPosition new_traj = new_pos - curr_pos;
    actionlog(240) << "Shifting pt " << pt_idx << " of path " << path_idx << ", "
		   << orig_pos.getX() << " -> "
		   << new_pos.getX()
		   << ende;

    // Note that sends could be in here, but we'll never have a problem that
    // a send becomes a too short pass
    if (new_traj.getMagnitude2() < min_pass_dist2) {
      actionlog(240) << "Shifting point " << pt_idx << " of path " << path_idx
		     << " makes length too short " << new_traj.getMagnitude2() << " < " << min_pass_dist2
		     << ende;
      std::vector<VecPosition> vsol = 
	onside_rect.circleIntersect(Circle(curr_pos, CoachParam::instance()->getSppMinPassDist()));

      if (vsol.empty()) {
	errorlog << "HillClimbWP::MoveAllPointsToValid: how can there be no solutions? "
		 << paths[path_idx].offsides_line[pt_idx-1] << " "
		 << CoachParam::instance()->getSppMinPassDist() << " " 
		 << curr_pos << " "
		 << orig_pos << " "
		 << new_pos << " "
		 << new_traj.getMagnitude() << ende;
	/*
	  cout << "info: " 
	  << path_idx 
	  << curr_pos
	  << (curr_pos + paths[path_idx].pt_list[pt_idx])
	  << PtToOnsidePt(my_side, curr_pos + paths[path_idx].pt_list[pt_idx], paths[path_idx].offsides_line[pt_idx-1])
	  << (new_pos - curr_pos) 
	  << endl;
	*/
      } else {
	VecPosition best_point(10000,10000);
	float best_dist2 = std::numeric_limits<float>::max();
	for (std::vector<VecPosition>::iterator iter = vsol.begin();
	     iter != vsol.end();
	     ++iter) {
	  float this_dist2 = iter->getDistanceTo2(orig_pos);
	  
	  if (this_dist2 < best_dist2) {
	    best_point = *iter;
	    best_dist2 = this_dist2;
	  }
	}

	new_pos = best_point;
	actionlog(240) << "New pt is " << new_pos << ende;
	new_traj = new_pos - curr_pos;
      }
      
    }

    paths[path_idx].pt_list[pt_idx] = new_traj;
    
    curr_pos = new_pos;
  }
}



int HillClimbWP::LoadPathsForProblem(Problem* theProb, PlayerDistribution* pDist,
				     OppModel* pOM)
{
  bool flipY;
  int num_paths = 0;
  BallMovement bm;
  VecPosition curr_pos;
  int path_idx, pt_idx;

  //The default waypoint are for bottom plans. If this is a top plan, let's flip
  switch (theProb->mode) {
  case SPM_My_Free_Kick:
    //this might be made into a kick_in
    if (theProb->spec_mode == SSPM_My_Kick_In_Def ||
	theProb->spec_mode == SSPM_My_Kick_In_Mid ||
	theProb->spec_mode == SSPM_My_Kick_In_Off) 
      flipY = true;
    else 
      flipY = false;
    break;

  case SPM_My_Kick_Off:
  case SPM_My_Goalie_Catch:
    //these plans are not sided
    flipY = false;
    break;

  case SPM_My_Kick_In:
  case SPM_My_Corner_Kick:
  case SPM_My_Goal_Kick:
    if (theProb->ball_pos.getY() < 0)
      flipY = true;
    else
      flipY = false;
    break;

  case SPM_No_Mode:
  default:
    flipY = false;
    errorlog << "What the theProb->mode " << theProb->mode << ende;
    break;
  }

  actionlog(200) << "Finished calculating flips" << ende;

  if (starting_paths[theProb->spec_mode] == NULL)
    errorlog << "There aren't any starting paths for mode "
	     << GetSpecSetPlayModeString(theProb->spec_mode) << "("
	     << theProb->spec_mode << ")"
	     << ende;
  
  for (path_idx = 0; path_idx < CoachParam::instance()->getSppHillclimbNumPaths(); path_idx++) {
    if (starting_paths[theProb->spec_mode][path_idx].num_pts <= 0) {
      //we've gotten all the paths
      //clear out the remaining paths
      for (int i=num_paths; i < CoachParam::instance()->getSppHillclimbNumPaths(); i++)
	paths[i].num_pts = 0;
      break;
    }
    
    actionlog(200) << "Starting load of path " << path_idx << ende;
    //actionlog(250) << "curr 1 num_paths: " << num_paths << ende;
     
    paths[num_paths].num_pts = starting_paths[theProb->spec_mode][path_idx].num_pts;
    
    if (theProb->mode == SPM_My_Goalie_Catch)
      paths[num_paths].pt_list[0] = starting_paths[theProb->spec_mode][path_idx].pt_list[0];
    else
      paths[num_paths].pt_list[0] = theProb->ball_pos;
    
    //we want to make sure that our points are on the field
    curr_pos = paths[num_paths].pt_list[0];
    for (pt_idx = 1; 
	 pt_idx < paths[num_paths].num_pts;
	 pt_idx++) {
      
      VecPosition next_pos = 
	curr_pos + 
	starting_paths[theProb->spec_mode][path_idx].pt_list[pt_idx].flipCoords(false, flipY);

      bool orig_pass = ( (next_pos - curr_pos).getMagnitude2() < max_pass_dist2);

      actionlog(200) << "Adjusting path " << path_idx << ende;
    
      next_pos = shrunkField.adjustToWithin(next_pos);
      paths[num_paths].pt_list[pt_idx] = next_pos - curr_pos;
      if (orig_pass && paths[num_paths].pt_list[pt_idx].getMagnitude2() < min_pass_dist2) {
	actionlog(70) << "Throwing out path " << path_idx
		      << " because pass too short now: " << pt_idx << ", "
		      << paths[num_paths].pt_list[pt_idx].getMagnitude2()
		      << ende;
	num_paths--; //this will be ++ in a second
	pt_idx += 1000;
      }
      
      curr_pos = next_pos;
    }

    //actionlog(250) << "curr 2 num_paths: " <<  num_paths << ende;

    num_paths++;
  }
  actionlog(250) << "loaded num_paths: " << num_paths << ende;
  DEBUG_MSG(cout << "First part of load complete" << endl);
  DEBUG_MSG(print_all_paths());
  
  //NOTE: We're only going to set offsides, then revise positions once,
  //even though both offsides and valid
  //locations rely on each other. One will hopefully be good enough
  SetOffsidesLines(theProb, pDist, pOM, num_paths);

  for (path_idx = 0; path_idx < num_paths; path_idx++) {

    //shift positions to onside positions
    //we actually allow target position to be offsides, but we'll let the hillclimbing
    // handle that
    MoveAllPointsToValid(path_idx);
  }

  EvalAllPaths(theProb, pDist, pOM, num_paths);

#ifndef NO_ACTION_LOG
  for (path_idx = 0; path_idx < num_paths; path_idx++) {
    actionlog(220) << "loaded path " << path_idx << " of " << paths[path_idx].num_pts
		   << " steps has value " << paths[path_idx].value
		   << ende;
  }
#endif  

  return num_paths;
}

bool HillClimbWP::run(Problem* theProb, 
		      PlayerDistribution* pDist, GetPlayerDistribution* pGetPD, 
		      WaypointList* output)
{
  if (pTimeLeft && *pTimeLeft == TL_Abort) {
    actionlog(60) << "HillClimbWP: aborting at beginning of computation" << ende;
    return false;
  }

  //check that this is an offensive set play
  if (!IsSetPlayOffensive(theProb->mode)) {
    actionlog(60) << "HillClimbWP: not an offensive set play" << ende;
    return false;
  }

  //printf("HillClimbWP::run, my this is 0x%x\n", this);  
  OppModel* pOM = pmSetplayPlan->getBestOppModel();
  if (pOM == NULL) {
    errorlog << "Why Could I not get a best opponent model?" << ende;
    return false;
  }
  
  int numNoMoves = 0; //number of times in a row we haven't moved a point
  int numSteps = 0; //number of hillclimbing steps we have done
  int path_idx, pt_idx;
  BallMovement bm;
  VecPosition curr_pos;
  int num_paths;

  actionlog(60) << "HillClimbWP: about to load seeds" << ende;

  num_paths = LoadPathsForProblem(theProb, pDist, pOM);
  if (num_paths <= 0) {
    errorlog << "Why couldn't I get any seeds here " << theProb->mode << ende;
    return false;
  }

  DEBUG_MSG(cout << "loaded seeds" << endl);
  DEBUG_MSG(print_all_paths());
  
  actionlog(60) << "HillClimbWP: loaded the preset plans, now climbing" << ende;

  /* While we are not in hurry mode and while we are still accomplishing something
     do a hillclimb on a node in a path */
  if (CoachParam::instance()->getSppHillclimb()) {
    while(1) {

      if (CoachParam::instance()->getSppHillclimbMaxSteps() >= 0 &&
	  numSteps > CoachParam::instance()->getSppHillclimbMaxSteps()) {
	actionlog(50) << "HillClimbWP: got to max_steps, ending after " << numNoMoves
		      << ", " << numSteps << " steps" << ende;
	break;
      }

      if (numNoMoves > CoachParam::instance()->getSppHillclimbMaxStatSteps()) {
	actionlog(50) << "HillClimbWP: stationary for " << numNoMoves
		      << " steps, ending after " << numSteps << " steps" << ende;
	break;
      }

      if (pTimeLeft && (*pTimeLeft == TL_Hurry || *pTimeLeft == TL_Abort)) {
	actionlog(50) << "HillClimbWP: got hurry or abort message: "
		      << *pTimeLeft << ", ending after " << numSteps <<" steps"
		      << ende;
	break;
      }

      /* now check to see if we need to resample the player distribution */
      if (CoachParam::instance()->getSppHillclimbResampleFreq() > 0 &&
	  (numSteps + 1) % CoachParam::instance()->getSppHillclimbResampleFreq() == 0) {

	actionlog(200) << "HCCHECK: Resampling player distribution after " << numSteps << " steps" << ende;
	pGetPD->SetPositions(pDist);
	
	/* Now we need to set all the offsides lines and reeval paths */
	actionlog(220) << "Resampling: resetting offsides lines" << ende;
	SetOffsidesLines(theProb, pDist, pOM, num_paths);
	actionlog(220) << "Resampling: reeval paths" << ende;
	EvalAllPaths(theProb, pDist, pOM, num_paths);

	/* let's make sure we didn't put paths in a flat 0 region by 
	   the offsides shifting */
	for (path_idx = 0; path_idx < num_paths; path_idx++) {
	  if (paths[path_idx].value == 0.0) {
	    actionlog(210) << "Resampling: shifting path " << path_idx << " back onsides" << ende;
	    MoveAllPointsToValid(path_idx);
	    float pathVal = path_eval->evaluate_path(&paths[path_idx], pDist, pOM, NULL);
	    paths[path_idx].value = pathVal;
	    if (pathVal == 0.0) {
	      errorlog << "Path " << path_idx << " still has 0 value after shift!" << ende;
	    }
	  }
	} //path_idx
      } //resample player distribution
      
      if (pTimeLeft && *pTimeLeft <= TL_MidWay1) {
	actionlog(240) << "At midway point, only climbing best path " << bestPath << ende;
	path_idx = bestPath;
      } else {
	path_idx = int_random(num_paths);
      }
      pt_idx = int_random(paths[path_idx].num_pts - 1) + 1; //don't want to get 0
    

      //climb this pt
      /* First get the surronding values: 
	 0 1 2
	 3 4 5
	 6 7 8 */
      bool only_climb_one = ((numSteps % 2) == 0);
      float bestNeighValue = paths[path_idx].value;
      VecPosition bestDisp(0,0);
      float bestNewOffsides[CoachParam::instance()->getSppHillclimbMaxLen()];

      bool move = false;
      VecPosition orig_val = paths[path_idx].pt_list[pt_idx];
      bool orig_pass = !(orig_val.getMagnitude2() > max_pass_dist2);
      VecPosition orig_next_val(0,0);
      if (only_climb_one && pt_idx < paths[path_idx].num_pts - 1)
	orig_next_val = paths[path_idx].pt_list[pt_idx+1];
      else 
	//if it doesn't exist, we just want a value that's not going to violate pass lens
	orig_next_val = VecPosition(CoachParam::instance()->getSppMaxPassDist() - 5, 0);
      bool orig_next_pass = !(orig_next_val.getMagnitude2() > max_pass_dist2);

      for (int neigh = 0; neigh < 9; neigh++) {
	if (neigh == 4)
	  continue; //we already have our current value;
	VecPosition disp( (neigh % 3) - 1, (neigh / 3) - 1);
	VecPosition newpt = orig_val + disp;
	VecPosition newpt_next = orig_next_val - disp;

	//we'll leave in field checking for the evaluation function
	//we'll leave the offsides checking to the eval function also
      
	//check that we don't make a pass into a send or shrink a pass too much
	float len2 = newpt.getMagnitude2();
	if (orig_pass) {
	  if (len2 < min_pass_dist2) {
	    actionlog(240) << "neigh " << neigh << " makes pass too small " << len2 << ende;
	    continue; //too small a pass
	  }
	  if (len2 > max_pass_dist2) {
	    actionlog(240) << "neigh " << neigh << " makes pass too long " << len2 << ende;
	    continue; //too long a pass
	  }
	} else {
	  //max_pass is the same as min_send
	  if (len2 < max_pass_dist2) {
	    actionlog(240) << "neigh " << neigh << " makes send too small " << len2 << ende;
	    continue; //too small a send
	  }
	  if (len2 > max_send_dist2) {
	    actionlog(240) << "neigh " << neigh << " makes send too long " << len2 << ende;
	    continue; //too long a send
	  }
	}
	if (only_climb_one) {
	  len2 = newpt_next.getMagnitude2();
	  if (orig_next_pass) {
	    if (len2 < min_pass_dist2) {
	      actionlog(240) << "neigh " << neigh << " makes next pass too small " << len2 << ende;
	      continue; //too small a pass
	    }
	    if (len2 > max_pass_dist2) {
	      actionlog(240) << "neigh " << neigh << " makes next pass too long " << len2 << ende;
	      continue; //too long a pass
	    }
	  } else {
	    //max_pass is the same as min_send
	    if (len2 < max_pass_dist2) {
	      actionlog(240) << "neigh " << neigh << " makes next send too small " << len2 << ende;
	      continue; //too small a send
	    }
	    if (len2 > max_send_dist2) {
	      actionlog(240) << "neigh " << neigh << " makes next send too long " << len2 << ende;
	      continue; //too long a send
	    }
	  }
	} //only_climb one
	
	paths[path_idx].pt_list[pt_idx] = newpt;
	if (only_climb_one)
	  paths[path_idx].pt_list[pt_idx+1] = newpt_next;
      
	float new_offsides[CoachParam::instance()->getSppHillclimbMaxLen()];
	float pathVal = path_eval->evaluate_path(&paths[path_idx], 
						 pDist, pOM, new_offsides);
	if (bestNeighValue < pathVal) {
	  bestNeighValue = pathVal;
	  bestDisp = disp;
	  move = TRUE;
	  for (int i=0; i <paths[path_idx].num_pts; i++)
	    bestNewOffsides[i] = new_offsides[i];
	}
      }

      if (bestPathValue < bestNeighValue) {
	actionlog(200) << "HCCHECK: new best path (" << numSteps << " steps) is "
		       << path_idx << " (old: " << bestPath << "), value "
		       << bestNeighValue << " (old: " << bestPathValue << ")"
		       << ende;
	bestPathValue = bestNeighValue;
	bestPath = path_idx;
      }

      numNoMoves = (move) ? 0 : (numNoMoves+1);
    
      numSteps++;
    
      //now actually change the path
      if (move) {
	DEBUG_MSG(paths[path_idx].pt_list[pt_idx] = orig_val);
	DEBUG_MSG(paths[path_idx].pt_list[pt_idx+1] = orig_next_val);
	DEBUG_MSG(cout << "old path: "; print_path(&paths[path_idx]));
	DEBUG_MSG(cout << "disp: " << bestDisp << " at " << pt_idx << " (" << only_climb_one << ") " << endl);
	DEBUG_MSG(cout << "origval: " << orig_val << "\tnewval: " << (orig_val + bestDisp) << endl);
	paths[path_idx].pt_list[pt_idx] = orig_val + bestDisp;
	if (only_climb_one)
	  paths[path_idx].pt_list[pt_idx+1] = orig_next_val - bestDisp;
	DEBUG_MSG(cout << "new path: "; print_path(&paths[path_idx]));
	
	paths[path_idx].value = bestNeighValue;
	int i;
	if (theProb->mode == SPM_My_Kick_Off) {
	  //force players to start on the correct half of the field for kickoffs
	  paths[path_idx].offsides_line[0] = 0;
	  i=1;
	} else {
	  i=0;
	}
	for (; i <paths[path_idx].num_pts; i++)
	  paths[path_idx].offsides_line[i] = bestNewOffsides[i];
      } else {
	paths[path_idx].pt_list[pt_idx] = orig_val;
	if (only_climb_one)
	  paths[path_idx].pt_list[pt_idx+1] = orig_next_val;
      }
      
    
      actionlog(230) << "Hillclimbed " << path_idx << "," << pt_idx
		     << " with displacement " << bestDisp << ", newval = "
		     << paths[path_idx].value << ende;
      
      if (numSteps % 100 == 1) {
	actionlog(90) << "Hillclimbing " << numSteps << " steps, bestPath is " << bestPath
		      << " with value " << bestPathValue << ende;
      }

      /*
      sigaction ioact, alrmact;
      sigaction(SIGIO, NULL, &ioact);
      sigaction(SIGALRM, NULL, &alrmact);
      printf("steps: %d, iohandler: %x, alrmhandler: %x", numSteps, ioact.sa_handler, alrmact.sa_handler);
      */

    }
  }
  
  //DONE WITH HILLCLIMBING!
  cout << "Hillclimbing steps: " << numSteps << endl;

  if (CoachParam::instance()->getSppHillclimbAlwaysFirst()) {
    actionlog(50) << "Changing to use the first path regardless of eval" << ende;
    bestPath = 0;
  }
  
  if (bestPath >= 0) {
    actionlog(70) << "HillClimbWP (steps " << numSteps << "): best path is "
		  << bestPath << ", with value " << bestPathValue << ende;
  } else {
    errorlog << "Why isn't there a best path: " << bestPath << ende;
  }

  output->Clear();
  
  /* Copy the waypoints and probability distributions to the output */
  if (!output->SetWaypoint(0, paths[bestPath].pt_list[0], pDist))
    errorlog << "HCWP: failed to write waypoint 0" << ende;
  
  curr_pos = paths[bestPath].pt_list[0];
  bm.clear();
  bm.addBallPosition(paths[bestPath].pt_list[0], 0);
  
  for (int i=1; i<paths[bestPath].num_pts; i++) {
    VecPosition new_pos = curr_pos + paths[bestPath].pt_list[i];
    int movement_time = 
      AvgTimeForKickToPoint(curr_pos, new_pos,
			    CoachParam::instance()->getSppPassTargetSpeed());

    bm.addBallPosition(new_pos, movement_time);
    
    pOM->predictMovement(output->GetPlayDist(i-1), output->GetPlayDist(i), &bm);

    //this will shift over the thing in location 1
    bm.removeBallPosition(0);

    if (!output->SetWaypoint(i, new_pos))
      errorlog << "HCWP: failed to write waypoint " << i << ende;
    
    curr_pos = new_pos;
  }
  
  return true;
}

void HillClimbWP::PrintStartingPaths(bool show_absolute)
{
  VecPosition pos;
  
  for (int sp_idx=0; sp_idx<num_SpecSetPlayMode; sp_idx++) {
    if (!starting_paths[sp_idx])
      continue;
    cout << "SetPlayMode: " << GetSpecSetPlayModeString((SpecSetPlayMode)sp_idx) 
	 << " (" <<  sp_idx << ") " 
	 << endl;
    for (int path_idx=0; path_idx < CoachParam::instance()->getSppHillclimbNumPaths(); path_idx++) {
      if (starting_paths[sp_idx][path_idx].num_pts > 0) {
	pos = VecPosition(0,0);
	for (int pt_idx=0; pt_idx < starting_paths[sp_idx][path_idx].num_pts; pt_idx++) {
	  pos += starting_paths[sp_idx][path_idx].pt_list[pt_idx];
	  if (show_absolute)
	    cout << pos << " ";
	  else
	    cout << starting_paths[sp_idx][path_idx].pt_list[pt_idx] << " ";
	}
	cout << endl;
      }
    }
  }
}

void HillClimbWP::DrawStartingPaths(const char* dir)
{
  char fn[MAX_FILE_LEN];
  //FieldImage fi(1);
  FieldImage fi(1,6.0);
  FieldImage::Color cPass(FieldImage::COLOR_GREEN);
  FieldImage::Color cSend(FieldImage::COLOR_BLUE);
  VecPosition curr_pos;

  cout << "Drawing waypoints..." << endl;
  
  for (int sp_idx=0; sp_idx<num_SpecSetPlayMode; sp_idx++) {
    if (!starting_paths[sp_idx])
      continue;
    
    cout << "\tSetPlayMode: " << GetSpecSetPlayModeString((SpecSetPlayMode)sp_idx) 
	 << " (" <<  sp_idx << ") " 
	 << endl;
    for (int path_idx=0; path_idx < CoachParam::instance()->getSppHillclimbNumPaths(); path_idx++) {
      if (starting_paths[sp_idx][path_idx].num_pts <= 0)
	continue;

      switch (sp_idx) {
      case SSPM_My_Kick_Off:
	curr_pos = VecPosition(0,0); 
	break;
      case SSPM_My_Kick_In_Def:
	curr_pos = VecPosition(-30, ServerParam::instance()->getSPPitchWidth()/2); 
	break;
      case SSPM_My_Kick_In_Mid:
	curr_pos = VecPosition(0, ServerParam::instance()->getSPPitchWidth()/2); 
	break;
      case SSPM_My_Kick_In_Off:
	curr_pos = VecPosition(25, ServerParam::instance()->getSPPitchWidth()/2); 
	break;
      case SSPM_My_Corner_Kick:
	curr_pos = VecPosition(ServerParam::instance()->getSPPitchLength()/2, ServerParam::instance()->getSPPitchWidth()/2); 
	break;
      case SSPM_My_Goal_Kick:
	curr_pos = VecPosition(-ServerParam::instance()->getSPPitchLength()/2+ServerParam::instance()->getSPGoalAreaLength(), ServerParam::instance()->getSPGoalAreaWidth()/2); 
	break;
      case SSPM_My_Free_Kick_Upper:
	curr_pos = VecPosition(-30, -20);
	break;
      case SSPM_My_Free_Kick_Mid:
	curr_pos = VecPosition(-30, 0);
	break;
      case SSPM_My_Free_Kick_Lower:
	curr_pos = VecPosition(-30, 20);
	break;
      case SSPM_My_Goalie_Catch:
	curr_pos = starting_paths[sp_idx][path_idx].pt_list[0];
	break;
      default:
	errorlog << "What is sp_idx: " << sp_idx << ende;
	break;
      }

      fi.erase();
      fi.addFieldLines();
      
      //we initialize pt_idx in the switch above
      for (int pt_idx=1; pt_idx < starting_paths[sp_idx][path_idx].num_pts; pt_idx++) {
	VecPosition next_pos = curr_pos + starting_paths[sp_idx][path_idx].pt_list[pt_idx];
	bool is_pass = curr_pos.getDistanceTo2(next_pos) <= path_eval->max_pass_dist2;
	fi.addArrow(curr_pos, next_pos, is_pass ? cPass : cSend);
	curr_pos = next_pos;
      }
      
      //sprintf(fn, "%s/%s-%d.gif", dir, GetSpecSetPlayModeString((SpecSetPlayMode)sp_idx), path_idx);
      sprintf(fn, "%s/%s-%d", dir, GetSpecSetPlayModeString((SpecSetPlayMode)sp_idx), path_idx);
      fi.addLegendLine(fn);
      // 9 is the compression level
      fi.writeTo(fn, 9);

    }
  }
}


void HillClimbWP::print_all_paths()
{
  for (int path_idx=0; path_idx < CoachParam::instance()->getSppHillclimbNumPaths(); path_idx++) {
    if (paths[path_idx].num_pts > 0) {
      print_path(&paths[path_idx]);
    }
  }
}



/************************* Level 2: RoleMaker ****************************/

bool SimpleAtPointRM::run(WaypointList* pWaypoints, Problem* theProb, AgentBallMovements* output)
{
  if (pTimeLeft && *pTimeLeft == TL_Abort) {
    actionlog(60) << "SimpleAtPointRM: aborting at beginning of computation" << ende;
    return false;
  }

  output->Clear();
  
  //First let's set the initial positions  
  for (int i=0; i<pWaypoints->GetNumPts(); i++) {
    //We have to watch for the end of send targets
    if ( i==0 || 
	pWaypoints->GetPt(i-1).getDistanceTo(pWaypoints->GetPt(i)) <=
	 CoachParam::instance()->getSppMaxPassDist()) {
      output->AddPlayerLoc(i+1, pWaypoints->GetPt(i), ABMO_NoConstraint);
    } else if (i != pWaypoints->GetNumPts()-1) {
      errorlog << "Why do I have a send in the middle of the waypoints?" << ende;
    }
  }

  //Now we set the initial ball position
  output->AddBallLoc(pWaypoints->GetPt(0));

  int player_num = 1;  
  //Now set each pass or send up
  //We have to set the player pos, then the ball pos to the right location
  for (int i=1; i<pWaypoints->GetNumPts(); i++) {
    if (pWaypoints->GetPt(i-1).getDistanceTo(pWaypoints->GetPt(i)) > 
	CoachParam::instance()->getSppMaxPassDist()) {
      //this is a send
      output->AddBallLoc(pWaypoints->GetPt(i));    
    } else {
      //this is a pass
      //player_num is for the player with the ball
      output->AddPlayerLoc(++player_num, pWaypoints->GetPt(i), ABMO_NoConstraint);
      output->AddBallLoc(pWaypoints->GetPt(i));    
    }
  }  

  return true;  
}

bool AtPointWithReuseRM::run(WaypointList* pWaypoints, Problem* theProb, 
			     AgentBallMovements* output)
{
  if (pTimeLeft && *pTimeLeft == TL_Abort) {
    actionlog(60) << "AtPointWithReuseRM: aborting at beginning of computation" << ende;
    return false;
  }

  Unum agentForPt[MAX_WAYPOINTS];
  //timeForWP[i] = # of cycles (avg) to go from wp[i-1] to wp[i]
  int timeForWP[MAX_WAYPOINTS];
	       
  timeForWP[0] = 0;
  for (int i=1; i < pWaypoints->GetNumPts(); i++) {
    timeForWP[i] = AvgTimeForKickToPoint(pWaypoints->GetPt(i-1), 
					 pWaypoints->GetPt(i),
					 CoachParam::instance()->getSppPassTargetSpeed());
  }
  
  //This is a comepletely greedy strategy, at every waypoint, decide if any earlier
  // agent (other then the one immediately before) could fill this role
  int lastPIdx = 0;
  for (int wpidx=0; wpidx < pWaypoints->GetNumPts(); wpidx++) {
    agentForPt[wpidx] = Unum_Unknown;
    //Look for a previous agent to fill this
    if (wpidx==0 || 
	pWaypoints->GetPt(wpidx-1).getDistanceTo(pWaypoints->GetPt(wpidx)) <=
	CoachParam::instance()->getSppMaxPassDist()) {
      if (wpidx >= 5)
	errorlog << "Watch Out! AtPointwithReuseRM may do weird things with >= 4 passes" << ende;
      int timeForBall = timeForWP[0];
      for (int prev=0; prev<= wpidx-2; prev++) {
	timeForBall += timeForWP[prev];
	if (agentForPt[prev] == agentForPt[wpidx-1])
	  continue; //don't let most recent agent fill it
	//if time for agent < time for ball, use this agent
	//SMURF: UpperB or avg?
	if (UpperBForPlayerToPoint(pWaypoints->GetPt(prev), pWaypoints->GetPt(wpidx)) <=
	    timeForBall) {
	  //we're going to use the agent who did waypoint prev
	  agentForPt[wpidx] = agentForPt[prev];
	  actionlog(60) << "4LPWorker: Reusing agent " << agentForPt[prev] << " from spot "
			<< prev << " for waypoint " << wpidx << ende;
	  break; //we're done with this loop
	}
      } //prev idx loop
      
      if (agentForPt[wpidx] == Unum_Unknown) {
	//assign a new agent
	agentForPt[wpidx] = ++lastPIdx;
	actionlog(60) << "4LPWorker: Assigned a new agent " << lastPIdx << " for waypoint "
		      << wpidx << ende;
      }
    } //this is a pass
  }
  
  output->Clear();

  //Now, we need to generate the AgentBallMovement stuff
  //This is quite similar to the SimpleAtPointRM from above
  /* First let's set the initial positions: loop through agents, at find first
     waypoint at whihc they are used */
  for (int num=1; num<=lastPIdx; num++) {
    for (int wpidx = 0; wpidx <= pWaypoints->GetNumPts(); wpidx++) {
      if (wpidx == pWaypoints->GetNumPts()) {
	errorlog << "I couldn't find where agent " << num << " is involved!" << ende;
      }
      if (agentForPt[wpidx] == num) {
	//This is the first waypoint for this agent
	output->AddPlayerLoc(agentForPt[wpidx], pWaypoints->GetPt(wpidx), 
			     ABMO_NoConstraint);
	break; //break to next player
      }
    }
  }
  
  //Now we set the initial ball position
  output->AddBallLoc(pWaypoints->GetPt(0));

  //Now set each pass or send up
  for (int i=1; i<pWaypoints->GetNumPts(); i++) {
    if (pWaypoints->GetPt(i-1).getDistanceTo(pWaypoints->GetPt(i)) > 
	CoachParam::instance()->getSppMaxPassDist()) {
      //this is a send
      output->AddBallLoc(pWaypoints->GetPt(i));    
    } else {
      //this is a pass
      output->AddPlayerLoc(agentForPt[i], pWaypoints->GetPt(i), ABMO_NoConstraint);
      output->AddBallLoc(pWaypoints->GetPt(i));    
    }
  }  

  return true;  
}

/* the basic algorithm goes like this: Store the locations the roles need to be
   before each waypoint *ends* in a 2d array.
   Step 1: Set the agent's positions to where they receive passes (agent at waypoint i
           is marked for level 
   Step 1.5: add some extra players for various plays
   Step 2: back up values for offsides and for starting players off the point
           Store *why* the point was backed up
   Step 3: try to reuse agents for roles
   Step 3.5: handle kick off (get on correct side of field)
   Step 4: compile into AgentBallMovements
*/
/* note that we use role numbers 0 to 10 */
bool OffPointOffsidesRM::run(WaypointList* pWaypoints, Problem* theProb, 
			     AgentBallMovements* output)
{
  if (pTimeLeft && *pTimeLeft == TL_Abort) {
    actionlog(60) << "OffPointOffsidesRM: aborting at beginning of computation" << ende;
    return false;
  }

  int wp_idx, pnum;
  float offsides_line[MAX_WAYPOINTS];
  int num_roles = 0;
  //VAR: you can change this buffer
  Rectangle shrunkField = ServerParam::instance()->getSPFieldRectangle().shrink(1);
  
  ClearLocInfo();

  //set offsides_lines
  for (wp_idx = 0; wp_idx < pWaypoints->GetNumPts(); wp_idx++) {
    offsides_line[wp_idx] = GetOffsidesLine(my_side,
					    pWaypoints->GetPlayDist(wp_idx),
					    pWaypoints->GetPt(wp_idx));
  }

  /* Step 1 */
  for (wp_idx = 0; wp_idx < pWaypoints->GetNumPts(); wp_idx++) {
    if (wp_idx == 0 ||
	pWaypoints->GetPt(wp_idx-1).getDistanceTo(pWaypoints->GetPt(wp_idx)) <=
	CoachParam::instance()->getSppMaxPassDist()) {
      //we'll assume that all passes are onside and fix it below
      //when backing up locations
      loc_info[wp_idx][wp_idx].type = LT_ReceiverOnside;
      loc_info[wp_idx][wp_idx].pos = pWaypoints->GetPt(wp_idx);
      loc_info[wp_idx][wp_idx].first_pos_set = true;    
      actionlog(90) << "Setting receiver at " << loc_info[wp_idx][wp_idx].pos
		    << ", wp " << wp_idx << ende;
      num_roles = wp_idx+1;
    } else {
      //this is a send, let's put a player in the right position on the offsides line
      if (wp_idx != pWaypoints->GetNumPts() - 1) {
	errorlog << "Why is there a send in the middle of the waypoints " << wp_idx << ende;
      } else {
	loc_info[wp_idx-1][wp_idx].type = LT_JustOff;
	loc_info[wp_idx-1][wp_idx].pos = 
	  PtToOnsidePt(my_side, pWaypoints->GetPt(wp_idx), offsides_line[wp_idx-1]);
	loc_info[wp_idx-1][wp_idx].first_pos_set = true;    
	actionlog(90) << "Putting player on offsides line " << loc_info[wp_idx-1][wp_idx].pos
		      << ", wp " << wp_idx << ende;
	num_roles = wp_idx+1;
      }
    }
  }

  /* Step 1.5 */
  switch (theProb->spec_mode) {
  case SSPM_My_Kick_Off:
  case SSPM_My_Kick_In_Def:
  case SSPM_My_Kick_In_Mid:
  case SSPM_My_Free_Kick_Upper:
  case SSPM_My_Free_Kick_Mid:
  case SSPM_My_Free_Kick_Lower:
    //we don't add extra players for this mode
    break;
    
  case SSPM_My_Kick_In_Off:
    //put one extra player at the end
    loc_info[0][num_roles].type = LT_JustOff;
    loc_info[0][num_roles].pos = 
      pWaypoints->GetPt(pWaypoints->GetNumPts()-1) + 
      VecPosition(0, -signf(theProb->ball_pos.getY())*3);
    loc_info[0][num_roles].first_pos_set = true;
    actionlog(80) << "Added extra kick in pos: " << loc_info[0][num_roles].pos << ende;
    num_roles++;
    break;
    
  case SSPM_My_Corner_Kick:
    switch (pWaypoints->GetNumPts()) {
    case 2:
      for (int i=-1; i<=1; i++) {
	loc_info[0][num_roles].type = LT_JustOff;
	loc_info[0][num_roles].pos = 
	  VecPosition(ServerParam::instance()->getSPPitchLength() / 2 - ServerParam::instance()->getSPPenaltyAreaLength(), 
		 10 * i);
	if (my_side == TS_Right) {
	  loc_info[0][num_roles].pos = loc_info[0][num_roles].pos.flipX();
	}
	loc_info[0][num_roles].first_pos_set = true;
	num_roles++;
      }

      break;
    case 3:
      //put a player at the midpoint between the two last positions;
      loc_info[0][num_roles].type = LT_JustOff;
      loc_info[0][num_roles].pos = 
	VecPosition::getPointInBetween(pWaypoints->GetPt(pWaypoints->GetNumPts()-1),
				       pWaypoints->GetPt(pWaypoints->GetNumPts()-2),
				       .5);
      loc_info[0][num_roles].pos += VecPosition((my_side==TS_Left)?-5:5,0);
      loc_info[0][num_roles].first_pos_set = true;
      actionlog(80) << "Added extra corner kick pos (3): " << loc_info[0][num_roles].pos << ende;
      num_roles++;
      ////FALL THROUGH TO NEXT CASE!
      
    case 4:
      //put one extra player at the end
      loc_info[0][num_roles].type = LT_JustOff;
      loc_info[0][num_roles].pos = 
	pWaypoints->GetPt(pWaypoints->GetNumPts()-1) + 
	VecPosition(0, -signf(theProb->ball_pos.getY())*3);
      loc_info[0][num_roles].first_pos_set = true;
      actionlog(80) << "Added extra corner kick pos (4): " << loc_info[0][num_roles].pos << ende;
      num_roles++;
      break;
    case 5:
      //enough players involved already
      break;
    default:
      errorlog << "How many waypoints for this corner kick? " << pWaypoints->GetNumPts() << ende;
      break;
    }
    
    break;

  case SSPM_My_Goal_Kick:
  case SSPM_My_Goalie_Catch: {
    //Let's bring back some players for safety
    //two players near the top of the box
    for (int i=-1; i<=1; i+= 2) {
      loc_info[0][num_roles].type = LT_JustOff;
      loc_info[0][num_roles].pos = 
	VecPosition(-ServerParam::instance()->getSPPitchLength() / 2 + ServerParam::instance()->getSPPenaltyAreaLength() - 2, 
	       6 * i);
      if (my_side == TS_Right) {
	loc_info[0][num_roles].pos = loc_info[0][num_roles].pos.flipX();
      }
      loc_info[0][num_roles].first_pos_set = true;
      num_roles++;
    }
    /* pfr 6/26/04: added because goalie catches can be especially dangerous for the coachable team */
#define EXTRA_PLAYER_IN_MIDDLE
#ifdef EXTRA_PLAYER_IN_MIDDLE
    loc_info[0][num_roles].type = LT_JustOff;
    loc_info[0][num_roles].pos = 
      VecPosition(-ServerParam::instance()->getSPPitchLength() / 2 + 8, 0);
      if (my_side == TS_Right) {
	loc_info[0][num_roles].pos = loc_info[0][num_roles].pos.flipX();
      }
      loc_info[0][num_roles].first_pos_set = true;
      num_roles++;
#endif	
    /* pfr: 7/24/01: this worked for the 2000 team, but it seems to be causing problems
       now (the extra player gets in the way all the time, so I took it out */
#ifdef EXTRA_PLAYER_AT_SIDE 
    //one player on the side by where the kick is
    float side_sign = 0;
    if (theProb->spec_mode == SSPM_My_Goal_Kick) 
      side_sign = signf(theProb->ball_pos.getY());
    else if (theProb->spec_mode == SSPM_My_Goalie_Catch) 
      side_sign = signf(pWaypoints->GetPt(1).getY());
    else
      errorlog << "What is spec_mode? " << theProb->spec_mode << ende;

    loc_info[0][num_roles].type = LT_JustOff;
    loc_info[0][num_roles].pos = 
      VecPosition(-ServerParam::instance()->getSPPitchLength() / 2 + 10, side_sign * 25);
    if (my_side == TS_Right) {
      loc_info[0][num_roles].pos = loc_info[0][num_roles].pos.flipX();
    }
    loc_info[0][num_roles].first_pos_set = true;
    
    actionlog(70) << "4LPworker: gk/gc adding " << loc_info[0][num_roles].pos
		  << " side_sign: " << side_sign << ende;

    num_roles++;
#endif      

  } break;

  default:
    errorlog << "What kind of spec SPM is this? " << theProb->spec_mode << ende;
    break;
  }
  

  /* Step 2 */
  //Notice the that we loop from the end so that values can be backed up multiple times
  //we don't include 0 because we can't back up to -1
  for (wp_idx = pWaypoints->GetNumPts() - 1; wp_idx > 0; wp_idx--) {
    VecPosition wp = pWaypoints->GetPt(wp_idx);
    for (pnum = 0; pnum < NUM_PLAYERS; pnum++) {
      //we do this first in order to succesfully do the below error check
      if (loc_info[wp_idx][pnum].type == LT_Empty)
	continue;
      
      if (loc_info[wp_idx-1][pnum].type != LT_Empty) {
	errorlog << "I'm trying to back up, but there's something there! "
		 << wp_idx << " " 
		 << pnum << " " 
		 << loc_info[wp_idx-1][pnum].type << ende;
	continue;
      }
      
      switch (loc_info[wp_idx][pnum].type) {
      case LT_Empty:
	continue; //nothing to back up

	//in these two cases we'll only set the previous type and let the actual pos
	// be set below
      case LT_ReceiverOnside:
	if (IsOffsidesPosition(my_side, wp, offsides_line[wp_idx-1])) {
	  loc_info[wp_idx][pnum].type = LT_ReceiverOffside;
	  loc_info[wp_idx-1][pnum].type = LT_Onside;
	} else {
	  loc_info[wp_idx-1][pnum].type = LT_JustOff;
	}
	break;
	
      case LT_Onside: {
	float dist_to_pass = 
	  Line::getDistToClosestPtInBetween(loc_info[wp_idx][pnum].pos, 
					    pWaypoints->GetPt(wp_idx-1),
					    wp);
	if (IsOffsidesPosition(my_side, loc_info[wp_idx][pnum].pos, offsides_line[wp_idx-1]) &&
	    dist_to_pass < ServerParam::instance()->getSPOffsideActiveAreaSize() + CoachParam::instance()->getSppOffsideWorryDist()) {
	  loc_info[wp_idx-1][pnum].type = LT_Onside;
	} else {
	  loc_info[wp_idx-1][pnum].type = LT_JustOff;
	}
      } break;
	
      case LT_ReceiverOffside:
	//we only init to ReceiverOnside
	errorlog << "Why am I examining a ReceivedOffside node for backup?" << ende;
	continue;
	
      case LT_JustOff:
	continue; //once we back off just for fun, we stop
	
      default:
	errorlog << "What is loc_info[" << wp_idx << "][" << pnum << "]: "
		 << loc_info[wp_idx][pnum].type << ende;
	continue;
      }

      switch (loc_info[wp_idx-1][pnum].type) {
      case LT_JustOff: {
	int ball_cyc = MinTimeForKickToPoint(pWaypoints->GetPt(wp_idx-1), wp,
					     CoachParam::instance()->getSppPassTargetSpeed());
	float disp_dist = (ball_cyc - CoachParam::instance()->getSppOffsetDelay()) * 
	  ServerParam::instance()->getSPPlayerSpeedMax() * CoachParam::instance()->getSppOffsetPercSpeed();
	if (disp_dist < 0) 
	  disp_dist = 0;
	double ang;
	if (loc_info[wp_idx][pnum].type == LT_ReceiverOnside) {
	  //if this is a just off for a pass, we want to make sure that we back away from
	  // the pass
	  //we'll try a few times and then just give up and shift on field
	  //VAR: play with the number of tries
	  VecPosition prev_pass_vec = pWaypoints->GetPt(wp_idx-1) - pWaypoints->GetPt(wp_idx);
	  for (int i=0; i < 10; i++) {
	    ang = range_random(-80, 80); 
	    if (my_side == TS_Left)
	      ang += 180;
	    VecPosition disp_vec = VecPosition::getVecPositionFromPolar(disp_dist, ang);
	    loc_info[wp_idx-1][pnum].pos = loc_info[wp_idx][pnum].pos + disp_vec;
	    if (shrunkField.isInside(loc_info[wp_idx-1][pnum].pos) &&
		VecPosition::dotProduct(disp_vec, prev_pass_vec) < 0)
	      break;
	  }
	  
	  if (!shrunkField.isInside(loc_info[wp_idx-1][pnum].pos)) {
	    actionlog(90) << "Role " << pnum << " just off (receiver) wp "
			  << wp_idx << ", dist " << disp_dist << ", ang " << ang
			  << ", moving on field" << ende;
	    loc_info[wp_idx-1][pnum].pos = 
	      shrunkField.adjustToWithin(loc_info[wp_idx-1][pnum].pos);
	  }

	  actionlog(90) << "Starting role " << pnum << " just off (receiver) wp " << wp_idx
			<< ", dist " << disp_dist << ", ang " << ang << ende;
	} else {
	  //we'll try a few times to get the point on the field, then we'll just shift 
	  // it on
	  //VAR: play with the number of times
	  for (int i=0; i < 10; i++) {
	    ang = range_random(-80, 80); 
	    if (my_side == TS_Left)
	      ang += 180;
	    loc_info[wp_idx-1][pnum].pos = 
	      loc_info[wp_idx][pnum].pos + VecPosition::getVecPositionFromPolar(disp_dist, ang);
	    if (shrunkField.isInside(loc_info[wp_idx-1][pnum].pos))
	      break;
	  }
	  
	  if (!shrunkField.isInside(loc_info[wp_idx-1][pnum].pos)) {
	    actionlog(90) << "Role " << pnum << " just off wp " << wp_idx << ", dist " << disp_dist
			  << ", ang " << ang << ", moving on field" << ende;
	    loc_info[wp_idx-1][pnum].pos = 
	      shrunkField.adjustToWithin(loc_info[wp_idx-1][pnum].pos);
	  }
	  
	  actionlog(90) << "Starting role " << pnum << " just off wp " << wp_idx
			<< ", dist " << disp_dist << ", ang " << ang << ende;
	}
      } break;

      case LT_Onside: 
	loc_info[wp_idx-1][pnum].pos = PtToOnsidePt(my_side,
						    loc_info[wp_idx][pnum].pos, 
						    offsides_line[wp_idx-1]);
	break;

      default:
	errorlog << "What am I trying to back up here? " <<  loc_info[wp_idx-1][pnum].type << ende;
	loc_info[wp_idx-1][pnum].type = LT_Empty;
	continue;
      }
      
      loc_info[wp_idx-1][pnum].first_pos_set = true;
      loc_info[wp_idx][pnum].first_pos_set = false;

      actionlog(80) << "Backed up role " << pnum << " from wp " << wp_idx << " (because of "
		    << loc_info[wp_idx-1][pnum].type << " to " << loc_info[wp_idx-1][pnum].pos
		    << ende;

    } //pnum loop
  } //wp_idx loop

  /* Step 3: Agent Reuse */
  if (CoachParam::instance()->getSppReuseAgents()) {
    int timeForWP[MAX_WAYPOINTS];
	       
    timeForWP[0] = 0;
    for (int i=1; i < pWaypoints->GetNumPts(); i++) {
      timeForWP[i] = AvgTimeForKickToPoint(pWaypoints->GetPt(i-1), 
					   pWaypoints->GetPt(i),
					   CoachParam::instance()->getSppPassTargetSpeed());
    }
    /* here's the idea, loop through the roles, and look for a role which 
     could be merged with the current role */
    int role_num;
    //we don't want to resuse the goalie
    if (theProb->mode == SPM_My_Goalie_Catch) 
      role_num = 1;
    else
      role_num = 0;
    for (/*init done above*/; role_num < NUM_PLAYERS; role_num++) {
      bool roleDone = FALSE;
      if (!IsRoleUsed(role_num, pWaypoints->GetNumPts()))
	continue; //can't merge with unused role
      for (int search_role_num = role_num + 2; 
	   search_role_num < NUM_PLAYERS && !roleDone; 
	   search_role_num++) {
	if (!IsRoleUsed(search_role_num, pWaypoints->GetNumPts()))
	  continue; //can't merge with unused role
	//now we'll loop through the waypoints and find out if there's enough time to
	//merge the first with the second. Note there is an implicit assumption here
	//that the first role will be done first, then the second
	int last_used_wp = -1;
	int time_since_used_wp = 0;
	for (int wp_idx=0; wp_idx < pWaypoints->GetNumPts(); wp_idx++) {

	  if (loc_info[wp_idx][role_num].type == LT_Empty) {
	    if (last_used_wp < 0) {
	      if (loc_info[wp_idx][search_role_num].type != LT_Empty) {
		actionlog(60) << "RM: Can't reuse role " << role_num << " for role "
			      << search_role_num << " because of wp " << wp_idx << " usage"
			      << ende;
		break; 
	      }
	      continue; //haven't started this role yet
	    }
	  
	    time_since_used_wp += timeForWP[wp_idx];
	    if (loc_info[wp_idx][search_role_num].type == LT_Empty ||
		loc_info[wp_idx][search_role_num].type == LT_JustOff)
	      continue; //second role hasn't really started yet
	    int time_for_player = UpperBForPlayerToPoint(loc_info[last_used_wp][role_num].pos,
							 loc_info[wp_idx][search_role_num].pos);
	    if ( time_for_player <= time_since_used_wp) {
	      actionlog(60) << "RM: Reusing role " << role_num << " for role " << search_role_num
			    << " (" << time_for_player << " <= " << time_since_used_wp << ")"
			    << ende;
	      actionlog(70) << "RM: extra dist is "
			    << loc_info[last_used_wp][role_num].pos
			    << " to "
			    << loc_info[wp_idx][search_role_num].pos
			    << ende;
	      //NOTE: that we reuse wp_idx- that is intentional- we want to drop out
	      // out the wp_idx loop anyway
	      for (int wp_idx=0; wp_idx < pWaypoints->GetNumPts(); wp_idx++) {
		if (loc_info[wp_idx][role_num].type == LT_Empty) 
		  loc_info[wp_idx][role_num] = loc_info[wp_idx][search_role_num];
		loc_info[wp_idx][search_role_num].type = LT_Empty;
	      }
	      roleDone = TRUE;
	      break;
	    } else {
	      actionlog(60) << "RM: Can't reuse role " << role_num << " for role "
			    << search_role_num << " (" << time_for_player
			    << " > " << time_since_used_wp << ")" << ende;
	      break;
	    }
	  }

	  //now we know that first role is not empty
	  if (loc_info[wp_idx][search_role_num].type != LT_Empty) {
	    actionlog(240) << "Role " << role_num << " can not be combined with role "
			   << search_role_num << " because both have spot at wp "
			   << wp_idx << ende;
	    break;
	  }
	  last_used_wp = wp_idx;
	} //wp_idx loop

      }
    }
  } //try to reuse agents
  
  /* Step 4: compile into AgentBallMovements */
  output->Clear();
  //First, set all the initial poses
  for (pnum=0; pnum < NUM_PLAYERS; pnum++) {
    if (loc_info[0][pnum].type == LT_ReceiverOnside ||
	loc_info[0][pnum].type == LT_ReceiverOffside)
      continue; // this is the kicker! we'll handle him below
    for (wp_idx=0; wp_idx < pWaypoints->GetNumPts(); wp_idx++) {
      if (loc_info[wp_idx][pnum].type != LT_Empty) {
	VecPosition pos = loc_info[wp_idx][pnum].pos;
	if (theProb->mode == SPM_My_Kick_Off) {
	  actionlog(90) << "RM: Moving " << pnum << " onto correct half, "
			<< pos.getX() << " -> " << PtToOnHalfPt(my_side, pos).getX() << ende;
	  pos = PtToOnHalfPt(my_side, pos, 1.0);
	}
	actionlog(70) << "RM: Adding initial PlayerLoc " << pnum << " const: "
		      << ABMO_NoConstraint << " " << loc_info[wp_idx][pnum].pos 
		      << ende;
	output->AddPlayerLoc(pnum+1, pos, ABMO_NoConstraint);
	break;
      }
    }
  }

  //now compile the levels
  for (wp_idx=0; wp_idx < pWaypoints->GetNumPts(); wp_idx++) {

    //First set the kicker node
    int kicker = -1;
    //first do the kicker node
    for (pnum = 0; pnum < NUM_PLAYERS; pnum++) {
      if (loc_info[wp_idx][pnum].type == LT_ReceiverOnside ||
	  loc_info[wp_idx][pnum].type == LT_ReceiverOffside) {
	kicker = pnum;
	break;
      }
    }
    
    if (kicker != -1) {

      if (loc_info[wp_idx][kicker].type == LT_ReceiverOnside) {
	actionlog(70) << "RM: Waypoint " << wp_idx << " has onside receiver " << kicker << " at "
		      << loc_info[wp_idx][kicker].pos << ende;
	output->AddPlayerLoc(kicker+1, loc_info[wp_idx][kicker].pos, ABMO_AfterLastEnd);
      } else if (loc_info[wp_idx][kicker].type == LT_ReceiverOffside) {
	actionlog(70) << "RM: Waypoint " << wp_idx << " has offside receiver " << kicker << " at "
		      << loc_info[wp_idx][kicker].pos << ende;
	output->AddPlayerLoc(kicker+1, loc_info[wp_idx][kicker].pos, ABMO_AfterThisStart);
      } else {
	errorlog << "What type is the kick node? "
		 << wp_idx << " "
		 << kicker << " " 
		 << loc_info[wp_idx][kicker].type << ende;
      }

      //and then the ball pos, so everyone knows this is the kicker/receiver
      output->AddBallLoc(loc_info[wp_idx][kicker].pos);

    } else {
      //this is the final send. let's just make sure that this is really the end
      actionlog(70) << "Waypoint " << wp_idx << " is a send" << ende;
      output->AddBallLoc(pWaypoints->GetPt(wp_idx));
      if (wp_idx != pWaypoints->GetNumPts() - 1)
	errorlog << "RM: It looks like there is a send in the middle of the waypoints..." << ende;
    }

    //now handle all these extra gotos
    kicker = -1;
    for (pnum = 0; pnum < NUM_PLAYERS; pnum++) {
      switch (loc_info[wp_idx][pnum].type) {
      case LT_Empty:
	break;
      case LT_ReceiverOnside:
      case LT_ReceiverOffside:
	if (kicker != -1)
	  errorlog << "Do have two kickers? "
		   << kicker << " "
		   << pnum << " "
		   << wp_idx << ende;
	kicker = pnum;
	break;
      case LT_Onside:
	if (wp_idx == 0) {
	  if (!loc_info[wp_idx][pnum].first_pos_set) 
	    errorlog << "first_pos_set should be true for level 0 OnSide node" << ende;
	  /* we don't add nodes for level 0 waypoints!
	     They are already there because of initial pos
	  actionlog(70) << "RM: Adding onside (0) PlayerLoc %d const: % << ende; (%.2f %.2f)",
		     pnum, ABMO_NoConstraint, loc_info[wp_idx][pnum].pos.getX(), 
		     loc_info[wp_idx][pnum].pos.getY());
	  output->AddPlayerLoc(pnum+1, loc_info[wp_idx][pnum].pos, ABMO_NoConstraint);
	  */
	} else {
	  if (!loc_info[wp_idx][pnum].first_pos_set) {
	    actionlog(70) << "RM: Adding onside (1) PlayerLoc " << pnum << " const: "
			  << ABMO_AfterLastStart << ", " << loc_info[wp_idx][pnum].pos
			  << ende;
	    output->AddPlayerLoc(pnum+1, loc_info[wp_idx][pnum].pos, ABMO_AfterLastStart);
	  }
	}
	break;
      case LT_JustOff:
	if (wp_idx == 0) {
	  if (!loc_info[wp_idx][pnum].first_pos_set) 
	    errorlog << "first_pos_set should be true for level 0 OnSide node" << ende;
	  /* we don't add nodes for level 0 waypoints!
	     They are already there because of initial pos
	  actionlog(70) << "RM: Adding justoff (0) PlayerLoc %d const: % << ende; (%.2f %.2f)",
		     pnum, ABMO_NoConstraint, loc_info[wp_idx][pnum].pos.getX(), 
		     loc_info[wp_idx][pnum].pos.getY());
	  output->AddPlayerLoc(pnum+1, loc_info[wp_idx][pnum].pos, ABMO_NoConstraint);
	  */
	} else {
	  if (!loc_info[wp_idx][pnum].first_pos_set) {
	    actionlog(70) << "RM: Adding justoff (1) PlayerLoc " << pnum << " const: "
			  << ABMO_AfterLastLastEnd << ", " << loc_info[wp_idx][pnum].pos
			  << ende;
	    output->AddPlayerLoc(pnum+1, loc_info[wp_idx][pnum].pos, ABMO_AfterLastLastEnd);
	  }
	}
	break;
      default:
	errorlog << "What is the loc info type? "
		 << wp_idx << " "
		 << pnum << " " 
		 << loc_info[wp_idx][pnum].type << ende;
	break;
      }
    } //pnum loop

  }
  
  return true;
}

  
void OffPointOffsidesRM::ClearLocInfo(int num_wp)
{
  for (int wp=0; wp < num_wp; wp++) {
    for (int pnum=0; pnum < NUM_PLAYERS; pnum++) {
      loc_info[wp][pnum].type = LT_Empty;
    }
  }
}

bool OffPointOffsidesRM::IsRoleUsed(int role, int num_wp)
{
  for (int wp=0; wp < num_wp; wp++) {
    if (loc_info[wp][role].type != LT_Empty)
      return true;
  }
  return false;
}



/************************* Level 3: STNCompilation ****************************/

/************************* Level 4: AgentInstantiation ****************************/
FixedPlanAI::FixedPlanAI(TeamSide my_side, int goalie_num)
  : my_side(my_side), goalie_num(goalie_num)
{
  //std::ifstream infile("Plans/example2");
  //std::ifstream infile("Plans/justinit");
  //std::ifstream infile("Plans/justsend");
  //std::ifstream infile("Plans/sendgo");
  //std::ifstream infile("Plans/1pass");
  //std::ifstream infile("Plans/1passsend");
  //std::ifstream infile("Plans/justdribble");
  std::ifstream infile("Plans/passdribpass");

  if (!infile)
    errorlog << "FixedPlanAI: could not open plan file" << ende;  

  fixedPlan = new Plan;
  if (!fixedPlan)
    errorlog << "FixedPlanAI: plan failed to allocate" << ende;  

  if (!fixedPlan->Read(infile))
    errorlog << "FixedPlanAI: couldn't read plan" << ende;

  if (!(my_side == TS_Left || my_side == TS_Right))
    errorlog << "FixedPlanAI: don't know what side I'm on" << ende;
  
  //the fixed plan is given for left side. If we are right, we have to flip
  if (my_side == TS_Right)
    fixedPlan->FlipCoords();

  cout << "Original Plan:" << endl;
  fixedPlan->Print(cout);
  cout << "message: " << fixedPlan->PrintStrCompact() << endl;  

  /*
  fixedPlan->ComputeAllPairsDistances();
  cout << "All Pair distances:" << endl;
  fixedPlan->PrintOnlyEdgeLens(cout);
  cout << "message: " << fixedPlan->PrintStrCompact() << endl;  

  fixedPlan->MakeMinimalDispatchable();  
  cout << "Min Dispatch:" << endl;
  fixedPlan->PrintOnlyEdgeLens(cout);
  cout << "message: " << fixedPlan->PrintStrCompact() << endl;  
  */

}

FixedPlanAI::~FixedPlanAI()
{
  delete fixedPlan;  
}

bool FixedPlanAI::run(Plan* thePlan, Problem* theProb)
{
  thePlan->copyFrom(fixedPlan);

  while (pTimeLeft && *pTimeLeft != TL_Normal)
    ; //spin until we are forced to give an answer

  return true;  
}


GreedyByRoleAI::GreedyByRoleAI(ModFormation* p, TeamSide my_side, int goalie_num)
  : AgentInstantiation(), my_side(my_side), goalie_num(goalie_num), pmFormation(p) 
{
  pSeqAI = new SequentialAI(1);
  arrRep = new int[ServerParam::instance()->getSPTeamSize()+1];
  posAssigned = new bool[ServerParam::instance()->getSPTeamSize()+1];
}

GreedyByRoleAI::~GreedyByRoleAI()
{
  delete pSeqAI;
  delete [] arrRep;  
  delete [] posAssigned;
}


bool GreedyByRoleAI::run(Plan* thePlan, Problem* theProb)
{
  /* Go through the positions in order, finding the closest player
     to that position, and mark the player as going there       */
  /* When it's me set my mode to Insetplay                      */

  actionlog(60) << "GreedyByRoleAI: running" << ende;
  
  if (!pmFormation) {
    errorlog << "GreedyByRoleAI needs ModFormation" << ende;
    return false;
  }

  //this will insure that our numbering start at 1 and it continuous
  if (!pSeqAI->run(thePlan, theProb)) {
    errorlog << "GreedyByRoleAI: SequentialAI failed" << ende;
    return false;
  }
  
  float minDist;
  int closestNum=0, i;
  int num_roles = 0;
  InitialPosPN* pIP = thePlan->GetInitPos();  

  for (i=1; i<=ServerParam::instance()->getSPTeamSize(); i++) 
    if (thePlan->IsPlayerInvolved(i,0)) 
      num_roles++;      

  for (i=1; i<=ServerParam::instance()->getSPTeamSize(); i++) 
    posAssigned[i] = FALSE;

  for (i=1; i<=num_roles; i++){
    minDist = HUGE;
    if ( i==1 && theProb->mode == SPM_My_Goalie_Catch ){
      /* Goalie starts the setplay after a catch */
      closestNum = goalie_num;
    }
    else {
      for (int num=1; num<=ServerParam::instance()->getSPTeamSize(); num++){
	if (goalie_num == num)
	  continue; /* goalie not in most set plays */
	if (posAssigned[num])
	  continue;
	VecPosition home_pt = pmFormation->getCurrentFormation()->getHomeRegionCentralPoint(num);
	if (my_side == TS_Right)
	  home_pt = home_pt.flipCoords();
	float dist = home_pt.getDistanceTo(pIP->GetPlayerPos(i));

	/* SMURF: do we want some way to exclude certain agents? */
	if (dist < minDist){
	  minDist = dist;
	  closestNum = num;
	}
      }
      if (minDist >= HUGE - 1) {
	errorlog << "GreedyByRoleAI: doesn't look like position got assigned" << ende;
      }
    }

    arrRep[i] = closestNum;      
    posAssigned[closestNum] = TRUE;

  } //role loop
  

  thePlan->ReplacePlayerNums(arrRep);

  return true;
}
