/* This is the module that controls the various functions */

//This is currently a bit specific in that it expects the planner to be the
//only module to return NotReady

#include "PlannerController.h"
#include "Logger.h"
using namespace spades;


PlannerController::PlannerController() 
{
  const_helper();  
}

PlannerController::PlannerController(Monitor* monitor, Diagnosis* diagnosis, 
				     Planner* planner, Executor* executor, 
				     Plan* defaultPlan) 
{
  const_helper();  

  this->monitor_list[0] = monitor;
  this->diagnosis_list[0] = diagnosis;
  this->planner_list[0] = planner;
  this->executor_list[0] = executor;
  this->defaultPlan = defaultPlan;
  defaultPlanExec->Load(defaultPlan, NULL);
}

void PlannerController::const_helper()
{
  for (int i=0; i<PC_max_num_modules; i++) {
    monitor_list[i] = NULL;
    diagnosis_list[i] = NULL;
    planner_list[i] = NULL;
    executor_list[i] = NULL;
  }
  
  rePlan = false;

  defaultPlan = NULL;
  problem = NULL;  

  thePlan = new Plan;
  thePlanExec = new PlanExecution; 
  
  defaultPlanExec = new PlanExecution;  
}




PlannerController::~PlannerController() 
{
  for (int i=0; i<PC_max_num_modules; i++) {
    if (monitor_list[i])   delete monitor_list[i];
    if (diagnosis_list[i]) delete diagnosis_list[i];
    if (planner_list[i])   delete planner_list[i];
    if (executor_list[i])  delete executor_list[i];
  }  
  delete defaultPlan;
  delete defaultPlanExec;
  delete thePlan;
  delete thePlanExec;
  delete problem;  
}

    
  // public utility functions	
bool PlannerController::addMonitor(Monitor* monitor) 
{
  if (monitor == NULL)
    errorlog << "Trying to add null monitor!" << ende;  
  for (int i=0; i<PC_max_num_modules; i++) {
    if (monitor_list[i] != NULL) continue;    
    monitor_list[i] = monitor;    
    return true;    
  }
  errorlog << "PC: Tried to add too many monitor!" << ende;
  return false;
}

bool PlannerController::addDiagnosis(Diagnosis* diagnosis)
{
  if (diagnosis == NULL)
    errorlog << "Trying to add null diagnosis!" << ende;  
  for (int i=0; i<PC_max_num_modules; i++) {
    if (diagnosis_list[i] != NULL) continue;    
    diagnosis_list[i] = diagnosis;    
    return true;    
  }
  errorlog << "PC: Tried to add too many diagnosis!" << ende;
  return false;
}

bool PlannerController::addPlanner(Planner* planner)
{
  if (planner == NULL)
    errorlog << "Trying to add null planner!" << ende;  
  for (int i=0; i<PC_max_num_modules; i++) {
    if (planner_list[i] != NULL) continue;    
    planner_list[i] = planner;    
    return true;    
  }
  errorlog << "PC: Tried to add too many planner!" << ende;
  return false;
}

bool PlannerController::addExecutor(Executor* executor)
{
  if (executor == NULL)
    errorlog << "Trying to add null executor!" << ende;  
  for (int i=0; i<PC_max_num_modules; i++) {
    if (executor_list[i] != NULL) continue;    
    executor_list[i] = executor;    
    return true;    
  }
  errorlog << "PC: Tried to add too many executor!" << ende;
  return false;
}


void PlannerController::setProblem(Problem *problem) 
{
  if (this->problem)
    delete this->problem;
  this->problem = problem;
  rePlan = true;  
}

void PlannerController::setDefaultPlan(Plan * defPlan) 
{
  if (this->defaultPlan)
    delete this->defaultPlan;
  this->defaultPlan = defPlan;
  initDefaultPlan();  
}

void PlannerController::initDefaultPlan()
{
  defaultPlanExec->Load(defaultPlan, problem);  
}



bool PlannerController::run_planner(PlanExecution** ppPE) 
{
  for (int i=0; i<PC_max_num_modules; i++) {
    if (!planner_list[i]) continue;	
    switch (planner_list[i]->run(problem, thePlan)) {
    case PlnRet_Complete: 
      actionlog(50) << "Planner " << i << " completed it's plan" << ende;      
      thePlanExec->Load(thePlan, problem);      
      *ppPE = thePlanExec; 
      rePlan = false; 
      thePlan->Log(60);      
      return true;
    case PlnRet_Failure: 
      actionlog(50) << "Planner " << i << " failed" << ende;      
      continue; //move to next planner      
    case PlnRet_NotReadyYet: 
      actionlog(50) << "Planner " << i << " not finished yet" << ende;      
      *ppPE = defaultPlanExec; return true;      
    case PlnRet_None:
    default:
      errorlog << "PC: What is planner " << i << " returning?" << ende;
      return false;      
      break;
    } //switch	
  }

  actionlog(50) << "Ran out of planners, returning false" << ende;  
  return false;  
}


PCReturn_t PlannerController::run()
{
  if (!monitor_list[0] || !diagnosis_list[0] || !planner_list[0] || !executor_list[0]) {
    errorlog << "PC: I have a NULL module!" << ende;
    return PCR_Error;    
  }

  PlanExecution* currPlanExec; //the plan we are going to actually do stuff with
  
  if (problem == NULL) {
    //there is no problem, so we'll just stick with the default plan
    actionlog(40) << "PC: null problem "
		  << defaultPlan << " "
		  << defaultPlanExec << " "
		  << defaultPlanExec->GetUnderlyingPlan()
		  << ende;
    currPlanExec = defaultPlanExec;    
  } else if (rePlan) { 
    //a new problem got presented, or more info came in for planning
    actionlog(40) << "PC: rePlan flag set" << ende;    
    initDefaultPlan();
    if (!run_planner(&currPlanExec)) {
      errorlog << "PC: run_planner failed" << ende;      
      return PCR_Abort;
    }    
  } else {
    currPlanExec = thePlanExec; //we can execute what we have stored    
  }
  
  //run the monitors
  for (int mon_idx=0; mon_idx<PC_max_num_modules; mon_idx++) {
    if (!monitor_list[mon_idx]) continue;	
    switch (monitor_list[mon_idx]->run(currPlanExec)) {
    case MR_NotReadyYet:
      errorlog << "PC: Why is monitor not ready yet?" << ende;
      return PCR_Error; break;
    case MR_Normal:
      //nothing to do but keep executing
      actionlog(40) << "PC: monitor " << mon_idx << " normal" << ende;    
      break;    

    case MR_Deviation:
      actionlog(40) << "PC: monitor " << mon_idx << " said deviation" << ende;    
      //Now we need to call diagnosis modules
      int diag_idx;      
      for (diag_idx=0; diag_idx<PC_max_num_modules; diag_idx++) {
	if (!diagnosis_list[diag_idx]) continue;	
	switch (diagnosis_list[diag_idx]->run(currPlanExec)) {
	case DR_NotReadyYet:
	  errorlog << "PC: Why is diagnosis not ready yet?" << ende;
	  return PCR_Error; break;
	case DR_Modified:
	  actionlog(40) << "PC: diagnosis " << diag_idx << ": modified" << ende;    
	  diag_idx = -1; //signify that we are done	  
	  break;      
	case DR_NoChange:
	  //all that's left is to let execution have it
	  actionlog(40) << "PC: diagnosis " << diag_idx << ": no change" << ende;    
	  diag_idx = -1; //signify that we are done	  
	  break;
	case DR_RePlan:
	  actionlog(40) << "PC: diagnosis " << diag_idx << " requested replan" << ende;   
	  rePlan = true;      
	  //currPlanExec->Reset(); //this plan is gone
	  currPlanExec = defaultPlanExec;	  
	  if (!run_planner(&currPlanExec)) {
	    errorlog << "PC: run_planner failed" << ende;      
	    return PCR_Abort;
	  }    
	case DR_Failure:
	  actionlog(40) << "PC: diagnosis " << diag_idx << " said failure" << ende;    
	  //we'll move on to the next one
	  break;
	case DR_None:
	default:
	  errorlog << "PC: What is diagnosis returning?" << ende;
	  return PCR_Error; break;    
	} //diagnosis switch

	if (diag_idx == -1)
	  break; //we finished going through this loop

      } //diagnosis list loop

      if (diag_idx >= PC_max_num_modules) {
	actionlog(40) << "PC: all diagnosis failure" << ende;
	return PCR_Abort;	
      }
      break; //case MR_Deviation
    
    case MR_None:
    default:
      errorlog << "PC: What is the monitor " << mon_idx << " returning?" << ende;
      return PCR_Error; break;    
    } //monitor switch
  } //monitor loop
  
  
  for (int exec_idx=0; exec_idx<PC_max_num_modules; exec_idx++) {
    if (!executor_list[exec_idx]) continue;	
    switch (executor_list[exec_idx]->run(currPlanExec)) {
    case ER_Normal:
      actionlog(40) << "PC: execution normal" << ende;    
      //the actions are already sent
      break;    
    case ER_Complete:
      actionlog(40) << "PC: execution completed" << ende;    
      return PCR_Complete; break;
    case ER_Error:
      actionlog(40) << "PC: execution error" << ende;    
      return PCR_Error; break;
    case ER_Abort:
      actionlog(40) << "PC: execution wants abort" << ende;    
      return PCR_Abort; break;
    case ER_None:
    default:
      errorlog << "PC: What is the executor returning?" << ende;
      return PCR_Error; break;
    }
  } //executor list loop
    
  return PCR_Normal;
}

    
